"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CompositeDependable = exports.Grant = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("@aws-cdk/core");
const policy_statement_1 = require("./policy-statement");
/**
 * Result of a grant() operation
 *
 * This class is not instantiable by consumers on purpose, so that they will be
 * required to call the Grant factory functions.
 */
class Grant {
    constructor(props) {
        this.options = props.options;
        this.principalStatement = props.principalStatement;
        this.resourceStatement = props.resourceStatement;
        cdk.DependableTrait.implement(this, {
            get dependencyRoots() {
                return props.policyDependable ? cdk.DependableTrait.get(props.policyDependable).dependencyRoots : [];
            },
        });
    }
    /**
     * Grant the given permissions to the principal
     *
     * The permissions will be added to the principal policy primarily, falling
     * back to the resource policy if necessary. The permissions must be granted
     * somewhere.
     *
     * - Trying to grant permissions to a principal that does not admit adding to
     *   the principal policy while not providing a resource with a resource policy
     *   is an error.
     * - Trying to grant permissions to an absent principal (possible in the
     *   case of imported resources) leads to a warning being added to the
     *   resource construct.
     */
    static addToPrincipalOrResource(options) {
        var _c;
        jsiiDeprecationWarnings._aws_cdk_aws_iam_GrantWithResourceOptions(options);
        const result = Grant.addToPrincipal({
            ...options,
            scope: options.resource,
        });
        const resourceAndPrincipalAccountComparison = options.grantee.grantPrincipal.principalAccount
            ? cdk.Token.compareStrings(options.resource.env.account, options.grantee.grantPrincipal.principalAccount)
            : undefined;
        // if both accounts are tokens, we assume here they are the same
        const equalOrBothUnresolved = resourceAndPrincipalAccountComparison === cdk.TokenComparison.SAME
            || resourceAndPrincipalAccountComparison == cdk.TokenComparison.BOTH_UNRESOLVED;
        const sameAccount = resourceAndPrincipalAccountComparison
            ? equalOrBothUnresolved
            // if the principal doesn't have an account (for example, a service principal),
            // we should modify the resource's trust policy
            : false;
        // If we added to the principal AND we're in the same account, then we're done.
        // If not, it's a different account and we must also add a trust policy on the resource.
        if (result.success && sameAccount) {
            return result;
        }
        const statement = new policy_statement_1.PolicyStatement({
            actions: options.actions,
            resources: (options.resourceSelfArns || options.resourceArns),
            principals: [options.grantee.grantPrincipal],
        });
        const resourceResult = options.resource.addToResourcePolicy(statement);
        return new Grant({
            resourceStatement: statement,
            options,
            policyDependable: resourceResult.statementAdded ? (_c = resourceResult.policyDependable) !== null && _c !== void 0 ? _c : options.resource : undefined,
        });
    }
    /**
     * Try to grant the given permissions to the given principal
     *
     * Absence of a principal leads to a warning, but failing to add
     * the permissions to a present principal is not an error.
     */
    static addToPrincipal(options) {
        jsiiDeprecationWarnings._aws_cdk_aws_iam_GrantOnPrincipalOptions(options);
        const statement = new policy_statement_1.PolicyStatement({
            actions: options.actions,
            resources: options.resourceArns,
        });
        const addedToPrincipal = options.grantee.grantPrincipal.addToPrincipalPolicy(statement);
        if (!addedToPrincipal.statementAdded) {
            return new Grant({ principalStatement: undefined, options });
        }
        if (!addedToPrincipal.policyDependable) {
            throw new Error('Contract violation: when Principal returns statementAdded=true, it should return a dependable');
        }
        return new Grant({ principalStatement: statement, options, policyDependable: addedToPrincipal.policyDependable });
    }
    /**
     * Add a grant both on the principal and on the resource
     *
     * As long as any principal is given, granting on the principal may fail (in
     * case of a non-identity principal), but granting on the resource will
     * never fail.
     *
     * Statement will be the resource statement.
     */
    static addToPrincipalAndResource(options) {
        var _c;
        jsiiDeprecationWarnings._aws_cdk_aws_iam_GrantOnPrincipalAndResourceOptions(options);
        const result = Grant.addToPrincipal({
            ...options,
            scope: options.resource,
        });
        const statement = new policy_statement_1.PolicyStatement({
            actions: options.actions,
            resources: (options.resourceSelfArns || options.resourceArns),
            principals: [options.resourcePolicyPrincipal || options.grantee.grantPrincipal],
        });
        const resourceResult = options.resource.addToResourcePolicy(statement);
        const resourceDependable = resourceResult.statementAdded ? (_c = resourceResult.policyDependable) !== null && _c !== void 0 ? _c : options.resource : undefined;
        return new Grant({
            principalStatement: statement,
            resourceStatement: result.resourceStatement,
            options,
            policyDependable: resourceDependable ? new CompositeDependable(result, resourceDependable) : result,
        });
    }
    /**
     * Returns a "no-op" `Grant` object which represents a "dropped grant".
     *
     * This can be used for e.g. imported resources where you may not be able to modify
     * the resource's policy or some underlying policy which you don't know about.
     *
     * @param grantee The intended grantee
     * @param _intent The user's intent (will be ignored at the moment)
     */
    static drop(grantee, _intent) {
        jsiiDeprecationWarnings._aws_cdk_aws_iam_IGrantable(grantee);
        return new Grant({
            options: { grantee, actions: [], resourceArns: [] },
        });
    }
    /**
     * Whether the grant operation was successful
     */
    get success() {
        return this.principalStatement !== undefined || this.resourceStatement !== undefined;
    }
    /**
     * Throw an error if this grant wasn't successful
     */
    assertSuccess() {
        if (!this.success) {
            // eslint-disable-next-line max-len
            throw new Error(`${describeGrant(this.options)} could not be added on either identity or resource policy.`);
        }
    }
    /**
     * Make sure this grant is applied before the given constructs are deployed
     *
     * The same as construct.node.addDependency(grant), but slightly nicer to read.
     */
    applyBefore(...constructs) {
        for (const construct of constructs) {
            construct.node.addDependency(this);
        }
    }
}
exports.Grant = Grant;
_a = JSII_RTTI_SYMBOL_1;
Grant[_a] = { fqn: "@aws-cdk/aws-iam.Grant", version: "1.152.0" };
function describeGrant(options) {
    return `Permissions for '${options.grantee}' to call '${options.actions}' on '${options.resourceArns}'`;
}
/**
 * Composite dependable
 *
 * Not as simple as eagerly getting the dependency roots from the
 * inner dependables, as they may be mutable so we need to defer
 * the query.
 */
class CompositeDependable {
    constructor(...dependables) {
        cdk.DependableTrait.implement(this, {
            get dependencyRoots() {
                return Array.prototype.concat.apply([], dependables.map(d => cdk.DependableTrait.get(d).dependencyRoots));
            },
        });
    }
}
exports.CompositeDependable = CompositeDependable;
_b = JSII_RTTI_SYMBOL_1;
CompositeDependable[_b] = { fqn: "@aws-cdk/aws-iam.CompositeDependable", version: "1.152.0" };
//# sourceMappingURL=data:application/json;base64,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