"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Effect = exports.PolicyStatement = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("@aws-cdk/core");
const group_1 = require("./group");
const principals_1 = require("./principals");
const postprocess_policy_document_1 = require("./private/postprocess-policy-document");
const util_1 = require("./util");
const ensureArrayOrUndefined = (field) => {
    if (field === undefined) {
        return undefined;
    }
    if (typeof (field) !== 'string' && !Array.isArray(field)) {
        throw new Error('Fields must be either a string or an array of strings');
    }
    if (Array.isArray(field) && !!field.find((f) => typeof (f) !== 'string')) {
        throw new Error('Fields must be either a string or an array of strings');
    }
    return Array.isArray(field) ? field : [field];
};
/**
 * Represents a statement in an IAM policy document.
 */
class PolicyStatement {
    constructor(props = {}) {
        this.action = new Array();
        this.notAction = new Array();
        this.principal = {};
        this.notPrincipal = {};
        this.resource = new Array();
        this.notResource = new Array();
        this.condition = {};
        // Hold on to those principals
        this._principals = new Array();
        jsiiDeprecationWarnings._aws_cdk_aws_iam_PolicyStatementProps(props);
        // Validate actions
        for (const action of [...props.actions || [], ...props.notActions || []]) {
            if (!/^(\*|[a-zA-Z0-9-]+:[a-zA-Z0-9*]+)$/.test(action) && !cdk.Token.isUnresolved(action)) {
                throw new Error(`Action '${action}' is invalid. An action string consists of a service namespace, a colon, and the name of an action. Action names can include wildcards.`);
            }
        }
        this.sid = props.sid;
        this.effect = props.effect || Effect.ALLOW;
        this.addActions(...props.actions || []);
        this.addNotActions(...props.notActions || []);
        this.addPrincipals(...props.principals || []);
        this.addNotPrincipals(...props.notPrincipals || []);
        this.addResources(...props.resources || []);
        this.addNotResources(...props.notResources || []);
        if (props.conditions !== undefined) {
            this.addConditions(props.conditions);
        }
    }
    /**
     * Creates a new PolicyStatement based on the object provided.
     * This will accept an object created from the `.toJSON()` call
     * @param obj the PolicyStatement in object form.
     */
    static fromJson(obj) {
        const ret = new PolicyStatement({
            sid: obj.Sid,
            actions: ensureArrayOrUndefined(obj.Action),
            resources: ensureArrayOrUndefined(obj.Resource),
            conditions: obj.Condition,
            effect: obj.Effect,
            notActions: ensureArrayOrUndefined(obj.NotAction),
            notResources: ensureArrayOrUndefined(obj.NotResource),
            principals: obj.Principal ? [new JsonPrincipal(obj.Principal)] : undefined,
            notPrincipals: obj.NotPrincipal ? [new JsonPrincipal(obj.NotPrincipal)] : undefined,
        });
        // validate that the PolicyStatement has the correct shape
        const errors = ret.validateForAnyPolicy();
        if (errors.length > 0) {
            throw new Error('Incorrect Policy Statement: ' + errors.join('\n'));
        }
        return ret;
    }
    //
    // Actions
    //
    /**
     * Specify allowed actions into the "Action" section of the policy statement.
     *
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_action.html
     *
     * @param actions actions that will be allowed.
     */
    addActions(...actions) {
        if (actions.length > 0 && this.notAction.length > 0) {
            throw new Error('Cannot add \'Actions\' to policy statement if \'NotActions\' have been added');
        }
        this.action.push(...actions);
    }
    /**
     * Explicitly allow all actions except the specified list of actions into the "NotAction" section
     * of the policy document.
     *
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notaction.html
     *
     * @param notActions actions that will be denied. All other actions will be permitted.
     */
    addNotActions(...notActions) {
        if (notActions.length > 0 && this.action.length > 0) {
            throw new Error('Cannot add \'NotActions\' to policy statement if \'Actions\' have been added');
        }
        this.notAction.push(...notActions);
    }
    //
    // Principal
    //
    /**
     * Indicates if this permission has a "Principal" section.
     */
    get hasPrincipal() {
        return Object.keys(this.principal).length > 0 || Object.keys(this.notPrincipal).length > 0;
    }
    /**
     * Adds principals to the "Principal" section of a policy statement.
     *
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html
     *
     * @param principals IAM principals that will be added
     */
    addPrincipals(...principals) {
        jsiiDeprecationWarnings._aws_cdk_aws_iam_IPrincipal(principals);
        this._principals.push(...principals);
        if (Object.keys(principals).length > 0 && Object.keys(this.notPrincipal).length > 0) {
            throw new Error('Cannot add \'Principals\' to policy statement if \'NotPrincipals\' have been added');
        }
        for (const principal of principals) {
            this.validatePolicyPrincipal(principal);
            const fragment = principal.policyFragment;
            util_1.mergePrincipal(this.principal, fragment.principalJson);
            this.addPrincipalConditions(fragment.conditions);
        }
    }
    /**
     * Expose principals to allow their ARNs to be replaced by account ID strings
     * in policy statements for resources policies that don't allow full account ARNs,
     * such as AWS::Logs::ResourcePolicy.
     */
    get principals() {
        return [...this._principals];
    }
    /**
     * Specify principals that is not allowed or denied access to the "NotPrincipal" section of
     * a policy statement.
     *
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notprincipal.html
     *
     * @param notPrincipals IAM principals that will be denied access
     */
    addNotPrincipals(...notPrincipals) {
        jsiiDeprecationWarnings._aws_cdk_aws_iam_IPrincipal(notPrincipals);
        if (Object.keys(notPrincipals).length > 0 && Object.keys(this.principal).length > 0) {
            throw new Error('Cannot add \'NotPrincipals\' to policy statement if \'Principals\' have been added');
        }
        for (const notPrincipal of notPrincipals) {
            this.validatePolicyPrincipal(notPrincipal);
            const fragment = notPrincipal.policyFragment;
            util_1.mergePrincipal(this.notPrincipal, fragment.principalJson);
            this.addPrincipalConditions(fragment.conditions);
        }
    }
    validatePolicyPrincipal(principal) {
        if (principal instanceof group_1.Group) {
            throw new Error('Cannot use an IAM Group as the \'Principal\' or \'NotPrincipal\' in an IAM Policy');
        }
    }
    /**
     * Specify AWS account ID as the principal entity to the "Principal" section of a policy statement.
     */
    addAwsAccountPrincipal(accountId) {
        this.addPrincipals(new principals_1.AccountPrincipal(accountId));
    }
    /**
     * Specify a principal using the ARN  identifier of the principal.
     * You cannot specify IAM groups and instance profiles as principals.
     *
     * @param arn ARN identifier of AWS account, IAM user, or IAM role (i.e. arn:aws:iam::123456789012:user/user-name)
     */
    addArnPrincipal(arn) {
        this.addPrincipals(new principals_1.ArnPrincipal(arn));
    }
    /**
     * Adds a service principal to this policy statement.
     *
     * @param service the service name for which a service principal is requested (e.g: `s3.amazonaws.com`).
     * @param opts    options for adding the service principal (such as specifying a principal in a different region)
     */
    addServicePrincipal(service, opts) {
        jsiiDeprecationWarnings._aws_cdk_aws_iam_ServicePrincipalOpts(opts);
        this.addPrincipals(new principals_1.ServicePrincipal(service, opts));
    }
    /**
     * Adds a federated identity provider such as Amazon Cognito to this policy statement.
     *
     * @param federated federated identity provider (i.e. 'cognito-identity.amazonaws.com')
     * @param conditions The conditions under which the policy is in effect.
     *   See [the IAM documentation](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
     */
    addFederatedPrincipal(federated, conditions) {
        this.addPrincipals(new principals_1.FederatedPrincipal(federated, conditions));
    }
    /**
     * Adds an AWS account root user principal to this policy statement
     */
    addAccountRootPrincipal() {
        this.addPrincipals(new principals_1.AccountRootPrincipal());
    }
    /**
     * Adds a canonical user ID principal to this policy document
     *
     * @param canonicalUserId unique identifier assigned by AWS for every account
     */
    addCanonicalUserPrincipal(canonicalUserId) {
        this.addPrincipals(new principals_1.CanonicalUserPrincipal(canonicalUserId));
    }
    /**
     * Adds all identities in all accounts ("*") to this policy statement
     */
    addAnyPrincipal() {
        this.addPrincipals(new principals_1.AnyPrincipal());
    }
    //
    // Resources
    //
    /**
     * Specify resources that this policy statement applies into the "Resource" section of
     * this policy statement.
     *
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_resource.html
     *
     * @param arns Amazon Resource Names (ARNs) of the resources that this policy statement applies to
     */
    addResources(...arns) {
        if (arns.length > 0 && this.notResource.length > 0) {
            throw new Error('Cannot add \'Resources\' to policy statement if \'NotResources\' have been added');
        }
        this.resource.push(...arns);
    }
    /**
     * Specify resources that this policy statement will not apply to in the "NotResource" section
     * of this policy statement. All resources except the specified list will be matched.
     *
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notresource.html
     *
     * @param arns Amazon Resource Names (ARNs) of the resources that this policy statement does not apply to
     */
    addNotResources(...arns) {
        if (arns.length > 0 && this.resource.length > 0) {
            throw new Error('Cannot add \'NotResources\' to policy statement if \'Resources\' have been added');
        }
        this.notResource.push(...arns);
    }
    /**
     * Adds a ``"*"`` resource to this statement.
     */
    addAllResources() {
        this.addResources('*');
    }
    /**
     * Indicates if this permission has at least one resource associated with it.
     */
    get hasResource() {
        return this.resource && this.resource.length > 0;
    }
    //
    // Condition
    //
    /**
     * Add a condition to the Policy
     */
    addCondition(key, value) {
        const existingValue = this.condition[key];
        this.condition[key] = existingValue ? { ...existingValue, ...value } : value;
    }
    /**
     * Add multiple conditions to the Policy
     */
    addConditions(conditions) {
        Object.keys(conditions).map(key => {
            this.addCondition(key, conditions[key]);
        });
    }
    /**
     * Add a condition that limits to a given account
     */
    addAccountCondition(accountId) {
        this.addCondition('StringEquals', { 'sts:ExternalId': accountId });
    }
    /**
     * Create a new `PolicyStatement` with the same exact properties
     * as this one, except for the overrides
     */
    copy(overrides = {}) {
        var _b, _c, _d, _e, _f, _g;
        jsiiDeprecationWarnings._aws_cdk_aws_iam_PolicyStatementProps(overrides);
        return new PolicyStatement({
            sid: (_b = overrides.sid) !== null && _b !== void 0 ? _b : this.sid,
            effect: (_c = overrides.effect) !== null && _c !== void 0 ? _c : this.effect,
            actions: (_d = overrides.actions) !== null && _d !== void 0 ? _d : this.action,
            notActions: (_e = overrides.notActions) !== null && _e !== void 0 ? _e : this.notAction,
            principals: overrides.principals,
            notPrincipals: overrides.notPrincipals,
            resources: (_f = overrides.resources) !== null && _f !== void 0 ? _f : this.resource,
            notResources: (_g = overrides.notResources) !== null && _g !== void 0 ? _g : this.notResource,
        });
    }
    /**
     * JSON-ify the policy statement
     *
     * Used when JSON.stringify() is called
     */
    toStatementJson() {
        return postprocess_policy_document_1.normalizeStatement({
            Action: this.action,
            NotAction: this.notAction,
            Condition: this.condition,
            Effect: this.effect,
            Principal: this.principal,
            NotPrincipal: this.notPrincipal,
            Resource: this.resource,
            NotResource: this.notResource,
            Sid: this.sid,
        });
    }
    /**
     * String representation of this policy statement
     */
    toString() {
        return cdk.Token.asString(this, {
            displayHint: 'PolicyStatement',
        });
    }
    /**
     * JSON-ify the statement
     *
     * Used when JSON.stringify() is called
     */
    toJSON() {
        return this.toStatementJson();
    }
    /**
     * Add a principal's conditions
     *
     * For convenience, principals have been modeled as both a principal
     * and a set of conditions. This makes it possible to have a single
     * object represent e.g. an "SNS Topic" (SNS service principal + aws:SourcArn
     * condition) or an Organization member (* + aws:OrgId condition).
     *
     * However, when using multiple principals in the same policy statement,
     * they must all have the same conditions or the OR samentics
     * implied by a list of principals cannot be guaranteed (user needs to
     * add multiple statements in that case).
     */
    addPrincipalConditions(conditions) {
        // Stringifying the conditions is an easy way to do deep equality
        const theseConditions = JSON.stringify(conditions);
        if (this.principalConditionsJson === undefined) {
            // First principal, anything goes
            this.principalConditionsJson = theseConditions;
        }
        else {
            if (this.principalConditionsJson !== theseConditions) {
                throw new Error(`All principals in a PolicyStatement must have the same Conditions (got '${this.principalConditionsJson}' and '${theseConditions}'). Use multiple statements instead.`);
            }
        }
        this.addConditions(conditions);
    }
    /**
     * Validate that the policy statement satisfies base requirements for a policy.
     */
    validateForAnyPolicy() {
        const errors = new Array();
        if (this.action.length === 0 && this.notAction.length === 0) {
            errors.push('A PolicyStatement must specify at least one \'action\' or \'notAction\'.');
        }
        return errors;
    }
    /**
     * Validate that the policy statement satisfies all requirements for a resource-based policy.
     */
    validateForResourcePolicy() {
        const errors = this.validateForAnyPolicy();
        if (Object.keys(this.principal).length === 0 && Object.keys(this.notPrincipal).length === 0) {
            errors.push('A PolicyStatement used in a resource-based policy must specify at least one IAM principal.');
        }
        return errors;
    }
    /**
     * Validate that the policy statement satisfies all requirements for an identity-based policy.
     */
    validateForIdentityPolicy() {
        const errors = this.validateForAnyPolicy();
        if (Object.keys(this.principal).length > 0 || Object.keys(this.notPrincipal).length > 0) {
            errors.push('A PolicyStatement used in an identity-based policy cannot specify any IAM principals.');
        }
        if (Object.keys(this.resource).length === 0 && Object.keys(this.notResource).length === 0) {
            errors.push('A PolicyStatement used in an identity-based policy must specify at least one resource.');
        }
        return errors;
    }
}
exports.PolicyStatement = PolicyStatement;
_a = JSII_RTTI_SYMBOL_1;
PolicyStatement[_a] = { fqn: "@aws-cdk/aws-iam.PolicyStatement", version: "1.152.0" };
/**
 * The Effect element of an IAM policy
 *
 * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_effect.html
 */
var Effect;
(function (Effect) {
    /**
     * Allows access to a resource in an IAM policy statement. By default, access to resources are denied.
     */
    Effect["ALLOW"] = "Allow";
    /**
     * Explicitly deny access to a resource. By default, all requests are denied implicitly.
     *
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_evaluation-logic.html
     */
    Effect["DENY"] = "Deny";
})(Effect = exports.Effect || (exports.Effect = {}));
class JsonPrincipal extends principals_1.PrincipalBase {
    constructor(json = {}) {
        super();
        // special case: if principal is a string, turn it into a "LiteralString" principal,
        // so we render the exact same string back out.
        if (typeof (json) === 'string') {
            json = { [util_1.LITERAL_STRING_KEY]: [json] };
        }
        if (typeof (json) !== 'object') {
            throw new Error(`JSON IAM principal should be an object, got ${JSON.stringify(json)}`);
        }
        this.policyFragment = {
            principalJson: json,
            conditions: {},
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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