#!/usr/bin/env python3
"""
Keymaster HJY CLI - Interactive setup and management tools.

This CLI provides user-friendly commands for initializing and managing
your Keymaster HJY installation.
"""

import os
import sys
import textwrap
from pathlib import Path
from typing import Optional, Dict, Any

try:
    import click
except ImportError:
    click = None


def _ensure_click() -> None:
    """Ensure click is available, provide helpful error if not."""
    if click is None:
        print("❌ CLI functionality requires the 'click' package.")
        print("📦 Install it with: pip install click")
        print("🔧 Or install with CLI support: pip install keymaster_hjy[cli]")
        sys.exit(1)


def _print_banner() -> None:
    """Print the Keymaster banner."""
    banner = """
🔐 Keymaster HJY
Zero-config, professional-grade API security manager
    """
    print(banner.strip())


def _print_success(message: str) -> None:
    """Print a success message."""
    print(f"✅ {message}")


def _print_info(message: str) -> None:
    """Print an info message."""
    print(f"💡 {message}")


def _print_warning(message: str) -> None:
    """Print a warning message."""
    print(f"⚠️ {message}")


def _print_error(message: str) -> None:
    """Print an error message."""
    print(f"❌ {message}")


def _create_env_file(config: Dict[str, Any], file_path: Path) -> None:
    """Create the mysql.env configuration file."""
    env_content = textwrap.dedent(f"""
    # Keymaster HJY Configuration
    # Generated by: keymaster init
    
    # Database Configuration (Required)
    MYSQL_HOST={config['mysql_host']}
    MYSQL_PORT={config['mysql_port']}
    MYSQL_USER={config['mysql_user']}
    MYSQL_PASSWORD={config['mysql_password']}
    MYSQL_DATABASE={config['mysql_database']}
    
    # Redis Configuration (Optional - for distributed rate limiting)
    # Uncomment and configure if you have Redis available
    """).strip()
    
    if config.get('redis_enabled'):
        env_content += textwrap.dedent(f"""
        
        REDIS_HOST={config['redis_host']}
        REDIS_PORT={config['redis_port']}
        REDIS_PASSWORD={config['redis_password']}
        """)
    else:
        env_content += textwrap.dedent("""
        
        # REDIS_HOST=your-redis-host
        # REDIS_PORT=6379
        # REDIS_PASSWORD=your-redis-password
        """)
    
    file_path.write_text(env_content)


def _create_example_app(framework: str, project_dir: Path) -> None:
    """Create an example application file."""
    
    if framework == "fastapi":
        app_content = textwrap.dedent("""
        # example_app.py - FastAPI example with Keymaster HJY
        from fastapi import FastAPI
        from keymaster_hjy.integrations import fastapi_guard
        
        app = FastAPI(title="My Secure API", version="1.0.0")
        
        @app.get("/")
        async def root():
            return {"message": "Welcome to your secure API!"}
        
        @app.get("/api/public", dependencies=[fastapi_guard()])
        async def public_data():
            return {"message": "This endpoint requires a valid API key"}
        
        @app.get("/api/users", dependencies=[fastapi_guard("read:users")])
        async def get_users():
            return {"users": ["alice", "bob", "charlie"]}
        
        @app.post("/api/users", dependencies=[fastapi_guard("write:users")])
        async def create_user():
            return {"message": "User created successfully"}
        
        if __name__ == "__main__":
            import uvicorn
            print("🚀 Starting FastAPI server...")
            print("📖 API docs available at: http://127.0.0.1:8000/docs")
            uvicorn.run(app, host="127.0.0.1", port=8000, reload=True)
        """).strip()
        
    elif framework == "flask":
        app_content = textwrap.dedent("""
        # example_app.py - Flask example with Keymaster HJY
        from flask import Flask, jsonify
        from keymaster_hjy.integrations import flask_guard
        
        app = Flask(__name__)
        
        @app.route("/")
        def root():
            return {"message": "Welcome to your secure API!"}
        
        @app.route("/api/public")
        @flask_guard()
        def public_data():
            return jsonify({"message": "This endpoint requires a valid API key"})
        
        @app.route("/api/users")
        @flask_guard("read:users")
        def get_users():
            return jsonify({"users": ["alice", "bob", "charlie"]})
        
        @app.route("/api/users", methods=["POST"])
        @flask_guard("write:users")
        def create_user():
            return jsonify({"message": "User created successfully"})
        
        if __name__ == "__main__":
            print("🚀 Starting Flask server...")
            print("🔧 Debug mode enabled")
            app.run(host="127.0.0.1", port=5000, debug=True)
        """).strip()
    
    else:  # minimal
        app_content = textwrap.dedent("""
        # example_usage.py - Basic Keymaster HJY usage
        from keymaster_hjy import master
        
        def main():
            print("🔐 Keymaster HJY - Basic Usage Example")
            
            # Create an API key
            print("\\n📝 Creating a new API key...")
            key_info = master.keys.create(
                description="Example API key",
                tags=["example", "test"],
                scopes=["read:data", "write:data"]
            )
            
            print(f"✅ Key created successfully!")
            print(f"   ID: {key_info['id']}")
            print(f"   Key: {key_info['key']}")
            print(f"   🔒 Store this key securely!")
            
            # Validate the key
            print("\\n🔍 Validating the API key...")
            try:
                master.auth.validate_key(
                    key_info['key'],
                    required_scope="read:data"
                )
                print("✅ Key validation successful!")
            except Exception as e:
                print(f"❌ Key validation failed: {e}")
            
            print("\\n🎉 Setup complete! Your Keymaster HJY is ready to use.")
        
        if __name__ == "__main__":
            main()
        """).strip()
    
    (project_dir / "example_app.py").write_text(app_content)


def _interactive_setup() -> Dict[str, Any]:
    """Run interactive setup to gather configuration."""
    _ensure_click()
    
    config = {}
    
    print("\n🔧 Let's set up your Keymaster HJY configuration!\n")
    
    # Database configuration
    print("📊 Database Configuration (MySQL/MariaDB)")
    config['mysql_host'] = click.prompt("Database host", default="localhost")
    config['mysql_port'] = click.prompt("Database port", default=3306, type=int)
    config['mysql_user'] = click.prompt("Database username")
    config['mysql_password'] = click.prompt("Database password", hide_input=True)
    config['mysql_database'] = click.prompt("Database name")
    
    # Redis configuration
    print("\n🚀 Redis Configuration (Optional - for distributed rate limiting)")
    config['redis_enabled'] = click.confirm("Do you have Redis available?", default=False)
    
    if config['redis_enabled']:
        config['redis_host'] = click.prompt("Redis host", default="localhost")
        config['redis_port'] = click.prompt("Redis port", default=6379, type=int)
        config['redis_password'] = click.prompt("Redis password (leave empty if none)", default="", show_default=False)
    
    # Framework choice
    print("\n🎯 Framework Integration")
    framework_choices = ["fastapi", "flask", "minimal"]
    config['framework'] = click.prompt(
        "Which framework example would you like?",
        type=click.Choice(framework_choices),
        default="fastapi"
    )
    
    return config


def init_command(project_dir: Optional[str] = None) -> None:
    """Initialize a new Keymaster HJY project."""
    _print_banner()
    
    # Determine project directory
    if project_dir:
        project_path = Path(project_dir).resolve()
        if not project_path.exists():
            project_path.mkdir(parents=True)
    else:
        project_path = Path.cwd()
    
    print(f"📁 Initializing Keymaster HJY in: {project_path}")
    
    # Check if already initialized
    env_file = project_path / "mysql.env"
    if env_file.exists():
        _print_warning("mysql.env already exists!")
        if click and not click.confirm("Overwrite existing configuration?"):
            _print_info("Initialization cancelled.")
            return
    
    # Interactive setup
    config = _interactive_setup()
    
    # Create configuration files
    print("\n📝 Creating configuration files...")
    
    try:
        # Create mysql.env
        _create_env_file(config, env_file)
        _print_success(f"Created mysql.env")
        
        # Create example application
        _create_example_app(config['framework'], project_path)
        _print_success(f"Created example_app.py ({config['framework']} example)")
        
        # Create .env.example for reference
        example_env = project_path / ".env.example"
        example_env.write_text(textwrap.dedent("""
        # Example environment configuration for Keymaster HJY
        # Copy this to mysql.env and fill in your actual values
        
        MYSQL_HOST=your-mysql-host
        MYSQL_PORT=3306
        MYSQL_USER=your-username
        MYSQL_PASSWORD=your-password
        MYSQL_DATABASE=your-database
        
        # Optional Redis configuration
        REDIS_HOST=your-redis-host
        REDIS_PORT=6379
        REDIS_PASSWORD=your-redis-password
        """).strip())
        _print_success("Created .env.example")
        
        # Success message with next steps
        print("\n🎉 Keymaster HJY initialized successfully!")
        print("\n📋 Next steps:")
        print("   1. Review and update mysql.env with your actual database credentials")
        print("   2. Run your example application:")
        
        if config['framework'] == "fastapi":
            print("      pip install uvicorn")
            print("      python example_app.py")
        elif config['framework'] == "flask":
            print("      python example_app.py")
        else:
            print("      python example_app.py")
        
        print("   3. Create your first API key and start protecting your endpoints!")
        print("\n💡 Need help? Check the documentation at: https://github.com/hjy/keymaster_hjy#readme")
        
    except Exception as e:
        _print_error(f"Failed to create configuration: {e}")
        sys.exit(1)


def main() -> None:
    """Main CLI entry point."""
    if click is None:
        # Fallback for when click is not available
        if len(sys.argv) > 1 and sys.argv[1] == "init":
            project_dir = sys.argv[2] if len(sys.argv) > 2 else None
            init_command(project_dir)
        else:
            _print_banner()
            print("\n🚀 Available commands:")
            print("   keymaster init [directory]  - Initialize a new project")
            print("\n💡 For full CLI functionality, install: pip install keymaster_hjy[cli]")
        return
    
    # Full click-based CLI
    @click.group()
    @click.version_option(version="0.1.0", prog_name="keymaster")
    def cli():
        """Keymaster HJY - Professional API security manager."""
        pass
    
    @cli.command()
    @click.argument('directory', required=False)
    @click.option('--framework', type=click.Choice(['fastapi', 'flask', 'minimal']), 
                  help='Framework for example application')
    def init(directory: Optional[str], framework: Optional[str]) -> None:
        """Initialize a new Keymaster HJY project."""
        init_command(directory)
    
    @cli.command()
    def version() -> None:
        """Show version information."""
        _print_banner()
        print("Version: 0.1.0")
        print("Python package for zero-config API security")
    
    cli()


if __name__ == "__main__":
    main()
