import enum
import typing

from sqlalchemy.sql import ColumnElement, Select

from devtools.attrs import define
from devtools.query import interface


class OrderDirection(enum.Enum):
    ASC = "asc"
    DESC = "desc"


@define
class OrderBy:
    field: typing.Optional[str]
    direction: OrderDirection
    type_ = interface.ClauseType.APPLY

    @property
    def _should_apply(self):
        return self.field is not None

    @classmethod
    def none(cls):
        return cls(field=None, direction=OrderDirection.ASC)

    @classmethod
    def asc(cls, field: str):
        return cls(field, OrderDirection.ASC)

    @classmethod
    def desc(cls, field: str):
        return cls(field, OrderDirection.DESC)

    def apply(self, query: Select) -> Select:
        return (
            query.order_by(self._apply_order(self._find_column(query)))
            if self._should_apply
            else query
        )

    def _find_column(self, query: Select) -> ColumnElement:
        try:
            return next(col for col in query.selected_columns if col.key == self.field)
        except StopIteration:
            raise ValueError(f"Field {self.field} does not exist in query") from None

    def _apply_order(self, col: ColumnElement):
        return col.asc() if self.direction is OrderDirection.ASC else col.desc()
