"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const core_1 = require("@aws-solutions-constructs/core");
const lambda = require("aws-cdk-lib/aws-lambda");
const elb = require("aws-cdk-lib/aws-elasticloadbalancingv2");
const defaults = require("@aws-solutions-constructs/core");
// Note: All integration tests for alb are for HTTP APIs, as certificates require
// validation through DNS and email. This validation is impossible during our integration
// tests and the stack will fail to launch with an unvalidated certificate.
// Setup
const app = new aws_cdk_lib_1.App();
const stack = new aws_cdk_lib_1.Stack(app, core_1.generateIntegStackName(__filename), {
    env: { account: aws_cdk_lib_1.Aws.ACCOUNT_ID, region: 'us-east-1' },
});
stack.templateOptions.description = 'Integration test for alb with 2 Lambda targets';
const props = {
    lambdaFunctionProps: {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: 'index.handler'
    },
    listenerProps: {
        protocol: 'HTTP'
    },
    publicApi: true,
    albLoggingBucketProps: {
        removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        autoDeleteObjects: true
    }
};
const firstConstruct = new lib_1.AlbToLambda(stack, 'test-one', props);
const secondProps = {
    lambdaFunctionProps: {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: 'index.handler'
    },
    ruleProps: {
        conditions: [elb.ListenerCondition.pathPatterns(["*admin*"])],
        priority: 10
    },
    existingVpc: firstConstruct.vpc,
    existingLoadBalancerObj: firstConstruct.loadBalancer,
    publicApi: true
};
new lib_1.AlbToLambda(stack, 'test-two', secondProps);
defaults.addCfnSuppressRules(firstConstruct.listener, [
    { id: 'W56', reason: 'All integration tests must be HTTP because of certificate limitations.' },
]);
const newSecurityGroup = firstConstruct.loadBalancer.connections.securityGroups[0].node.defaultChild;
defaults.addCfnSuppressRules(newSecurityGroup, [
    { id: 'W29', reason: 'CDK created rule that blocks all traffic.' },
    { id: 'W2', reason: 'Rule does not apply for ELB.' },
    { id: 'W9', reason: 'Rule does not apply for ELB.' }
]);
defaults.suppressAutoDeleteHandlerWarnings(stack);
// Synth
app.synth();
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW50ZWcudHdvVGFyZ2V0cy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImludGVnLnR3b1RhcmdldHMudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBOzs7Ozs7Ozs7OztHQVdHOztBQUVILFVBQVU7QUFDViw2Q0FBNkQ7QUFDN0QsZ0NBQXVEO0FBQ3ZELHlEQUF3RTtBQUN4RSxpREFBaUQ7QUFDakQsOERBQThEO0FBQzlELDJEQUEyRDtBQUczRCxpRkFBaUY7QUFDakYseUZBQXlGO0FBQ3pGLDJFQUEyRTtBQUUzRSxRQUFRO0FBQ1IsTUFBTSxHQUFHLEdBQUcsSUFBSSxpQkFBRyxFQUFFLENBQUM7QUFDdEIsTUFBTSxLQUFLLEdBQUcsSUFBSSxtQkFBSyxDQUFDLEdBQUcsRUFBRSw2QkFBc0IsQ0FBQyxVQUFVLENBQUMsRUFBRTtJQUMvRCxHQUFHLEVBQUUsRUFBRSxPQUFPLEVBQUUsaUJBQUcsQ0FBQyxVQUFVLEVBQUUsTUFBTSxFQUFFLFdBQVcsRUFBRTtDQUN0RCxDQUFDLENBQUM7QUFDSCxLQUFLLENBQUMsZUFBZSxDQUFDLFdBQVcsR0FBRyxnREFBZ0QsQ0FBQztBQUVyRixNQUFNLEtBQUssR0FBcUI7SUFDOUIsbUJBQW1CLEVBQUU7UUFDbkIsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsU0FBUyxTQUFTLENBQUM7UUFDbEQsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsV0FBVztRQUNuQyxPQUFPLEVBQUUsZUFBZTtLQUN6QjtJQUNELGFBQWEsRUFBRTtRQUNiLFFBQVEsRUFBRSxNQUFNO0tBQ2pCO0lBQ0QsU0FBUyxFQUFFLElBQUk7SUFDZixxQkFBcUIsRUFBRTtRQUNyQixhQUFhLEVBQUUsMkJBQWEsQ0FBQyxPQUFPO1FBQ3BDLGlCQUFpQixFQUFFLElBQUk7S0FDeEI7Q0FDRixDQUFDO0FBQ0YsTUFBTSxjQUFjLEdBQUcsSUFBSSxpQkFBVyxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUUsS0FBSyxDQUFDLENBQUM7QUFFakUsTUFBTSxXQUFXLEdBQXFCO0lBQ3BDLG1CQUFtQixFQUFFO1FBQ25CLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLFNBQVMsU0FBUyxDQUFDO1FBQ2xELE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7UUFDbkMsT0FBTyxFQUFFLGVBQWU7S0FDekI7SUFDRCxTQUFTLEVBQUU7UUFDVCxVQUFVLEVBQUUsQ0FBQyxHQUFHLENBQUMsaUJBQWlCLENBQUMsWUFBWSxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQztRQUM3RCxRQUFRLEVBQUUsRUFBRTtLQUNiO0lBQ0QsV0FBVyxFQUFFLGNBQWMsQ0FBQyxHQUFHO0lBQy9CLHVCQUF1QixFQUFFLGNBQWMsQ0FBQyxZQUFZO0lBQ3BELFNBQVMsRUFBRSxJQUFJO0NBQ2hCLENBQUM7QUFDRixJQUFJLGlCQUFXLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRSxXQUFXLENBQUMsQ0FBQztBQUVoRCxRQUFRLENBQUMsbUJBQW1CLENBQUMsY0FBYyxDQUFDLFFBQVEsRUFBRTtJQUNwRCxFQUFFLEVBQUUsRUFBRSxLQUFLLEVBQUUsTUFBTSxFQUFFLHdFQUF3RSxFQUFFO0NBQ2hHLENBQUMsQ0FBQztBQUVILE1BQU0sZ0JBQWdCLEdBQUcsY0FBYyxDQUFDLFlBQVksQ0FBQyxXQUFXLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxZQUFnQyxDQUFDO0FBQ3pILFFBQVEsQ0FBQyxtQkFBbUIsQ0FBQyxnQkFBZ0IsRUFBRTtJQUM3QyxFQUFFLEVBQUUsRUFBRSxLQUFLLEVBQUUsTUFBTSxFQUFFLDJDQUEyQyxFQUFDO0lBQ2pFLEVBQUUsRUFBRSxFQUFFLElBQUksRUFBRSxNQUFNLEVBQUUsOEJBQThCLEVBQUM7SUFDbkQsRUFBRSxFQUFFLEVBQUUsSUFBSSxFQUFFLE1BQU0sRUFBRSw4QkFBOEIsRUFBQztDQUNwRCxDQUFDLENBQUM7QUFFSCxRQUFRLENBQUMsaUNBQWlDLENBQUMsS0FBSyxDQUFDLENBQUM7QUFDbEQsUUFBUTtBQUNSLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogIENvcHlyaWdodCBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICpcbiAqICBMaWNlbnNlZCB1bmRlciB0aGUgQXBhY2hlIExpY2Vuc2UsIFZlcnNpb24gMi4wICh0aGUgXCJMaWNlbnNlXCIpLiBZb3UgbWF5IG5vdCB1c2UgdGhpcyBmaWxlIGV4Y2VwdCBpbiBjb21wbGlhbmNlXG4gKiAgd2l0aCB0aGUgTGljZW5zZS4gQSBjb3B5IG9mIHRoZSBMaWNlbnNlIGlzIGxvY2F0ZWQgYXRcbiAqXG4gKiAgICAgIGh0dHA6Ly93d3cuYXBhY2hlLm9yZy9saWNlbnNlcy9MSUNFTlNFLTIuMFxuICpcbiAqICBvciBpbiB0aGUgJ2xpY2Vuc2UnIGZpbGUgYWNjb21wYW55aW5nIHRoaXMgZmlsZS4gVGhpcyBmaWxlIGlzIGRpc3RyaWJ1dGVkIG9uIGFuICdBUyBJUycgQkFTSVMsIFdJVEhPVVQgV0FSUkFOVElFU1xuICogIE9SIENPTkRJVElPTlMgT0YgQU5ZIEtJTkQsIGV4cHJlc3Mgb3IgaW1wbGllZC4gU2VlIHRoZSBMaWNlbnNlIGZvciB0aGUgc3BlY2lmaWMgbGFuZ3VhZ2UgZ292ZXJuaW5nIHBlcm1pc3Npb25zXG4gKiAgYW5kIGxpbWl0YXRpb25zIHVuZGVyIHRoZSBMaWNlbnNlLlxuICovXG5cbi8vIEltcG9ydHNcbmltcG9ydCB7IEF3cywgQXBwLCBTdGFjaywgUmVtb3ZhbFBvbGljeSB9IGZyb20gXCJhd3MtY2RrLWxpYlwiO1xuaW1wb3J0IHsgQWxiVG9MYW1iZGEsIEFsYlRvTGFtYmRhUHJvcHMgfSBmcm9tIFwiLi4vbGliXCI7XG5pbXBvcnQgeyBnZW5lcmF0ZUludGVnU3RhY2tOYW1lIH0gZnJvbSAnQGF3cy1zb2x1dGlvbnMtY29uc3RydWN0cy9jb3JlJztcbmltcG9ydCAqIGFzIGxhbWJkYSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtbGFtYmRhJztcbmltcG9ydCAqIGFzIGVsYiBmcm9tICdhd3MtY2RrLWxpYi9hd3MtZWxhc3RpY2xvYWRiYWxhbmNpbmd2Mic7XG5pbXBvcnQgKiBhcyBkZWZhdWx0cyBmcm9tICdAYXdzLXNvbHV0aW9ucy1jb25zdHJ1Y3RzL2NvcmUnO1xuaW1wb3J0IHsgQ2ZuU2VjdXJpdHlHcm91cCB9IGZyb20gXCJhd3MtY2RrLWxpYi9hd3MtZWMyXCI7XG5cbi8vIE5vdGU6IEFsbCBpbnRlZ3JhdGlvbiB0ZXN0cyBmb3IgYWxiIGFyZSBmb3IgSFRUUCBBUElzLCBhcyBjZXJ0aWZpY2F0ZXMgcmVxdWlyZVxuLy8gdmFsaWRhdGlvbiB0aHJvdWdoIEROUyBhbmQgZW1haWwuIFRoaXMgdmFsaWRhdGlvbiBpcyBpbXBvc3NpYmxlIGR1cmluZyBvdXIgaW50ZWdyYXRpb25cbi8vIHRlc3RzIGFuZCB0aGUgc3RhY2sgd2lsbCBmYWlsIHRvIGxhdW5jaCB3aXRoIGFuIHVudmFsaWRhdGVkIGNlcnRpZmljYXRlLlxuXG4vLyBTZXR1cFxuY29uc3QgYXBwID0gbmV3IEFwcCgpO1xuY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soYXBwLCBnZW5lcmF0ZUludGVnU3RhY2tOYW1lKF9fZmlsZW5hbWUpLCB7XG4gIGVudjogeyBhY2NvdW50OiBBd3MuQUNDT1VOVF9JRCwgcmVnaW9uOiAndXMtZWFzdC0xJyB9LFxufSk7XG5zdGFjay50ZW1wbGF0ZU9wdGlvbnMuZGVzY3JpcHRpb24gPSAnSW50ZWdyYXRpb24gdGVzdCBmb3IgYWxiIHdpdGggMiBMYW1iZGEgdGFyZ2V0cyc7XG5cbmNvbnN0IHByb3BzOiBBbGJUb0xhbWJkYVByb3BzID0ge1xuICBsYW1iZGFGdW5jdGlvblByb3BzOiB7XG4gICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KGAke19fZGlybmFtZX0vbGFtYmRhYCksXG4gICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzE2X1gsXG4gICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInXG4gIH0sXG4gIGxpc3RlbmVyUHJvcHM6IHtcbiAgICBwcm90b2NvbDogJ0hUVFAnXG4gIH0sXG4gIHB1YmxpY0FwaTogdHJ1ZSxcbiAgYWxiTG9nZ2luZ0J1Y2tldFByb3BzOiB7XG4gICAgcmVtb3ZhbFBvbGljeTogUmVtb3ZhbFBvbGljeS5ERVNUUk9ZLFxuICAgIGF1dG9EZWxldGVPYmplY3RzOiB0cnVlXG4gIH1cbn07XG5jb25zdCBmaXJzdENvbnN0cnVjdCA9IG5ldyBBbGJUb0xhbWJkYShzdGFjaywgJ3Rlc3Qtb25lJywgcHJvcHMpO1xuXG5jb25zdCBzZWNvbmRQcm9wczogQWxiVG9MYW1iZGFQcm9wcyA9IHtcbiAgbGFtYmRhRnVuY3Rpb25Qcm9wczoge1xuICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChgJHtfX2Rpcm5hbWV9L2xhbWJkYWApLFxuICAgIHJ1bnRpbWU6IGxhbWJkYS5SdW50aW1lLk5PREVKU18xNl9YLFxuICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJ1xuICB9LFxuICBydWxlUHJvcHM6IHtcbiAgICBjb25kaXRpb25zOiBbZWxiLkxpc3RlbmVyQ29uZGl0aW9uLnBhdGhQYXR0ZXJucyhbXCIqYWRtaW4qXCJdKV0sXG4gICAgcHJpb3JpdHk6IDEwXG4gIH0sXG4gIGV4aXN0aW5nVnBjOiBmaXJzdENvbnN0cnVjdC52cGMsXG4gIGV4aXN0aW5nTG9hZEJhbGFuY2VyT2JqOiBmaXJzdENvbnN0cnVjdC5sb2FkQmFsYW5jZXIsXG4gIHB1YmxpY0FwaTogdHJ1ZVxufTtcbm5ldyBBbGJUb0xhbWJkYShzdGFjaywgJ3Rlc3QtdHdvJywgc2Vjb25kUHJvcHMpO1xuXG5kZWZhdWx0cy5hZGRDZm5TdXBwcmVzc1J1bGVzKGZpcnN0Q29uc3RydWN0Lmxpc3RlbmVyLCBbXG4gIHsgaWQ6ICdXNTYnLCByZWFzb246ICdBbGwgaW50ZWdyYXRpb24gdGVzdHMgbXVzdCBiZSBIVFRQIGJlY2F1c2Ugb2YgY2VydGlmaWNhdGUgbGltaXRhdGlvbnMuJyB9LFxuXSk7XG5cbmNvbnN0IG5ld1NlY3VyaXR5R3JvdXAgPSBmaXJzdENvbnN0cnVjdC5sb2FkQmFsYW5jZXIuY29ubmVjdGlvbnMuc2VjdXJpdHlHcm91cHNbMF0ubm9kZS5kZWZhdWx0Q2hpbGQgYXMgQ2ZuU2VjdXJpdHlHcm91cDtcbmRlZmF1bHRzLmFkZENmblN1cHByZXNzUnVsZXMobmV3U2VjdXJpdHlHcm91cCwgW1xuICB7IGlkOiAnVzI5JywgcmVhc29uOiAnQ0RLIGNyZWF0ZWQgcnVsZSB0aGF0IGJsb2NrcyBhbGwgdHJhZmZpYy4nfSxcbiAgeyBpZDogJ1cyJywgcmVhc29uOiAnUnVsZSBkb2VzIG5vdCBhcHBseSBmb3IgRUxCLid9LFxuICB7IGlkOiAnVzknLCByZWFzb246ICdSdWxlIGRvZXMgbm90IGFwcGx5IGZvciBFTEIuJ31cbl0pO1xuXG5kZWZhdWx0cy5zdXBwcmVzc0F1dG9EZWxldGVIYW5kbGVyV2FybmluZ3Moc3RhY2spO1xuLy8gU3ludGhcbmFwcC5zeW50aCgpO1xuIl19