"""Execution context for NLQL queries."""

from dataclasses import dataclass, field
from typing import Any

from nlql.adapters.base import BaseAdapter
from nlql.config import NLQLConfig
from nlql.registry.embedding import EmbeddingProvider
from nlql.types.meta import MetaFieldRegistry


@dataclass
class ExecutionContext:
    """Context for query execution.

    This holds all the runtime state and configuration needed during query execution.

    Attributes:
        adapter: Data source adapter
        config: NLQL configuration
        meta_registry: META field type registry
        embedding_provider: Embedding provider for semantic search
        function_registry: Instance-level function registry (optional)
        operator_registry: Instance-level operator registry (optional)
        custom_data: Additional user-defined context data.
                    Reserved for future extensibility.
    """

    adapter: BaseAdapter
    config: NLQLConfig = field(default_factory=NLQLConfig.default)
    meta_registry: MetaFieldRegistry | None = None
    embedding_provider: EmbeddingProvider | None = None
    function_registry: Any = None  # FunctionRegistry | None
    operator_registry: Any = None  # OperatorRegistry | None
    custom_data: dict[str, Any] = field(default_factory=dict)  # TODO: Not implemented yet

    def __post_init__(self) -> None:
        """Initialize default values after dataclass initialization."""
        if self.meta_registry is None:
            from nlql.types.meta import get_meta_registry

            self.meta_registry = get_meta_registry()

        if self.embedding_provider is None:
            from nlql.registry.embedding import get_embedding_provider

            self.embedding_provider = get_embedding_provider()

