"""Exception hierarchy for NLQL.

This module defines all custom exceptions used throughout the NLQL library,
providing clear error messages with context information.
"""



class NLQLError(Exception):
    """Base exception for all NLQL errors."""

    pass


class NLQLParseError(NLQLError):
    """Raised when NLQL query parsing fails.

    Attributes:
        message: Error description
        line: Line number where error occurred (1-based)
        column: Column number where error occurred (1-based)
        context: Surrounding source text for context
    """

    def __init__(
        self,
        message: str,
        line: int | None = None,
        column: int | None = None,
        context: str | None = None,
    ) -> None:
        self.message = message
        self.line = line
        self.column = column
        self.context = context

        # Build detailed error message
        parts = [message]
        if line is not None and column is not None:
            parts.append(f"at line {line}, column {column}")
        if context:
            parts.append(f"\nContext:\n{context}")

        super().__init__(" ".join(parts))


class NLQLExecutionError(NLQLError):
    """Raised when query execution fails."""

    pass


class NLQLTypeError(NLQLError):
    """Raised when type checking or comparison fails.

    This includes:
    - Incompatible type comparisons
    - Missing type information for META fields
    - Invalid type conversions
    """

    pass


class NLQLAdapterError(NLQLError):
    """Raised when data source adapter encounters an error.

    This includes:
    - Connection failures
    - Unsupported operations
    - Data format issues
    """

    pass


class NLQLRegistryError(NLQLError):
    """Raised when registry operations fail.

    This includes:
    - Duplicate registrations
    - Missing required operators/functions
    - Invalid registration parameters
    """

    pass


class NLQLConfigError(NLQLError):
    """Raised when configuration is invalid or incomplete."""

    pass

