from typing import Dict, Optional, Union

from clabe import Clabe
from pydantic import BaseModel, Extra, StrictStr

from ..types.enums import CardStatus
from ..typing import DictStrAny
from .card import PaymentCardNumber, StrictPaymentCardNumber
from .general import StrictPositiveInt


class ExtraForbidBaseModel(BaseModel):
    class Config:
        extra = Extra.forbid


class TransferRequest(ExtraForbidBaseModel):
    recipient_name: StrictStr
    account_number: Union[Clabe, PaymentCardNumber]
    amount: StrictPositiveInt  # in centavos
    descriptor: StrictStr  # how it'll appear for the recipient
    idempotency_key: str  # must be unique for each transfer


class StrictTransferRequest(TransferRequest):
    account_number: Union[Clabe, StrictPaymentCardNumber]


class CardUpdateRequest(ExtraForbidBaseModel):
    user_id: Optional[str]
    ledger_account_id: Optional[str]
    status: Optional[CardStatus]


class CardRequest(ExtraForbidBaseModel):
    user_id: str
    ledger_account_id: str


class ApiKeyUpdateRequest(ExtraForbidBaseModel):
    user_id: Optional[str]
    metadata: Optional[Dict]


class ApiKeyRequest(ExtraForbidBaseModel):
    id: StrictStr
    secret: StrictStr
    user_id: StrictStr
    metadata: DictStrAny
