"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractDependencies = exports.tryGetModuleVersionFromPkg = exports.tryGetModuleVersionFromRequire = exports.exec = exports.findUp = exports.callsites = void 0;
const child_process_1 = require("child_process");
const fs = require("fs");
const path = require("path");
/**
 * Get callsites from the V8 stack trace API
 *
 * https://github.com/sindresorhus/callsites
 */
function callsites() {
    var _a;
    const _prepareStackTrace = Error.prepareStackTrace;
    Error.prepareStackTrace = (_, stack) => stack;
    const stack = (_a = new Error().stack) === null || _a === void 0 ? void 0 : _a.slice(1);
    Error.prepareStackTrace = _prepareStackTrace;
    return stack;
}
exports.callsites = callsites;
/**
 * Find a file by walking up parent directories
 */
function findUp(name, directory = process.cwd()) {
    const absoluteDirectory = path.resolve(directory);
    const file = path.join(directory, name);
    if (fs.existsSync(file)) {
        return file;
    }
    const { root } = path.parse(absoluteDirectory);
    if (absoluteDirectory === root) {
        return undefined;
    }
    return findUp(name, path.dirname(absoluteDirectory));
}
exports.findUp = findUp;
/**
 * Spawn sync with error handling
 */
function exec(cmd, args, options) {
    var _a, _b;
    const proc = child_process_1.spawnSync(cmd, args, options);
    if (proc.error) {
        throw proc.error;
    }
    if (proc.status !== 0) {
        if (proc.stdout || proc.stderr) {
            throw new Error(`[Status ${proc.status}] stdout: ${(_a = proc.stdout) === null || _a === void 0 ? void 0 : _a.toString().trim()}\n\n\nstderr: ${(_b = proc.stderr) === null || _b === void 0 ? void 0 : _b.toString().trim()}`);
        }
        throw new Error(`${cmd} ${args.join(' ')} ${(options === null || options === void 0 ? void 0 : options.cwd) ? `run in directory ${options.cwd}` : ''} exited with status ${proc.status}`);
    }
    return proc;
}
exports.exec = exec;
/**
 * Returns a module version by requiring its package.json file
 */
function tryGetModuleVersionFromRequire(mod) {
    try {
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        return require(`${mod}/package.json`).version;
    }
    catch (err) {
        return undefined;
    }
}
exports.tryGetModuleVersionFromRequire = tryGetModuleVersionFromRequire;
/**
 * Gets module version from package.json content
 */
function tryGetModuleVersionFromPkg(mod, pkgJson, pkgPath) {
    var _a, _b, _c;
    const dependencies = {
        ...(_a = pkgJson.dependencies) !== null && _a !== void 0 ? _a : {},
        ...(_b = pkgJson.devDependencies) !== null && _b !== void 0 ? _b : {},
        ...(_c = pkgJson.peerDependencies) !== null && _c !== void 0 ? _c : {},
    };
    if (!dependencies[mod]) {
        return undefined;
    }
    // If it's a "file:" version, make it absolute
    const fileMatch = dependencies[mod].match(/file:(.+)/);
    if (fileMatch && !path.isAbsolute(fileMatch[1])) {
        const absoluteFilePath = path.join(path.dirname(pkgPath), fileMatch[1]);
        return `file:${absoluteFilePath}`;
    }
    ;
    return dependencies[mod];
}
exports.tryGetModuleVersionFromPkg = tryGetModuleVersionFromPkg;
/**
 * Extract versions for a list of modules.
 *
 * First lookup the version in the package.json and then fallback to requiring
 * the module's package.json. The fallback is needed for transitive dependencies.
 */
function extractDependencies(pkgPath, modules) {
    var _a;
    const dependencies = {};
    // Use require for cache
    const pkgJson = require(pkgPath); // eslint-disable-line @typescript-eslint/no-require-imports
    for (const mod of modules) {
        const version = (_a = tryGetModuleVersionFromPkg(mod, pkgJson, pkgPath)) !== null && _a !== void 0 ? _a : tryGetModuleVersionFromRequire(mod);
        if (!version) {
            throw new Error(`Cannot extract version for module '${mod}'. Check that it's referenced in your package.json or installed.`);
        }
        dependencies[mod] = version;
    }
    return dependencies;
}
exports.extractDependencies = extractDependencies;
//# sourceMappingURL=data:application/json;base64,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