"""
Integral Interface
------------------

An interface to libraries generated by
:func:`pySecDec.code_writer.make_package` or
:func:`pySecDec.loop_integral.loop_package`.

"""

from ctypes import CDLL, c_void_p, c_char_p, c_bool, c_int, c_uint, c_longlong, c_double, c_ulonglong, c_size_t
from threading import Thread
try:
    from Queue import Queue
except ImportError:
    from queue import Queue
import os
import os.path
from .misc import version

def _parse_series_coefficient(text):
    """
    Parse a textual representation of a single coefficient
    in a series. Return a float, a complex number, a tuple
    consisting of the mean and the standard deviation if it
    is given, or a tuple of three elements (via _parse_series)
    if the coefficient is a nested series.
    """
    if text.startswith(" + "):
        return _parse_series(text)
    if "+/-" in text:
        mean, stdev = text.split("+/-")
        return _parse_series_coefficient(mean.strip()), _parse_series_coefficient(stdev.strip())
    if text.startswith("(") and text.endswith(")") and "," in text:
        real, imag = text[1:-1].split(",")
        return complex(float(real), float(imag))
    return float(text)

def _parse_series(text):
    """
    Parse a textual representation of a series of complex numbers,
    according to the format of secdecutil/series.hpp. Return
    a list of terms, the variable name, and the truncation
    order. The last two can be None.
    """
    def iter_terms(text):
        i = j = 0
        while True:
            j = text.find(" + ", j)
            if j == -1: break
            if text.count("(", i, j) != text.count(")", i, j):
                j = j + 3
                continue
            yield text[i:j]
            i = j = j + 3
        yield text[i:]
    import re
    rx_term = re.compile(r"[(](.*)[)]([*]([^^]+)(?:\^([0-9+-]+))?)?")
    rx_order = re.compile(r"O[(]([^^]+)(?:\^([0-9+-]+))?[)]")
    def series(text):
        terms = []
        order = None
        variable = None
        for part in iter_terms(text):
            part = part.strip()
            if part == "": continue
            m = rx_term.fullmatch(part)
            if m is not None:
                value = _parse_series_coefficient(m.group(1))
                if m.group(3): variable = m.group(3)
                power = 0 if m.group(2) is None else \
                        1 if m.group(4) is None else \
                        int(m.group(4))
                terms.append((value, power))
                continue
            m = rx_order.fullmatch(part)
            if m is not None:
                variable = m.group(1)
                order = 1 if m.group(2) is None else int(m.group(2))
                continue
            if part.startswith("( + ") and part.endswith(")"):
                return series(part[1:-1])
            raise ValueError("Can't parse this term: " + part)
        return terms, variable, order
    return series(text)

def _parse_disteval_series(text):
    """
    Parse a series in the disteval format, return a list of entries
    arranged the same way as in _parse_series().
    """
    import re
    rx_term = re.compile(r"\+([^(]*)\*\(([+-].*[0-9])([+-].*[0-9])j\)(\*plusminus)?")
    result = []
    # Parse the text into a {(expo, ...): (value, error)} dictionary.
    coeffs = {}
    variables = None
    for line in text.splitlines():
        line = line.strip()
        m = rx_term.fullmatch(line)
        if m is not None:
            stem, real, imag, pm = m.groups()
            if variables is None:
                variables = [x.split("^")[0] for x in stem.split("*")]
            powers = tuple(int(x.split("^")[1]) for x in stem.split("*"))
            real = float(real)
            imag = float(imag)
            coeffs.setdefault(powers, [0, 0])
            coeffs[powers][0 if pm is None else 1] = \
                real if imag == 0 else complex(real, imag)
        elif line == "),":
            result.append({k:tuple(v) for k,v in coeffs.items()})
            coeffs = {}
        else:
            pass
    result.append({k:tuple(v) for k,v in coeffs.items()})
    # Turn it into a nested series.
    def rec(expmap, i):
        if i >= len(variables):
            assert(len(expmap) == 1)
            return next(iter(expmap.values()))
        co = {}
        for powers, coeff in expmap.items():
            power = powers[i]
            co.setdefault(power, {})
            co[power][powers] = coeff
        return [(rec(c, i+1), p) for p, c in co.items()], variables[i], max(co.keys()) + 1
    return [rec(expr, 0) for expr in result]

def _convert_series(series, power, Order):
    def fmt_value(val, mean):
        if isinstance(val, complex):
            return "(%.18g%+.18g*I)" % (val.real, val.imag) if mean else "0"
        if isinstance(val, complex) or isinstance(val, float):
            return "%.18g" % val  if mean else "0"
        if isinstance(val, tuple) and len(val) == 2:
            return fmt_value(val[0] if mean else val[1], True)
        if isinstance(val, tuple) and len(val) == 3:
            return "(" + fmt_series(*val, mean) + ")"
        raise ValueError(f"Can't convert value: {val}")
    def fmt_term(val, var, exp, mean):
        val = fmt_value(val, mean)
        return "%s" % val if exp == 0 else \
               "%s*%s" % (val, var) if exp == 1 else \
               "%s/%s" % (val, var) if exp == -1 else \
               "%s/%s%s%d" % (val, var, power, -exp) if exp < 0 else \
               "%s*%s%s%d" % (val, var, power, exp)
    def fmt_series(terms, var, order, mean):
        order = "" if order is None else \
                " + %s(%s)" % (Order, var,) if order == 1 else \
                " + %s(%s%s%d)" % (Order, var, power, order) if order > 0 else \
                " + %s(%s%s(%d))" % (Order, var, power, order)
        return " + ".join(fmt_term(val, var, exp, mean) for val, exp in terms) + order
    tvolist = \
            _parse_disteval_series(series) if series.startswith("[") else \
            [_parse_series(line) for line in series.splitlines()]
    results = [
        ( fmt_series(terms, variable, order, True), fmt_series(terms, variable, order, False) )
        for terms, variable, order in tvolist
    ]
    return results if len(results) > 1 else results[0]

def series_to_ginac(series):
    """
    Convert a textual representation of a series into GiNaC format.

    :param series:
        Any of the series obtained by calling an :class:`.IntegralLibrary` object.
    :type series: str
    :returns:
        Two strings: the series of mean values, and the series of standard deviations.
        The format of each returned value may look like this::

            (0+0.012665*I)/eps + (0+0.028632*I) + Order(eps)

        If there are multiple series specified, return a list of string pairs.
    """
    def fmt_order(var, order):
        return " + Order(%s)" % (var,) if order == 1 else \
               " + Order(%s^%d)" % (var, order)
    return _convert_series(series, "^", "Order")

def series_to_sympy(series):
    """
    Convert a textual representation of a series into SymPy format.

    :param series:
        Any of the series obtained by calling an :class:`.IntegralLibrary` object.
    :type series: str
    :returns:
        Two strings: the series of mean values, and the series of standard deviations.
        The format of each returned value may look like this::

            (0+0.012665*I)/eps + (0+0.028632*I) + O(eps)

        If there are multiple series specified, return a list of string pairs.
    """
    return _convert_series(series, "**", "O")

def series_to_mathematica(series):
    """
    Convert a textual representation of a series into Mathematica format.

    :param series:
        Any of the series obtained by calling an :class:`.IntegralLibrary` object.
    :type series: str
    :returns:
        Two strings: the series of mean values, and the series of standard deviations.
        The format of each returned value may look like this::

            (0+0.012665*I)/eps + (0+0.028632*I) + O[eps]

        If there are multiple series specified, return a list of string pairs.
    """
    def fmt_value(val, mean):
        if isinstance(val, float):
            return ("%.18g" % val).replace("e", "*10^") if mean else "0"
        if isinstance(val, complex):
            return ("(%.18g%+.18g*I)" % (val.real, val.imag)).replace("e", "*10^") if mean else "0"
        if isinstance(val, tuple) and len(val) == 2:
            return fmt_value(val[0] if mean else val[1], True)
        if isinstance(val, tuple) and len(val) == 3:
            return fmt_series(*val, mean)
        raise ValueError(f"Can't convert value: {val}")
    def fmt_term(val, var, exp, mean):
        val = fmt_value(val, mean)
        return "%s" % val if exp == 0 else \
               "%s*%s" % (val, var) if exp == 1 else \
               "%s/%s" % (val, var) if exp == -1 else \
               "%s/%s^%d" % (val, var, -exp) if exp < 0 else \
               "%s*%s^%d" % (val, var, exp)
    def fmt_series(terms, var, order, mean):
        if order is None:
            return " + ".join(fmt_term(val, var, exp, mean) for val, exp in terms)
        if any(isinstance(val, tuple) and len(val) == 3 for val, exp in terms):
            minexp = min(exp for val, exp in terms)
            if [exp for val, exp in terms] != list(range(minexp, order)):
                raise ValueError(f"Gaps in the series: {terms} + O({var}^{order})")
            return f"SeriesData[{var}, 0, {{" + \
                   ", ".join(fmt_term(val, var, 0, mean) for val, exp in terms) + \
                   f"}}, {minexp}, {order}, 1]"
        else:
            order = "" if order is None else \
                    " + O[%s]" % (var,) if order == 1 else \
                    " + O[%s]^%d" % (var, order) if order > 0 else \
                    " + O[%s]/%s^%d" % (var, var, 1-order)
            return " + ".join(fmt_term(val, var, exp, mean) for val, exp in terms) + order
    tvolist = \
            _parse_disteval_series(series) if series.startswith("[") else \
            [_parse_series(line) for line in series.splitlines()]
    results = [
        ( fmt_series(terms, variable, order, True), fmt_series(terms, variable, order, False) )
        for terms, variable, order in tvolist
    ]
    return results if len(results) > 1 else results[0]

def series_to_maple(series):
    """
    Convert a textual representation of a series into Maple format.

    :param series:
        Any of the series obtained by calling an :class:`.IntegralLibrary` object.
    :type series: str
    :returns:
        Two strings: the series of mean values, and the series of standard deviations.
        The format of each returned value may look like this::

            (0+0.012665*I)/eps + (0+0.028632*I) + O(eps)

        If there are multiple series specified, return a list of string pairs.
    """
    return _convert_series(series, "^", "O")

def series_to_json(series, name="sum"):
    """
    Convert a textual representation of a series into json format.

    :param series:
        str;
        Any of the series obtained by calling an :class:`.IntegralLibrary` object.
    :param name:
        str;
        The name of the series.
        Default: ``"sum"``.
    :returns:
        A dictionary containing the ``'regulators'`` and the value of the series
        stored as (exponent, value) key-value pairs.
    """
    def parse_terms(t, bottom_is_tuple, var=[], order=[], all_terms={'regulators':[]}):
        def is_bottom(t):
            return (not isinstance(t,tuple) and not isinstance(t,list))
        if is_bottom(t[0] if bottom_is_tuple else t):
            all_terms[tuple(order)] = t if bottom_is_tuple else (t,0.)
            all_terms['regulators'] = var.copy()
        elif isinstance(t,tuple):
            if len(t) == 3:
                var.append(t[1])
                parse_terms(t[0], bottom_is_tuple, var, order, all_terms)
                var.pop()
            elif len(t) == 2:
                order.append(t[1])
                parse_terms(t[0], bottom_is_tuple, var, order, all_terms)
                order.pop()
            else:
                raise RuntimeError(f'Unexpected tuple in series: {t}')
        elif isinstance(t,list):
            for e in t:
                parse_terms(e, bottom_is_tuple, var, order, all_terms)
        else:
            raise RuntimeError(f'Unexpected term in series: {t}')
        return all_terms
    terms = parse_terms(_parse_series(series), True if "+/-" in series else False)
    regulators = terms['regulators']
    del terms['regulators']
    result = {}
    result['regulators'] = regulators
    result['sums'] = {name: terms}
    return result

# assuming
# enum qmc_transform_t : int
# {
#     no_transform = -1,
#
#     baker = -2,
#
#     korobov1x1 = 1, korobov1x2 = 2, korobov1x3 = 3, korobov1x4 = 4, korobov1x5 = 5, korobov1x6 = 6,
#     korobov2x1 = 7, korobov2x2 = 8, korobov2x3 = 9, korobov2x4 = 10, korobov2x5 = 11, korobov2x6 = 12,
#     korobov3x1 = 13, korobov3x2 = 14, korobov3x3 = 15, korobov3x4 = 16, korobov3x5 = 17, korobov3x6 = 18,
#     korobov4x1 = 19, korobov4x2 = 20, korobov4x3 = 21, korobov4x4 = 22, korobov4x5 = 23, korobov4x6 = 24,
#     korobov5x1 = 25, korobov5x2 = 26, korobov5x3 = 27, korobov5x4 = 28, korobov5x5 = 29, korobov5x6 = 30,
#     korobov6x1 = 31, korobov6x2 = 32, korobov6x3 = 33, korobov6x4 = 34, korobov6x5 = 35, korobov6x6 = 36,
#
#     sidi1 = -11,
#     sidi2 = -12,
#     sidi3 = -13,
#     sidi4 = -14,
#     sidi5 = -15,
#     sidi6 = -16
# };
known_qmc_transforms = dict(
    none = -1,

    baker = -2,

    korobov1x1 = 1, korobov1x2 = 2, korobov1x3 = 3, korobov1x4 = 4, korobov1x5 = 5, korobov1x6 = 6,
    korobov2x1 = 7, korobov2x2 = 8, korobov2x3 = 9, korobov2x4 = 10, korobov2x5 = 11, korobov2x6 = 12,
    korobov3x1 = 13, korobov3x2 = 14, korobov3x3 = 15, korobov3x4 = 16, korobov3x5 = 17, korobov3x6 = 18,
    korobov4x1 = 19, korobov4x2 = 20, korobov4x3 = 21, korobov4x4 = 22, korobov4x5 = 23, korobov4x6 = 24,
    korobov5x1 = 25, korobov5x2 = 26, korobov5x3 = 27, korobov5x4 = 28, korobov5x5 = 29, korobov5x6 = 30,
    korobov6x1 = 31, korobov6x2 = 32, korobov6x3 = 33, korobov6x4 = 34, korobov6x5 = 35, korobov6x6 = 36,

    sidi1 = -11,
    sidi2 = -12,
    sidi3 = -13,
    sidi4 = -14,
    sidi5 = -15,
    sidi6 = -16
)
for i in range(1,7):
    known_qmc_transforms['korobov' + str(i)] = known_qmc_transforms['korobov%ix%i'%(i,i)]

# assuming
# enum qmc_fitfunction_t : int
# {
#     default_fitfunction = 0,
#
#     none = -1,
#     polysingular = 1
# };
known_qmc_fitfunctions = dict(
    default = 0,

    none = -1,
    polysingular = 1
)

# assuming
# enum qmc_generatingvectors_t: int
# {
#     default_generatingvectors = 0,
#
#     cbcpt_dn1_100 = 1,
#     cbcpt_dn2_6 = 2,
#     cbcpt_cfftw1_6 = 3,
#     cbcpt_cfftw2_10 = 4
# };
known_qmc_generatingvectors = dict(
    default = 0,

    cbcpt_dn1_100 = 1,
    cbcpt_dn2_6 = 2,
    cbcpt_cfftw1_6 = 3,
    cbcpt_cfftw2_10 = 4,
    none = 5
)

class CPPIntegrator(object):
    '''
    Abstract base class for integrators to be used with
    an :class:`.IntegralLibrary`.
    This class holds a pointer to the c++ integrator and
    defines the destructor.

    '''
    def __del__(self):
        if hasattr(self, 'c_integrator_ptr'):
            self.c_lib.free_integrator.restype = None
            self.c_lib.free_integrator.argtypes = [c_void_p]
            self.c_lib.free_integrator(self.c_integrator_ptr)

class MultiIntegrator(CPPIntegrator):
    '''
    .. versionadded:: 1.3.1

    Wrapper for the :cpp:class:`secdecutil::MultiIntegrator`.

    :param integral_library:
        :class:`IntegralLibrary`;
        The integral to be computed with this integrator.

    :param low_dim_integrator:
        :class:`CPPIntegrator`;
        The integrator to be used if the integrand is lower
        dimensional than `critical_dim`.

    :param high_dim_integrator:
        :class:`CPPIntegrator`;
        The integrator to be used if the integrand has dimension
        `critical_dim` or higher.

    :param critical_dim:
        integer;
        The dimension below which the `low_dimensional_integrator`
        is used.

    Use this class to switch between integrators based on the
    dimension of the integrand when integrating the `integral_ibrary`.
    For example, ":class:`CQuad` for 1D and :class:`Vegas` otherwise"
    is implemented as::

        integral_library.integrator = MultiIntegrator(integral_library,CQuad(integral_library),Vegas(integral_library),2)

    :class:`MultiIntegrator` can be nested to implement multiple
    critical dimensions. To use e.g. :class:`CQuad` for 1D,
    :class:`Cuhre` for 2D and 3D, and :class:`Vegas` otherwise, do::

        integral_library.integrator = MultiIntegrator(integral_library,CQuad(integral_library),MultiIntegrator(integral_library,Cuhre(integral_library),Vegas(integral_library),4),2)

    .. warning::
        The `integral_library` passed to the integrators must be the
        same for all of them. Furthermore, an integrator can only be
        used to integrate the `integral_library` it has been
        constructed with.

    .. warning::
        The :class:`MultiIntegrator` cannot be used with :class:`.CudaQmc`.

    '''
    def __init__(self,integral_library,low_dim_integrator,high_dim_integrator,critical_dim):
        self.low_dim_integrator = low_dim_integrator # keep reference to avoid deallocation
        self.high_dim_integrator = high_dim_integrator # keep reference to avoid deallocation
        self.c_lib = integral_library.c_lib
        self.c_lib_path = integral_library.c_lib_path
        self.c_lib.allocate_MultiIntegrator.restype = c_void_p
        self.c_lib.allocate_MultiIntegrator.argtypes = [c_void_p, c_void_p, c_int]
        self.c_integrator_ptr = self.c_lib.allocate_MultiIntegrator(low_dim_integrator.c_integrator_ptr,high_dim_integrator.c_integrator_ptr,critical_dim)

class CQuad(CPPIntegrator):
    '''
    Wrapper for the cquad integrator defined in the gsl
    library.

    :param integral_library:
        :class:`IntegralLibrary`;
        The integral to be computed with this integrator.

    The other options are defined in :numref:`chapter_cpp_cquad`
    and in the gsl manual.

    '''
    def __init__(self,integral_library,epsrel=1e-2,epsabs=1e-7,n=100,verbose=False,zero_border=0.0):
        self.c_lib = integral_library.c_lib
        self.c_lib_path = integral_library.c_lib_path
        self.c_lib.allocate_gsl_cquad.restype = c_void_p
        self.c_lib.allocate_gsl_cquad.argtypes = [c_double, c_double, c_uint, c_bool, c_double]
        self.c_integrator_ptr = self.c_lib.allocate_gsl_cquad(epsrel,epsabs,n,verbose,zero_border)
        self._epsrel=epsrel
        self._epsabs=epsabs

class Vegas(CPPIntegrator):
    '''
    Wrapper for the Vegas integrator defined in the cuba
    library.

    :param integral_library:
        :class:`IntegralLibrary`;
        The integral to be computed with this integrator.

    The other options are defined in :numref:`chapter_cpp_cuba`
    and in the cuba manual.

    '''
    def __init__(self,integral_library,epsrel=1e-2,epsabs=1e-7,flags=0,seed=0,mineval=10000,maxeval=4611686018427387903,zero_border=0.0,nstart=10000,nincrease=5000,nbatch=1000,real_complex_together=False):
        self.c_lib = integral_library.c_lib
        self.c_lib_path = integral_library.c_lib_path
        self.c_lib.allocate_cuba_Vegas.restype = c_void_p
        self.c_lib.allocate_cuba_Vegas.argtypes = [c_double, c_double, c_int, c_int, c_longlong, c_longlong, c_double, c_longlong, c_longlong, c_longlong, c_bool]
        self.c_integrator_ptr = self.c_lib.allocate_cuba_Vegas(epsrel,epsabs,flags,seed,mineval,maxeval,zero_border,nstart,nincrease,nbatch,real_complex_together)
        self._epsrel=epsrel
        self._epsabs=epsabs
        self._mineval=mineval
        self._maxeval=maxeval

class Suave(CPPIntegrator):
    '''
    Wrapper for the Suave integrator defined in the cuba
    library.

    :param integral_library:
        :class:`IntegralLibrary`;
        The integral to be computed with this integrator.

    The other options are defined in :numref:`chapter_cpp_cuba`
    and in the cuba manual.

    '''
    def __init__(self,integral_library,epsrel=1e-2,epsabs=1e-7,flags=0,seed=0,mineval=10000,maxeval=4611686018427387903,zero_border=0.0,nnew=1000,nmin=10,flatness=25.,real_complex_together=False):
        self.c_lib = integral_library.c_lib
        self.c_lib_path = integral_library.c_lib_path
        self.c_lib.allocate_cuba_Suave.restype = c_void_p
        self.c_lib.allocate_cuba_Suave.argtypes = [c_double, c_double, c_int, c_int, c_longlong, c_longlong, c_double, c_longlong, c_longlong, c_double, c_bool]
        self.c_integrator_ptr = self.c_lib.allocate_cuba_Suave(epsrel,epsabs,flags,seed,mineval,maxeval,zero_border,nnew,nmin,flatness,real_complex_together)
        self._epsrel=epsrel
        self._epsabs=epsabs
        self._mineval=mineval
        self._maxeval=maxeval

class Divonne(CPPIntegrator):
    '''
    Wrapper for the Divonne integrator defined in the cuba
    library.

    :param integral_library:
        :class:`IntegralLibrary`;
        The integral to be computed with this integrator.

    The other options are defined in :numref:`chapter_cpp_cuba`
    and in the cuba manual.

    '''
    def __init__(self, integral_library, epsrel=1e-2, epsabs=1e-7, flags=0, seed=0, mineval=10000, maxeval=4611686018427387903,zero_border=0.0,
                                         key1=2000, key2=1, key3=1, maxpass=4, border=0., maxchisq=1.,
                                         mindeviation=.15, real_complex_together=False):
        self.c_lib = integral_library.c_lib
        self.c_lib_path = integral_library.c_lib_path
        self.c_lib.allocate_cuba_Divonne.restype = c_void_p
        self.c_lib.allocate_cuba_Divonne.argtypes = [c_double, c_double, c_int, c_int, c_longlong, c_longlong,
                                                     c_double, c_int, c_int, c_int, c_int, c_double, c_double,
                                                     c_double, c_bool]
        self.c_integrator_ptr = self.c_lib.allocate_cuba_Divonne(epsrel, epsabs, flags, seed, mineval,maxeval,
                                                                 zero_border, key1, key2, key3, maxpass, border,
                                                                 maxchisq, mindeviation, real_complex_together)
        self._epsrel=epsrel
        self._epsabs=epsabs
        self._mineval=mineval
        self._maxeval=maxeval

class Cuhre(CPPIntegrator):
    '''
    Wrapper for the Cuhre integrator defined in the cuba
    library.

    :param integral_library:
        :class:`IntegralLibrary`;
        The integral to be computed with this integrator.

    The other options are defined in :numref:`chapter_cpp_cuba`
    and in the cuba manual.

    '''
    def __init__(self,integral_library,epsrel=1e-2,epsabs=1e-7,flags=0,mineval=10000,maxeval=4611686018427387903,zero_border=0.0,key=0,real_complex_together=False):
        self.c_lib = integral_library.c_lib
        self.c_lib_path = integral_library.c_lib_path
        self.c_lib.allocate_cuba_Cuhre.restype = c_void_p
        self.c_lib.allocate_cuba_Cuhre.argtypes = [c_double, c_double, c_int, c_longlong, c_longlong, c_double, c_int, c_bool]
        self.c_integrator_ptr = self.c_lib.allocate_cuba_Cuhre(epsrel,epsabs,flags,mineval,maxeval,zero_border,key,real_complex_together)
        self._epsrel=epsrel
        self._epsabs=epsabs
        self._mineval=mineval
        self._maxeval=maxeval

class Qmc(CPPIntegrator):
    '''
    Wrapper for the Qmc integrator defined in the integrators
    library.

    :param integral_library:
        :class:`IntegralLibrary`;
        The integral to be computed with this integrator.

    :param errormode:
        string;
        The `errormode` parameter of the Qmc, can be
        ``"default"``, ``"all"``, and ``"largest"``.
        ``"default"`` takes the default from the Qmc
        library. See the Qmc docs for details on the
        other settings.

    :param transform:
        string;
        An integral transform related to the parameter
        `P` of the Qmc. The possible choices
        correspond to the integral transforms of the
        underlying Qmc implementation. Possible values
        are, ``"none"``, ``"baker"``, ``sidi#``,
        ``"korobov#"``, and ``korobov#x#`` where
        any ``#`` (the rank of the Korobov/Sidi transform)
        must be an integer between 1 and 6.

    :param fitfunction:
        string;
        An integral transform related to the parameter
        `F` of the Qmc. The possible choices
        correspond to the integral transforms of the
        underlying Qmc implementation. Possible values
        are ``"default"``, ``"none"``, ``"polysingular"``.

    :param generatingvectors:
        string;
        The name of a set of generating vectors.
        The possible choices correspond to the available generating
        vectors of the underlying Qmc implementation. Possible values
        are ``"default"``, ``"cbcpt_dn1_100"``, ``"cbcpt_dn2_6"``,
        ``"cbcpt_cfftw1_6"``, and ``"cbcpt_cfftw2_10"``, ``"none"``.

        The ``"default"`` value will use all available generating
        vectors suitable for the highest dimension integral
        appearing in the library.

    :param lattice_candidates:
        int;
        Number of generating vector candidates used for median QMC rule.
        If standard_lattices=True, the median QMC is only used once the standard lattices are exhausted
        lattice_candidates=0 disables the use of the median QMC rule.
        Default: ``"0"``

    :param standard_lattices:
        bool; experimental
        Use pre-computed lattices instead of median QMC.
        Setting this parameter to ``"False"`` is equal to setting ``"generatingvectors=none"``
        Default: ``"False"``

    :param keep_lattices:
        bool; experimental
        Specifies if list of generating vectors generated using
        median Qmc rule should be kept for other integrals
        Default: ``"False"``


    :param cputhreads:
        int;
        The number of CPU threads that should be used to evaluate
        the integrand function.

        The default is the number of logical CPUs allocated to the
        current process (that is, ``len(os.sched_getaffinity(0))`` )
        on platforms that expose this information (i.e. Linux+glibc),
        or ``os.cpu_count()``.

        If GPUs are used, one additional CPU thread per device
        will be launched for communicating with the device. One
        can set ``cputhreads`` to zero to disable CPU evaluation
        in this case.

    .. seealso::
        The most important options are described in
        :numref:`chapter_cpp_qmc`.

    The other options are defined in the Qmc docs. If
    an argument is omitted then the default of the
    underlying Qmc implementation is used.

    '''
    def __init__(self,integral_library,transform='korobov3',fitfunction='default',generatingvectors='default',epsrel=1e-2,epsabs=1e-7,maxeval=4611686018427387903,errormode='default',evaluateminn=0,
                      minn=10000,minm=0,maxnperpackage=0,maxmperpackage=0,cputhreads=None,cudablocks=0,cudathreadsperblock=0,verbosity=0,seed=0,devices=[],lattice_candidates=0,standard_lattices=False,keep_lattices=False):
        if cputhreads is None:
            try:
                cputhreads = len(os.sched_getaffinity(0))
            except AttributeError:
                cputhreads = os.cpu_count()
        self.c_lib = integral_library.c_lib
        self.c_lib_path = integral_library.c_lib_path
        self.c_lib.allocate_integrators_Qmc.restype = c_void_p
        self.c_lib.allocate_integrators_Qmc.argtypes = [
                                                            c_double, # epsrel
                                                            c_double, # epsabs
                                                            c_ulonglong, # maxeval
                                                            c_int, # errormode
                                                            c_ulonglong, # evaluateminn
                                                            c_ulonglong, # minn
                                                            c_ulonglong, # minm
                                                            c_ulonglong, # maxnperpackage
                                                            c_ulonglong, # maxmperpackage
                                                            c_longlong, # cputhreads
                                                            c_ulonglong, # cudablocks
                                                            c_ulonglong, # cudathreadsperblock
                                                            c_ulonglong, # verbosity
                                                            c_longlong, # seed
                                                            c_int, # transform_id
                                                            c_int, # fitfunction_id
                                                            c_int, # generatingvectors_id
                                                            c_ulonglong, # lattice_candidates
                                                            c_bool, # standard_lattices
                                                            c_bool # keep_lattices
                                                      ]

        # assuming:
        # enum ErrorMode : int
        # {
        #     all = 1,
        #     largest = 2
        # };
        if errormode == 'default':
            errormode_enum = 0
        elif errormode == 'all':
            errormode_enum = 1
        elif errormode == 'largest':
            errormode_enum = 2
        else:
            raise ValueError('Unknown `errormode` "' + str(errormode) + '"')

        self.c_integrator_ptr = self.c_lib.allocate_integrators_Qmc(epsrel,epsabs,maxeval,errormode_enum,evaluateminn,minn,
                                                                    minm,maxnperpackage,maxmperpackage,cputhreads,
                                                                    cudablocks,cudathreadsperblock,verbosity,
                                                                    seed,known_qmc_transforms[str(transform).lower()],
                                                                    known_qmc_fitfunctions[str(fitfunction).lower()],
                                                                    known_qmc_generatingvectors[str(generatingvectors).lower()],
                                                                    lattice_candidates,standard_lattices,keep_lattices
                                                                   )
        self._epsrel=epsrel
        self._epsabs=epsabs
        self._mineval=minn
        self._maxeval=maxeval

class CudaQmc(object):
    '''
    Wrapper for the Qmc integrator defined in the integrators
    library for GPU use.

    :param integral_library:
        :class:`IntegralLibrary`;
        The integral to be computed with this integrator.

    :param errormode:
        string;
        The `errormode` parameter of the Qmc, can be
        ``"default"``, ``"all"``, and ``"largest"``.
        ``"default"`` takes the default from the Qmc
        library. See the Qmc docs for details on the
        other settings.

    :param transform:
        string;
        An integral transform related to the parameter
        `P` of the Qmc. The possible choices
        correspond to the integral transforms of the
        underlying Qmc implementation. Possible values
        are, ``"none"``, ``"baker"``, ``sidi#``,
        ``"korobov#"``, and ``korobov#x#`` where
        any ``#`` (the rank of the Korobov/Sidi transform)
        must be an integer between 1 and 6.

    :param fitfunction:
        string;
        An integral transform related to the parameter
        `F` of the Qmc. The possible choices
        correspond to the integral transforms of the
        underlying Qmc implementation. Possible values
        are ``"default"``, ``"none"``, ``"polysingular"``.

    :param generatingvectors:
        string;
        The name of a set of generating vectors.
        The possible choices correspond to the available generating
        vectors of the underlying Qmc implementation. Possible values
        are ``"default"``, ``"cbcpt_dn1_100"``, ``"cbcpt_dn2_6"``,
        ``"cbcpt_cfftw1_6"``, and ``"cbcpt_cfftw2_10"``.

    :param lattice_candidates:
        int;
        Number of generating vector candidates used for median QMC rule.
        If standard_lattices=True, the median QMC is only used once the standard lattices are exhausted
        lattice_candidates=0 disables the use of the median QMC rule.
        Default: ``"0"``

    :param standard_lattices:
        bool;
        Use pre-computed lattices instead of median QMC.
        Setting this parameter to ``"False"`` is equal to setting ``"generatingvectors=none"``
        Default: ``"False"``

    :param keep_lattices:
        bool;
        Specifies if list of generating vectors generated using
        median Qmc rule should be kept for other integrals
        Default: ``"False"``

    :param cputhreads:
        int;
        The number of CPU threads that should be used to evaluate
        the integrand function.

        The default is the number of logical CPUs allocated to the
        current process (that is, ``len(os.sched_getaffinity(0))``)
        on platforms that expose this information (i.e. Linux+glibc),
        or ``os.cpu_count()`` .

        If GPUs are used, one additional CPU thread per device
        will be launched for communicating with the device. One
        can set ``cputhreads`` to zero to disable CPU evaluation
        in this case.

    .. seealso::
        The most important options are described in
        :numref:`chapter_cpp_qmc`.

    The other options are defined in the Qmc docs. If
    an argument is omitted then the default of the
    underlying Qmc implementation is used.

    '''
    def __init__(self,integral_library,transform='korobov3',fitfunction='default',generatingvectors='default',epsrel=1e-2,epsabs=1e-7,maxeval=4611686018427387903,errormode='default',evaluateminn=0,
                      minn=10000,minm=0,maxnperpackage=0,maxmperpackage=0,cputhreads=None,cudablocks=0,cudathreadsperblock=0,verbosity=0,seed=0,devices=[],lattice_candidates=0,standard_lattices=False,keep_lattices=False):
        devices_t = c_int * len(devices)
        if cputhreads is None:
            try:
                cputhreads = len(os.sched_getaffinity(0))
            except AttributeError:
                cputhreads = os.cpu_count()
        argtypes = [
                        c_double, # epsrel
                        c_double, # epsabs
                        c_ulonglong, # maxeval
                        c_int, # errormode
                        c_ulonglong, # evaluateminn
                        c_ulonglong, # minn
                        c_ulonglong, # minm
                        c_ulonglong, # maxnperpackage
                        c_ulonglong, # maxmperpackage
                        c_longlong, # cputhreads
                        c_ulonglong, # cudablocks
                        c_ulonglong, # cudathreadsperblock
                        c_ulonglong, # verbosity
                        c_longlong, # seed
                        c_int, # transform_id
                        c_int, # fitfunction_id
                        c_int, # generatingvectors_id
                        c_ulonglong, # lattice_candidates
                        c_bool, # standard_lattices
                        c_bool, # keep_lattices
                        c_ulonglong, # number_of_devices
                        devices_t # devices[]
                   ]
        self.c_lib = integral_library.c_lib
        self.c_lib_path = integral_library.c_lib_path
        self.c_lib.allocate_cuda_integrators_Qmc_together.restype = self.c_lib.allocate_cuda_integrators_Qmc_separate.restype = c_void_p
        self.c_lib.allocate_cuda_integrators_Qmc_together.argtypes = self.c_lib.allocate_cuda_integrators_Qmc_separate.argtypes = argtypes

        # assuming:
        # enum ErrorMode : int
        # {
        #     all = 1,
        #     largest = 2
        # };
        if errormode == 'default':
            errormode_enum = 0
        elif errormode == 'all':
            errormode_enum = 1
        elif errormode == 'largest':
            errormode_enum = 2
        else:
            raise ValueError('Unknown `errormode` "' + str(errormode) + '"')

        self.c_integrator_ptr_together = self.c_lib.allocate_cuda_integrators_Qmc_together(
                                                                                               epsrel,epsabs,maxeval,errormode_enum,evaluateminn,minn,
                                                                                               minm,maxnperpackage,maxmperpackage,cputhreads,
                                                                                               cudablocks,cudathreadsperblock,verbosity,
                                                                                               seed,known_qmc_transforms[str(transform).lower()],
                                                                                               known_qmc_fitfunctions[str(fitfunction).lower()],
                                                                                               known_qmc_generatingvectors[str(generatingvectors).lower()],
                                                                                               lattice_candidates,standard_lattices,keep_lattices,
                                                                                               len(devices),devices_t(*devices)
                                                                                          )
        self.c_integrator_ptr_separate = self.c_lib.allocate_cuda_integrators_Qmc_separate(
                                                                                               epsrel,epsabs,maxeval,errormode_enum,evaluateminn,minn,
                                                                                               minm,maxnperpackage,maxmperpackage,cputhreads,
                                                                                               cudablocks,cudathreadsperblock,verbosity,
                                                                                               seed,known_qmc_transforms[str(transform).lower()],
                                                                                               known_qmc_fitfunctions[str(fitfunction).lower()],
                                                                                               known_qmc_generatingvectors[str(generatingvectors).lower()],
                                                                                               lattice_candidates,standard_lattices,keep_lattices,
                                                                                               len(devices),devices_t(*devices)
                                                                                          )
        self._epsrel=epsrel
        self._epsabs=epsabs
        self._mineval=minn
        self._maxeval=maxeval

    def __del__(self):
        if hasattr(self, 'c_integrator_ptr_together'):
            self.c_lib.free_cuda_together_integrator.restype = None
            self.c_lib.free_cuda_together_integrator.argtypes = [c_void_p]
            self.c_lib.free_cuda_together_integrator(self.c_integrator_ptr_together)

        if hasattr(self, 'c_integrator_ptr_separate'):
            self.c_lib.free_cuda_separate_integrator.restype = None
            self.c_lib.free_cuda_separate_integrator.argtypes = [c_void_p]
            self.c_lib.free_cuda_separate_integrator(self.c_integrator_ptr_separate)

class IntegralLibrary(object):
    r'''
    Interface to a c++ library produced by
    :func:`.code_writer.make_package` or :func:`.loop_package`.

    :param shared_object_path:
        str;
        The path to the file "<name>_pylink.so"
        that can be built by the command

        .. code::

            $ make pylink

        in the root directory of the c++ library.

    Instances of this class can be called with the
    following arguments:

    :param real_parameters:
        iterable of float;
        The real_parameters of the library.

    :param complex_parameters:
        iterable of complex;
        The complex parameters of the library.

    :param together:
        bool, optional;
        Whether to integrate the sum of all sectors
        or to integrate the sectors separately.
        Default: ``True``.

    :param number_of_presamples:
        unsigned int, optional;
        The number of samples used for the
        contour optimization.
        A larger value here may resolve a sign
        check error (sign_check_error).
        This option is ignored if the integral
        library was created without deformation.
        Default: ``100000``.

    :param deformation_parameters_maximum:
        float, optional;
        The maximal value the deformation parameters
        :math:`\lambda_i` can obtain.
        Lower this value if you get a sign check
        error (sign_check_error).
        If ``number_of_presamples=0``, all
        :math:`\lambda_i` are set to this value.
        This option is ignored if the integral
        library was created without deformation.
        Default: ``1.0``.

    :param deformation_parameters_minimum:
        float, optional;
        The minimal value the deformation parameters
        :math:`\lambda_i` can obtain.
        Lower this value if you get a sign check
        error (sign_check_error).
        If ``number_of_presamples=0``, all
        :math:`\lambda_i` are set to this value.
        This option is ignored if the integral
        library was created without deformation.
        Default: ``1e-5``.

    :param deformation_parameters_decrease_factor:
        float, optional;
        If the sign check with the optimized
        :math:`\lambda_i` fails during the presampling
        stage, all :math:`\lambda_i` are multiplied
        by this value until the sign check passes.
        We recommend to rather change
        ``number_of_presamples``,
        ``deformation_parameters_maximum``,
        and ``deformation_parameters_minimum``
        in case of a sign check error.
        This option is ignored if the integral
        library was created without deformation.
        Default: ``0.9``.

    :param epsrel:
        float, optional;
        The desired relative accuracy for the numerical
        evaluation of the weighted sum of the sectors.
        Default: epsrel of integrator (default 0.2).

    :param epsabs:
        float, optional;
        The desired absolute accuracy for the numerical
        evaluation of the weighted sum of the sectors.
        Default: epsabs of integrator (default 1e-7).

    :param maxeval:
        unsigned int, optional;
        The maximal number of integrand evaluations for
        each sector.
        Default: maxeval of integrator (default 2**62-1).

    :param mineval:
        unsigned int, optional;
        The minimal number of integrand evaluations for
        each sector.
        Default: mineval/minn of integrator (default 10000).

    :param maxincreasefac:
        float, optional;
        The maximum factor by which the number of
        integrand evaluations will be increased in a
        single refinement
        iteration.
        Default: ``20``.

    :param min_epsrel:
        float, optional;
        The minimum relative accuracy required for
        each individual sector.
        Default: ``0.2``

    :param min_epsabs:
        float, optional;
        The minimum absolute accuracy required for
        each individual sector.
        Default: ``1.e-4``.

    :param max_epsrel:
        float, optional;
        The maximum relative accuracy assumed possible
        for each individual sector. Any sector known to
        this precision will not be refined further.
        Note: if this condition is met this means that
        the expected precision will not match the
        desired precision.
        Default: ``1.e-14``.

    :param max_epsabs:
        float, optional;
        The maximum absolute accuracy assumed possible
        for each individual sector. Any sector known to
        this precision will not be refined further. Note:
        if this condition is met this means that the
        expected precision will not match the desired
        precision.
        Default: ``1.e-20``.

    :param min_decrease_factor:
        float, optional;
        If the next refinement iteration is expected to
        make the total time taken for the code to run
        longer than ``wall_clock_limit`` then the
        number of points to be requested in the next
        iteration will be reduced by
        at least ``min_decrease_factor``.
        Default: ``0.9``.

    :param decrease_to_percentage:
        float, optional;
        If
        ``remaining_time * decrease_to_percentage > time_for_next_iteration``
        then the number of points requested in
        the next refinement iteration will be reduced.
        Here: ``remaining_time = wall_clock_limit - elapsed_time``
        and ``time_for_next_iteration`` is the estimated
        time required for the next refinement iteration.
        Note: if this condition is met this means that
        the expected precision will not match the desired
        precision.
        Default: ``0.7``.

    :param wall_clock_limit:
        float, optional;
        If the current elapsed time has passed
        ``wall_clock`` limit and a refinement
        iteration finishes then a new refinement iteration
        will not be started. Instead, the code will return
        the current result and exit.
        Default: ``DBL_MAX (1.7976931348623158e+308)``.

    :param number_of_threads:
        int, optional;
        The number of threads used to compute integrals
        concurrently. Note: The integrals themselves may
        also be computed with multiple threads
        irrespective of this option.
        Default: ``0``.

    :param reset_cuda_after:
        int, optional;
        The cuda driver does not automatically remove
        unnecessary functions from the device memory
        such that the device may run out of memory after
        some time. This option controls after how many
        integrals ``cudaDeviceReset()`` is called to clear
        the memory. With the default ``0``,
        ``cudaDeviceReset()`` is never called. This option
        is ignored if compiled without cuda.
        Default: ``0 (never)``.

    :param verbose:
        bool, optional;
        Controls the verbosity of the output of the amplitude.
        Default: ``False``.

    :param errormode:
        str, optional;
        Allowed values: ``abs``, ``all``, ``largest``, ``real``, ``imag``.
        Defines how epsrel and epsabs should be applied to complex values.
        With the choice  ``largest``, the relative uncertainty is defined as
        ``max( |Re(error)|, |Im(error)|)/max( |Re(result)|, |Im(result)|)``.
        Choosing ``all`` will apply epsrel and epsabs to both the real
        and imaginary part separately.
        Note: If either the real or imaginary part integrate to 0,
        the choices ``all``, ``real`` or ``imag`` might prevent the integration
        from stopping since the requested precision epsrel cannot be reached.
        Default: ``abs``.

    :param format:
        string;
        The format of the returned result, ``"series"``,
        ``"ginac"``, ``"sympy"``, ``"mathematica"``,
        ``"maple"``, or ``"json"``. Default: ``"series"``.

    .. seealso::
        A more detailed description of these parameters and
        how they affect timing/precision is given in
        :numref:`chapter_secdecutil_amplitude`.

    The call operator returns three strings:
    * The integral without its prefactor
    * The prefactor
    * The integral multiplied by the prefactor

    The integrator can be configured by calling the
    member methods :meth:`.use_Vegas`, :meth:`.use_Suave`,
    :meth:`.use_Divonne`, :meth:`.use_Cuhre`,
    :meth:`.use_CQuad`, and :meth:`.use_Qmc`.
    The available options are listed in the documentation of
    :class:`.Vegas`, :class:`.Suave`, :class:`.Divonne`,
    :class:`.Cuhre`, :class:`.CQuad`, :class:`.Qmc`
    (:class:`.CudaQmc` for GPU version), respectively.
    :class:`CQuad` can only be used for one dimensional
    integrals. A call to :meth:`use_CQuad` configures the
    integrator to use :class:`CQuad` if possible (1D) and the
    previously defined integrator otherwise.
    By default, :class:`CQuad` (1D only) and :class:`Vegas`
    are used with their default arguments.
    For details about the options, refer to the cuba and the
    gsl manual.

    Further information about the library is stored in
    the member variable `info` of type :class:`dict`.

    '''
    def __init__(self, shared_object_path):
        self._cuda = False

        # import c++ library
        c_lib = self.c_lib = CDLL(shared_object_path)
        basename, ext = os.path.splitext(shared_object_path)
        self.c_lib_path = \
                basename[:-7] + "_data" if basename.endswith("_pylink") else \
                basename + "_data"

        # set c prototypes
        c_lib.allocate_string.restype = c_void_p
        c_lib.allocate_string.argtypes = None

        c_lib.free_string.restype = None
        c_lib.free_string.argtypes = [c_void_p]

        c_lib.get_integral_info.restype = c_void_p
        c_lib.get_integral_info.argtypes = [c_void_p]

        c_lib.string2charptr.restype = c_char_p
        c_lib.string2charptr.argtypes = [c_void_p]


        # get integral info
        cpp_str_integral_info = c_lib.allocate_string()
        c_lib.get_integral_info(cpp_str_integral_info)
        str_integral_info = c_lib.string2charptr(cpp_str_integral_info)
        c_lib.free_string(cpp_str_integral_info)
        if not isinstance(str_integral_info, str):
            str_integral_info = str_integral_info.decode('ASCII')


        # store the integral info in a dictionary
        integral_info = self.info = dict()
        for line in str_integral_info.split('\n'):
            key, value = line.split('=')
            integral_info[key.strip()] = value.strip(' ,')


        # continue set c prototypes
        self.real_parameter_t = c_double * int(integral_info['number_of_real_parameters'])
        self.complex_parameter_t = c_double * (2*int(integral_info['number_of_complex_parameters'])) # flattened as: ``real(x0), imag(x0), real(x1), imag(x1), ...``

        c_lib.compute_integral.restype = c_int
        c_lib.compute_integral.argtypes = [
                                               c_void_p, c_void_p, c_void_p, # output strings
                                               c_void_p, # integrator
                                               self.real_parameter_t, # double array
                                               self.complex_parameter_t, # double array as real(x0), imag(x0), real(x1), imag(x1), ...
                                               c_bool, # together
                                               c_uint, # number_of_presamples
                                               c_double, # deformation_parameters_maximum
                                               c_double, # deformation_parameters_minimum
                                               c_double, # deformation_parameters_decrease_factor
                                               c_double, # epsrel
                                               c_double, # epsabs
                                               c_ulonglong, # maxeval
                                               c_ulonglong, # mineval
                                               c_double, # maxincreasefac
                                               c_double, # min_epsrel
                                               c_double, # min_epsabs
                                               c_double, # max_epsrel
                                               c_double, # max_epsabs
                                               c_double, # min_decrease_factor
                                               c_double, # decrease_to_percentage
                                               c_double, # wall_clock_limit
                                               c_size_t, # number_of_threads
                                               c_size_t, # reset_cuda_after
                                               c_bool, # verbose
                                               c_int, # errormode
                                               c_char_p  # lib_path
        ]

        # set cuda integrate types if applicable
        try:
            c_lib.cuda_compute_integral.restype = c_int
            c_lib.cuda_compute_integral.argtypes = [
                                                        c_void_p, c_void_p, c_void_p, # output strings
                                                        c_void_p, # together integrator
                                                        c_void_p, # separate integrator
                                                        self.real_parameter_t, # double array
                                                        self.complex_parameter_t, # double array as real(x0), imag(x0), real(x1), imag(x1), ...
                                                        c_bool, # together
                                                        c_uint, # number_of_presamples
                                                        c_double, # deformation_parameters_maximum
                                                        c_double, # deformation_parameters_minimum
                                                        c_double, # deformation_parameters_decrease_factor
                                                        c_double,  # epsrel
                                                        c_double,  # epsabs
                                                        c_ulonglong,  # maxeval
                                                        c_ulonglong,  # mineval
                                                        c_double,  # maxincreasefac
                                                        c_double,  # min_epsrel
                                                        c_double,  # min_epsabs
                                                        c_double,  # max_epsrel
                                                        c_double,  # max_epsabs
                                                        c_double,  # min_decrease_factor
                                                        c_double,  # decrease_to_percentage
                                                        c_double,  # wall_clock_limit
                                                        c_size_t,  # number_of_threads
                                                        c_size_t,  # reset_cuda_after
                                                        c_bool, # verbose
                                                        c_int, # errormode
                                                        c_char_p  # lib_path
                                                   ]
        except AttributeError:
            # c_lib has been compiled without cuda
            pass

    def __call__(
                     self, real_parameters=[], complex_parameters=[], together=True,
                     number_of_presamples=100000, deformation_parameters_maximum=1.,
                     deformation_parameters_minimum=1.e-5,
                     deformation_parameters_decrease_factor=0.9,
                     epsrel=None, epsabs=None, maxeval=None,
                     mineval=None, maxincreasefac=20., min_epsrel=0.2, min_epsabs=1.e-4,
                     max_epsrel=1.e-14, max_epsabs=1.e-20, min_decrease_factor=0.9,
                     decrease_to_percentage=0.7, wall_clock_limit=1.7976931348623158e+308, # 1.7976931348623158e+308 max double
                     number_of_threads=0, reset_cuda_after=0, verbose=False, errormode='abs', format="series"
                ):
        # Set the default integrator
        if getattr(self, "integrator", None) is None:
            self.use_Qmc()

        # Set default epsrel,epsabs to integrator.epsrel,epsabs
        if (epsrel is None): epsrel = self.high_dimensional_integrator._epsrel
        if (epsabs is None): epsabs = self.high_dimensional_integrator._epsabs
        if (mineval is None): mineval = self.high_dimensional_integrator._mineval
        if (maxeval is None): maxeval = self.high_dimensional_integrator._maxeval

        if errormode == 'abs':
            errormode_enum = 0
        elif errormode == 'all':
            errormode_enum = 1
        elif errormode == 'largest':
            errormode_enum = 2
        elif errormode == 'real':
            errormode_enum = 3
        elif errormode == 'imag':
            errormode_enum = 4
        else:
            raise ValueError('Unknown `errormode` "' + str(errormode) + '"')

        if verbose:
            print(version)


        # Initialize and launch the underlying c routines in a subprocess
        # to enable KeyboardInterrupt and avoid crashing the primary python
        # interpreter on error.
        queue = Queue()
        return_value_queue = Queue()
        integration_thread = Thread(
                                         target=self._call_implementation,
                                         args=(
                                                  queue, return_value_queue, real_parameters,
                                                  complex_parameters, together,
                                                  number_of_presamples,
                                                  deformation_parameters_maximum,
                                                  deformation_parameters_minimum,
                                                  deformation_parameters_decrease_factor,
                                                  epsrel, epsabs, maxeval,
                                                  mineval, maxincreasefac, min_epsrel, min_epsabs,
                                                  max_epsrel, max_epsabs, min_decrease_factor,
                                                  decrease_to_percentage, wall_clock_limit,
                                                  number_of_threads, reset_cuda_after, verbose, errormode_enum
                                              )
                                     )
        integration_thread.daemon = True # daemonize worker to have it killed when the main thread is killed
        integration_thread.start()
        while integration_thread.is_alive(): # keep joining worker until it is finished
            integration_thread.join(5) # call `join` with `timeout` to keep the main thread interruptible
        return_value = return_value_queue.get()
        if return_value != 0:
            raise RuntimeError("Integration failed, see error message above.")
        else:
            str_integral_without_prefactor, str_prefactor, str_integral_with_prefactor = queue.get()
            if format == "ginac":
                str_integral_without_prefactor = series_to_ginac(str_integral_without_prefactor)
                str_integral_with_prefactor = series_to_ginac(str_integral_with_prefactor)
            elif format == "sympy":
                str_integral_without_prefactor = series_to_sympy(str_integral_without_prefactor)
                str_integral_with_prefactor = series_to_sympy(str_integral_with_prefactor)
            elif format == "mathematica":
                str_integral_without_prefactor = series_to_mathematica(str_integral_without_prefactor)
                str_integral_with_prefactor = series_to_mathematica(str_integral_with_prefactor)
            elif format == "maple":
                str_integral_without_prefactor = series_to_maple(str_integral_without_prefactor)
                str_integral_with_prefactor = series_to_maple(str_integral_with_prefactor)
            elif format == "json":
                str_integral_without_prefactor = series_to_json(str_integral_without_prefactor, self.info['name'])
                str_integral_with_prefactor = series_to_json(str_integral_with_prefactor, self.info['name'])
            return (str_integral_without_prefactor, str_prefactor, str_integral_with_prefactor)

    def _call_implementation(
                                self, queue, return_value_queue, real_parameters, complex_parameters, together,
                                number_of_presamples, deformation_parameters_maximum,
                                deformation_parameters_minimum,
                                deformation_parameters_decrease_factor,
                                epsrel, epsabs, maxeval,
                                mineval, maxincreasefac, min_epsrel, min_epsabs,
                                max_epsrel, max_epsabs, min_decrease_factor,
                                decrease_to_percentage, wall_clock_limit,
                                number_of_threads, reset_cuda_after, verbose, errormode_enum
                            ):
        # Passed in correct number of parameters?
        assert len(real_parameters) == int(self.info['number_of_real_parameters']), \
            'Passed %i `real_parameters` but %s needs %i.' % (len(real_parameters),self.info['name'],int(self.info['number_of_real_parameters']))
        assert len(complex_parameters) == int(self.info['number_of_complex_parameters']), \
            'Passed %i `complex_parameters` but `%s` needs %i.' % (len(complex_parameters),self.info['name'],int(self.info['number_of_complex_parameters']))

        # set parameter values
        #   - real parameters
        c_real_parameters = self.real_parameter_t(*real_parameters)

        #   - complex parameters
        flattened_complex_parameters = []
        for c in complex_parameters:
            flattened_complex_parameters.append(c.real)
            flattened_complex_parameters.append(c.imag)
        c_complex_parameters = self.complex_parameter_t(*flattened_complex_parameters)

        # allocate c++ strings
        cpp_str_integral_without_prefactor = self.c_lib.allocate_string()
        cpp_str_prefactor = self.c_lib.allocate_string()
        cpp_str_integral_with_prefactor = self.c_lib.allocate_string()

        # call the underlying c routine
        if self._cuda:
            compute_integral_return_value = self.c_lib.cuda_compute_integral(
                                                 cpp_str_integral_without_prefactor,
                                                 cpp_str_prefactor, cpp_str_integral_with_prefactor,
                                                 self.integrator.c_integrator_ptr_together,
                                                 self.integrator.c_integrator_ptr_separate, c_real_parameters,
                                                 c_complex_parameters, together,
                                                 number_of_presamples, deformation_parameters_maximum,
                                                 deformation_parameters_minimum,
                                                 deformation_parameters_decrease_factor,
                                                 epsrel, epsabs, maxeval,
                                                 mineval, maxincreasefac, min_epsrel, min_epsabs,
                                                 max_epsrel, max_epsabs, min_decrease_factor,
                                                 decrease_to_percentage, wall_clock_limit,
                                                 number_of_threads, reset_cuda_after, verbose,errormode_enum,
                                                 self.c_lib_path.encode("utf-8")
                                            )
        else:
            compute_integral_return_value = self.c_lib.compute_integral(
                                            cpp_str_integral_without_prefactor,
                                            cpp_str_prefactor, cpp_str_integral_with_prefactor,
                                            self.integrator.c_integrator_ptr, c_real_parameters,
                                            c_complex_parameters, together,
                                            number_of_presamples, deformation_parameters_maximum,
                                            deformation_parameters_minimum,
                                            deformation_parameters_decrease_factor,
                                            epsrel, epsabs, maxeval,
                                            mineval, maxincreasefac, min_epsrel, min_epsabs,
                                            max_epsrel, max_epsabs, min_decrease_factor,
                                            decrease_to_percentage, wall_clock_limit,
                                            number_of_threads, reset_cuda_after, verbose,errormode_enum,
                                            self.c_lib_path.encode("utf-8")
                                    )
        return_value_queue.put(compute_integral_return_value)
        if compute_integral_return_value != 0:
            return

        # convert c++ strings to python strings or bytes (depending on whether we use python2 or python3)
        str_integral_without_prefactor = self.c_lib.string2charptr(cpp_str_integral_without_prefactor)
        str_prefactor = self.c_lib.string2charptr(cpp_str_prefactor)
        str_integral_with_prefactor = self.c_lib.string2charptr(cpp_str_integral_with_prefactor)

        # free allocated c++ strings
        self.c_lib.free_string(cpp_str_integral_without_prefactor)
        self.c_lib.free_string(cpp_str_prefactor)
        self.c_lib.free_string(cpp_str_integral_with_prefactor)

        # python 2/3 compatibility: make sure the strings read from c++ have type "str" with ASCII encoding
        if not isinstance(str_integral_without_prefactor, str):
            str_integral_without_prefactor = str_integral_without_prefactor.decode('ASCII')
        if not isinstance(str_prefactor, str):
            str_prefactor = str_prefactor.decode('ASCII')
        if not isinstance(str_integral_with_prefactor, str):
            str_integral_with_prefactor = str_integral_with_prefactor.decode('ASCII')

        queue.put( (str_integral_without_prefactor, str_prefactor, str_integral_with_prefactor) )

    def use_Vegas(self, *args, **kwargs):
        self._cuda = False
        self.high_dimensional_integrator = self.integrator = Vegas(self,*args,**kwargs)

    def use_Suave(self, *args, **kwargs):
        self._cuda = False
        self.high_dimensional_integrator = self.integrator = Suave(self,*args,**kwargs)

    def use_Divonne(self, *args, **kwargs):
        self._cuda = False
        self.high_dimensional_integrator = self.integrator = Divonne(self,*args,**kwargs)

    def use_Cuhre(self, *args, **kwargs):
        self.high_dimensional_integrator = self.integrator = Cuhre(self,*args,**kwargs)

    def use_CQuad(self, *args, **kwargs):
        if self._cuda:
            raise RuntimeError('Cannot use `CQuad` together with `CudaQmc`.')
        self.cquad = CQuad(self, *args, **kwargs)
        # If no high_dimensional integrator set, use vegas
        if getattr(self, "high_dimensional_integrator", None) is None:
            self.use_Vegas()
        self.integrator = MultiIntegrator(self,self.cquad,self.high_dimensional_integrator,2)

    def use_Qmc(self, *args, **kwargs):
        if hasattr(self.c_lib,'allocate_integrators_Qmc'):
            self._cuda = False
            self.high_dimensional_integrator = self.integrator = Qmc(self,*args,**kwargs)
        else:
            self._cuda = True
            self.high_dimensional_integrator = self.integrator = CudaQmc(self,*args,**kwargs)

def _runlength_encode(values):
    result = []
    prev_val = None
    repeat = 1
    for val in values:
        if val == prev_val:
            repeat += 1
        else:
            if prev_val is not None:
                result.append((repeat, prev_val))
            prev_val = val
            repeat = 1
    if prev_val is not None:
        result.append((repeat, prev_val))
    return result

class DevNullWriter:
    def write(self, s): pass
    def flush(self): pass

class DistevalLibrary(object):
    r'''
    Interface to the integration library produced by
    :func:`.code_writer.make_package` or :func:`.loop_package` and built by
    ``make disteval``.

    :param specification_path:
        str;
        The path to the file ``disteval/<name>.json`` that can
        be built by the command

        .. code::

            $ make disteval

        in the root directory of the integration library.

    :param workers:
        list of string or list of list of string, optional;
        List of commands that start pySecDec workers.
        Default: one ``"nice python3 -m pySecDecContrib pysecdec_cpuworker"``
        per available CPU, or one
        ``"nice python3 -m pySecDecContrib pysecdec_cudaworker -d <i>"``
        for each available GPU.

    :param verbose:
        bool, optional;
        Print the set up and the integration log.
        Default: ``True``.

    Instances of this class can be called with the
    following arguments:

    :param parameters:
        dict of str to object, optional;
        A map from parameter names to their values.
        A value can be any object that can be converted to
        complex() and to str(). To make sure floating point
        imprecision does not spoil the results, it is best to
        pass the values of the parameters as integers, strings,
        sympy expressions, or fractions.Fraction() objects.
        Floating point numbers are supported, but not encouraged.

    :param real_parameters:
        iterable of float, optional;
        The values of the real parameters of the library in
        the same order as the real_parameters argument of
        :func:`.code_writer.make_package`.
        (Not needed if ``parameters`` are provided).

    :param complex_parameters:
        iterable of complex, optional;
        The values of the complex parameters of the library in
        the same order as the complex_parameters argument of
        :func:`.code_writer.make_package`.
        (Not needed if ``parameters`` are provided).

    :param number_of_presamples:
        unsigned int, optional;
        The number of samples used for the
        contour optimization.
        This option is ignored if the integral
        library was created without deformation.
        Default: ``10000``.

    :param epsrel:
        float, optional;
        The desired relative accuracy for the numerical
        evaluation of the weighted sum of the sectors.
        Default: ``1e-4``.

    :param epsabs:
        float, optional;
        The desired absolute accuracy for the numerical
        evaluation of the weighted sum of the sectors.
        Default: epsabs of integrator (default 1e-10).

    :param timeout:
        float, optional;
        The maximal integration time (in seconds) after which
        the integration will stop, and the current result will
        be returned (no matter which precision is reached).
        Default: ``None``.

    :param points:
        unsigned int, optional;
        The initial QMC lattice size.
        Default: ``1e4``.

    :param shifts:
        unsigned int, optional;
        The number of shifts of the QMC lattice.
        Default: ``32``.

    :param lattice_candidates:
        unsigned int, optional;
        The number of generating vector candidates used for median QMC rule.
        lattice_candidates=0 disables the use of the median QMC rule.
        Default: ``0``.

    :param verbose:
        bool, optional;
        Print the integration log.
        Default: whatever was used in the __init__() call.

    :param coefficients:
        string, optional;
        Alternative path to the directory with the files containing
        the coefficients for the evaluated amplitude.
        Default: the ``coefficients/`` directory next to the
        specification file.

    :param format:
        string;
        The format of the returned result, ``"sympy"``,
        ``"mathematica"``, or ``"json"``. Default: ``"sympy"``.

    The call operator returns a single string with the resulting
    value as a series in the regulator powers.
    '''

    def __init__(self, specification_path, workers=None, verbose=True):
        import asyncio
        import sys
        from . import disteval
        # Normally this code is not supposed to be called from
        # within an asyncio event loop, but e.g. Jupyter notebook
        # kernel insists on starting it automatically, breaking
        # any internal asyncio usage. See [1,2,3].
        #
        # The only solution at the moment is to patch asyncio to
        # allow nested event loop execution via nest_asyncio.
        #
        # [1] https://github.com/jupyter/notebook/issues/3397
        # [2] https://github.com/ipython/ipykernel/issues/548
        # [3] https://github.com/python/cpython/issues/66435
        import nest_asyncio
        nest_asyncio.apply()
        disteval.log_file = sys.stderr if verbose else DevNullWriter()
        disteval.log(version)
        dirname = os.path.dirname(specification_path)
        if workers is None:
            workers = disteval.default_worker_commands(dirname)
        self.filename = specification_path
        self.dirname = dirname
        self.verbose = verbose
        self.prepared = asyncio.run(disteval.prepare_eval(workers, dirname, specification_path))

    def __call__(self,
            parameters={}, real_parameters=[], complex_parameters=[],
            epsabs=1e-10, epsrel=1e-4, timeout=None, points=1e4,
            number_of_presamples=1e4, shifts=32,
            lattice_candidates=0, standard_lattices=False,
            coefficients=None, verbose=None, format="sympy"):
        import asyncio
        import json
        import math
        import sys
        import time
        from . import disteval

        if real_parameters or complex_parameters:
            with open(self.filename) as f:
                spec = json.load(f)
            realp = spec["realp"]
            for i, val in enumerate(real_parameters):
                parameters[realp[i]] = val
            complexp = spec["complexp"]
            for i, val in enumerate(complex_parameters):
                parameters[complexp[i]] = val
        valuemap_int = {}
        valuemap_coeff = {}
        for key, value in parameters.items():
            fvalue = complex(value)
            if fvalue.imag == 0:
                valuemap_int[key] = fvalue.real
                valuemap_coeff[key] = str(value.real) if isinstance(value, complex) else str(value)
            else:
                valuemap_int[key] = fvalue
                valuemap_coeff[key] = f"{str(value.real)} + ({str(value.imag)})*I" if isinstance(value, complex) else str(value)
        if not isinstance(epsabs, list) and not isinstance(epsabs, tuple):
            epsabs = [epsabs]
        if not isinstance(epsrel, list) and not isinstance(epsrel, tuple):
            epsrel = [epsrel]
        if coefficients is None:
            coefficients = os.path.join(self.dirname, "coefficients")
        deadline = math.inf if timeout is None else time.time() + timeout
        if verbose is None: verbose = self.verbose
        disteval.log_file = sys.stderr if verbose else DevNullWriter()
        result = asyncio.run(disteval.do_eval(
            self.prepared, coefficients, epsabs, epsrel,
            int(number_of_presamples), int(points), int(shifts),
            lattice_candidates, standard_lattices,
            valuemap_int, valuemap_coeff, deadline))

        if format == "raw":
            return result
        elif format == "sympy":
            return disteval.result_to_sympy(result)
        elif format == "mathematica":
            return disteval.result_to_mathematica(result)
        else:
            return disteval.result_to_json(result)

    def close(self):
        """Shutdown the workers and release all resources."""
        import asyncio
        from . import disteval
        asyncio.run(disteval.clear_eval(self.prepared))
