"""Routines to Feynman parametrize a loop integral"""

from ..algebra import Polynomial, ExponentiatedPolynomial, Product
from ..misc import cached_property, sympify_symbols, assert_degree_at_most_max_degree, sympify_expression
import sympy as sp
import numpy as np
from math import floor

class LoopIntegral(object):
    r'''
    Container class for loop integrals.
    The main purpose of this class is to convert a
    loop integral from the momentum representation
    to the Feynman parameter representation.

    It is possible to provide either the graph
    of the loop integrals as adjacency list,
    or the propagators.

    The Feynman parametrized integral is a product
    of the following expressions that are accessible
    as member properties:

    * ``self.Gamma_factor``
    * ``self.exponentiated_U``
    * ``self.exponentiated_F``
    * ``self.numerator``
    * ``self.measure``,

    where ``self`` is an instance of either
    :class:`.LoopIntegralFromGraph` or
    :class:`.LoopIntegralFromPropagators`.

    When inverse propagators or nonnumerical propagator
    powers are present (see `powerlist`), some
    `Feynman_parameters` drop out of the integral. The
    variables to integrate over can be accessed as
    ``self.integration_variables``.

    While ``self.numerator`` describes the numerator
    polynomial generated by tensor numerators or inverse
    propagators, ``self.measure`` contains the monomial
    associated with the integration measure in the case
    of propagator powers :math:`\neq 1`. The Gamma functions
    in the denominator belonging to the measure, however,
    are multiplied to the overall Gamma factor given by
    ``self.Gamma_factor``.

    .. versionchanged:: 1.2.2
        The overall sign :math:`(-1)^{N_\nu}` is included
        in ``self.Gamma_factor``.

    .. seealso::
        * input as graph: :class:`.LoopIntegralFromGraph`
        * input as list of propagators: :class:`.LoopIntegralFromPropagators`

    '''
    def __init__(self, *args, **kwargs):
        raise AttributeError('Use one of the derived classes.')

    @cached_property
    def exponent_U(self):
        return self.N_nu - self.U_derivatives - self.dimensionality / 2 * (self.L + 1)\
            - self.highest_rank

    @property # not cached on purpose --> this is just making copies
    def exponentiated_U(self):
        return ExponentiatedPolynomial(self.U.expolist, self.U.coeffs, self.exponent_U, self.U.polysymbols)

    @cached_property
    def exponent_F(self):
        return self.dimensionality / 2 * self.L - (self.N_nu + self.F_derivatives)

    @property # not cached on purpose --> this is just making copies
    def exponentiated_F(self):
        return ExponentiatedPolynomial(self.F.expolist, self.F.coeffs, self.exponent_F, self.F.polysymbols)

    common_properties_doc = r'''
    :param replacement_rules:
        iterable of iterables with two strings or sympy
        expressions, optional;
        Symbolic replacements to be made for the external
        momenta, e.g. definition of Mandelstam variables.
        Example: ``[('p1*p2', 's'), ('p1**2', 0)]`` where
        ``p1`` and ``p2`` are external momenta.
        It is also possible to specify vector replacements,
        for example ``[('p4', '-(p1+p2+p3)')]``.

    :param Feynman_parameters:
        iterable or string, optional;
        The symbols to be used for the Feynman parameters.
        If a string is passed, the Feynman parameter
        variables will be consecutively numbered starting
        from zero.

    :param regulators:
        list of strings or sympy symbol, optional;
        The symbols to be used for the regulators
        (typically :math:`\epsilon` or :math:`\epsilon_D`)

        .. note::
            If you change the dimensional regulator symbol,
            you have to adapt the `dimensionality` accordingly.

    :param regulator:
        a string or a sympy symbol, optional;
        Deprecated; same as setting `regulators` to a list
        of a single element.

    :param dimensionality:
        string or sympy expression, optional;
        The dimensionality; typically :math:`4-2\epsilon`,
        which is the default value.

    :param powerlist:
        iterable, optional;
        The powers of the propergators, possibly dependent on
        the `regulators`. In case of negative powers, the
        `numerator` is constructed by taking derivatives with
        respect to the corresponding Feynman parameters as
        explained in Section 3.2.4 of Ref. [BHJ+15]_.
        If negative powers are combined with a tensor numerator,
        the derivatives act on the Feynman-parametrized tensor
        numerator as well, which leads to a consistent result.

    '''

    def set_common_properties(self, replacement_rules, Feynman_parameters, regulators,
                              dimensionality, powerlist):
        # sympify and store `regulators`
        self.regulators = sympify_symbols(regulators, '`regulators` must be symbols.')

        # sympify and store `dimensionality`
        self.dimensionality = sympify_expression(dimensionality)

        # check and store replacement rules
        if not isinstance(replacement_rules, list):
            replacement_rules = list(replacement_rules)
        if replacement_rules:
            replacement_rules = np.array(replacement_rules)
            assert len(replacement_rules.shape) == 2, "The `replacement_rules` should be a list of pairs."
            assert replacement_rules.shape[1] == 2 , "The `replacement_rules` should be a list of pairs."
            for rule in replacement_rules:
                for expression in rule:
                    assert_degree_at_most_max_degree(expression, self.all_momenta, 2, 'Each of the `replacement_rules` must be polynomial and at most quadratic in the momenta.')
            self.replacement_rules = [(sympify_expression(rule[0]),sympify_expression(rule[1])) for rule in replacement_rules]
        else:
            self.replacement_rules = []

        # sympify and store `Feynman_parameters`
        # There should be one Feynman parameter for each propagator.
        if isinstance(Feynman_parameters, str):
            self.Feynman_parameters = [sympify_expression(Feynman_parameters + str(i)) for i in range(self.P)]
        else:
            self.Feynman_parameters = sympify_expression(list(Feynman_parameters))
            assert len(self.Feynman_parameters) == self.P, \
                'Mismatch between the number of `propagators` (%i) and the number of `Feynman_parameters` (%i).' % \
                ( len(self.propagators) , len(self.Feynman_parameters) )

        # check and store `powerlist`
        # If there are negative powers, determine the number of derivatives neseccary to make them positive
        # and store them in derivativelist.
        if not powerlist:
            self.powerlist = [sympify_expression(1)] * self.P
            self.derivativelist = [0] * self.P
            self.number_of_derivatives = 0
        else:
            if not isinstance(powerlist,list):
                powerlist = list(powerlist)
            assert len(powerlist)==self.P, "The length of the powerlist must equal the number of propagators."

            self.powerlist=[]
            self.derivativelist=[]

            for power in powerlist:
                power_sp = sympify_expression(power)
                power0 = power_sp
                for regulator in regulators:
                    power0 = power0.subs(regulator,0)
                assert power0.is_Number, "The propagator powers must be numbers for vanishing regulators."
                self.powerlist.append(power_sp)
                if power0 >= 1:
                    self.derivativelist.append(0)
                elif power0.is_Integer:
                    self.derivativelist.append(abs(power0))
                else:
                    self.derivativelist.append(int(abs(floor(power0))) + 1)

            self.number_of_derivatives = sum(self.derivativelist)

        self.U_derivatives = self.number_of_derivatives
        self.F_derivatives = self.number_of_derivatives

        self.N_nu = sum(self.powerlist)

    @cached_property
    def U(self):
        # returns U with all Feynman parameters of inverse propagators set to zero
        U = self.preliminary_U
        for i in range(self.P-1,-1,-1):
            if self.powerlist[i].is_integer and self.powerlist[i].is_nonpositive:
                U = U.replace(i,0,remove=True).simplify()
        return U

    @cached_property
    def F(self):
        # returns F with all Feynman parameters of inverse propagators set to zero
        F = self.preliminary_F
        for i in range(self.P-1,-1,-1):
            if self.powerlist[i].is_integer and self.powerlist[i].is_nonpositive:
                F = F.replace(i,0,remove=True).simplify()
        return F

    @cached_property
    def numerator(self):

        Feynman_parameters_U_F = self.Feynman_parameters + sympify_expression(['U', 'F'])

        extended_expolist = []
        for exponents in self.preliminary_U.expolist:
            extended_expolist.append(np.concatenate([exponents,[0,0]]))

        U_explicit = Polynomial(extended_expolist, self.preliminary_U.coeffs, polysymbols=Feynman_parameters_U_F)

        extended_expolist = []
        for exponents in self.preliminary_F.expolist:
            extended_expolist.append(np.concatenate([exponents,[0,0]]))

        F_explicit = Polynomial(extended_expolist, self.preliminary_F.coeffs, polysymbols=Feynman_parameters_U_F)

        Nu = self.preliminary_numerator
        U = Polynomial.from_expression('U', Feynman_parameters_U_F)
        F = Polynomial.from_expression('F', Feynman_parameters_U_F)

        n = self.N_nu - self.dimensionality / 2 * (self.L + 1) - self.highest_rank
        m = self.N_nu - self.dimensionality / 2 * self.L

        # Loop backwards over Feynman parameters so that removing one parameter does not change the indexing
        # of the ones still to come.
        for i in range(self.P-1,-1,-1):

            # calculate k-fold derivative of U^n/F^m*Nu with respect to Feynman_parameters[i]
            # keeping F and U symbolic but calculating their derivatives explicitly
            # In each step factor out U^(n-1)/F^(m+1) or U^n/F^(m+1).
            k = self.derivativelist[i]

            if k != 0:
                dFdx = F_explicit.derive(i)
                dUdx = U_explicit.derive(i)
                for _ in range(k):
                    # The derivative of U^n/F^m*Nu is given by the sum of three terms:
                    # term1 = n * U^(n-1)/F^m * dU/dx * Nu
                    # term2 = -m * U^n/F^(m+1) * dF/dx * Nu
                    # term3 = U^n/F^m * ( dNu/dx + dNu/dF * dF/dx + dNu/dU * dU/dx )

                    # terms with all factors of U and F stripped off:
                    term1 = n*dUdx*Nu
                    term2 = - m*dFdx*Nu
                    term3 = Nu.derive(i) + Nu.derive(-1)*dFdx + Nu.derive(-2)*dUdx

                    # If term1 vanishes, we can factor out U^n.
                    if dUdx.coeffs.any() != 0:
                        # -> term1 non-zero -> factor out U^(n-1) rather than U^n
                        n -= 1
                        term2 *= U
                        term3 *= U

                    # dFdx will in practice never be the zero polynomial
                    # -> term2 always present -> factor out F^-(m+1) rather than F^-m
                    m += 1
                    term1 *= F
                    term3 *= F

                    Nu = term1 + term2 + term3

            # The k-fold derivative effectively increments the power of the propagator by k.
            # If the new 'effective power' is exactly zero, the corresponding parameter has to be set to zero.
            # Note that in the case where the power is zero from the start, this applies as well.
            newpower = self.powerlist[i] + k
            if newpower == 0:
                F_explicit = F_explicit.replace(i,0,remove=True).simplify()
                U_explicit = U_explicit.replace(i,0,remove=True).simplify()
                F = F.replace(i,0,remove=True).simplify()
                U = U.replace(i,0,remove=True).simplify()
                Nu = Nu.replace(i,0,remove=True).simplify()

        return Nu

    @cached_property
    def measure(self):
        # The monomials x_i^(nu_i-1) multiplying the integration measure.
        # The factors of 1/Gamma(nu_i) are implemented in `Gamma_factor` together with the global Gamma.
        measure_factors = []

        Feynman_parameters_U_F = self.Feynman_parameters + sympify_expression(['U', 'F'])

        # The effective power to be used in the measure has to be increased by the number of derivatives.
        for i in range(self.P):
            eff_power = self.powerlist[i] + self.derivativelist[i]
            if eff_power not in (0,1):
                expolist = np.zeros([1,len(Feynman_parameters_U_F)], dtype=int)
                expolist[0][i] = 1
                measure_factors.append(
                                          ExponentiatedPolynomial(
                                                                     expolist, np.array([1]), exponent=eff_power-1,
                                                                     polysymbols=Feynman_parameters_U_F, copy=False
                                                                 )
                                      )

        if measure_factors:
            measure = Product(*measure_factors, copy=False)
        else:
            measure = Product(
                              ExponentiatedPolynomial(
                                                         np.zeros([1,len(Feynman_parameters_U_F)], dtype=int),
                                                         np.array([1]), exponent=eff_power-1,
                                                         polysymbols=Feynman_parameters_U_F, copy=False
                                                     )
                             )

        for i in range(self.P-1,-1,-1):
            eff_power = self.powerlist[i] + self.derivativelist[i]
            if eff_power.is_integer and eff_power.is_zero:
                measure = measure.replace(i,0,remove=True)

        return measure

    @cached_property
    def Gamma_argument(self):
        return self.N_nu - self.dimensionality * self.L/2 - self.highest_rank//2

    @cached_property
    def Gamma_factor(self):
        # Every term factor in the sum of equation (2.5) in arXiv:1010.1667v1 comes with
        # the scalar factor `1/(-2)**(r/2)*Gamma(N_nu - dim*L/2 - r/2)*F**(r/2)`.
        # In order to keep the `numerator` free of poles in the dimensional regulator, we divide it
        # by the Gamma function with the smallest argument `N_nu - dim*L/2 - highest_rank//2`,
        # where `//` means integer division, and put it here.
        gamma_fac = sp.gamma(self.Gamma_argument)

        # Multiply by the 1/Gamma(nu_i) factors belonging to the integration measure.
        # The effective power to be used in the gamma functions has to be increased by the number of derivatives.
        for i in range(self.P):
            eff_power = self.powerlist[i] + self.derivativelist[i]
            if eff_power != 0 :
                gamma_fac *= 1/sp.gamma(eff_power)

        # Multiply the global factor of `(-1)**N_nu` in equation (2.15) of arXiv:1010.1667v1.
        # Since we need the (nonstandard) negative prescription, we must rewrite:
        #   ``(-1-I*delta)**N_nu = exp(Log(-1-I*delta) * N_nu) = exp(-I*pi * N_nu)``
        gamma_fac *= sp.exp( - sp.I * sp.pi * (self.N_nu + self.number_of_derivatives) )

        return gamma_fac

    @cached_property
    def integration_variables(self):
        variables = []
        for FP,power in zip(self.Feynman_parameters, self.powerlist):
            if not (power.is_integer and power.is_nonpositive):
                variables.append(FP)
        return variables
