"""
peakrdl-python is a tool to generate Python Register Access Layer (RAL) from SystemRDL
Copyright (C) 2021 - 2025

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <https://www.gnu.org/licenses/>.

The methods for building the main iterator that is used in the templates for building the
code.
"""
from typing import Optional
from dataclasses import dataclass, field
from collections.abc import Iterator
from itertools import filterfalse

from logging import getLogger

from systemrdl.node import Node
from systemrdl.node import RegNode
from systemrdl.node import FieldNode
from systemrdl.node import MemNode
from systemrdl.node import AddrmapNode
from systemrdl.node import RegfileNode
from systemrdl.node import SignalNode
from systemrdl import RDLListener, WalkerAction
from systemrdl.rdltypes.user_enum import UserEnumMeta

from .systemrdl_node_hashes import node_hash, enum_hash, NodeHashingMethod
from .systemrdl_node_utility_functions import HideNodeCallback
from .systemrdl_node_utility_functions import ShowUDPCallback
from .systemrdl_node_utility_functions import get_properties_to_include
from .systemrdl_node_utility_functions import get_reg_regwidth, get_reg_accesswidth
from .systemrdl_node_utility_functions import get_memory_accesswidth
from .class_names import get_base_class_name

@dataclass(frozen=True)
class PeakRDLPythonUniqueComponents:
    """
    Dataclass to hold a node that needs to be made into a python class
    """
    instance: Node
    instance_hash: int
    parent_walker: 'UniqueComponents'
    python_class_name: str = field(init=False)
    optimal_python_class_name: bool =  field(init=False)

    def __post_init__(self) -> None:
        if not isinstance(self.instance, Node):
            raise TypeError(f'instance must be a node got {type(self.instance)}')
        python_class_name, optimal_python_class_name = self.__determine_python_class_name()
        object.__setattr__(self, 'python_class_name', python_class_name)
        object.__setattr__(self, 'optimal_python_class_name', optimal_python_class_name)


    def base_class(self, async_library_classes: bool) -> str:
        """
        Return the python base class name from the library to use with the component generated
        in the register model
        """
        return get_base_class_name(node=self.instance,
                                   async_library_classes=async_library_classes)

    @property
    def properties_to_include(self) -> list[str]:
        """
        Provide a list of the User Defined Properties to include in the Register Model for a given
        Node
        """
        return get_properties_to_include(
            node=self.instance,
            udp_to_include_callback=self.parent_walker.udp_include_func)

    def __determine_python_class_name(self) -> tuple[str, bool]:
        """
        Returns the fully qualified class type name with a pre-calculated hash to save time
        """
        scope_path = self.instance.inst.get_scope_path(scope_separator='_')

        # the node.inst.type_name may include a suffix for the reset value, peak_rdl python passes
        # the reset value in when the component is initialised so this is not needed. Therefore,
        # the orginal_def version plus the peakrdl_python hash needs to be used
        if self.instance.inst.original_def is None:
            inst_type_name = self.instance.inst_name
            ideal_class_name = False
        else:
            inbound_inst_type_name = self.instance.inst.original_def.type_name
            ideal_class_name = True
            if inbound_inst_type_name is None:
                inst_type_name = self.instance.inst_name
                ideal_class_name = False
            else:
                inst_type_name = inbound_inst_type_name

        if self.instance_hash < 0:
            if (scope_path == '') or (scope_path is None):
                return (inst_type_name + '_neg_' + hex(-self.instance_hash) + '_cls',
                        ideal_class_name)

            return (scope_path + '_' + inst_type_name + '_neg_' + hex(-self.instance_hash) + '_cls',
                    ideal_class_name)

        if (scope_path == '') or (scope_path is None):
            return (inst_type_name + '_' + hex(self.instance_hash) + '_cls',
                    ideal_class_name)

        return (scope_path + '_' + inst_type_name + '_' + hex(self.instance_hash) + '_cls',
                ideal_class_name)

    def children(self, unroll:bool, exclude_signals:bool=True) -> Iterator[Node]:
        """
        Iterator for all the systemRDL nodes which are not hidden
        """
        if exclude_signals:
            child_nodes = filter(lambda node : not isinstance(node, SignalNode),
                                 self.instance.children(unroll=unroll))
            yield from filterfalse(self.parent_walker.hide_node_callback,
                                   child_nodes)
        else:
            yield from filterfalse(self.parent_walker.hide_node_callback,
                                   self.instance.children(unroll=unroll))

    @property
    def zero_children(self) -> int:
        """
        This condition can happen if the children are hidden
        """
        return len(tuple(self.children(unroll=False))) == 0


@dataclass(frozen=True)
class PeakRDLPythonUniqueRegisterComponents(PeakRDLPythonUniqueComponents):
    """
    Dataclass to hold a register node that needs to be made into a python class
    """
    instance: RegNode

    def __post_init__(self) -> None:
        super().__post_init__()
        if not isinstance(self.instance, RegNode):
            raise TypeError(f'instance must be a RegNode got {type(self.instance)}')

    @property
    def read_write(self) -> bool:
        """
        Determine if the register is read-write
        """
        return self.instance.has_sw_readable and self.instance.has_sw_writable

    @property
    def read_only(self) -> bool:
        """
        Determine if the register is read-only
        """
        return self.instance.has_sw_readable and not self.instance.has_sw_writable

    @property
    def write_only(self) -> bool:
        """
        Determine if the register is write-only
        """
        return not self.instance.has_sw_readable and self.instance.has_sw_writable

    @property
    def regwidth(self) -> int:
        """
        register width in bits
        """
        return get_reg_regwidth(self.instance)

    @property
    def accesswidth(self) -> int:
        """
        register access width in bits
        """
        return get_reg_accesswidth(self.instance)

    def lookup_field_data_python_class(self, field_node: FieldNode) -> str:
        """
        Helper function to lookup the class name of a field node
        """

        if not isinstance(field_node, FieldNode):
            raise TypeError(f'Only fields should use this method, got {type(field_node)}')

        field_hash = self.parent_walker.calculate_or_lookup_hash(node=field_node)
        if field_hash is None:
            return 'int'

        field_component = self.parent_walker.nodes[field_hash]

        if not isinstance(field_component, (PeakRDLPythonUniqueFieldComponents,
                                            PeakRDLPythonUniqueEnumFieldComponents)):
            raise TypeError(f'Unexpected Component Type:{type(field_component)}')

        return field_component.data_type_python_class_name

    def fields(self) -> Iterator[FieldNode]:
        """
        Iterator for all the systemRDL nodes which are not hidden
        """
        yield from filterfalse(self.parent_walker.hide_node_callback,
                               self.instance.fields())

@dataclass(frozen=True)
class PeakRDLPythonUniqueMemoryComponents(PeakRDLPythonUniqueComponents):
    """
    Dataclass to hold a register node that needs to be made into a python class
    """
    instance: MemNode

    def __post_init__(self) -> None:
        super().__post_init__()
        if not isinstance(self.instance, MemNode):
            raise TypeError(f'instance must be a MemNode got {type(self.instance)}')

    @property
    def accesswidth(self) -> int:
        """
        memory access width in bits
        """
        return get_memory_accesswidth(self.instance)

    def registers(self, unroll:bool) -> Iterator[RegNode]:
        """
        Iterator for all the systemRDL nodes which are not hidden
        """
        for child in self.children(unroll=unroll, exclude_signals=True):
            if not isinstance(child, RegNode):
                raise TypeError(f'child must be a RegNode got {type(child)}')
            yield child

@dataclass(frozen=True)
class PeakRDLPythonUniqueFieldComponents(PeakRDLPythonUniqueComponents):
    """
    Dataclass to hold a register node that needs to be made into a python class
    """
    instance: FieldNode
    data_type_python_class_name : str = field(init=False)

    def __post_init__(self) -> None:
        super().__post_init__()
        if not isinstance(self.instance, FieldNode):
            raise TypeError(f'instance must be a FieldNode got {type(self.instance)}')
        object.__setattr__(self, 'data_type_python_class_name', 'int')

@dataclass(frozen=True)
class PeakRDLPythonUniqueFieldEnum:
    """
    Dataclass to hold a field encoding definition that needs to be made into a python class
    """
    instance: UserEnumMeta
    instance_hash: int
    parent_walker: 'UniqueComponents'
    python_class_name: str = field(init=False)

    def __post_init__(self) -> None:
        if not isinstance(self.instance, UserEnumMeta):
            raise TypeError(f'instance must be a UserEnumMeta got {type(self.instance)}')
        class_name = self.__fully_qualified_enum_type()
        object.__setattr__(self, 'python_class_name', class_name)

    def __fully_qualified_enum_type(self) -> str:
        """
        Returns the fully qualified class type name, for an enum
        """
        enum_hash_value = self.parent_walker.enum_hash(self.instance)
        full_scope_path = self.instance.get_scope_path('_')
        if enum_hash_value < 0:
            return full_scope_path + '_' + self.instance.type_name + '_neg_' + hex(-enum_hash_value)
        return full_scope_path + '_' + self.instance.type_name + hex(enum_hash_value) + '_enumcls'

@dataclass(frozen=True)
class PeakRDLPythonUniqueEnumFieldComponents(PeakRDLPythonUniqueFieldComponents):
    """
    Dataclass to hold a register node that needs to be made into a python class
    """
    instance: FieldNode
    encoding: PeakRDLPythonUniqueFieldEnum

    def __post_init__(self) -> None:
        super().__post_init__()
        if not isinstance(self.instance, FieldNode):
            raise TypeError(f'instance must be a FieldNode got {type(self.instance)}')
        data_type_class = self.encoding.python_class_name
        object.__setattr__(self, 'data_type_python_class_name', data_type_class)

class UniqueComponents(RDLListener):
    """
    class intended to be used as part of the walker/listener protocol to find all the items
    non-hidden nodes
    """
    # pylint:disable=too-many-instance-attributes

    def __init__(self,
                 hide_node_callback: HideNodeCallback,
                 include_name_and_desc: bool,
                 udp_include_func: ShowUDPCallback,
                 hashing_method: NodeHashingMethod = NodeHashingMethod.SHA256) -> None:
        super().__init__()

        self.__hide_node_callback = hide_node_callback
        self.__udp_include_func = udp_include_func
        self.__include_name_and_desc = include_name_and_desc
        self.__hashing_method = hashing_method
        self.nodes: dict[int, PeakRDLPythonUniqueComponents] = {}
        self.field_enum: dict[int, PeakRDLPythonUniqueFieldEnum] = {}
        self.__name_hash_cache: dict[str, Optional[int]] = {}
        self.__logger = getLogger('peakrdl_python.UniqueComponents')

    @property
    def hide_node_callback(self) -> HideNodeCallback:
        """
        Callback to determine if a node is hidden or not
        """
        return self.__hide_node_callback

    @property
    def udp_include_func(self) -> ShowUDPCallback:
        """
        Callback to determine if a UDP shown
        """
        return self.__udp_include_func

    @property
    def include_name_and_desc(self) -> bool:
        """
        Whether to consider the systemRDL name and desc when determine the uniqueness of the
        components
        """
        return self.__include_name_and_desc

    @property
    def hashing_method(self) -> NodeHashingMethod:
        """
        Hashing method to use
        """
        return self.__hashing_method

    def __test_and_add(self, potential_unique_node:PeakRDLPythonUniqueComponents) -> bool:
        """
        Tests whether a unique component is in the set of nodes to generate already and add it
        if it new.

        Args:
            potential_unique_node: A potential component to add

        Returns: True if the component has been added, False if it has not been added (which
                 allows descendants to be skipped

        """
        self.__logger.debug(f'Node under test hash:{potential_unique_node.instance_hash}')

        if potential_unique_node.instance_hash in self.nodes:
            # The node is already in the node set, however, if the new node has a better
            # python class name use the new one
            if self.nodes[potential_unique_node.instance_hash].optimal_python_class_name is False:
                if potential_unique_node.optimal_python_class_name is True:
                    self.nodes[potential_unique_node.instance_hash] = potential_unique_node

            return True

        self.nodes[potential_unique_node.instance_hash] = potential_unique_node
        return False

    def __build_peak_rdl_unique_component(self, node: Node) -> \
            Optional[PeakRDLPythonUniqueComponents]:

        nodal_hash_result = self.calculate_or_lookup_hash(node)

        if nodal_hash_result is None:
            return None

        if isinstance(node, RegNode):
            return PeakRDLPythonUniqueRegisterComponents(instance=node,
                                                         instance_hash=nodal_hash_result,
                                                         parent_walker=self)
        if isinstance(node, MemNode):
            return PeakRDLPythonUniqueMemoryComponents(instance=node,
                                                       instance_hash=nodal_hash_result,
                                                       parent_walker=self)

        if isinstance(node, FieldNode):
            # depending on whether the field is encoded or not the component generated changes
            # and may need to have a field encoding entry added to the table
            encoding = node.get_property('encode', default=None)
            if encoding is None:
                return PeakRDLPythonUniqueFieldComponents(instance=node,
                                                              instance_hash=nodal_hash_result,
                                                              parent_walker=self)

            encoding_hash = enum_hash(enum=encoding,
                                      method=self.hashing_method,
                                      include_name_and_desc=self.include_name_and_desc)

            if encoding_hash not in self.field_enum:
                self.field_enum[encoding_hash] = PeakRDLPythonUniqueFieldEnum(
                    instance=encoding,
                    instance_hash=encoding_hash,
                    parent_walker=self)

            return PeakRDLPythonUniqueEnumFieldComponents(instance=node,
                                                          instance_hash=nodal_hash_result,
                                                          encoding=self.field_enum[encoding_hash],
                                                          parent_walker=self)

        return PeakRDLPythonUniqueComponents(instance=node,
                                             instance_hash=nodal_hash_result,
                                             parent_walker=self)

    def __enter_non_field_node(self, node: Node) -> Optional[WalkerAction]:
        """
        Handler for all node types other than Field
        """
        full_node_name = '.'.join(node.get_path_segments())
        self.__logger.debug(f'Analysing node:{full_node_name}')

        if self.__hide_node_callback(node):
            return WalkerAction.SkipDescendants

        potential_unique_node = self.__build_peak_rdl_unique_component(node)

        if potential_unique_node is None:
            raise RuntimeError('This node type should not have a hash of None')

        if self.__test_and_add(potential_unique_node):
            return WalkerAction.SkipDescendants

        return WalkerAction.Continue

    def enter_Reg(self, node: RegNode) -> Optional[WalkerAction]:
        return self.__enter_non_field_node(node)

    def enter_Mem(self, node: MemNode) -> Optional[WalkerAction]:
        return self.__enter_non_field_node(node)

    def enter_Field(self, node: FieldNode) -> Optional[WalkerAction]:

        full_node_name = '.'.join(node.get_path_segments())
        self.__logger.debug(f'Analysing node:{full_node_name}')

        if self.__hide_node_callback(node):
            return WalkerAction.SkipDescendants

        potential_unique_node = self.__build_peak_rdl_unique_component(node)

        # The Filed Hash can be None if a base class from the library can be used directly
        if potential_unique_node is None:
            return WalkerAction.SkipDescendants

        if self.__test_and_add(potential_unique_node):
            return WalkerAction.SkipDescendants

        return WalkerAction.Continue

    def enter_Addrmap(self, node: AddrmapNode) -> Optional[WalkerAction]:
        return self.__enter_non_field_node(node)

    def enter_Regfile(self, node: RegfileNode) -> Optional[WalkerAction]:
        return self.__enter_non_field_node(node)

    def python_class_name(self, node: Node, async_library_classes: bool) -> str:
        """
        Lookup the python class name to be used for a given node

        Args:
            node: node
            async_library_classes: whether base classes returned are async or not

        Returns: classname as a string

        """
        nodal_hash_result = self.calculate_or_lookup_hash(node)

        if nodal_hash_result is None:
            # This is special case where the field has no attributes that need a field definition
            # to be created so it is not included in the list of things to construct, therefore the
            # base classes are directly used
            if not isinstance(node, FieldNode):
                raise TypeError(f'This code should occur for a FieldNode, got {type(node)}')
            return get_base_class_name(node,
                                       async_library_classes=async_library_classes)

        if nodal_hash_result not in self.nodes:
            raise RuntimeError(f'The node hash for {node.inst_name} is not in the table')
        python_class_name = self.nodes[nodal_hash_result].python_class_name
        return python_class_name

    def calculate_or_lookup_hash(self, node: Node) -> Optional[int]:
        """
        Calculates the hash for a node with an option to retrieve it from the cache to avoid it
        being redone if possible
        """

        full_instance_name = '.'.join(node.get_path_segments())
        if full_instance_name in self.__name_hash_cache:
            return self.__name_hash_cache[full_instance_name]

        nodal_hash_result = node_hash(node=node, udp_include_func=self.udp_include_func,
                                      hide_node_callback=self.hide_node_callback,
                                      include_name_and_desc=self.include_name_and_desc,
                                      method=self.hashing_method)
        self.__name_hash_cache[full_instance_name] = nodal_hash_result
        return nodal_hash_result

    def register_nodes(self) -> Iterator[PeakRDLPythonUniqueRegisterComponents]:
        """
        Iterator though all the unique register nodes
        """
        yield from filter(
            lambda component: isinstance(component, PeakRDLPythonUniqueRegisterComponents),
            self.nodes.values())

    def memory_nodes(self) -> Iterator[PeakRDLPythonUniqueMemoryComponents]:
        """
        Iterator though all the unique memory nodes
        """
        yield from filter(
            lambda component: isinstance(component, PeakRDLPythonUniqueMemoryComponents),
            self.nodes.values())

    def enum_hash(self, enum: UserEnumMeta) -> int:
        """
        Calculate the hash for a system RDL field encoding
        """
        return enum_hash(enum=enum,
                         include_name_and_desc=self.include_name_and_desc,
                         method=self.hashing_method)
