"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const grant_1 = require("./grant");
const iam_generated_1 = require("./iam.generated");
const policy_1 = require("./policy");
const policy_document_1 = require("./policy-document");
const policy_statement_1 = require("./policy-statement");
const principals_1 = require("./principals");
const util_1 = require("./util");
/**
 * IAM Role
 *
 * Defines an IAM role. The role is created with an assume policy document associated with
 * the specified AWS service principal defined in `serviceAssumeRole`.
 */
class Role extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.roleName,
        });
        this.grantPrincipal = this;
        this.assumeRoleAction = 'sts:AssumeRole';
        this.managedPolicies = [];
        this.attachedPolicies = new util_1.AttachedPolicies();
        const externalIds = props.externalIds || [];
        if (props.externalId) {
            externalIds.push(props.externalId);
        }
        this.assumeRolePolicy = createAssumeRolePolicy(props.assumedBy, externalIds);
        this.managedPolicies.push(...props.managedPolicies || []);
        this.permissionsBoundary = props.permissionsBoundary;
        const maxSessionDuration = props.maxSessionDuration && props.maxSessionDuration.toSeconds();
        validateMaxSessionDuration(maxSessionDuration);
        const role = new iam_generated_1.CfnRole(this, 'Resource', {
            assumeRolePolicyDocument: this.assumeRolePolicy,
            managedPolicyArns: core_1.Lazy.listValue({ produce: () => this.managedPolicies.map(p => p.managedPolicyArn) }, { omitEmpty: true }),
            policies: _flatten(props.inlinePolicies),
            path: props.path,
            permissionsBoundary: this.permissionsBoundary ? this.permissionsBoundary.managedPolicyArn : undefined,
            roleName: this.physicalName,
            maxSessionDuration,
        });
        this.roleId = role.attrRoleId;
        this.roleArn = this.getResourceArnAttribute(role.attrArn, {
            region: '',
            service: 'iam',
            resource: 'role',
            resourceName: this.physicalName,
        });
        this.roleName = this.getResourceNameAttribute(role.ref);
        this.policyFragment = new principals_1.ArnPrincipal(this.roleArn).policyFragment;
        function _flatten(policies) {
            if (policies == null || Object.keys(policies).length === 0) {
                return undefined;
            }
            const result = new Array();
            for (const policyName of Object.keys(policies)) {
                const policyDocument = policies[policyName];
                result.push({ policyName, policyDocument });
            }
            return result;
        }
    }
    /**
     * Imports an external role by ARN.
     *
     * @param scope construct scope
     * @param id construct id
     * @param roleArn the ARN of the role to import
     * @param options allow customizing the behavior of the returned role
     */
    static fromRoleArn(scope, id, roleArn, options = {}) {
        const scopeStack = core_1.Stack.of(scope);
        const parsedArn = scopeStack.parseArn(roleArn);
        const roleName = parsedArn.resourceName;
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.grantPrincipal = this;
                this.assumeRoleAction = 'sts:AssumeRole';
                this.policyFragment = new principals_1.ArnPrincipal(roleArn).policyFragment;
                this.roleArn = roleArn;
                this.roleName = roleName;
            }
            addManagedPolicy(_policy) {
                // FIXME: Add warning that we're ignoring this
            }
            /**
             * Grant permissions to the given principal to pass this role.
             */
            grantPassRole(identity) {
                return this.grant(identity, 'iam:PassRole');
            }
            /**
             * Grant the actions defined in actions to the identity Principal on this resource.
             */
            grant(grantee, ...actions) {
                return grant_1.Grant.addToPrincipal({
                    grantee,
                    actions,
                    resourceArns: [this.roleArn],
                    scope: this,
                });
            }
        }
        const roleAccount = parsedArn.account;
        class MutableImport extends Import {
            constructor() {
                super(...arguments);
                this.attachedPolicies = new util_1.AttachedPolicies();
            }
            addToPolicy(statement) {
                if (!this.defaultPolicy) {
                    this.defaultPolicy = new policy_1.Policy(this, 'Policy');
                    this.attachInlinePolicy(this.defaultPolicy);
                }
                this.defaultPolicy.addStatements(statement);
                return true;
            }
            attachInlinePolicy(policy) {
                const policyAccount = core_1.Stack.of(policy).account;
                if (accountsAreEqualOrOneIsUnresolved(policyAccount, roleAccount)) {
                    this.attachedPolicies.attach(policy);
                    policy.attachToRole(this);
                }
            }
        }
        class ImmutableImport extends Import {
            addToPolicy(_statement) {
                return true;
            }
            attachInlinePolicy(_policy) {
                // do nothing
            }
        }
        const scopeAccount = scopeStack.account;
        return options.mutable !== false && accountsAreEqualOrOneIsUnresolved(scopeAccount, roleAccount)
            ? new MutableImport(scope, id)
            : new ImmutableImport(scope, id);
        function accountsAreEqualOrOneIsUnresolved(account1, account2) {
            return core_1.Token.isUnresolved(account1) || core_1.Token.isUnresolved(account2) ||
                account1 === account2;
        }
    }
    /**
     * Adds a permission to the role's default policy document.
     * If there is no default policy attached to this role, it will be created.
     * @param statement The permission statement to add to the policy document
     */
    addToPolicy(statement) {
        if (!this.defaultPolicy) {
            this.defaultPolicy = new policy_1.Policy(this, 'DefaultPolicy');
            this.attachInlinePolicy(this.defaultPolicy);
        }
        this.defaultPolicy.addStatements(statement);
        return true;
    }
    /**
     * Attaches a managed policy to this role.
     * @param policy The the managed policy to attach.
     */
    addManagedPolicy(policy) {
        if (this.managedPolicies.find(mp => mp === policy)) {
            return;
        }
        this.managedPolicies.push(policy);
    }
    /**
     * Attaches a policy to this role.
     * @param policy The policy to attach
     */
    attachInlinePolicy(policy) {
        this.attachedPolicies.attach(policy);
        policy.attachToRole(this);
    }
    /**
     * Grant the actions defined in actions to the identity Principal on this resource.
     */
    grant(grantee, ...actions) {
        return grant_1.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [this.roleArn],
            scope: this
        });
    }
    /**
     * Grant permissions to the given principal to pass this role.
     */
    grantPassRole(identity) {
        return this.grant(identity, 'iam:PassRole');
    }
}
exports.Role = Role;
function createAssumeRolePolicy(principal, externalIds) {
    const statement = new policy_statement_1.PolicyStatement();
    statement.addPrincipals(principal);
    statement.addActions(principal.assumeRoleAction);
    if (externalIds.length) {
        statement.addCondition('StringEquals', { 'sts:ExternalId': externalIds.length === 1 ? externalIds[0] : externalIds });
    }
    const doc = new policy_document_1.PolicyDocument();
    doc.addStatements(statement);
    return doc;
}
function validateMaxSessionDuration(duration) {
    if (duration === undefined) {
        return;
    }
    if (duration < 3600 || duration > 43200) {
        throw new Error(`maxSessionDuration is set to ${duration}, but must be >= 3600sec (1hr) and <= 43200sec (12hrs)`);
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicm9sZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInJvbGUudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQSx3Q0FBa0Y7QUFDbEYsbUNBQWdDO0FBQ2hDLG1EQUEwQztBQUcxQyxxQ0FBa0M7QUFDbEMsdURBQW1EO0FBQ25ELHlEQUFxRDtBQUNyRCw2Q0FBaUY7QUFDakYsaUNBQTBDO0FBa0kxQzs7Ozs7R0FLRztBQUNILE1BQWEsSUFBSyxTQUFRLGVBQVE7SUEwSWhDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBZ0I7UUFDeEQsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDZixZQUFZLEVBQUUsS0FBSyxDQUFDLFFBQVE7U0FDN0IsQ0FBQyxDQUFDO1FBNUNXLG1CQUFjLEdBQWUsSUFBSSxDQUFDO1FBRWxDLHFCQUFnQixHQUFXLGdCQUFnQixDQUFDO1FBb0MzQyxvQkFBZSxHQUFxQixFQUFFLENBQUM7UUFDdkMscUJBQWdCLEdBQUcsSUFBSSx1QkFBZ0IsRUFBRSxDQUFDO1FBT3pELE1BQU0sV0FBVyxHQUFHLEtBQUssQ0FBQyxXQUFXLElBQUksRUFBRSxDQUFDO1FBQzVDLElBQUksS0FBSyxDQUFDLFVBQVUsRUFBRTtZQUNwQixXQUFXLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQztTQUNwQztRQUVELElBQUksQ0FBQyxnQkFBZ0IsR0FBRyxzQkFBc0IsQ0FBQyxLQUFLLENBQUMsU0FBUyxFQUFFLFdBQVcsQ0FBQyxDQUFDO1FBQzdFLElBQUksQ0FBQyxlQUFlLENBQUMsSUFBSSxDQUFDLEdBQUcsS0FBSyxDQUFDLGVBQWUsSUFBSSxFQUFFLENBQUMsQ0FBQztRQUMxRCxJQUFJLENBQUMsbUJBQW1CLEdBQUcsS0FBSyxDQUFDLG1CQUFtQixDQUFDO1FBQ3JELE1BQU0sa0JBQWtCLEdBQUcsS0FBSyxDQUFDLGtCQUFrQixJQUFJLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxTQUFTLEVBQUUsQ0FBQztRQUM1RiwwQkFBMEIsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO1FBRS9DLE1BQU0sSUFBSSxHQUFHLElBQUksdUJBQU8sQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQ3pDLHdCQUF3QixFQUFFLElBQUksQ0FBQyxnQkFBdUI7WUFDdEQsaUJBQWlCLEVBQUUsV0FBSSxDQUFDLFNBQVMsQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsZUFBZSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxnQkFBZ0IsQ0FBQyxFQUFFLEVBQUUsRUFBRSxTQUFTLEVBQUUsSUFBSSxFQUFFLENBQUM7WUFDNUgsUUFBUSxFQUFFLFFBQVEsQ0FBQyxLQUFLLENBQUMsY0FBYyxDQUFDO1lBQ3hDLElBQUksRUFBRSxLQUFLLENBQUMsSUFBSTtZQUNoQixtQkFBbUIsRUFBRSxJQUFJLENBQUMsbUJBQW1CLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDLENBQUMsU0FBUztZQUNyRyxRQUFRLEVBQUUsSUFBSSxDQUFDLFlBQVk7WUFDM0Isa0JBQWtCO1NBQ25CLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxNQUFNLEdBQUcsSUFBSSxDQUFDLFVBQVUsQ0FBQztRQUM5QixJQUFJLENBQUMsT0FBTyxHQUFHLElBQUksQ0FBQyx1QkFBdUIsQ0FBQyxJQUFJLENBQUMsT0FBTyxFQUFFO1lBQ3hELE1BQU0sRUFBRSxFQUFFO1lBQ1YsT0FBTyxFQUFFLEtBQUs7WUFDZCxRQUFRLEVBQUUsTUFBTTtZQUNoQixZQUFZLEVBQUUsSUFBSSxDQUFDLFlBQVk7U0FDaEMsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFFBQVEsR0FBRyxJQUFJLENBQUMsd0JBQXdCLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ3hELElBQUksQ0FBQyxjQUFjLEdBQUcsSUFBSSx5QkFBWSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQyxjQUFjLENBQUM7UUFFcEUsU0FBUyxRQUFRLENBQUMsUUFBNkM7WUFDN0QsSUFBSSxRQUFRLElBQUksSUFBSSxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtnQkFDMUQsT0FBTyxTQUFTLENBQUM7YUFDbEI7WUFDRCxNQUFNLE1BQU0sR0FBRyxJQUFJLEtBQUssRUFBMEIsQ0FBQztZQUNuRCxLQUFLLE1BQU0sVUFBVSxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLEVBQUU7Z0JBQzlDLE1BQU0sY0FBYyxHQUFHLFFBQVEsQ0FBQyxVQUFVLENBQUMsQ0FBQztnQkFDNUMsTUFBTSxDQUFDLElBQUksQ0FBQyxFQUFFLFVBQVUsRUFBRSxjQUFjLEVBQUUsQ0FBQyxDQUFDO2FBQzdDO1lBQ0QsT0FBTyxNQUFNLENBQUM7UUFDaEIsQ0FBQztJQUNILENBQUM7SUF4TEQ7Ozs7Ozs7T0FPRztJQUNJLE1BQU0sQ0FBQyxXQUFXLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsT0FBZSxFQUFFLFVBQThCLEVBQUU7UUFDdkcsTUFBTSxVQUFVLEdBQUcsWUFBSyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUNuQyxNQUFNLFNBQVMsR0FBRyxVQUFVLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBQy9DLE1BQU0sUUFBUSxHQUFHLFNBQVMsQ0FBQyxZQUFhLENBQUM7UUFFekMsTUFBZSxNQUFPLFNBQVEsZUFBUTtZQUF0Qzs7Z0JBQ2tCLG1CQUFjLEdBQWUsSUFBSSxDQUFDO2dCQUNsQyxxQkFBZ0IsR0FBVyxnQkFBZ0IsQ0FBQztnQkFDNUMsbUJBQWMsR0FBRyxJQUFJLHlCQUFZLENBQUMsT0FBTyxDQUFDLENBQUMsY0FBYyxDQUFDO2dCQUMxRCxZQUFPLEdBQUcsT0FBTyxDQUFDO2dCQUNsQixhQUFRLEdBQUcsUUFBUSxDQUFDO1lBNEJ0QyxDQUFDO1lBdEJRLGdCQUFnQixDQUFDLE9BQXVCO2dCQUM3Qyw4Q0FBOEM7WUFDaEQsQ0FBQztZQUVEOztlQUVHO1lBQ0ksYUFBYSxDQUFDLFFBQW9CO2dCQUN2QyxPQUFPLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxFQUFFLGNBQWMsQ0FBQyxDQUFDO1lBQzlDLENBQUM7WUFFRDs7ZUFFRztZQUNJLEtBQUssQ0FBQyxPQUFtQixFQUFFLEdBQUcsT0FBaUI7Z0JBQ3BELE9BQU8sYUFBSyxDQUFDLGNBQWMsQ0FBQztvQkFDMUIsT0FBTztvQkFDUCxPQUFPO29CQUNQLFlBQVksRUFBRSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUM7b0JBQzVCLEtBQUssRUFBRSxJQUFJO2lCQUNaLENBQUMsQ0FBQztZQUNMLENBQUM7U0FDRjtRQUVELE1BQU0sV0FBVyxHQUFHLFNBQVMsQ0FBQyxPQUFPLENBQUM7UUFFdEMsTUFBTSxhQUFjLFNBQVEsTUFBTTtZQUFsQzs7Z0JBQ21CLHFCQUFnQixHQUFHLElBQUksdUJBQWdCLEVBQUUsQ0FBQztZQW9CN0QsQ0FBQztZQWpCUSxXQUFXLENBQUMsU0FBMEI7Z0JBQzNDLElBQUksQ0FBQyxJQUFJLENBQUMsYUFBYSxFQUFFO29CQUN2QixJQUFJLENBQUMsYUFBYSxHQUFHLElBQUksZUFBTSxDQUFDLElBQUksRUFBRSxRQUFRLENBQUMsQ0FBQztvQkFDaEQsSUFBSSxDQUFDLGtCQUFrQixDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQztpQkFDN0M7Z0JBQ0QsSUFBSSxDQUFDLGFBQWEsQ0FBQyxhQUFhLENBQUMsU0FBUyxDQUFDLENBQUM7Z0JBQzVDLE9BQU8sSUFBSSxDQUFDO1lBQ2QsQ0FBQztZQUVNLGtCQUFrQixDQUFDLE1BQWM7Z0JBQ3RDLE1BQU0sYUFBYSxHQUFHLFlBQUssQ0FBQyxFQUFFLENBQUMsTUFBTSxDQUFDLENBQUMsT0FBTyxDQUFDO2dCQUUvQyxJQUFJLGlDQUFpQyxDQUFDLGFBQWEsRUFBRSxXQUFXLENBQUMsRUFBRTtvQkFDakUsSUFBSSxDQUFDLGdCQUFnQixDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQztvQkFDckMsTUFBTSxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsQ0FBQztpQkFDM0I7WUFDSCxDQUFDO1NBQ0Y7UUFFRCxNQUFNLGVBQWdCLFNBQVEsTUFBTTtZQUMzQixXQUFXLENBQUMsVUFBMkI7Z0JBQzVDLE9BQU8sSUFBSSxDQUFDO1lBQ2QsQ0FBQztZQUVNLGtCQUFrQixDQUFDLE9BQWU7Z0JBQ3ZDLGFBQWE7WUFDZixDQUFDO1NBQ0Y7UUFFRCxNQUFNLFlBQVksR0FBRyxVQUFVLENBQUMsT0FBTyxDQUFDO1FBRXhDLE9BQU8sT0FBTyxDQUFDLE9BQU8sS0FBSyxLQUFLLElBQUksaUNBQWlDLENBQUMsWUFBWSxFQUFFLFdBQVcsQ0FBQztZQUM5RixDQUFDLENBQUMsSUFBSSxhQUFhLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQztZQUM5QixDQUFDLENBQUMsSUFBSSxlQUFlLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRW5DLFNBQVMsaUNBQWlDLENBQUMsUUFBNEIsRUFDNUIsUUFBNEI7WUFDckUsT0FBTyxZQUFLLENBQUMsWUFBWSxDQUFDLFFBQVEsQ0FBQyxJQUFJLFlBQUssQ0FBQyxZQUFZLENBQUMsUUFBUSxDQUFDO2dCQUNqRSxRQUFRLEtBQUssUUFBUSxDQUFDO1FBQzFCLENBQUM7SUFDSCxDQUFDO0lBNEZEOzs7O09BSUc7SUFDSSxXQUFXLENBQUMsU0FBMEI7UUFDM0MsSUFBSSxDQUFDLElBQUksQ0FBQyxhQUFhLEVBQUU7WUFDdkIsSUFBSSxDQUFDLGFBQWEsR0FBRyxJQUFJLGVBQU0sQ0FBQyxJQUFJLEVBQUUsZUFBZSxDQUFDLENBQUM7WUFDdkQsSUFBSSxDQUFDLGtCQUFrQixDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQztTQUM3QztRQUNELElBQUksQ0FBQyxhQUFhLENBQUMsYUFBYSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBQzVDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7T0FHRztJQUNJLGdCQUFnQixDQUFDLE1BQXNCO1FBQzVDLElBQUksSUFBSSxDQUFDLGVBQWUsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxFQUFFLEtBQUssTUFBTSxDQUFDLEVBQUU7WUFBRSxPQUFPO1NBQUU7UUFDL0QsSUFBSSxDQUFDLGVBQWUsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7SUFDcEMsQ0FBQztJQUVEOzs7T0FHRztJQUNJLGtCQUFrQixDQUFDLE1BQWM7UUFDdEMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUNyQyxNQUFNLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQzVCLENBQUM7SUFFRDs7T0FFRztJQUNJLEtBQUssQ0FBQyxPQUFtQixFQUFFLEdBQUcsT0FBaUI7UUFDcEQsT0FBTyxhQUFLLENBQUMsY0FBYyxDQUFDO1lBQzFCLE9BQU87WUFDUCxPQUFPO1lBQ1AsWUFBWSxFQUFFLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQztZQUM1QixLQUFLLEVBQUUsSUFBSTtTQUNaLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7T0FFRztJQUNJLGFBQWEsQ0FBQyxRQUFvQjtRQUN2QyxPQUFPLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxFQUFFLGNBQWMsQ0FBQyxDQUFDO0lBQzlDLENBQUM7Q0FDRjtBQTdPRCxvQkE2T0M7QUErQkQsU0FBUyxzQkFBc0IsQ0FBQyxTQUFxQixFQUFFLFdBQXFCO0lBQzFFLE1BQU0sU0FBUyxHQUFHLElBQUksa0NBQWUsRUFBRSxDQUFDO0lBQ3hDLFNBQVMsQ0FBQyxhQUFhLENBQUMsU0FBUyxDQUFDLENBQUM7SUFDbkMsU0FBUyxDQUFDLFVBQVUsQ0FBQyxTQUFTLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztJQUVqRCxJQUFJLFdBQVcsQ0FBQyxNQUFNLEVBQUU7UUFDdEIsU0FBUyxDQUFDLFlBQVksQ0FBQyxjQUFjLEVBQUUsRUFBRSxnQkFBZ0IsRUFBRSxXQUFXLENBQUMsTUFBTSxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxXQUFXLEVBQUUsQ0FBQyxDQUFDO0tBQ3ZIO0lBRUQsTUFBTSxHQUFHLEdBQUcsSUFBSSxnQ0FBYyxFQUFFLENBQUM7SUFDakMsR0FBRyxDQUFDLGFBQWEsQ0FBQyxTQUFTLENBQUMsQ0FBQztJQUM3QixPQUFPLEdBQUcsQ0FBQztBQUNiLENBQUM7QUFFRCxTQUFTLDBCQUEwQixDQUFDLFFBQWlCO0lBQ25ELElBQUksUUFBUSxLQUFLLFNBQVMsRUFBRTtRQUMxQixPQUFPO0tBQ1I7SUFFRCxJQUFJLFFBQVEsR0FBRyxJQUFJLElBQUksUUFBUSxHQUFHLEtBQUssRUFBRTtRQUN2QyxNQUFNLElBQUksS0FBSyxDQUFDLGdDQUFnQyxRQUFRLHdEQUF3RCxDQUFDLENBQUM7S0FDbkg7QUFDSCxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQ29uc3RydWN0LCBEdXJhdGlvbiwgTGF6eSwgUmVzb3VyY2UsIFN0YWNrLCBUb2tlbiB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgR3JhbnQgfSBmcm9tICcuL2dyYW50JztcbmltcG9ydCB7IENmblJvbGUgfSBmcm9tICcuL2lhbS5nZW5lcmF0ZWQnO1xuaW1wb3J0IHsgSUlkZW50aXR5IH0gZnJvbSAnLi9pZGVudGl0eS1iYXNlJztcbmltcG9ydCB7IElNYW5hZ2VkUG9saWN5IH0gZnJvbSAnLi9tYW5hZ2VkLXBvbGljeSc7XG5pbXBvcnQgeyBQb2xpY3kgfSBmcm9tICcuL3BvbGljeSc7XG5pbXBvcnQgeyBQb2xpY3lEb2N1bWVudCB9IGZyb20gJy4vcG9saWN5LWRvY3VtZW50JztcbmltcG9ydCB7IFBvbGljeVN0YXRlbWVudCB9IGZyb20gJy4vcG9saWN5LXN0YXRlbWVudCc7XG5pbXBvcnQgeyBBcm5QcmluY2lwYWwsIElQcmluY2lwYWwsIFByaW5jaXBhbFBvbGljeUZyYWdtZW50IH0gZnJvbSAnLi9wcmluY2lwYWxzJztcbmltcG9ydCB7IEF0dGFjaGVkUG9saWNpZXMgfSBmcm9tICcuL3V0aWwnO1xuXG5leHBvcnQgaW50ZXJmYWNlIFJvbGVQcm9wcyB7XG4gIC8qKlxuICAgKiBUaGUgSUFNIHByaW5jaXBhbCAoaS5lLiBgbmV3IFNlcnZpY2VQcmluY2lwYWwoJ3Nucy5hbWF6b25hd3MuY29tJylgKVxuICAgKiB3aGljaCBjYW4gYXNzdW1lIHRoaXMgcm9sZS5cbiAgICpcbiAgICogWW91IGNhbiBsYXRlciBtb2RpZnkgdGhlIGFzc3VtZSByb2xlIHBvbGljeSBkb2N1bWVudCBieSBhY2Nlc3NpbmcgaXQgdmlhXG4gICAqIHRoZSBgYXNzdW1lUm9sZVBvbGljeWAgcHJvcGVydHkuXG4gICAqL1xuICByZWFkb25seSBhc3N1bWVkQnk6IElQcmluY2lwYWw7XG5cbiAgLyoqXG4gICAqIElEIHRoYXQgdGhlIHJvbGUgYXNzdW1lciBuZWVkcyB0byBwcm92aWRlIHdoZW4gYXNzdW1pbmcgdGhpcyByb2xlXG4gICAqXG4gICAqIElmIHRoZSBjb25maWd1cmVkIGFuZCBwcm92aWRlZCBleHRlcm5hbCBJRHMgZG8gbm90IG1hdGNoLCB0aGVcbiAgICogQXNzdW1lUm9sZSBvcGVyYXRpb24gd2lsbCBmYWlsLlxuICAgKlxuICAgKiBAZGVwcmVjYXRlZCBzZWUge0BsaW5rIGV4dGVybmFsSWRzfVxuICAgKlxuICAgKiBAZGVmYXVsdCBObyBleHRlcm5hbCBJRCByZXF1aXJlZFxuICAgKi9cbiAgcmVhZG9ubHkgZXh0ZXJuYWxJZD86IHN0cmluZztcblxuICAvKipcbiAgICogTGlzdCBvZiBJRHMgdGhhdCB0aGUgcm9sZSBhc3N1bWVyIG5lZWRzIHRvIHByb3ZpZGUgb25lIG9mIHdoZW4gYXNzdW1pbmcgdGhpcyByb2xlXG4gICAqXG4gICAqIElmIHRoZSBjb25maWd1cmVkIGFuZCBwcm92aWRlZCBleHRlcm5hbCBJRHMgZG8gbm90IG1hdGNoLCB0aGVcbiAgICogQXNzdW1lUm9sZSBvcGVyYXRpb24gd2lsbCBmYWlsLlxuICAgKlxuICAgKiBAZGVmYXVsdCBObyBleHRlcm5hbCBJRCByZXF1aXJlZFxuICAgKi9cbiAgcmVhZG9ubHkgZXh0ZXJuYWxJZHM/OiBzdHJpbmdbXTtcblxuICAvKipcbiAgICogQSBsaXN0IG9mIG1hbmFnZWQgcG9saWNpZXMgYXNzb2NpYXRlZCB3aXRoIHRoaXMgcm9sZS5cbiAgICpcbiAgICogWW91IGNhbiBhZGQgbWFuYWdlZCBwb2xpY2llcyBsYXRlciB1c2luZ1xuICAgKiBgYWRkTWFuYWdlZFBvbGljeShNYW5hZ2VkUG9saWN5LmZyb21Bd3NNYW5hZ2VkUG9saWN5TmFtZShwb2xpY3lOYW1lKSlgLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIG1hbmFnZWQgcG9saWNpZXMuXG4gICAqL1xuICByZWFkb25seSBtYW5hZ2VkUG9saWNpZXM/OiBJTWFuYWdlZFBvbGljeVtdO1xuXG4gIC8qKlxuICAgKiBBIGxpc3Qgb2YgbmFtZWQgcG9saWNpZXMgdG8gaW5saW5lIGludG8gdGhpcyByb2xlLiBUaGVzZSBwb2xpY2llcyB3aWxsIGJlXG4gICAqIGNyZWF0ZWQgd2l0aCB0aGUgcm9sZSwgd2hlcmVhcyB0aG9zZSBhZGRlZCBieSBgYGFkZFRvUG9saWN5YGAgYXJlIGFkZGVkXG4gICAqIHVzaW5nIGEgc2VwYXJhdGUgQ2xvdWRGb3JtYXRpb24gcmVzb3VyY2UgKGFsbG93aW5nIGEgd2F5IGFyb3VuZCBjaXJjdWxhclxuICAgKiBkZXBlbmRlbmNpZXMgdGhhdCBjb3VsZCBvdGhlcndpc2UgYmUgaW50cm9kdWNlZCkuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gcG9saWN5IGlzIGlubGluZWQgaW4gdGhlIFJvbGUgcmVzb3VyY2UuXG4gICAqL1xuICByZWFkb25seSBpbmxpbmVQb2xpY2llcz86IHsgW25hbWU6IHN0cmluZ106IFBvbGljeURvY3VtZW50IH07XG5cbiAgLyoqXG4gICAqIFRoZSBwYXRoIGFzc29jaWF0ZWQgd2l0aCB0aGlzIHJvbGUuIEZvciBpbmZvcm1hdGlvbiBhYm91dCBJQU0gcGF0aHMsIHNlZVxuICAgKiBGcmllbmRseSBOYW1lcyBhbmQgUGF0aHMgaW4gSUFNIFVzZXIgR3VpZGUuXG4gICAqXG4gICAqIEBkZWZhdWx0IC9cbiAgICovXG4gIHJlYWRvbmx5IHBhdGg/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEFXUyBzdXBwb3J0cyBwZXJtaXNzaW9ucyBib3VuZGFyaWVzIGZvciBJQU0gZW50aXRpZXMgKHVzZXJzIG9yIHJvbGVzKS5cbiAgICogQSBwZXJtaXNzaW9ucyBib3VuZGFyeSBpcyBhbiBhZHZhbmNlZCBmZWF0dXJlIGZvciB1c2luZyBhIG1hbmFnZWQgcG9saWN5XG4gICAqIHRvIHNldCB0aGUgbWF4aW11bSBwZXJtaXNzaW9ucyB0aGF0IGFuIGlkZW50aXR5LWJhc2VkIHBvbGljeSBjYW4gZ3JhbnQgdG9cbiAgICogYW4gSUFNIGVudGl0eS4gQW4gZW50aXR5J3MgcGVybWlzc2lvbnMgYm91bmRhcnkgYWxsb3dzIGl0IHRvIHBlcmZvcm0gb25seVxuICAgKiB0aGUgYWN0aW9ucyB0aGF0IGFyZSBhbGxvd2VkIGJ5IGJvdGggaXRzIGlkZW50aXR5LWJhc2VkIHBvbGljaWVzIGFuZCBpdHNcbiAgICogcGVybWlzc2lvbnMgYm91bmRhcmllcy5cbiAgICpcbiAgICogQGxpbmsgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU0Nsb3VkRm9ybWF0aW9uL2xhdGVzdC9Vc2VyR3VpZGUvYXdzLXJlc291cmNlLWlhbS1yb2xlLmh0bWwjY2ZuLWlhbS1yb2xlLXBlcm1pc3Npb25zYm91bmRhcnlcbiAgICogQGxpbmsgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL2FjY2Vzc19wb2xpY2llc19ib3VuZGFyaWVzLmh0bWxcbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBwZXJtaXNzaW9ucyBib3VuZGFyeS5cbiAgICovXG4gIHJlYWRvbmx5IHBlcm1pc3Npb25zQm91bmRhcnk/OiBJTWFuYWdlZFBvbGljeTtcblxuICAvKipcbiAgICogQSBuYW1lIGZvciB0aGUgSUFNIHJvbGUuIEZvciB2YWxpZCB2YWx1ZXMsIHNlZSB0aGUgUm9sZU5hbWUgcGFyYW1ldGVyIGZvclxuICAgKiB0aGUgQ3JlYXRlUm9sZSBhY3Rpb24gaW4gdGhlIElBTSBBUEkgUmVmZXJlbmNlLlxuICAgKlxuICAgKiBJTVBPUlRBTlQ6IElmIHlvdSBzcGVjaWZ5IGEgbmFtZSwgeW91IGNhbm5vdCBwZXJmb3JtIHVwZGF0ZXMgdGhhdCByZXF1aXJlXG4gICAqIHJlcGxhY2VtZW50IG9mIHRoaXMgcmVzb3VyY2UuIFlvdSBjYW4gcGVyZm9ybSB1cGRhdGVzIHRoYXQgcmVxdWlyZSBubyBvclxuICAgKiBzb21lIGludGVycnVwdGlvbi4gSWYgeW91IG11c3QgcmVwbGFjZSB0aGUgcmVzb3VyY2UsIHNwZWNpZnkgYSBuZXcgbmFtZS5cbiAgICpcbiAgICogSWYgeW91IHNwZWNpZnkgYSBuYW1lLCB5b3UgbXVzdCBzcGVjaWZ5IHRoZSBDQVBBQklMSVRZX05BTUVEX0lBTSB2YWx1ZSB0b1xuICAgKiBhY2tub3dsZWRnZSB5b3VyIHRlbXBsYXRlJ3MgY2FwYWJpbGl0aWVzLiBGb3IgbW9yZSBpbmZvcm1hdGlvbiwgc2VlXG4gICAqIEFja25vd2xlZGdpbmcgSUFNIFJlc291cmNlcyBpbiBBV1MgQ2xvdWRGb3JtYXRpb24gVGVtcGxhdGVzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIEFXUyBDbG91ZEZvcm1hdGlvbiBnZW5lcmF0ZXMgYSB1bmlxdWUgcGh5c2ljYWwgSUQgYW5kIHVzZXMgdGhhdCBJRFxuICAgKiBmb3IgdGhlIGdyb3VwIG5hbWUuXG4gICAqL1xuICByZWFkb25seSByb2xlTmFtZT86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIG1heGltdW0gc2Vzc2lvbiBkdXJhdGlvbiB0aGF0IHlvdSB3YW50IHRvIHNldCBmb3IgdGhlIHNwZWNpZmllZCByb2xlLlxuICAgKiBUaGlzIHNldHRpbmcgY2FuIGhhdmUgYSB2YWx1ZSBmcm9tIDEgaG91ciAoMzYwMHNlYykgdG8gMTIgKDQzMjAwc2VjKSBob3Vycy5cbiAgICpcbiAgICogQW55b25lIHdobyBhc3N1bWVzIHRoZSByb2xlIGZyb20gdGhlIEFXUyBDTEkgb3IgQVBJIGNhbiB1c2UgdGhlXG4gICAqIER1cmF0aW9uU2Vjb25kcyBBUEkgcGFyYW1ldGVyIG9yIHRoZSBkdXJhdGlvbi1zZWNvbmRzIENMSSBwYXJhbWV0ZXIgdG9cbiAgICogcmVxdWVzdCBhIGxvbmdlciBzZXNzaW9uLiBUaGUgTWF4U2Vzc2lvbkR1cmF0aW9uIHNldHRpbmcgZGV0ZXJtaW5lcyB0aGVcbiAgICogbWF4aW11bSBkdXJhdGlvbiB0aGF0IGNhbiBiZSByZXF1ZXN0ZWQgdXNpbmcgdGhlIER1cmF0aW9uU2Vjb25kc1xuICAgKiBwYXJhbWV0ZXIuXG4gICAqXG4gICAqIElmIHVzZXJzIGRvbid0IHNwZWNpZnkgYSB2YWx1ZSBmb3IgdGhlIER1cmF0aW9uU2Vjb25kcyBwYXJhbWV0ZXIsIHRoZWlyXG4gICAqIHNlY3VyaXR5IGNyZWRlbnRpYWxzIGFyZSB2YWxpZCBmb3Igb25lIGhvdXIgYnkgZGVmYXVsdC4gVGhpcyBhcHBsaWVzIHdoZW5cbiAgICogeW91IHVzZSB0aGUgQXNzdW1lUm9sZSogQVBJIG9wZXJhdGlvbnMgb3IgdGhlIGFzc3VtZS1yb2xlKiBDTEkgb3BlcmF0aW9uc1xuICAgKiBidXQgZG9lcyBub3QgYXBwbHkgd2hlbiB5b3UgdXNlIHRob3NlIG9wZXJhdGlvbnMgdG8gY3JlYXRlIGEgY29uc29sZSBVUkwuXG4gICAqXG4gICAqIEBsaW5rIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9pZF9yb2xlc191c2UuaHRtbFxuICAgKlxuICAgKiBAZGVmYXVsdCBEdXJhdGlvbi5ob3VycygxKVxuICAgKi9cbiAgcmVhZG9ubHkgbWF4U2Vzc2lvbkR1cmF0aW9uPzogRHVyYXRpb247XG59XG5cbi8qKlxuICogT3B0aW9ucyBhbGxvd2luZyBjdXN0b21pemluZyB0aGUgYmVoYXZpb3Igb2Yge0BsaW5rIFJvbGUuZnJvbVJvbGVBcm59LlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEZyb21Sb2xlQXJuT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBXaGV0aGVyIHRoZSBpbXBvcnRlZCByb2xlIGNhbiBiZSBtb2RpZmllZCBieSBhdHRhY2hpbmcgcG9saWN5IHJlc291cmNlcyB0byBpdC5cbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKlxuICAgKiBAZXhwZXJpbWVudGFsXG4gICAqL1xuICByZWFkb25seSBtdXRhYmxlPzogYm9vbGVhbjtcbn1cblxuLyoqXG4gKiBJQU0gUm9sZVxuICpcbiAqIERlZmluZXMgYW4gSUFNIHJvbGUuIFRoZSByb2xlIGlzIGNyZWF0ZWQgd2l0aCBhbiBhc3N1bWUgcG9saWN5IGRvY3VtZW50IGFzc29jaWF0ZWQgd2l0aFxuICogdGhlIHNwZWNpZmllZCBBV1Mgc2VydmljZSBwcmluY2lwYWwgZGVmaW5lZCBpbiBgc2VydmljZUFzc3VtZVJvbGVgLlxuICovXG5leHBvcnQgY2xhc3MgUm9sZSBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSVJvbGUge1xuICAvKipcbiAgICogSW1wb3J0cyBhbiBleHRlcm5hbCByb2xlIGJ5IEFSTi5cbiAgICpcbiAgICogQHBhcmFtIHNjb3BlIGNvbnN0cnVjdCBzY29wZVxuICAgKiBAcGFyYW0gaWQgY29uc3RydWN0IGlkXG4gICAqIEBwYXJhbSByb2xlQXJuIHRoZSBBUk4gb2YgdGhlIHJvbGUgdG8gaW1wb3J0XG4gICAqIEBwYXJhbSBvcHRpb25zIGFsbG93IGN1c3RvbWl6aW5nIHRoZSBiZWhhdmlvciBvZiB0aGUgcmV0dXJuZWQgcm9sZVxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tUm9sZUFybihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCByb2xlQXJuOiBzdHJpbmcsIG9wdGlvbnM6IEZyb21Sb2xlQXJuT3B0aW9ucyA9IHt9KTogSVJvbGUge1xuICAgIGNvbnN0IHNjb3BlU3RhY2sgPSBTdGFjay5vZihzY29wZSk7XG4gICAgY29uc3QgcGFyc2VkQXJuID0gc2NvcGVTdGFjay5wYXJzZUFybihyb2xlQXJuKTtcbiAgICBjb25zdCByb2xlTmFtZSA9IHBhcnNlZEFybi5yZXNvdXJjZU5hbWUhO1xuXG4gICAgYWJzdHJhY3QgY2xhc3MgSW1wb3J0IGV4dGVuZHMgUmVzb3VyY2UgaW1wbGVtZW50cyBJUm9sZSB7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgZ3JhbnRQcmluY2lwYWw6IElQcmluY2lwYWwgPSB0aGlzO1xuICAgICAgcHVibGljIHJlYWRvbmx5IGFzc3VtZVJvbGVBY3Rpb246IHN0cmluZyA9ICdzdHM6QXNzdW1lUm9sZSc7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgcG9saWN5RnJhZ21lbnQgPSBuZXcgQXJuUHJpbmNpcGFsKHJvbGVBcm4pLnBvbGljeUZyYWdtZW50O1xuICAgICAgcHVibGljIHJlYWRvbmx5IHJvbGVBcm4gPSByb2xlQXJuO1xuICAgICAgcHVibGljIHJlYWRvbmx5IHJvbGVOYW1lID0gcm9sZU5hbWU7XG5cbiAgICAgIHB1YmxpYyBhYnN0cmFjdCBhZGRUb1BvbGljeShzdGF0ZW1lbnQ6IFBvbGljeVN0YXRlbWVudCk6IGJvb2xlYW47XG5cbiAgICAgIHB1YmxpYyBhYnN0cmFjdCBhdHRhY2hJbmxpbmVQb2xpY3kocG9saWN5OiBQb2xpY3kpOiB2b2lkO1xuXG4gICAgICBwdWJsaWMgYWRkTWFuYWdlZFBvbGljeShfcG9saWN5OiBJTWFuYWdlZFBvbGljeSk6IHZvaWQge1xuICAgICAgICAvLyBGSVhNRTogQWRkIHdhcm5pbmcgdGhhdCB3ZSdyZSBpZ25vcmluZyB0aGlzXG4gICAgICB9XG5cbiAgICAgIC8qKlxuICAgICAgICogR3JhbnQgcGVybWlzc2lvbnMgdG8gdGhlIGdpdmVuIHByaW5jaXBhbCB0byBwYXNzIHRoaXMgcm9sZS5cbiAgICAgICAqL1xuICAgICAgcHVibGljIGdyYW50UGFzc1JvbGUoaWRlbnRpdHk6IElQcmluY2lwYWwpOiBHcmFudCB7XG4gICAgICAgIHJldHVybiB0aGlzLmdyYW50KGlkZW50aXR5LCAnaWFtOlBhc3NSb2xlJyk7XG4gICAgICB9XG5cbiAgICAgIC8qKlxuICAgICAgICogR3JhbnQgdGhlIGFjdGlvbnMgZGVmaW5lZCBpbiBhY3Rpb25zIHRvIHRoZSBpZGVudGl0eSBQcmluY2lwYWwgb24gdGhpcyByZXNvdXJjZS5cbiAgICAgICAqL1xuICAgICAgcHVibGljIGdyYW50KGdyYW50ZWU6IElQcmluY2lwYWwsIC4uLmFjdGlvbnM6IHN0cmluZ1tdKTogR3JhbnQge1xuICAgICAgICByZXR1cm4gR3JhbnQuYWRkVG9QcmluY2lwYWwoe1xuICAgICAgICAgIGdyYW50ZWUsXG4gICAgICAgICAgYWN0aW9ucyxcbiAgICAgICAgICByZXNvdXJjZUFybnM6IFt0aGlzLnJvbGVBcm5dLFxuICAgICAgICAgIHNjb3BlOiB0aGlzLFxuICAgICAgICB9KTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICBjb25zdCByb2xlQWNjb3VudCA9IHBhcnNlZEFybi5hY2NvdW50O1xuXG4gICAgY2xhc3MgTXV0YWJsZUltcG9ydCBleHRlbmRzIEltcG9ydCB7XG4gICAgICBwcml2YXRlIHJlYWRvbmx5IGF0dGFjaGVkUG9saWNpZXMgPSBuZXcgQXR0YWNoZWRQb2xpY2llcygpO1xuICAgICAgcHJpdmF0ZSBkZWZhdWx0UG9saWN5PzogUG9saWN5O1xuXG4gICAgICBwdWJsaWMgYWRkVG9Qb2xpY3koc3RhdGVtZW50OiBQb2xpY3lTdGF0ZW1lbnQpOiBib29sZWFuIHtcbiAgICAgICAgaWYgKCF0aGlzLmRlZmF1bHRQb2xpY3kpIHtcbiAgICAgICAgICB0aGlzLmRlZmF1bHRQb2xpY3kgPSBuZXcgUG9saWN5KHRoaXMsICdQb2xpY3knKTtcbiAgICAgICAgICB0aGlzLmF0dGFjaElubGluZVBvbGljeSh0aGlzLmRlZmF1bHRQb2xpY3kpO1xuICAgICAgICB9XG4gICAgICAgIHRoaXMuZGVmYXVsdFBvbGljeS5hZGRTdGF0ZW1lbnRzKHN0YXRlbWVudCk7XG4gICAgICAgIHJldHVybiB0cnVlO1xuICAgICAgfVxuXG4gICAgICBwdWJsaWMgYXR0YWNoSW5saW5lUG9saWN5KHBvbGljeTogUG9saWN5KTogdm9pZCB7XG4gICAgICAgIGNvbnN0IHBvbGljeUFjY291bnQgPSBTdGFjay5vZihwb2xpY3kpLmFjY291bnQ7XG5cbiAgICAgICAgaWYgKGFjY291bnRzQXJlRXF1YWxPck9uZUlzVW5yZXNvbHZlZChwb2xpY3lBY2NvdW50LCByb2xlQWNjb3VudCkpIHtcbiAgICAgICAgICB0aGlzLmF0dGFjaGVkUG9saWNpZXMuYXR0YWNoKHBvbGljeSk7XG4gICAgICAgICAgcG9saWN5LmF0dGFjaFRvUm9sZSh0aGlzKTtcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH1cblxuICAgIGNsYXNzIEltbXV0YWJsZUltcG9ydCBleHRlbmRzIEltcG9ydCB7XG4gICAgICBwdWJsaWMgYWRkVG9Qb2xpY3koX3N0YXRlbWVudDogUG9saWN5U3RhdGVtZW50KTogYm9vbGVhbiB7XG4gICAgICAgIHJldHVybiB0cnVlO1xuICAgICAgfVxuXG4gICAgICBwdWJsaWMgYXR0YWNoSW5saW5lUG9saWN5KF9wb2xpY3k6IFBvbGljeSk6IHZvaWQge1xuICAgICAgICAvLyBkbyBub3RoaW5nXG4gICAgICB9XG4gICAgfVxuXG4gICAgY29uc3Qgc2NvcGVBY2NvdW50ID0gc2NvcGVTdGFjay5hY2NvdW50O1xuXG4gICAgcmV0dXJuIG9wdGlvbnMubXV0YWJsZSAhPT0gZmFsc2UgJiYgYWNjb3VudHNBcmVFcXVhbE9yT25lSXNVbnJlc29sdmVkKHNjb3BlQWNjb3VudCwgcm9sZUFjY291bnQpXG4gICAgICA/IG5ldyBNdXRhYmxlSW1wb3J0KHNjb3BlLCBpZClcbiAgICAgIDogbmV3IEltbXV0YWJsZUltcG9ydChzY29wZSwgaWQpO1xuXG4gICAgZnVuY3Rpb24gYWNjb3VudHNBcmVFcXVhbE9yT25lSXNVbnJlc29sdmVkKGFjY291bnQxOiBzdHJpbmcgfCB1bmRlZmluZWQsXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIGFjY291bnQyOiBzdHJpbmcgfCB1bmRlZmluZWQpOiBib29sZWFuIHtcbiAgICAgIHJldHVybiBUb2tlbi5pc1VucmVzb2x2ZWQoYWNjb3VudDEpIHx8IFRva2VuLmlzVW5yZXNvbHZlZChhY2NvdW50MikgfHxcbiAgICAgICAgYWNjb3VudDEgPT09IGFjY291bnQyO1xuICAgIH1cbiAgfVxuXG4gIHB1YmxpYyByZWFkb25seSBncmFudFByaW5jaXBhbDogSVByaW5jaXBhbCA9IHRoaXM7XG5cbiAgcHVibGljIHJlYWRvbmx5IGFzc3VtZVJvbGVBY3Rpb246IHN0cmluZyA9ICdzdHM6QXNzdW1lUm9sZSc7XG5cbiAgLyoqXG4gICAqIFRoZSBhc3N1bWUgcm9sZSBwb2xpY3kgZG9jdW1lbnQgYXNzb2NpYXRlZCB3aXRoIHRoaXMgcm9sZS5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBhc3N1bWVSb2xlUG9saWN5PzogUG9saWN5RG9jdW1lbnQ7XG5cbiAgLyoqXG4gICAqIFJldHVybnMgdGhlIEFSTiBvZiB0aGlzIHJvbGUuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgcm9sZUFybjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBSZXR1cm5zIHRoZSBzdGFibGUgYW5kIHVuaXF1ZSBzdHJpbmcgaWRlbnRpZnlpbmcgdGhlIHJvbGUuIEZvciBleGFtcGxlLFxuICAgKiBBSURBSlFBQkxaUzRBM1FEVTU3NlEuXG4gICAqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSByb2xlSWQ6IHN0cmluZztcblxuICAvKipcbiAgICogUmV0dXJucyB0aGUgbmFtZSBvZiB0aGUgcm9sZS5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSByb2xlTmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBSZXR1cm5zIHRoZSByb2xlLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHBvbGljeUZyYWdtZW50OiBQcmluY2lwYWxQb2xpY3lGcmFnbWVudDtcblxuICAvKipcbiAgICogUmV0dXJucyB0aGUgcGVybWlzc2lvbnMgYm91bmRhcnkgYXR0YWNoZWQgdG8gdGhpcyByb2xlXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgcGVybWlzc2lvbnNCb3VuZGFyeT86IElNYW5hZ2VkUG9saWN5O1xuXG4gIHByaXZhdGUgZGVmYXVsdFBvbGljeT86IFBvbGljeTtcbiAgcHJpdmF0ZSByZWFkb25seSBtYW5hZ2VkUG9saWNpZXM6IElNYW5hZ2VkUG9saWN5W10gPSBbXTtcbiAgcHJpdmF0ZSByZWFkb25seSBhdHRhY2hlZFBvbGljaWVzID0gbmV3IEF0dGFjaGVkUG9saWNpZXMoKTtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogUm9sZVByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkLCB7XG4gICAgICBwaHlzaWNhbE5hbWU6IHByb3BzLnJvbGVOYW1lLFxuICAgIH0pO1xuXG4gICAgY29uc3QgZXh0ZXJuYWxJZHMgPSBwcm9wcy5leHRlcm5hbElkcyB8fCBbXTtcbiAgICBpZiAocHJvcHMuZXh0ZXJuYWxJZCkge1xuICAgICAgZXh0ZXJuYWxJZHMucHVzaChwcm9wcy5leHRlcm5hbElkKTtcbiAgICB9XG5cbiAgICB0aGlzLmFzc3VtZVJvbGVQb2xpY3kgPSBjcmVhdGVBc3N1bWVSb2xlUG9saWN5KHByb3BzLmFzc3VtZWRCeSwgZXh0ZXJuYWxJZHMpO1xuICAgIHRoaXMubWFuYWdlZFBvbGljaWVzLnB1c2goLi4ucHJvcHMubWFuYWdlZFBvbGljaWVzIHx8IFtdKTtcbiAgICB0aGlzLnBlcm1pc3Npb25zQm91bmRhcnkgPSBwcm9wcy5wZXJtaXNzaW9uc0JvdW5kYXJ5O1xuICAgIGNvbnN0IG1heFNlc3Npb25EdXJhdGlvbiA9IHByb3BzLm1heFNlc3Npb25EdXJhdGlvbiAmJiBwcm9wcy5tYXhTZXNzaW9uRHVyYXRpb24udG9TZWNvbmRzKCk7XG4gICAgdmFsaWRhdGVNYXhTZXNzaW9uRHVyYXRpb24obWF4U2Vzc2lvbkR1cmF0aW9uKTtcblxuICAgIGNvbnN0IHJvbGUgPSBuZXcgQ2ZuUm9sZSh0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICBhc3N1bWVSb2xlUG9saWN5RG9jdW1lbnQ6IHRoaXMuYXNzdW1lUm9sZVBvbGljeSBhcyBhbnksXG4gICAgICBtYW5hZ2VkUG9saWN5QXJuczogTGF6eS5saXN0VmFsdWUoeyBwcm9kdWNlOiAoKSA9PiB0aGlzLm1hbmFnZWRQb2xpY2llcy5tYXAocCA9PiBwLm1hbmFnZWRQb2xpY3lBcm4pIH0sIHsgb21pdEVtcHR5OiB0cnVlIH0pLFxuICAgICAgcG9saWNpZXM6IF9mbGF0dGVuKHByb3BzLmlubGluZVBvbGljaWVzKSxcbiAgICAgIHBhdGg6IHByb3BzLnBhdGgsXG4gICAgICBwZXJtaXNzaW9uc0JvdW5kYXJ5OiB0aGlzLnBlcm1pc3Npb25zQm91bmRhcnkgPyB0aGlzLnBlcm1pc3Npb25zQm91bmRhcnkubWFuYWdlZFBvbGljeUFybiA6IHVuZGVmaW5lZCxcbiAgICAgIHJvbGVOYW1lOiB0aGlzLnBoeXNpY2FsTmFtZSxcbiAgICAgIG1heFNlc3Npb25EdXJhdGlvbixcbiAgICB9KTtcblxuICAgIHRoaXMucm9sZUlkID0gcm9sZS5hdHRyUm9sZUlkO1xuICAgIHRoaXMucm9sZUFybiA9IHRoaXMuZ2V0UmVzb3VyY2VBcm5BdHRyaWJ1dGUocm9sZS5hdHRyQXJuLCB7XG4gICAgICByZWdpb246ICcnLCAvLyBJQU0gaXMgZ2xvYmFsIGluIGVhY2ggcGFydGl0aW9uXG4gICAgICBzZXJ2aWNlOiAnaWFtJyxcbiAgICAgIHJlc291cmNlOiAncm9sZScsXG4gICAgICByZXNvdXJjZU5hbWU6IHRoaXMucGh5c2ljYWxOYW1lLFxuICAgIH0pO1xuICAgIHRoaXMucm9sZU5hbWUgPSB0aGlzLmdldFJlc291cmNlTmFtZUF0dHJpYnV0ZShyb2xlLnJlZik7XG4gICAgdGhpcy5wb2xpY3lGcmFnbWVudCA9IG5ldyBBcm5QcmluY2lwYWwodGhpcy5yb2xlQXJuKS5wb2xpY3lGcmFnbWVudDtcblxuICAgIGZ1bmN0aW9uIF9mbGF0dGVuKHBvbGljaWVzPzogeyBbbmFtZTogc3RyaW5nXTogUG9saWN5RG9jdW1lbnQgfSkge1xuICAgICAgaWYgKHBvbGljaWVzID09IG51bGwgfHwgT2JqZWN0LmtleXMocG9saWNpZXMpLmxlbmd0aCA9PT0gMCkge1xuICAgICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgICAgfVxuICAgICAgY29uc3QgcmVzdWx0ID0gbmV3IEFycmF5PENmblJvbGUuUG9saWN5UHJvcGVydHk+KCk7XG4gICAgICBmb3IgKGNvbnN0IHBvbGljeU5hbWUgb2YgT2JqZWN0LmtleXMocG9saWNpZXMpKSB7XG4gICAgICAgIGNvbnN0IHBvbGljeURvY3VtZW50ID0gcG9saWNpZXNbcG9saWN5TmFtZV07XG4gICAgICAgIHJlc3VsdC5wdXNoKHsgcG9saWN5TmFtZSwgcG9saWN5RG9jdW1lbnQgfSk7XG4gICAgICB9XG4gICAgICByZXR1cm4gcmVzdWx0O1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcGVybWlzc2lvbiB0byB0aGUgcm9sZSdzIGRlZmF1bHQgcG9saWN5IGRvY3VtZW50LlxuICAgKiBJZiB0aGVyZSBpcyBubyBkZWZhdWx0IHBvbGljeSBhdHRhY2hlZCB0byB0aGlzIHJvbGUsIGl0IHdpbGwgYmUgY3JlYXRlZC5cbiAgICogQHBhcmFtIHN0YXRlbWVudCBUaGUgcGVybWlzc2lvbiBzdGF0ZW1lbnQgdG8gYWRkIHRvIHRoZSBwb2xpY3kgZG9jdW1lbnRcbiAgICovXG4gIHB1YmxpYyBhZGRUb1BvbGljeShzdGF0ZW1lbnQ6IFBvbGljeVN0YXRlbWVudCk6IGJvb2xlYW4ge1xuICAgIGlmICghdGhpcy5kZWZhdWx0UG9saWN5KSB7XG4gICAgICB0aGlzLmRlZmF1bHRQb2xpY3kgPSBuZXcgUG9saWN5KHRoaXMsICdEZWZhdWx0UG9saWN5Jyk7XG4gICAgICB0aGlzLmF0dGFjaElubGluZVBvbGljeSh0aGlzLmRlZmF1bHRQb2xpY3kpO1xuICAgIH1cbiAgICB0aGlzLmRlZmF1bHRQb2xpY3kuYWRkU3RhdGVtZW50cyhzdGF0ZW1lbnQpO1xuICAgIHJldHVybiB0cnVlO1xuICB9XG5cbiAgLyoqXG4gICAqIEF0dGFjaGVzIGEgbWFuYWdlZCBwb2xpY3kgdG8gdGhpcyByb2xlLlxuICAgKiBAcGFyYW0gcG9saWN5IFRoZSB0aGUgbWFuYWdlZCBwb2xpY3kgdG8gYXR0YWNoLlxuICAgKi9cbiAgcHVibGljIGFkZE1hbmFnZWRQb2xpY3kocG9saWN5OiBJTWFuYWdlZFBvbGljeSkge1xuICAgIGlmICh0aGlzLm1hbmFnZWRQb2xpY2llcy5maW5kKG1wID0+IG1wID09PSBwb2xpY3kpKSB7IHJldHVybjsgfVxuICAgIHRoaXMubWFuYWdlZFBvbGljaWVzLnB1c2gocG9saWN5KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBdHRhY2hlcyBhIHBvbGljeSB0byB0aGlzIHJvbGUuXG4gICAqIEBwYXJhbSBwb2xpY3kgVGhlIHBvbGljeSB0byBhdHRhY2hcbiAgICovXG4gIHB1YmxpYyBhdHRhY2hJbmxpbmVQb2xpY3kocG9saWN5OiBQb2xpY3kpIHtcbiAgICB0aGlzLmF0dGFjaGVkUG9saWNpZXMuYXR0YWNoKHBvbGljeSk7XG4gICAgcG9saWN5LmF0dGFjaFRvUm9sZSh0aGlzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudCB0aGUgYWN0aW9ucyBkZWZpbmVkIGluIGFjdGlvbnMgdG8gdGhlIGlkZW50aXR5IFByaW5jaXBhbCBvbiB0aGlzIHJlc291cmNlLlxuICAgKi9cbiAgcHVibGljIGdyYW50KGdyYW50ZWU6IElQcmluY2lwYWwsIC4uLmFjdGlvbnM6IHN0cmluZ1tdKSB7XG4gICAgcmV0dXJuIEdyYW50LmFkZFRvUHJpbmNpcGFsKHtcbiAgICAgIGdyYW50ZWUsXG4gICAgICBhY3Rpb25zLFxuICAgICAgcmVzb3VyY2VBcm5zOiBbdGhpcy5yb2xlQXJuXSxcbiAgICAgIHNjb3BlOiB0aGlzXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnQgcGVybWlzc2lvbnMgdG8gdGhlIGdpdmVuIHByaW5jaXBhbCB0byBwYXNzIHRoaXMgcm9sZS5cbiAgICovXG4gIHB1YmxpYyBncmFudFBhc3NSb2xlKGlkZW50aXR5OiBJUHJpbmNpcGFsKSB7XG4gICAgcmV0dXJuIHRoaXMuZ3JhbnQoaWRlbnRpdHksICdpYW06UGFzc1JvbGUnKTtcbiAgfVxufVxuXG4vKipcbiAqIEEgUm9sZSBvYmplY3RcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJUm9sZSBleHRlbmRzIElJZGVudGl0eSB7XG4gIC8qKlxuICAgKiBSZXR1cm5zIHRoZSBBUk4gb2YgdGhpcyByb2xlLlxuICAgKlxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICByZWFkb25seSByb2xlQXJuOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFJldHVybnMgdGhlIG5hbWUgb2YgdGhpcyByb2xlLlxuICAgKlxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICByZWFkb25seSByb2xlTmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBHcmFudCB0aGUgYWN0aW9ucyBkZWZpbmVkIGluIGFjdGlvbnMgdG8gdGhlIGlkZW50aXR5IFByaW5jaXBhbCBvbiB0aGlzIHJlc291cmNlLlxuICAgKi9cbiAgZ3JhbnQoZ3JhbnRlZTogSVByaW5jaXBhbCwgLi4uYWN0aW9uczogc3RyaW5nW10pOiBHcmFudDtcblxuICAvKipcbiAgICogR3JhbnQgcGVybWlzc2lvbnMgdG8gdGhlIGdpdmVuIHByaW5jaXBhbCB0byBwYXNzIHRoaXMgcm9sZS5cbiAgICovXG4gIGdyYW50UGFzc1JvbGUoZ3JhbnRlZTogSVByaW5jaXBhbCk6IEdyYW50O1xufVxuXG5mdW5jdGlvbiBjcmVhdGVBc3N1bWVSb2xlUG9saWN5KHByaW5jaXBhbDogSVByaW5jaXBhbCwgZXh0ZXJuYWxJZHM6IHN0cmluZ1tdKSB7XG4gIGNvbnN0IHN0YXRlbWVudCA9IG5ldyBQb2xpY3lTdGF0ZW1lbnQoKTtcbiAgc3RhdGVtZW50LmFkZFByaW5jaXBhbHMocHJpbmNpcGFsKTtcbiAgc3RhdGVtZW50LmFkZEFjdGlvbnMocHJpbmNpcGFsLmFzc3VtZVJvbGVBY3Rpb24pO1xuXG4gIGlmIChleHRlcm5hbElkcy5sZW5ndGgpIHtcbiAgICBzdGF0ZW1lbnQuYWRkQ29uZGl0aW9uKCdTdHJpbmdFcXVhbHMnLCB7ICdzdHM6RXh0ZXJuYWxJZCc6IGV4dGVybmFsSWRzLmxlbmd0aCA9PT0gMSA/IGV4dGVybmFsSWRzWzBdIDogZXh0ZXJuYWxJZHMgfSk7XG4gIH1cblxuICBjb25zdCBkb2MgPSBuZXcgUG9saWN5RG9jdW1lbnQoKTtcbiAgZG9jLmFkZFN0YXRlbWVudHMoc3RhdGVtZW50KTtcbiAgcmV0dXJuIGRvYztcbn1cblxuZnVuY3Rpb24gdmFsaWRhdGVNYXhTZXNzaW9uRHVyYXRpb24oZHVyYXRpb24/OiBudW1iZXIpIHtcbiAgaWYgKGR1cmF0aW9uID09PSB1bmRlZmluZWQpIHtcbiAgICByZXR1cm47XG4gIH1cblxuICBpZiAoZHVyYXRpb24gPCAzNjAwIHx8IGR1cmF0aW9uID4gNDMyMDApIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoYG1heFNlc3Npb25EdXJhdGlvbiBpcyBzZXQgdG8gJHtkdXJhdGlvbn0sIGJ1dCBtdXN0IGJlID49IDM2MDBzZWMgKDFocikgYW5kIDw9IDQzMjAwc2VjICgxMmhycylgKTtcbiAgfVxufVxuIl19