"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
/**
 * A principal for use in resources that need to have a role but it's unknown
 *
 * Some resources have roles associated with them which they assume, such as
 * Lambda Functions, CodeBuild projects, StepFunctions machines, etc.
 *
 * When those resources are imported, their actual roles are not always
 * imported with them. When that happens, we use an instance of this class
 * instead, which will add user warnings when statements are attempted to be
 * added to it.
 */
class UnknownPrincipal {
    constructor(props) {
        this.assumeRoleAction = 'sts:AssumeRole';
        this.resource = props.resource;
        this.grantPrincipal = this;
    }
    get policyFragment() {
        throw new Error(`Cannot get policy fragment of ${this.resource.node.path}, resource imported without a role`);
    }
    addToPolicy(statement) {
        const stack = core_1.Stack.of(this.resource);
        const repr = JSON.stringify(stack.resolve(statement));
        this.resource.node.addWarning(`Add statement to this resource's role: ${repr}`);
        return true; // Pretend we did the work. The human will do it for us, eventually.
    }
}
exports.UnknownPrincipal = UnknownPrincipal;
//# sourceMappingURL=data:application/json;base64,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