"""
Tornado integration for Telescope client.
"""

import sys
from typing import TYPE_CHECKING, Any, Dict, Optional

from opentelemetry import trace
from opentelemetry.instrumentation.tornado import TornadoInstrumentor

if TYPE_CHECKING:
    from ..client import TelescopeClient
    from ..integrations.base import Integration


class TornadoIntegration:
    """
    Tornado integration for Telescope client.
    
    Automatically captures exceptions and provides tracing for Tornado applications.
    """
    
    identifier = "tornado"
    
    def __init__(self, **options: Any):
        """Initialize Tornado integration."""
        self.options = options
    
    def setup(self, client: "TelescopeClient"):
        """
        Setup Tornado integration with the Telescope client.
        
        Args:
            client: TelescopeClient instance
        """
        try:
            from tornado.web import RequestHandler, Application
            from tornado.gen import coroutine
            import tornado.web
            
            class TelescopeRequestHandler(RequestHandler):
                """Base request handler with Telescope integration."""
                
                def write_error(self, status_code: int, **kwargs):
                    """Handle errors with Telescope."""
                    if "exc_info" in kwargs:
                        exc_type, exc_value, exc_traceback = kwargs["exc_info"]
                        
                        user_context = {}
                        if hasattr(self, "current_user") and self.current_user:
                            user_context = {
                                "id": str(getattr(self.current_user, "id", "")),
                                "username": getattr(self.current_user, "username", ""),
                                "email": getattr(self.current_user, "email", ""),
                            }
                        
                        tags = {
                            "url": self.request.uri,
                            "method": self.request.method,
                            "user_agent": self.request.headers.get("User-Agent", ""),
                        }
                        
                        extra = {
                            "request": {
                                "url": self.request.uri,
                                "method": self.request.method,
                                "headers": dict(self.request.headers),
                                "query_string": dict(self.request.arguments),
                            }
                        }
                        
                        event_id = client.capture_exception(
                            exc_value,
                            tags=tags,
                            extra=extra,
                            user=user_context,
                        )
                        
                        self.write(f"Internal server error (Event ID: {event_id})")
                    else:
                        self.write(f"Error {status_code}")
            
            def init_app(app: Application):
                """Initialize Tornado app with Telescope."""
                TornadoInstrumentor().instrument()
                
                # Replace default RequestHandler with TelescopeRequestHandler
                app.default_handler_class = TelescopeRequestHandler
            
            return init_app
            
        except ImportError:
            from ..integrations.base import DidNotEnable
            raise DidNotEnable("Tornado not found, skipping Tornado integration")


def setup_tornado_integration(client: "TelescopeClient"):
    """
    Legacy function for backward compatibility.
    
    Args:
        client: TelescopeClient instance
    """
    return TornadoIntegration().setup(client)
