"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const crypto = require("crypto");
const fs = require("fs");
const path = require("path");
const options_1 = require("./options");
const utils_1 = require("./utils");
const BUFFER_SIZE = 8 * 1024;
const CTRL_SOH = '\x01';
const CTRL_SOT = '\x02';
const CTRL_ETX = '\x03';
/**
 * Produces fingerprint based on the contents of a single file or an entire directory tree.
 *
 * The fingerprint will also include:
 * 1. An extra string if defined in `options.extra`.
 * 2. The set of exclude patterns, if defined in `options.exclude`
 * 3. The symlink follow mode value.
 *
 * @param fileOrDirectory The directory or file to fingerprint
 * @param options Fingerprinting options
 */
function fingerprint(fileOrDirectory, options = {}) {
    const hash = crypto.createHash('sha256');
    _hashField(hash, 'options.extra', options.extraHash || '');
    const follow = options.follow || options_1.SymlinkFollowMode.EXTERNAL;
    _hashField(hash, 'options.follow', follow);
    const rootDirectory = fs.statSync(fileOrDirectory).isDirectory()
        ? fileOrDirectory
        : path.dirname(fileOrDirectory);
    const exclude = options.exclude || [];
    if (exclude.length) {
        _hashField(hash, 'options.exclude', JSON.stringify(exclude));
    }
    const isDir = fs.statSync(fileOrDirectory).isDirectory();
    _processFileOrDirectory(fileOrDirectory, isDir);
    return hash.digest('hex');
    function _processFileOrDirectory(symbolicPath, isRootDir = false, realPath = symbolicPath) {
        if (!isRootDir && utils_1.shouldExclude(exclude, symbolicPath)) {
            return;
        }
        const stat = fs.lstatSync(realPath);
        const relativePath = path.relative(fileOrDirectory, symbolicPath);
        if (stat.isSymbolicLink()) {
            const linkTarget = fs.readlinkSync(realPath);
            const resolvedLinkTarget = path.resolve(path.dirname(realPath), linkTarget);
            if (utils_1.shouldFollow(follow, rootDirectory, resolvedLinkTarget)) {
                _processFileOrDirectory(symbolicPath, false, resolvedLinkTarget);
            }
            else {
                _hashField(hash, `link:${relativePath}`, linkTarget);
            }
        }
        else if (stat.isFile()) {
            _hashField(hash, `file:${relativePath}`, _contentFingerprint(realPath, stat));
        }
        else if (stat.isDirectory()) {
            for (const item of fs.readdirSync(realPath).sort()) {
                _processFileOrDirectory(path.join(symbolicPath, item), false, path.join(realPath, item));
            }
        }
        else {
            throw new Error(`Unable to hash ${symbolicPath}: it is neither a file nor a directory`);
        }
    }
}
exports.fingerprint = fingerprint;
function _contentFingerprint(file, stat) {
    const hash = crypto.createHash('sha256');
    const buffer = Buffer.alloc(BUFFER_SIZE);
    // tslint:disable-next-line: no-bitwise
    const fd = fs.openSync(file, fs.constants.O_DSYNC | fs.constants.O_RDONLY | fs.constants.O_SYNC);
    try {
        let read = 0;
        // tslint:disable-next-line: no-conditional-assignment
        while ((read = fs.readSync(fd, buffer, 0, BUFFER_SIZE, null)) !== 0) {
            hash.update(buffer.slice(0, read));
        }
    }
    finally {
        fs.closeSync(fd);
    }
    return `${stat.size}:${hash.digest('hex')}`;
}
function _hashField(hash, header, value) {
    hash.update(CTRL_SOH).update(header).update(CTRL_SOT).update(value).update(CTRL_ETX);
}
//# sourceMappingURL=data:application/json;base64,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