"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const core_1 = require("@aws-solutions-constructs/core");
const integ_tests_alpha_1 = require("@aws-cdk/integ-tests-alpha");
const lambda = require("aws-cdk-lib/aws-lambda");
const defaults = require("@aws-solutions-constructs/core");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
// Note: All integration tests for alb are for HTTP APIs, as certificates require
// validation through DNS and email. This validation is impossible during our integration
// tests and the stack will fail to launch with an unvalidated certificate.
// Setup
const app = new aws_cdk_lib_1.App();
const stack = new aws_cdk_lib_1.Stack(app, core_1.generateIntegStackName(__filename), {
    env: { account: aws_cdk_lib_1.Aws.ACCOUNT_ID, region: 'us-east-1' },
});
stack.templateOptions.description = 'Integration Test for private HTTP API with a existing function and ALB';
const myVpc = defaults.getTestVpc(stack);
const testSg = new aws_ec2_1.SecurityGroup(stack, 'lambda-sg', { vpc: myVpc, allowAllOutbound: false });
const lambdaFunction = defaults.buildLambdaFunction(stack, {
    lambdaFunctionProps: {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler',
        vpc: myVpc,
        securityGroups: [testSg]
    }
});
const loadBalancer = defaults.ObtainAlb(stack, 'existing-alb', {
    vpc: myVpc,
    publicApi: false,
    loadBalancerProps: {
        internetFacing: false,
        vpc: myVpc,
    },
    logAccessLogs: true,
    loggingBucketProps: {
        removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        autoDeleteObjects: true
    }
});
const props = {
    existingLambdaObj: lambdaFunction,
    existingLoadBalancerObj: loadBalancer,
    existingVpc: myVpc,
    listenerProps: {
        protocol: 'HTTP'
    },
    publicApi: false
};
const albToLambda = new lib_1.AlbToLambda(stack, 'test-one', props);
defaults.addCfnSuppressRules(albToLambda.listener, [
    { id: 'W56', reason: 'All integration tests must be HTTP because of certificate limitations.' },
]);
const newSecurityGroup = albToLambda.loadBalancer.connections.securityGroups[0].node.defaultChild;
defaults.addCfnSuppressRules(newSecurityGroup, [
    { id: 'W29', reason: 'CDK created rule that blocks all traffic.' },
    { id: 'W2', reason: 'Rule does not apply for ELB.' },
    { id: 'W9', reason: 'Rule does not apply for ELB.' }
]);
defaults.addCfnSuppressRules(testSg, [
    { id: 'W29', reason: 'CDK created rule that blocks all traffic.' },
]);
defaults.suppressCustomHandlerCfnNagWarnings(stack, 'Custom::S3AutoDeleteObjectsCustomResourceProvider');
defaults.suppressCustomHandlerCfnNagWarnings(stack, 'Custom::VpcRestrictDefaultSGCustomResourceProvider');
// Synth
new integ_tests_alpha_1.IntegTest(stack, 'Integ', { testCases: [
        stack
    ] });
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW50ZWcuYWxibGFtLXByaXZhdGVBcGlFeGlzdGluZ1Jlc291cmNlcy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImludGVnLmFsYmxhbS1wcml2YXRlQXBpRXhpc3RpbmdSZXNvdXJjZXMudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBOzs7Ozs7Ozs7OztHQVdHOztBQUVILFVBQVU7QUFDViw2Q0FBNkQ7QUFDN0QsZ0NBQXVEO0FBQ3ZELHlEQUF3RTtBQUN4RSxrRUFBdUQ7QUFDdkQsaURBQWlEO0FBQ2pELDJEQUEyRDtBQUMzRCxpREFBc0U7QUFFdEUsaUZBQWlGO0FBQ2pGLHlGQUF5RjtBQUN6RiwyRUFBMkU7QUFFM0UsUUFBUTtBQUNSLE1BQU0sR0FBRyxHQUFHLElBQUksaUJBQUcsRUFBRSxDQUFDO0FBQ3RCLE1BQU0sS0FBSyxHQUFHLElBQUksbUJBQUssQ0FBQyxHQUFHLEVBQUUsNkJBQXNCLENBQUMsVUFBVSxDQUFDLEVBQUU7SUFDL0QsR0FBRyxFQUFFLEVBQUUsT0FBTyxFQUFFLGlCQUFHLENBQUMsVUFBVSxFQUFFLE1BQU0sRUFBRSxXQUFXLEVBQUU7Q0FDdEQsQ0FBQyxDQUFDO0FBQ0gsS0FBSyxDQUFDLGVBQWUsQ0FBQyxXQUFXLEdBQUcsd0VBQXdFLENBQUM7QUFFN0csTUFBTSxLQUFLLEdBQUcsUUFBUSxDQUFDLFVBQVUsQ0FBQyxLQUFLLENBQUMsQ0FBQztBQUV6QyxNQUFNLE1BQU0sR0FBRyxJQUFJLHVCQUFhLENBQUMsS0FBSyxFQUFFLFdBQVcsRUFBRSxFQUFFLEdBQUcsRUFBRSxLQUFLLEVBQUUsZ0JBQWdCLEVBQUUsS0FBSyxFQUFDLENBQUMsQ0FBQztBQUU3RixNQUFNLGNBQWMsR0FBRyxRQUFRLENBQUMsbUJBQW1CLENBQUMsS0FBSyxFQUFFO0lBQ3pELG1CQUFtQixFQUFFO1FBQ25CLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLFNBQVMsU0FBUyxDQUFDO1FBQ2xELE9BQU8sRUFBRSxRQUFRLENBQUMscUNBQXFDO1FBQ3ZELE9BQU8sRUFBRSxlQUFlO1FBQ3hCLEdBQUcsRUFBRSxLQUFLO1FBQ1YsY0FBYyxFQUFFLENBQUUsTUFBTSxDQUFFO0tBQzNCO0NBQ0YsQ0FBQyxDQUFDO0FBRUgsTUFBTSxZQUFZLEdBQUcsUUFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLEVBQUUsY0FBYyxFQUFFO0lBQzdELEdBQUcsRUFBRSxLQUFLO0lBQ1YsU0FBUyxFQUFFLEtBQUs7SUFDaEIsaUJBQWlCLEVBQUU7UUFDakIsY0FBYyxFQUFFLEtBQUs7UUFDckIsR0FBRyxFQUFFLEtBQUs7S0FDWDtJQUNELGFBQWEsRUFBRSxJQUFJO0lBQ25CLGtCQUFrQixFQUFFO1FBQ2xCLGFBQWEsRUFBRSwyQkFBYSxDQUFDLE9BQU87UUFDcEMsaUJBQWlCLEVBQUUsSUFBSTtLQUN4QjtDQUNGLENBQUMsQ0FBQztBQUVILE1BQU0sS0FBSyxHQUFxQjtJQUM5QixpQkFBaUIsRUFBRSxjQUFjO0lBQ2pDLHVCQUF1QixFQUFFLFlBQVk7SUFDckMsV0FBVyxFQUFFLEtBQUs7SUFDbEIsYUFBYSxFQUFFO1FBQ2IsUUFBUSxFQUFFLE1BQU07S0FDakI7SUFDRCxTQUFTLEVBQUUsS0FBSztDQUNqQixDQUFDO0FBRUYsTUFBTSxXQUFXLEdBQUcsSUFBSSxpQkFBVyxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUUsS0FBSyxDQUFDLENBQUM7QUFFOUQsUUFBUSxDQUFDLG1CQUFtQixDQUFDLFdBQVcsQ0FBQyxRQUFRLEVBQUU7SUFDakQsRUFBRSxFQUFFLEVBQUUsS0FBSyxFQUFFLE1BQU0sRUFBRSx3RUFBd0UsRUFBRTtDQUNoRyxDQUFDLENBQUM7QUFFSCxNQUFNLGdCQUFnQixHQUFHLFdBQVcsQ0FBQyxZQUFZLENBQUMsV0FBVyxDQUFDLGNBQWMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsWUFBZ0MsQ0FBQztBQUN0SCxRQUFRLENBQUMsbUJBQW1CLENBQUMsZ0JBQWdCLEVBQUU7SUFDN0MsRUFBRSxFQUFFLEVBQUUsS0FBSyxFQUFFLE1BQU0sRUFBRSwyQ0FBMkMsRUFBQztJQUNqRSxFQUFFLEVBQUUsRUFBRSxJQUFJLEVBQUUsTUFBTSxFQUFFLDhCQUE4QixFQUFDO0lBQ25ELEVBQUUsRUFBRSxFQUFFLElBQUksRUFBRSxNQUFNLEVBQUUsOEJBQThCLEVBQUM7Q0FDcEQsQ0FBQyxDQUFDO0FBRUgsUUFBUSxDQUFDLG1CQUFtQixDQUFDLE1BQU0sRUFBRTtJQUNuQyxFQUFFLEVBQUUsRUFBRSxLQUFLLEVBQUUsTUFBTSxFQUFFLDJDQUEyQyxFQUFDO0NBQ2xFLENBQUMsQ0FBQztBQUVILFFBQVEsQ0FBQyxtQ0FBbUMsQ0FBQyxLQUFLLEVBQUUsbURBQW1ELENBQUMsQ0FBQztBQUN6RyxRQUFRLENBQUMsbUNBQW1DLENBQUMsS0FBSyxFQUFFLG9EQUFvRCxDQUFDLENBQUM7QUFFMUcsUUFBUTtBQUNSLElBQUksNkJBQVMsQ0FBQyxLQUFLLEVBQUUsT0FBTyxFQUFFLEVBQUUsU0FBUyxFQUFFO1FBQ3pDLEtBQUs7S0FDTixFQUFFLENBQUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogIENvcHlyaWdodCBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICpcbiAqICBMaWNlbnNlZCB1bmRlciB0aGUgQXBhY2hlIExpY2Vuc2UsIFZlcnNpb24gMi4wICh0aGUgXCJMaWNlbnNlXCIpLiBZb3UgbWF5IG5vdCB1c2UgdGhpcyBmaWxlIGV4Y2VwdCBpbiBjb21wbGlhbmNlXG4gKiAgd2l0aCB0aGUgTGljZW5zZS4gQSBjb3B5IG9mIHRoZSBMaWNlbnNlIGlzIGxvY2F0ZWQgYXRcbiAqXG4gKiAgICAgIGh0dHA6Ly93d3cuYXBhY2hlLm9yZy9saWNlbnNlcy9MSUNFTlNFLTIuMFxuICpcbiAqICBvciBpbiB0aGUgJ2xpY2Vuc2UnIGZpbGUgYWNjb21wYW55aW5nIHRoaXMgZmlsZS4gVGhpcyBmaWxlIGlzIGRpc3RyaWJ1dGVkIG9uIGFuICdBUyBJUycgQkFTSVMsIFdJVEhPVVQgV0FSUkFOVElFU1xuICogIE9SIENPTkRJVElPTlMgT0YgQU5ZIEtJTkQsIGV4cHJlc3Mgb3IgaW1wbGllZC4gU2VlIHRoZSBMaWNlbnNlIGZvciB0aGUgc3BlY2lmaWMgbGFuZ3VhZ2UgZ292ZXJuaW5nIHBlcm1pc3Npb25zXG4gKiAgYW5kIGxpbWl0YXRpb25zIHVuZGVyIHRoZSBMaWNlbnNlLlxuICovXG5cbi8vIEltcG9ydHNcbmltcG9ydCB7IEF3cywgQXBwLCBTdGFjaywgUmVtb3ZhbFBvbGljeSB9IGZyb20gXCJhd3MtY2RrLWxpYlwiO1xuaW1wb3J0IHsgQWxiVG9MYW1iZGEsIEFsYlRvTGFtYmRhUHJvcHMgfSBmcm9tIFwiLi4vbGliXCI7XG5pbXBvcnQgeyBnZW5lcmF0ZUludGVnU3RhY2tOYW1lIH0gZnJvbSAnQGF3cy1zb2x1dGlvbnMtY29uc3RydWN0cy9jb3JlJztcbmltcG9ydCB7IEludGVnVGVzdCB9IGZyb20gJ0Bhd3MtY2RrL2ludGVnLXRlc3RzLWFscGhhJztcbmltcG9ydCAqIGFzIGxhbWJkYSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtbGFtYmRhJztcbmltcG9ydCAqIGFzIGRlZmF1bHRzIGZyb20gJ0Bhd3Mtc29sdXRpb25zLWNvbnN0cnVjdHMvY29yZSc7XG5pbXBvcnQgeyBTZWN1cml0eUdyb3VwLCBDZm5TZWN1cml0eUdyb3VwIH0gZnJvbSBcImF3cy1jZGstbGliL2F3cy1lYzJcIjtcblxuLy8gTm90ZTogQWxsIGludGVncmF0aW9uIHRlc3RzIGZvciBhbGIgYXJlIGZvciBIVFRQIEFQSXMsIGFzIGNlcnRpZmljYXRlcyByZXF1aXJlXG4vLyB2YWxpZGF0aW9uIHRocm91Z2ggRE5TIGFuZCBlbWFpbC4gVGhpcyB2YWxpZGF0aW9uIGlzIGltcG9zc2libGUgZHVyaW5nIG91ciBpbnRlZ3JhdGlvblxuLy8gdGVzdHMgYW5kIHRoZSBzdGFjayB3aWxsIGZhaWwgdG8gbGF1bmNoIHdpdGggYW4gdW52YWxpZGF0ZWQgY2VydGlmaWNhdGUuXG5cbi8vIFNldHVwXG5jb25zdCBhcHAgPSBuZXcgQXBwKCk7XG5jb25zdCBzdGFjayA9IG5ldyBTdGFjayhhcHAsIGdlbmVyYXRlSW50ZWdTdGFja05hbWUoX19maWxlbmFtZSksIHtcbiAgZW52OiB7IGFjY291bnQ6IEF3cy5BQ0NPVU5UX0lELCByZWdpb246ICd1cy1lYXN0LTEnIH0sXG59KTtcbnN0YWNrLnRlbXBsYXRlT3B0aW9ucy5kZXNjcmlwdGlvbiA9ICdJbnRlZ3JhdGlvbiBUZXN0IGZvciBwcml2YXRlIEhUVFAgQVBJIHdpdGggYSBleGlzdGluZyBmdW5jdGlvbiBhbmQgQUxCJztcblxuY29uc3QgbXlWcGMgPSBkZWZhdWx0cy5nZXRUZXN0VnBjKHN0YWNrKTtcblxuY29uc3QgdGVzdFNnID0gbmV3IFNlY3VyaXR5R3JvdXAoc3RhY2ssICdsYW1iZGEtc2cnLCB7IHZwYzogbXlWcGMsIGFsbG93QWxsT3V0Ym91bmQ6IGZhbHNlfSk7XG5cbmNvbnN0IGxhbWJkYUZ1bmN0aW9uID0gZGVmYXVsdHMuYnVpbGRMYW1iZGFGdW5jdGlvbihzdGFjaywge1xuICBsYW1iZGFGdW5jdGlvblByb3BzOiB7XG4gICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KGAke19fZGlybmFtZX0vbGFtYmRhYCksXG4gICAgcnVudGltZTogZGVmYXVsdHMuQ09NTUVSQ0lBTF9SRUdJT05fTEFNQkRBX05PREVfUlVOVElNRSxcbiAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcicsXG4gICAgdnBjOiBteVZwYyxcbiAgICBzZWN1cml0eUdyb3VwczogWyB0ZXN0U2cgXVxuICB9XG59KTtcblxuY29uc3QgbG9hZEJhbGFuY2VyID0gZGVmYXVsdHMuT2J0YWluQWxiKHN0YWNrLCAnZXhpc3RpbmctYWxiJywge1xuICB2cGM6IG15VnBjLFxuICBwdWJsaWNBcGk6IGZhbHNlLFxuICBsb2FkQmFsYW5jZXJQcm9wczoge1xuICAgIGludGVybmV0RmFjaW5nOiBmYWxzZSxcbiAgICB2cGM6IG15VnBjLFxuICB9LFxuICBsb2dBY2Nlc3NMb2dzOiB0cnVlLFxuICBsb2dnaW5nQnVja2V0UHJvcHM6IHtcbiAgICByZW1vdmFsUG9saWN5OiBSZW1vdmFsUG9saWN5LkRFU1RST1ksXG4gICAgYXV0b0RlbGV0ZU9iamVjdHM6IHRydWVcbiAgfVxufSk7XG5cbmNvbnN0IHByb3BzOiBBbGJUb0xhbWJkYVByb3BzID0ge1xuICBleGlzdGluZ0xhbWJkYU9iajogbGFtYmRhRnVuY3Rpb24sXG4gIGV4aXN0aW5nTG9hZEJhbGFuY2VyT2JqOiBsb2FkQmFsYW5jZXIsXG4gIGV4aXN0aW5nVnBjOiBteVZwYyxcbiAgbGlzdGVuZXJQcm9wczoge1xuICAgIHByb3RvY29sOiAnSFRUUCdcbiAgfSxcbiAgcHVibGljQXBpOiBmYWxzZVxufTtcblxuY29uc3QgYWxiVG9MYW1iZGEgPSBuZXcgQWxiVG9MYW1iZGEoc3RhY2ssICd0ZXN0LW9uZScsIHByb3BzKTtcblxuZGVmYXVsdHMuYWRkQ2ZuU3VwcHJlc3NSdWxlcyhhbGJUb0xhbWJkYS5saXN0ZW5lciwgW1xuICB7IGlkOiAnVzU2JywgcmVhc29uOiAnQWxsIGludGVncmF0aW9uIHRlc3RzIG11c3QgYmUgSFRUUCBiZWNhdXNlIG9mIGNlcnRpZmljYXRlIGxpbWl0YXRpb25zLicgfSxcbl0pO1xuXG5jb25zdCBuZXdTZWN1cml0eUdyb3VwID0gYWxiVG9MYW1iZGEubG9hZEJhbGFuY2VyLmNvbm5lY3Rpb25zLnNlY3VyaXR5R3JvdXBzWzBdLm5vZGUuZGVmYXVsdENoaWxkIGFzIENmblNlY3VyaXR5R3JvdXA7XG5kZWZhdWx0cy5hZGRDZm5TdXBwcmVzc1J1bGVzKG5ld1NlY3VyaXR5R3JvdXAsIFtcbiAgeyBpZDogJ1cyOScsIHJlYXNvbjogJ0NESyBjcmVhdGVkIHJ1bGUgdGhhdCBibG9ja3MgYWxsIHRyYWZmaWMuJ30sXG4gIHsgaWQ6ICdXMicsIHJlYXNvbjogJ1J1bGUgZG9lcyBub3QgYXBwbHkgZm9yIEVMQi4nfSxcbiAgeyBpZDogJ1c5JywgcmVhc29uOiAnUnVsZSBkb2VzIG5vdCBhcHBseSBmb3IgRUxCLid9XG5dKTtcblxuZGVmYXVsdHMuYWRkQ2ZuU3VwcHJlc3NSdWxlcyh0ZXN0U2csIFtcbiAgeyBpZDogJ1cyOScsIHJlYXNvbjogJ0NESyBjcmVhdGVkIHJ1bGUgdGhhdCBibG9ja3MgYWxsIHRyYWZmaWMuJ30sXG5dKTtcblxuZGVmYXVsdHMuc3VwcHJlc3NDdXN0b21IYW5kbGVyQ2ZuTmFnV2FybmluZ3Moc3RhY2ssICdDdXN0b206OlMzQXV0b0RlbGV0ZU9iamVjdHNDdXN0b21SZXNvdXJjZVByb3ZpZGVyJyk7XG5kZWZhdWx0cy5zdXBwcmVzc0N1c3RvbUhhbmRsZXJDZm5OYWdXYXJuaW5ncyhzdGFjaywgJ0N1c3RvbTo6VnBjUmVzdHJpY3REZWZhdWx0U0dDdXN0b21SZXNvdXJjZVByb3ZpZGVyJyk7XG5cbi8vIFN5bnRoXG5uZXcgSW50ZWdUZXN0KHN0YWNrLCAnSW50ZWcnLCB7IHRlc3RDYXNlczogW1xuICBzdGFja1xuXSB9KTtcbiJdfQ==