"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const core_1 = require("@aws-solutions-constructs/core");
const integ_tests_alpha_1 = require("@aws-cdk/integ-tests-alpha");
const lambda = require("aws-cdk-lib/aws-lambda");
const defaults = require("@aws-solutions-constructs/core");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
// Note: All integration tests for alb are for HTTP APIs, as certificates require
// validation through DNS and email. This validation is impossible during our integration
// tests and the stack will fail to launch with an unvalidated certificate.
// Setup
const app = new aws_cdk_lib_1.App();
const stack = new aws_cdk_lib_1.Stack(app, core_1.generateIntegStackName(__filename), {
    env: { account: aws_cdk_lib_1.Aws.ACCOUNT_ID, region: "us-east-1" },
});
stack.templateOptions.description =
    "Integration Test for public HTTP API with a existing function and ALB";
const myVpc = defaults.getTestVpc(stack);
const testSg = new aws_ec2_1.SecurityGroup(stack, "lambda-sg", {
    vpc: myVpc,
    allowAllOutbound: false,
});
const lambdaFunction = defaults.buildLambdaFunction(stack, {
    lambdaFunctionProps: {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: "index.handler",
        vpc: myVpc,
        securityGroups: [testSg],
    },
});
const loadBalancer = defaults.ObtainAlb(stack, "existing-alb", {
    vpc: myVpc,
    publicApi: false,
    loadBalancerProps: {
        internetFacing: false,
        vpc: myVpc,
    },
    logAccessLogs: true,
    loggingBucketProps: {
        removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        autoDeleteObjects: true,
    }
});
const props = {
    existingLambdaObj: lambdaFunction,
    existingLoadBalancerObj: loadBalancer,
    existingVpc: myVpc,
    listenerProps: {
        protocol: "HTTP",
    },
    publicApi: true,
};
const albToLambda = new lib_1.AlbToLambda(stack, "test-one", props);
defaults.addCfnSuppressRules(albToLambda.listener, [
    {
        id: "W56",
        reason: "All integration tests must be HTTP because of certificate limitations.",
    },
]);
const newSecurityGroup = albToLambda.loadBalancer.connections.securityGroups[0]
    .node.defaultChild;
defaults.addCfnSuppressRules(newSecurityGroup, [
    { id: "W29", reason: "CDK created rule that blocks all traffic." },
    { id: "W2", reason: "Rule does not apply for ELB." },
    { id: "W9", reason: "Rule does not apply for ELB." },
]);
defaults.addCfnSuppressRules(testSg, [
    { id: "W29", reason: "CDK created rule that blocks all traffic." },
]);
defaults.suppressCustomHandlerCfnNagWarnings(stack, 'Custom::S3AutoDeleteObjectsCustomResourceProvider');
defaults.suppressCustomHandlerCfnNagWarnings(stack, 'Custom::VpcRestrictDefaultSGCustomResourceProvider');
// Synth
new integ_tests_alpha_1.IntegTest(stack, 'Integ', { testCases: [
        stack
    ] });
//# sourceMappingURL=data:application/json;base64,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