"use strict";
const fs = require("fs");
const os = require("os");
const path = require("path");
const libfs = require("../../lib/fs");
module.exports = {
    files: {
        'does not change with the file name'(test) {
            // GIVEN
            const workdir = fs.mkdtempSync(path.join(os.tmpdir(), 'hash-tests'));
            const content = 'Hello, world!';
            const input1 = path.join(workdir, 'input1.txt');
            const input2 = path.join(workdir, 'input2.txt');
            const input3 = path.join(workdir, 'input3.txt');
            fs.writeFileSync(input1, content);
            fs.writeFileSync(input2, content);
            fs.writeFileSync(input3, content + '.'); // add one character, hash should be different
            // WHEN
            const hash1 = libfs.fingerprint(input1);
            const hash2 = libfs.fingerprint(input2);
            const hash3 = libfs.fingerprint(input3);
            // THEN
            test.deepEqual(hash1, hash2);
            test.notDeepEqual(hash3, hash1);
            test.done();
        },
        'works on empty files'(test) {
            // GIVEN
            const workdir = fs.mkdtempSync(path.join(os.tmpdir(), 'hash-tests'));
            const input1 = path.join(workdir, 'empty');
            const input2 = path.join(workdir, 'empty');
            fs.writeFileSync(input1, '');
            fs.writeFileSync(input2, '');
            // WHEN
            const hash1 = libfs.fingerprint(input1);
            const hash2 = libfs.fingerprint(input2);
            // THEN
            test.deepEqual(hash1, hash2);
            test.done();
        },
    },
    directories: {
        'works on directories'(test) {
            // GIVEN
            const srcdir = path.join(__dirname, 'fixtures', 'symlinks');
            const outdir = fs.mkdtempSync(path.join(os.tmpdir(), 'copy-tests'));
            libfs.copyDirectory(srcdir, outdir);
            // WHEN
            const hashSrc = libfs.fingerprint(srcdir);
            const hashCopy = libfs.fingerprint(outdir);
            // THEN
            test.deepEqual(hashSrc, hashCopy);
            test.done();
        },
        'ignores requested files'(test) {
            // GIVEN
            const srcdir = path.join(__dirname, 'fixtures', 'symlinks');
            const outdir = fs.mkdtempSync(path.join(os.tmpdir(), 'copy-tests'));
            libfs.copyDirectory(srcdir, outdir);
            // WHEN
            const hashSrc = libfs.fingerprint(srcdir);
            fs.writeFileSync(path.join(outdir, `${hashSrc}.ignoreme`), 'Ignore me!');
            const hashCopy = libfs.fingerprint(outdir, { exclude: ['*.ignoreme'] });
            // THEN
            test.deepEqual(hashSrc, hashCopy);
            test.done();
        },
        'changes with file names'(test) {
            // GIVEN
            const srcdir = path.join(__dirname, 'fixtures', 'symlinks');
            const cpydir = fs.mkdtempSync(path.join(os.tmpdir(), 'fingerprint-tests'));
            libfs.copyDirectory(srcdir, cpydir);
            // be careful not to break a symlink
            fs.renameSync(path.join(cpydir, 'normal-dir', 'file-in-subdir.txt'), path.join(cpydir, 'move-me.txt'));
            // WHEN
            const hashSrc = libfs.fingerprint(srcdir);
            const hashCopy = libfs.fingerprint(cpydir);
            // THEN
            test.notDeepEqual(hashSrc, hashCopy);
            test.done();
        },
    },
    symlinks: {
        'changes with the contents of followed symlink referent'(test) {
            // GIVEN
            const dir1 = fs.mkdtempSync(path.join(os.tmpdir(), 'fingerprint-tests'));
            const dir2 = fs.mkdtempSync(path.join(os.tmpdir(), 'fingerprint-tests'));
            const target = path.join(dir1, 'boom.txt');
            const content = 'boom';
            fs.writeFileSync(target, content);
            fs.symlinkSync(target, path.join(dir2, 'link-to-boom.txt'));
            // now dir2 contains a symlink to a file in dir1
            // WHEN
            const original = libfs.fingerprint(dir2);
            // now change the contents of the target
            fs.writeFileSync(target, 'changning you!');
            const afterChange = libfs.fingerprint(dir2);
            // revert the content to original and expect hash to be reverted
            fs.writeFileSync(target, content);
            const afterRevert = libfs.fingerprint(dir2);
            // THEN
            test.notDeepEqual(original, afterChange);
            test.deepEqual(afterRevert, original);
            test.done();
        },
        'does not change with the contents of un-followed symlink referent'(test) {
            // GIVEN
            const dir1 = fs.mkdtempSync(path.join(os.tmpdir(), 'fingerprint-tests'));
            const dir2 = fs.mkdtempSync(path.join(os.tmpdir(), 'fingerprint-tests'));
            const target = path.join(dir1, 'boom.txt');
            const content = 'boom';
            fs.writeFileSync(target, content);
            fs.symlinkSync(target, path.join(dir2, 'link-to-boom.txt'));
            // now dir2 contains a symlink to a file in dir1
            // WHEN
            const original = libfs.fingerprint(dir2, { follow: libfs.FollowMode.NEVER });
            // now change the contents of the target
            fs.writeFileSync(target, 'changning you!');
            const afterChange = libfs.fingerprint(dir2, { follow: libfs.FollowMode.NEVER });
            // revert the content to original and expect hash to be reverted
            fs.writeFileSync(target, content);
            const afterRevert = libfs.fingerprint(dir2, { follow: libfs.FollowMode.NEVER });
            // THEN
            test.deepEqual(original, afterChange);
            test.deepEqual(afterRevert, original);
            test.done();
        }
    }
};
//# sourceMappingURL=data:application/json;base64,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