"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
/**
 * A PolicyDocument is a collection of statements
 */
class PolicyDocument {
    constructor(props = {}) {
        this.statements = new Array();
        this.creationStack = cdk.captureStackTrace();
        this.autoAssignSids = !!props.assignSids;
        this.addStatements(...props.statements || []);
    }
    resolve(context) {
        context.registerPostProcessor(new RemoveDuplicateStatements(this.autoAssignSids));
        return this.render();
    }
    get isEmpty() {
        return this.statements.length === 0;
    }
    /**
     * The number of statements already added to this policy.
     * Can be used, for example, to generate uniuqe "sid"s within the policy.
     */
    get statementCount() {
        return this.statements.length;
    }
    /**
     * Adds a statement to the policy document.
     *
     * @param statement the statement to add.
     */
    addStatements(...statement) {
        this.statements.push(...statement);
    }
    /**
     * Encode the policy document as a string
     */
    toString() {
        return cdk.Token.asString(this, {
            displayHint: 'PolicyDocument'
        });
    }
    /**
     * JSON-ify the document
     *
     * Used when JSON.stringify() is called
     */
    toJSON() {
        return this.render();
    }
    render() {
        if (this.isEmpty) {
            return undefined;
        }
        const doc = {
            Statement: this.statements.map(s => s.toStatementJson()),
            Version: '2012-10-17'
        };
        return doc;
    }
}
exports.PolicyDocument = PolicyDocument;
/**
 * Removes duplicate statements and assign Sids if necessary
 */
class RemoveDuplicateStatements {
    constructor(autoAssignSids) {
        this.autoAssignSids = autoAssignSids;
    }
    postProcess(input, _context) {
        if (!input || !input.Statement) {
            return input;
        }
        const jsonStatements = new Set();
        const uniqueStatements = [];
        for (const statement of input.Statement) {
            const jsonStatement = JSON.stringify(statement);
            if (!jsonStatements.has(jsonStatement)) {
                uniqueStatements.push(statement);
                jsonStatements.add(jsonStatement);
            }
        }
        // assign unique SIDs (the statement index) if `autoAssignSids` is enabled
        const statements = uniqueStatements.map((s, i) => {
            if (this.autoAssignSids && !s.Sid) {
                s.Sid = i.toString();
            }
            return s;
        });
        return {
            ...input,
            Statement: statements
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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