from abc import abstractmethod

import numpy as np

from pymoo.model.population import Population


class Crossover:
    """
    The crossover combines parents to offsprings. Some crossover are problem specific and use additional information.
    This class must be inherited from to provide a crossover method to an algorithm.
    """

    def __init__(self, n_parents, n_offsprings, vectorized):
        self.n_parents = n_parents
        self.n_offsprings = n_offsprings
        self.vectorized = vectorized

    def do(self, problem, pop, parents, **kwargs):
        """

        This method executes the crossover on the parents. This class wraps the implementation of the class
        that implements the crossover.

        Parameters
        ----------
        problem: class
            The problem to be solved. Provides information such as lower and upper bounds or feasibility
            conditions for custom crossovers.

        pop : Population
            The population as an object

        parents: numpy.array
            The select parents of the population for the crossover

        kwargs : dict
            Any additional data that might be necessary to perform the crossover. E.g. constants of an algorithm.

        Returns
        -------
        offsprings : Population
            The off as a matrix. n_children rows and the number of columns is equal to the variable
            length of the problem.

        """

        if self.n_parents != parents.shape[1]:
            raise ValueError('Exception during crossover: Number of parents differs from defined at crossover.')

        off = self._do(problem, pop, parents, **kwargs)
        return off

    @abstractmethod
    def _do(self, problem, parents, children, **kwargs):
        n_children = 0
        for k in range(parents.shape[1]):

            _children = np.full((self.n_offsprings, problem.n_var), np.inf)

            self._mating(problem, parents[:, k, :], _children, **kwargs)

            for i in range(_children.shape[0]):
                children[n_children + i, :] = _children[i, :]

            n_children += self.n_offsprings

    def _mating(self, problem, parents, children, **kwargs):
        pass
