# Copyright 2021-2024 NVIDIA Corporation.  All rights reserved.
#
# Please refer to the NVIDIA end user license agreement (EULA) associated
# with this source code for terms and conditions that govern your use of
# this software. Any use, reproduction, disclosure, or distribution of
# this software and related documentation outside the terms of the EULA
# is strictly prohibited.
from typing import List, Tuple, Any, Optional
from enum import IntEnum
import cython
import ctypes
from libc.stdlib cimport calloc, free
from libc cimport string
from libc.stdint cimport int32_t, uint32_t, int64_t, uint64_t
from libc.stddef cimport wchar_t
from libc.limits cimport CHAR_MIN
from libcpp.vector cimport vector
from cpython.buffer cimport PyObject_CheckBuffer, PyObject_GetBuffer, PyBuffer_Release, PyBUF_SIMPLE, PyBUF_ANY_CONTIGUOUS
from cpython.bytes cimport PyBytes_FromStringAndSize

ctypedef unsigned long long signed_char_ptr
ctypedef unsigned long long unsigned_char_ptr
ctypedef unsigned long long char_ptr
ctypedef unsigned long long short_ptr
ctypedef unsigned long long unsigned_short_ptr
ctypedef unsigned long long int_ptr
ctypedef unsigned long long long_int_ptr
ctypedef unsigned long long long_long_int_ptr
ctypedef unsigned long long unsigned_int_ptr
ctypedef unsigned long long unsigned_long_int_ptr
ctypedef unsigned long long unsigned_long_long_int_ptr
ctypedef unsigned long long uint32_t_ptr
ctypedef unsigned long long uint64_t_ptr
ctypedef unsigned long long int32_t_ptr
ctypedef unsigned long long int64_t_ptr
ctypedef unsigned long long unsigned_ptr
ctypedef unsigned long long unsigned_long_long_ptr
ctypedef unsigned long long long_long_ptr
ctypedef unsigned long long size_t_ptr
ctypedef unsigned long long float_ptr
ctypedef unsigned long long double_ptr
ctypedef unsigned long long void_ptr

#: CUDA API version number
CUDA_VERSION = cydriver.CUDA_VERSION

#: CUDA IPC handle size
CU_IPC_HANDLE_SIZE = cydriver.CU_IPC_HANDLE_SIZE

#: Legacy stream handle
#:
#: Stream handle that can be passed as a CUstream to use an implicit stream
#: with legacy synchronization behavior.
#:
#: See details of the \link_sync_behavior
CU_STREAM_LEGACY = cydriver.CU_STREAM_LEGACY

#: Per-thread stream handle
#:
#: Stream handle that can be passed as a CUstream to use an implicit stream
#: with per-thread synchronization behavior.
#:
#: See details of the \link_sync_behavior
CU_STREAM_PER_THREAD = cydriver.CU_STREAM_PER_THREAD

CU_KERNEL_NODE_ATTRIBUTE_ACCESS_POLICY_WINDOW = cydriver.CU_KERNEL_NODE_ATTRIBUTE_ACCESS_POLICY_WINDOW

CU_KERNEL_NODE_ATTRIBUTE_COOPERATIVE = cydriver.CU_KERNEL_NODE_ATTRIBUTE_COOPERATIVE

CU_KERNEL_NODE_ATTRIBUTE_CLUSTER_DIMENSION = cydriver.CU_KERNEL_NODE_ATTRIBUTE_CLUSTER_DIMENSION

CU_KERNEL_NODE_ATTRIBUTE_CLUSTER_SCHEDULING_POLICY_PREFERENCE = cydriver.CU_KERNEL_NODE_ATTRIBUTE_CLUSTER_SCHEDULING_POLICY_PREFERENCE

CU_KERNEL_NODE_ATTRIBUTE_PRIORITY = cydriver.CU_KERNEL_NODE_ATTRIBUTE_PRIORITY

CU_STREAM_ATTRIBUTE_ACCESS_POLICY_WINDOW = cydriver.CU_STREAM_ATTRIBUTE_ACCESS_POLICY_WINDOW

CU_STREAM_ATTRIBUTE_SYNCHRONIZATION_POLICY = cydriver.CU_STREAM_ATTRIBUTE_SYNCHRONIZATION_POLICY

#: If set, host memory is portable between CUDA contexts. Flag for
#: :py:obj:`~.cuMemHostAlloc()`
CU_MEMHOSTALLOC_PORTABLE = cydriver.CU_MEMHOSTALLOC_PORTABLE

#: If set, host memory is mapped into CUDA address space and
#: :py:obj:`~.cuMemHostGetDevicePointer()` may be called on the host
#: pointer. Flag for :py:obj:`~.cuMemHostAlloc()`
CU_MEMHOSTALLOC_DEVICEMAP = cydriver.CU_MEMHOSTALLOC_DEVICEMAP

#: If set, host memory is allocated as write-combined - fast to write,
#: faster to DMA, slow to read except via SSE4 streaming load instruction
#: (MOVNTDQA). Flag for :py:obj:`~.cuMemHostAlloc()`
CU_MEMHOSTALLOC_WRITECOMBINED = cydriver.CU_MEMHOSTALLOC_WRITECOMBINED

#: If set, host memory is portable between CUDA contexts. Flag for
#: :py:obj:`~.cuMemHostRegister()`
CU_MEMHOSTREGISTER_PORTABLE = cydriver.CU_MEMHOSTREGISTER_PORTABLE

#: If set, host memory is mapped into CUDA address space and
#: :py:obj:`~.cuMemHostGetDevicePointer()` may be called on the host
#: pointer. Flag for :py:obj:`~.cuMemHostRegister()`
CU_MEMHOSTREGISTER_DEVICEMAP = cydriver.CU_MEMHOSTREGISTER_DEVICEMAP

#: If set, the passed memory pointer is treated as pointing to some memory-
#: mapped I/O space, e.g. belonging to a third-party PCIe device. On
#: Windows the flag is a no-op. On Linux that memory is marked as non
#: cache-coherent for the GPU and is expected to be physically contiguous.
#: It may return :py:obj:`~.CUDA_ERROR_NOT_PERMITTED` if run as an
#: unprivileged user, :py:obj:`~.CUDA_ERROR_NOT_SUPPORTED` on older Linux
#: kernel versions. On all other platforms, it is not supported and
#: :py:obj:`~.CUDA_ERROR_NOT_SUPPORTED` is returned. Flag for
#: :py:obj:`~.cuMemHostRegister()`
CU_MEMHOSTREGISTER_IOMEMORY = cydriver.CU_MEMHOSTREGISTER_IOMEMORY

#: If set, the passed memory pointer is treated as pointing to memory that
#: is considered read-only by the device. On platforms without
#: :py:obj:`~.CU_DEVICE_ATTRIBUTE_PAGEABLE_MEMORY_ACCESS_USES_HOST_PAGE_TABLES`,
#: this flag is required in order to register memory mapped to the CPU as
#: read-only. Support for the use of this flag can be queried from the
#: device attribute
#: :py:obj:`~.CU_DEVICE_ATTRIBUTE_READ_ONLY_HOST_REGISTER_SUPPORTED`. Using
#: this flag with a current context associated with a device that does not
#: have this attribute set will cause :py:obj:`~.cuMemHostRegister` to
#: error with :py:obj:`~.CUDA_ERROR_NOT_SUPPORTED`.
CU_MEMHOSTREGISTER_READ_ONLY = cydriver.CU_MEMHOSTREGISTER_READ_ONLY

#: Indicates that the layered sparse CUDA array or CUDA mipmapped array has
#: a single mip tail region for all layers
CU_ARRAY_SPARSE_PROPERTIES_SINGLE_MIPTAIL = cydriver.CU_ARRAY_SPARSE_PROPERTIES_SINGLE_MIPTAIL

#: Indicates that the external memory object is a dedicated resource
CUDA_EXTERNAL_MEMORY_DEDICATED = cydriver.CUDA_EXTERNAL_MEMORY_DEDICATED

#: When the `flags` parameter of
#: :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS` contains this flag, it
#: indicates that signaling an external semaphore object should skip
#: performing appropriate memory synchronization operations over all the
#: external memory objects that are imported as
#: :py:obj:`~.CU_EXTERNAL_MEMORY_HANDLE_TYPE_NVSCIBUF`, which otherwise are
#: performed by default to ensure data coherency with other importers of
#: the same NvSciBuf memory objects.
CUDA_EXTERNAL_SEMAPHORE_SIGNAL_SKIP_NVSCIBUF_MEMSYNC = cydriver.CUDA_EXTERNAL_SEMAPHORE_SIGNAL_SKIP_NVSCIBUF_MEMSYNC

#: When the `flags` parameter of
#: :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_WAIT_PARAMS` contains this flag, it
#: indicates that waiting on an external semaphore object should skip
#: performing appropriate memory synchronization operations over all the
#: external memory objects that are imported as
#: :py:obj:`~.CU_EXTERNAL_MEMORY_HANDLE_TYPE_NVSCIBUF`, which otherwise are
#: performed by default to ensure data coherency with other importers of
#: the same NvSciBuf memory objects.
CUDA_EXTERNAL_SEMAPHORE_WAIT_SKIP_NVSCIBUF_MEMSYNC = cydriver.CUDA_EXTERNAL_SEMAPHORE_WAIT_SKIP_NVSCIBUF_MEMSYNC

#: When `flags` of :py:obj:`~.cuDeviceGetNvSciSyncAttributes` is set to
#: this, it indicates that application needs signaler specific
#: NvSciSyncAttr to be filled by
#: :py:obj:`~.cuDeviceGetNvSciSyncAttributes`.
CUDA_NVSCISYNC_ATTR_SIGNAL = cydriver.CUDA_NVSCISYNC_ATTR_SIGNAL

#: When `flags` of :py:obj:`~.cuDeviceGetNvSciSyncAttributes` is set to
#: this, it indicates that application needs waiter specific NvSciSyncAttr
#: to be filled by :py:obj:`~.cuDeviceGetNvSciSyncAttributes`.
CUDA_NVSCISYNC_ATTR_WAIT = cydriver.CUDA_NVSCISYNC_ATTR_WAIT

#: This flag if set indicates that the memory will be used as a tile pool.
CU_MEM_CREATE_USAGE_TILE_POOL = cydriver.CU_MEM_CREATE_USAGE_TILE_POOL

#: If set, each kernel launched as part of
#: :py:obj:`~.cuLaunchCooperativeKernelMultiDevice` only waits for prior
#: work in the stream corresponding to that GPU to complete before the
#: kernel begins execution.
CUDA_COOPERATIVE_LAUNCH_MULTI_DEVICE_NO_PRE_LAUNCH_SYNC = cydriver.CUDA_COOPERATIVE_LAUNCH_MULTI_DEVICE_NO_PRE_LAUNCH_SYNC

#: If set, any subsequent work pushed in a stream that participated in a
#: call to :py:obj:`~.cuLaunchCooperativeKernelMultiDevice` will only wait
#: for the kernel launched on the GPU corresponding to that stream to
#: complete before it begins execution.
CUDA_COOPERATIVE_LAUNCH_MULTI_DEVICE_NO_POST_LAUNCH_SYNC = cydriver.CUDA_COOPERATIVE_LAUNCH_MULTI_DEVICE_NO_POST_LAUNCH_SYNC

#: If set, the CUDA array is a collection of layers, where each layer is
#: either a 1D or a 2D array and the Depth member of
#: CUDA_ARRAY3D_DESCRIPTOR specifies the number of layers, not the depth of
#: a 3D array.
CUDA_ARRAY3D_LAYERED = cydriver.CUDA_ARRAY3D_LAYERED

#: Deprecated, use CUDA_ARRAY3D_LAYERED
CUDA_ARRAY3D_2DARRAY = cydriver.CUDA_ARRAY3D_2DARRAY

#: This flag must be set in order to bind a surface reference to the CUDA
#: array
CUDA_ARRAY3D_SURFACE_LDST = cydriver.CUDA_ARRAY3D_SURFACE_LDST

#: If set, the CUDA array is a collection of six 2D arrays, representing
#: faces of a cube. The width of such a CUDA array must be equal to its
#: height, and Depth must be six. If :py:obj:`~.CUDA_ARRAY3D_LAYERED` flag
#: is also set, then the CUDA array is a collection of cubemaps and Depth
#: must be a multiple of six.
CUDA_ARRAY3D_CUBEMAP = cydriver.CUDA_ARRAY3D_CUBEMAP

#: This flag must be set in order to perform texture gather operations on a
#: CUDA array.
CUDA_ARRAY3D_TEXTURE_GATHER = cydriver.CUDA_ARRAY3D_TEXTURE_GATHER

#: This flag if set indicates that the CUDA array is a DEPTH_TEXTURE.
CUDA_ARRAY3D_DEPTH_TEXTURE = cydriver.CUDA_ARRAY3D_DEPTH_TEXTURE

#: This flag indicates that the CUDA array may be bound as a color target
#: in an external graphics API
CUDA_ARRAY3D_COLOR_ATTACHMENT = cydriver.CUDA_ARRAY3D_COLOR_ATTACHMENT

#: This flag if set indicates that the CUDA array or CUDA mipmapped array
#: is a sparse CUDA array or CUDA mipmapped array respectively
CUDA_ARRAY3D_SPARSE = cydriver.CUDA_ARRAY3D_SPARSE

#: This flag if set indicates that the CUDA array or CUDA mipmapped array
#: will allow deferred memory mapping
CUDA_ARRAY3D_DEFERRED_MAPPING = cydriver.CUDA_ARRAY3D_DEFERRED_MAPPING

#: Override the texref format with a format inferred from the array. Flag
#: for :py:obj:`~.cuTexRefSetArray()`
CU_TRSA_OVERRIDE_FORMAT = cydriver.CU_TRSA_OVERRIDE_FORMAT

#: Read the texture as integers rather than promoting the values to floats
#: in the range [0,1]. Flag for :py:obj:`~.cuTexRefSetFlags()` and
#: :py:obj:`~.cuTexObjectCreate()`
CU_TRSF_READ_AS_INTEGER = cydriver.CU_TRSF_READ_AS_INTEGER

#: Use normalized texture coordinates in the range [0,1) instead of
#: [0,dim). Flag for :py:obj:`~.cuTexRefSetFlags()` and
#: :py:obj:`~.cuTexObjectCreate()`
CU_TRSF_NORMALIZED_COORDINATES = cydriver.CU_TRSF_NORMALIZED_COORDINATES

#: Perform sRGB->linear conversion during texture read. Flag for
#: :py:obj:`~.cuTexRefSetFlags()` and :py:obj:`~.cuTexObjectCreate()`
CU_TRSF_SRGB = cydriver.CU_TRSF_SRGB

#: Disable any trilinear filtering optimizations. Flag for
#: :py:obj:`~.cuTexRefSetFlags()` and :py:obj:`~.cuTexObjectCreate()`
CU_TRSF_DISABLE_TRILINEAR_OPTIMIZATION = cydriver.CU_TRSF_DISABLE_TRILINEAR_OPTIMIZATION

#: Enable seamless cube map filtering. Flag for
#: :py:obj:`~.cuTexObjectCreate()`
CU_TRSF_SEAMLESS_CUBEMAP = cydriver.CU_TRSF_SEAMLESS_CUBEMAP

#: C++ compile time constant for CU_LAUNCH_PARAM_END
CU_LAUNCH_PARAM_END_AS_INT = cydriver.CU_LAUNCH_PARAM_END_AS_INT

#: End of array terminator for the `extra` parameter to
#: :py:obj:`~.cuLaunchKernel`
CU_LAUNCH_PARAM_END = cydriver.CU_LAUNCH_PARAM_END

#: C++ compile time constant for CU_LAUNCH_PARAM_BUFFER_POINTER
CU_LAUNCH_PARAM_BUFFER_POINTER_AS_INT = cydriver.CU_LAUNCH_PARAM_BUFFER_POINTER_AS_INT

#: Indicator that the next value in the `extra` parameter to
#: :py:obj:`~.cuLaunchKernel` will be a pointer to a buffer containing all
#: kernel parameters used for launching kernel `f`. This buffer needs to
#: honor all alignment/padding requirements of the individual parameters.
#: If :py:obj:`~.CU_LAUNCH_PARAM_BUFFER_SIZE` is not also specified in the
#: `extra` array, then :py:obj:`~.CU_LAUNCH_PARAM_BUFFER_POINTER` will have
#: no effect.
CU_LAUNCH_PARAM_BUFFER_POINTER = cydriver.CU_LAUNCH_PARAM_BUFFER_POINTER

#: C++ compile time constant for CU_LAUNCH_PARAM_BUFFER_SIZE
CU_LAUNCH_PARAM_BUFFER_SIZE_AS_INT = cydriver.CU_LAUNCH_PARAM_BUFFER_SIZE_AS_INT

#: Indicator that the next value in the `extra` parameter to
#: :py:obj:`~.cuLaunchKernel` will be a pointer to a size_t which contains
#: the size of the buffer specified with
#: :py:obj:`~.CU_LAUNCH_PARAM_BUFFER_POINTER`. It is required that
#: :py:obj:`~.CU_LAUNCH_PARAM_BUFFER_POINTER` also be specified in the
#: `extra` array if the value associated with
#: :py:obj:`~.CU_LAUNCH_PARAM_BUFFER_SIZE` is not zero.
CU_LAUNCH_PARAM_BUFFER_SIZE = cydriver.CU_LAUNCH_PARAM_BUFFER_SIZE

#: For texture references loaded into the module, use default texunit from
#: texture reference.
CU_PARAM_TR_DEFAULT = cydriver.CU_PARAM_TR_DEFAULT

#: Device that represents the CPU
CU_DEVICE_CPU = cydriver.CU_DEVICE_CPU

#: Device that represents an invalid device
CU_DEVICE_INVALID = cydriver.CU_DEVICE_INVALID

#: Maximum number of planes per frame
MAX_PLANES = cydriver.MAX_PLANES

#: Indicates that timeout for :py:obj:`~.cuEGLStreamConsumerAcquireFrame`
#: is infinite.
CUDA_EGL_INFINITE_TIMEOUT = cydriver.CUDA_EGL_INFINITE_TIMEOUT

class CUipcMem_flags(IntEnum):
    """
    CUDA Ipc Mem Flags
    """

    #: Automatically enable peer access between remote devices as needed
    CU_IPC_MEM_LAZY_ENABLE_PEER_ACCESS = cydriver.CUipcMem_flags_enum.CU_IPC_MEM_LAZY_ENABLE_PEER_ACCESS

class CUmemAttach_flags(IntEnum):
    """
    CUDA Mem Attach Flags
    """

    #: Memory can be accessed by any stream on any device
    CU_MEM_ATTACH_GLOBAL = cydriver.CUmemAttach_flags_enum.CU_MEM_ATTACH_GLOBAL

    #: Memory cannot be accessed by any stream on any device
    CU_MEM_ATTACH_HOST = cydriver.CUmemAttach_flags_enum.CU_MEM_ATTACH_HOST

    #: Memory can only be accessed by a single stream on the associated
    #: device
    CU_MEM_ATTACH_SINGLE = cydriver.CUmemAttach_flags_enum.CU_MEM_ATTACH_SINGLE

class CUctx_flags(IntEnum):
    """
    Context creation flags
    """

    #: Automatic scheduling
    CU_CTX_SCHED_AUTO = cydriver.CUctx_flags_enum.CU_CTX_SCHED_AUTO

    #: Set spin as default scheduling
    CU_CTX_SCHED_SPIN = cydriver.CUctx_flags_enum.CU_CTX_SCHED_SPIN

    #: Set yield as default scheduling
    CU_CTX_SCHED_YIELD = cydriver.CUctx_flags_enum.CU_CTX_SCHED_YIELD

    #: Set blocking synchronization as default scheduling
    CU_CTX_SCHED_BLOCKING_SYNC = cydriver.CUctx_flags_enum.CU_CTX_SCHED_BLOCKING_SYNC

    #: Set blocking synchronization as default scheduling [Deprecated]
    CU_CTX_BLOCKING_SYNC = cydriver.CUctx_flags_enum.CU_CTX_BLOCKING_SYNC
    CU_CTX_SCHED_MASK = cydriver.CUctx_flags_enum.CU_CTX_SCHED_MASK

    #: [Deprecated]
    CU_CTX_MAP_HOST = cydriver.CUctx_flags_enum.CU_CTX_MAP_HOST

    #: Keep local memory allocation after launch
    CU_CTX_LMEM_RESIZE_TO_MAX = cydriver.CUctx_flags_enum.CU_CTX_LMEM_RESIZE_TO_MAX
    CU_CTX_FLAGS_MASK = cydriver.CUctx_flags_enum.CU_CTX_FLAGS_MASK

class CUevent_sched_flags(IntEnum):
    """
    Event sched flags
    """

    #: Automatic scheduling
    CU_EVENT_SCHED_AUTO = cydriver.CUevent_sched_flags_enum.CU_EVENT_SCHED_AUTO

    #: Set spin as default scheduling
    CU_EVENT_SCHED_SPIN = cydriver.CUevent_sched_flags_enum.CU_EVENT_SCHED_SPIN

    #: Set yield as default scheduling
    CU_EVENT_SCHED_YIELD = cydriver.CUevent_sched_flags_enum.CU_EVENT_SCHED_YIELD

    #: Set blocking synchronization as default scheduling
    CU_EVENT_SCHED_BLOCKING_SYNC = cydriver.CUevent_sched_flags_enum.CU_EVENT_SCHED_BLOCKING_SYNC

class cl_event_flags(IntEnum):
    """
    NVCL event scheduling flags
    """

    #: Automatic scheduling
    NVCL_EVENT_SCHED_AUTO = cydriver.cl_event_flags_enum.NVCL_EVENT_SCHED_AUTO

    #: Set spin as default scheduling
    NVCL_EVENT_SCHED_SPIN = cydriver.cl_event_flags_enum.NVCL_EVENT_SCHED_SPIN

    #: Set yield as default scheduling
    NVCL_EVENT_SCHED_YIELD = cydriver.cl_event_flags_enum.NVCL_EVENT_SCHED_YIELD

    #: Set blocking synchronization as default scheduling
    NVCL_EVENT_SCHED_BLOCKING_SYNC = cydriver.cl_event_flags_enum.NVCL_EVENT_SCHED_BLOCKING_SYNC

class cl_context_flags(IntEnum):
    """
    NVCL context scheduling flags
    """

    #: Automatic scheduling
    NVCL_CTX_SCHED_AUTO = cydriver.cl_context_flags_enum.NVCL_CTX_SCHED_AUTO

    #: Set spin as default scheduling
    NVCL_CTX_SCHED_SPIN = cydriver.cl_context_flags_enum.NVCL_CTX_SCHED_SPIN

    #: Set yield as default scheduling
    NVCL_CTX_SCHED_YIELD = cydriver.cl_context_flags_enum.NVCL_CTX_SCHED_YIELD

    #: Set blocking synchronization as default scheduling
    NVCL_CTX_SCHED_BLOCKING_SYNC = cydriver.cl_context_flags_enum.NVCL_CTX_SCHED_BLOCKING_SYNC

class CUstream_flags(IntEnum):
    """
    Stream creation flags
    """

    #: Default stream flag
    CU_STREAM_DEFAULT = cydriver.CUstream_flags_enum.CU_STREAM_DEFAULT

    #: Stream does not synchronize with stream 0 (the NULL stream)
    CU_STREAM_NON_BLOCKING = cydriver.CUstream_flags_enum.CU_STREAM_NON_BLOCKING

class CUevent_flags(IntEnum):
    """
    Event creation flags
    """

    #: Default event flag
    CU_EVENT_DEFAULT = cydriver.CUevent_flags_enum.CU_EVENT_DEFAULT

    #: Event uses blocking synchronization
    CU_EVENT_BLOCKING_SYNC = cydriver.CUevent_flags_enum.CU_EVENT_BLOCKING_SYNC

    #: Event will not record timing data
    CU_EVENT_DISABLE_TIMING = cydriver.CUevent_flags_enum.CU_EVENT_DISABLE_TIMING

    #: Event is suitable for interprocess use. CU_EVENT_DISABLE_TIMING must
    #: be set
    CU_EVENT_INTERPROCESS = cydriver.CUevent_flags_enum.CU_EVENT_INTERPROCESS

class CUevent_record_flags(IntEnum):
    """
    Event record flags
    """

    #: Default event record flag
    CU_EVENT_RECORD_DEFAULT = cydriver.CUevent_record_flags_enum.CU_EVENT_RECORD_DEFAULT

    #: When using stream capture, create an event record node instead of
    #: the default behavior. This flag is invalid when used outside of
    #: capture.
    CU_EVENT_RECORD_EXTERNAL = cydriver.CUevent_record_flags_enum.CU_EVENT_RECORD_EXTERNAL

class CUevent_wait_flags(IntEnum):
    """
    Event wait flags
    """

    #: Default event wait flag
    CU_EVENT_WAIT_DEFAULT = cydriver.CUevent_wait_flags_enum.CU_EVENT_WAIT_DEFAULT

    #: When using stream capture, create an event wait node instead of the
    #: default behavior. This flag is invalid when used outside of capture.
    CU_EVENT_WAIT_EXTERNAL = cydriver.CUevent_wait_flags_enum.CU_EVENT_WAIT_EXTERNAL

class CUstreamWaitValue_flags(IntEnum):
    """
    Flags for :py:obj:`~.cuStreamWaitValue32` and
    :py:obj:`~.cuStreamWaitValue64`
    """

    #: Wait until (int32_t)(*addr - value) >= 0 (or int64_t for 64 bit
    #: values). Note this is a cyclic comparison which ignores wraparound.
    #: (Default behavior.)
    CU_STREAM_WAIT_VALUE_GEQ = cydriver.CUstreamWaitValue_flags_enum.CU_STREAM_WAIT_VALUE_GEQ

    #: Wait until *addr == value.
    CU_STREAM_WAIT_VALUE_EQ = cydriver.CUstreamWaitValue_flags_enum.CU_STREAM_WAIT_VALUE_EQ

    #: Wait until (*addr & value) != 0.
    CU_STREAM_WAIT_VALUE_AND = cydriver.CUstreamWaitValue_flags_enum.CU_STREAM_WAIT_VALUE_AND

    #: Wait until ~(*addr | value) != 0. Support for this operation can be
    #: queried with :py:obj:`~.cuDeviceGetAttribute()` and
    #: :py:obj:`~.CU_DEVICE_ATTRIBUTE_CAN_USE_STREAM_WAIT_VALUE_NOR`.
    CU_STREAM_WAIT_VALUE_NOR = cydriver.CUstreamWaitValue_flags_enum.CU_STREAM_WAIT_VALUE_NOR

    #: Follow the wait operation with a flush of outstanding remote writes.
    #: This means that, if a remote write operation is guaranteed to have
    #: reached the device before the wait can be satisfied, that write is
    #: guaranteed to be visible to downstream device work. The device is
    #: permitted to reorder remote writes internally. For example, this
    #: flag would be required if two remote writes arrive in a defined
    #: order, the wait is satisfied by the second write, and downstream
    #: work needs to observe the first write. Support for this operation is
    #: restricted to selected platforms and can be queried with
    #: :py:obj:`~.CU_DEVICE_ATTRIBUTE_CAN_FLUSH_REMOTE_WRITES`.
    CU_STREAM_WAIT_VALUE_FLUSH = cydriver.CUstreamWaitValue_flags_enum.CU_STREAM_WAIT_VALUE_FLUSH

class CUstreamWriteValue_flags(IntEnum):
    """
    Flags for :py:obj:`~.cuStreamWriteValue32`
    """

    #: Default behavior
    CU_STREAM_WRITE_VALUE_DEFAULT = cydriver.CUstreamWriteValue_flags_enum.CU_STREAM_WRITE_VALUE_DEFAULT

    #: Permits the write to be reordered with writes which were issued
    #: before it, as a performance optimization. Normally,
    #: :py:obj:`~.cuStreamWriteValue32` will provide a memory fence before
    #: the write, which has similar semantics to __threadfence_system() but
    #: is scoped to the stream rather than a CUDA thread. This flag is not
    #: supported in the v2 API.
    CU_STREAM_WRITE_VALUE_NO_MEMORY_BARRIER = cydriver.CUstreamWriteValue_flags_enum.CU_STREAM_WRITE_VALUE_NO_MEMORY_BARRIER

class CUstreamBatchMemOpType(IntEnum):
    """
    Operations for :py:obj:`~.cuStreamBatchMemOp`
    """

    #: Represents a :py:obj:`~.cuStreamWaitValue32` operation
    CU_STREAM_MEM_OP_WAIT_VALUE_32 = cydriver.CUstreamBatchMemOpType_enum.CU_STREAM_MEM_OP_WAIT_VALUE_32

    #: Represents a :py:obj:`~.cuStreamWriteValue32` operation
    CU_STREAM_MEM_OP_WRITE_VALUE_32 = cydriver.CUstreamBatchMemOpType_enum.CU_STREAM_MEM_OP_WRITE_VALUE_32

    #: This has the same effect as :py:obj:`~.CU_STREAM_WAIT_VALUE_FLUSH`,
    #: but as a standalone operation.
    CU_STREAM_MEM_OP_FLUSH_REMOTE_WRITES = cydriver.CUstreamBatchMemOpType_enum.CU_STREAM_MEM_OP_FLUSH_REMOTE_WRITES

    #: Represents a :py:obj:`~.cuStreamWaitValue64` operation
    CU_STREAM_MEM_OP_WAIT_VALUE_64 = cydriver.CUstreamBatchMemOpType_enum.CU_STREAM_MEM_OP_WAIT_VALUE_64

    #: Represents a :py:obj:`~.cuStreamWriteValue64` operation
    CU_STREAM_MEM_OP_WRITE_VALUE_64 = cydriver.CUstreamBatchMemOpType_enum.CU_STREAM_MEM_OP_WRITE_VALUE_64

    #: Insert a memory barrier of the specified type
    CU_STREAM_MEM_OP_BARRIER = cydriver.CUstreamBatchMemOpType_enum.CU_STREAM_MEM_OP_BARRIER

class CUstreamMemoryBarrier_flags(IntEnum):
    """
    Flags for :py:obj:`~.cuStreamMemoryBarrier`
    """

    #: System-wide memory barrier.
    CU_STREAM_MEMORY_BARRIER_TYPE_SYS = cydriver.CUstreamMemoryBarrier_flags_enum.CU_STREAM_MEMORY_BARRIER_TYPE_SYS

    #: Limit memory barrier scope to the GPU.
    CU_STREAM_MEMORY_BARRIER_TYPE_GPU = cydriver.CUstreamMemoryBarrier_flags_enum.CU_STREAM_MEMORY_BARRIER_TYPE_GPU

class CUoccupancy_flags(IntEnum):
    """
    Occupancy calculator flag
    """

    #: Default behavior
    CU_OCCUPANCY_DEFAULT = cydriver.CUoccupancy_flags_enum.CU_OCCUPANCY_DEFAULT

    #: Assume global caching is enabled and cannot be automatically turned
    #: off
    CU_OCCUPANCY_DISABLE_CACHING_OVERRIDE = cydriver.CUoccupancy_flags_enum.CU_OCCUPANCY_DISABLE_CACHING_OVERRIDE

class CUstreamUpdateCaptureDependencies_flags(IntEnum):
    """
    Flags for :py:obj:`~.cuStreamUpdateCaptureDependencies`
    """

    #: Add new nodes to the dependency set
    CU_STREAM_ADD_CAPTURE_DEPENDENCIES = cydriver.CUstreamUpdateCaptureDependencies_flags_enum.CU_STREAM_ADD_CAPTURE_DEPENDENCIES

    #: Replace the dependency set with the new nodes
    CU_STREAM_SET_CAPTURE_DEPENDENCIES = cydriver.CUstreamUpdateCaptureDependencies_flags_enum.CU_STREAM_SET_CAPTURE_DEPENDENCIES

class CUarray_format(IntEnum):
    """
    Array formats
    """

    #: Unsigned 8-bit integers
    CU_AD_FORMAT_UNSIGNED_INT8 = cydriver.CUarray_format_enum.CU_AD_FORMAT_UNSIGNED_INT8

    #: Unsigned 16-bit integers
    CU_AD_FORMAT_UNSIGNED_INT16 = cydriver.CUarray_format_enum.CU_AD_FORMAT_UNSIGNED_INT16

    #: Unsigned 32-bit integers
    CU_AD_FORMAT_UNSIGNED_INT32 = cydriver.CUarray_format_enum.CU_AD_FORMAT_UNSIGNED_INT32

    #: Signed 8-bit integers
    CU_AD_FORMAT_SIGNED_INT8 = cydriver.CUarray_format_enum.CU_AD_FORMAT_SIGNED_INT8

    #: Signed 16-bit integers
    CU_AD_FORMAT_SIGNED_INT16 = cydriver.CUarray_format_enum.CU_AD_FORMAT_SIGNED_INT16

    #: Signed 32-bit integers
    CU_AD_FORMAT_SIGNED_INT32 = cydriver.CUarray_format_enum.CU_AD_FORMAT_SIGNED_INT32

    #: 16-bit floating point
    CU_AD_FORMAT_HALF = cydriver.CUarray_format_enum.CU_AD_FORMAT_HALF

    #: 32-bit floating point
    CU_AD_FORMAT_FLOAT = cydriver.CUarray_format_enum.CU_AD_FORMAT_FLOAT

    #: 4 channel unsigned normalized block-compressed (BC1 compression)
    #: format
    CU_AD_FORMAT_BC1_UNORM = cydriver.CUarray_format_enum.CU_AD_FORMAT_BC1_UNORM

    #: 4 channel unsigned normalized block-compressed (BC1 compression)
    #: format with sRGB encoding
    CU_AD_FORMAT_BC1_UNORM_SRGB = cydriver.CUarray_format_enum.CU_AD_FORMAT_BC1_UNORM_SRGB

    #: 4 channel unsigned normalized block-compressed (BC2 compression)
    #: format
    CU_AD_FORMAT_BC2_UNORM = cydriver.CUarray_format_enum.CU_AD_FORMAT_BC2_UNORM

    #: 4 channel unsigned normalized block-compressed (BC2 compression)
    #: format with sRGB encoding
    CU_AD_FORMAT_BC2_UNORM_SRGB = cydriver.CUarray_format_enum.CU_AD_FORMAT_BC2_UNORM_SRGB

    #: 4 channel unsigned normalized block-compressed (BC3 compression)
    #: format
    CU_AD_FORMAT_BC3_UNORM = cydriver.CUarray_format_enum.CU_AD_FORMAT_BC3_UNORM

    #: 4 channel unsigned normalized block-compressed (BC3 compression)
    #: format with sRGB encoding
    CU_AD_FORMAT_BC3_UNORM_SRGB = cydriver.CUarray_format_enum.CU_AD_FORMAT_BC3_UNORM_SRGB

    #: 1 channel unsigned normalized block-compressed (BC4 compression)
    #: format
    CU_AD_FORMAT_BC4_UNORM = cydriver.CUarray_format_enum.CU_AD_FORMAT_BC4_UNORM

    #: 1 channel signed normalized block-compressed (BC4 compression)
    #: format
    CU_AD_FORMAT_BC4_SNORM = cydriver.CUarray_format_enum.CU_AD_FORMAT_BC4_SNORM

    #: 2 channel unsigned normalized block-compressed (BC5 compression)
    #: format
    CU_AD_FORMAT_BC5_UNORM = cydriver.CUarray_format_enum.CU_AD_FORMAT_BC5_UNORM

    #: 2 channel signed normalized block-compressed (BC5 compression)
    #: format
    CU_AD_FORMAT_BC5_SNORM = cydriver.CUarray_format_enum.CU_AD_FORMAT_BC5_SNORM

    #: 3 channel unsigned half-float block-compressed (BC6H compression)
    #: format
    CU_AD_FORMAT_BC6H_UF16 = cydriver.CUarray_format_enum.CU_AD_FORMAT_BC6H_UF16

    #: 3 channel signed half-float block-compressed (BC6H compression)
    #: format
    CU_AD_FORMAT_BC6H_SF16 = cydriver.CUarray_format_enum.CU_AD_FORMAT_BC6H_SF16

    #: 4 channel unsigned normalized block-compressed (BC7 compression)
    #: format
    CU_AD_FORMAT_BC7_UNORM = cydriver.CUarray_format_enum.CU_AD_FORMAT_BC7_UNORM

    #: 4 channel unsigned normalized block-compressed (BC7 compression)
    #: format with sRGB encoding
    CU_AD_FORMAT_BC7_UNORM_SRGB = cydriver.CUarray_format_enum.CU_AD_FORMAT_BC7_UNORM_SRGB

    #: 8-bit YUV planar format, with 4:2:0 sampling
    CU_AD_FORMAT_NV12 = cydriver.CUarray_format_enum.CU_AD_FORMAT_NV12

    #: 1 channel unsigned 8-bit normalized integer
    CU_AD_FORMAT_UNORM_INT8X1 = cydriver.CUarray_format_enum.CU_AD_FORMAT_UNORM_INT8X1

    #: 2 channel unsigned 8-bit normalized integer
    CU_AD_FORMAT_UNORM_INT8X2 = cydriver.CUarray_format_enum.CU_AD_FORMAT_UNORM_INT8X2

    #: 4 channel unsigned 8-bit normalized integer
    CU_AD_FORMAT_UNORM_INT8X4 = cydriver.CUarray_format_enum.CU_AD_FORMAT_UNORM_INT8X4

    #: 1 channel unsigned 16-bit normalized integer
    CU_AD_FORMAT_UNORM_INT16X1 = cydriver.CUarray_format_enum.CU_AD_FORMAT_UNORM_INT16X1

    #: 2 channel unsigned 16-bit normalized integer
    CU_AD_FORMAT_UNORM_INT16X2 = cydriver.CUarray_format_enum.CU_AD_FORMAT_UNORM_INT16X2

    #: 4 channel unsigned 16-bit normalized integer
    CU_AD_FORMAT_UNORM_INT16X4 = cydriver.CUarray_format_enum.CU_AD_FORMAT_UNORM_INT16X4

    #: 1 channel signed 8-bit normalized integer
    CU_AD_FORMAT_SNORM_INT8X1 = cydriver.CUarray_format_enum.CU_AD_FORMAT_SNORM_INT8X1

    #: 2 channel signed 8-bit normalized integer
    CU_AD_FORMAT_SNORM_INT8X2 = cydriver.CUarray_format_enum.CU_AD_FORMAT_SNORM_INT8X2

    #: 4 channel signed 8-bit normalized integer
    CU_AD_FORMAT_SNORM_INT8X4 = cydriver.CUarray_format_enum.CU_AD_FORMAT_SNORM_INT8X4

    #: 1 channel signed 16-bit normalized integer
    CU_AD_FORMAT_SNORM_INT16X1 = cydriver.CUarray_format_enum.CU_AD_FORMAT_SNORM_INT16X1

    #: 2 channel signed 16-bit normalized integer
    CU_AD_FORMAT_SNORM_INT16X2 = cydriver.CUarray_format_enum.CU_AD_FORMAT_SNORM_INT16X2

    #: 4 channel signed 16-bit normalized integer
    CU_AD_FORMAT_SNORM_INT16X4 = cydriver.CUarray_format_enum.CU_AD_FORMAT_SNORM_INT16X4

class CUaddress_mode(IntEnum):
    """
    Texture reference addressing modes
    """

    #: Wrapping address mode
    CU_TR_ADDRESS_MODE_WRAP = cydriver.CUaddress_mode_enum.CU_TR_ADDRESS_MODE_WRAP

    #: Clamp to edge address mode
    CU_TR_ADDRESS_MODE_CLAMP = cydriver.CUaddress_mode_enum.CU_TR_ADDRESS_MODE_CLAMP

    #: Mirror address mode
    CU_TR_ADDRESS_MODE_MIRROR = cydriver.CUaddress_mode_enum.CU_TR_ADDRESS_MODE_MIRROR

    #: Border address mode
    CU_TR_ADDRESS_MODE_BORDER = cydriver.CUaddress_mode_enum.CU_TR_ADDRESS_MODE_BORDER

class CUfilter_mode(IntEnum):
    """
    Texture reference filtering modes
    """

    #: Point filter mode
    CU_TR_FILTER_MODE_POINT = cydriver.CUfilter_mode_enum.CU_TR_FILTER_MODE_POINT

    #: Linear filter mode
    CU_TR_FILTER_MODE_LINEAR = cydriver.CUfilter_mode_enum.CU_TR_FILTER_MODE_LINEAR

class CUdevice_attribute(IntEnum):
    """
    Device properties
    """

    #: Maximum number of threads per block
    CU_DEVICE_ATTRIBUTE_MAX_THREADS_PER_BLOCK = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAX_THREADS_PER_BLOCK

    #: Maximum block dimension X
    CU_DEVICE_ATTRIBUTE_MAX_BLOCK_DIM_X = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAX_BLOCK_DIM_X

    #: Maximum block dimension Y
    CU_DEVICE_ATTRIBUTE_MAX_BLOCK_DIM_Y = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAX_BLOCK_DIM_Y

    #: Maximum block dimension Z
    CU_DEVICE_ATTRIBUTE_MAX_BLOCK_DIM_Z = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAX_BLOCK_DIM_Z

    #: Maximum grid dimension X
    CU_DEVICE_ATTRIBUTE_MAX_GRID_DIM_X = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAX_GRID_DIM_X

    #: Maximum grid dimension Y
    CU_DEVICE_ATTRIBUTE_MAX_GRID_DIM_Y = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAX_GRID_DIM_Y

    #: Maximum grid dimension Z
    CU_DEVICE_ATTRIBUTE_MAX_GRID_DIM_Z = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAX_GRID_DIM_Z

    #: Maximum shared memory available per block in bytes
    CU_DEVICE_ATTRIBUTE_MAX_SHARED_MEMORY_PER_BLOCK = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAX_SHARED_MEMORY_PER_BLOCK

    #: Deprecated, use CU_DEVICE_ATTRIBUTE_MAX_SHARED_MEMORY_PER_BLOCK
    CU_DEVICE_ATTRIBUTE_SHARED_MEMORY_PER_BLOCK = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_SHARED_MEMORY_PER_BLOCK

    #: Memory available on device for constant variables in a CUDA C kernel
    #: in bytes
    CU_DEVICE_ATTRIBUTE_TOTAL_CONSTANT_MEMORY = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_TOTAL_CONSTANT_MEMORY

    #: Warp size in threads
    CU_DEVICE_ATTRIBUTE_WARP_SIZE = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_WARP_SIZE

    #: Maximum pitch in bytes allowed by memory copies
    CU_DEVICE_ATTRIBUTE_MAX_PITCH = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAX_PITCH

    #: Maximum number of 32-bit registers available per block
    CU_DEVICE_ATTRIBUTE_MAX_REGISTERS_PER_BLOCK = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAX_REGISTERS_PER_BLOCK

    #: Deprecated, use CU_DEVICE_ATTRIBUTE_MAX_REGISTERS_PER_BLOCK
    CU_DEVICE_ATTRIBUTE_REGISTERS_PER_BLOCK = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_REGISTERS_PER_BLOCK

    #: Typical clock frequency in kilohertz
    CU_DEVICE_ATTRIBUTE_CLOCK_RATE = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_CLOCK_RATE

    #: Alignment requirement for textures
    CU_DEVICE_ATTRIBUTE_TEXTURE_ALIGNMENT = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_TEXTURE_ALIGNMENT

    #: Device can possibly copy memory and execute a kernel concurrently.
    #: Deprecated. Use instead CU_DEVICE_ATTRIBUTE_ASYNC_ENGINE_COUNT.
    CU_DEVICE_ATTRIBUTE_GPU_OVERLAP = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_GPU_OVERLAP

    #: Number of multiprocessors on device
    CU_DEVICE_ATTRIBUTE_MULTIPROCESSOR_COUNT = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MULTIPROCESSOR_COUNT

    #: Specifies whether there is a run time limit on kernels
    CU_DEVICE_ATTRIBUTE_KERNEL_EXEC_TIMEOUT = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_KERNEL_EXEC_TIMEOUT

    #: Device is integrated with host memory
    CU_DEVICE_ATTRIBUTE_INTEGRATED = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_INTEGRATED

    #: Device can map host memory into CUDA address space
    CU_DEVICE_ATTRIBUTE_CAN_MAP_HOST_MEMORY = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_CAN_MAP_HOST_MEMORY

    #: Compute mode (See :py:obj:`~.CUcomputemode` for details)
    CU_DEVICE_ATTRIBUTE_COMPUTE_MODE = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_COMPUTE_MODE

    #: Maximum 1D texture width
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE1D_WIDTH = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE1D_WIDTH

    #: Maximum 2D texture width
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_WIDTH = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_WIDTH

    #: Maximum 2D texture height
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_HEIGHT = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_HEIGHT

    #: Maximum 3D texture width
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_WIDTH = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_WIDTH

    #: Maximum 3D texture height
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_HEIGHT = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_HEIGHT

    #: Maximum 3D texture depth
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_DEPTH = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_DEPTH

    #: Maximum 2D layered texture width
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LAYERED_WIDTH = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LAYERED_WIDTH

    #: Deprecated, use CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LAYERED_WIDTH
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_ARRAY_WIDTH = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_ARRAY_WIDTH

    #: Maximum 2D layered texture height
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LAYERED_HEIGHT = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LAYERED_HEIGHT

    #: Deprecated, use CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LAYERED_HEIGHT
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_ARRAY_HEIGHT = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_ARRAY_HEIGHT

    #: Maximum layers in a 2D layered texture
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LAYERED_LAYERS = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LAYERED_LAYERS

    #: Deprecated, use CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LAYERED_LAYERS
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_ARRAY_NUMSLICES = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_ARRAY_NUMSLICES

    #: Alignment requirement for surfaces
    CU_DEVICE_ATTRIBUTE_SURFACE_ALIGNMENT = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_SURFACE_ALIGNMENT

    #: Device can possibly execute multiple kernels concurrently
    CU_DEVICE_ATTRIBUTE_CONCURRENT_KERNELS = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_CONCURRENT_KERNELS

    #: Device has ECC support enabled
    CU_DEVICE_ATTRIBUTE_ECC_ENABLED = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_ECC_ENABLED

    #: PCI bus ID of the device
    CU_DEVICE_ATTRIBUTE_PCI_BUS_ID = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_PCI_BUS_ID

    #: PCI device ID of the device
    CU_DEVICE_ATTRIBUTE_PCI_DEVICE_ID = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_PCI_DEVICE_ID

    #: Device is using TCC driver model
    CU_DEVICE_ATTRIBUTE_TCC_DRIVER = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_TCC_DRIVER

    #: Peak memory clock frequency in kilohertz
    CU_DEVICE_ATTRIBUTE_MEMORY_CLOCK_RATE = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MEMORY_CLOCK_RATE

    #: Global memory bus width in bits
    CU_DEVICE_ATTRIBUTE_GLOBAL_MEMORY_BUS_WIDTH = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_GLOBAL_MEMORY_BUS_WIDTH

    #: Size of L2 cache in bytes
    CU_DEVICE_ATTRIBUTE_L2_CACHE_SIZE = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_L2_CACHE_SIZE

    #: Maximum resident threads per multiprocessor
    CU_DEVICE_ATTRIBUTE_MAX_THREADS_PER_MULTIPROCESSOR = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAX_THREADS_PER_MULTIPROCESSOR

    #: Number of asynchronous engines
    CU_DEVICE_ATTRIBUTE_ASYNC_ENGINE_COUNT = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_ASYNC_ENGINE_COUNT

    #: Device shares a unified address space with the host
    CU_DEVICE_ATTRIBUTE_UNIFIED_ADDRESSING = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_UNIFIED_ADDRESSING

    #: Maximum 1D layered texture width
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE1D_LAYERED_WIDTH = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE1D_LAYERED_WIDTH

    #: Maximum layers in a 1D layered texture
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE1D_LAYERED_LAYERS = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE1D_LAYERED_LAYERS

    #: Deprecated, do not use.
    CU_DEVICE_ATTRIBUTE_CAN_TEX2D_GATHER = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_CAN_TEX2D_GATHER

    #: Maximum 2D texture width if CUDA_ARRAY3D_TEXTURE_GATHER is set
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_GATHER_WIDTH = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_GATHER_WIDTH

    #: Maximum 2D texture height if CUDA_ARRAY3D_TEXTURE_GATHER is set
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_GATHER_HEIGHT = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_GATHER_HEIGHT

    #: Alternate maximum 3D texture width
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_WIDTH_ALTERNATE = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_WIDTH_ALTERNATE

    #: Alternate maximum 3D texture height
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_HEIGHT_ALTERNATE = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_HEIGHT_ALTERNATE

    #: Alternate maximum 3D texture depth
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_DEPTH_ALTERNATE = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_DEPTH_ALTERNATE

    #: PCI domain ID of the device
    CU_DEVICE_ATTRIBUTE_PCI_DOMAIN_ID = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_PCI_DOMAIN_ID

    #: Pitch alignment requirement for textures
    CU_DEVICE_ATTRIBUTE_TEXTURE_PITCH_ALIGNMENT = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_TEXTURE_PITCH_ALIGNMENT

    #: Maximum cubemap texture width/height
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURECUBEMAP_WIDTH = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURECUBEMAP_WIDTH

    #: Maximum cubemap layered texture width/height
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURECUBEMAP_LAYERED_WIDTH = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURECUBEMAP_LAYERED_WIDTH

    #: Maximum layers in a cubemap layered texture
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURECUBEMAP_LAYERED_LAYERS = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURECUBEMAP_LAYERED_LAYERS

    #: Maximum 1D surface width
    CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE1D_WIDTH = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE1D_WIDTH

    #: Maximum 2D surface width
    CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE2D_WIDTH = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE2D_WIDTH

    #: Maximum 2D surface height
    CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE2D_HEIGHT = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE2D_HEIGHT

    #: Maximum 3D surface width
    CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE3D_WIDTH = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE3D_WIDTH

    #: Maximum 3D surface height
    CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE3D_HEIGHT = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE3D_HEIGHT

    #: Maximum 3D surface depth
    CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE3D_DEPTH = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE3D_DEPTH

    #: Maximum 1D layered surface width
    CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE1D_LAYERED_WIDTH = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE1D_LAYERED_WIDTH

    #: Maximum layers in a 1D layered surface
    CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE1D_LAYERED_LAYERS = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE1D_LAYERED_LAYERS

    #: Maximum 2D layered surface width
    CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE2D_LAYERED_WIDTH = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE2D_LAYERED_WIDTH

    #: Maximum 2D layered surface height
    CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE2D_LAYERED_HEIGHT = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE2D_LAYERED_HEIGHT

    #: Maximum layers in a 2D layered surface
    CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE2D_LAYERED_LAYERS = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE2D_LAYERED_LAYERS

    #: Maximum cubemap surface width
    CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACECUBEMAP_WIDTH = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACECUBEMAP_WIDTH

    #: Maximum cubemap layered surface width
    CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACECUBEMAP_LAYERED_WIDTH = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACECUBEMAP_LAYERED_WIDTH

    #: Maximum layers in a cubemap layered surface
    CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACECUBEMAP_LAYERED_LAYERS = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACECUBEMAP_LAYERED_LAYERS

    #: Deprecated, do not use. Use cudaDeviceGetTexture1DLinearMaxWidth()
    #: or :py:obj:`~.cuDeviceGetTexture1DLinearMaxWidth()` instead.
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE1D_LINEAR_WIDTH = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE1D_LINEAR_WIDTH

    #: Maximum 2D linear texture width
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LINEAR_WIDTH = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LINEAR_WIDTH

    #: Maximum 2D linear texture height
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LINEAR_HEIGHT = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LINEAR_HEIGHT

    #: Maximum 2D linear texture pitch in bytes
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LINEAR_PITCH = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LINEAR_PITCH

    #: Maximum mipmapped 2D texture width
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_MIPMAPPED_WIDTH = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_MIPMAPPED_WIDTH

    #: Maximum mipmapped 2D texture height
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_MIPMAPPED_HEIGHT = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_MIPMAPPED_HEIGHT

    #: Major compute capability version number
    CU_DEVICE_ATTRIBUTE_COMPUTE_CAPABILITY_MAJOR = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_COMPUTE_CAPABILITY_MAJOR

    #: Minor compute capability version number
    CU_DEVICE_ATTRIBUTE_COMPUTE_CAPABILITY_MINOR = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_COMPUTE_CAPABILITY_MINOR

    #: Maximum mipmapped 1D texture width
    CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE1D_MIPMAPPED_WIDTH = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE1D_MIPMAPPED_WIDTH

    #: Device supports stream priorities
    CU_DEVICE_ATTRIBUTE_STREAM_PRIORITIES_SUPPORTED = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_STREAM_PRIORITIES_SUPPORTED

    #: Device supports caching globals in L1
    CU_DEVICE_ATTRIBUTE_GLOBAL_L1_CACHE_SUPPORTED = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_GLOBAL_L1_CACHE_SUPPORTED

    #: Device supports caching locals in L1
    CU_DEVICE_ATTRIBUTE_LOCAL_L1_CACHE_SUPPORTED = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_LOCAL_L1_CACHE_SUPPORTED

    #: Maximum shared memory available per multiprocessor in bytes
    CU_DEVICE_ATTRIBUTE_MAX_SHARED_MEMORY_PER_MULTIPROCESSOR = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAX_SHARED_MEMORY_PER_MULTIPROCESSOR

    #: Maximum number of 32-bit registers available per multiprocessor
    CU_DEVICE_ATTRIBUTE_MAX_REGISTERS_PER_MULTIPROCESSOR = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAX_REGISTERS_PER_MULTIPROCESSOR

    #: Device can allocate managed memory on this system
    CU_DEVICE_ATTRIBUTE_MANAGED_MEMORY = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MANAGED_MEMORY

    #: Device is on a multi-GPU board
    CU_DEVICE_ATTRIBUTE_MULTI_GPU_BOARD = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MULTI_GPU_BOARD

    #: Unique id for a group of devices on the same multi-GPU board
    CU_DEVICE_ATTRIBUTE_MULTI_GPU_BOARD_GROUP_ID = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MULTI_GPU_BOARD_GROUP_ID

    #: Link between the device and the host supports native atomic
    #: operations (this is a placeholder attribute, and is not supported on
    #: any current hardware)
    CU_DEVICE_ATTRIBUTE_HOST_NATIVE_ATOMIC_SUPPORTED = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_HOST_NATIVE_ATOMIC_SUPPORTED

    #: Ratio of single precision performance (in floating-point operations
    #: per second) to double precision performance
    CU_DEVICE_ATTRIBUTE_SINGLE_TO_DOUBLE_PRECISION_PERF_RATIO = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_SINGLE_TO_DOUBLE_PRECISION_PERF_RATIO

    #: Device supports coherently accessing pageable memory without calling
    #: cudaHostRegister on it
    CU_DEVICE_ATTRIBUTE_PAGEABLE_MEMORY_ACCESS = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_PAGEABLE_MEMORY_ACCESS

    #: Device can coherently access managed memory concurrently with the
    #: CPU
    CU_DEVICE_ATTRIBUTE_CONCURRENT_MANAGED_ACCESS = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_CONCURRENT_MANAGED_ACCESS

    #: Device supports compute preemption.
    CU_DEVICE_ATTRIBUTE_COMPUTE_PREEMPTION_SUPPORTED = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_COMPUTE_PREEMPTION_SUPPORTED

    #: Device can access host registered memory at the same virtual address
    #: as the CPU
    CU_DEVICE_ATTRIBUTE_CAN_USE_HOST_POINTER_FOR_REGISTERED_MEM = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_CAN_USE_HOST_POINTER_FOR_REGISTERED_MEM

    #: :py:obj:`~.cuStreamBatchMemOp` and related APIs are supported.
    CU_DEVICE_ATTRIBUTE_CAN_USE_STREAM_MEM_OPS = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_CAN_USE_STREAM_MEM_OPS

    #: 64-bit operations are supported in :py:obj:`~.cuStreamBatchMemOp`
    #: and related APIs.
    CU_DEVICE_ATTRIBUTE_CAN_USE_64_BIT_STREAM_MEM_OPS = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_CAN_USE_64_BIT_STREAM_MEM_OPS

    #: :py:obj:`~.CU_STREAM_WAIT_VALUE_NOR` is supported.
    CU_DEVICE_ATTRIBUTE_CAN_USE_STREAM_WAIT_VALUE_NOR = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_CAN_USE_STREAM_WAIT_VALUE_NOR

    #: Device supports launching cooperative kernels via
    #: :py:obj:`~.cuLaunchCooperativeKernel`
    CU_DEVICE_ATTRIBUTE_COOPERATIVE_LAUNCH = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_COOPERATIVE_LAUNCH

    #: Deprecated, :py:obj:`~.cuLaunchCooperativeKernelMultiDevice` is
    #: deprecated.
    CU_DEVICE_ATTRIBUTE_COOPERATIVE_MULTI_DEVICE_LAUNCH = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_COOPERATIVE_MULTI_DEVICE_LAUNCH

    #: Maximum optin shared memory per block
    CU_DEVICE_ATTRIBUTE_MAX_SHARED_MEMORY_PER_BLOCK_OPTIN = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAX_SHARED_MEMORY_PER_BLOCK_OPTIN

    #: The :py:obj:`~.CU_STREAM_WAIT_VALUE_FLUSH` flag and the
    #: :py:obj:`~.CU_STREAM_MEM_OP_FLUSH_REMOTE_WRITES` MemOp are supported
    #: on the device. See :py:obj:`~.Stream Memory Operations` for
    #: additional details.
    CU_DEVICE_ATTRIBUTE_CAN_FLUSH_REMOTE_WRITES = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_CAN_FLUSH_REMOTE_WRITES

    #: Device supports host memory registration via
    #: :py:obj:`~.cudaHostRegister`.
    CU_DEVICE_ATTRIBUTE_HOST_REGISTER_SUPPORTED = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_HOST_REGISTER_SUPPORTED

    #: Device accesses pageable memory via the host's page tables.
    CU_DEVICE_ATTRIBUTE_PAGEABLE_MEMORY_ACCESS_USES_HOST_PAGE_TABLES = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_PAGEABLE_MEMORY_ACCESS_USES_HOST_PAGE_TABLES

    #: The host can directly access managed memory on the device without
    #: migration.
    CU_DEVICE_ATTRIBUTE_DIRECT_MANAGED_MEM_ACCESS_FROM_HOST = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_DIRECT_MANAGED_MEM_ACCESS_FROM_HOST

    #: Deprecated, Use
    #: CU_DEVICE_ATTRIBUTE_VIRTUAL_MEMORY_MANAGEMENT_SUPPORTED
    CU_DEVICE_ATTRIBUTE_VIRTUAL_ADDRESS_MANAGEMENT_SUPPORTED = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_VIRTUAL_ADDRESS_MANAGEMENT_SUPPORTED

    #: Device supports virtual memory management APIs like
    #: :py:obj:`~.cuMemAddressReserve`, :py:obj:`~.cuMemCreate`,
    #: :py:obj:`~.cuMemMap` and related APIs
    CU_DEVICE_ATTRIBUTE_VIRTUAL_MEMORY_MANAGEMENT_SUPPORTED = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_VIRTUAL_MEMORY_MANAGEMENT_SUPPORTED

    #: Device supports exporting memory to a posix file descriptor with
    #: :py:obj:`~.cuMemExportToShareableHandle`, if requested via
    #: :py:obj:`~.cuMemCreate`
    CU_DEVICE_ATTRIBUTE_HANDLE_TYPE_POSIX_FILE_DESCRIPTOR_SUPPORTED = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_HANDLE_TYPE_POSIX_FILE_DESCRIPTOR_SUPPORTED

    #: Device supports exporting memory to a Win32 NT handle with
    #: :py:obj:`~.cuMemExportToShareableHandle`, if requested via
    #: :py:obj:`~.cuMemCreate`
    CU_DEVICE_ATTRIBUTE_HANDLE_TYPE_WIN32_HANDLE_SUPPORTED = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_HANDLE_TYPE_WIN32_HANDLE_SUPPORTED

    #: Device supports exporting memory to a Win32 KMT handle with
    #: :py:obj:`~.cuMemExportToShareableHandle`, if requested via
    #: :py:obj:`~.cuMemCreate`
    CU_DEVICE_ATTRIBUTE_HANDLE_TYPE_WIN32_KMT_HANDLE_SUPPORTED = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_HANDLE_TYPE_WIN32_KMT_HANDLE_SUPPORTED

    #: Maximum number of blocks per multiprocessor
    CU_DEVICE_ATTRIBUTE_MAX_BLOCKS_PER_MULTIPROCESSOR = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAX_BLOCKS_PER_MULTIPROCESSOR

    #: Device supports compression of memory
    CU_DEVICE_ATTRIBUTE_GENERIC_COMPRESSION_SUPPORTED = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_GENERIC_COMPRESSION_SUPPORTED

    #: Maximum L2 persisting lines capacity setting in bytes.
    CU_DEVICE_ATTRIBUTE_MAX_PERSISTING_L2_CACHE_SIZE = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAX_PERSISTING_L2_CACHE_SIZE

    #: Maximum value of :py:obj:`~.CUaccessPolicyWindow.num_bytes`.
    CU_DEVICE_ATTRIBUTE_MAX_ACCESS_POLICY_WINDOW_SIZE = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAX_ACCESS_POLICY_WINDOW_SIZE

    #: Device supports specifying the GPUDirect RDMA flag with
    #: :py:obj:`~.cuMemCreate`
    CU_DEVICE_ATTRIBUTE_GPU_DIRECT_RDMA_WITH_CUDA_VMM_SUPPORTED = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_GPU_DIRECT_RDMA_WITH_CUDA_VMM_SUPPORTED

    #: Shared memory reserved by CUDA driver per block in bytes
    CU_DEVICE_ATTRIBUTE_RESERVED_SHARED_MEMORY_PER_BLOCK = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_RESERVED_SHARED_MEMORY_PER_BLOCK

    #: Device supports sparse CUDA arrays and sparse CUDA mipmapped arrays
    CU_DEVICE_ATTRIBUTE_SPARSE_CUDA_ARRAY_SUPPORTED = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_SPARSE_CUDA_ARRAY_SUPPORTED

    #: Device supports using the :py:obj:`~.cuMemHostRegister` flag
    #: :py:obj:`~.CU_MEMHOSTERGISTER_READ_ONLY` to register memory that
    #: must be mapped as read-only to the GPU
    CU_DEVICE_ATTRIBUTE_READ_ONLY_HOST_REGISTER_SUPPORTED = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_READ_ONLY_HOST_REGISTER_SUPPORTED

    #: External timeline semaphore interop is supported on the device
    CU_DEVICE_ATTRIBUTE_TIMELINE_SEMAPHORE_INTEROP_SUPPORTED = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_TIMELINE_SEMAPHORE_INTEROP_SUPPORTED

    #: Device supports using the :py:obj:`~.cuMemAllocAsync` and
    #: :py:obj:`~.cuMemPool` family of APIs
    CU_DEVICE_ATTRIBUTE_MEMORY_POOLS_SUPPORTED = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MEMORY_POOLS_SUPPORTED

    #: Device supports GPUDirect RDMA APIs, like nvidia_p2p_get_pages (see
    #: https://docs.nvidia.com/cuda/gpudirect-rdma for more information)
    CU_DEVICE_ATTRIBUTE_GPU_DIRECT_RDMA_SUPPORTED = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_GPU_DIRECT_RDMA_SUPPORTED

    #: The returned attribute shall be interpreted as a bitmask, where the
    #: individual bits are described by the
    #: :py:obj:`~.CUflushGPUDirectRDMAWritesOptions` enum
    CU_DEVICE_ATTRIBUTE_GPU_DIRECT_RDMA_FLUSH_WRITES_OPTIONS = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_GPU_DIRECT_RDMA_FLUSH_WRITES_OPTIONS

    #: GPUDirect RDMA writes to the device do not need to be flushed for
    #: consumers within the scope indicated by the returned attribute. See
    #: :py:obj:`~.CUGPUDirectRDMAWritesOrdering` for the numerical values
    #: returned here.
    CU_DEVICE_ATTRIBUTE_GPU_DIRECT_RDMA_WRITES_ORDERING = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_GPU_DIRECT_RDMA_WRITES_ORDERING

    #: Handle types supported with mempool based IPC
    CU_DEVICE_ATTRIBUTE_MEMPOOL_SUPPORTED_HANDLE_TYPES = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MEMPOOL_SUPPORTED_HANDLE_TYPES

    #: Indicates device supports cluster launch
    CU_DEVICE_ATTRIBUTE_CLUSTER_LAUNCH = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_CLUSTER_LAUNCH

    #: Device supports deferred mapping CUDA arrays and CUDA mipmapped
    #: arrays
    CU_DEVICE_ATTRIBUTE_DEFERRED_MAPPING_CUDA_ARRAY_SUPPORTED = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_DEFERRED_MAPPING_CUDA_ARRAY_SUPPORTED

    #: 64-bit operations are supported in :py:obj:`~.cuStreamBatchMemOp_v2`
    #: and related v2 MemOp APIs.
    CU_DEVICE_ATTRIBUTE_CAN_USE_64_BIT_STREAM_MEM_OPS_V2 = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_CAN_USE_64_BIT_STREAM_MEM_OPS_V2

    #: :py:obj:`~.CU_STREAM_WAIT_VALUE_NOR` is supported by v2 MemOp APIs.
    CU_DEVICE_ATTRIBUTE_CAN_USE_STREAM_WAIT_VALUE_NOR_V2 = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_CAN_USE_STREAM_WAIT_VALUE_NOR_V2

    #: Device supports buffer sharing with dma_buf mechanism.
    CU_DEVICE_ATTRIBUTE_DMA_BUF_SUPPORTED = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_DMA_BUF_SUPPORTED
    CU_DEVICE_ATTRIBUTE_MAX = cydriver.CUdevice_attribute_enum.CU_DEVICE_ATTRIBUTE_MAX

class CUpointer_attribute(IntEnum):
    """
    Pointer information
    """

    #: The :py:obj:`~.CUcontext` on which a pointer was allocated or
    #: registered
    CU_POINTER_ATTRIBUTE_CONTEXT = cydriver.CUpointer_attribute_enum.CU_POINTER_ATTRIBUTE_CONTEXT

    #: The :py:obj:`~.CUmemorytype` describing the physical location of a
    #: pointer
    CU_POINTER_ATTRIBUTE_MEMORY_TYPE = cydriver.CUpointer_attribute_enum.CU_POINTER_ATTRIBUTE_MEMORY_TYPE

    #: The address at which a pointer's memory may be accessed on the
    #: device
    CU_POINTER_ATTRIBUTE_DEVICE_POINTER = cydriver.CUpointer_attribute_enum.CU_POINTER_ATTRIBUTE_DEVICE_POINTER

    #: The address at which a pointer's memory may be accessed on the host
    CU_POINTER_ATTRIBUTE_HOST_POINTER = cydriver.CUpointer_attribute_enum.CU_POINTER_ATTRIBUTE_HOST_POINTER

    #: A pair of tokens for use with the nv-p2p.h Linux kernel interface
    CU_POINTER_ATTRIBUTE_P2P_TOKENS = cydriver.CUpointer_attribute_enum.CU_POINTER_ATTRIBUTE_P2P_TOKENS

    #: Synchronize every synchronous memory operation initiated on this
    #: region
    CU_POINTER_ATTRIBUTE_SYNC_MEMOPS = cydriver.CUpointer_attribute_enum.CU_POINTER_ATTRIBUTE_SYNC_MEMOPS

    #: A process-wide unique ID for an allocated memory region
    CU_POINTER_ATTRIBUTE_BUFFER_ID = cydriver.CUpointer_attribute_enum.CU_POINTER_ATTRIBUTE_BUFFER_ID

    #: Indicates if the pointer points to managed memory
    CU_POINTER_ATTRIBUTE_IS_MANAGED = cydriver.CUpointer_attribute_enum.CU_POINTER_ATTRIBUTE_IS_MANAGED

    #: A device ordinal of a device on which a pointer was allocated or
    #: registered
    CU_POINTER_ATTRIBUTE_DEVICE_ORDINAL = cydriver.CUpointer_attribute_enum.CU_POINTER_ATTRIBUTE_DEVICE_ORDINAL

    #: 1 if this pointer maps to an allocation that is suitable for
    #: :py:obj:`~.cudaIpcGetMemHandle`, 0 otherwise
    CU_POINTER_ATTRIBUTE_IS_LEGACY_CUDA_IPC_CAPABLE = cydriver.CUpointer_attribute_enum.CU_POINTER_ATTRIBUTE_IS_LEGACY_CUDA_IPC_CAPABLE

    #: Starting address for this requested pointer
    CU_POINTER_ATTRIBUTE_RANGE_START_ADDR = cydriver.CUpointer_attribute_enum.CU_POINTER_ATTRIBUTE_RANGE_START_ADDR

    #: Size of the address range for this requested pointer
    CU_POINTER_ATTRIBUTE_RANGE_SIZE = cydriver.CUpointer_attribute_enum.CU_POINTER_ATTRIBUTE_RANGE_SIZE

    #: 1 if this pointer is in a valid address range that is mapped to a
    #: backing allocation, 0 otherwise
    CU_POINTER_ATTRIBUTE_MAPPED = cydriver.CUpointer_attribute_enum.CU_POINTER_ATTRIBUTE_MAPPED

    #: Bitmask of allowed :py:obj:`~.CUmemAllocationHandleType` for this
    #: allocation
    CU_POINTER_ATTRIBUTE_ALLOWED_HANDLE_TYPES = cydriver.CUpointer_attribute_enum.CU_POINTER_ATTRIBUTE_ALLOWED_HANDLE_TYPES

    #: 1 if the memory this pointer is referencing can be used with the
    #: GPUDirect RDMA API
    CU_POINTER_ATTRIBUTE_IS_GPU_DIRECT_RDMA_CAPABLE = cydriver.CUpointer_attribute_enum.CU_POINTER_ATTRIBUTE_IS_GPU_DIRECT_RDMA_CAPABLE

    #: Returns the access flags the device associated with the current
    #: context has on the corresponding memory referenced by the pointer
    #: given
    CU_POINTER_ATTRIBUTE_ACCESS_FLAGS = cydriver.CUpointer_attribute_enum.CU_POINTER_ATTRIBUTE_ACCESS_FLAGS

    #: Returns the mempool handle for the allocation if it was allocated
    #: from a mempool. Otherwise returns NULL.
    CU_POINTER_ATTRIBUTE_MEMPOOL_HANDLE = cydriver.CUpointer_attribute_enum.CU_POINTER_ATTRIBUTE_MEMPOOL_HANDLE

    #: Size of the actual underlying mapping that the pointer belongs to
    CU_POINTER_ATTRIBUTE_MAPPING_SIZE = cydriver.CUpointer_attribute_enum.CU_POINTER_ATTRIBUTE_MAPPING_SIZE

    #: The start address of the mapping that the pointer belongs to
    CU_POINTER_ATTRIBUTE_MAPPING_BASE_ADDR = cydriver.CUpointer_attribute_enum.CU_POINTER_ATTRIBUTE_MAPPING_BASE_ADDR

    #: A process-wide unique id corresponding to the physical allocation
    #: the pointer belongs to
    CU_POINTER_ATTRIBUTE_MEMORY_BLOCK_ID = cydriver.CUpointer_attribute_enum.CU_POINTER_ATTRIBUTE_MEMORY_BLOCK_ID

class CUfunction_attribute(IntEnum):
    """
    Function properties
    """

    #: The maximum number of threads per block, beyond which a launch of
    #: the function would fail. This number depends on both the function
    #: and the device on which the function is currently loaded.
    CU_FUNC_ATTRIBUTE_MAX_THREADS_PER_BLOCK = cydriver.CUfunction_attribute_enum.CU_FUNC_ATTRIBUTE_MAX_THREADS_PER_BLOCK

    #: The size in bytes of statically-allocated shared memory required by
    #: this function. This does not include dynamically-allocated shared
    #: memory requested by the user at runtime.
    CU_FUNC_ATTRIBUTE_SHARED_SIZE_BYTES = cydriver.CUfunction_attribute_enum.CU_FUNC_ATTRIBUTE_SHARED_SIZE_BYTES

    #: The size in bytes of user-allocated constant memory required by this
    #: function.
    CU_FUNC_ATTRIBUTE_CONST_SIZE_BYTES = cydriver.CUfunction_attribute_enum.CU_FUNC_ATTRIBUTE_CONST_SIZE_BYTES

    #: The size in bytes of local memory used by each thread of this
    #: function.
    CU_FUNC_ATTRIBUTE_LOCAL_SIZE_BYTES = cydriver.CUfunction_attribute_enum.CU_FUNC_ATTRIBUTE_LOCAL_SIZE_BYTES

    #: The number of registers used by each thread of this function.
    CU_FUNC_ATTRIBUTE_NUM_REGS = cydriver.CUfunction_attribute_enum.CU_FUNC_ATTRIBUTE_NUM_REGS

    #: The PTX virtual architecture version for which the function was
    #: compiled. This value is the major PTX version * 10 + the minor PTX
    #: version, so a PTX version 1.3 function would return the value 13.
    #: Note that this may return the undefined value of 0 for cubins
    #: compiled prior to CUDA 3.0.
    CU_FUNC_ATTRIBUTE_PTX_VERSION = cydriver.CUfunction_attribute_enum.CU_FUNC_ATTRIBUTE_PTX_VERSION

    #: The binary architecture version for which the function was compiled.
    #: This value is the major binary version * 10 + the minor binary
    #: version, so a binary version 1.3 function would return the value 13.
    #: Note that this will return a value of 10 for legacy cubins that do
    #: not have a properly-encoded binary architecture version.
    CU_FUNC_ATTRIBUTE_BINARY_VERSION = cydriver.CUfunction_attribute_enum.CU_FUNC_ATTRIBUTE_BINARY_VERSION

    #: The attribute to indicate whether the function has been compiled
    #: with user specified option "-Xptxas --dlcm=ca" set .
    CU_FUNC_ATTRIBUTE_CACHE_MODE_CA = cydriver.CUfunction_attribute_enum.CU_FUNC_ATTRIBUTE_CACHE_MODE_CA

    #: The maximum size in bytes of dynamically-allocated shared memory
    #: that can be used by this function. If the user-specified dynamic
    #: shared memory size is larger than this value, the launch will fail.
    #: See :py:obj:`~.cuFuncSetAttribute`
    CU_FUNC_ATTRIBUTE_MAX_DYNAMIC_SHARED_SIZE_BYTES = cydriver.CUfunction_attribute_enum.CU_FUNC_ATTRIBUTE_MAX_DYNAMIC_SHARED_SIZE_BYTES

    #: On devices where the L1 cache and shared memory use the same
    #: hardware resources, this sets the shared memory carveout preference,
    #: in percent of the total shared memory. Refer to
    #: :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_SHARED_MEMORY_PER_MULTIPROCESSOR`.
    #: This is only a hint, and the driver can choose a different ratio if
    #: required to execute the function. See :py:obj:`~.cuFuncSetAttribute`
    CU_FUNC_ATTRIBUTE_PREFERRED_SHARED_MEMORY_CARVEOUT = cydriver.CUfunction_attribute_enum.CU_FUNC_ATTRIBUTE_PREFERRED_SHARED_MEMORY_CARVEOUT

    #: If this attribute is set, the kernel must launch with a valid
    #: cluster size specified. See :py:obj:`~.cuFuncSetAttribute`
    CU_FUNC_ATTRIBUTE_CLUSTER_SIZE_MUST_BE_SET = cydriver.CUfunction_attribute_enum.CU_FUNC_ATTRIBUTE_CLUSTER_SIZE_MUST_BE_SET

    #: The required cluster width in blocks. The values must either all be
    #: 0 or all be positive. The validity of the cluster dimensions is
    #: otherwise checked at launch time.
    #:
    #: If the value is set during compile time, it cannot be set at
    #: runtime. Setting it at runtime will return CUDA_ERROR_NOT_PERMITTED.
    #: See :py:obj:`~.cuFuncSetAttribute`
    CU_FUNC_ATTRIBUTE_REQUIRED_CLUSTER_WIDTH = cydriver.CUfunction_attribute_enum.CU_FUNC_ATTRIBUTE_REQUIRED_CLUSTER_WIDTH

    #: The required cluster height in blocks. The values must either all be
    #: 0 or all be positive. The validity of the cluster dimensions is
    #: otherwise checked at launch time.
    #:
    #: If the value is set during compile time, it cannot be set at
    #: runtime. Setting it at runtime should return
    #: CUDA_ERROR_NOT_PERMITTED. See :py:obj:`~.cuFuncSetAttribute`
    CU_FUNC_ATTRIBUTE_REQUIRED_CLUSTER_HEIGHT = cydriver.CUfunction_attribute_enum.CU_FUNC_ATTRIBUTE_REQUIRED_CLUSTER_HEIGHT

    #: The required cluster depth in blocks. The values must either all be
    #: 0 or all be positive. The validity of the cluster dimensions is
    #: otherwise checked at launch time.
    #:
    #: If the value is set during compile time, it cannot be set at
    #: runtime. Setting it at runtime should return
    #: CUDA_ERROR_NOT_PERMITTED. See :py:obj:`~.cuFuncSetAttribute`
    CU_FUNC_ATTRIBUTE_REQUIRED_CLUSTER_DEPTH = cydriver.CUfunction_attribute_enum.CU_FUNC_ATTRIBUTE_REQUIRED_CLUSTER_DEPTH

    #: Whether the function can be launched with non-portable cluster size.
    #: 1 is allowed, 0 is disallowed. A non-portable cluster size may only
    #: function on the specific SKUs the program is tested on. The launch
    #: might fail if the program is run on a different hardware platform.
    #:
    #: CUDA API provides cudaOccupancyMaxActiveClusters to assist with
    #: checking whether the desired size can be launched on the current
    #: device.
    #:
    #: Portable Cluster Size
    #:
    #: A portable cluster size is guaranteed to be functional on all
    #: compute capabilities higher than the target compute capability. The
    #: portable cluster size for sm_90 is 8 blocks per cluster. This value
    #: may increase for future compute capabilities.
    #:
    #: The specific hardware unit may support higher cluster sizes that’s
    #: not guaranteed to be portable. See :py:obj:`~.cuFuncSetAttribute`
    CU_FUNC_ATTRIBUTE_NON_PORTABLE_CLUSTER_SIZE_ALLOWED = cydriver.CUfunction_attribute_enum.CU_FUNC_ATTRIBUTE_NON_PORTABLE_CLUSTER_SIZE_ALLOWED

    #: The block scheduling policy of a function. The value type is
    #: CUclusterSchedulingPolicy / cudaClusterSchedulingPolicy. See
    #: :py:obj:`~.cuFuncSetAttribute`
    CU_FUNC_ATTRIBUTE_CLUSTER_SCHEDULING_POLICY_PREFERENCE = cydriver.CUfunction_attribute_enum.CU_FUNC_ATTRIBUTE_CLUSTER_SCHEDULING_POLICY_PREFERENCE
    CU_FUNC_ATTRIBUTE_MAX = cydriver.CUfunction_attribute_enum.CU_FUNC_ATTRIBUTE_MAX

class CUfunc_cache(IntEnum):
    """
    Function cache configurations
    """

    #: no preference for shared memory or L1 (default)
    CU_FUNC_CACHE_PREFER_NONE = cydriver.CUfunc_cache_enum.CU_FUNC_CACHE_PREFER_NONE

    #: prefer larger shared memory and smaller L1 cache
    CU_FUNC_CACHE_PREFER_SHARED = cydriver.CUfunc_cache_enum.CU_FUNC_CACHE_PREFER_SHARED

    #: prefer larger L1 cache and smaller shared memory
    CU_FUNC_CACHE_PREFER_L1 = cydriver.CUfunc_cache_enum.CU_FUNC_CACHE_PREFER_L1

    #: prefer equal sized L1 cache and shared memory
    CU_FUNC_CACHE_PREFER_EQUAL = cydriver.CUfunc_cache_enum.CU_FUNC_CACHE_PREFER_EQUAL

class CUsharedconfig(IntEnum):
    """
    Shared memory configurations
    """

    #: set default shared memory bank size
    CU_SHARED_MEM_CONFIG_DEFAULT_BANK_SIZE = cydriver.CUsharedconfig_enum.CU_SHARED_MEM_CONFIG_DEFAULT_BANK_SIZE

    #: set shared memory bank width to four bytes
    CU_SHARED_MEM_CONFIG_FOUR_BYTE_BANK_SIZE = cydriver.CUsharedconfig_enum.CU_SHARED_MEM_CONFIG_FOUR_BYTE_BANK_SIZE

    #: set shared memory bank width to eight bytes
    CU_SHARED_MEM_CONFIG_EIGHT_BYTE_BANK_SIZE = cydriver.CUsharedconfig_enum.CU_SHARED_MEM_CONFIG_EIGHT_BYTE_BANK_SIZE

class CUshared_carveout(IntEnum):
    """
    Shared memory carveout configurations. These may be passed to
    :py:obj:`~.cuFuncSetAttribute`
    """

    #: No preference for shared memory or L1 (default)
    CU_SHAREDMEM_CARVEOUT_DEFAULT = cydriver.CUshared_carveout_enum.CU_SHAREDMEM_CARVEOUT_DEFAULT

    #: Prefer maximum available L1 cache, minimum shared memory
    CU_SHAREDMEM_CARVEOUT_MAX_L1 = cydriver.CUshared_carveout_enum.CU_SHAREDMEM_CARVEOUT_MAX_L1

    #: Prefer maximum available shared memory, minimum L1 cache
    CU_SHAREDMEM_CARVEOUT_MAX_SHARED = cydriver.CUshared_carveout_enum.CU_SHAREDMEM_CARVEOUT_MAX_SHARED

class CUmemorytype(IntEnum):
    """
    Memory types
    """

    #: Host memory
    CU_MEMORYTYPE_HOST = cydriver.CUmemorytype_enum.CU_MEMORYTYPE_HOST

    #: Device memory
    CU_MEMORYTYPE_DEVICE = cydriver.CUmemorytype_enum.CU_MEMORYTYPE_DEVICE

    #: Array memory
    CU_MEMORYTYPE_ARRAY = cydriver.CUmemorytype_enum.CU_MEMORYTYPE_ARRAY

    #: Unified device or host memory
    CU_MEMORYTYPE_UNIFIED = cydriver.CUmemorytype_enum.CU_MEMORYTYPE_UNIFIED

class CUcomputemode(IntEnum):
    """
    Compute Modes
    """

    #: Default compute mode (Multiple contexts allowed per device)
    CU_COMPUTEMODE_DEFAULT = cydriver.CUcomputemode_enum.CU_COMPUTEMODE_DEFAULT

    #: Compute-prohibited mode (No contexts can be created on this device
    #: at this time)
    CU_COMPUTEMODE_PROHIBITED = cydriver.CUcomputemode_enum.CU_COMPUTEMODE_PROHIBITED

    #: Compute-exclusive-process mode (Only one context used by a single
    #: process can be present on this device at a time)
    CU_COMPUTEMODE_EXCLUSIVE_PROCESS = cydriver.CUcomputemode_enum.CU_COMPUTEMODE_EXCLUSIVE_PROCESS

class CUmem_advise(IntEnum):
    """
    Memory advise values
    """

    #: Data will mostly be read and only occassionally be written to
    CU_MEM_ADVISE_SET_READ_MOSTLY = cydriver.CUmem_advise_enum.CU_MEM_ADVISE_SET_READ_MOSTLY

    #: Undo the effect of :py:obj:`~.CU_MEM_ADVISE_SET_READ_MOSTLY`
    CU_MEM_ADVISE_UNSET_READ_MOSTLY = cydriver.CUmem_advise_enum.CU_MEM_ADVISE_UNSET_READ_MOSTLY

    #: Set the preferred location for the data as the specified device
    CU_MEM_ADVISE_SET_PREFERRED_LOCATION = cydriver.CUmem_advise_enum.CU_MEM_ADVISE_SET_PREFERRED_LOCATION

    #: Clear the preferred location for the data
    CU_MEM_ADVISE_UNSET_PREFERRED_LOCATION = cydriver.CUmem_advise_enum.CU_MEM_ADVISE_UNSET_PREFERRED_LOCATION

    #: Data will be accessed by the specified device, so prevent page
    #: faults as much as possible
    CU_MEM_ADVISE_SET_ACCESSED_BY = cydriver.CUmem_advise_enum.CU_MEM_ADVISE_SET_ACCESSED_BY

    #: Let the Unified Memory subsystem decide on the page faulting policy
    #: for the specified device
    CU_MEM_ADVISE_UNSET_ACCESSED_BY = cydriver.CUmem_advise_enum.CU_MEM_ADVISE_UNSET_ACCESSED_BY

class CUmem_range_attribute(IntEnum):
    """

    """

    #: Whether the range will mostly be read and only occassionally be
    #: written to
    CU_MEM_RANGE_ATTRIBUTE_READ_MOSTLY = cydriver.CUmem_range_attribute_enum.CU_MEM_RANGE_ATTRIBUTE_READ_MOSTLY

    #: The preferred location of the range
    CU_MEM_RANGE_ATTRIBUTE_PREFERRED_LOCATION = cydriver.CUmem_range_attribute_enum.CU_MEM_RANGE_ATTRIBUTE_PREFERRED_LOCATION

    #: Memory range has :py:obj:`~.CU_MEM_ADVISE_SET_ACCESSED_BY` set for
    #: specified device
    CU_MEM_RANGE_ATTRIBUTE_ACCESSED_BY = cydriver.CUmem_range_attribute_enum.CU_MEM_RANGE_ATTRIBUTE_ACCESSED_BY

    #: The last location to which the range was prefetched
    CU_MEM_RANGE_ATTRIBUTE_LAST_PREFETCH_LOCATION = cydriver.CUmem_range_attribute_enum.CU_MEM_RANGE_ATTRIBUTE_LAST_PREFETCH_LOCATION

class CUjit_option(IntEnum):
    """
    Online compiler and linker options
    """

    #: Max number of registers that a thread may use.
    #: Option type: unsigned int
    #: Applies to: compiler only
    CU_JIT_MAX_REGISTERS = cydriver.CUjit_option_enum.CU_JIT_MAX_REGISTERS

    #: IN: Specifies minimum number of threads per block to target
    #: compilation for
    #: OUT: Returns the number of threads the compiler actually targeted.
    #: This restricts the resource utilization fo the compiler (e.g. max
    #: registers) such that a block with the given number of threads should
    #: be able to launch based on register limitations. Note, this option
    #: does not currently take into account any other resource limitations,
    #: such as shared memory utilization.
    #: Cannot be combined with :py:obj:`~.CU_JIT_TARGET`.
    #: Option type: unsigned int
    #: Applies to: compiler only
    CU_JIT_THREADS_PER_BLOCK = cydriver.CUjit_option_enum.CU_JIT_THREADS_PER_BLOCK

    #: Overwrites the option value with the total wall clock time, in
    #: milliseconds, spent in the compiler and linker
    #: Option type: float
    #: Applies to: compiler and linker
    CU_JIT_WALL_TIME = cydriver.CUjit_option_enum.CU_JIT_WALL_TIME

    #: Pointer to a buffer in which to print any log messages that are
    #: informational in nature (the buffer size is specified via option
    #: :py:obj:`~.CU_JIT_INFO_LOG_BUFFER_SIZE_BYTES`)
    #: Option type: char *
    #: Applies to: compiler and linker
    CU_JIT_INFO_LOG_BUFFER = cydriver.CUjit_option_enum.CU_JIT_INFO_LOG_BUFFER

    #: IN: Log buffer size in bytes. Log messages will be capped at this
    #: size (including null terminator)
    #: OUT: Amount of log buffer filled with messages
    #: Option type: unsigned int
    #: Applies to: compiler and linker
    CU_JIT_INFO_LOG_BUFFER_SIZE_BYTES = cydriver.CUjit_option_enum.CU_JIT_INFO_LOG_BUFFER_SIZE_BYTES

    #: Pointer to a buffer in which to print any log messages that reflect
    #: errors (the buffer size is specified via option
    #: :py:obj:`~.CU_JIT_ERROR_LOG_BUFFER_SIZE_BYTES`)
    #: Option type: char *
    #: Applies to: compiler and linker
    CU_JIT_ERROR_LOG_BUFFER = cydriver.CUjit_option_enum.CU_JIT_ERROR_LOG_BUFFER

    #: IN: Log buffer size in bytes. Log messages will be capped at this
    #: size (including null terminator)
    #: OUT: Amount of log buffer filled with messages
    #: Option type: unsigned int
    #: Applies to: compiler and linker
    CU_JIT_ERROR_LOG_BUFFER_SIZE_BYTES = cydriver.CUjit_option_enum.CU_JIT_ERROR_LOG_BUFFER_SIZE_BYTES

    #: Level of optimizations to apply to generated code (0 - 4), with 4
    #: being the default and highest level of optimizations.
    #: Option type: unsigned int
    #: Applies to: compiler only
    CU_JIT_OPTIMIZATION_LEVEL = cydriver.CUjit_option_enum.CU_JIT_OPTIMIZATION_LEVEL

    #: No option value required. Determines the target based on the current
    #: attached context (default)
    #: Option type: No option value needed
    #: Applies to: compiler and linker
    CU_JIT_TARGET_FROM_CUCONTEXT = cydriver.CUjit_option_enum.CU_JIT_TARGET_FROM_CUCONTEXT

    #: Target is chosen based on supplied :py:obj:`~.CUjit_target`. Cannot
    #: be combined with :py:obj:`~.CU_JIT_THREADS_PER_BLOCK`.
    #: Option type: unsigned int for enumerated type
    #: :py:obj:`~.CUjit_target`
    #: Applies to: compiler and linker
    CU_JIT_TARGET = cydriver.CUjit_option_enum.CU_JIT_TARGET

    #: Specifies choice of fallback strategy if matching cubin is not
    #: found. Choice is based on supplied :py:obj:`~.CUjit_fallback`. This
    #: option cannot be used with cuLink* APIs as the linker requires exact
    #: matches.
    #: Option type: unsigned int for enumerated type
    #: :py:obj:`~.CUjit_fallback`
    #: Applies to: compiler only
    CU_JIT_FALLBACK_STRATEGY = cydriver.CUjit_option_enum.CU_JIT_FALLBACK_STRATEGY

    #: Specifies whether to create debug information in output (-g) (0:
    #: false, default)
    #: Option type: int
    #: Applies to: compiler and linker
    CU_JIT_GENERATE_DEBUG_INFO = cydriver.CUjit_option_enum.CU_JIT_GENERATE_DEBUG_INFO

    #: Generate verbose log messages (0: false, default)
    #: Option type: int
    #: Applies to: compiler and linker
    CU_JIT_LOG_VERBOSE = cydriver.CUjit_option_enum.CU_JIT_LOG_VERBOSE

    #: Generate line number information (-lineinfo) (0: false, default)
    #: Option type: int
    #: Applies to: compiler only
    CU_JIT_GENERATE_LINE_INFO = cydriver.CUjit_option_enum.CU_JIT_GENERATE_LINE_INFO

    #: Specifies whether to enable caching explicitly (-dlcm)
    #: Choice is based on supplied :py:obj:`~.CUjit_cacheMode_enum`.
    #: Option type: unsigned int for enumerated type
    #: :py:obj:`~.CUjit_cacheMode_enum`
    #: Applies to: compiler only
    CU_JIT_CACHE_MODE = cydriver.CUjit_option_enum.CU_JIT_CACHE_MODE

    #: [Deprecated]
    CU_JIT_NEW_SM3X_OPT = cydriver.CUjit_option_enum.CU_JIT_NEW_SM3X_OPT

    #: This jit option is used for internal purpose only.
    CU_JIT_FAST_COMPILE = cydriver.CUjit_option_enum.CU_JIT_FAST_COMPILE

    #: Array of device symbol names that will be relocated to the
    #: corresponing host addresses stored in
    #: :py:obj:`~.CU_JIT_GLOBAL_SYMBOL_ADDRESSES`.
    #: Must contain :py:obj:`~.CU_JIT_GLOBAL_SYMBOL_COUNT` entries.
    #: When loding a device module, driver will relocate all encountered
    #: unresolved symbols to the host addresses.
    #: It is only allowed to register symbols that correspond to unresolved
    #: global variables.
    #: It is illegal to register the same device symbol at multiple
    #: addresses.
    #: Option type: const char **
    #: Applies to: dynamic linker only
    CU_JIT_GLOBAL_SYMBOL_NAMES = cydriver.CUjit_option_enum.CU_JIT_GLOBAL_SYMBOL_NAMES

    #: Array of host addresses that will be used to relocate corresponding
    #: device symbols stored in :py:obj:`~.CU_JIT_GLOBAL_SYMBOL_NAMES`.
    #: Must contain :py:obj:`~.CU_JIT_GLOBAL_SYMBOL_COUNT` entries.
    #: Option type: void **
    #: Applies to: dynamic linker only
    CU_JIT_GLOBAL_SYMBOL_ADDRESSES = cydriver.CUjit_option_enum.CU_JIT_GLOBAL_SYMBOL_ADDRESSES

    #: Number of entries in :py:obj:`~.CU_JIT_GLOBAL_SYMBOL_NAMES` and
    #: :py:obj:`~.CU_JIT_GLOBAL_SYMBOL_ADDRESSES` arrays.
    #: Option type: unsigned int
    #: Applies to: dynamic linker only
    CU_JIT_GLOBAL_SYMBOL_COUNT = cydriver.CUjit_option_enum.CU_JIT_GLOBAL_SYMBOL_COUNT

    #: Enable link-time optimization (-dlto) for device code (Disabled by
    #: default).
    #: This option is not supported on 32-bit platforms.
    #: Option type: int
    #: Applies to: compiler and linker
    CU_JIT_LTO = cydriver.CUjit_option_enum.CU_JIT_LTO

    #: Control single-precision denormals (-ftz) support (0: false,
    #: default). 1 : flushes denormal values to zero 0 : preserves denormal
    #: values Option type: int
    #: Applies to: link-time optimization specified with CU_JIT_LTO
    CU_JIT_FTZ = cydriver.CUjit_option_enum.CU_JIT_FTZ

    #: Control single-precision floating-point division and reciprocals
    #: (-prec-div) support (1: true, default). 1 : Enables the IEEE round-
    #: to-nearest mode 0 : Enables the fast approximation mode Option type:
    #: int
    #: Applies to: link-time optimization specified with CU_JIT_LTO
    CU_JIT_PREC_DIV = cydriver.CUjit_option_enum.CU_JIT_PREC_DIV

    #: Control single-precision floating-point square root (-prec-sqrt)
    #: support (1: true, default). 1 : Enables the IEEE round-to-nearest
    #: mode 0 : Enables the fast approximation mode Option type: int
    #: Applies to: link-time optimization specified with CU_JIT_LTO
    CU_JIT_PREC_SQRT = cydriver.CUjit_option_enum.CU_JIT_PREC_SQRT

    #: Enable/Disable the contraction of floating-point multiplies and
    #: adds/subtracts into floating-point multiply-add (-fma) operations
    #: (1: Enable, default; 0: Disable). Option type: int
    #: Applies to: link-time optimization specified with CU_JIT_LTO
    CU_JIT_FMA = cydriver.CUjit_option_enum.CU_JIT_FMA

    #: Array of kernel names that should be preserved at link time while
    #: others can be removed.
    #: Must contain :py:obj:`~.CU_JIT_REFERENCED_KERNEL_COUNT` entries.
    #: Note that kernel names can be mangled by the compiler in which case
    #: the mangled name needs to be specified.
    #: Wildcard "*" can be used to represent zero or more characters
    #: instead of specifying the full or mangled name.
    #: It is important to note that the wildcard "*" is also added
    #: implicitly. For example, specifying "foo" will match "foobaz",
    #: "barfoo", "barfoobaz" and thus preserve all kernels with those
    #: names. This can be avoided by providing a more specific name like
    #: "barfoobaz".
    #: Option type: const char **
    #: Applies to: dynamic linker only
    CU_JIT_REFERENCED_KERNEL_NAMES = cydriver.CUjit_option_enum.CU_JIT_REFERENCED_KERNEL_NAMES

    #: Number of entries in :py:obj:`~.CU_JIT_REFERENCED_KERNEL_NAMES`
    #: array.
    #: Option type: unsigned int
    #: Applies to: dynamic linker only
    CU_JIT_REFERENCED_KERNEL_COUNT = cydriver.CUjit_option_enum.CU_JIT_REFERENCED_KERNEL_COUNT

    #: Array of variable names (device and/or constant) that should be
    #: preserved at link time while others can be removed.
    #: Must contain :py:obj:`~.CU_JIT_REFERENCED_VARIABLE_COUNT` entries.
    #: Note that variable names can be mangled by the compiler in which
    #: case the mangled name needs to be specified.
    #: Wildcard "*" can be used to represent zero or more characters
    #: instead of specifying the full or mangled name.
    #: It is important to note that the wildcard "*" is also added
    #: implicitly. For example, specifying "foo" will match "foobaz",
    #: "barfoo", "barfoobaz" and thus preserve all variables with those
    #: names. This can be avoided by providing a more specific name like
    #: "barfoobaz".
    #: Option type: const char **
    #: Applies to: link-time optimization specified with CU_JIT_LTO
    CU_JIT_REFERENCED_VARIABLE_NAMES = cydriver.CUjit_option_enum.CU_JIT_REFERENCED_VARIABLE_NAMES

    #: Number of entries in :py:obj:`~.CU_JIT_REFERENCED_VARIABLE_NAMES`
    #: array.
    #: Option type: unsigned int
    #: Applies to: link-time optimization specified with CU_JIT_LTO
    CU_JIT_REFERENCED_VARIABLE_COUNT = cydriver.CUjit_option_enum.CU_JIT_REFERENCED_VARIABLE_COUNT

    #: This option serves as a hint to enable the JIT compiler/linker to
    #: remove constant (constant) and device (device) variables
    #: unreferenced in device code (Disabled by default).
    #: Note that host references to constant and device variables using
    #: APIs like :py:obj:`~.cuModuleGetGlobal()` with this option specified
    #: may result in undefined behavior unless the variables are explicitly
    #: specified using :py:obj:`~.CU_JIT_REFERENCED_VARIABLE_NAMES`.
    #: Option type: int
    #: Applies to: link-time optimization specified with CU_JIT_LTO
    CU_JIT_OPTIMIZE_UNUSED_DEVICE_VARIABLES = cydriver.CUjit_option_enum.CU_JIT_OPTIMIZE_UNUSED_DEVICE_VARIABLES
    CU_JIT_NUM_OPTIONS = cydriver.CUjit_option_enum.CU_JIT_NUM_OPTIONS

class CUjit_target(IntEnum):
    """
    Online compilation targets
    """

    #: Compute device class 2.0
    CU_TARGET_COMPUTE_20 = cydriver.CUjit_target_enum.CU_TARGET_COMPUTE_20

    #: Compute device class 2.1
    CU_TARGET_COMPUTE_21 = cydriver.CUjit_target_enum.CU_TARGET_COMPUTE_21

    #: Compute device class 3.0
    CU_TARGET_COMPUTE_30 = cydriver.CUjit_target_enum.CU_TARGET_COMPUTE_30

    #: Compute device class 3.2
    CU_TARGET_COMPUTE_32 = cydriver.CUjit_target_enum.CU_TARGET_COMPUTE_32

    #: Compute device class 3.5
    CU_TARGET_COMPUTE_35 = cydriver.CUjit_target_enum.CU_TARGET_COMPUTE_35

    #: Compute device class 3.7
    CU_TARGET_COMPUTE_37 = cydriver.CUjit_target_enum.CU_TARGET_COMPUTE_37

    #: Compute device class 5.0
    CU_TARGET_COMPUTE_50 = cydriver.CUjit_target_enum.CU_TARGET_COMPUTE_50

    #: Compute device class 5.2
    CU_TARGET_COMPUTE_52 = cydriver.CUjit_target_enum.CU_TARGET_COMPUTE_52

    #: Compute device class 5.3
    CU_TARGET_COMPUTE_53 = cydriver.CUjit_target_enum.CU_TARGET_COMPUTE_53

    #: Compute device class 6.0.
    CU_TARGET_COMPUTE_60 = cydriver.CUjit_target_enum.CU_TARGET_COMPUTE_60

    #: Compute device class 6.1.
    CU_TARGET_COMPUTE_61 = cydriver.CUjit_target_enum.CU_TARGET_COMPUTE_61

    #: Compute device class 6.2.
    CU_TARGET_COMPUTE_62 = cydriver.CUjit_target_enum.CU_TARGET_COMPUTE_62

    #: Compute device class 7.0.
    CU_TARGET_COMPUTE_70 = cydriver.CUjit_target_enum.CU_TARGET_COMPUTE_70

    #: Compute device class 7.2.
    CU_TARGET_COMPUTE_72 = cydriver.CUjit_target_enum.CU_TARGET_COMPUTE_72

    #: Compute device class 7.5.
    CU_TARGET_COMPUTE_75 = cydriver.CUjit_target_enum.CU_TARGET_COMPUTE_75

    #: Compute device class 8.0.
    CU_TARGET_COMPUTE_80 = cydriver.CUjit_target_enum.CU_TARGET_COMPUTE_80

    #: Compute device class 8.6.
    CU_TARGET_COMPUTE_86 = cydriver.CUjit_target_enum.CU_TARGET_COMPUTE_86

    #: Compute device class 8.7.
    CU_TARGET_COMPUTE_87 = cydriver.CUjit_target_enum.CU_TARGET_COMPUTE_87

    #: Compute device class 8.9.
    CU_TARGET_COMPUTE_89 = cydriver.CUjit_target_enum.CU_TARGET_COMPUTE_89

    #: Compute device class 9.0.
    CU_TARGET_COMPUTE_90 = cydriver.CUjit_target_enum.CU_TARGET_COMPUTE_90

class CUjit_fallback(IntEnum):
    """
    Cubin matching fallback strategies
    """

    #: Prefer to compile ptx if exact binary match not found
    CU_PREFER_PTX = cydriver.CUjit_fallback_enum.CU_PREFER_PTX

    #: Prefer to fall back to compatible binary code if exact match not
    #: found
    CU_PREFER_BINARY = cydriver.CUjit_fallback_enum.CU_PREFER_BINARY

class CUjit_cacheMode(IntEnum):
    """
    Caching modes for dlcm
    """

    #: Compile with no -dlcm flag specified
    CU_JIT_CACHE_OPTION_NONE = cydriver.CUjit_cacheMode_enum.CU_JIT_CACHE_OPTION_NONE

    #: Compile with L1 cache disabled
    CU_JIT_CACHE_OPTION_CG = cydriver.CUjit_cacheMode_enum.CU_JIT_CACHE_OPTION_CG

    #: Compile with L1 cache enabled
    CU_JIT_CACHE_OPTION_CA = cydriver.CUjit_cacheMode_enum.CU_JIT_CACHE_OPTION_CA

class CUjitInputType(IntEnum):
    """
    Device code formats
    """

    #: Compiled device-class-specific device code
    #: Applicable options: none
    CU_JIT_INPUT_CUBIN = cydriver.CUjitInputType_enum.CU_JIT_INPUT_CUBIN

    #: PTX source code
    #: Applicable options: PTX compiler options
    CU_JIT_INPUT_PTX = cydriver.CUjitInputType_enum.CU_JIT_INPUT_PTX

    #: Bundle of multiple cubins and/or PTX of some device code
    #: Applicable options: PTX compiler options,
    #: :py:obj:`~.CU_JIT_FALLBACK_STRATEGY`
    CU_JIT_INPUT_FATBINARY = cydriver.CUjitInputType_enum.CU_JIT_INPUT_FATBINARY

    #: Host object with embedded device code
    #: Applicable options: PTX compiler options,
    #: :py:obj:`~.CU_JIT_FALLBACK_STRATEGY`
    CU_JIT_INPUT_OBJECT = cydriver.CUjitInputType_enum.CU_JIT_INPUT_OBJECT

    #: Archive of host objects with embedded device code
    #: Applicable options: PTX compiler options,
    #: :py:obj:`~.CU_JIT_FALLBACK_STRATEGY`
    CU_JIT_INPUT_LIBRARY = cydriver.CUjitInputType_enum.CU_JIT_INPUT_LIBRARY

    #: High-level intermediate code for link-time optimization
    #: Applicable options: NVVM compiler options, PTX compiler options
    CU_JIT_INPUT_NVVM = cydriver.CUjitInputType_enum.CU_JIT_INPUT_NVVM
    CU_JIT_NUM_INPUT_TYPES = cydriver.CUjitInputType_enum.CU_JIT_NUM_INPUT_TYPES

class CUgraphicsRegisterFlags(IntEnum):
    """
    Flags to register a graphics resource
    """
    CU_GRAPHICS_REGISTER_FLAGS_NONE = cydriver.CUgraphicsRegisterFlags_enum.CU_GRAPHICS_REGISTER_FLAGS_NONE
    CU_GRAPHICS_REGISTER_FLAGS_READ_ONLY = cydriver.CUgraphicsRegisterFlags_enum.CU_GRAPHICS_REGISTER_FLAGS_READ_ONLY
    CU_GRAPHICS_REGISTER_FLAGS_WRITE_DISCARD = cydriver.CUgraphicsRegisterFlags_enum.CU_GRAPHICS_REGISTER_FLAGS_WRITE_DISCARD
    CU_GRAPHICS_REGISTER_FLAGS_SURFACE_LDST = cydriver.CUgraphicsRegisterFlags_enum.CU_GRAPHICS_REGISTER_FLAGS_SURFACE_LDST
    CU_GRAPHICS_REGISTER_FLAGS_TEXTURE_GATHER = cydriver.CUgraphicsRegisterFlags_enum.CU_GRAPHICS_REGISTER_FLAGS_TEXTURE_GATHER

class CUgraphicsMapResourceFlags(IntEnum):
    """
    Flags for mapping and unmapping interop resources
    """
    CU_GRAPHICS_MAP_RESOURCE_FLAGS_NONE = cydriver.CUgraphicsMapResourceFlags_enum.CU_GRAPHICS_MAP_RESOURCE_FLAGS_NONE
    CU_GRAPHICS_MAP_RESOURCE_FLAGS_READ_ONLY = cydriver.CUgraphicsMapResourceFlags_enum.CU_GRAPHICS_MAP_RESOURCE_FLAGS_READ_ONLY
    CU_GRAPHICS_MAP_RESOURCE_FLAGS_WRITE_DISCARD = cydriver.CUgraphicsMapResourceFlags_enum.CU_GRAPHICS_MAP_RESOURCE_FLAGS_WRITE_DISCARD

class CUarray_cubemap_face(IntEnum):
    """
    Array indices for cube faces
    """

    #: Positive X face of cubemap
    CU_CUBEMAP_FACE_POSITIVE_X = cydriver.CUarray_cubemap_face_enum.CU_CUBEMAP_FACE_POSITIVE_X

    #: Negative X face of cubemap
    CU_CUBEMAP_FACE_NEGATIVE_X = cydriver.CUarray_cubemap_face_enum.CU_CUBEMAP_FACE_NEGATIVE_X

    #: Positive Y face of cubemap
    CU_CUBEMAP_FACE_POSITIVE_Y = cydriver.CUarray_cubemap_face_enum.CU_CUBEMAP_FACE_POSITIVE_Y

    #: Negative Y face of cubemap
    CU_CUBEMAP_FACE_NEGATIVE_Y = cydriver.CUarray_cubemap_face_enum.CU_CUBEMAP_FACE_NEGATIVE_Y

    #: Positive Z face of cubemap
    CU_CUBEMAP_FACE_POSITIVE_Z = cydriver.CUarray_cubemap_face_enum.CU_CUBEMAP_FACE_POSITIVE_Z

    #: Negative Z face of cubemap
    CU_CUBEMAP_FACE_NEGATIVE_Z = cydriver.CUarray_cubemap_face_enum.CU_CUBEMAP_FACE_NEGATIVE_Z

class CUlimit(IntEnum):
    """
    Limits
    """

    #: GPU thread stack size
    CU_LIMIT_STACK_SIZE = cydriver.CUlimit_enum.CU_LIMIT_STACK_SIZE

    #: GPU printf FIFO size
    CU_LIMIT_PRINTF_FIFO_SIZE = cydriver.CUlimit_enum.CU_LIMIT_PRINTF_FIFO_SIZE

    #: GPU malloc heap size
    CU_LIMIT_MALLOC_HEAP_SIZE = cydriver.CUlimit_enum.CU_LIMIT_MALLOC_HEAP_SIZE

    #: GPU device runtime launch synchronize depth
    CU_LIMIT_DEV_RUNTIME_SYNC_DEPTH = cydriver.CUlimit_enum.CU_LIMIT_DEV_RUNTIME_SYNC_DEPTH

    #: GPU device runtime pending launch count
    CU_LIMIT_DEV_RUNTIME_PENDING_LAUNCH_COUNT = cydriver.CUlimit_enum.CU_LIMIT_DEV_RUNTIME_PENDING_LAUNCH_COUNT

    #: A value between 0 and 128 that indicates the maximum fetch
    #: granularity of L2 (in Bytes). This is a hint
    CU_LIMIT_MAX_L2_FETCH_GRANULARITY = cydriver.CUlimit_enum.CU_LIMIT_MAX_L2_FETCH_GRANULARITY

    #: A size in bytes for L2 persisting lines cache size
    CU_LIMIT_PERSISTING_L2_CACHE_SIZE = cydriver.CUlimit_enum.CU_LIMIT_PERSISTING_L2_CACHE_SIZE
    CU_LIMIT_MAX = cydriver.CUlimit_enum.CU_LIMIT_MAX

class CUresourcetype(IntEnum):
    """
    Resource types
    """

    #: Array resoure
    CU_RESOURCE_TYPE_ARRAY = cydriver.CUresourcetype_enum.CU_RESOURCE_TYPE_ARRAY

    #: Mipmapped array resource
    CU_RESOURCE_TYPE_MIPMAPPED_ARRAY = cydriver.CUresourcetype_enum.CU_RESOURCE_TYPE_MIPMAPPED_ARRAY

    #: Linear resource
    CU_RESOURCE_TYPE_LINEAR = cydriver.CUresourcetype_enum.CU_RESOURCE_TYPE_LINEAR

    #: Pitch 2D resource
    CU_RESOURCE_TYPE_PITCH2D = cydriver.CUresourcetype_enum.CU_RESOURCE_TYPE_PITCH2D

class CUaccessProperty(IntEnum):
    """
    Specifies performance hint with :py:obj:`~.CUaccessPolicyWindow`
    for hitProp and missProp members.
    """

    #: Normal cache persistence.
    CU_ACCESS_PROPERTY_NORMAL = cydriver.CUaccessProperty_enum.CU_ACCESS_PROPERTY_NORMAL

    #: Streaming access is less likely to persit from cache.
    CU_ACCESS_PROPERTY_STREAMING = cydriver.CUaccessProperty_enum.CU_ACCESS_PROPERTY_STREAMING

    #: Persisting access is more likely to persist in cache.
    CU_ACCESS_PROPERTY_PERSISTING = cydriver.CUaccessProperty_enum.CU_ACCESS_PROPERTY_PERSISTING

class CUgraphNodeType(IntEnum):
    """
    Graph node types
    """

    #: GPU kernel node
    CU_GRAPH_NODE_TYPE_KERNEL = cydriver.CUgraphNodeType_enum.CU_GRAPH_NODE_TYPE_KERNEL

    #: Memcpy node
    CU_GRAPH_NODE_TYPE_MEMCPY = cydriver.CUgraphNodeType_enum.CU_GRAPH_NODE_TYPE_MEMCPY

    #: Memset node
    CU_GRAPH_NODE_TYPE_MEMSET = cydriver.CUgraphNodeType_enum.CU_GRAPH_NODE_TYPE_MEMSET

    #: Host (executable) node
    CU_GRAPH_NODE_TYPE_HOST = cydriver.CUgraphNodeType_enum.CU_GRAPH_NODE_TYPE_HOST

    #: Node which executes an embedded graph
    CU_GRAPH_NODE_TYPE_GRAPH = cydriver.CUgraphNodeType_enum.CU_GRAPH_NODE_TYPE_GRAPH

    #: Empty (no-op) node
    CU_GRAPH_NODE_TYPE_EMPTY = cydriver.CUgraphNodeType_enum.CU_GRAPH_NODE_TYPE_EMPTY

    #: External event wait node
    CU_GRAPH_NODE_TYPE_WAIT_EVENT = cydriver.CUgraphNodeType_enum.CU_GRAPH_NODE_TYPE_WAIT_EVENT

    #: External event record node
    CU_GRAPH_NODE_TYPE_EVENT_RECORD = cydriver.CUgraphNodeType_enum.CU_GRAPH_NODE_TYPE_EVENT_RECORD

    #: External semaphore signal node
    CU_GRAPH_NODE_TYPE_EXT_SEMAS_SIGNAL = cydriver.CUgraphNodeType_enum.CU_GRAPH_NODE_TYPE_EXT_SEMAS_SIGNAL

    #: External semaphore wait node
    CU_GRAPH_NODE_TYPE_EXT_SEMAS_WAIT = cydriver.CUgraphNodeType_enum.CU_GRAPH_NODE_TYPE_EXT_SEMAS_WAIT

    #: Memory Allocation Node
    CU_GRAPH_NODE_TYPE_MEM_ALLOC = cydriver.CUgraphNodeType_enum.CU_GRAPH_NODE_TYPE_MEM_ALLOC

    #: Memory Free Node
    CU_GRAPH_NODE_TYPE_MEM_FREE = cydriver.CUgraphNodeType_enum.CU_GRAPH_NODE_TYPE_MEM_FREE

    #: Batch MemOp Node
    CU_GRAPH_NODE_TYPE_BATCH_MEM_OP = cydriver.CUgraphNodeType_enum.CU_GRAPH_NODE_TYPE_BATCH_MEM_OP

class CUsynchronizationPolicy(IntEnum):
    """

    """
    CU_SYNC_POLICY_AUTO = cydriver.CUsynchronizationPolicy_enum.CU_SYNC_POLICY_AUTO
    CU_SYNC_POLICY_SPIN = cydriver.CUsynchronizationPolicy_enum.CU_SYNC_POLICY_SPIN
    CU_SYNC_POLICY_YIELD = cydriver.CUsynchronizationPolicy_enum.CU_SYNC_POLICY_YIELD
    CU_SYNC_POLICY_BLOCKING_SYNC = cydriver.CUsynchronizationPolicy_enum.CU_SYNC_POLICY_BLOCKING_SYNC

class CUclusterSchedulingPolicy(IntEnum):
    """
    Cluster scheduling policies. These may be passed to
    :py:obj:`~.cuFuncSetAttribute`
    """

    #: the default policy
    CU_CLUSTER_SCHEDULING_POLICY_DEFAULT = cydriver.CUclusterSchedulingPolicy_enum.CU_CLUSTER_SCHEDULING_POLICY_DEFAULT

    #: spread the blocks within a cluster to the SMs
    CU_CLUSTER_SCHEDULING_POLICY_SPREAD = cydriver.CUclusterSchedulingPolicy_enum.CU_CLUSTER_SCHEDULING_POLICY_SPREAD

    #: allow the hardware to load-balance the blocks in a cluster to the
    #: SMs
    CU_CLUSTER_SCHEDULING_POLICY_LOAD_BALANCING = cydriver.CUclusterSchedulingPolicy_enum.CU_CLUSTER_SCHEDULING_POLICY_LOAD_BALANCING

class CUlaunchAttributeID(IntEnum):
    """

    """

    #: Ignored entry, for convenient composition
    CU_LAUNCH_ATTRIBUTE_IGNORE = cydriver.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_IGNORE

    #: Valid for streams, graph nodes, launches.
    CU_LAUNCH_ATTRIBUTE_ACCESS_POLICY_WINDOW = cydriver.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_ACCESS_POLICY_WINDOW

    #: Valid for graph nodes, launches.
    CU_LAUNCH_ATTRIBUTE_COOPERATIVE = cydriver.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_COOPERATIVE

    #: Valid for streams.
    CU_LAUNCH_ATTRIBUTE_SYNCHRONIZATION_POLICY = cydriver.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_SYNCHRONIZATION_POLICY

    #: Valid for graph nodes, launches.
    CU_LAUNCH_ATTRIBUTE_CLUSTER_DIMENSION = cydriver.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_CLUSTER_DIMENSION

    #: Valid for graph nodes, launches.
    CU_LAUNCH_ATTRIBUTE_CLUSTER_SCHEDULING_POLICY_PREFERENCE = cydriver.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_CLUSTER_SCHEDULING_POLICY_PREFERENCE

    #: Valid for launches. Setting programmaticStreamSerializationAllowed
    #: to non-0 signals that the kernel will use programmatic means to
    #: resolve its stream dependency, so that the CUDA runtime should
    #: opportunistically allow the grid's execution to overlap with the
    #: previous kernel in the stream, if that kernel requests the overlap.
    CU_LAUNCH_ATTRIBUTE_PROGRAMMATIC_STREAM_SERIALIZATION = cydriver.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_PROGRAMMATIC_STREAM_SERIALIZATION

    #: Valid for launches. Event recorded through this launch attribute is
    #: guaranteed to only trigger after all block in the associated kernel
    #: trigger the event. A block can trigger the event through PTX
    #: griddepcontrol.launch_dependents. A trigger can also be inserted at
    #: the beginning of each block's execution if triggerAtBlockStart is
    #: set to non-0. Note that dependents (including the CPU thread calling
    #: :py:obj:`~.cuEventSynchronize()`) are not guaranteed to observe the
    #: release precisely when it is released. For example,
    #: :py:obj:`~.cuEventSynchronize()` may only observe the event trigger
    #: long after the associated kernel has completed. This recording type
    #: is primarily meant for establishing programmatic dependency between
    #: device tasks. The event supplied must not be an interprocess or
    #: interop event. The event must disable timing (i.e. created with
    #: :py:obj:`~.CU_EVENT_DISABLE_TIMING` flag set).
    CU_LAUNCH_ATTRIBUTE_PROGRAMMATIC_EVENT = cydriver.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_PROGRAMMATIC_EVENT

    #: Valid for graph nodes.
    CU_LAUNCH_ATTRIBUTE_PRIORITY = cydriver.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_PRIORITY

class CUstreamCaptureStatus(IntEnum):
    """
    Possible stream capture statuses returned by
    :py:obj:`~.cuStreamIsCapturing`
    """

    #: Stream is not capturing
    CU_STREAM_CAPTURE_STATUS_NONE = cydriver.CUstreamCaptureStatus_enum.CU_STREAM_CAPTURE_STATUS_NONE

    #: Stream is actively capturing
    CU_STREAM_CAPTURE_STATUS_ACTIVE = cydriver.CUstreamCaptureStatus_enum.CU_STREAM_CAPTURE_STATUS_ACTIVE

    #: Stream is part of a capture sequence that has been invalidated, but
    #: not terminated
    CU_STREAM_CAPTURE_STATUS_INVALIDATED = cydriver.CUstreamCaptureStatus_enum.CU_STREAM_CAPTURE_STATUS_INVALIDATED

class CUstreamCaptureMode(IntEnum):
    """
    Possible modes for stream capture thread interactions. For more
    details see :py:obj:`~.cuStreamBeginCapture` and
    :py:obj:`~.cuThreadExchangeStreamCaptureMode`
    """
    CU_STREAM_CAPTURE_MODE_GLOBAL = cydriver.CUstreamCaptureMode_enum.CU_STREAM_CAPTURE_MODE_GLOBAL
    CU_STREAM_CAPTURE_MODE_THREAD_LOCAL = cydriver.CUstreamCaptureMode_enum.CU_STREAM_CAPTURE_MODE_THREAD_LOCAL
    CU_STREAM_CAPTURE_MODE_RELAXED = cydriver.CUstreamCaptureMode_enum.CU_STREAM_CAPTURE_MODE_RELAXED

class CUdriverProcAddress_flags(IntEnum):
    """
    Flags to specify search options. For more details see
    :py:obj:`~.cuGetProcAddress`
    """

    #: Default search mode for driver symbols.
    CU_GET_PROC_ADDRESS_DEFAULT = cydriver.CUdriverProcAddress_flags_enum.CU_GET_PROC_ADDRESS_DEFAULT

    #: Search for legacy versions of driver symbols.
    CU_GET_PROC_ADDRESS_LEGACY_STREAM = cydriver.CUdriverProcAddress_flags_enum.CU_GET_PROC_ADDRESS_LEGACY_STREAM

    #: Search for per-thread versions of driver symbols.
    CU_GET_PROC_ADDRESS_PER_THREAD_DEFAULT_STREAM = cydriver.CUdriverProcAddress_flags_enum.CU_GET_PROC_ADDRESS_PER_THREAD_DEFAULT_STREAM

class CUexecAffinityType(IntEnum):
    """
    Execution Affinity Types
    """

    #: Create a context with limited SMs.
    CU_EXEC_AFFINITY_TYPE_SM_COUNT = cydriver.CUexecAffinityType_enum.CU_EXEC_AFFINITY_TYPE_SM_COUNT
    CU_EXEC_AFFINITY_TYPE_MAX = cydriver.CUexecAffinityType_enum.CU_EXEC_AFFINITY_TYPE_MAX

class CUresult(IntEnum):
    """
    Error codes
    """

    #: The API call returned with no errors. In the case of query calls,
    #: this also means that the operation being queried is complete (see
    #: :py:obj:`~.cuEventQuery()` and :py:obj:`~.cuStreamQuery()`).
    CUDA_SUCCESS = cydriver.cudaError_enum.CUDA_SUCCESS

    #: This indicates that one or more of the parameters passed to the API
    #: call is not within an acceptable range of values.
    CUDA_ERROR_INVALID_VALUE = cydriver.cudaError_enum.CUDA_ERROR_INVALID_VALUE

    #: The API call failed because it was unable to allocate enough memory
    #: to perform the requested operation.
    CUDA_ERROR_OUT_OF_MEMORY = cydriver.cudaError_enum.CUDA_ERROR_OUT_OF_MEMORY

    #: This indicates that the CUDA driver has not been initialized with
    #: :py:obj:`~.cuInit()` or that initialization has failed.
    CUDA_ERROR_NOT_INITIALIZED = cydriver.cudaError_enum.CUDA_ERROR_NOT_INITIALIZED

    #: This indicates that the CUDA driver is in the process of shutting
    #: down.
    CUDA_ERROR_DEINITIALIZED = cydriver.cudaError_enum.CUDA_ERROR_DEINITIALIZED

    #: This indicates profiler is not initialized for this run. This can
    #: happen when the application is running with external profiling tools
    #: like visual profiler.
    CUDA_ERROR_PROFILER_DISABLED = cydriver.cudaError_enum.CUDA_ERROR_PROFILER_DISABLED

    #: [Deprecated]
    CUDA_ERROR_PROFILER_NOT_INITIALIZED = cydriver.cudaError_enum.CUDA_ERROR_PROFILER_NOT_INITIALIZED

    #: [Deprecated]
    CUDA_ERROR_PROFILER_ALREADY_STARTED = cydriver.cudaError_enum.CUDA_ERROR_PROFILER_ALREADY_STARTED

    #: [Deprecated]
    CUDA_ERROR_PROFILER_ALREADY_STOPPED = cydriver.cudaError_enum.CUDA_ERROR_PROFILER_ALREADY_STOPPED

    #: This indicates that the CUDA driver that the application has loaded
    #: is a stub library. Applications that run with the stub rather than a
    #: real driver loaded will result in CUDA API returning this error.
    CUDA_ERROR_STUB_LIBRARY = cydriver.cudaError_enum.CUDA_ERROR_STUB_LIBRARY

    #: This indicates that requested CUDA device is unavailable at the
    #: current time. Devices are often unavailable due to use of
    #: :py:obj:`~.CU_COMPUTEMODE_EXCLUSIVE_PROCESS` or
    #: :py:obj:`~.CU_COMPUTEMODE_PROHIBITED`.
    CUDA_ERROR_DEVICE_UNAVAILABLE = cydriver.cudaError_enum.CUDA_ERROR_DEVICE_UNAVAILABLE

    #: This indicates that no CUDA-capable devices were detected by the
    #: installed CUDA driver.
    CUDA_ERROR_NO_DEVICE = cydriver.cudaError_enum.CUDA_ERROR_NO_DEVICE

    #: This indicates that the device ordinal supplied by the user does not
    #: correspond to a valid CUDA device or that the action requested is
    #: invalid for the specified device.
    CUDA_ERROR_INVALID_DEVICE = cydriver.cudaError_enum.CUDA_ERROR_INVALID_DEVICE

    #: This error indicates that the Grid license is not applied.
    CUDA_ERROR_DEVICE_NOT_LICENSED = cydriver.cudaError_enum.CUDA_ERROR_DEVICE_NOT_LICENSED

    #: This indicates that the device kernel image is invalid. This can
    #: also indicate an invalid CUDA module.
    CUDA_ERROR_INVALID_IMAGE = cydriver.cudaError_enum.CUDA_ERROR_INVALID_IMAGE

    #: This most frequently indicates that there is no context bound to the
    #: current thread. This can also be returned if the context passed to
    #: an API call is not a valid handle (such as a context that has had
    #: :py:obj:`~.cuCtxDestroy()` invoked on it). This can also be returned
    #: if a user mixes different API versions (i.e. 3010 context with 3020
    #: API calls). See :py:obj:`~.cuCtxGetApiVersion()` for more details.
    CUDA_ERROR_INVALID_CONTEXT = cydriver.cudaError_enum.CUDA_ERROR_INVALID_CONTEXT

    #: This indicated that the context being supplied as a parameter to the
    #: API call was already the active context. [Deprecated]
    CUDA_ERROR_CONTEXT_ALREADY_CURRENT = cydriver.cudaError_enum.CUDA_ERROR_CONTEXT_ALREADY_CURRENT

    #: This indicates that a map or register operation has failed.
    CUDA_ERROR_MAP_FAILED = cydriver.cudaError_enum.CUDA_ERROR_MAP_FAILED

    #: This indicates that an unmap or unregister operation has failed.
    CUDA_ERROR_UNMAP_FAILED = cydriver.cudaError_enum.CUDA_ERROR_UNMAP_FAILED

    #: This indicates that the specified array is currently mapped and thus
    #: cannot be destroyed.
    CUDA_ERROR_ARRAY_IS_MAPPED = cydriver.cudaError_enum.CUDA_ERROR_ARRAY_IS_MAPPED

    #: This indicates that the resource is already mapped.
    CUDA_ERROR_ALREADY_MAPPED = cydriver.cudaError_enum.CUDA_ERROR_ALREADY_MAPPED

    #: This indicates that there is no kernel image available that is
    #: suitable for the device. This can occur when a user specifies code
    #: generation options for a particular CUDA source file that do not
    #: include the corresponding device configuration.
    CUDA_ERROR_NO_BINARY_FOR_GPU = cydriver.cudaError_enum.CUDA_ERROR_NO_BINARY_FOR_GPU

    #: This indicates that a resource has already been acquired.
    CUDA_ERROR_ALREADY_ACQUIRED = cydriver.cudaError_enum.CUDA_ERROR_ALREADY_ACQUIRED

    #: This indicates that a resource is not mapped.
    CUDA_ERROR_NOT_MAPPED = cydriver.cudaError_enum.CUDA_ERROR_NOT_MAPPED

    #: This indicates that a mapped resource is not available for access as
    #: an array.
    CUDA_ERROR_NOT_MAPPED_AS_ARRAY = cydriver.cudaError_enum.CUDA_ERROR_NOT_MAPPED_AS_ARRAY

    #: This indicates that a mapped resource is not available for access as
    #: a pointer.
    CUDA_ERROR_NOT_MAPPED_AS_POINTER = cydriver.cudaError_enum.CUDA_ERROR_NOT_MAPPED_AS_POINTER

    #: This indicates that an uncorrectable ECC error was detected during
    #: execution.
    CUDA_ERROR_ECC_UNCORRECTABLE = cydriver.cudaError_enum.CUDA_ERROR_ECC_UNCORRECTABLE

    #: This indicates that the :py:obj:`~.CUlimit` passed to the API call
    #: is not supported by the active device.
    CUDA_ERROR_UNSUPPORTED_LIMIT = cydriver.cudaError_enum.CUDA_ERROR_UNSUPPORTED_LIMIT

    #: This indicates that the :py:obj:`~.CUcontext` passed to the API call
    #: can only be bound to a single CPU thread at a time but is already
    #: bound to a CPU thread.
    CUDA_ERROR_CONTEXT_ALREADY_IN_USE = cydriver.cudaError_enum.CUDA_ERROR_CONTEXT_ALREADY_IN_USE

    #: This indicates that peer access is not supported across the given
    #: devices.
    CUDA_ERROR_PEER_ACCESS_UNSUPPORTED = cydriver.cudaError_enum.CUDA_ERROR_PEER_ACCESS_UNSUPPORTED

    #: This indicates that a PTX JIT compilation failed.
    CUDA_ERROR_INVALID_PTX = cydriver.cudaError_enum.CUDA_ERROR_INVALID_PTX

    #: This indicates an error with OpenGL or DirectX context.
    CUDA_ERROR_INVALID_GRAPHICS_CONTEXT = cydriver.cudaError_enum.CUDA_ERROR_INVALID_GRAPHICS_CONTEXT

    #: This indicates that an uncorrectable NVLink error was detected
    #: during the execution.
    CUDA_ERROR_NVLINK_UNCORRECTABLE = cydriver.cudaError_enum.CUDA_ERROR_NVLINK_UNCORRECTABLE

    #: This indicates that the PTX JIT compiler library was not found.
    CUDA_ERROR_JIT_COMPILER_NOT_FOUND = cydriver.cudaError_enum.CUDA_ERROR_JIT_COMPILER_NOT_FOUND

    #: This indicates that the provided PTX was compiled with an
    #: unsupported toolchain.
    CUDA_ERROR_UNSUPPORTED_PTX_VERSION = cydriver.cudaError_enum.CUDA_ERROR_UNSUPPORTED_PTX_VERSION

    #: This indicates that the PTX JIT compilation was disabled.
    CUDA_ERROR_JIT_COMPILATION_DISABLED = cydriver.cudaError_enum.CUDA_ERROR_JIT_COMPILATION_DISABLED

    #: This indicates that the :py:obj:`~.CUexecAffinityType` passed to the
    #: API call is not supported by the active device.
    CUDA_ERROR_UNSUPPORTED_EXEC_AFFINITY = cydriver.cudaError_enum.CUDA_ERROR_UNSUPPORTED_EXEC_AFFINITY

    #: This indicates that the device kernel source is invalid. This
    #: includes compilation/linker errors encountered in device code or
    #: user error.
    CUDA_ERROR_INVALID_SOURCE = cydriver.cudaError_enum.CUDA_ERROR_INVALID_SOURCE

    #: This indicates that the file specified was not found.
    CUDA_ERROR_FILE_NOT_FOUND = cydriver.cudaError_enum.CUDA_ERROR_FILE_NOT_FOUND

    #: This indicates that a link to a shared object failed to resolve.
    CUDA_ERROR_SHARED_OBJECT_SYMBOL_NOT_FOUND = cydriver.cudaError_enum.CUDA_ERROR_SHARED_OBJECT_SYMBOL_NOT_FOUND

    #: This indicates that initialization of a shared object failed.
    CUDA_ERROR_SHARED_OBJECT_INIT_FAILED = cydriver.cudaError_enum.CUDA_ERROR_SHARED_OBJECT_INIT_FAILED

    #: This indicates that an OS call failed.
    CUDA_ERROR_OPERATING_SYSTEM = cydriver.cudaError_enum.CUDA_ERROR_OPERATING_SYSTEM

    #: This indicates that a resource handle passed to the API call was not
    #: valid. Resource handles are opaque types like :py:obj:`~.CUstream`
    #: and :py:obj:`~.CUevent`.
    CUDA_ERROR_INVALID_HANDLE = cydriver.cudaError_enum.CUDA_ERROR_INVALID_HANDLE

    #: This indicates that a resource required by the API call is not in a
    #: valid state to perform the requested operation.
    CUDA_ERROR_ILLEGAL_STATE = cydriver.cudaError_enum.CUDA_ERROR_ILLEGAL_STATE

    #: This indicates that a named symbol was not found. Examples of
    #: symbols are global/constant variable names, driver function names,
    #: texture names, and surface names.
    CUDA_ERROR_NOT_FOUND = cydriver.cudaError_enum.CUDA_ERROR_NOT_FOUND

    #: This indicates that asynchronous operations issued previously have
    #: not completed yet. This result is not actually an error, but must be
    #: indicated differently than :py:obj:`~.CUDA_SUCCESS` (which indicates
    #: completion). Calls that may return this value include
    #: :py:obj:`~.cuEventQuery()` and :py:obj:`~.cuStreamQuery()`.
    CUDA_ERROR_NOT_READY = cydriver.cudaError_enum.CUDA_ERROR_NOT_READY

    #: While executing a kernel, the device encountered a load or store
    #: instruction on an invalid memory address. This leaves the process in
    #: an inconsistent state and any further CUDA work will return the same
    #: error. To continue using CUDA, the process must be terminated and
    #: relaunched.
    CUDA_ERROR_ILLEGAL_ADDRESS = cydriver.cudaError_enum.CUDA_ERROR_ILLEGAL_ADDRESS

    #: This indicates that a launch did not occur because it did not have
    #: appropriate resources. This error usually indicates that the user
    #: has attempted to pass too many arguments to the device kernel, or
    #: the kernel launch specifies too many threads for the kernel's
    #: register count. Passing arguments of the wrong size (i.e. a 64-bit
    #: pointer when a 32-bit int is expected) is equivalent to passing too
    #: many arguments and can also result in this error.
    CUDA_ERROR_LAUNCH_OUT_OF_RESOURCES = cydriver.cudaError_enum.CUDA_ERROR_LAUNCH_OUT_OF_RESOURCES

    #: This indicates that the device kernel took too long to execute. This
    #: can only occur if timeouts are enabled - see the device attribute
    #: :py:obj:`~.CU_DEVICE_ATTRIBUTE_KERNEL_EXEC_TIMEOUT` for more
    #: information. This leaves the process in an inconsistent state and
    #: any further CUDA work will return the same error. To continue using
    #: CUDA, the process must be terminated and relaunched.
    CUDA_ERROR_LAUNCH_TIMEOUT = cydriver.cudaError_enum.CUDA_ERROR_LAUNCH_TIMEOUT

    #: This error indicates a kernel launch that uses an incompatible
    #: texturing mode.
    CUDA_ERROR_LAUNCH_INCOMPATIBLE_TEXTURING = cydriver.cudaError_enum.CUDA_ERROR_LAUNCH_INCOMPATIBLE_TEXTURING

    #: This error indicates that a call to
    #: :py:obj:`~.cuCtxEnablePeerAccess()` is trying to re-enable peer
    #: access to a context which has already had peer access to it enabled.
    CUDA_ERROR_PEER_ACCESS_ALREADY_ENABLED = cydriver.cudaError_enum.CUDA_ERROR_PEER_ACCESS_ALREADY_ENABLED

    #: This error indicates that :py:obj:`~.cuCtxDisablePeerAccess()` is
    #: trying to disable peer access which has not been enabled yet via
    #: :py:obj:`~.cuCtxEnablePeerAccess()`.
    CUDA_ERROR_PEER_ACCESS_NOT_ENABLED = cydriver.cudaError_enum.CUDA_ERROR_PEER_ACCESS_NOT_ENABLED

    #: This error indicates that the primary context for the specified
    #: device has already been initialized.
    CUDA_ERROR_PRIMARY_CONTEXT_ACTIVE = cydriver.cudaError_enum.CUDA_ERROR_PRIMARY_CONTEXT_ACTIVE

    #: This error indicates that the context current to the calling thread
    #: has been destroyed using :py:obj:`~.cuCtxDestroy`, or is a primary
    #: context which has not yet been initialized.
    CUDA_ERROR_CONTEXT_IS_DESTROYED = cydriver.cudaError_enum.CUDA_ERROR_CONTEXT_IS_DESTROYED

    #: A device-side assert triggered during kernel execution. The context
    #: cannot be used anymore, and must be destroyed. All existing device
    #: memory allocations from this context are invalid and must be
    #: reconstructed if the program is to continue using CUDA.
    CUDA_ERROR_ASSERT = cydriver.cudaError_enum.CUDA_ERROR_ASSERT

    #: This error indicates that the hardware resources required to enable
    #: peer access have been exhausted for one or more of the devices
    #: passed to :py:obj:`~.cuCtxEnablePeerAccess()`.
    CUDA_ERROR_TOO_MANY_PEERS = cydriver.cudaError_enum.CUDA_ERROR_TOO_MANY_PEERS

    #: This error indicates that the memory range passed to
    #: :py:obj:`~.cuMemHostRegister()` has already been registered.
    CUDA_ERROR_HOST_MEMORY_ALREADY_REGISTERED = cydriver.cudaError_enum.CUDA_ERROR_HOST_MEMORY_ALREADY_REGISTERED

    #: This error indicates that the pointer passed to
    #: :py:obj:`~.cuMemHostUnregister()` does not correspond to any
    #: currently registered memory region.
    CUDA_ERROR_HOST_MEMORY_NOT_REGISTERED = cydriver.cudaError_enum.CUDA_ERROR_HOST_MEMORY_NOT_REGISTERED

    #: While executing a kernel, the device encountered a stack error. This
    #: can be due to stack corruption or exceeding the stack size limit.
    #: This leaves the process in an inconsistent state and any further
    #: CUDA work will return the same error. To continue using CUDA, the
    #: process must be terminated and relaunched.
    CUDA_ERROR_HARDWARE_STACK_ERROR = cydriver.cudaError_enum.CUDA_ERROR_HARDWARE_STACK_ERROR

    #: While executing a kernel, the device encountered an illegal
    #: instruction. This leaves the process in an inconsistent state and
    #: any further CUDA work will return the same error. To continue using
    #: CUDA, the process must be terminated and relaunched.
    CUDA_ERROR_ILLEGAL_INSTRUCTION = cydriver.cudaError_enum.CUDA_ERROR_ILLEGAL_INSTRUCTION

    #: While executing a kernel, the device encountered a load or store
    #: instruction on a memory address which is not aligned. This leaves
    #: the process in an inconsistent state and any further CUDA work will
    #: return the same error. To continue using CUDA, the process must be
    #: terminated and relaunched.
    CUDA_ERROR_MISALIGNED_ADDRESS = cydriver.cudaError_enum.CUDA_ERROR_MISALIGNED_ADDRESS

    #: While executing a kernel, the device encountered an instruction
    #: which can only operate on memory locations in certain address spaces
    #: (global, shared, or local), but was supplied a memory address not
    #: belonging to an allowed address space. This leaves the process in an
    #: inconsistent state and any further CUDA work will return the same
    #: error. To continue using CUDA, the process must be terminated and
    #: relaunched.
    CUDA_ERROR_INVALID_ADDRESS_SPACE = cydriver.cudaError_enum.CUDA_ERROR_INVALID_ADDRESS_SPACE

    #: While executing a kernel, the device program counter wrapped its
    #: address space. This leaves the process in an inconsistent state and
    #: any further CUDA work will return the same error. To continue using
    #: CUDA, the process must be terminated and relaunched.
    CUDA_ERROR_INVALID_PC = cydriver.cudaError_enum.CUDA_ERROR_INVALID_PC

    #: An exception occurred on the device while executing a kernel. Common
    #: causes include dereferencing an invalid device pointer and accessing
    #: out of bounds shared memory. Less common cases can be system
    #: specific - more information about these cases can be found in the
    #: system specific user guide. This leaves the process in an
    #: inconsistent state and any further CUDA work will return the same
    #: error. To continue using CUDA, the process must be terminated and
    #: relaunched.
    CUDA_ERROR_LAUNCH_FAILED = cydriver.cudaError_enum.CUDA_ERROR_LAUNCH_FAILED

    #: This error indicates that the number of blocks launched per grid for
    #: a kernel that was launched via either
    #: :py:obj:`~.cuLaunchCooperativeKernel` or
    #: :py:obj:`~.cuLaunchCooperativeKernelMultiDevice` exceeds the maximum
    #: number of blocks as allowed by
    #: :py:obj:`~.cuOccupancyMaxActiveBlocksPerMultiprocessor` or
    #: :py:obj:`~.cuOccupancyMaxActiveBlocksPerMultiprocessorWithFlags`
    #: times the number of multiprocessors as specified by the device
    #: attribute :py:obj:`~.CU_DEVICE_ATTRIBUTE_MULTIPROCESSOR_COUNT`.
    CUDA_ERROR_COOPERATIVE_LAUNCH_TOO_LARGE = cydriver.cudaError_enum.CUDA_ERROR_COOPERATIVE_LAUNCH_TOO_LARGE

    #: This error indicates that the attempted operation is not permitted.
    CUDA_ERROR_NOT_PERMITTED = cydriver.cudaError_enum.CUDA_ERROR_NOT_PERMITTED

    #: This error indicates that the attempted operation is not supported
    #: on the current system or device.
    CUDA_ERROR_NOT_SUPPORTED = cydriver.cudaError_enum.CUDA_ERROR_NOT_SUPPORTED

    #: This error indicates that the system is not yet ready to start any
    #: CUDA work. To continue using CUDA, verify the system configuration
    #: is in a valid state and all required driver daemons are actively
    #: running. More information about this error can be found in the
    #: system specific user guide.
    CUDA_ERROR_SYSTEM_NOT_READY = cydriver.cudaError_enum.CUDA_ERROR_SYSTEM_NOT_READY

    #: This error indicates that there is a mismatch between the versions
    #: of the display driver and the CUDA driver. Refer to the
    #: compatibility documentation for supported versions.
    CUDA_ERROR_SYSTEM_DRIVER_MISMATCH = cydriver.cudaError_enum.CUDA_ERROR_SYSTEM_DRIVER_MISMATCH

    #: This error indicates that the system was upgraded to run with
    #: forward compatibility but the visible hardware detected by CUDA does
    #: not support this configuration. Refer to the compatibility
    #: documentation for the supported hardware matrix or ensure that only
    #: supported hardware is visible during initialization via the
    #: CUDA_VISIBLE_DEVICES environment variable.
    CUDA_ERROR_COMPAT_NOT_SUPPORTED_ON_DEVICE = cydriver.cudaError_enum.CUDA_ERROR_COMPAT_NOT_SUPPORTED_ON_DEVICE

    #: This error indicates that the MPS client failed to connect to the
    #: MPS control daemon or the MPS server.
    CUDA_ERROR_MPS_CONNECTION_FAILED = cydriver.cudaError_enum.CUDA_ERROR_MPS_CONNECTION_FAILED

    #: This error indicates that the remote procedural call between the MPS
    #: server and the MPS client failed.
    CUDA_ERROR_MPS_RPC_FAILURE = cydriver.cudaError_enum.CUDA_ERROR_MPS_RPC_FAILURE

    #: This error indicates that the MPS server is not ready to accept new
    #: MPS client requests. This error can be returned when the MPS server
    #: is in the process of recovering from a fatal failure.
    CUDA_ERROR_MPS_SERVER_NOT_READY = cydriver.cudaError_enum.CUDA_ERROR_MPS_SERVER_NOT_READY

    #: This error indicates that the hardware resources required to create
    #: MPS client have been exhausted.
    CUDA_ERROR_MPS_MAX_CLIENTS_REACHED = cydriver.cudaError_enum.CUDA_ERROR_MPS_MAX_CLIENTS_REACHED

    #: This error indicates the the hardware resources required to support
    #: device connections have been exhausted.
    CUDA_ERROR_MPS_MAX_CONNECTIONS_REACHED = cydriver.cudaError_enum.CUDA_ERROR_MPS_MAX_CONNECTIONS_REACHED

    #: This error indicates that the MPS client has been terminated by the
    #: server. To continue using CUDA, the process must be terminated and
    #: relaunched.
    CUDA_ERROR_MPS_CLIENT_TERMINATED = cydriver.cudaError_enum.CUDA_ERROR_MPS_CLIENT_TERMINATED

    #: This error indicates that the operation is not permitted when the
    #: stream is capturing.
    CUDA_ERROR_STREAM_CAPTURE_UNSUPPORTED = cydriver.cudaError_enum.CUDA_ERROR_STREAM_CAPTURE_UNSUPPORTED

    #: This error indicates that the current capture sequence on the stream
    #: has been invalidated due to a previous error.
    CUDA_ERROR_STREAM_CAPTURE_INVALIDATED = cydriver.cudaError_enum.CUDA_ERROR_STREAM_CAPTURE_INVALIDATED

    #: This error indicates that the operation would have resulted in a
    #: merge of two independent capture sequences.
    CUDA_ERROR_STREAM_CAPTURE_MERGE = cydriver.cudaError_enum.CUDA_ERROR_STREAM_CAPTURE_MERGE

    #: This error indicates that the capture was not initiated in this
    #: stream.
    CUDA_ERROR_STREAM_CAPTURE_UNMATCHED = cydriver.cudaError_enum.CUDA_ERROR_STREAM_CAPTURE_UNMATCHED

    #: This error indicates that the capture sequence contains a fork that
    #: was not joined to the primary stream.
    CUDA_ERROR_STREAM_CAPTURE_UNJOINED = cydriver.cudaError_enum.CUDA_ERROR_STREAM_CAPTURE_UNJOINED

    #: This error indicates that a dependency would have been created which
    #: crosses the capture sequence boundary. Only implicit in-stream
    #: ordering dependencies are allowed to cross the boundary.
    CUDA_ERROR_STREAM_CAPTURE_ISOLATION = cydriver.cudaError_enum.CUDA_ERROR_STREAM_CAPTURE_ISOLATION

    #: This error indicates a disallowed implicit dependency on a current
    #: capture sequence from cudaStreamLegacy.
    CUDA_ERROR_STREAM_CAPTURE_IMPLICIT = cydriver.cudaError_enum.CUDA_ERROR_STREAM_CAPTURE_IMPLICIT

    #: This error indicates that the operation is not permitted on an event
    #: which was last recorded in a capturing stream.
    CUDA_ERROR_CAPTURED_EVENT = cydriver.cudaError_enum.CUDA_ERROR_CAPTURED_EVENT

    #: A stream capture sequence not initiated with the
    #: :py:obj:`~.CU_STREAM_CAPTURE_MODE_RELAXED` argument to
    #: :py:obj:`~.cuStreamBeginCapture` was passed to
    #: :py:obj:`~.cuStreamEndCapture` in a different thread.
    CUDA_ERROR_STREAM_CAPTURE_WRONG_THREAD = cydriver.cudaError_enum.CUDA_ERROR_STREAM_CAPTURE_WRONG_THREAD

    #: This error indicates that the timeout specified for the wait
    #: operation has lapsed.
    CUDA_ERROR_TIMEOUT = cydriver.cudaError_enum.CUDA_ERROR_TIMEOUT

    #: This error indicates that the graph update was not performed because
    #: it included changes which violated constraints specific to
    #: instantiated graph update.
    CUDA_ERROR_GRAPH_EXEC_UPDATE_FAILURE = cydriver.cudaError_enum.CUDA_ERROR_GRAPH_EXEC_UPDATE_FAILURE

    #: This indicates that an async error has occurred in a device outside
    #: of CUDA. If CUDA was waiting for an external device's signal before
    #: consuming shared data, the external device signaled an error
    #: indicating that the data is not valid for consumption. This leaves
    #: the process in an inconsistent state and any further CUDA work will
    #: return the same error. To continue using CUDA, the process must be
    #: terminated and relaunched.
    CUDA_ERROR_EXTERNAL_DEVICE = cydriver.cudaError_enum.CUDA_ERROR_EXTERNAL_DEVICE

    #: Indicates a kernel launch error due to cluster misconfiguration.
    CUDA_ERROR_INVALID_CLUSTER_SIZE = cydriver.cudaError_enum.CUDA_ERROR_INVALID_CLUSTER_SIZE

    #: This indicates that an unknown internal error has occurred.
    CUDA_ERROR_UNKNOWN = cydriver.cudaError_enum.CUDA_ERROR_UNKNOWN

class CUdevice_P2PAttribute(IntEnum):
    """
    P2P Attributes
    """

    #: A relative value indicating the performance of the link between two
    #: devices
    CU_DEVICE_P2P_ATTRIBUTE_PERFORMANCE_RANK = cydriver.CUdevice_P2PAttribute_enum.CU_DEVICE_P2P_ATTRIBUTE_PERFORMANCE_RANK

    #: P2P Access is enable
    CU_DEVICE_P2P_ATTRIBUTE_ACCESS_SUPPORTED = cydriver.CUdevice_P2PAttribute_enum.CU_DEVICE_P2P_ATTRIBUTE_ACCESS_SUPPORTED

    #: Atomic operation over the link supported
    CU_DEVICE_P2P_ATTRIBUTE_NATIVE_ATOMIC_SUPPORTED = cydriver.CUdevice_P2PAttribute_enum.CU_DEVICE_P2P_ATTRIBUTE_NATIVE_ATOMIC_SUPPORTED

    #: [Deprecated]
    CU_DEVICE_P2P_ATTRIBUTE_ACCESS_ACCESS_SUPPORTED = cydriver.CUdevice_P2PAttribute_enum.CU_DEVICE_P2P_ATTRIBUTE_ACCESS_ACCESS_SUPPORTED

    #: Accessing CUDA arrays over the link supported
    CU_DEVICE_P2P_ATTRIBUTE_CUDA_ARRAY_ACCESS_SUPPORTED = cydriver.CUdevice_P2PAttribute_enum.CU_DEVICE_P2P_ATTRIBUTE_CUDA_ARRAY_ACCESS_SUPPORTED

class CUresourceViewFormat(IntEnum):
    """
    Resource view format
    """

    #: No resource view format (use underlying resource format)
    CU_RES_VIEW_FORMAT_NONE = cydriver.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_NONE

    #: 1 channel unsigned 8-bit integers
    CU_RES_VIEW_FORMAT_UINT_1X8 = cydriver.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_UINT_1X8

    #: 2 channel unsigned 8-bit integers
    CU_RES_VIEW_FORMAT_UINT_2X8 = cydriver.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_UINT_2X8

    #: 4 channel unsigned 8-bit integers
    CU_RES_VIEW_FORMAT_UINT_4X8 = cydriver.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_UINT_4X8

    #: 1 channel signed 8-bit integers
    CU_RES_VIEW_FORMAT_SINT_1X8 = cydriver.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_SINT_1X8

    #: 2 channel signed 8-bit integers
    CU_RES_VIEW_FORMAT_SINT_2X8 = cydriver.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_SINT_2X8

    #: 4 channel signed 8-bit integers
    CU_RES_VIEW_FORMAT_SINT_4X8 = cydriver.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_SINT_4X8

    #: 1 channel unsigned 16-bit integers
    CU_RES_VIEW_FORMAT_UINT_1X16 = cydriver.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_UINT_1X16

    #: 2 channel unsigned 16-bit integers
    CU_RES_VIEW_FORMAT_UINT_2X16 = cydriver.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_UINT_2X16

    #: 4 channel unsigned 16-bit integers
    CU_RES_VIEW_FORMAT_UINT_4X16 = cydriver.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_UINT_4X16

    #: 1 channel signed 16-bit integers
    CU_RES_VIEW_FORMAT_SINT_1X16 = cydriver.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_SINT_1X16

    #: 2 channel signed 16-bit integers
    CU_RES_VIEW_FORMAT_SINT_2X16 = cydriver.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_SINT_2X16

    #: 4 channel signed 16-bit integers
    CU_RES_VIEW_FORMAT_SINT_4X16 = cydriver.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_SINT_4X16

    #: 1 channel unsigned 32-bit integers
    CU_RES_VIEW_FORMAT_UINT_1X32 = cydriver.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_UINT_1X32

    #: 2 channel unsigned 32-bit integers
    CU_RES_VIEW_FORMAT_UINT_2X32 = cydriver.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_UINT_2X32

    #: 4 channel unsigned 32-bit integers
    CU_RES_VIEW_FORMAT_UINT_4X32 = cydriver.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_UINT_4X32

    #: 1 channel signed 32-bit integers
    CU_RES_VIEW_FORMAT_SINT_1X32 = cydriver.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_SINT_1X32

    #: 2 channel signed 32-bit integers
    CU_RES_VIEW_FORMAT_SINT_2X32 = cydriver.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_SINT_2X32

    #: 4 channel signed 32-bit integers
    CU_RES_VIEW_FORMAT_SINT_4X32 = cydriver.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_SINT_4X32

    #: 1 channel 16-bit floating point
    CU_RES_VIEW_FORMAT_FLOAT_1X16 = cydriver.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_FLOAT_1X16

    #: 2 channel 16-bit floating point
    CU_RES_VIEW_FORMAT_FLOAT_2X16 = cydriver.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_FLOAT_2X16

    #: 4 channel 16-bit floating point
    CU_RES_VIEW_FORMAT_FLOAT_4X16 = cydriver.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_FLOAT_4X16

    #: 1 channel 32-bit floating point
    CU_RES_VIEW_FORMAT_FLOAT_1X32 = cydriver.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_FLOAT_1X32

    #: 2 channel 32-bit floating point
    CU_RES_VIEW_FORMAT_FLOAT_2X32 = cydriver.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_FLOAT_2X32

    #: 4 channel 32-bit floating point
    CU_RES_VIEW_FORMAT_FLOAT_4X32 = cydriver.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_FLOAT_4X32

    #: Block compressed 1
    CU_RES_VIEW_FORMAT_UNSIGNED_BC1 = cydriver.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_UNSIGNED_BC1

    #: Block compressed 2
    CU_RES_VIEW_FORMAT_UNSIGNED_BC2 = cydriver.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_UNSIGNED_BC2

    #: Block compressed 3
    CU_RES_VIEW_FORMAT_UNSIGNED_BC3 = cydriver.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_UNSIGNED_BC3

    #: Block compressed 4 unsigned
    CU_RES_VIEW_FORMAT_UNSIGNED_BC4 = cydriver.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_UNSIGNED_BC4

    #: Block compressed 4 signed
    CU_RES_VIEW_FORMAT_SIGNED_BC4 = cydriver.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_SIGNED_BC4

    #: Block compressed 5 unsigned
    CU_RES_VIEW_FORMAT_UNSIGNED_BC5 = cydriver.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_UNSIGNED_BC5

    #: Block compressed 5 signed
    CU_RES_VIEW_FORMAT_SIGNED_BC5 = cydriver.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_SIGNED_BC5

    #: Block compressed 6 unsigned half-float
    CU_RES_VIEW_FORMAT_UNSIGNED_BC6H = cydriver.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_UNSIGNED_BC6H

    #: Block compressed 6 signed half-float
    CU_RES_VIEW_FORMAT_SIGNED_BC6H = cydriver.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_SIGNED_BC6H

    #: Block compressed 7
    CU_RES_VIEW_FORMAT_UNSIGNED_BC7 = cydriver.CUresourceViewFormat_enum.CU_RES_VIEW_FORMAT_UNSIGNED_BC7

class CUDA_POINTER_ATTRIBUTE_ACCESS_FLAGS(IntEnum):
    """
    Access flags that specify the level of access the current context's
    device has on the memory referenced.
    """

    #: No access, meaning the device cannot access this memory at all, thus
    #: must be staged through accessible memory in order to complete
    #: certain operations
    CU_POINTER_ATTRIBUTE_ACCESS_FLAG_NONE = cydriver.CUDA_POINTER_ATTRIBUTE_ACCESS_FLAGS_enum.CU_POINTER_ATTRIBUTE_ACCESS_FLAG_NONE

    #: Read-only access, meaning writes to this memory are considered
    #: invalid accesses and thus return error in that case.
    CU_POINTER_ATTRIBUTE_ACCESS_FLAG_READ = cydriver.CUDA_POINTER_ATTRIBUTE_ACCESS_FLAGS_enum.CU_POINTER_ATTRIBUTE_ACCESS_FLAG_READ

    #: Read-write access, the device has full read-write access to the
    #: memory
    CU_POINTER_ATTRIBUTE_ACCESS_FLAG_READWRITE = cydriver.CUDA_POINTER_ATTRIBUTE_ACCESS_FLAGS_enum.CU_POINTER_ATTRIBUTE_ACCESS_FLAG_READWRITE

class CUexternalMemoryHandleType(IntEnum):
    """
    External memory handle types
    """

    #: Handle is an opaque file descriptor
    CU_EXTERNAL_MEMORY_HANDLE_TYPE_OPAQUE_FD = cydriver.CUexternalMemoryHandleType_enum.CU_EXTERNAL_MEMORY_HANDLE_TYPE_OPAQUE_FD

    #: Handle is an opaque shared NT handle
    CU_EXTERNAL_MEMORY_HANDLE_TYPE_OPAQUE_WIN32 = cydriver.CUexternalMemoryHandleType_enum.CU_EXTERNAL_MEMORY_HANDLE_TYPE_OPAQUE_WIN32

    #: Handle is an opaque, globally shared handle
    CU_EXTERNAL_MEMORY_HANDLE_TYPE_OPAQUE_WIN32_KMT = cydriver.CUexternalMemoryHandleType_enum.CU_EXTERNAL_MEMORY_HANDLE_TYPE_OPAQUE_WIN32_KMT

    #: Handle is a D3D12 heap object
    CU_EXTERNAL_MEMORY_HANDLE_TYPE_D3D12_HEAP = cydriver.CUexternalMemoryHandleType_enum.CU_EXTERNAL_MEMORY_HANDLE_TYPE_D3D12_HEAP

    #: Handle is a D3D12 committed resource
    CU_EXTERNAL_MEMORY_HANDLE_TYPE_D3D12_RESOURCE = cydriver.CUexternalMemoryHandleType_enum.CU_EXTERNAL_MEMORY_HANDLE_TYPE_D3D12_RESOURCE

    #: Handle is a shared NT handle to a D3D11 resource
    CU_EXTERNAL_MEMORY_HANDLE_TYPE_D3D11_RESOURCE = cydriver.CUexternalMemoryHandleType_enum.CU_EXTERNAL_MEMORY_HANDLE_TYPE_D3D11_RESOURCE

    #: Handle is a globally shared handle to a D3D11 resource
    CU_EXTERNAL_MEMORY_HANDLE_TYPE_D3D11_RESOURCE_KMT = cydriver.CUexternalMemoryHandleType_enum.CU_EXTERNAL_MEMORY_HANDLE_TYPE_D3D11_RESOURCE_KMT

    #: Handle is an NvSciBuf object
    CU_EXTERNAL_MEMORY_HANDLE_TYPE_NVSCIBUF = cydriver.CUexternalMemoryHandleType_enum.CU_EXTERNAL_MEMORY_HANDLE_TYPE_NVSCIBUF

class CUexternalSemaphoreHandleType(IntEnum):
    """
    External semaphore handle types
    """

    #: Handle is an opaque file descriptor
    CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_OPAQUE_FD = cydriver.CUexternalSemaphoreHandleType_enum.CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_OPAQUE_FD

    #: Handle is an opaque shared NT handle
    CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_OPAQUE_WIN32 = cydriver.CUexternalSemaphoreHandleType_enum.CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_OPAQUE_WIN32

    #: Handle is an opaque, globally shared handle
    CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_OPAQUE_WIN32_KMT = cydriver.CUexternalSemaphoreHandleType_enum.CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_OPAQUE_WIN32_KMT

    #: Handle is a shared NT handle referencing a D3D12 fence object
    CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_D3D12_FENCE = cydriver.CUexternalSemaphoreHandleType_enum.CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_D3D12_FENCE

    #: Handle is a shared NT handle referencing a D3D11 fence object
    CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_D3D11_FENCE = cydriver.CUexternalSemaphoreHandleType_enum.CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_D3D11_FENCE

    #: Opaque handle to NvSciSync Object
    CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_NVSCISYNC = cydriver.CUexternalSemaphoreHandleType_enum.CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_NVSCISYNC

    #: Handle is a shared NT handle referencing a D3D11 keyed mutex object
    CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_D3D11_KEYED_MUTEX = cydriver.CUexternalSemaphoreHandleType_enum.CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_D3D11_KEYED_MUTEX

    #: Handle is a globally shared handle referencing a D3D11 keyed mutex
    #: object
    CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_D3D11_KEYED_MUTEX_KMT = cydriver.CUexternalSemaphoreHandleType_enum.CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_D3D11_KEYED_MUTEX_KMT

    #: Handle is an opaque file descriptor referencing a timeline semaphore
    CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_TIMELINE_SEMAPHORE_FD = cydriver.CUexternalSemaphoreHandleType_enum.CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_TIMELINE_SEMAPHORE_FD

    #: Handle is an opaque shared NT handle referencing a timeline
    #: semaphore
    CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_TIMELINE_SEMAPHORE_WIN32 = cydriver.CUexternalSemaphoreHandleType_enum.CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_TIMELINE_SEMAPHORE_WIN32

class CUmemAllocationHandleType(IntEnum):
    """
    Flags for specifying particular handle types
    """

    #: Does not allow any export mechanism. >
    CU_MEM_HANDLE_TYPE_NONE = cydriver.CUmemAllocationHandleType_enum.CU_MEM_HANDLE_TYPE_NONE

    #: Allows a file descriptor to be used for exporting. Permitted only on
    #: POSIX systems. (int)
    CU_MEM_HANDLE_TYPE_POSIX_FILE_DESCRIPTOR = cydriver.CUmemAllocationHandleType_enum.CU_MEM_HANDLE_TYPE_POSIX_FILE_DESCRIPTOR

    #: Allows a Win32 NT handle to be used for exporting. (HANDLE)
    CU_MEM_HANDLE_TYPE_WIN32 = cydriver.CUmemAllocationHandleType_enum.CU_MEM_HANDLE_TYPE_WIN32

    #: Allows a Win32 KMT handle to be used for exporting. (D3DKMT_HANDLE)
    CU_MEM_HANDLE_TYPE_WIN32_KMT = cydriver.CUmemAllocationHandleType_enum.CU_MEM_HANDLE_TYPE_WIN32_KMT
    CU_MEM_HANDLE_TYPE_MAX = cydriver.CUmemAllocationHandleType_enum.CU_MEM_HANDLE_TYPE_MAX

class CUmemAccess_flags(IntEnum):
    """
    Specifies the memory protection flags for mapping.
    """

    #: Default, make the address range not accessible
    CU_MEM_ACCESS_FLAGS_PROT_NONE = cydriver.CUmemAccess_flags_enum.CU_MEM_ACCESS_FLAGS_PROT_NONE

    #: Make the address range read accessible
    CU_MEM_ACCESS_FLAGS_PROT_READ = cydriver.CUmemAccess_flags_enum.CU_MEM_ACCESS_FLAGS_PROT_READ

    #: Make the address range read-write accessible
    CU_MEM_ACCESS_FLAGS_PROT_READWRITE = cydriver.CUmemAccess_flags_enum.CU_MEM_ACCESS_FLAGS_PROT_READWRITE
    CU_MEM_ACCESS_FLAGS_PROT_MAX = cydriver.CUmemAccess_flags_enum.CU_MEM_ACCESS_FLAGS_PROT_MAX

class CUmemLocationType(IntEnum):
    """
    Specifies the type of location
    """
    CU_MEM_LOCATION_TYPE_INVALID = cydriver.CUmemLocationType_enum.CU_MEM_LOCATION_TYPE_INVALID

    #: Location is a device location, thus id is a device ordinal
    CU_MEM_LOCATION_TYPE_DEVICE = cydriver.CUmemLocationType_enum.CU_MEM_LOCATION_TYPE_DEVICE
    CU_MEM_LOCATION_TYPE_MAX = cydriver.CUmemLocationType_enum.CU_MEM_LOCATION_TYPE_MAX

class CUmemAllocationType(IntEnum):
    """
    Defines the allocation types available
    """
    CU_MEM_ALLOCATION_TYPE_INVALID = cydriver.CUmemAllocationType_enum.CU_MEM_ALLOCATION_TYPE_INVALID

    #: This allocation type is 'pinned', i.e. cannot migrate from its
    #: current location while the application is actively using it
    CU_MEM_ALLOCATION_TYPE_PINNED = cydriver.CUmemAllocationType_enum.CU_MEM_ALLOCATION_TYPE_PINNED
    CU_MEM_ALLOCATION_TYPE_MAX = cydriver.CUmemAllocationType_enum.CU_MEM_ALLOCATION_TYPE_MAX

class CUmemAllocationGranularity_flags(IntEnum):
    """
    Flag for requesting different optimal and required granularities
    for an allocation.
    """

    #: Minimum required granularity for allocation
    CU_MEM_ALLOC_GRANULARITY_MINIMUM = cydriver.CUmemAllocationGranularity_flags_enum.CU_MEM_ALLOC_GRANULARITY_MINIMUM

    #: Recommended granularity for allocation for best performance
    CU_MEM_ALLOC_GRANULARITY_RECOMMENDED = cydriver.CUmemAllocationGranularity_flags_enum.CU_MEM_ALLOC_GRANULARITY_RECOMMENDED

class CUmemRangeHandleType(IntEnum):
    """
    Specifies the handle type for address range
    """
    CU_MEM_RANGE_HANDLE_TYPE_DMA_BUF_FD = cydriver.CUmemRangeHandleType_enum.CU_MEM_RANGE_HANDLE_TYPE_DMA_BUF_FD
    CU_MEM_RANGE_HANDLE_TYPE_MAX = cydriver.CUmemRangeHandleType_enum.CU_MEM_RANGE_HANDLE_TYPE_MAX

class CUarraySparseSubresourceType(IntEnum):
    """
    Sparse subresource types
    """
    CU_ARRAY_SPARSE_SUBRESOURCE_TYPE_SPARSE_LEVEL = cydriver.CUarraySparseSubresourceType_enum.CU_ARRAY_SPARSE_SUBRESOURCE_TYPE_SPARSE_LEVEL
    CU_ARRAY_SPARSE_SUBRESOURCE_TYPE_MIPTAIL = cydriver.CUarraySparseSubresourceType_enum.CU_ARRAY_SPARSE_SUBRESOURCE_TYPE_MIPTAIL

class CUmemOperationType(IntEnum):
    """
    Memory operation types
    """
    CU_MEM_OPERATION_TYPE_MAP = cydriver.CUmemOperationType_enum.CU_MEM_OPERATION_TYPE_MAP
    CU_MEM_OPERATION_TYPE_UNMAP = cydriver.CUmemOperationType_enum.CU_MEM_OPERATION_TYPE_UNMAP

class CUmemHandleType(IntEnum):
    """
    Memory handle types
    """
    CU_MEM_HANDLE_TYPE_GENERIC = cydriver.CUmemHandleType_enum.CU_MEM_HANDLE_TYPE_GENERIC

class CUmemAllocationCompType(IntEnum):
    """
    Specifies compression attribute for an allocation.
    """

    #: Allocating non-compressible memory
    CU_MEM_ALLOCATION_COMP_NONE = cydriver.CUmemAllocationCompType_enum.CU_MEM_ALLOCATION_COMP_NONE

    #: Allocating compressible memory
    CU_MEM_ALLOCATION_COMP_GENERIC = cydriver.CUmemAllocationCompType_enum.CU_MEM_ALLOCATION_COMP_GENERIC

class CUgraphExecUpdateResult(IntEnum):
    """

    """

    #: The update succeeded
    CU_GRAPH_EXEC_UPDATE_SUCCESS = cydriver.CUgraphExecUpdateResult_enum.CU_GRAPH_EXEC_UPDATE_SUCCESS

    #: The update failed for an unexpected reason which is described in the
    #: return value of the function
    CU_GRAPH_EXEC_UPDATE_ERROR = cydriver.CUgraphExecUpdateResult_enum.CU_GRAPH_EXEC_UPDATE_ERROR

    #: The update failed because the topology changed
    CU_GRAPH_EXEC_UPDATE_ERROR_TOPOLOGY_CHANGED = cydriver.CUgraphExecUpdateResult_enum.CU_GRAPH_EXEC_UPDATE_ERROR_TOPOLOGY_CHANGED

    #: The update failed because a node type changed
    CU_GRAPH_EXEC_UPDATE_ERROR_NODE_TYPE_CHANGED = cydriver.CUgraphExecUpdateResult_enum.CU_GRAPH_EXEC_UPDATE_ERROR_NODE_TYPE_CHANGED

    #: The update failed because the function of a kernel node changed
    #: (CUDA driver < 11.2)
    CU_GRAPH_EXEC_UPDATE_ERROR_FUNCTION_CHANGED = cydriver.CUgraphExecUpdateResult_enum.CU_GRAPH_EXEC_UPDATE_ERROR_FUNCTION_CHANGED

    #: The update failed because the parameters changed in a way that is
    #: not supported
    CU_GRAPH_EXEC_UPDATE_ERROR_PARAMETERS_CHANGED = cydriver.CUgraphExecUpdateResult_enum.CU_GRAPH_EXEC_UPDATE_ERROR_PARAMETERS_CHANGED

    #: The update failed because something about the node is not supported
    CU_GRAPH_EXEC_UPDATE_ERROR_NOT_SUPPORTED = cydriver.CUgraphExecUpdateResult_enum.CU_GRAPH_EXEC_UPDATE_ERROR_NOT_SUPPORTED

    #: The update failed because the function of a kernel node changed in
    #: an unsupported way
    CU_GRAPH_EXEC_UPDATE_ERROR_UNSUPPORTED_FUNCTION_CHANGE = cydriver.CUgraphExecUpdateResult_enum.CU_GRAPH_EXEC_UPDATE_ERROR_UNSUPPORTED_FUNCTION_CHANGE

    #: The update failed because the node attributes changed in a way that
    #: is not supported
    CU_GRAPH_EXEC_UPDATE_ERROR_ATTRIBUTES_CHANGED = cydriver.CUgraphExecUpdateResult_enum.CU_GRAPH_EXEC_UPDATE_ERROR_ATTRIBUTES_CHANGED

class CUmemPool_attribute(IntEnum):
    """
    CUDA memory pool attributes
    """

    #: (value type = int) Allow cuMemAllocAsync to use memory
    #: asynchronously freed in another streams as long as a stream ordering
    #: dependency of the allocating stream on the free action exists. Cuda
    #: events and null stream interactions can create the required stream
    #: ordered dependencies. (default enabled)
    CU_MEMPOOL_ATTR_REUSE_FOLLOW_EVENT_DEPENDENCIES = cydriver.CUmemPool_attribute_enum.CU_MEMPOOL_ATTR_REUSE_FOLLOW_EVENT_DEPENDENCIES

    #: (value type = int) Allow reuse of already completed frees when there
    #: is no dependency between the free and allocation. (default enabled)
    CU_MEMPOOL_ATTR_REUSE_ALLOW_OPPORTUNISTIC = cydriver.CUmemPool_attribute_enum.CU_MEMPOOL_ATTR_REUSE_ALLOW_OPPORTUNISTIC

    #: (value type = int) Allow cuMemAllocAsync to insert new stream
    #: dependencies in order to establish the stream ordering required to
    #: reuse a piece of memory released by cuFreeAsync (default enabled).
    CU_MEMPOOL_ATTR_REUSE_ALLOW_INTERNAL_DEPENDENCIES = cydriver.CUmemPool_attribute_enum.CU_MEMPOOL_ATTR_REUSE_ALLOW_INTERNAL_DEPENDENCIES

    #: (value type = cuuint64_t) Amount of reserved memory in bytes to hold
    #: onto before trying to release memory back to the OS. When more than
    #: the release threshold bytes of memory are held by the memory pool,
    #: the allocator will try to release memory back to the OS on the next
    #: call to stream, event or context synchronize. (default 0)
    CU_MEMPOOL_ATTR_RELEASE_THRESHOLD = cydriver.CUmemPool_attribute_enum.CU_MEMPOOL_ATTR_RELEASE_THRESHOLD

    #: (value type = cuuint64_t) Amount of backing memory currently
    #: allocated for the mempool.
    CU_MEMPOOL_ATTR_RESERVED_MEM_CURRENT = cydriver.CUmemPool_attribute_enum.CU_MEMPOOL_ATTR_RESERVED_MEM_CURRENT

    #: (value type = cuuint64_t) High watermark of backing memory allocated
    #: for the mempool since the last time it was reset. High watermark can
    #: only be reset to zero.
    CU_MEMPOOL_ATTR_RESERVED_MEM_HIGH = cydriver.CUmemPool_attribute_enum.CU_MEMPOOL_ATTR_RESERVED_MEM_HIGH

    #: (value type = cuuint64_t) Amount of memory from the pool that is
    #: currently in use by the application.
    CU_MEMPOOL_ATTR_USED_MEM_CURRENT = cydriver.CUmemPool_attribute_enum.CU_MEMPOOL_ATTR_USED_MEM_CURRENT

    #: (value type = cuuint64_t) High watermark of the amount of memory
    #: from the pool that was in use by the application since the last time
    #: it was reset. High watermark can only be reset to zero.
    CU_MEMPOOL_ATTR_USED_MEM_HIGH = cydriver.CUmemPool_attribute_enum.CU_MEMPOOL_ATTR_USED_MEM_HIGH

class CUgraphMem_attribute(IntEnum):
    """

    """

    #: (value type = cuuint64_t) Amount of memory, in bytes, currently
    #: associated with graphs
    CU_GRAPH_MEM_ATTR_USED_MEM_CURRENT = cydriver.CUgraphMem_attribute_enum.CU_GRAPH_MEM_ATTR_USED_MEM_CURRENT

    #: (value type = cuuint64_t) High watermark of memory, in bytes,
    #: associated with graphs since the last time it was reset. High
    #: watermark can only be reset to zero.
    CU_GRAPH_MEM_ATTR_USED_MEM_HIGH = cydriver.CUgraphMem_attribute_enum.CU_GRAPH_MEM_ATTR_USED_MEM_HIGH

    #: (value type = cuuint64_t) Amount of memory, in bytes, currently
    #: allocated for use by the CUDA graphs asynchronous allocator.
    CU_GRAPH_MEM_ATTR_RESERVED_MEM_CURRENT = cydriver.CUgraphMem_attribute_enum.CU_GRAPH_MEM_ATTR_RESERVED_MEM_CURRENT

    #: (value type = cuuint64_t) High watermark of memory, in bytes,
    #: currently allocated for use by the CUDA graphs asynchronous
    #: allocator.
    CU_GRAPH_MEM_ATTR_RESERVED_MEM_HIGH = cydriver.CUgraphMem_attribute_enum.CU_GRAPH_MEM_ATTR_RESERVED_MEM_HIGH

class CUflushGPUDirectRDMAWritesOptions(IntEnum):
    """
    Bitmasks for
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_GPU_DIRECT_RDMA_FLUSH_WRITES_OPTIONS`
    """

    #: :py:obj:`~.cuFlushGPUDirectRDMAWrites()` and its CUDA Runtime API
    #: counterpart are supported on the device.
    CU_FLUSH_GPU_DIRECT_RDMA_WRITES_OPTION_HOST = cydriver.CUflushGPUDirectRDMAWritesOptions_enum.CU_FLUSH_GPU_DIRECT_RDMA_WRITES_OPTION_HOST

    #: The :py:obj:`~.CU_STREAM_WAIT_VALUE_FLUSH` flag and the
    #: :py:obj:`~.CU_STREAM_MEM_OP_FLUSH_REMOTE_WRITES` MemOp are supported
    #: on the device.
    CU_FLUSH_GPU_DIRECT_RDMA_WRITES_OPTION_MEMOPS = cydriver.CUflushGPUDirectRDMAWritesOptions_enum.CU_FLUSH_GPU_DIRECT_RDMA_WRITES_OPTION_MEMOPS

class CUGPUDirectRDMAWritesOrdering(IntEnum):
    """
    Platform native ordering for GPUDirect RDMA writes
    """

    #: The device does not natively support ordering of remote writes.
    #: :py:obj:`~.cuFlushGPUDirectRDMAWrites()` can be leveraged if
    #: supported.
    CU_GPU_DIRECT_RDMA_WRITES_ORDERING_NONE = cydriver.CUGPUDirectRDMAWritesOrdering_enum.CU_GPU_DIRECT_RDMA_WRITES_ORDERING_NONE

    #: Natively, the device can consistently consume remote writes,
    #: although other CUDA devices may not.
    CU_GPU_DIRECT_RDMA_WRITES_ORDERING_OWNER = cydriver.CUGPUDirectRDMAWritesOrdering_enum.CU_GPU_DIRECT_RDMA_WRITES_ORDERING_OWNER

    #: Any CUDA device in the system can consistently consume remote writes
    #: to this device.
    CU_GPU_DIRECT_RDMA_WRITES_ORDERING_ALL_DEVICES = cydriver.CUGPUDirectRDMAWritesOrdering_enum.CU_GPU_DIRECT_RDMA_WRITES_ORDERING_ALL_DEVICES

class CUflushGPUDirectRDMAWritesScope(IntEnum):
    """
    The scopes for :py:obj:`~.cuFlushGPUDirectRDMAWrites`
    """

    #: Blocks until remote writes are visible to the CUDA device context
    #: owning the data.
    CU_FLUSH_GPU_DIRECT_RDMA_WRITES_TO_OWNER = cydriver.CUflushGPUDirectRDMAWritesScope_enum.CU_FLUSH_GPU_DIRECT_RDMA_WRITES_TO_OWNER

    #: Blocks until remote writes are visible to all CUDA device contexts.
    CU_FLUSH_GPU_DIRECT_RDMA_WRITES_TO_ALL_DEVICES = cydriver.CUflushGPUDirectRDMAWritesScope_enum.CU_FLUSH_GPU_DIRECT_RDMA_WRITES_TO_ALL_DEVICES

class CUflushGPUDirectRDMAWritesTarget(IntEnum):
    """
    The targets for :py:obj:`~.cuFlushGPUDirectRDMAWrites`
    """

    #: Sets the target for :py:obj:`~.cuFlushGPUDirectRDMAWrites()` to the
    #: currently active CUDA device context.
    CU_FLUSH_GPU_DIRECT_RDMA_WRITES_TARGET_CURRENT_CTX = cydriver.CUflushGPUDirectRDMAWritesTarget_enum.CU_FLUSH_GPU_DIRECT_RDMA_WRITES_TARGET_CURRENT_CTX

class CUgraphDebugDot_flags(IntEnum):
    """
    The additional write options for :py:obj:`~.cuGraphDebugDotPrint`
    """
    CU_GRAPH_DEBUG_DOT_FLAGS_VERBOSE = cydriver.CUgraphDebugDot_flags_enum.CU_GRAPH_DEBUG_DOT_FLAGS_VERBOSE

    #: Output all debug data as if every debug flag is enabled
    CU_GRAPH_DEBUG_DOT_FLAGS_RUNTIME_TYPES = cydriver.CUgraphDebugDot_flags_enum.CU_GRAPH_DEBUG_DOT_FLAGS_RUNTIME_TYPES

    #: Use CUDA Runtime structures for output
    CU_GRAPH_DEBUG_DOT_FLAGS_KERNEL_NODE_PARAMS = cydriver.CUgraphDebugDot_flags_enum.CU_GRAPH_DEBUG_DOT_FLAGS_KERNEL_NODE_PARAMS

    #: Adds CUDA_KERNEL_NODE_PARAMS values to output
    CU_GRAPH_DEBUG_DOT_FLAGS_MEMCPY_NODE_PARAMS = cydriver.CUgraphDebugDot_flags_enum.CU_GRAPH_DEBUG_DOT_FLAGS_MEMCPY_NODE_PARAMS

    #: Adds CUDA_MEMCPY3D values to output
    CU_GRAPH_DEBUG_DOT_FLAGS_MEMSET_NODE_PARAMS = cydriver.CUgraphDebugDot_flags_enum.CU_GRAPH_DEBUG_DOT_FLAGS_MEMSET_NODE_PARAMS

    #: Adds CUDA_MEMSET_NODE_PARAMS values to output
    CU_GRAPH_DEBUG_DOT_FLAGS_HOST_NODE_PARAMS = cydriver.CUgraphDebugDot_flags_enum.CU_GRAPH_DEBUG_DOT_FLAGS_HOST_NODE_PARAMS

    #: Adds CUDA_HOST_NODE_PARAMS values to output
    CU_GRAPH_DEBUG_DOT_FLAGS_EVENT_NODE_PARAMS = cydriver.CUgraphDebugDot_flags_enum.CU_GRAPH_DEBUG_DOT_FLAGS_EVENT_NODE_PARAMS

    #: Adds CUevent handle from record and wait nodes to output
    CU_GRAPH_DEBUG_DOT_FLAGS_EXT_SEMAS_SIGNAL_NODE_PARAMS = cydriver.CUgraphDebugDot_flags_enum.CU_GRAPH_DEBUG_DOT_FLAGS_EXT_SEMAS_SIGNAL_NODE_PARAMS

    #: Adds CUDA_EXT_SEM_SIGNAL_NODE_PARAMS values to output
    CU_GRAPH_DEBUG_DOT_FLAGS_EXT_SEMAS_WAIT_NODE_PARAMS = cydriver.CUgraphDebugDot_flags_enum.CU_GRAPH_DEBUG_DOT_FLAGS_EXT_SEMAS_WAIT_NODE_PARAMS

    #: Adds CUDA_EXT_SEM_WAIT_NODE_PARAMS values to output
    CU_GRAPH_DEBUG_DOT_FLAGS_KERNEL_NODE_ATTRIBUTES = cydriver.CUgraphDebugDot_flags_enum.CU_GRAPH_DEBUG_DOT_FLAGS_KERNEL_NODE_ATTRIBUTES

    #: Adds CUkernelNodeAttrValue values to output
    CU_GRAPH_DEBUG_DOT_FLAGS_HANDLES = cydriver.CUgraphDebugDot_flags_enum.CU_GRAPH_DEBUG_DOT_FLAGS_HANDLES

    #: Adds node handles and every kernel function handle to output
    CU_GRAPH_DEBUG_DOT_FLAGS_MEM_ALLOC_NODE_PARAMS = cydriver.CUgraphDebugDot_flags_enum.CU_GRAPH_DEBUG_DOT_FLAGS_MEM_ALLOC_NODE_PARAMS

    #: Adds memory alloc node parameters to output
    CU_GRAPH_DEBUG_DOT_FLAGS_MEM_FREE_NODE_PARAMS = cydriver.CUgraphDebugDot_flags_enum.CU_GRAPH_DEBUG_DOT_FLAGS_MEM_FREE_NODE_PARAMS
    CU_GRAPH_DEBUG_DOT_FLAGS_BATCH_MEM_OP_NODE_PARAMS = cydriver.CUgraphDebugDot_flags_enum.CU_GRAPH_DEBUG_DOT_FLAGS_BATCH_MEM_OP_NODE_PARAMS

class CUuserObject_flags(IntEnum):
    """
    Flags for user objects for graphs
    """

    #: Indicates the destructor execution is not synchronized by any CUDA
    #: handle.
    CU_USER_OBJECT_NO_DESTRUCTOR_SYNC = cydriver.CUuserObject_flags_enum.CU_USER_OBJECT_NO_DESTRUCTOR_SYNC

class CUuserObjectRetain_flags(IntEnum):
    """
    Flags for retaining user object references for graphs
    """

    #: Transfer references from the caller rather than creating new
    #: references.
    CU_GRAPH_USER_OBJECT_MOVE = cydriver.CUuserObjectRetain_flags_enum.CU_GRAPH_USER_OBJECT_MOVE

class CUgraphInstantiate_flags(IntEnum):
    """
    Flags for instantiating a graph
    """

    #: Automatically free memory allocated in a graph before relaunching.
    CUDA_GRAPH_INSTANTIATE_FLAG_AUTO_FREE_ON_LAUNCH = cydriver.CUgraphInstantiate_flags_enum.CUDA_GRAPH_INSTANTIATE_FLAG_AUTO_FREE_ON_LAUNCH

    #: Run the graph using the per-node priority attributes rather than the
    #: priority of the stream it is launched into.
    CUDA_GRAPH_INSTANTIATE_FLAG_USE_NODE_PRIORITY = cydriver.CUgraphInstantiate_flags_enum.CUDA_GRAPH_INSTANTIATE_FLAG_USE_NODE_PRIORITY

class CUmoduleLoadingMode(IntEnum):
    """
    CUDA Lazy Loading status
    """

    #: Lazy Kernel Loading is not enabled
    CU_MODULE_EAGER_LOADING = cydriver.CUmoduleLoadingMode_enum.CU_MODULE_EAGER_LOADING

    #: Lazy Kernel Loading is enabled
    CU_MODULE_LAZY_LOADING = cydriver.CUmoduleLoadingMode_enum.CU_MODULE_LAZY_LOADING

class CUoutput_mode(IntEnum):
    """
    Profiler Output Modes
    """

    #: Output mode Key-Value pair format.
    CU_OUT_KEY_VALUE_PAIR = cydriver.CUoutput_mode_enum.CU_OUT_KEY_VALUE_PAIR

    #: Output mode Comma separated values format.
    CU_OUT_CSV = cydriver.CUoutput_mode_enum.CU_OUT_CSV

class CUeglFrameType(IntEnum):
    """
    CUDA EglFrame type - array or pointer
    """

    #: Frame type CUDA array
    CU_EGL_FRAME_TYPE_ARRAY = cydriver.CUeglFrameType_enum.CU_EGL_FRAME_TYPE_ARRAY

    #: Frame type pointer
    CU_EGL_FRAME_TYPE_PITCH = cydriver.CUeglFrameType_enum.CU_EGL_FRAME_TYPE_PITCH

class CUeglResourceLocationFlags(IntEnum):
    """
    Resource location flags- sysmem or vidmem  For CUDA context on
    iGPU, since video and system memory are equivalent - these flags
    will not have an effect on the execution.  For CUDA context on
    dGPU, applications can use the flag
    :py:obj:`~.CUeglResourceLocationFlags` to give a hint about the
    desired location.  :py:obj:`~.CU_EGL_RESOURCE_LOCATION_SYSMEM` -
    the frame data is made resident on the system memory to be accessed
    by CUDA.  :py:obj:`~.CU_EGL_RESOURCE_LOCATION_VIDMEM` - the frame
    data is made resident on the dedicated video memory to be accessed
    by CUDA.  There may be an additional latency due to new allocation
    and data migration, if the frame is produced on a different memory.
    """

    #: Resource location sysmem
    CU_EGL_RESOURCE_LOCATION_SYSMEM = cydriver.CUeglResourceLocationFlags_enum.CU_EGL_RESOURCE_LOCATION_SYSMEM

    #: Resource location vidmem
    CU_EGL_RESOURCE_LOCATION_VIDMEM = cydriver.CUeglResourceLocationFlags_enum.CU_EGL_RESOURCE_LOCATION_VIDMEM

class CUeglColorFormat(IntEnum):
    """
    CUDA EGL Color Format - The different planar and multiplanar
    formats currently supported for CUDA_EGL interops. Three channel
    formats are currently not supported for
    :py:obj:`~.CU_EGL_FRAME_TYPE_ARRAY`
    """

    #: Y, U, V in three surfaces, each in a separate surface, U/V width =
    #: 1/2 Y width, U/V height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_YUV420_PLANAR = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUV420_PLANAR

    #: Y, UV in two surfaces (UV as one surface) with VU byte ordering,
    #: width, height ratio same as YUV420Planar.
    CU_EGL_COLOR_FORMAT_YUV420_SEMIPLANAR = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUV420_SEMIPLANAR

    #: Y, U, V each in a separate surface, U/V width = 1/2 Y width, U/V
    #: height = Y height.
    CU_EGL_COLOR_FORMAT_YUV422_PLANAR = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUV422_PLANAR

    #: Y, UV in two surfaces with VU byte ordering, width, height ratio
    #: same as YUV422Planar.
    CU_EGL_COLOR_FORMAT_YUV422_SEMIPLANAR = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUV422_SEMIPLANAR

    #: R/G/B three channels in one surface with BGR byte ordering. Only
    #: pitch linear format supported.
    CU_EGL_COLOR_FORMAT_RGB = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_RGB

    #: R/G/B three channels in one surface with RGB byte ordering. Only
    #: pitch linear format supported.
    CU_EGL_COLOR_FORMAT_BGR = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BGR

    #: R/G/B/A four channels in one surface with BGRA byte ordering.
    CU_EGL_COLOR_FORMAT_ARGB = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_ARGB

    #: R/G/B/A four channels in one surface with ABGR byte ordering.
    CU_EGL_COLOR_FORMAT_RGBA = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_RGBA

    #: single luminance channel in one surface.
    CU_EGL_COLOR_FORMAT_L = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_L

    #: single color channel in one surface.
    CU_EGL_COLOR_FORMAT_R = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_R

    #: Y, U, V in three surfaces, each in a separate surface, U/V width = Y
    #: width, U/V height = Y height.
    CU_EGL_COLOR_FORMAT_YUV444_PLANAR = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUV444_PLANAR

    #: Y, UV in two surfaces (UV as one surface) with VU byte ordering,
    #: width, height ratio same as YUV444Planar.
    CU_EGL_COLOR_FORMAT_YUV444_SEMIPLANAR = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUV444_SEMIPLANAR

    #: Y, U, V in one surface, interleaved as UYVY in one channel.
    CU_EGL_COLOR_FORMAT_YUYV_422 = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUYV_422

    #: Y, U, V in one surface, interleaved as YUYV in one channel.
    CU_EGL_COLOR_FORMAT_UYVY_422 = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_UYVY_422

    #: R/G/B/A four channels in one surface with RGBA byte ordering.
    CU_EGL_COLOR_FORMAT_ABGR = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_ABGR

    #: R/G/B/A four channels in one surface with ARGB byte ordering.
    CU_EGL_COLOR_FORMAT_BGRA = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BGRA

    #: Alpha color format - one channel in one surface.
    CU_EGL_COLOR_FORMAT_A = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_A

    #: R/G color format - two channels in one surface with GR byte ordering
    CU_EGL_COLOR_FORMAT_RG = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_RG

    #: Y, U, V, A four channels in one surface, interleaved as VUYA.
    CU_EGL_COLOR_FORMAT_AYUV = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_AYUV

    #: Y, VU in two surfaces (VU as one surface) with UV byte ordering, U/V
    #: width = Y width, U/V height = Y height.
    CU_EGL_COLOR_FORMAT_YVU444_SEMIPLANAR = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YVU444_SEMIPLANAR

    #: Y, VU in two surfaces (VU as one surface) with UV byte ordering, U/V
    #: width = 1/2 Y width, U/V height = Y height.
    CU_EGL_COLOR_FORMAT_YVU422_SEMIPLANAR = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YVU422_SEMIPLANAR

    #: Y, VU in two surfaces (VU as one surface) with UV byte ordering, U/V
    #: width = 1/2 Y width, U/V height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_YVU420_SEMIPLANAR = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YVU420_SEMIPLANAR

    #: Y10, V10U10 in two surfaces (VU as one surface) with UV byte
    #: ordering, U/V width = Y width, U/V height = Y height.
    CU_EGL_COLOR_FORMAT_Y10V10U10_444_SEMIPLANAR = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y10V10U10_444_SEMIPLANAR

    #: Y10, V10U10 in two surfaces (VU as one surface) with UV byte
    #: ordering, U/V width = 1/2 Y width, U/V height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_Y10V10U10_420_SEMIPLANAR = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y10V10U10_420_SEMIPLANAR

    #: Y12, V12U12 in two surfaces (VU as one surface) with UV byte
    #: ordering, U/V width = Y width, U/V height = Y height.
    CU_EGL_COLOR_FORMAT_Y12V12U12_444_SEMIPLANAR = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y12V12U12_444_SEMIPLANAR

    #: Y12, V12U12 in two surfaces (VU as one surface) with UV byte
    #: ordering, U/V width = 1/2 Y width, U/V height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_Y12V12U12_420_SEMIPLANAR = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y12V12U12_420_SEMIPLANAR

    #: Extended Range Y, U, V in one surface, interleaved as YVYU in one
    #: channel.
    CU_EGL_COLOR_FORMAT_VYUY_ER = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_VYUY_ER

    #: Extended Range Y, U, V in one surface, interleaved as YUYV in one
    #: channel.
    CU_EGL_COLOR_FORMAT_UYVY_ER = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_UYVY_ER

    #: Extended Range Y, U, V in one surface, interleaved as UYVY in one
    #: channel.
    CU_EGL_COLOR_FORMAT_YUYV_ER = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUYV_ER

    #: Extended Range Y, U, V in one surface, interleaved as VYUY in one
    #: channel.
    CU_EGL_COLOR_FORMAT_YVYU_ER = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YVYU_ER

    #: Extended Range Y, U, V three channels in one surface, interleaved as
    #: VUY. Only pitch linear format supported.
    CU_EGL_COLOR_FORMAT_YUV_ER = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUV_ER

    #: Extended Range Y, U, V, A four channels in one surface, interleaved
    #: as AVUY.
    CU_EGL_COLOR_FORMAT_YUVA_ER = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUVA_ER

    #: Extended Range Y, U, V, A four channels in one surface, interleaved
    #: as VUYA.
    CU_EGL_COLOR_FORMAT_AYUV_ER = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_AYUV_ER

    #: Extended Range Y, U, V in three surfaces, U/V width = Y width, U/V
    #: height = Y height.
    CU_EGL_COLOR_FORMAT_YUV444_PLANAR_ER = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUV444_PLANAR_ER

    #: Extended Range Y, U, V in three surfaces, U/V width = 1/2 Y width,
    #: U/V height = Y height.
    CU_EGL_COLOR_FORMAT_YUV422_PLANAR_ER = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUV422_PLANAR_ER

    #: Extended Range Y, U, V in three surfaces, U/V width = 1/2 Y width,
    #: U/V height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_YUV420_PLANAR_ER = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUV420_PLANAR_ER

    #: Extended Range Y, UV in two surfaces (UV as one surface) with VU
    #: byte ordering, U/V width = Y width, U/V height = Y height.
    CU_EGL_COLOR_FORMAT_YUV444_SEMIPLANAR_ER = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUV444_SEMIPLANAR_ER

    #: Extended Range Y, UV in two surfaces (UV as one surface) with VU
    #: byte ordering, U/V width = 1/2 Y width, U/V height = Y height.
    CU_EGL_COLOR_FORMAT_YUV422_SEMIPLANAR_ER = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUV422_SEMIPLANAR_ER

    #: Extended Range Y, UV in two surfaces (UV as one surface) with VU
    #: byte ordering, U/V width = 1/2 Y width, U/V height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_YUV420_SEMIPLANAR_ER = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUV420_SEMIPLANAR_ER

    #: Extended Range Y, V, U in three surfaces, U/V width = Y width, U/V
    #: height = Y height.
    CU_EGL_COLOR_FORMAT_YVU444_PLANAR_ER = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YVU444_PLANAR_ER

    #: Extended Range Y, V, U in three surfaces, U/V width = 1/2 Y width,
    #: U/V height = Y height.
    CU_EGL_COLOR_FORMAT_YVU422_PLANAR_ER = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YVU422_PLANAR_ER

    #: Extended Range Y, V, U in three surfaces, U/V width = 1/2 Y width,
    #: U/V height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_YVU420_PLANAR_ER = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YVU420_PLANAR_ER

    #: Extended Range Y, VU in two surfaces (VU as one surface) with UV
    #: byte ordering, U/V width = Y width, U/V height = Y height.
    CU_EGL_COLOR_FORMAT_YVU444_SEMIPLANAR_ER = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YVU444_SEMIPLANAR_ER

    #: Extended Range Y, VU in two surfaces (VU as one surface) with UV
    #: byte ordering, U/V width = 1/2 Y width, U/V height = Y height.
    CU_EGL_COLOR_FORMAT_YVU422_SEMIPLANAR_ER = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YVU422_SEMIPLANAR_ER

    #: Extended Range Y, VU in two surfaces (VU as one surface) with UV
    #: byte ordering, U/V width = 1/2 Y width, U/V height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_YVU420_SEMIPLANAR_ER = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YVU420_SEMIPLANAR_ER

    #: Bayer format - one channel in one surface with interleaved RGGB
    #: ordering.
    CU_EGL_COLOR_FORMAT_BAYER_RGGB = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER_RGGB

    #: Bayer format - one channel in one surface with interleaved BGGR
    #: ordering.
    CU_EGL_COLOR_FORMAT_BAYER_BGGR = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER_BGGR

    #: Bayer format - one channel in one surface with interleaved GRBG
    #: ordering.
    CU_EGL_COLOR_FORMAT_BAYER_GRBG = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER_GRBG

    #: Bayer format - one channel in one surface with interleaved GBRG
    #: ordering.
    CU_EGL_COLOR_FORMAT_BAYER_GBRG = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER_GBRG

    #: Bayer10 format - one channel in one surface with interleaved RGGB
    #: ordering. Out of 16 bits, 10 bits used 6 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER10_RGGB = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER10_RGGB

    #: Bayer10 format - one channel in one surface with interleaved BGGR
    #: ordering. Out of 16 bits, 10 bits used 6 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER10_BGGR = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER10_BGGR

    #: Bayer10 format - one channel in one surface with interleaved GRBG
    #: ordering. Out of 16 bits, 10 bits used 6 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER10_GRBG = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER10_GRBG

    #: Bayer10 format - one channel in one surface with interleaved GBRG
    #: ordering. Out of 16 bits, 10 bits used 6 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER10_GBRG = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER10_GBRG

    #: Bayer12 format - one channel in one surface with interleaved RGGB
    #: ordering. Out of 16 bits, 12 bits used 4 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER12_RGGB = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER12_RGGB

    #: Bayer12 format - one channel in one surface with interleaved BGGR
    #: ordering. Out of 16 bits, 12 bits used 4 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER12_BGGR = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER12_BGGR

    #: Bayer12 format - one channel in one surface with interleaved GRBG
    #: ordering. Out of 16 bits, 12 bits used 4 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER12_GRBG = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER12_GRBG

    #: Bayer12 format - one channel in one surface with interleaved GBRG
    #: ordering. Out of 16 bits, 12 bits used 4 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER12_GBRG = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER12_GBRG

    #: Bayer14 format - one channel in one surface with interleaved RGGB
    #: ordering. Out of 16 bits, 14 bits used 2 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER14_RGGB = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER14_RGGB

    #: Bayer14 format - one channel in one surface with interleaved BGGR
    #: ordering. Out of 16 bits, 14 bits used 2 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER14_BGGR = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER14_BGGR

    #: Bayer14 format - one channel in one surface with interleaved GRBG
    #: ordering. Out of 16 bits, 14 bits used 2 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER14_GRBG = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER14_GRBG

    #: Bayer14 format - one channel in one surface with interleaved GBRG
    #: ordering. Out of 16 bits, 14 bits used 2 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER14_GBRG = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER14_GBRG

    #: Bayer20 format - one channel in one surface with interleaved RGGB
    #: ordering. Out of 32 bits, 20 bits used 12 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER20_RGGB = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER20_RGGB

    #: Bayer20 format - one channel in one surface with interleaved BGGR
    #: ordering. Out of 32 bits, 20 bits used 12 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER20_BGGR = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER20_BGGR

    #: Bayer20 format - one channel in one surface with interleaved GRBG
    #: ordering. Out of 32 bits, 20 bits used 12 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER20_GRBG = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER20_GRBG

    #: Bayer20 format - one channel in one surface with interleaved GBRG
    #: ordering. Out of 32 bits, 20 bits used 12 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER20_GBRG = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER20_GBRG

    #: Y, V, U in three surfaces, each in a separate surface, U/V width = Y
    #: width, U/V height = Y height.
    CU_EGL_COLOR_FORMAT_YVU444_PLANAR = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YVU444_PLANAR

    #: Y, V, U in three surfaces, each in a separate surface, U/V width =
    #: 1/2 Y width, U/V height = Y height.
    CU_EGL_COLOR_FORMAT_YVU422_PLANAR = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YVU422_PLANAR

    #: Y, V, U in three surfaces, each in a separate surface, U/V width =
    #: 1/2 Y width, U/V height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_YVU420_PLANAR = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YVU420_PLANAR

    #: Nvidia proprietary Bayer ISP format - one channel in one surface
    #: with interleaved RGGB ordering and mapped to opaque integer
    #: datatype.
    CU_EGL_COLOR_FORMAT_BAYER_ISP_RGGB = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER_ISP_RGGB

    #: Nvidia proprietary Bayer ISP format - one channel in one surface
    #: with interleaved BGGR ordering and mapped to opaque integer
    #: datatype.
    CU_EGL_COLOR_FORMAT_BAYER_ISP_BGGR = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER_ISP_BGGR

    #: Nvidia proprietary Bayer ISP format - one channel in one surface
    #: with interleaved GRBG ordering and mapped to opaque integer
    #: datatype.
    CU_EGL_COLOR_FORMAT_BAYER_ISP_GRBG = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER_ISP_GRBG

    #: Nvidia proprietary Bayer ISP format - one channel in one surface
    #: with interleaved GBRG ordering and mapped to opaque integer
    #: datatype.
    CU_EGL_COLOR_FORMAT_BAYER_ISP_GBRG = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER_ISP_GBRG

    #: Bayer format - one channel in one surface with interleaved BCCR
    #: ordering.
    CU_EGL_COLOR_FORMAT_BAYER_BCCR = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER_BCCR

    #: Bayer format - one channel in one surface with interleaved RCCB
    #: ordering.
    CU_EGL_COLOR_FORMAT_BAYER_RCCB = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER_RCCB

    #: Bayer format - one channel in one surface with interleaved CRBC
    #: ordering.
    CU_EGL_COLOR_FORMAT_BAYER_CRBC = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER_CRBC

    #: Bayer format - one channel in one surface with interleaved CBRC
    #: ordering.
    CU_EGL_COLOR_FORMAT_BAYER_CBRC = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER_CBRC

    #: Bayer10 format - one channel in one surface with interleaved CCCC
    #: ordering. Out of 16 bits, 10 bits used 6 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER10_CCCC = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER10_CCCC

    #: Bayer12 format - one channel in one surface with interleaved BCCR
    #: ordering. Out of 16 bits, 12 bits used 4 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER12_BCCR = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER12_BCCR

    #: Bayer12 format - one channel in one surface with interleaved RCCB
    #: ordering. Out of 16 bits, 12 bits used 4 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER12_RCCB = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER12_RCCB

    #: Bayer12 format - one channel in one surface with interleaved CRBC
    #: ordering. Out of 16 bits, 12 bits used 4 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER12_CRBC = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER12_CRBC

    #: Bayer12 format - one channel in one surface with interleaved CBRC
    #: ordering. Out of 16 bits, 12 bits used 4 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER12_CBRC = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER12_CBRC

    #: Bayer12 format - one channel in one surface with interleaved CCCC
    #: ordering. Out of 16 bits, 12 bits used 4 bits No-op.
    CU_EGL_COLOR_FORMAT_BAYER12_CCCC = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_BAYER12_CCCC

    #: Color format for single Y plane.
    CU_EGL_COLOR_FORMAT_Y = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y

    #: Y, UV in two surfaces (UV as one surface) U/V width = 1/2 Y width,
    #: U/V height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_YUV420_SEMIPLANAR_2020 = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUV420_SEMIPLANAR_2020

    #: Y, VU in two surfaces (VU as one surface) U/V width = 1/2 Y width,
    #: U/V height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_YVU420_SEMIPLANAR_2020 = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YVU420_SEMIPLANAR_2020

    #: Y, U, V each in a separate surface, U/V width = 1/2 Y width, U/V
    #: height= 1/2 Y height.
    CU_EGL_COLOR_FORMAT_YUV420_PLANAR_2020 = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUV420_PLANAR_2020

    #: Y, V, U each in a separate surface, U/V width = 1/2 Y width, U/V
    #: height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_YVU420_PLANAR_2020 = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YVU420_PLANAR_2020

    #: Y, UV in two surfaces (UV as one surface) U/V width = 1/2 Y width,
    #: U/V height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_YUV420_SEMIPLANAR_709 = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUV420_SEMIPLANAR_709

    #: Y, VU in two surfaces (VU as one surface) U/V width = 1/2 Y width,
    #: U/V height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_YVU420_SEMIPLANAR_709 = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YVU420_SEMIPLANAR_709

    #: Y, U, V each in a separate surface, U/V width = 1/2 Y width, U/V
    #: height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_YUV420_PLANAR_709 = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUV420_PLANAR_709

    #: Y, V, U each in a separate surface, U/V width = 1/2 Y width, U/V
    #: height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_YVU420_PLANAR_709 = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YVU420_PLANAR_709

    #: Y10, V10U10 in two surfaces (VU as one surface), U/V width = 1/2 Y
    #: width, U/V height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_Y10V10U10_420_SEMIPLANAR_709 = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y10V10U10_420_SEMIPLANAR_709

    #: Y10, V10U10 in two surfaces (VU as one surface), U/V width = 1/2 Y
    #: width, U/V height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_Y10V10U10_420_SEMIPLANAR_2020 = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y10V10U10_420_SEMIPLANAR_2020

    #: Y10, V10U10 in two surfaces(VU as one surface) U/V width = 1/2 Y
    #: width, U/V height = Y height.
    CU_EGL_COLOR_FORMAT_Y10V10U10_422_SEMIPLANAR_2020 = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y10V10U10_422_SEMIPLANAR_2020

    #: Y10, V10U10 in two surfaces(VU as one surface) U/V width = 1/2 Y
    #: width, U/V height = Y height.
    CU_EGL_COLOR_FORMAT_Y10V10U10_422_SEMIPLANAR = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y10V10U10_422_SEMIPLANAR

    #: Y10, V10U10 in two surfaces(VU as one surface) U/V width = 1/2 Y
    #: width, U/V height = Y height.
    CU_EGL_COLOR_FORMAT_Y10V10U10_422_SEMIPLANAR_709 = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y10V10U10_422_SEMIPLANAR_709

    #: Extended Range Color format for single Y plane.
    CU_EGL_COLOR_FORMAT_Y_ER = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y_ER

    #: Extended Range Color format for single Y plane.
    CU_EGL_COLOR_FORMAT_Y_709_ER = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y_709_ER

    #: Extended Range Color format for single Y10 plane.
    CU_EGL_COLOR_FORMAT_Y10_ER = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y10_ER

    #: Extended Range Color format for single Y10 plane.
    CU_EGL_COLOR_FORMAT_Y10_709_ER = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y10_709_ER

    #: Extended Range Color format for single Y12 plane.
    CU_EGL_COLOR_FORMAT_Y12_ER = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y12_ER

    #: Extended Range Color format for single Y12 plane.
    CU_EGL_COLOR_FORMAT_Y12_709_ER = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y12_709_ER

    #: Y, U, V, A four channels in one surface, interleaved as AVUY.
    CU_EGL_COLOR_FORMAT_YUVA = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUVA

    #: Y, U, V three channels in one surface, interleaved as VUY. Only
    #: pitch linear format supported.
    CU_EGL_COLOR_FORMAT_YUV = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YUV

    #: Y, U, V in one surface, interleaved as YVYU in one channel.
    CU_EGL_COLOR_FORMAT_YVYU = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_YVYU

    #: Y, U, V in one surface, interleaved as VYUY in one channel.
    CU_EGL_COLOR_FORMAT_VYUY = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_VYUY

    #: Extended Range Y10, V10U10 in two surfaces(VU as one surface) U/V
    #: width = 1/2 Y width, U/V height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_Y10V10U10_420_SEMIPLANAR_ER = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y10V10U10_420_SEMIPLANAR_ER

    #: Extended Range Y10, V10U10 in two surfaces(VU as one surface) U/V
    #: width = 1/2 Y width, U/V height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_Y10V10U10_420_SEMIPLANAR_709_ER = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y10V10U10_420_SEMIPLANAR_709_ER

    #: Extended Range Y10, V10U10 in two surfaces (VU as one surface) U/V
    #: width = Y width, U/V height = Y height.
    CU_EGL_COLOR_FORMAT_Y10V10U10_444_SEMIPLANAR_ER = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y10V10U10_444_SEMIPLANAR_ER

    #: Extended Range Y10, V10U10 in two surfaces (VU as one surface) U/V
    #: width = Y width, U/V height = Y height.
    CU_EGL_COLOR_FORMAT_Y10V10U10_444_SEMIPLANAR_709_ER = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y10V10U10_444_SEMIPLANAR_709_ER

    #: Extended Range Y12, V12U12 in two surfaces (VU as one surface) U/V
    #: width = 1/2 Y width, U/V height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_Y12V12U12_420_SEMIPLANAR_ER = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y12V12U12_420_SEMIPLANAR_ER

    #: Extended Range Y12, V12U12 in two surfaces (VU as one surface) U/V
    #: width = 1/2 Y width, U/V height = 1/2 Y height.
    CU_EGL_COLOR_FORMAT_Y12V12U12_420_SEMIPLANAR_709_ER = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y12V12U12_420_SEMIPLANAR_709_ER

    #: Extended Range Y12, V12U12 in two surfaces (VU as one surface) U/V
    #: width = Y width, U/V height = Y height.
    CU_EGL_COLOR_FORMAT_Y12V12U12_444_SEMIPLANAR_ER = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y12V12U12_444_SEMIPLANAR_ER

    #: Extended Range Y12, V12U12 in two surfaces (VU as one surface) U/V
    #: width = Y width, U/V height = Y height.
    CU_EGL_COLOR_FORMAT_Y12V12U12_444_SEMIPLANAR_709_ER = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_Y12V12U12_444_SEMIPLANAR_709_ER
    CU_EGL_COLOR_FORMAT_MAX = cydriver.CUeglColorFormat_enum.CU_EGL_COLOR_FORMAT_MAX

class CUGLDeviceList(IntEnum):
    """
    CUDA devices corresponding to an OpenGL device
    """

    #: The CUDA devices for all GPUs used by the current OpenGL context
    CU_GL_DEVICE_LIST_ALL = cydriver.CUGLDeviceList_enum.CU_GL_DEVICE_LIST_ALL

    #: The CUDA devices for the GPUs used by the current OpenGL context in
    #: its currently rendering frame
    CU_GL_DEVICE_LIST_CURRENT_FRAME = cydriver.CUGLDeviceList_enum.CU_GL_DEVICE_LIST_CURRENT_FRAME

    #: The CUDA devices for the GPUs to be used by the current OpenGL
    #: context in the next frame
    CU_GL_DEVICE_LIST_NEXT_FRAME = cydriver.CUGLDeviceList_enum.CU_GL_DEVICE_LIST_NEXT_FRAME

class CUGLmap_flags(IntEnum):
    """
    Flags to map or unmap a resource
    """
    CU_GL_MAP_RESOURCE_FLAGS_NONE = cydriver.CUGLmap_flags_enum.CU_GL_MAP_RESOURCE_FLAGS_NONE
    CU_GL_MAP_RESOURCE_FLAGS_READ_ONLY = cydriver.CUGLmap_flags_enum.CU_GL_MAP_RESOURCE_FLAGS_READ_ONLY
    CU_GL_MAP_RESOURCE_FLAGS_WRITE_DISCARD = cydriver.CUGLmap_flags_enum.CU_GL_MAP_RESOURCE_FLAGS_WRITE_DISCARD

cdef class CUdeviceptr:
    """

    CUDA device pointer CUdeviceptr is defined as an unsigned integer type whose size matches the size of a pointer on the target platform.

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, unsigned long long init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUdeviceptr *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<CUdeviceptr ' + str(self.__int__()) + '>'
    def __int__(self):
        return <unsigned long long>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUdevice:
    """

    CUDA device

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, int init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUdevice *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<CUdevice ' + str(self.__int__()) + '>'
    def __int__(self):
        return <int>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUtexObject:
    """

    An opaque value that represents a CUDA texture object

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, unsigned long long init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUtexObject *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<CUtexObject ' + str(self.__int__()) + '>'
    def __int__(self):
        return <unsigned long long>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUsurfObject:
    """

    An opaque value that represents a CUDA surface object

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, unsigned long long init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUsurfObject *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<CUsurfObject ' + str(self.__int__()) + '>'
    def __int__(self):
        return <unsigned long long>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

class CUkernelNodeAttrID(IntEnum):
    """

    """

    #: Ignored entry, for convenient composition
    CU_LAUNCH_ATTRIBUTE_IGNORE = cydriver.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_IGNORE

    #: Valid for streams, graph nodes, launches.
    CU_LAUNCH_ATTRIBUTE_ACCESS_POLICY_WINDOW = cydriver.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_ACCESS_POLICY_WINDOW

    #: Valid for graph nodes, launches.
    CU_LAUNCH_ATTRIBUTE_COOPERATIVE = cydriver.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_COOPERATIVE

    #: Valid for streams.
    CU_LAUNCH_ATTRIBUTE_SYNCHRONIZATION_POLICY = cydriver.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_SYNCHRONIZATION_POLICY

    #: Valid for graph nodes, launches.
    CU_LAUNCH_ATTRIBUTE_CLUSTER_DIMENSION = cydriver.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_CLUSTER_DIMENSION

    #: Valid for graph nodes, launches.
    CU_LAUNCH_ATTRIBUTE_CLUSTER_SCHEDULING_POLICY_PREFERENCE = cydriver.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_CLUSTER_SCHEDULING_POLICY_PREFERENCE

    #: Valid for launches. Setting programmaticStreamSerializationAllowed
    #: to non-0 signals that the kernel will use programmatic means to
    #: resolve its stream dependency, so that the CUDA runtime should
    #: opportunistically allow the grid's execution to overlap with the
    #: previous kernel in the stream, if that kernel requests the overlap.
    CU_LAUNCH_ATTRIBUTE_PROGRAMMATIC_STREAM_SERIALIZATION = cydriver.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_PROGRAMMATIC_STREAM_SERIALIZATION

    #: Valid for launches. Event recorded through this launch attribute is
    #: guaranteed to only trigger after all block in the associated kernel
    #: trigger the event. A block can trigger the event through PTX
    #: griddepcontrol.launch_dependents. A trigger can also be inserted at
    #: the beginning of each block's execution if triggerAtBlockStart is
    #: set to non-0. Note that dependents (including the CPU thread calling
    #: :py:obj:`~.cuEventSynchronize()`) are not guaranteed to observe the
    #: release precisely when it is released. For example,
    #: :py:obj:`~.cuEventSynchronize()` may only observe the event trigger
    #: long after the associated kernel has completed. This recording type
    #: is primarily meant for establishing programmatic dependency between
    #: device tasks. The event supplied must not be an interprocess or
    #: interop event. The event must disable timing (i.e. created with
    #: :py:obj:`~.CU_EVENT_DISABLE_TIMING` flag set).
    CU_LAUNCH_ATTRIBUTE_PROGRAMMATIC_EVENT = cydriver.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_PROGRAMMATIC_EVENT

    #: Valid for graph nodes.
    CU_LAUNCH_ATTRIBUTE_PRIORITY = cydriver.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_PRIORITY

class CUstreamAttrID(IntEnum):
    """

    """

    #: Ignored entry, for convenient composition
    CU_LAUNCH_ATTRIBUTE_IGNORE = cydriver.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_IGNORE

    #: Valid for streams, graph nodes, launches.
    CU_LAUNCH_ATTRIBUTE_ACCESS_POLICY_WINDOW = cydriver.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_ACCESS_POLICY_WINDOW

    #: Valid for graph nodes, launches.
    CU_LAUNCH_ATTRIBUTE_COOPERATIVE = cydriver.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_COOPERATIVE

    #: Valid for streams.
    CU_LAUNCH_ATTRIBUTE_SYNCHRONIZATION_POLICY = cydriver.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_SYNCHRONIZATION_POLICY

    #: Valid for graph nodes, launches.
    CU_LAUNCH_ATTRIBUTE_CLUSTER_DIMENSION = cydriver.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_CLUSTER_DIMENSION

    #: Valid for graph nodes, launches.
    CU_LAUNCH_ATTRIBUTE_CLUSTER_SCHEDULING_POLICY_PREFERENCE = cydriver.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_CLUSTER_SCHEDULING_POLICY_PREFERENCE

    #: Valid for launches. Setting programmaticStreamSerializationAllowed
    #: to non-0 signals that the kernel will use programmatic means to
    #: resolve its stream dependency, so that the CUDA runtime should
    #: opportunistically allow the grid's execution to overlap with the
    #: previous kernel in the stream, if that kernel requests the overlap.
    CU_LAUNCH_ATTRIBUTE_PROGRAMMATIC_STREAM_SERIALIZATION = cydriver.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_PROGRAMMATIC_STREAM_SERIALIZATION

    #: Valid for launches. Event recorded through this launch attribute is
    #: guaranteed to only trigger after all block in the associated kernel
    #: trigger the event. A block can trigger the event through PTX
    #: griddepcontrol.launch_dependents. A trigger can also be inserted at
    #: the beginning of each block's execution if triggerAtBlockStart is
    #: set to non-0. Note that dependents (including the CPU thread calling
    #: :py:obj:`~.cuEventSynchronize()`) are not guaranteed to observe the
    #: release precisely when it is released. For example,
    #: :py:obj:`~.cuEventSynchronize()` may only observe the event trigger
    #: long after the associated kernel has completed. This recording type
    #: is primarily meant for establishing programmatic dependency between
    #: device tasks. The event supplied must not be an interprocess or
    #: interop event. The event must disable timing (i.e. created with
    #: :py:obj:`~.CU_EVENT_DISABLE_TIMING` flag set).
    CU_LAUNCH_ATTRIBUTE_PROGRAMMATIC_EVENT = cydriver.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_PROGRAMMATIC_EVENT

    #: Valid for graph nodes.
    CU_LAUNCH_ATTRIBUTE_PRIORITY = cydriver.CUlaunchAttributeID_enum.CU_LAUNCH_ATTRIBUTE_PRIORITY

cdef class CUmemGenericAllocationHandle:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, unsigned long long init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUmemGenericAllocationHandle *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<CUmemGenericAllocationHandle ' + str(self.__int__()) + '>'
    def __int__(self):
        return <unsigned long long>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUcontext:
    """

    CUDA context

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <cydriver.CUcontext>init_value
        else:
            self._ptr = <cydriver.CUcontext *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUcontext ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUmodule:
    """

    CUDA module

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <cydriver.CUmodule>init_value
        else:
            self._ptr = <cydriver.CUmodule *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUmodule ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUfunction:
    """

    CUDA function

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <cydriver.CUfunction>init_value
        else:
            self._ptr = <cydriver.CUfunction *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUfunction ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUarray:
    """

    CUDA array

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <cydriver.CUarray>init_value
        else:
            self._ptr = <cydriver.CUarray *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUarray ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUmipmappedArray:
    """

    CUDA mipmapped array

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <cydriver.CUmipmappedArray>init_value
        else:
            self._ptr = <cydriver.CUmipmappedArray *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUmipmappedArray ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUtexref:
    """

    CUDA texture reference

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <cydriver.CUtexref>init_value
        else:
            self._ptr = <cydriver.CUtexref *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUtexref ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUsurfref:
    """

    CUDA surface reference

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <cydriver.CUsurfref>init_value
        else:
            self._ptr = <cydriver.CUsurfref *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUsurfref ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUevent:
    """

    CUDA event

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <cydriver.CUevent>init_value
        else:
            self._ptr = <cydriver.CUevent *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUevent ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUstream:
    """

    CUDA stream

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <cydriver.CUstream>init_value
        else:
            self._ptr = <cydriver.CUstream *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUstream ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUgraphicsResource:
    """

    CUDA graphics interop resource

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <cydriver.CUgraphicsResource>init_value
        else:
            self._ptr = <cydriver.CUgraphicsResource *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUgraphicsResource ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUexternalMemory:
    """

    CUDA external memory

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <cydriver.CUexternalMemory>init_value
        else:
            self._ptr = <cydriver.CUexternalMemory *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUexternalMemory ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUexternalSemaphore:
    """

    CUDA external semaphore

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <cydriver.CUexternalSemaphore>init_value
        else:
            self._ptr = <cydriver.CUexternalSemaphore *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUexternalSemaphore ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUgraph:
    """

    CUDA graph

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <cydriver.CUgraph>init_value
        else:
            self._ptr = <cydriver.CUgraph *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUgraph ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUgraphNode:
    """

    CUDA graph node

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <cydriver.CUgraphNode>init_value
        else:
            self._ptr = <cydriver.CUgraphNode *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUgraphNode ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUgraphExec:
    """

    CUDA executable graph

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <cydriver.CUgraphExec>init_value
        else:
            self._ptr = <cydriver.CUgraphExec *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUgraphExec ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUmemoryPool:
    """

    CUDA memory pool

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <cydriver.CUmemoryPool>init_value
        else:
            self._ptr = <cydriver.CUmemoryPool *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUmemoryPool ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUuserObject:
    """

    CUDA user object for graphs

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <cydriver.CUuserObject>init_value
        else:
            self._ptr = <cydriver.CUuserObject *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUuserObject ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUlinkState:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <cydriver.CUlinkState>init_value
        else:
            self._ptr = <cydriver.CUlinkState *>_ptr
    def __init__(self, *args, **kwargs):
        self._keepalive = []
    def __repr__(self):
        return '<CUlinkState ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUeglStreamConnection:
    """

    CUDA EGLSream Connection

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <cydriver.CUeglStreamConnection>init_value
        else:
            self._ptr = <cydriver.CUeglStreamConnection *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUeglStreamConnection ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class EGLImageKHR:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <cydriver.EGLImageKHR>init_value
        else:
            self._ptr = <cydriver.EGLImageKHR *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<EGLImageKHR ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class EGLStreamKHR:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <cydriver.EGLStreamKHR>init_value
        else:
            self._ptr = <cydriver.EGLStreamKHR *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<EGLStreamKHR ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class EGLSyncKHR:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <cydriver.EGLSyncKHR>init_value
        else:
            self._ptr = <cydriver.EGLSyncKHR *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<EGLSyncKHR ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUhostFn:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <cydriver.CUhostFn>init_value
        else:
            self._ptr = <cydriver.CUhostFn *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUhostFn ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUstreamCallback:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <cydriver.CUstreamCallback>init_value
        else:
            self._ptr = <cydriver.CUstreamCallback *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUstreamCallback ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUoccupancyB2DSize:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
            self._ptr[0] = <cydriver.CUoccupancyB2DSize>init_value
        else:
            self._ptr = <cydriver.CUoccupancyB2DSize *>_ptr
    def __init__(self, *args, **kwargs):
        pass
    def __repr__(self):
        return '<CUoccupancyB2DSize ' + str(hex(self.__int__())) + '>'
    def __index__(self):
        return self.__int__()
    def __int__(self):
        return <void_ptr>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUuuid_st:
    """
    Attributes
    ----------
    bytes : bytes
        < CUDA definition of UUID

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUuuid_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['bytes : ' + str(self.bytes.hex())]
            except ValueError:
                str_list += ['bytes : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def bytes(self):
        return PyBytes_FromStringAndSize(self._ptr[0].bytes, 16)

cdef class CUipcEventHandle_st:
    """
    CUDA IPC event handle

    Attributes
    ----------
    reserved : bytes


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUipcEventHandle_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def reserved(self):
        return PyBytes_FromStringAndSize(self._ptr[0].reserved, 64)
    @reserved.setter
    def reserved(self, reserved):
        if len(reserved) != 64:
            raise ValueError("reserved length must be 64, is " + str(len(reserved)))
        if CHAR_MIN == 0:
            for i, b in enumerate(reserved):
                if b < 0 and b > -129:
                    b = b + 256
                self._ptr[0].reserved[i] = b
        else:
            for i, b in enumerate(reserved):
                if b > 127 and b < 256:
                    b = b - 256
                self._ptr[0].reserved[i] = b

cdef class CUipcMemHandle_st:
    """
    CUDA IPC mem handle

    Attributes
    ----------
    reserved : bytes


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUipcMemHandle_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def reserved(self):
        return PyBytes_FromStringAndSize(self._ptr[0].reserved, 64)
    @reserved.setter
    def reserved(self, reserved):
        if len(reserved) != 64:
            raise ValueError("reserved length must be 64, is " + str(len(reserved)))
        if CHAR_MIN == 0:
            for i, b in enumerate(reserved):
                if b < 0 and b > -129:
                    b = b + 256
                self._ptr[0].reserved[i] = b
        else:
            for i, b in enumerate(reserved):
                if b > 127 and b < 256:
                    b = b - 256
                self._ptr[0].reserved[i] = b

cdef class CUstreamMemOpWaitValueParams_st:
    """
    Attributes
    ----------
    operation : CUstreamBatchMemOpType

    address : CUdeviceptr

    value : cuuint32_t

    value64 : cuuint64_t

    flags : unsigned int

    alias : CUdeviceptr
        For driver internal use. Initial value is unimportant.

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <cydriver.CUstreamBatchMemOpParams_union *>_ptr

    def __init__(self, void_ptr _ptr):
        self._address = CUdeviceptr(_ptr=<void_ptr>&self._ptr[0].waitValue.address)
        self._value = cuuint32_t(_ptr=<void_ptr>&self._ptr[0].waitValue.value)
        self._value64 = cuuint64_t(_ptr=<void_ptr>&self._ptr[0].waitValue.value64)
        self._alias = CUdeviceptr(_ptr=<void_ptr>&self._ptr[0].waitValue.alias)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].waitValue
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['operation : ' + str(self.operation)]
            except ValueError:
                str_list += ['operation : <ValueError>']
            try:
                str_list += ['address : ' + str(self.address)]
            except ValueError:
                str_list += ['address : <ValueError>']
            try:
                str_list += ['value : ' + str(self.value)]
            except ValueError:
                str_list += ['value : <ValueError>']
            try:
                str_list += ['value64 : ' + str(self.value64)]
            except ValueError:
                str_list += ['value64 : <ValueError>']
            try:
                str_list += ['flags : ' + str(self.flags)]
            except ValueError:
                str_list += ['flags : <ValueError>']
            try:
                str_list += ['alias : ' + str(self.alias)]
            except ValueError:
                str_list += ['alias : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def operation(self):
        return CUstreamBatchMemOpType(self._ptr[0].waitValue.operation)
    @operation.setter
    def operation(self, operation not None : CUstreamBatchMemOpType):
        self._ptr[0].waitValue.operation = operation.value
    @property
    def address(self):
        return self._address
    @address.setter
    def address(self, address):
        cdef cydriver.CUdeviceptr cyaddress
        if address is None:
            cyaddress = <cydriver.CUdeviceptr><void_ptr>0
        elif isinstance(address, (CUdeviceptr)):
            paddress = int(address)
            cyaddress = <cydriver.CUdeviceptr><void_ptr>paddress
        else:
            paddress = int(CUdeviceptr(address))
            cyaddress = <cydriver.CUdeviceptr><void_ptr>paddress
        self._address._ptr[0] = cyaddress

    @property
    def value(self):
        return self._value
    @value.setter
    def value(self, value):
        cdef cydriver.cuuint32_t cyvalue
        if value is None:
            cyvalue = <cydriver.cuuint32_t><void_ptr>0
        elif isinstance(value, (cuuint32_t)):
            pvalue = int(value)
            cyvalue = <cydriver.cuuint32_t><void_ptr>pvalue
        else:
            pvalue = int(cuuint32_t(value))
            cyvalue = <cydriver.cuuint32_t><void_ptr>pvalue
        self._value._ptr[0] = cyvalue

    @property
    def value64(self):
        return self._value64
    @value64.setter
    def value64(self, value64):
        cdef cydriver.cuuint64_t cyvalue64
        if value64 is None:
            cyvalue64 = <cydriver.cuuint64_t><void_ptr>0
        elif isinstance(value64, (cuuint64_t)):
            pvalue64 = int(value64)
            cyvalue64 = <cydriver.cuuint64_t><void_ptr>pvalue64
        else:
            pvalue64 = int(cuuint64_t(value64))
            cyvalue64 = <cydriver.cuuint64_t><void_ptr>pvalue64
        self._value64._ptr[0] = cyvalue64

    @property
    def flags(self):
        return self._ptr[0].waitValue.flags
    @flags.setter
    def flags(self, unsigned int flags):
        self._ptr[0].waitValue.flags = flags
    @property
    def alias(self):
        return self._alias
    @alias.setter
    def alias(self, alias):
        cdef cydriver.CUdeviceptr cyalias
        if alias is None:
            cyalias = <cydriver.CUdeviceptr><void_ptr>0
        elif isinstance(alias, (CUdeviceptr)):
            palias = int(alias)
            cyalias = <cydriver.CUdeviceptr><void_ptr>palias
        else:
            palias = int(CUdeviceptr(alias))
            cyalias = <cydriver.CUdeviceptr><void_ptr>palias
        self._alias._ptr[0] = cyalias

cdef class CUstreamMemOpWriteValueParams_st:
    """
    Attributes
    ----------
    operation : CUstreamBatchMemOpType

    address : CUdeviceptr

    value : cuuint32_t

    value64 : cuuint64_t

    flags : unsigned int

    alias : CUdeviceptr
        For driver internal use. Initial value is unimportant.

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <cydriver.CUstreamBatchMemOpParams_union *>_ptr

    def __init__(self, void_ptr _ptr):
        self._address = CUdeviceptr(_ptr=<void_ptr>&self._ptr[0].writeValue.address)
        self._value = cuuint32_t(_ptr=<void_ptr>&self._ptr[0].writeValue.value)
        self._value64 = cuuint64_t(_ptr=<void_ptr>&self._ptr[0].writeValue.value64)
        self._alias = CUdeviceptr(_ptr=<void_ptr>&self._ptr[0].writeValue.alias)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].writeValue
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['operation : ' + str(self.operation)]
            except ValueError:
                str_list += ['operation : <ValueError>']
            try:
                str_list += ['address : ' + str(self.address)]
            except ValueError:
                str_list += ['address : <ValueError>']
            try:
                str_list += ['value : ' + str(self.value)]
            except ValueError:
                str_list += ['value : <ValueError>']
            try:
                str_list += ['value64 : ' + str(self.value64)]
            except ValueError:
                str_list += ['value64 : <ValueError>']
            try:
                str_list += ['flags : ' + str(self.flags)]
            except ValueError:
                str_list += ['flags : <ValueError>']
            try:
                str_list += ['alias : ' + str(self.alias)]
            except ValueError:
                str_list += ['alias : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def operation(self):
        return CUstreamBatchMemOpType(self._ptr[0].writeValue.operation)
    @operation.setter
    def operation(self, operation not None : CUstreamBatchMemOpType):
        self._ptr[0].writeValue.operation = operation.value
    @property
    def address(self):
        return self._address
    @address.setter
    def address(self, address):
        cdef cydriver.CUdeviceptr cyaddress
        if address is None:
            cyaddress = <cydriver.CUdeviceptr><void_ptr>0
        elif isinstance(address, (CUdeviceptr)):
            paddress = int(address)
            cyaddress = <cydriver.CUdeviceptr><void_ptr>paddress
        else:
            paddress = int(CUdeviceptr(address))
            cyaddress = <cydriver.CUdeviceptr><void_ptr>paddress
        self._address._ptr[0] = cyaddress

    @property
    def value(self):
        return self._value
    @value.setter
    def value(self, value):
        cdef cydriver.cuuint32_t cyvalue
        if value is None:
            cyvalue = <cydriver.cuuint32_t><void_ptr>0
        elif isinstance(value, (cuuint32_t)):
            pvalue = int(value)
            cyvalue = <cydriver.cuuint32_t><void_ptr>pvalue
        else:
            pvalue = int(cuuint32_t(value))
            cyvalue = <cydriver.cuuint32_t><void_ptr>pvalue
        self._value._ptr[0] = cyvalue

    @property
    def value64(self):
        return self._value64
    @value64.setter
    def value64(self, value64):
        cdef cydriver.cuuint64_t cyvalue64
        if value64 is None:
            cyvalue64 = <cydriver.cuuint64_t><void_ptr>0
        elif isinstance(value64, (cuuint64_t)):
            pvalue64 = int(value64)
            cyvalue64 = <cydriver.cuuint64_t><void_ptr>pvalue64
        else:
            pvalue64 = int(cuuint64_t(value64))
            cyvalue64 = <cydriver.cuuint64_t><void_ptr>pvalue64
        self._value64._ptr[0] = cyvalue64

    @property
    def flags(self):
        return self._ptr[0].writeValue.flags
    @flags.setter
    def flags(self, unsigned int flags):
        self._ptr[0].writeValue.flags = flags
    @property
    def alias(self):
        return self._alias
    @alias.setter
    def alias(self, alias):
        cdef cydriver.CUdeviceptr cyalias
        if alias is None:
            cyalias = <cydriver.CUdeviceptr><void_ptr>0
        elif isinstance(alias, (CUdeviceptr)):
            palias = int(alias)
            cyalias = <cydriver.CUdeviceptr><void_ptr>palias
        else:
            palias = int(CUdeviceptr(alias))
            cyalias = <cydriver.CUdeviceptr><void_ptr>palias
        self._alias._ptr[0] = cyalias

cdef class CUstreamMemOpFlushRemoteWritesParams_st:
    """
    Attributes
    ----------
    operation : CUstreamBatchMemOpType

    flags : unsigned int


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <cydriver.CUstreamBatchMemOpParams_union *>_ptr

    def __init__(self, void_ptr _ptr):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].flushRemoteWrites
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['operation : ' + str(self.operation)]
            except ValueError:
                str_list += ['operation : <ValueError>']
            try:
                str_list += ['flags : ' + str(self.flags)]
            except ValueError:
                str_list += ['flags : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def operation(self):
        return CUstreamBatchMemOpType(self._ptr[0].flushRemoteWrites.operation)
    @operation.setter
    def operation(self, operation not None : CUstreamBatchMemOpType):
        self._ptr[0].flushRemoteWrites.operation = operation.value
    @property
    def flags(self):
        return self._ptr[0].flushRemoteWrites.flags
    @flags.setter
    def flags(self, unsigned int flags):
        self._ptr[0].flushRemoteWrites.flags = flags

cdef class CUstreamMemOpMemoryBarrierParams_st:
    """
    Attributes
    ----------
    operation : CUstreamBatchMemOpType
        < Only supported in the _v2 API
    flags : unsigned int


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <cydriver.CUstreamBatchMemOpParams_union *>_ptr

    def __init__(self, void_ptr _ptr):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].memoryBarrier
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['operation : ' + str(self.operation)]
            except ValueError:
                str_list += ['operation : <ValueError>']
            try:
                str_list += ['flags : ' + str(self.flags)]
            except ValueError:
                str_list += ['flags : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def operation(self):
        return CUstreamBatchMemOpType(self._ptr[0].memoryBarrier.operation)
    @operation.setter
    def operation(self, operation not None : CUstreamBatchMemOpType):
        self._ptr[0].memoryBarrier.operation = operation.value
    @property
    def flags(self):
        return self._ptr[0].memoryBarrier.flags
    @flags.setter
    def flags(self, unsigned int flags):
        self._ptr[0].memoryBarrier.flags = flags

cdef class CUstreamBatchMemOpParams_union:
    """
    Per-operation parameters for cuStreamBatchMemOp

    Attributes
    ----------
    operation : CUstreamBatchMemOpType

    waitValue : CUstreamMemOpWaitValueParams_st

    writeValue : CUstreamMemOpWriteValueParams_st

    flushRemoteWrites : CUstreamMemOpFlushRemoteWritesParams_st

    memoryBarrier : CUstreamMemOpMemoryBarrierParams_st

    pad : List[cuuint64_t]


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUstreamBatchMemOpParams_union *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._waitValue = CUstreamMemOpWaitValueParams_st(_ptr=<void_ptr>self._ptr)
        self._writeValue = CUstreamMemOpWriteValueParams_st(_ptr=<void_ptr>self._ptr)
        self._flushRemoteWrites = CUstreamMemOpFlushRemoteWritesParams_st(_ptr=<void_ptr>self._ptr)
        self._memoryBarrier = CUstreamMemOpMemoryBarrierParams_st(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['operation : ' + str(self.operation)]
            except ValueError:
                str_list += ['operation : <ValueError>']
            try:
                str_list += ['waitValue :\n' + '\n'.join(['    ' + line for line in str(self.waitValue).splitlines()])]
            except ValueError:
                str_list += ['waitValue : <ValueError>']
            try:
                str_list += ['writeValue :\n' + '\n'.join(['    ' + line for line in str(self.writeValue).splitlines()])]
            except ValueError:
                str_list += ['writeValue : <ValueError>']
            try:
                str_list += ['flushRemoteWrites :\n' + '\n'.join(['    ' + line for line in str(self.flushRemoteWrites).splitlines()])]
            except ValueError:
                str_list += ['flushRemoteWrites : <ValueError>']
            try:
                str_list += ['memoryBarrier :\n' + '\n'.join(['    ' + line for line in str(self.memoryBarrier).splitlines()])]
            except ValueError:
                str_list += ['memoryBarrier : <ValueError>']
            try:
                str_list += ['pad : ' + str(self.pad)]
            except ValueError:
                str_list += ['pad : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def operation(self):
        return CUstreamBatchMemOpType(self._ptr[0].operation)
    @operation.setter
    def operation(self, operation not None : CUstreamBatchMemOpType):
        self._ptr[0].operation = operation.value
    @property
    def waitValue(self):
        return self._waitValue
    @waitValue.setter
    def waitValue(self, waitValue not None : CUstreamMemOpWaitValueParams_st):
        string.memcpy(&self._ptr[0].waitValue, <cydriver.CUstreamMemOpWaitValueParams_st*><void_ptr>waitValue.getPtr(), sizeof(self._ptr[0].waitValue))
    @property
    def writeValue(self):
        return self._writeValue
    @writeValue.setter
    def writeValue(self, writeValue not None : CUstreamMemOpWriteValueParams_st):
        string.memcpy(&self._ptr[0].writeValue, <cydriver.CUstreamMemOpWriteValueParams_st*><void_ptr>writeValue.getPtr(), sizeof(self._ptr[0].writeValue))
    @property
    def flushRemoteWrites(self):
        return self._flushRemoteWrites
    @flushRemoteWrites.setter
    def flushRemoteWrites(self, flushRemoteWrites not None : CUstreamMemOpFlushRemoteWritesParams_st):
        string.memcpy(&self._ptr[0].flushRemoteWrites, <cydriver.CUstreamMemOpFlushRemoteWritesParams_st*><void_ptr>flushRemoteWrites.getPtr(), sizeof(self._ptr[0].flushRemoteWrites))
    @property
    def memoryBarrier(self):
        return self._memoryBarrier
    @memoryBarrier.setter
    def memoryBarrier(self, memoryBarrier not None : CUstreamMemOpMemoryBarrierParams_st):
        string.memcpy(&self._ptr[0].memoryBarrier, <cydriver.CUstreamMemOpMemoryBarrierParams_st*><void_ptr>memoryBarrier.getPtr(), sizeof(self._ptr[0].memoryBarrier))
    @property
    def pad(self):
        return [cuuint64_t(init_value=_pad) for _pad in self._ptr[0].pad]
    @pad.setter
    def pad(self, pad):
        self._ptr[0].pad = pad

cdef class CUDA_BATCH_MEM_OP_NODE_PARAMS_st:
    """
    Attributes
    ----------
    ctx : CUcontext

    count : unsigned int

    paramArray : CUstreamBatchMemOpParams

    flags : unsigned int


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUDA_BATCH_MEM_OP_NODE_PARAMS_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._ctx = CUcontext(_ptr=<void_ptr>&self._ptr[0].ctx)
    def __dealloc__(self):
        if self._paramArray is not NULL:
            free(self._paramArray)
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['ctx : ' + str(self.ctx)]
            except ValueError:
                str_list += ['ctx : <ValueError>']
            try:
                str_list += ['count : ' + str(self.count)]
            except ValueError:
                str_list += ['count : <ValueError>']
            try:
                str_list += ['paramArray : ' + str(self.paramArray)]
            except ValueError:
                str_list += ['paramArray : <ValueError>']
            try:
                str_list += ['flags : ' + str(self.flags)]
            except ValueError:
                str_list += ['flags : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def ctx(self):
        return self._ctx
    @ctx.setter
    def ctx(self, ctx):
        cdef cydriver.CUcontext cyctx
        if ctx is None:
            cyctx = <cydriver.CUcontext><void_ptr>0
        elif isinstance(ctx, (CUcontext,)):
            pctx = int(ctx)
            cyctx = <cydriver.CUcontext><void_ptr>pctx
        else:
            pctx = int(CUcontext(ctx))
            cyctx = <cydriver.CUcontext><void_ptr>pctx
        self._ctx._ptr[0] = cyctx
    @property
    def count(self):
        return self._ptr[0].count
    @count.setter
    def count(self, unsigned int count):
        self._ptr[0].count = count
    @property
    def paramArray(self):
        arrs = [<void_ptr>self._ptr[0].paramArray + x*sizeof(cydriver.CUstreamBatchMemOpParams) for x in range(self._paramArray_length)]
        return [CUstreamBatchMemOpParams(_ptr=arr) for arr in arrs]
    @paramArray.setter
    def paramArray(self, val):
        if len(val) == 0:
            free(self._paramArray)
            self._paramArray_length = 0
            self._ptr[0].paramArray = NULL
        else:
            if self._paramArray_length != <size_t>len(val):
                free(self._paramArray)
                self._paramArray = <cydriver.CUstreamBatchMemOpParams*> calloc(len(val), sizeof(cydriver.CUstreamBatchMemOpParams))
                if self._paramArray is NULL:
                    raise MemoryError('Failed to allocate length x size memory: ' + str(len(val)) + 'x' + str(sizeof(cydriver.CUstreamBatchMemOpParams)))
                self._paramArray_length = <size_t>len(val)
                self._ptr[0].paramArray = self._paramArray
            for idx in range(len(val)):
                string.memcpy(&self._paramArray[idx], (<CUstreamBatchMemOpParams>val[idx])._ptr, sizeof(cydriver.CUstreamBatchMemOpParams))

    @property
    def flags(self):
        return self._ptr[0].flags
    @flags.setter
    def flags(self, unsigned int flags):
        self._ptr[0].flags = flags

cdef class CUdevprop_st:
    """
    Legacy device properties

    Attributes
    ----------
    maxThreadsPerBlock : int
        Maximum number of threads per block
    maxThreadsDim : List[int]
        Maximum size of each dimension of a block
    maxGridSize : List[int]
        Maximum size of each dimension of a grid
    sharedMemPerBlock : int
        Shared memory available per block in bytes
    totalConstantMemory : int
        Constant memory available on device in bytes
    SIMDWidth : int
        Warp size in threads
    memPitch : int
        Maximum pitch in bytes allowed by memory copies
    regsPerBlock : int
        32-bit registers available per block
    clockRate : int
        Clock frequency in kilohertz
    textureAlign : int
        Alignment requirement for textures

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUdevprop_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['maxThreadsPerBlock : ' + str(self.maxThreadsPerBlock)]
            except ValueError:
                str_list += ['maxThreadsPerBlock : <ValueError>']
            try:
                str_list += ['maxThreadsDim : ' + str(self.maxThreadsDim)]
            except ValueError:
                str_list += ['maxThreadsDim : <ValueError>']
            try:
                str_list += ['maxGridSize : ' + str(self.maxGridSize)]
            except ValueError:
                str_list += ['maxGridSize : <ValueError>']
            try:
                str_list += ['sharedMemPerBlock : ' + str(self.sharedMemPerBlock)]
            except ValueError:
                str_list += ['sharedMemPerBlock : <ValueError>']
            try:
                str_list += ['totalConstantMemory : ' + str(self.totalConstantMemory)]
            except ValueError:
                str_list += ['totalConstantMemory : <ValueError>']
            try:
                str_list += ['SIMDWidth : ' + str(self.SIMDWidth)]
            except ValueError:
                str_list += ['SIMDWidth : <ValueError>']
            try:
                str_list += ['memPitch : ' + str(self.memPitch)]
            except ValueError:
                str_list += ['memPitch : <ValueError>']
            try:
                str_list += ['regsPerBlock : ' + str(self.regsPerBlock)]
            except ValueError:
                str_list += ['regsPerBlock : <ValueError>']
            try:
                str_list += ['clockRate : ' + str(self.clockRate)]
            except ValueError:
                str_list += ['clockRate : <ValueError>']
            try:
                str_list += ['textureAlign : ' + str(self.textureAlign)]
            except ValueError:
                str_list += ['textureAlign : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def maxThreadsPerBlock(self):
        return self._ptr[0].maxThreadsPerBlock
    @maxThreadsPerBlock.setter
    def maxThreadsPerBlock(self, int maxThreadsPerBlock):
        self._ptr[0].maxThreadsPerBlock = maxThreadsPerBlock
    @property
    def maxThreadsDim(self):
        return self._ptr[0].maxThreadsDim
    @maxThreadsDim.setter
    def maxThreadsDim(self, maxThreadsDim):
        self._ptr[0].maxThreadsDim = maxThreadsDim
    @property
    def maxGridSize(self):
        return self._ptr[0].maxGridSize
    @maxGridSize.setter
    def maxGridSize(self, maxGridSize):
        self._ptr[0].maxGridSize = maxGridSize
    @property
    def sharedMemPerBlock(self):
        return self._ptr[0].sharedMemPerBlock
    @sharedMemPerBlock.setter
    def sharedMemPerBlock(self, int sharedMemPerBlock):
        self._ptr[0].sharedMemPerBlock = sharedMemPerBlock
    @property
    def totalConstantMemory(self):
        return self._ptr[0].totalConstantMemory
    @totalConstantMemory.setter
    def totalConstantMemory(self, int totalConstantMemory):
        self._ptr[0].totalConstantMemory = totalConstantMemory
    @property
    def SIMDWidth(self):
        return self._ptr[0].SIMDWidth
    @SIMDWidth.setter
    def SIMDWidth(self, int SIMDWidth):
        self._ptr[0].SIMDWidth = SIMDWidth
    @property
    def memPitch(self):
        return self._ptr[0].memPitch
    @memPitch.setter
    def memPitch(self, int memPitch):
        self._ptr[0].memPitch = memPitch
    @property
    def regsPerBlock(self):
        return self._ptr[0].regsPerBlock
    @regsPerBlock.setter
    def regsPerBlock(self, int regsPerBlock):
        self._ptr[0].regsPerBlock = regsPerBlock
    @property
    def clockRate(self):
        return self._ptr[0].clockRate
    @clockRate.setter
    def clockRate(self, int clockRate):
        self._ptr[0].clockRate = clockRate
    @property
    def textureAlign(self):
        return self._ptr[0].textureAlign
    @textureAlign.setter
    def textureAlign(self, int textureAlign):
        self._ptr[0].textureAlign = textureAlign

cdef class CUaccessPolicyWindow_st:
    """
    Specifies an access policy for a window, a contiguous extent of
    memory beginning at base_ptr and ending at base_ptr + num_bytes.
    num_bytes is limited by
    CU_DEVICE_ATTRIBUTE_MAX_ACCESS_POLICY_WINDOW_SIZE. Partition into
    many segments and assign segments such that: sum of "hit segments"
    / window == approx. ratio. sum of "miss segments" / window ==
    approx 1-ratio. Segments and ratio specifications are fitted to the
    capabilities of the architecture. Accesses in a hit segment apply
    the hitProp access policy. Accesses in a miss segment apply the
    missProp access policy.

    Attributes
    ----------
    base_ptr : Any
        Starting address of the access policy window. CUDA driver may align
        it.
    num_bytes : size_t
        Size in bytes of the window policy. CUDA driver may restrict the
        maximum size and alignment.
    hitRatio : float
        hitRatio specifies percentage of lines assigned hitProp, rest are
        assigned missProp.
    hitProp : CUaccessProperty
        CUaccessProperty set for hit.
    missProp : CUaccessProperty
        CUaccessProperty set for miss. Must be either NORMAL or STREAMING

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUaccessPolicyWindow_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['base_ptr : ' + hex(self.base_ptr)]
            except ValueError:
                str_list += ['base_ptr : <ValueError>']
            try:
                str_list += ['num_bytes : ' + str(self.num_bytes)]
            except ValueError:
                str_list += ['num_bytes : <ValueError>']
            try:
                str_list += ['hitRatio : ' + str(self.hitRatio)]
            except ValueError:
                str_list += ['hitRatio : <ValueError>']
            try:
                str_list += ['hitProp : ' + str(self.hitProp)]
            except ValueError:
                str_list += ['hitProp : <ValueError>']
            try:
                str_list += ['missProp : ' + str(self.missProp)]
            except ValueError:
                str_list += ['missProp : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def base_ptr(self):
        return <void_ptr>self._ptr[0].base_ptr
    @base_ptr.setter
    def base_ptr(self, base_ptr):
        _cybase_ptr = utils.HelperInputVoidPtr(base_ptr)
        self._ptr[0].base_ptr = <void*><void_ptr>_cybase_ptr.cptr
    @property
    def num_bytes(self):
        return self._ptr[0].num_bytes
    @num_bytes.setter
    def num_bytes(self, size_t num_bytes):
        self._ptr[0].num_bytes = num_bytes
    @property
    def hitRatio(self):
        return self._ptr[0].hitRatio
    @hitRatio.setter
    def hitRatio(self, float hitRatio):
        self._ptr[0].hitRatio = hitRatio
    @property
    def hitProp(self):
        return CUaccessProperty(self._ptr[0].hitProp)
    @hitProp.setter
    def hitProp(self, hitProp not None : CUaccessProperty):
        self._ptr[0].hitProp = hitProp.value
    @property
    def missProp(self):
        return CUaccessProperty(self._ptr[0].missProp)
    @missProp.setter
    def missProp(self, missProp not None : CUaccessProperty):
        self._ptr[0].missProp = missProp.value

cdef class CUDA_KERNEL_NODE_PARAMS_st:
    """
    GPU kernel node parameters

    Attributes
    ----------
    func : CUfunction
        Kernel to launch
    gridDimX : unsigned int
        Width of grid in blocks
    gridDimY : unsigned int
        Height of grid in blocks
    gridDimZ : unsigned int
        Depth of grid in blocks
    blockDimX : unsigned int
        X dimension of each thread block
    blockDimY : unsigned int
        Y dimension of each thread block
    blockDimZ : unsigned int
        Z dimension of each thread block
    sharedMemBytes : unsigned int
        Dynamic shared-memory size per thread block in bytes
    kernelParams : Any
        Array of pointers to kernel parameters
    extra : Any
        Extra options

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUDA_KERNEL_NODE_PARAMS_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._func = CUfunction(_ptr=<void_ptr>&self._ptr[0].func)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['func : ' + str(self.func)]
            except ValueError:
                str_list += ['func : <ValueError>']
            try:
                str_list += ['gridDimX : ' + str(self.gridDimX)]
            except ValueError:
                str_list += ['gridDimX : <ValueError>']
            try:
                str_list += ['gridDimY : ' + str(self.gridDimY)]
            except ValueError:
                str_list += ['gridDimY : <ValueError>']
            try:
                str_list += ['gridDimZ : ' + str(self.gridDimZ)]
            except ValueError:
                str_list += ['gridDimZ : <ValueError>']
            try:
                str_list += ['blockDimX : ' + str(self.blockDimX)]
            except ValueError:
                str_list += ['blockDimX : <ValueError>']
            try:
                str_list += ['blockDimY : ' + str(self.blockDimY)]
            except ValueError:
                str_list += ['blockDimY : <ValueError>']
            try:
                str_list += ['blockDimZ : ' + str(self.blockDimZ)]
            except ValueError:
                str_list += ['blockDimZ : <ValueError>']
            try:
                str_list += ['sharedMemBytes : ' + str(self.sharedMemBytes)]
            except ValueError:
                str_list += ['sharedMemBytes : <ValueError>']
            try:
                str_list += ['kernelParams : ' + str(self.kernelParams)]
            except ValueError:
                str_list += ['kernelParams : <ValueError>']
            try:
                str_list += ['extra : ' + str(self.extra)]
            except ValueError:
                str_list += ['extra : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def func(self):
        return self._func
    @func.setter
    def func(self, func):
        cdef cydriver.CUfunction cyfunc
        if func is None:
            cyfunc = <cydriver.CUfunction><void_ptr>0
        elif isinstance(func, (CUfunction,)):
            pfunc = int(func)
            cyfunc = <cydriver.CUfunction><void_ptr>pfunc
        else:
            pfunc = int(CUfunction(func))
            cyfunc = <cydriver.CUfunction><void_ptr>pfunc
        self._func._ptr[0] = cyfunc
    @property
    def gridDimX(self):
        return self._ptr[0].gridDimX
    @gridDimX.setter
    def gridDimX(self, unsigned int gridDimX):
        self._ptr[0].gridDimX = gridDimX
    @property
    def gridDimY(self):
        return self._ptr[0].gridDimY
    @gridDimY.setter
    def gridDimY(self, unsigned int gridDimY):
        self._ptr[0].gridDimY = gridDimY
    @property
    def gridDimZ(self):
        return self._ptr[0].gridDimZ
    @gridDimZ.setter
    def gridDimZ(self, unsigned int gridDimZ):
        self._ptr[0].gridDimZ = gridDimZ
    @property
    def blockDimX(self):
        return self._ptr[0].blockDimX
    @blockDimX.setter
    def blockDimX(self, unsigned int blockDimX):
        self._ptr[0].blockDimX = blockDimX
    @property
    def blockDimY(self):
        return self._ptr[0].blockDimY
    @blockDimY.setter
    def blockDimY(self, unsigned int blockDimY):
        self._ptr[0].blockDimY = blockDimY
    @property
    def blockDimZ(self):
        return self._ptr[0].blockDimZ
    @blockDimZ.setter
    def blockDimZ(self, unsigned int blockDimZ):
        self._ptr[0].blockDimZ = blockDimZ
    @property
    def sharedMemBytes(self):
        return self._ptr[0].sharedMemBytes
    @sharedMemBytes.setter
    def sharedMemBytes(self, unsigned int sharedMemBytes):
        self._ptr[0].sharedMemBytes = sharedMemBytes
    @property
    def kernelParams(self):
        return <void_ptr>self._ptr[0].kernelParams
    @kernelParams.setter
    def kernelParams(self, kernelParams):
        self._cykernelParams = utils.HelperKernelParams(kernelParams)
        self._ptr[0].kernelParams = <void**><void_ptr>self._cykernelParams.ckernelParams
    @property
    def extra(self):
        return <void_ptr>self._ptr[0].extra
    @extra.setter
    def extra(self, void_ptr extra):
        self._ptr[0].extra = <void**>extra

cdef class CUDA_MEMSET_NODE_PARAMS_st:
    """
    Memset node parameters

    Attributes
    ----------
    dst : CUdeviceptr
        Destination device pointer
    pitch : size_t
        Pitch of destination device pointer. Unused if height is 1
    value : unsigned int
        Value to be set
    elementSize : unsigned int
        Size of each element in bytes. Must be 1, 2, or 4.
    width : size_t
        Width of the row in elements
    height : size_t
        Number of rows

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUDA_MEMSET_NODE_PARAMS_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._dst = CUdeviceptr(_ptr=<void_ptr>&self._ptr[0].dst)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['dst : ' + str(self.dst)]
            except ValueError:
                str_list += ['dst : <ValueError>']
            try:
                str_list += ['pitch : ' + str(self.pitch)]
            except ValueError:
                str_list += ['pitch : <ValueError>']
            try:
                str_list += ['value : ' + str(self.value)]
            except ValueError:
                str_list += ['value : <ValueError>']
            try:
                str_list += ['elementSize : ' + str(self.elementSize)]
            except ValueError:
                str_list += ['elementSize : <ValueError>']
            try:
                str_list += ['width : ' + str(self.width)]
            except ValueError:
                str_list += ['width : <ValueError>']
            try:
                str_list += ['height : ' + str(self.height)]
            except ValueError:
                str_list += ['height : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def dst(self):
        return self._dst
    @dst.setter
    def dst(self, dst):
        cdef cydriver.CUdeviceptr cydst
        if dst is None:
            cydst = <cydriver.CUdeviceptr><void_ptr>0
        elif isinstance(dst, (CUdeviceptr)):
            pdst = int(dst)
            cydst = <cydriver.CUdeviceptr><void_ptr>pdst
        else:
            pdst = int(CUdeviceptr(dst))
            cydst = <cydriver.CUdeviceptr><void_ptr>pdst
        self._dst._ptr[0] = cydst

    @property
    def pitch(self):
        return self._ptr[0].pitch
    @pitch.setter
    def pitch(self, size_t pitch):
        self._ptr[0].pitch = pitch
    @property
    def value(self):
        return self._ptr[0].value
    @value.setter
    def value(self, unsigned int value):
        self._ptr[0].value = value
    @property
    def elementSize(self):
        return self._ptr[0].elementSize
    @elementSize.setter
    def elementSize(self, unsigned int elementSize):
        self._ptr[0].elementSize = elementSize
    @property
    def width(self):
        return self._ptr[0].width
    @width.setter
    def width(self, size_t width):
        self._ptr[0].width = width
    @property
    def height(self):
        return self._ptr[0].height
    @height.setter
    def height(self, size_t height):
        self._ptr[0].height = height

cdef class CUDA_HOST_NODE_PARAMS_st:
    """
    Host node parameters

    Attributes
    ----------
    fn : CUhostFn
        The function to call when the node executes
    userData : Any
        Argument to pass to the function

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUDA_HOST_NODE_PARAMS_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._fn = CUhostFn(_ptr=<void_ptr>&self._ptr[0].fn)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['fn : ' + str(self.fn)]
            except ValueError:
                str_list += ['fn : <ValueError>']
            try:
                str_list += ['userData : ' + hex(self.userData)]
            except ValueError:
                str_list += ['userData : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def fn(self):
        return self._fn
    @fn.setter
    def fn(self, fn):
        cdef cydriver.CUhostFn cyfn
        if fn is None:
            cyfn = <cydriver.CUhostFn><void_ptr>0
        elif isinstance(fn, (CUhostFn)):
            pfn = int(fn)
            cyfn = <cydriver.CUhostFn><void_ptr>pfn
        else:
            pfn = int(CUhostFn(fn))
            cyfn = <cydriver.CUhostFn><void_ptr>pfn
        self._fn._ptr[0] = cyfn
    @property
    def userData(self):
        return <void_ptr>self._ptr[0].userData
    @userData.setter
    def userData(self, userData):
        _cyuserData = utils.HelperInputVoidPtr(userData)
        self._ptr[0].userData = <void*><void_ptr>_cyuserData.cptr

cdef class anon_struct0:
    """
    Attributes
    ----------
    x : unsigned int

    y : unsigned int

    z : unsigned int


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <cydriver.CUlaunchAttributeValue_union *>_ptr

    def __init__(self, void_ptr _ptr):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].clusterDim
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['x : ' + str(self.x)]
            except ValueError:
                str_list += ['x : <ValueError>']
            try:
                str_list += ['y : ' + str(self.y)]
            except ValueError:
                str_list += ['y : <ValueError>']
            try:
                str_list += ['z : ' + str(self.z)]
            except ValueError:
                str_list += ['z : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def x(self):
        return self._ptr[0].clusterDim.x
    @x.setter
    def x(self, unsigned int x):
        self._ptr[0].clusterDim.x = x
    @property
    def y(self):
        return self._ptr[0].clusterDim.y
    @y.setter
    def y(self, unsigned int y):
        self._ptr[0].clusterDim.y = y
    @property
    def z(self):
        return self._ptr[0].clusterDim.z
    @z.setter
    def z(self, unsigned int z):
        self._ptr[0].clusterDim.z = z

cdef class anon_struct1:
    """
    Attributes
    ----------
    event : CUevent

    flags : int

    triggerAtBlockStart : int


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <cydriver.CUlaunchAttributeValue_union *>_ptr

    def __init__(self, void_ptr _ptr):
        self._event = CUevent(_ptr=<void_ptr>&self._ptr[0].programmaticEvent.event)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].programmaticEvent
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['event : ' + str(self.event)]
            except ValueError:
                str_list += ['event : <ValueError>']
            try:
                str_list += ['flags : ' + str(self.flags)]
            except ValueError:
                str_list += ['flags : <ValueError>']
            try:
                str_list += ['triggerAtBlockStart : ' + str(self.triggerAtBlockStart)]
            except ValueError:
                str_list += ['triggerAtBlockStart : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def event(self):
        return self._event
    @event.setter
    def event(self, event):
        cdef cydriver.CUevent cyevent
        if event is None:
            cyevent = <cydriver.CUevent><void_ptr>0
        elif isinstance(event, (CUevent,)):
            pevent = int(event)
            cyevent = <cydriver.CUevent><void_ptr>pevent
        else:
            pevent = int(CUevent(event))
            cyevent = <cydriver.CUevent><void_ptr>pevent
        self._event._ptr[0] = cyevent
    @property
    def flags(self):
        return self._ptr[0].programmaticEvent.flags
    @flags.setter
    def flags(self, int flags):
        self._ptr[0].programmaticEvent.flags = flags
    @property
    def triggerAtBlockStart(self):
        return self._ptr[0].programmaticEvent.triggerAtBlockStart
    @triggerAtBlockStart.setter
    def triggerAtBlockStart(self, int triggerAtBlockStart):
        self._ptr[0].programmaticEvent.triggerAtBlockStart = triggerAtBlockStart

cdef class CUlaunchAttributeValue_union:
    """
    Attributes
    ----------
    pad : bytes
        Pad to 64 bytes
    accessPolicyWindow : CUaccessPolicyWindow

    cooperative : int

    syncPolicy : CUsynchronizationPolicy

    clusterDim : anon_struct0

    clusterSchedulingPolicyPreference : CUclusterSchedulingPolicy

    programmaticStreamSerializationAllowed : int

    programmaticEvent : anon_struct1

    priority : int


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUlaunchAttributeValue_union *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._accessPolicyWindow = CUaccessPolicyWindow(_ptr=<void_ptr>&self._ptr[0].accessPolicyWindow)
        self._clusterDim = anon_struct0(_ptr=<void_ptr>self._ptr)
        self._programmaticEvent = anon_struct1(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['pad : ' + str(self.pad)]
            except ValueError:
                str_list += ['pad : <ValueError>']
            try:
                str_list += ['accessPolicyWindow :\n' + '\n'.join(['    ' + line for line in str(self.accessPolicyWindow).splitlines()])]
            except ValueError:
                str_list += ['accessPolicyWindow : <ValueError>']
            try:
                str_list += ['cooperative : ' + str(self.cooperative)]
            except ValueError:
                str_list += ['cooperative : <ValueError>']
            try:
                str_list += ['syncPolicy : ' + str(self.syncPolicy)]
            except ValueError:
                str_list += ['syncPolicy : <ValueError>']
            try:
                str_list += ['clusterDim :\n' + '\n'.join(['    ' + line for line in str(self.clusterDim).splitlines()])]
            except ValueError:
                str_list += ['clusterDim : <ValueError>']
            try:
                str_list += ['clusterSchedulingPolicyPreference : ' + str(self.clusterSchedulingPolicyPreference)]
            except ValueError:
                str_list += ['clusterSchedulingPolicyPreference : <ValueError>']
            try:
                str_list += ['programmaticStreamSerializationAllowed : ' + str(self.programmaticStreamSerializationAllowed)]
            except ValueError:
                str_list += ['programmaticStreamSerializationAllowed : <ValueError>']
            try:
                str_list += ['programmaticEvent :\n' + '\n'.join(['    ' + line for line in str(self.programmaticEvent).splitlines()])]
            except ValueError:
                str_list += ['programmaticEvent : <ValueError>']
            try:
                str_list += ['priority : ' + str(self.priority)]
            except ValueError:
                str_list += ['priority : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def pad(self):
        return PyBytes_FromStringAndSize(self._ptr[0].pad, 64)
    @pad.setter
    def pad(self, pad):
        if len(pad) != 64:
            raise ValueError("pad length must be 64, is " + str(len(pad)))
        if CHAR_MIN == 0:
            for i, b in enumerate(pad):
                if b < 0 and b > -129:
                    b = b + 256
                self._ptr[0].pad[i] = b
        else:
            for i, b in enumerate(pad):
                if b > 127 and b < 256:
                    b = b - 256
                self._ptr[0].pad[i] = b
    @property
    def accessPolicyWindow(self):
        return self._accessPolicyWindow
    @accessPolicyWindow.setter
    def accessPolicyWindow(self, accessPolicyWindow not None : CUaccessPolicyWindow):
        string.memcpy(&self._ptr[0].accessPolicyWindow, <cydriver.CUaccessPolicyWindow*><void_ptr>accessPolicyWindow.getPtr(), sizeof(self._ptr[0].accessPolicyWindow))
    @property
    def cooperative(self):
        return self._ptr[0].cooperative
    @cooperative.setter
    def cooperative(self, int cooperative):
        self._ptr[0].cooperative = cooperative
    @property
    def syncPolicy(self):
        return CUsynchronizationPolicy(self._ptr[0].syncPolicy)
    @syncPolicy.setter
    def syncPolicy(self, syncPolicy not None : CUsynchronizationPolicy):
        self._ptr[0].syncPolicy = syncPolicy.value
    @property
    def clusterDim(self):
        return self._clusterDim
    @clusterDim.setter
    def clusterDim(self, clusterDim not None : anon_struct0):
        string.memcpy(&self._ptr[0].clusterDim, <cydriver.anon_struct0*><void_ptr>clusterDim.getPtr(), sizeof(self._ptr[0].clusterDim))
    @property
    def clusterSchedulingPolicyPreference(self):
        return CUclusterSchedulingPolicy(self._ptr[0].clusterSchedulingPolicyPreference)
    @clusterSchedulingPolicyPreference.setter
    def clusterSchedulingPolicyPreference(self, clusterSchedulingPolicyPreference not None : CUclusterSchedulingPolicy):
        self._ptr[0].clusterSchedulingPolicyPreference = clusterSchedulingPolicyPreference.value
    @property
    def programmaticStreamSerializationAllowed(self):
        return self._ptr[0].programmaticStreamSerializationAllowed
    @programmaticStreamSerializationAllowed.setter
    def programmaticStreamSerializationAllowed(self, int programmaticStreamSerializationAllowed):
        self._ptr[0].programmaticStreamSerializationAllowed = programmaticStreamSerializationAllowed
    @property
    def programmaticEvent(self):
        return self._programmaticEvent
    @programmaticEvent.setter
    def programmaticEvent(self, programmaticEvent not None : anon_struct1):
        string.memcpy(&self._ptr[0].programmaticEvent, <cydriver.anon_struct1*><void_ptr>programmaticEvent.getPtr(), sizeof(self._ptr[0].programmaticEvent))
    @property
    def priority(self):
        return self._ptr[0].priority
    @priority.setter
    def priority(self, int priority):
        self._ptr[0].priority = priority

cdef class CUlaunchAttribute_st:
    """
    Attributes
    ----------
    id : CUlaunchAttributeID

    value : CUlaunchAttributeValue


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUlaunchAttribute_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._value = CUlaunchAttributeValue(_ptr=<void_ptr>&self._ptr[0].value)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['id : ' + str(self.id)]
            except ValueError:
                str_list += ['id : <ValueError>']
            try:
                str_list += ['value :\n' + '\n'.join(['    ' + line for line in str(self.value).splitlines()])]
            except ValueError:
                str_list += ['value : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def id(self):
        return CUlaunchAttributeID(self._ptr[0].id)
    @id.setter
    def id(self, id not None : CUlaunchAttributeID):
        self._ptr[0].id = id.value
    @property
    def value(self):
        return self._value
    @value.setter
    def value(self, value not None : CUlaunchAttributeValue):
        string.memcpy(&self._ptr[0].value, <cydriver.CUlaunchAttributeValue*><void_ptr>value.getPtr(), sizeof(self._ptr[0].value))

cdef class CUlaunchConfig_st:
    """
    Attributes
    ----------
    gridDimX : unsigned int
        Width of grid in blocks
    gridDimY : unsigned int
        Height of grid in blocks
    gridDimZ : unsigned int
        Depth of grid in blocks
    blockDimX : unsigned int
        X dimension of each thread block
    blockDimY : unsigned int
        Y dimension of each thread block
    blockDimZ : unsigned int
        Z dimension of each thread block
    sharedMemBytes : unsigned int
        Dynamic shared-memory size per thread block in bytes
    hStream : CUstream
        Stream identifier
    attrs : CUlaunchAttribute
        nullable if numAttrs == 0
    numAttrs : unsigned int
        number of attributes populated in attrs

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUlaunchConfig_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._hStream = CUstream(_ptr=<void_ptr>&self._ptr[0].hStream)
    def __dealloc__(self):
        if self._attrs is not NULL:
            free(self._attrs)
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['gridDimX : ' + str(self.gridDimX)]
            except ValueError:
                str_list += ['gridDimX : <ValueError>']
            try:
                str_list += ['gridDimY : ' + str(self.gridDimY)]
            except ValueError:
                str_list += ['gridDimY : <ValueError>']
            try:
                str_list += ['gridDimZ : ' + str(self.gridDimZ)]
            except ValueError:
                str_list += ['gridDimZ : <ValueError>']
            try:
                str_list += ['blockDimX : ' + str(self.blockDimX)]
            except ValueError:
                str_list += ['blockDimX : <ValueError>']
            try:
                str_list += ['blockDimY : ' + str(self.blockDimY)]
            except ValueError:
                str_list += ['blockDimY : <ValueError>']
            try:
                str_list += ['blockDimZ : ' + str(self.blockDimZ)]
            except ValueError:
                str_list += ['blockDimZ : <ValueError>']
            try:
                str_list += ['sharedMemBytes : ' + str(self.sharedMemBytes)]
            except ValueError:
                str_list += ['sharedMemBytes : <ValueError>']
            try:
                str_list += ['hStream : ' + str(self.hStream)]
            except ValueError:
                str_list += ['hStream : <ValueError>']
            try:
                str_list += ['attrs : ' + str(self.attrs)]
            except ValueError:
                str_list += ['attrs : <ValueError>']
            try:
                str_list += ['numAttrs : ' + str(self.numAttrs)]
            except ValueError:
                str_list += ['numAttrs : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def gridDimX(self):
        return self._ptr[0].gridDimX
    @gridDimX.setter
    def gridDimX(self, unsigned int gridDimX):
        self._ptr[0].gridDimX = gridDimX
    @property
    def gridDimY(self):
        return self._ptr[0].gridDimY
    @gridDimY.setter
    def gridDimY(self, unsigned int gridDimY):
        self._ptr[0].gridDimY = gridDimY
    @property
    def gridDimZ(self):
        return self._ptr[0].gridDimZ
    @gridDimZ.setter
    def gridDimZ(self, unsigned int gridDimZ):
        self._ptr[0].gridDimZ = gridDimZ
    @property
    def blockDimX(self):
        return self._ptr[0].blockDimX
    @blockDimX.setter
    def blockDimX(self, unsigned int blockDimX):
        self._ptr[0].blockDimX = blockDimX
    @property
    def blockDimY(self):
        return self._ptr[0].blockDimY
    @blockDimY.setter
    def blockDimY(self, unsigned int blockDimY):
        self._ptr[0].blockDimY = blockDimY
    @property
    def blockDimZ(self):
        return self._ptr[0].blockDimZ
    @blockDimZ.setter
    def blockDimZ(self, unsigned int blockDimZ):
        self._ptr[0].blockDimZ = blockDimZ
    @property
    def sharedMemBytes(self):
        return self._ptr[0].sharedMemBytes
    @sharedMemBytes.setter
    def sharedMemBytes(self, unsigned int sharedMemBytes):
        self._ptr[0].sharedMemBytes = sharedMemBytes
    @property
    def hStream(self):
        return self._hStream
    @hStream.setter
    def hStream(self, hStream):
        cdef cydriver.CUstream cyhStream
        if hStream is None:
            cyhStream = <cydriver.CUstream><void_ptr>0
        elif isinstance(hStream, (CUstream,)):
            phStream = int(hStream)
            cyhStream = <cydriver.CUstream><void_ptr>phStream
        else:
            phStream = int(CUstream(hStream))
            cyhStream = <cydriver.CUstream><void_ptr>phStream
        self._hStream._ptr[0] = cyhStream
    @property
    def attrs(self):
        arrs = [<void_ptr>self._ptr[0].attrs + x*sizeof(cydriver.CUlaunchAttribute) for x in range(self._attrs_length)]
        return [CUlaunchAttribute(_ptr=arr) for arr in arrs]
    @attrs.setter
    def attrs(self, val):
        if len(val) == 0:
            free(self._attrs)
            self._attrs_length = 0
            self._ptr[0].attrs = NULL
        else:
            if self._attrs_length != <size_t>len(val):
                free(self._attrs)
                self._attrs = <cydriver.CUlaunchAttribute*> calloc(len(val), sizeof(cydriver.CUlaunchAttribute))
                if self._attrs is NULL:
                    raise MemoryError('Failed to allocate length x size memory: ' + str(len(val)) + 'x' + str(sizeof(cydriver.CUlaunchAttribute)))
                self._attrs_length = <size_t>len(val)
                self._ptr[0].attrs = self._attrs
            for idx in range(len(val)):
                string.memcpy(&self._attrs[idx], (<CUlaunchAttribute>val[idx])._ptr, sizeof(cydriver.CUlaunchAttribute))

    @property
    def numAttrs(self):
        return self._ptr[0].numAttrs
    @numAttrs.setter
    def numAttrs(self, unsigned int numAttrs):
        self._ptr[0].numAttrs = numAttrs

cdef class CUexecAffinitySmCount_st:
    """
    Value for CU_EXEC_AFFINITY_TYPE_SM_COUNT

    Attributes
    ----------
    val : unsigned int
        The number of SMs the context is limited to use.

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUexecAffinitySmCount_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['val : ' + str(self.val)]
            except ValueError:
                str_list += ['val : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def val(self):
        return self._ptr[0].val
    @val.setter
    def val(self, unsigned int val):
        self._ptr[0].val = val

cdef class anon_union2:
    """
    Attributes
    ----------
    smCount : CUexecAffinitySmCount


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <cydriver.CUexecAffinityParam_st *>_ptr

    def __init__(self, void_ptr _ptr):
        self._smCount = CUexecAffinitySmCount(_ptr=<void_ptr>&self._ptr[0].param.smCount)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].param
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['smCount :\n' + '\n'.join(['    ' + line for line in str(self.smCount).splitlines()])]
            except ValueError:
                str_list += ['smCount : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def smCount(self):
        return self._smCount
    @smCount.setter
    def smCount(self, smCount not None : CUexecAffinitySmCount):
        string.memcpy(&self._ptr[0].param.smCount, <cydriver.CUexecAffinitySmCount*><void_ptr>smCount.getPtr(), sizeof(self._ptr[0].param.smCount))

cdef class CUexecAffinityParam_st:
    """
    Execution Affinity Parameters

    Attributes
    ----------
    type : CUexecAffinityType

    param : anon_union2


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._val_ptr = <cydriver.CUexecAffinityParam_st *>calloc(1, sizeof(cydriver.CUexecAffinityParam_st))
            self._ptr = self._val_ptr
        else:
            self._ptr = <cydriver.CUexecAffinityParam_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._param = anon_union2(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        if self._val_ptr is not NULL:
            free(self._val_ptr)
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['type : ' + str(self.type)]
            except ValueError:
                str_list += ['type : <ValueError>']
            try:
                str_list += ['param :\n' + '\n'.join(['    ' + line for line in str(self.param).splitlines()])]
            except ValueError:
                str_list += ['param : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def type(self):
        return CUexecAffinityType(self._ptr[0].type)
    @type.setter
    def type(self, type not None : CUexecAffinityType):
        self._ptr[0].type = type.value
    @property
    def param(self):
        return self._param
    @param.setter
    def param(self, param not None : anon_union2):
        string.memcpy(&self._ptr[0].param, <cydriver.anon_union2*><void_ptr>param.getPtr(), sizeof(self._ptr[0].param))

cdef class CUDA_MEMCPY2D_st:
    """
    2D memory copy parameters

    Attributes
    ----------
    srcXInBytes : size_t
        Source X in bytes
    srcY : size_t
        Source Y
    srcMemoryType : CUmemorytype
        Source memory type (host, device, array)
    srcHost : Any
        Source host pointer
    srcDevice : CUdeviceptr
        Source device pointer
    srcArray : CUarray
        Source array reference
    srcPitch : size_t
        Source pitch (ignored when src is array)
    dstXInBytes : size_t
        Destination X in bytes
    dstY : size_t
        Destination Y
    dstMemoryType : CUmemorytype
        Destination memory type (host, device, array)
    dstHost : Any
        Destination host pointer
    dstDevice : CUdeviceptr
        Destination device pointer
    dstArray : CUarray
        Destination array reference
    dstPitch : size_t
        Destination pitch (ignored when dst is array)
    WidthInBytes : size_t
        Width of 2D memory copy in bytes
    Height : size_t
        Height of 2D memory copy

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUDA_MEMCPY2D_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._srcDevice = CUdeviceptr(_ptr=<void_ptr>&self._ptr[0].srcDevice)
        self._srcArray = CUarray(_ptr=<void_ptr>&self._ptr[0].srcArray)
        self._dstDevice = CUdeviceptr(_ptr=<void_ptr>&self._ptr[0].dstDevice)
        self._dstArray = CUarray(_ptr=<void_ptr>&self._ptr[0].dstArray)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['srcXInBytes : ' + str(self.srcXInBytes)]
            except ValueError:
                str_list += ['srcXInBytes : <ValueError>']
            try:
                str_list += ['srcY : ' + str(self.srcY)]
            except ValueError:
                str_list += ['srcY : <ValueError>']
            try:
                str_list += ['srcMemoryType : ' + str(self.srcMemoryType)]
            except ValueError:
                str_list += ['srcMemoryType : <ValueError>']
            try:
                str_list += ['srcHost : ' + hex(self.srcHost)]
            except ValueError:
                str_list += ['srcHost : <ValueError>']
            try:
                str_list += ['srcDevice : ' + str(self.srcDevice)]
            except ValueError:
                str_list += ['srcDevice : <ValueError>']
            try:
                str_list += ['srcArray : ' + str(self.srcArray)]
            except ValueError:
                str_list += ['srcArray : <ValueError>']
            try:
                str_list += ['srcPitch : ' + str(self.srcPitch)]
            except ValueError:
                str_list += ['srcPitch : <ValueError>']
            try:
                str_list += ['dstXInBytes : ' + str(self.dstXInBytes)]
            except ValueError:
                str_list += ['dstXInBytes : <ValueError>']
            try:
                str_list += ['dstY : ' + str(self.dstY)]
            except ValueError:
                str_list += ['dstY : <ValueError>']
            try:
                str_list += ['dstMemoryType : ' + str(self.dstMemoryType)]
            except ValueError:
                str_list += ['dstMemoryType : <ValueError>']
            try:
                str_list += ['dstHost : ' + hex(self.dstHost)]
            except ValueError:
                str_list += ['dstHost : <ValueError>']
            try:
                str_list += ['dstDevice : ' + str(self.dstDevice)]
            except ValueError:
                str_list += ['dstDevice : <ValueError>']
            try:
                str_list += ['dstArray : ' + str(self.dstArray)]
            except ValueError:
                str_list += ['dstArray : <ValueError>']
            try:
                str_list += ['dstPitch : ' + str(self.dstPitch)]
            except ValueError:
                str_list += ['dstPitch : <ValueError>']
            try:
                str_list += ['WidthInBytes : ' + str(self.WidthInBytes)]
            except ValueError:
                str_list += ['WidthInBytes : <ValueError>']
            try:
                str_list += ['Height : ' + str(self.Height)]
            except ValueError:
                str_list += ['Height : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def srcXInBytes(self):
        return self._ptr[0].srcXInBytes
    @srcXInBytes.setter
    def srcXInBytes(self, size_t srcXInBytes):
        self._ptr[0].srcXInBytes = srcXInBytes
    @property
    def srcY(self):
        return self._ptr[0].srcY
    @srcY.setter
    def srcY(self, size_t srcY):
        self._ptr[0].srcY = srcY
    @property
    def srcMemoryType(self):
        return CUmemorytype(self._ptr[0].srcMemoryType)
    @srcMemoryType.setter
    def srcMemoryType(self, srcMemoryType not None : CUmemorytype):
        self._ptr[0].srcMemoryType = srcMemoryType.value
    @property
    def srcHost(self):
        return <void_ptr>self._ptr[0].srcHost
    @srcHost.setter
    def srcHost(self, srcHost):
        _cysrcHost = utils.HelperInputVoidPtr(srcHost)
        self._ptr[0].srcHost = <void*><void_ptr>_cysrcHost.cptr
    @property
    def srcDevice(self):
        return self._srcDevice
    @srcDevice.setter
    def srcDevice(self, srcDevice):
        cdef cydriver.CUdeviceptr cysrcDevice
        if srcDevice is None:
            cysrcDevice = <cydriver.CUdeviceptr><void_ptr>0
        elif isinstance(srcDevice, (CUdeviceptr)):
            psrcDevice = int(srcDevice)
            cysrcDevice = <cydriver.CUdeviceptr><void_ptr>psrcDevice
        else:
            psrcDevice = int(CUdeviceptr(srcDevice))
            cysrcDevice = <cydriver.CUdeviceptr><void_ptr>psrcDevice
        self._srcDevice._ptr[0] = cysrcDevice

    @property
    def srcArray(self):
        return self._srcArray
    @srcArray.setter
    def srcArray(self, srcArray):
        cdef cydriver.CUarray cysrcArray
        if srcArray is None:
            cysrcArray = <cydriver.CUarray><void_ptr>0
        elif isinstance(srcArray, (CUarray,)):
            psrcArray = int(srcArray)
            cysrcArray = <cydriver.CUarray><void_ptr>psrcArray
        else:
            psrcArray = int(CUarray(srcArray))
            cysrcArray = <cydriver.CUarray><void_ptr>psrcArray
        self._srcArray._ptr[0] = cysrcArray
    @property
    def srcPitch(self):
        return self._ptr[0].srcPitch
    @srcPitch.setter
    def srcPitch(self, size_t srcPitch):
        self._ptr[0].srcPitch = srcPitch
    @property
    def dstXInBytes(self):
        return self._ptr[0].dstXInBytes
    @dstXInBytes.setter
    def dstXInBytes(self, size_t dstXInBytes):
        self._ptr[0].dstXInBytes = dstXInBytes
    @property
    def dstY(self):
        return self._ptr[0].dstY
    @dstY.setter
    def dstY(self, size_t dstY):
        self._ptr[0].dstY = dstY
    @property
    def dstMemoryType(self):
        return CUmemorytype(self._ptr[0].dstMemoryType)
    @dstMemoryType.setter
    def dstMemoryType(self, dstMemoryType not None : CUmemorytype):
        self._ptr[0].dstMemoryType = dstMemoryType.value
    @property
    def dstHost(self):
        return <void_ptr>self._ptr[0].dstHost
    @dstHost.setter
    def dstHost(self, dstHost):
        _cydstHost = utils.HelperInputVoidPtr(dstHost)
        self._ptr[0].dstHost = <void*><void_ptr>_cydstHost.cptr
    @property
    def dstDevice(self):
        return self._dstDevice
    @dstDevice.setter
    def dstDevice(self, dstDevice):
        cdef cydriver.CUdeviceptr cydstDevice
        if dstDevice is None:
            cydstDevice = <cydriver.CUdeviceptr><void_ptr>0
        elif isinstance(dstDevice, (CUdeviceptr)):
            pdstDevice = int(dstDevice)
            cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
        else:
            pdstDevice = int(CUdeviceptr(dstDevice))
            cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
        self._dstDevice._ptr[0] = cydstDevice

    @property
    def dstArray(self):
        return self._dstArray
    @dstArray.setter
    def dstArray(self, dstArray):
        cdef cydriver.CUarray cydstArray
        if dstArray is None:
            cydstArray = <cydriver.CUarray><void_ptr>0
        elif isinstance(dstArray, (CUarray,)):
            pdstArray = int(dstArray)
            cydstArray = <cydriver.CUarray><void_ptr>pdstArray
        else:
            pdstArray = int(CUarray(dstArray))
            cydstArray = <cydriver.CUarray><void_ptr>pdstArray
        self._dstArray._ptr[0] = cydstArray
    @property
    def dstPitch(self):
        return self._ptr[0].dstPitch
    @dstPitch.setter
    def dstPitch(self, size_t dstPitch):
        self._ptr[0].dstPitch = dstPitch
    @property
    def WidthInBytes(self):
        return self._ptr[0].WidthInBytes
    @WidthInBytes.setter
    def WidthInBytes(self, size_t WidthInBytes):
        self._ptr[0].WidthInBytes = WidthInBytes
    @property
    def Height(self):
        return self._ptr[0].Height
    @Height.setter
    def Height(self, size_t Height):
        self._ptr[0].Height = Height

cdef class CUDA_MEMCPY3D_st:
    """
    3D memory copy parameters

    Attributes
    ----------
    srcXInBytes : size_t
        Source X in bytes
    srcY : size_t
        Source Y
    srcZ : size_t
        Source Z
    srcLOD : size_t
        Source LOD
    srcMemoryType : CUmemorytype
        Source memory type (host, device, array)
    srcHost : Any
        Source host pointer
    srcDevice : CUdeviceptr
        Source device pointer
    srcArray : CUarray
        Source array reference
    reserved0 : Any
        Must be NULL
    srcPitch : size_t
        Source pitch (ignored when src is array)
    srcHeight : size_t
        Source height (ignored when src is array; may be 0 if Depth==1)
    dstXInBytes : size_t
        Destination X in bytes
    dstY : size_t
        Destination Y
    dstZ : size_t
        Destination Z
    dstLOD : size_t
        Destination LOD
    dstMemoryType : CUmemorytype
        Destination memory type (host, device, array)
    dstHost : Any
        Destination host pointer
    dstDevice : CUdeviceptr
        Destination device pointer
    dstArray : CUarray
        Destination array reference
    reserved1 : Any
        Must be NULL
    dstPitch : size_t
        Destination pitch (ignored when dst is array)
    dstHeight : size_t
        Destination height (ignored when dst is array; may be 0 if
        Depth==1)
    WidthInBytes : size_t
        Width of 3D memory copy in bytes
    Height : size_t
        Height of 3D memory copy
    Depth : size_t
        Depth of 3D memory copy

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUDA_MEMCPY3D_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._srcDevice = CUdeviceptr(_ptr=<void_ptr>&self._ptr[0].srcDevice)
        self._srcArray = CUarray(_ptr=<void_ptr>&self._ptr[0].srcArray)
        self._dstDevice = CUdeviceptr(_ptr=<void_ptr>&self._ptr[0].dstDevice)
        self._dstArray = CUarray(_ptr=<void_ptr>&self._ptr[0].dstArray)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['srcXInBytes : ' + str(self.srcXInBytes)]
            except ValueError:
                str_list += ['srcXInBytes : <ValueError>']
            try:
                str_list += ['srcY : ' + str(self.srcY)]
            except ValueError:
                str_list += ['srcY : <ValueError>']
            try:
                str_list += ['srcZ : ' + str(self.srcZ)]
            except ValueError:
                str_list += ['srcZ : <ValueError>']
            try:
                str_list += ['srcLOD : ' + str(self.srcLOD)]
            except ValueError:
                str_list += ['srcLOD : <ValueError>']
            try:
                str_list += ['srcMemoryType : ' + str(self.srcMemoryType)]
            except ValueError:
                str_list += ['srcMemoryType : <ValueError>']
            try:
                str_list += ['srcHost : ' + hex(self.srcHost)]
            except ValueError:
                str_list += ['srcHost : <ValueError>']
            try:
                str_list += ['srcDevice : ' + str(self.srcDevice)]
            except ValueError:
                str_list += ['srcDevice : <ValueError>']
            try:
                str_list += ['srcArray : ' + str(self.srcArray)]
            except ValueError:
                str_list += ['srcArray : <ValueError>']
            try:
                str_list += ['reserved0 : ' + hex(self.reserved0)]
            except ValueError:
                str_list += ['reserved0 : <ValueError>']
            try:
                str_list += ['srcPitch : ' + str(self.srcPitch)]
            except ValueError:
                str_list += ['srcPitch : <ValueError>']
            try:
                str_list += ['srcHeight : ' + str(self.srcHeight)]
            except ValueError:
                str_list += ['srcHeight : <ValueError>']
            try:
                str_list += ['dstXInBytes : ' + str(self.dstXInBytes)]
            except ValueError:
                str_list += ['dstXInBytes : <ValueError>']
            try:
                str_list += ['dstY : ' + str(self.dstY)]
            except ValueError:
                str_list += ['dstY : <ValueError>']
            try:
                str_list += ['dstZ : ' + str(self.dstZ)]
            except ValueError:
                str_list += ['dstZ : <ValueError>']
            try:
                str_list += ['dstLOD : ' + str(self.dstLOD)]
            except ValueError:
                str_list += ['dstLOD : <ValueError>']
            try:
                str_list += ['dstMemoryType : ' + str(self.dstMemoryType)]
            except ValueError:
                str_list += ['dstMemoryType : <ValueError>']
            try:
                str_list += ['dstHost : ' + hex(self.dstHost)]
            except ValueError:
                str_list += ['dstHost : <ValueError>']
            try:
                str_list += ['dstDevice : ' + str(self.dstDevice)]
            except ValueError:
                str_list += ['dstDevice : <ValueError>']
            try:
                str_list += ['dstArray : ' + str(self.dstArray)]
            except ValueError:
                str_list += ['dstArray : <ValueError>']
            try:
                str_list += ['reserved1 : ' + hex(self.reserved1)]
            except ValueError:
                str_list += ['reserved1 : <ValueError>']
            try:
                str_list += ['dstPitch : ' + str(self.dstPitch)]
            except ValueError:
                str_list += ['dstPitch : <ValueError>']
            try:
                str_list += ['dstHeight : ' + str(self.dstHeight)]
            except ValueError:
                str_list += ['dstHeight : <ValueError>']
            try:
                str_list += ['WidthInBytes : ' + str(self.WidthInBytes)]
            except ValueError:
                str_list += ['WidthInBytes : <ValueError>']
            try:
                str_list += ['Height : ' + str(self.Height)]
            except ValueError:
                str_list += ['Height : <ValueError>']
            try:
                str_list += ['Depth : ' + str(self.Depth)]
            except ValueError:
                str_list += ['Depth : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def srcXInBytes(self):
        return self._ptr[0].srcXInBytes
    @srcXInBytes.setter
    def srcXInBytes(self, size_t srcXInBytes):
        self._ptr[0].srcXInBytes = srcXInBytes
    @property
    def srcY(self):
        return self._ptr[0].srcY
    @srcY.setter
    def srcY(self, size_t srcY):
        self._ptr[0].srcY = srcY
    @property
    def srcZ(self):
        return self._ptr[0].srcZ
    @srcZ.setter
    def srcZ(self, size_t srcZ):
        self._ptr[0].srcZ = srcZ
    @property
    def srcLOD(self):
        return self._ptr[0].srcLOD
    @srcLOD.setter
    def srcLOD(self, size_t srcLOD):
        self._ptr[0].srcLOD = srcLOD
    @property
    def srcMemoryType(self):
        return CUmemorytype(self._ptr[0].srcMemoryType)
    @srcMemoryType.setter
    def srcMemoryType(self, srcMemoryType not None : CUmemorytype):
        self._ptr[0].srcMemoryType = srcMemoryType.value
    @property
    def srcHost(self):
        return <void_ptr>self._ptr[0].srcHost
    @srcHost.setter
    def srcHost(self, srcHost):
        _cysrcHost = utils.HelperInputVoidPtr(srcHost)
        self._ptr[0].srcHost = <void*><void_ptr>_cysrcHost.cptr
    @property
    def srcDevice(self):
        return self._srcDevice
    @srcDevice.setter
    def srcDevice(self, srcDevice):
        cdef cydriver.CUdeviceptr cysrcDevice
        if srcDevice is None:
            cysrcDevice = <cydriver.CUdeviceptr><void_ptr>0
        elif isinstance(srcDevice, (CUdeviceptr)):
            psrcDevice = int(srcDevice)
            cysrcDevice = <cydriver.CUdeviceptr><void_ptr>psrcDevice
        else:
            psrcDevice = int(CUdeviceptr(srcDevice))
            cysrcDevice = <cydriver.CUdeviceptr><void_ptr>psrcDevice
        self._srcDevice._ptr[0] = cysrcDevice

    @property
    def srcArray(self):
        return self._srcArray
    @srcArray.setter
    def srcArray(self, srcArray):
        cdef cydriver.CUarray cysrcArray
        if srcArray is None:
            cysrcArray = <cydriver.CUarray><void_ptr>0
        elif isinstance(srcArray, (CUarray,)):
            psrcArray = int(srcArray)
            cysrcArray = <cydriver.CUarray><void_ptr>psrcArray
        else:
            psrcArray = int(CUarray(srcArray))
            cysrcArray = <cydriver.CUarray><void_ptr>psrcArray
        self._srcArray._ptr[0] = cysrcArray
    @property
    def reserved0(self):
        return <void_ptr>self._ptr[0].reserved0
    @reserved0.setter
    def reserved0(self, reserved0):
        _cyreserved0 = utils.HelperInputVoidPtr(reserved0)
        self._ptr[0].reserved0 = <void*><void_ptr>_cyreserved0.cptr
    @property
    def srcPitch(self):
        return self._ptr[0].srcPitch
    @srcPitch.setter
    def srcPitch(self, size_t srcPitch):
        self._ptr[0].srcPitch = srcPitch
    @property
    def srcHeight(self):
        return self._ptr[0].srcHeight
    @srcHeight.setter
    def srcHeight(self, size_t srcHeight):
        self._ptr[0].srcHeight = srcHeight
    @property
    def dstXInBytes(self):
        return self._ptr[0].dstXInBytes
    @dstXInBytes.setter
    def dstXInBytes(self, size_t dstXInBytes):
        self._ptr[0].dstXInBytes = dstXInBytes
    @property
    def dstY(self):
        return self._ptr[0].dstY
    @dstY.setter
    def dstY(self, size_t dstY):
        self._ptr[0].dstY = dstY
    @property
    def dstZ(self):
        return self._ptr[0].dstZ
    @dstZ.setter
    def dstZ(self, size_t dstZ):
        self._ptr[0].dstZ = dstZ
    @property
    def dstLOD(self):
        return self._ptr[0].dstLOD
    @dstLOD.setter
    def dstLOD(self, size_t dstLOD):
        self._ptr[0].dstLOD = dstLOD
    @property
    def dstMemoryType(self):
        return CUmemorytype(self._ptr[0].dstMemoryType)
    @dstMemoryType.setter
    def dstMemoryType(self, dstMemoryType not None : CUmemorytype):
        self._ptr[0].dstMemoryType = dstMemoryType.value
    @property
    def dstHost(self):
        return <void_ptr>self._ptr[0].dstHost
    @dstHost.setter
    def dstHost(self, dstHost):
        _cydstHost = utils.HelperInputVoidPtr(dstHost)
        self._ptr[0].dstHost = <void*><void_ptr>_cydstHost.cptr
    @property
    def dstDevice(self):
        return self._dstDevice
    @dstDevice.setter
    def dstDevice(self, dstDevice):
        cdef cydriver.CUdeviceptr cydstDevice
        if dstDevice is None:
            cydstDevice = <cydriver.CUdeviceptr><void_ptr>0
        elif isinstance(dstDevice, (CUdeviceptr)):
            pdstDevice = int(dstDevice)
            cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
        else:
            pdstDevice = int(CUdeviceptr(dstDevice))
            cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
        self._dstDevice._ptr[0] = cydstDevice

    @property
    def dstArray(self):
        return self._dstArray
    @dstArray.setter
    def dstArray(self, dstArray):
        cdef cydriver.CUarray cydstArray
        if dstArray is None:
            cydstArray = <cydriver.CUarray><void_ptr>0
        elif isinstance(dstArray, (CUarray,)):
            pdstArray = int(dstArray)
            cydstArray = <cydriver.CUarray><void_ptr>pdstArray
        else:
            pdstArray = int(CUarray(dstArray))
            cydstArray = <cydriver.CUarray><void_ptr>pdstArray
        self._dstArray._ptr[0] = cydstArray
    @property
    def reserved1(self):
        return <void_ptr>self._ptr[0].reserved1
    @reserved1.setter
    def reserved1(self, reserved1):
        _cyreserved1 = utils.HelperInputVoidPtr(reserved1)
        self._ptr[0].reserved1 = <void*><void_ptr>_cyreserved1.cptr
    @property
    def dstPitch(self):
        return self._ptr[0].dstPitch
    @dstPitch.setter
    def dstPitch(self, size_t dstPitch):
        self._ptr[0].dstPitch = dstPitch
    @property
    def dstHeight(self):
        return self._ptr[0].dstHeight
    @dstHeight.setter
    def dstHeight(self, size_t dstHeight):
        self._ptr[0].dstHeight = dstHeight
    @property
    def WidthInBytes(self):
        return self._ptr[0].WidthInBytes
    @WidthInBytes.setter
    def WidthInBytes(self, size_t WidthInBytes):
        self._ptr[0].WidthInBytes = WidthInBytes
    @property
    def Height(self):
        return self._ptr[0].Height
    @Height.setter
    def Height(self, size_t Height):
        self._ptr[0].Height = Height
    @property
    def Depth(self):
        return self._ptr[0].Depth
    @Depth.setter
    def Depth(self, size_t Depth):
        self._ptr[0].Depth = Depth

cdef class CUDA_MEMCPY3D_PEER_st:
    """
    3D memory cross-context copy parameters

    Attributes
    ----------
    srcXInBytes : size_t
        Source X in bytes
    srcY : size_t
        Source Y
    srcZ : size_t
        Source Z
    srcLOD : size_t
        Source LOD
    srcMemoryType : CUmemorytype
        Source memory type (host, device, array)
    srcHost : Any
        Source host pointer
    srcDevice : CUdeviceptr
        Source device pointer
    srcArray : CUarray
        Source array reference
    srcContext : CUcontext
        Source context (ignored with srcMemoryType is CU_MEMORYTYPE_ARRAY)
    srcPitch : size_t
        Source pitch (ignored when src is array)
    srcHeight : size_t
        Source height (ignored when src is array; may be 0 if Depth==1)
    dstXInBytes : size_t
        Destination X in bytes
    dstY : size_t
        Destination Y
    dstZ : size_t
        Destination Z
    dstLOD : size_t
        Destination LOD
    dstMemoryType : CUmemorytype
        Destination memory type (host, device, array)
    dstHost : Any
        Destination host pointer
    dstDevice : CUdeviceptr
        Destination device pointer
    dstArray : CUarray
        Destination array reference
    dstContext : CUcontext
        Destination context (ignored with dstMemoryType is
        CU_MEMORYTYPE_ARRAY)
    dstPitch : size_t
        Destination pitch (ignored when dst is array)
    dstHeight : size_t
        Destination height (ignored when dst is array; may be 0 if
        Depth==1)
    WidthInBytes : size_t
        Width of 3D memory copy in bytes
    Height : size_t
        Height of 3D memory copy
    Depth : size_t
        Depth of 3D memory copy

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUDA_MEMCPY3D_PEER_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._srcDevice = CUdeviceptr(_ptr=<void_ptr>&self._ptr[0].srcDevice)
        self._srcArray = CUarray(_ptr=<void_ptr>&self._ptr[0].srcArray)
        self._srcContext = CUcontext(_ptr=<void_ptr>&self._ptr[0].srcContext)
        self._dstDevice = CUdeviceptr(_ptr=<void_ptr>&self._ptr[0].dstDevice)
        self._dstArray = CUarray(_ptr=<void_ptr>&self._ptr[0].dstArray)
        self._dstContext = CUcontext(_ptr=<void_ptr>&self._ptr[0].dstContext)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['srcXInBytes : ' + str(self.srcXInBytes)]
            except ValueError:
                str_list += ['srcXInBytes : <ValueError>']
            try:
                str_list += ['srcY : ' + str(self.srcY)]
            except ValueError:
                str_list += ['srcY : <ValueError>']
            try:
                str_list += ['srcZ : ' + str(self.srcZ)]
            except ValueError:
                str_list += ['srcZ : <ValueError>']
            try:
                str_list += ['srcLOD : ' + str(self.srcLOD)]
            except ValueError:
                str_list += ['srcLOD : <ValueError>']
            try:
                str_list += ['srcMemoryType : ' + str(self.srcMemoryType)]
            except ValueError:
                str_list += ['srcMemoryType : <ValueError>']
            try:
                str_list += ['srcHost : ' + hex(self.srcHost)]
            except ValueError:
                str_list += ['srcHost : <ValueError>']
            try:
                str_list += ['srcDevice : ' + str(self.srcDevice)]
            except ValueError:
                str_list += ['srcDevice : <ValueError>']
            try:
                str_list += ['srcArray : ' + str(self.srcArray)]
            except ValueError:
                str_list += ['srcArray : <ValueError>']
            try:
                str_list += ['srcContext : ' + str(self.srcContext)]
            except ValueError:
                str_list += ['srcContext : <ValueError>']
            try:
                str_list += ['srcPitch : ' + str(self.srcPitch)]
            except ValueError:
                str_list += ['srcPitch : <ValueError>']
            try:
                str_list += ['srcHeight : ' + str(self.srcHeight)]
            except ValueError:
                str_list += ['srcHeight : <ValueError>']
            try:
                str_list += ['dstXInBytes : ' + str(self.dstXInBytes)]
            except ValueError:
                str_list += ['dstXInBytes : <ValueError>']
            try:
                str_list += ['dstY : ' + str(self.dstY)]
            except ValueError:
                str_list += ['dstY : <ValueError>']
            try:
                str_list += ['dstZ : ' + str(self.dstZ)]
            except ValueError:
                str_list += ['dstZ : <ValueError>']
            try:
                str_list += ['dstLOD : ' + str(self.dstLOD)]
            except ValueError:
                str_list += ['dstLOD : <ValueError>']
            try:
                str_list += ['dstMemoryType : ' + str(self.dstMemoryType)]
            except ValueError:
                str_list += ['dstMemoryType : <ValueError>']
            try:
                str_list += ['dstHost : ' + hex(self.dstHost)]
            except ValueError:
                str_list += ['dstHost : <ValueError>']
            try:
                str_list += ['dstDevice : ' + str(self.dstDevice)]
            except ValueError:
                str_list += ['dstDevice : <ValueError>']
            try:
                str_list += ['dstArray : ' + str(self.dstArray)]
            except ValueError:
                str_list += ['dstArray : <ValueError>']
            try:
                str_list += ['dstContext : ' + str(self.dstContext)]
            except ValueError:
                str_list += ['dstContext : <ValueError>']
            try:
                str_list += ['dstPitch : ' + str(self.dstPitch)]
            except ValueError:
                str_list += ['dstPitch : <ValueError>']
            try:
                str_list += ['dstHeight : ' + str(self.dstHeight)]
            except ValueError:
                str_list += ['dstHeight : <ValueError>']
            try:
                str_list += ['WidthInBytes : ' + str(self.WidthInBytes)]
            except ValueError:
                str_list += ['WidthInBytes : <ValueError>']
            try:
                str_list += ['Height : ' + str(self.Height)]
            except ValueError:
                str_list += ['Height : <ValueError>']
            try:
                str_list += ['Depth : ' + str(self.Depth)]
            except ValueError:
                str_list += ['Depth : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def srcXInBytes(self):
        return self._ptr[0].srcXInBytes
    @srcXInBytes.setter
    def srcXInBytes(self, size_t srcXInBytes):
        self._ptr[0].srcXInBytes = srcXInBytes
    @property
    def srcY(self):
        return self._ptr[0].srcY
    @srcY.setter
    def srcY(self, size_t srcY):
        self._ptr[0].srcY = srcY
    @property
    def srcZ(self):
        return self._ptr[0].srcZ
    @srcZ.setter
    def srcZ(self, size_t srcZ):
        self._ptr[0].srcZ = srcZ
    @property
    def srcLOD(self):
        return self._ptr[0].srcLOD
    @srcLOD.setter
    def srcLOD(self, size_t srcLOD):
        self._ptr[0].srcLOD = srcLOD
    @property
    def srcMemoryType(self):
        return CUmemorytype(self._ptr[0].srcMemoryType)
    @srcMemoryType.setter
    def srcMemoryType(self, srcMemoryType not None : CUmemorytype):
        self._ptr[0].srcMemoryType = srcMemoryType.value
    @property
    def srcHost(self):
        return <void_ptr>self._ptr[0].srcHost
    @srcHost.setter
    def srcHost(self, srcHost):
        _cysrcHost = utils.HelperInputVoidPtr(srcHost)
        self._ptr[0].srcHost = <void*><void_ptr>_cysrcHost.cptr
    @property
    def srcDevice(self):
        return self._srcDevice
    @srcDevice.setter
    def srcDevice(self, srcDevice):
        cdef cydriver.CUdeviceptr cysrcDevice
        if srcDevice is None:
            cysrcDevice = <cydriver.CUdeviceptr><void_ptr>0
        elif isinstance(srcDevice, (CUdeviceptr)):
            psrcDevice = int(srcDevice)
            cysrcDevice = <cydriver.CUdeviceptr><void_ptr>psrcDevice
        else:
            psrcDevice = int(CUdeviceptr(srcDevice))
            cysrcDevice = <cydriver.CUdeviceptr><void_ptr>psrcDevice
        self._srcDevice._ptr[0] = cysrcDevice

    @property
    def srcArray(self):
        return self._srcArray
    @srcArray.setter
    def srcArray(self, srcArray):
        cdef cydriver.CUarray cysrcArray
        if srcArray is None:
            cysrcArray = <cydriver.CUarray><void_ptr>0
        elif isinstance(srcArray, (CUarray,)):
            psrcArray = int(srcArray)
            cysrcArray = <cydriver.CUarray><void_ptr>psrcArray
        else:
            psrcArray = int(CUarray(srcArray))
            cysrcArray = <cydriver.CUarray><void_ptr>psrcArray
        self._srcArray._ptr[0] = cysrcArray
    @property
    def srcContext(self):
        return self._srcContext
    @srcContext.setter
    def srcContext(self, srcContext):
        cdef cydriver.CUcontext cysrcContext
        if srcContext is None:
            cysrcContext = <cydriver.CUcontext><void_ptr>0
        elif isinstance(srcContext, (CUcontext,)):
            psrcContext = int(srcContext)
            cysrcContext = <cydriver.CUcontext><void_ptr>psrcContext
        else:
            psrcContext = int(CUcontext(srcContext))
            cysrcContext = <cydriver.CUcontext><void_ptr>psrcContext
        self._srcContext._ptr[0] = cysrcContext
    @property
    def srcPitch(self):
        return self._ptr[0].srcPitch
    @srcPitch.setter
    def srcPitch(self, size_t srcPitch):
        self._ptr[0].srcPitch = srcPitch
    @property
    def srcHeight(self):
        return self._ptr[0].srcHeight
    @srcHeight.setter
    def srcHeight(self, size_t srcHeight):
        self._ptr[0].srcHeight = srcHeight
    @property
    def dstXInBytes(self):
        return self._ptr[0].dstXInBytes
    @dstXInBytes.setter
    def dstXInBytes(self, size_t dstXInBytes):
        self._ptr[0].dstXInBytes = dstXInBytes
    @property
    def dstY(self):
        return self._ptr[0].dstY
    @dstY.setter
    def dstY(self, size_t dstY):
        self._ptr[0].dstY = dstY
    @property
    def dstZ(self):
        return self._ptr[0].dstZ
    @dstZ.setter
    def dstZ(self, size_t dstZ):
        self._ptr[0].dstZ = dstZ
    @property
    def dstLOD(self):
        return self._ptr[0].dstLOD
    @dstLOD.setter
    def dstLOD(self, size_t dstLOD):
        self._ptr[0].dstLOD = dstLOD
    @property
    def dstMemoryType(self):
        return CUmemorytype(self._ptr[0].dstMemoryType)
    @dstMemoryType.setter
    def dstMemoryType(self, dstMemoryType not None : CUmemorytype):
        self._ptr[0].dstMemoryType = dstMemoryType.value
    @property
    def dstHost(self):
        return <void_ptr>self._ptr[0].dstHost
    @dstHost.setter
    def dstHost(self, dstHost):
        _cydstHost = utils.HelperInputVoidPtr(dstHost)
        self._ptr[0].dstHost = <void*><void_ptr>_cydstHost.cptr
    @property
    def dstDevice(self):
        return self._dstDevice
    @dstDevice.setter
    def dstDevice(self, dstDevice):
        cdef cydriver.CUdeviceptr cydstDevice
        if dstDevice is None:
            cydstDevice = <cydriver.CUdeviceptr><void_ptr>0
        elif isinstance(dstDevice, (CUdeviceptr)):
            pdstDevice = int(dstDevice)
            cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
        else:
            pdstDevice = int(CUdeviceptr(dstDevice))
            cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
        self._dstDevice._ptr[0] = cydstDevice

    @property
    def dstArray(self):
        return self._dstArray
    @dstArray.setter
    def dstArray(self, dstArray):
        cdef cydriver.CUarray cydstArray
        if dstArray is None:
            cydstArray = <cydriver.CUarray><void_ptr>0
        elif isinstance(dstArray, (CUarray,)):
            pdstArray = int(dstArray)
            cydstArray = <cydriver.CUarray><void_ptr>pdstArray
        else:
            pdstArray = int(CUarray(dstArray))
            cydstArray = <cydriver.CUarray><void_ptr>pdstArray
        self._dstArray._ptr[0] = cydstArray
    @property
    def dstContext(self):
        return self._dstContext
    @dstContext.setter
    def dstContext(self, dstContext):
        cdef cydriver.CUcontext cydstContext
        if dstContext is None:
            cydstContext = <cydriver.CUcontext><void_ptr>0
        elif isinstance(dstContext, (CUcontext,)):
            pdstContext = int(dstContext)
            cydstContext = <cydriver.CUcontext><void_ptr>pdstContext
        else:
            pdstContext = int(CUcontext(dstContext))
            cydstContext = <cydriver.CUcontext><void_ptr>pdstContext
        self._dstContext._ptr[0] = cydstContext
    @property
    def dstPitch(self):
        return self._ptr[0].dstPitch
    @dstPitch.setter
    def dstPitch(self, size_t dstPitch):
        self._ptr[0].dstPitch = dstPitch
    @property
    def dstHeight(self):
        return self._ptr[0].dstHeight
    @dstHeight.setter
    def dstHeight(self, size_t dstHeight):
        self._ptr[0].dstHeight = dstHeight
    @property
    def WidthInBytes(self):
        return self._ptr[0].WidthInBytes
    @WidthInBytes.setter
    def WidthInBytes(self, size_t WidthInBytes):
        self._ptr[0].WidthInBytes = WidthInBytes
    @property
    def Height(self):
        return self._ptr[0].Height
    @Height.setter
    def Height(self, size_t Height):
        self._ptr[0].Height = Height
    @property
    def Depth(self):
        return self._ptr[0].Depth
    @Depth.setter
    def Depth(self, size_t Depth):
        self._ptr[0].Depth = Depth

cdef class CUDA_ARRAY_DESCRIPTOR_st:
    """
    Array descriptor

    Attributes
    ----------
    Width : size_t
        Width of array
    Height : size_t
        Height of array
    Format : CUarray_format
        Array format
    NumChannels : unsigned int
        Channels per array element

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUDA_ARRAY_DESCRIPTOR_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['Width : ' + str(self.Width)]
            except ValueError:
                str_list += ['Width : <ValueError>']
            try:
                str_list += ['Height : ' + str(self.Height)]
            except ValueError:
                str_list += ['Height : <ValueError>']
            try:
                str_list += ['Format : ' + str(self.Format)]
            except ValueError:
                str_list += ['Format : <ValueError>']
            try:
                str_list += ['NumChannels : ' + str(self.NumChannels)]
            except ValueError:
                str_list += ['NumChannels : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def Width(self):
        return self._ptr[0].Width
    @Width.setter
    def Width(self, size_t Width):
        self._ptr[0].Width = Width
    @property
    def Height(self):
        return self._ptr[0].Height
    @Height.setter
    def Height(self, size_t Height):
        self._ptr[0].Height = Height
    @property
    def Format(self):
        return CUarray_format(self._ptr[0].Format)
    @Format.setter
    def Format(self, Format not None : CUarray_format):
        self._ptr[0].Format = Format.value
    @property
    def NumChannels(self):
        return self._ptr[0].NumChannels
    @NumChannels.setter
    def NumChannels(self, unsigned int NumChannels):
        self._ptr[0].NumChannels = NumChannels

cdef class CUDA_ARRAY3D_DESCRIPTOR_st:
    """
    3D array descriptor

    Attributes
    ----------
    Width : size_t
        Width of 3D array
    Height : size_t
        Height of 3D array
    Depth : size_t
        Depth of 3D array
    Format : CUarray_format
        Array format
    NumChannels : unsigned int
        Channels per array element
    Flags : unsigned int
        Flags

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUDA_ARRAY3D_DESCRIPTOR_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['Width : ' + str(self.Width)]
            except ValueError:
                str_list += ['Width : <ValueError>']
            try:
                str_list += ['Height : ' + str(self.Height)]
            except ValueError:
                str_list += ['Height : <ValueError>']
            try:
                str_list += ['Depth : ' + str(self.Depth)]
            except ValueError:
                str_list += ['Depth : <ValueError>']
            try:
                str_list += ['Format : ' + str(self.Format)]
            except ValueError:
                str_list += ['Format : <ValueError>']
            try:
                str_list += ['NumChannels : ' + str(self.NumChannels)]
            except ValueError:
                str_list += ['NumChannels : <ValueError>']
            try:
                str_list += ['Flags : ' + str(self.Flags)]
            except ValueError:
                str_list += ['Flags : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def Width(self):
        return self._ptr[0].Width
    @Width.setter
    def Width(self, size_t Width):
        self._ptr[0].Width = Width
    @property
    def Height(self):
        return self._ptr[0].Height
    @Height.setter
    def Height(self, size_t Height):
        self._ptr[0].Height = Height
    @property
    def Depth(self):
        return self._ptr[0].Depth
    @Depth.setter
    def Depth(self, size_t Depth):
        self._ptr[0].Depth = Depth
    @property
    def Format(self):
        return CUarray_format(self._ptr[0].Format)
    @Format.setter
    def Format(self, Format not None : CUarray_format):
        self._ptr[0].Format = Format.value
    @property
    def NumChannels(self):
        return self._ptr[0].NumChannels
    @NumChannels.setter
    def NumChannels(self, unsigned int NumChannels):
        self._ptr[0].NumChannels = NumChannels
    @property
    def Flags(self):
        return self._ptr[0].Flags
    @Flags.setter
    def Flags(self, unsigned int Flags):
        self._ptr[0].Flags = Flags

cdef class anon_struct2:
    """
    Attributes
    ----------
    width : unsigned int

    height : unsigned int

    depth : unsigned int


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <cydriver.CUDA_ARRAY_SPARSE_PROPERTIES_st *>_ptr

    def __init__(self, void_ptr _ptr):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].tileExtent
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['width : ' + str(self.width)]
            except ValueError:
                str_list += ['width : <ValueError>']
            try:
                str_list += ['height : ' + str(self.height)]
            except ValueError:
                str_list += ['height : <ValueError>']
            try:
                str_list += ['depth : ' + str(self.depth)]
            except ValueError:
                str_list += ['depth : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def width(self):
        return self._ptr[0].tileExtent.width
    @width.setter
    def width(self, unsigned int width):
        self._ptr[0].tileExtent.width = width
    @property
    def height(self):
        return self._ptr[0].tileExtent.height
    @height.setter
    def height(self, unsigned int height):
        self._ptr[0].tileExtent.height = height
    @property
    def depth(self):
        return self._ptr[0].tileExtent.depth
    @depth.setter
    def depth(self, unsigned int depth):
        self._ptr[0].tileExtent.depth = depth

cdef class CUDA_ARRAY_SPARSE_PROPERTIES_st:
    """
    CUDA array sparse properties

    Attributes
    ----------
    tileExtent : anon_struct2

    miptailFirstLevel : unsigned int
        First mip level at which the mip tail begins.
    miptailSize : unsigned long long
        Total size of the mip tail.
    flags : unsigned int
        Flags will either be zero or
        CU_ARRAY_SPARSE_PROPERTIES_SINGLE_MIPTAIL
    reserved : List[unsigned int]


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUDA_ARRAY_SPARSE_PROPERTIES_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._tileExtent = anon_struct2(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['tileExtent :\n' + '\n'.join(['    ' + line for line in str(self.tileExtent).splitlines()])]
            except ValueError:
                str_list += ['tileExtent : <ValueError>']
            try:
                str_list += ['miptailFirstLevel : ' + str(self.miptailFirstLevel)]
            except ValueError:
                str_list += ['miptailFirstLevel : <ValueError>']
            try:
                str_list += ['miptailSize : ' + str(self.miptailSize)]
            except ValueError:
                str_list += ['miptailSize : <ValueError>']
            try:
                str_list += ['flags : ' + str(self.flags)]
            except ValueError:
                str_list += ['flags : <ValueError>']
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def tileExtent(self):
        return self._tileExtent
    @tileExtent.setter
    def tileExtent(self, tileExtent not None : anon_struct2):
        string.memcpy(&self._ptr[0].tileExtent, <cydriver.anon_struct2*><void_ptr>tileExtent.getPtr(), sizeof(self._ptr[0].tileExtent))
    @property
    def miptailFirstLevel(self):
        return self._ptr[0].miptailFirstLevel
    @miptailFirstLevel.setter
    def miptailFirstLevel(self, unsigned int miptailFirstLevel):
        self._ptr[0].miptailFirstLevel = miptailFirstLevel
    @property
    def miptailSize(self):
        return self._ptr[0].miptailSize
    @miptailSize.setter
    def miptailSize(self, unsigned long long miptailSize):
        self._ptr[0].miptailSize = miptailSize
    @property
    def flags(self):
        return self._ptr[0].flags
    @flags.setter
    def flags(self, unsigned int flags):
        self._ptr[0].flags = flags
    @property
    def reserved(self):
        return self._ptr[0].reserved
    @reserved.setter
    def reserved(self, reserved):
        self._ptr[0].reserved = reserved

cdef class CUDA_ARRAY_MEMORY_REQUIREMENTS_st:
    """
    CUDA array memory requirements

    Attributes
    ----------
    size : size_t
        Total required memory size
    alignment : size_t
        alignment requirement
    reserved : List[unsigned int]


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUDA_ARRAY_MEMORY_REQUIREMENTS_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['size : ' + str(self.size)]
            except ValueError:
                str_list += ['size : <ValueError>']
            try:
                str_list += ['alignment : ' + str(self.alignment)]
            except ValueError:
                str_list += ['alignment : <ValueError>']
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def size(self):
        return self._ptr[0].size
    @size.setter
    def size(self, size_t size):
        self._ptr[0].size = size
    @property
    def alignment(self):
        return self._ptr[0].alignment
    @alignment.setter
    def alignment(self, size_t alignment):
        self._ptr[0].alignment = alignment
    @property
    def reserved(self):
        return self._ptr[0].reserved
    @reserved.setter
    def reserved(self, reserved):
        self._ptr[0].reserved = reserved

cdef class anon_struct3:
    """
    Attributes
    ----------
    hArray : CUarray


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <cydriver.CUDA_RESOURCE_DESC_st *>_ptr

    def __init__(self, void_ptr _ptr):
        self._hArray = CUarray(_ptr=<void_ptr>&self._ptr[0].res.array.hArray)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].res.array
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['hArray : ' + str(self.hArray)]
            except ValueError:
                str_list += ['hArray : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def hArray(self):
        return self._hArray
    @hArray.setter
    def hArray(self, hArray):
        cdef cydriver.CUarray cyhArray
        if hArray is None:
            cyhArray = <cydriver.CUarray><void_ptr>0
        elif isinstance(hArray, (CUarray,)):
            phArray = int(hArray)
            cyhArray = <cydriver.CUarray><void_ptr>phArray
        else:
            phArray = int(CUarray(hArray))
            cyhArray = <cydriver.CUarray><void_ptr>phArray
        self._hArray._ptr[0] = cyhArray

cdef class anon_struct4:
    """
    Attributes
    ----------
    hMipmappedArray : CUmipmappedArray


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <cydriver.CUDA_RESOURCE_DESC_st *>_ptr

    def __init__(self, void_ptr _ptr):
        self._hMipmappedArray = CUmipmappedArray(_ptr=<void_ptr>&self._ptr[0].res.mipmap.hMipmappedArray)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].res.mipmap
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['hMipmappedArray : ' + str(self.hMipmappedArray)]
            except ValueError:
                str_list += ['hMipmappedArray : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def hMipmappedArray(self):
        return self._hMipmappedArray
    @hMipmappedArray.setter
    def hMipmappedArray(self, hMipmappedArray):
        cdef cydriver.CUmipmappedArray cyhMipmappedArray
        if hMipmappedArray is None:
            cyhMipmappedArray = <cydriver.CUmipmappedArray><void_ptr>0
        elif isinstance(hMipmappedArray, (CUmipmappedArray,)):
            phMipmappedArray = int(hMipmappedArray)
            cyhMipmappedArray = <cydriver.CUmipmappedArray><void_ptr>phMipmappedArray
        else:
            phMipmappedArray = int(CUmipmappedArray(hMipmappedArray))
            cyhMipmappedArray = <cydriver.CUmipmappedArray><void_ptr>phMipmappedArray
        self._hMipmappedArray._ptr[0] = cyhMipmappedArray

cdef class anon_struct5:
    """
    Attributes
    ----------
    devPtr : CUdeviceptr

    format : CUarray_format

    numChannels : unsigned int

    sizeInBytes : size_t


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <cydriver.CUDA_RESOURCE_DESC_st *>_ptr

    def __init__(self, void_ptr _ptr):
        self._devPtr = CUdeviceptr(_ptr=<void_ptr>&self._ptr[0].res.linear.devPtr)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].res.linear
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['devPtr : ' + str(self.devPtr)]
            except ValueError:
                str_list += ['devPtr : <ValueError>']
            try:
                str_list += ['format : ' + str(self.format)]
            except ValueError:
                str_list += ['format : <ValueError>']
            try:
                str_list += ['numChannels : ' + str(self.numChannels)]
            except ValueError:
                str_list += ['numChannels : <ValueError>']
            try:
                str_list += ['sizeInBytes : ' + str(self.sizeInBytes)]
            except ValueError:
                str_list += ['sizeInBytes : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def devPtr(self):
        return self._devPtr
    @devPtr.setter
    def devPtr(self, devPtr):
        cdef cydriver.CUdeviceptr cydevPtr
        if devPtr is None:
            cydevPtr = <cydriver.CUdeviceptr><void_ptr>0
        elif isinstance(devPtr, (CUdeviceptr)):
            pdevPtr = int(devPtr)
            cydevPtr = <cydriver.CUdeviceptr><void_ptr>pdevPtr
        else:
            pdevPtr = int(CUdeviceptr(devPtr))
            cydevPtr = <cydriver.CUdeviceptr><void_ptr>pdevPtr
        self._devPtr._ptr[0] = cydevPtr

    @property
    def format(self):
        return CUarray_format(self._ptr[0].res.linear.format)
    @format.setter
    def format(self, format not None : CUarray_format):
        self._ptr[0].res.linear.format = format.value
    @property
    def numChannels(self):
        return self._ptr[0].res.linear.numChannels
    @numChannels.setter
    def numChannels(self, unsigned int numChannels):
        self._ptr[0].res.linear.numChannels = numChannels
    @property
    def sizeInBytes(self):
        return self._ptr[0].res.linear.sizeInBytes
    @sizeInBytes.setter
    def sizeInBytes(self, size_t sizeInBytes):
        self._ptr[0].res.linear.sizeInBytes = sizeInBytes

cdef class anon_struct6:
    """
    Attributes
    ----------
    devPtr : CUdeviceptr

    format : CUarray_format

    numChannels : unsigned int

    width : size_t

    height : size_t

    pitchInBytes : size_t


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <cydriver.CUDA_RESOURCE_DESC_st *>_ptr

    def __init__(self, void_ptr _ptr):
        self._devPtr = CUdeviceptr(_ptr=<void_ptr>&self._ptr[0].res.pitch2D.devPtr)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].res.pitch2D
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['devPtr : ' + str(self.devPtr)]
            except ValueError:
                str_list += ['devPtr : <ValueError>']
            try:
                str_list += ['format : ' + str(self.format)]
            except ValueError:
                str_list += ['format : <ValueError>']
            try:
                str_list += ['numChannels : ' + str(self.numChannels)]
            except ValueError:
                str_list += ['numChannels : <ValueError>']
            try:
                str_list += ['width : ' + str(self.width)]
            except ValueError:
                str_list += ['width : <ValueError>']
            try:
                str_list += ['height : ' + str(self.height)]
            except ValueError:
                str_list += ['height : <ValueError>']
            try:
                str_list += ['pitchInBytes : ' + str(self.pitchInBytes)]
            except ValueError:
                str_list += ['pitchInBytes : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def devPtr(self):
        return self._devPtr
    @devPtr.setter
    def devPtr(self, devPtr):
        cdef cydriver.CUdeviceptr cydevPtr
        if devPtr is None:
            cydevPtr = <cydriver.CUdeviceptr><void_ptr>0
        elif isinstance(devPtr, (CUdeviceptr)):
            pdevPtr = int(devPtr)
            cydevPtr = <cydriver.CUdeviceptr><void_ptr>pdevPtr
        else:
            pdevPtr = int(CUdeviceptr(devPtr))
            cydevPtr = <cydriver.CUdeviceptr><void_ptr>pdevPtr
        self._devPtr._ptr[0] = cydevPtr

    @property
    def format(self):
        return CUarray_format(self._ptr[0].res.pitch2D.format)
    @format.setter
    def format(self, format not None : CUarray_format):
        self._ptr[0].res.pitch2D.format = format.value
    @property
    def numChannels(self):
        return self._ptr[0].res.pitch2D.numChannels
    @numChannels.setter
    def numChannels(self, unsigned int numChannels):
        self._ptr[0].res.pitch2D.numChannels = numChannels
    @property
    def width(self):
        return self._ptr[0].res.pitch2D.width
    @width.setter
    def width(self, size_t width):
        self._ptr[0].res.pitch2D.width = width
    @property
    def height(self):
        return self._ptr[0].res.pitch2D.height
    @height.setter
    def height(self, size_t height):
        self._ptr[0].res.pitch2D.height = height
    @property
    def pitchInBytes(self):
        return self._ptr[0].res.pitch2D.pitchInBytes
    @pitchInBytes.setter
    def pitchInBytes(self, size_t pitchInBytes):
        self._ptr[0].res.pitch2D.pitchInBytes = pitchInBytes

cdef class anon_struct7:
    """
    Attributes
    ----------
    reserved : List[int]


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <cydriver.CUDA_RESOURCE_DESC_st *>_ptr

    def __init__(self, void_ptr _ptr):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].res.reserved
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def reserved(self):
        return self._ptr[0].res.reserved.reserved
    @reserved.setter
    def reserved(self, reserved):
        self._ptr[0].res.reserved.reserved = reserved

cdef class anon_union3:
    """
    Attributes
    ----------
    array : anon_struct3

    mipmap : anon_struct4

    linear : anon_struct5

    pitch2D : anon_struct6

    reserved : anon_struct7


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <cydriver.CUDA_RESOURCE_DESC_st *>_ptr

    def __init__(self, void_ptr _ptr):
        self._array = anon_struct3(_ptr=<void_ptr>self._ptr)
        self._mipmap = anon_struct4(_ptr=<void_ptr>self._ptr)
        self._linear = anon_struct5(_ptr=<void_ptr>self._ptr)
        self._pitch2D = anon_struct6(_ptr=<void_ptr>self._ptr)
        self._reserved = anon_struct7(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].res
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['array :\n' + '\n'.join(['    ' + line for line in str(self.array).splitlines()])]
            except ValueError:
                str_list += ['array : <ValueError>']
            try:
                str_list += ['mipmap :\n' + '\n'.join(['    ' + line for line in str(self.mipmap).splitlines()])]
            except ValueError:
                str_list += ['mipmap : <ValueError>']
            try:
                str_list += ['linear :\n' + '\n'.join(['    ' + line for line in str(self.linear).splitlines()])]
            except ValueError:
                str_list += ['linear : <ValueError>']
            try:
                str_list += ['pitch2D :\n' + '\n'.join(['    ' + line for line in str(self.pitch2D).splitlines()])]
            except ValueError:
                str_list += ['pitch2D : <ValueError>']
            try:
                str_list += ['reserved :\n' + '\n'.join(['    ' + line for line in str(self.reserved).splitlines()])]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def array(self):
        return self._array
    @array.setter
    def array(self, array not None : anon_struct3):
        string.memcpy(&self._ptr[0].res.array, <cydriver.anon_struct3*><void_ptr>array.getPtr(), sizeof(self._ptr[0].res.array))
    @property
    def mipmap(self):
        return self._mipmap
    @mipmap.setter
    def mipmap(self, mipmap not None : anon_struct4):
        string.memcpy(&self._ptr[0].res.mipmap, <cydriver.anon_struct4*><void_ptr>mipmap.getPtr(), sizeof(self._ptr[0].res.mipmap))
    @property
    def linear(self):
        return self._linear
    @linear.setter
    def linear(self, linear not None : anon_struct5):
        string.memcpy(&self._ptr[0].res.linear, <cydriver.anon_struct5*><void_ptr>linear.getPtr(), sizeof(self._ptr[0].res.linear))
    @property
    def pitch2D(self):
        return self._pitch2D
    @pitch2D.setter
    def pitch2D(self, pitch2D not None : anon_struct6):
        string.memcpy(&self._ptr[0].res.pitch2D, <cydriver.anon_struct6*><void_ptr>pitch2D.getPtr(), sizeof(self._ptr[0].res.pitch2D))
    @property
    def reserved(self):
        return self._reserved
    @reserved.setter
    def reserved(self, reserved not None : anon_struct7):
        string.memcpy(&self._ptr[0].res.reserved, <cydriver.anon_struct7*><void_ptr>reserved.getPtr(), sizeof(self._ptr[0].res.reserved))

cdef class CUDA_RESOURCE_DESC_st:
    """
    CUDA Resource descriptor

    Attributes
    ----------
    resType : CUresourcetype
        Resource type
    res : anon_union3

    flags : unsigned int
        Flags (must be zero)

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._val_ptr = <cydriver.CUDA_RESOURCE_DESC_st *>calloc(1, sizeof(cydriver.CUDA_RESOURCE_DESC_st))
            self._ptr = self._val_ptr
        else:
            self._ptr = <cydriver.CUDA_RESOURCE_DESC_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._res = anon_union3(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        if self._val_ptr is not NULL:
            free(self._val_ptr)
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['resType : ' + str(self.resType)]
            except ValueError:
                str_list += ['resType : <ValueError>']
            try:
                str_list += ['res :\n' + '\n'.join(['    ' + line for line in str(self.res).splitlines()])]
            except ValueError:
                str_list += ['res : <ValueError>']
            try:
                str_list += ['flags : ' + str(self.flags)]
            except ValueError:
                str_list += ['flags : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def resType(self):
        return CUresourcetype(self._ptr[0].resType)
    @resType.setter
    def resType(self, resType not None : CUresourcetype):
        self._ptr[0].resType = resType.value
    @property
    def res(self):
        return self._res
    @res.setter
    def res(self, res not None : anon_union3):
        string.memcpy(&self._ptr[0].res, <cydriver.anon_union3*><void_ptr>res.getPtr(), sizeof(self._ptr[0].res))
    @property
    def flags(self):
        return self._ptr[0].flags
    @flags.setter
    def flags(self, unsigned int flags):
        self._ptr[0].flags = flags

cdef class CUDA_TEXTURE_DESC_st:
    """
    Texture descriptor

    Attributes
    ----------
    addressMode : List[CUaddress_mode]
        Address modes
    filterMode : CUfilter_mode
        Filter mode
    flags : unsigned int
        Flags
    maxAnisotropy : unsigned int
        Maximum anisotropy ratio
    mipmapFilterMode : CUfilter_mode
        Mipmap filter mode
    mipmapLevelBias : float
        Mipmap level bias
    minMipmapLevelClamp : float
        Mipmap minimum level clamp
    maxMipmapLevelClamp : float
        Mipmap maximum level clamp
    borderColor : List[float]
        Border Color
    reserved : List[int]


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUDA_TEXTURE_DESC_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['addressMode : ' + str(self.addressMode)]
            except ValueError:
                str_list += ['addressMode : <ValueError>']
            try:
                str_list += ['filterMode : ' + str(self.filterMode)]
            except ValueError:
                str_list += ['filterMode : <ValueError>']
            try:
                str_list += ['flags : ' + str(self.flags)]
            except ValueError:
                str_list += ['flags : <ValueError>']
            try:
                str_list += ['maxAnisotropy : ' + str(self.maxAnisotropy)]
            except ValueError:
                str_list += ['maxAnisotropy : <ValueError>']
            try:
                str_list += ['mipmapFilterMode : ' + str(self.mipmapFilterMode)]
            except ValueError:
                str_list += ['mipmapFilterMode : <ValueError>']
            try:
                str_list += ['mipmapLevelBias : ' + str(self.mipmapLevelBias)]
            except ValueError:
                str_list += ['mipmapLevelBias : <ValueError>']
            try:
                str_list += ['minMipmapLevelClamp : ' + str(self.minMipmapLevelClamp)]
            except ValueError:
                str_list += ['minMipmapLevelClamp : <ValueError>']
            try:
                str_list += ['maxMipmapLevelClamp : ' + str(self.maxMipmapLevelClamp)]
            except ValueError:
                str_list += ['maxMipmapLevelClamp : <ValueError>']
            try:
                str_list += ['borderColor : ' + str(self.borderColor)]
            except ValueError:
                str_list += ['borderColor : <ValueError>']
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def addressMode(self):
        return [CUaddress_mode(_x) for _x in list(self._ptr[0].addressMode)]
    @addressMode.setter
    def addressMode(self, addressMode):
        self._ptr[0].addressMode = [_x.value for _x in addressMode]
    @property
    def filterMode(self):
        return CUfilter_mode(self._ptr[0].filterMode)
    @filterMode.setter
    def filterMode(self, filterMode not None : CUfilter_mode):
        self._ptr[0].filterMode = filterMode.value
    @property
    def flags(self):
        return self._ptr[0].flags
    @flags.setter
    def flags(self, unsigned int flags):
        self._ptr[0].flags = flags
    @property
    def maxAnisotropy(self):
        return self._ptr[0].maxAnisotropy
    @maxAnisotropy.setter
    def maxAnisotropy(self, unsigned int maxAnisotropy):
        self._ptr[0].maxAnisotropy = maxAnisotropy
    @property
    def mipmapFilterMode(self):
        return CUfilter_mode(self._ptr[0].mipmapFilterMode)
    @mipmapFilterMode.setter
    def mipmapFilterMode(self, mipmapFilterMode not None : CUfilter_mode):
        self._ptr[0].mipmapFilterMode = mipmapFilterMode.value
    @property
    def mipmapLevelBias(self):
        return self._ptr[0].mipmapLevelBias
    @mipmapLevelBias.setter
    def mipmapLevelBias(self, float mipmapLevelBias):
        self._ptr[0].mipmapLevelBias = mipmapLevelBias
    @property
    def minMipmapLevelClamp(self):
        return self._ptr[0].minMipmapLevelClamp
    @minMipmapLevelClamp.setter
    def minMipmapLevelClamp(self, float minMipmapLevelClamp):
        self._ptr[0].minMipmapLevelClamp = minMipmapLevelClamp
    @property
    def maxMipmapLevelClamp(self):
        return self._ptr[0].maxMipmapLevelClamp
    @maxMipmapLevelClamp.setter
    def maxMipmapLevelClamp(self, float maxMipmapLevelClamp):
        self._ptr[0].maxMipmapLevelClamp = maxMipmapLevelClamp
    @property
    def borderColor(self):
        return self._ptr[0].borderColor
    @borderColor.setter
    def borderColor(self, borderColor):
        self._ptr[0].borderColor = borderColor
    @property
    def reserved(self):
        return self._ptr[0].reserved
    @reserved.setter
    def reserved(self, reserved):
        self._ptr[0].reserved = reserved

cdef class CUDA_RESOURCE_VIEW_DESC_st:
    """
    Resource view descriptor

    Attributes
    ----------
    format : CUresourceViewFormat
        Resource view format
    width : size_t
        Width of the resource view
    height : size_t
        Height of the resource view
    depth : size_t
        Depth of the resource view
    firstMipmapLevel : unsigned int
        First defined mipmap level
    lastMipmapLevel : unsigned int
        Last defined mipmap level
    firstLayer : unsigned int
        First layer index
    lastLayer : unsigned int
        Last layer index
    reserved : List[unsigned int]


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUDA_RESOURCE_VIEW_DESC_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['format : ' + str(self.format)]
            except ValueError:
                str_list += ['format : <ValueError>']
            try:
                str_list += ['width : ' + str(self.width)]
            except ValueError:
                str_list += ['width : <ValueError>']
            try:
                str_list += ['height : ' + str(self.height)]
            except ValueError:
                str_list += ['height : <ValueError>']
            try:
                str_list += ['depth : ' + str(self.depth)]
            except ValueError:
                str_list += ['depth : <ValueError>']
            try:
                str_list += ['firstMipmapLevel : ' + str(self.firstMipmapLevel)]
            except ValueError:
                str_list += ['firstMipmapLevel : <ValueError>']
            try:
                str_list += ['lastMipmapLevel : ' + str(self.lastMipmapLevel)]
            except ValueError:
                str_list += ['lastMipmapLevel : <ValueError>']
            try:
                str_list += ['firstLayer : ' + str(self.firstLayer)]
            except ValueError:
                str_list += ['firstLayer : <ValueError>']
            try:
                str_list += ['lastLayer : ' + str(self.lastLayer)]
            except ValueError:
                str_list += ['lastLayer : <ValueError>']
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def format(self):
        return CUresourceViewFormat(self._ptr[0].format)
    @format.setter
    def format(self, format not None : CUresourceViewFormat):
        self._ptr[0].format = format.value
    @property
    def width(self):
        return self._ptr[0].width
    @width.setter
    def width(self, size_t width):
        self._ptr[0].width = width
    @property
    def height(self):
        return self._ptr[0].height
    @height.setter
    def height(self, size_t height):
        self._ptr[0].height = height
    @property
    def depth(self):
        return self._ptr[0].depth
    @depth.setter
    def depth(self, size_t depth):
        self._ptr[0].depth = depth
    @property
    def firstMipmapLevel(self):
        return self._ptr[0].firstMipmapLevel
    @firstMipmapLevel.setter
    def firstMipmapLevel(self, unsigned int firstMipmapLevel):
        self._ptr[0].firstMipmapLevel = firstMipmapLevel
    @property
    def lastMipmapLevel(self):
        return self._ptr[0].lastMipmapLevel
    @lastMipmapLevel.setter
    def lastMipmapLevel(self, unsigned int lastMipmapLevel):
        self._ptr[0].lastMipmapLevel = lastMipmapLevel
    @property
    def firstLayer(self):
        return self._ptr[0].firstLayer
    @firstLayer.setter
    def firstLayer(self, unsigned int firstLayer):
        self._ptr[0].firstLayer = firstLayer
    @property
    def lastLayer(self):
        return self._ptr[0].lastLayer
    @lastLayer.setter
    def lastLayer(self, unsigned int lastLayer):
        self._ptr[0].lastLayer = lastLayer
    @property
    def reserved(self):
        return self._ptr[0].reserved
    @reserved.setter
    def reserved(self, reserved):
        self._ptr[0].reserved = reserved

cdef class CUDA_POINTER_ATTRIBUTE_P2P_TOKENS_st:
    """
    GPU Direct v3 tokens

    Attributes
    ----------
    p2pToken : unsigned long long

    vaSpaceToken : unsigned int


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUDA_POINTER_ATTRIBUTE_P2P_TOKENS_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['p2pToken : ' + str(self.p2pToken)]
            except ValueError:
                str_list += ['p2pToken : <ValueError>']
            try:
                str_list += ['vaSpaceToken : ' + str(self.vaSpaceToken)]
            except ValueError:
                str_list += ['vaSpaceToken : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def p2pToken(self):
        return self._ptr[0].p2pToken
    @p2pToken.setter
    def p2pToken(self, unsigned long long p2pToken):
        self._ptr[0].p2pToken = p2pToken
    @property
    def vaSpaceToken(self):
        return self._ptr[0].vaSpaceToken
    @vaSpaceToken.setter
    def vaSpaceToken(self, unsigned int vaSpaceToken):
        self._ptr[0].vaSpaceToken = vaSpaceToken

cdef class CUDA_LAUNCH_PARAMS_st:
    """
    Kernel launch parameters

    Attributes
    ----------
    function : CUfunction
        Kernel to launch
    gridDimX : unsigned int
        Width of grid in blocks
    gridDimY : unsigned int
        Height of grid in blocks
    gridDimZ : unsigned int
        Depth of grid in blocks
    blockDimX : unsigned int
        X dimension of each thread block
    blockDimY : unsigned int
        Y dimension of each thread block
    blockDimZ : unsigned int
        Z dimension of each thread block
    sharedMemBytes : unsigned int
        Dynamic shared-memory size per thread block in bytes
    hStream : CUstream
        Stream identifier
    kernelParams : Any
        Array of pointers to kernel parameters

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUDA_LAUNCH_PARAMS_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._function = CUfunction(_ptr=<void_ptr>&self._ptr[0].function)
        self._hStream = CUstream(_ptr=<void_ptr>&self._ptr[0].hStream)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['function : ' + str(self.function)]
            except ValueError:
                str_list += ['function : <ValueError>']
            try:
                str_list += ['gridDimX : ' + str(self.gridDimX)]
            except ValueError:
                str_list += ['gridDimX : <ValueError>']
            try:
                str_list += ['gridDimY : ' + str(self.gridDimY)]
            except ValueError:
                str_list += ['gridDimY : <ValueError>']
            try:
                str_list += ['gridDimZ : ' + str(self.gridDimZ)]
            except ValueError:
                str_list += ['gridDimZ : <ValueError>']
            try:
                str_list += ['blockDimX : ' + str(self.blockDimX)]
            except ValueError:
                str_list += ['blockDimX : <ValueError>']
            try:
                str_list += ['blockDimY : ' + str(self.blockDimY)]
            except ValueError:
                str_list += ['blockDimY : <ValueError>']
            try:
                str_list += ['blockDimZ : ' + str(self.blockDimZ)]
            except ValueError:
                str_list += ['blockDimZ : <ValueError>']
            try:
                str_list += ['sharedMemBytes : ' + str(self.sharedMemBytes)]
            except ValueError:
                str_list += ['sharedMemBytes : <ValueError>']
            try:
                str_list += ['hStream : ' + str(self.hStream)]
            except ValueError:
                str_list += ['hStream : <ValueError>']
            try:
                str_list += ['kernelParams : ' + str(self.kernelParams)]
            except ValueError:
                str_list += ['kernelParams : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def function(self):
        return self._function
    @function.setter
    def function(self, function):
        cdef cydriver.CUfunction cyfunction
        if function is None:
            cyfunction = <cydriver.CUfunction><void_ptr>0
        elif isinstance(function, (CUfunction,)):
            pfunction = int(function)
            cyfunction = <cydriver.CUfunction><void_ptr>pfunction
        else:
            pfunction = int(CUfunction(function))
            cyfunction = <cydriver.CUfunction><void_ptr>pfunction
        self._function._ptr[0] = cyfunction
    @property
    def gridDimX(self):
        return self._ptr[0].gridDimX
    @gridDimX.setter
    def gridDimX(self, unsigned int gridDimX):
        self._ptr[0].gridDimX = gridDimX
    @property
    def gridDimY(self):
        return self._ptr[0].gridDimY
    @gridDimY.setter
    def gridDimY(self, unsigned int gridDimY):
        self._ptr[0].gridDimY = gridDimY
    @property
    def gridDimZ(self):
        return self._ptr[0].gridDimZ
    @gridDimZ.setter
    def gridDimZ(self, unsigned int gridDimZ):
        self._ptr[0].gridDimZ = gridDimZ
    @property
    def blockDimX(self):
        return self._ptr[0].blockDimX
    @blockDimX.setter
    def blockDimX(self, unsigned int blockDimX):
        self._ptr[0].blockDimX = blockDimX
    @property
    def blockDimY(self):
        return self._ptr[0].blockDimY
    @blockDimY.setter
    def blockDimY(self, unsigned int blockDimY):
        self._ptr[0].blockDimY = blockDimY
    @property
    def blockDimZ(self):
        return self._ptr[0].blockDimZ
    @blockDimZ.setter
    def blockDimZ(self, unsigned int blockDimZ):
        self._ptr[0].blockDimZ = blockDimZ
    @property
    def sharedMemBytes(self):
        return self._ptr[0].sharedMemBytes
    @sharedMemBytes.setter
    def sharedMemBytes(self, unsigned int sharedMemBytes):
        self._ptr[0].sharedMemBytes = sharedMemBytes
    @property
    def hStream(self):
        return self._hStream
    @hStream.setter
    def hStream(self, hStream):
        cdef cydriver.CUstream cyhStream
        if hStream is None:
            cyhStream = <cydriver.CUstream><void_ptr>0
        elif isinstance(hStream, (CUstream,)):
            phStream = int(hStream)
            cyhStream = <cydriver.CUstream><void_ptr>phStream
        else:
            phStream = int(CUstream(hStream))
            cyhStream = <cydriver.CUstream><void_ptr>phStream
        self._hStream._ptr[0] = cyhStream
    @property
    def kernelParams(self):
        return <void_ptr>self._ptr[0].kernelParams
    @kernelParams.setter
    def kernelParams(self, kernelParams):
        self._cykernelParams = utils.HelperKernelParams(kernelParams)
        self._ptr[0].kernelParams = <void**><void_ptr>self._cykernelParams.ckernelParams

cdef class anon_struct8:
    """
    Attributes
    ----------
    handle : Any

    name : Any


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <cydriver.CUDA_EXTERNAL_MEMORY_HANDLE_DESC_st *>_ptr

    def __init__(self, void_ptr _ptr):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].handle.win32
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['handle : ' + hex(self.handle)]
            except ValueError:
                str_list += ['handle : <ValueError>']
            try:
                str_list += ['name : ' + hex(self.name)]
            except ValueError:
                str_list += ['name : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def handle(self):
        return <void_ptr>self._ptr[0].handle.win32.handle
    @handle.setter
    def handle(self, handle):
        _cyhandle = utils.HelperInputVoidPtr(handle)
        self._ptr[0].handle.win32.handle = <void*><void_ptr>_cyhandle.cptr
    @property
    def name(self):
        return <void_ptr>self._ptr[0].handle.win32.name
    @name.setter
    def name(self, name):
        _cyname = utils.HelperInputVoidPtr(name)
        self._ptr[0].handle.win32.name = <void*><void_ptr>_cyname.cptr

cdef class anon_union4:
    """
    Attributes
    ----------
    fd : int

    win32 : anon_struct8

    nvSciBufObject : Any


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <cydriver.CUDA_EXTERNAL_MEMORY_HANDLE_DESC_st *>_ptr

    def __init__(self, void_ptr _ptr):
        self._win32 = anon_struct8(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].handle
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['fd : ' + str(self.fd)]
            except ValueError:
                str_list += ['fd : <ValueError>']
            try:
                str_list += ['win32 :\n' + '\n'.join(['    ' + line for line in str(self.win32).splitlines()])]
            except ValueError:
                str_list += ['win32 : <ValueError>']
            try:
                str_list += ['nvSciBufObject : ' + hex(self.nvSciBufObject)]
            except ValueError:
                str_list += ['nvSciBufObject : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def fd(self):
        return self._ptr[0].handle.fd
    @fd.setter
    def fd(self, int fd):
        self._ptr[0].handle.fd = fd
    @property
    def win32(self):
        return self._win32
    @win32.setter
    def win32(self, win32 not None : anon_struct8):
        string.memcpy(&self._ptr[0].handle.win32, <cydriver.anon_struct8*><void_ptr>win32.getPtr(), sizeof(self._ptr[0].handle.win32))
    @property
    def nvSciBufObject(self):
        return <void_ptr>self._ptr[0].handle.nvSciBufObject
    @nvSciBufObject.setter
    def nvSciBufObject(self, nvSciBufObject):
        _cynvSciBufObject = utils.HelperInputVoidPtr(nvSciBufObject)
        self._ptr[0].handle.nvSciBufObject = <void*><void_ptr>_cynvSciBufObject.cptr

cdef class CUDA_EXTERNAL_MEMORY_HANDLE_DESC_st:
    """
    External memory handle descriptor

    Attributes
    ----------
    type : CUexternalMemoryHandleType
        Type of the handle
    handle : anon_union4

    size : unsigned long long
        Size of the memory allocation
    flags : unsigned int
        Flags must either be zero or CUDA_EXTERNAL_MEMORY_DEDICATED
    reserved : List[unsigned int]


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._val_ptr = <cydriver.CUDA_EXTERNAL_MEMORY_HANDLE_DESC_st *>calloc(1, sizeof(cydriver.CUDA_EXTERNAL_MEMORY_HANDLE_DESC_st))
            self._ptr = self._val_ptr
        else:
            self._ptr = <cydriver.CUDA_EXTERNAL_MEMORY_HANDLE_DESC_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._handle = anon_union4(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        if self._val_ptr is not NULL:
            free(self._val_ptr)
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['type : ' + str(self.type)]
            except ValueError:
                str_list += ['type : <ValueError>']
            try:
                str_list += ['handle :\n' + '\n'.join(['    ' + line for line in str(self.handle).splitlines()])]
            except ValueError:
                str_list += ['handle : <ValueError>']
            try:
                str_list += ['size : ' + str(self.size)]
            except ValueError:
                str_list += ['size : <ValueError>']
            try:
                str_list += ['flags : ' + str(self.flags)]
            except ValueError:
                str_list += ['flags : <ValueError>']
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def type(self):
        return CUexternalMemoryHandleType(self._ptr[0].type)
    @type.setter
    def type(self, type not None : CUexternalMemoryHandleType):
        self._ptr[0].type = type.value
    @property
    def handle(self):
        return self._handle
    @handle.setter
    def handle(self, handle not None : anon_union4):
        string.memcpy(&self._ptr[0].handle, <cydriver.anon_union4*><void_ptr>handle.getPtr(), sizeof(self._ptr[0].handle))
    @property
    def size(self):
        return self._ptr[0].size
    @size.setter
    def size(self, unsigned long long size):
        self._ptr[0].size = size
    @property
    def flags(self):
        return self._ptr[0].flags
    @flags.setter
    def flags(self, unsigned int flags):
        self._ptr[0].flags = flags
    @property
    def reserved(self):
        return self._ptr[0].reserved
    @reserved.setter
    def reserved(self, reserved):
        self._ptr[0].reserved = reserved

cdef class CUDA_EXTERNAL_MEMORY_BUFFER_DESC_st:
    """
    External memory buffer descriptor

    Attributes
    ----------
    offset : unsigned long long
        Offset into the memory object where the buffer's base is
    size : unsigned long long
        Size of the buffer
    flags : unsigned int
        Flags reserved for future use. Must be zero.
    reserved : List[unsigned int]


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUDA_EXTERNAL_MEMORY_BUFFER_DESC_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['offset : ' + str(self.offset)]
            except ValueError:
                str_list += ['offset : <ValueError>']
            try:
                str_list += ['size : ' + str(self.size)]
            except ValueError:
                str_list += ['size : <ValueError>']
            try:
                str_list += ['flags : ' + str(self.flags)]
            except ValueError:
                str_list += ['flags : <ValueError>']
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def offset(self):
        return self._ptr[0].offset
    @offset.setter
    def offset(self, unsigned long long offset):
        self._ptr[0].offset = offset
    @property
    def size(self):
        return self._ptr[0].size
    @size.setter
    def size(self, unsigned long long size):
        self._ptr[0].size = size
    @property
    def flags(self):
        return self._ptr[0].flags
    @flags.setter
    def flags(self, unsigned int flags):
        self._ptr[0].flags = flags
    @property
    def reserved(self):
        return self._ptr[0].reserved
    @reserved.setter
    def reserved(self, reserved):
        self._ptr[0].reserved = reserved

cdef class CUDA_EXTERNAL_MEMORY_MIPMAPPED_ARRAY_DESC_st:
    """
    External memory mipmap descriptor

    Attributes
    ----------
    offset : unsigned long long
        Offset into the memory object where the base level of the mipmap
        chain is.
    arrayDesc : CUDA_ARRAY3D_DESCRIPTOR
        Format, dimension and type of base level of the mipmap chain
    numLevels : unsigned int
        Total number of levels in the mipmap chain
    reserved : List[unsigned int]


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUDA_EXTERNAL_MEMORY_MIPMAPPED_ARRAY_DESC_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._arrayDesc = CUDA_ARRAY3D_DESCRIPTOR(_ptr=<void_ptr>&self._ptr[0].arrayDesc)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['offset : ' + str(self.offset)]
            except ValueError:
                str_list += ['offset : <ValueError>']
            try:
                str_list += ['arrayDesc :\n' + '\n'.join(['    ' + line for line in str(self.arrayDesc).splitlines()])]
            except ValueError:
                str_list += ['arrayDesc : <ValueError>']
            try:
                str_list += ['numLevels : ' + str(self.numLevels)]
            except ValueError:
                str_list += ['numLevels : <ValueError>']
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def offset(self):
        return self._ptr[0].offset
    @offset.setter
    def offset(self, unsigned long long offset):
        self._ptr[0].offset = offset
    @property
    def arrayDesc(self):
        return self._arrayDesc
    @arrayDesc.setter
    def arrayDesc(self, arrayDesc not None : CUDA_ARRAY3D_DESCRIPTOR):
        string.memcpy(&self._ptr[0].arrayDesc, <cydriver.CUDA_ARRAY3D_DESCRIPTOR*><void_ptr>arrayDesc.getPtr(), sizeof(self._ptr[0].arrayDesc))
    @property
    def numLevels(self):
        return self._ptr[0].numLevels
    @numLevels.setter
    def numLevels(self, unsigned int numLevels):
        self._ptr[0].numLevels = numLevels
    @property
    def reserved(self):
        return self._ptr[0].reserved
    @reserved.setter
    def reserved(self, reserved):
        self._ptr[0].reserved = reserved

cdef class anon_struct9:
    """
    Attributes
    ----------
    handle : Any

    name : Any


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <cydriver.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC_st *>_ptr

    def __init__(self, void_ptr _ptr):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].handle.win32
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['handle : ' + hex(self.handle)]
            except ValueError:
                str_list += ['handle : <ValueError>']
            try:
                str_list += ['name : ' + hex(self.name)]
            except ValueError:
                str_list += ['name : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def handle(self):
        return <void_ptr>self._ptr[0].handle.win32.handle
    @handle.setter
    def handle(self, handle):
        _cyhandle = utils.HelperInputVoidPtr(handle)
        self._ptr[0].handle.win32.handle = <void*><void_ptr>_cyhandle.cptr
    @property
    def name(self):
        return <void_ptr>self._ptr[0].handle.win32.name
    @name.setter
    def name(self, name):
        _cyname = utils.HelperInputVoidPtr(name)
        self._ptr[0].handle.win32.name = <void*><void_ptr>_cyname.cptr

cdef class anon_union5:
    """
    Attributes
    ----------
    fd : int

    win32 : anon_struct9

    nvSciSyncObj : Any


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <cydriver.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC_st *>_ptr

    def __init__(self, void_ptr _ptr):
        self._win32 = anon_struct9(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].handle
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['fd : ' + str(self.fd)]
            except ValueError:
                str_list += ['fd : <ValueError>']
            try:
                str_list += ['win32 :\n' + '\n'.join(['    ' + line for line in str(self.win32).splitlines()])]
            except ValueError:
                str_list += ['win32 : <ValueError>']
            try:
                str_list += ['nvSciSyncObj : ' + hex(self.nvSciSyncObj)]
            except ValueError:
                str_list += ['nvSciSyncObj : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def fd(self):
        return self._ptr[0].handle.fd
    @fd.setter
    def fd(self, int fd):
        self._ptr[0].handle.fd = fd
    @property
    def win32(self):
        return self._win32
    @win32.setter
    def win32(self, win32 not None : anon_struct9):
        string.memcpy(&self._ptr[0].handle.win32, <cydriver.anon_struct9*><void_ptr>win32.getPtr(), sizeof(self._ptr[0].handle.win32))
    @property
    def nvSciSyncObj(self):
        return <void_ptr>self._ptr[0].handle.nvSciSyncObj
    @nvSciSyncObj.setter
    def nvSciSyncObj(self, nvSciSyncObj):
        _cynvSciSyncObj = utils.HelperInputVoidPtr(nvSciSyncObj)
        self._ptr[0].handle.nvSciSyncObj = <void*><void_ptr>_cynvSciSyncObj.cptr

cdef class CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC_st:
    """
    External semaphore handle descriptor

    Attributes
    ----------
    type : CUexternalSemaphoreHandleType
        Type of the handle
    handle : anon_union5

    flags : unsigned int
        Flags reserved for the future. Must be zero.
    reserved : List[unsigned int]


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._val_ptr = <cydriver.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC_st *>calloc(1, sizeof(cydriver.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC_st))
            self._ptr = self._val_ptr
        else:
            self._ptr = <cydriver.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._handle = anon_union5(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        if self._val_ptr is not NULL:
            free(self._val_ptr)
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['type : ' + str(self.type)]
            except ValueError:
                str_list += ['type : <ValueError>']
            try:
                str_list += ['handle :\n' + '\n'.join(['    ' + line for line in str(self.handle).splitlines()])]
            except ValueError:
                str_list += ['handle : <ValueError>']
            try:
                str_list += ['flags : ' + str(self.flags)]
            except ValueError:
                str_list += ['flags : <ValueError>']
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def type(self):
        return CUexternalSemaphoreHandleType(self._ptr[0].type)
    @type.setter
    def type(self, type not None : CUexternalSemaphoreHandleType):
        self._ptr[0].type = type.value
    @property
    def handle(self):
        return self._handle
    @handle.setter
    def handle(self, handle not None : anon_union5):
        string.memcpy(&self._ptr[0].handle, <cydriver.anon_union5*><void_ptr>handle.getPtr(), sizeof(self._ptr[0].handle))
    @property
    def flags(self):
        return self._ptr[0].flags
    @flags.setter
    def flags(self, unsigned int flags):
        self._ptr[0].flags = flags
    @property
    def reserved(self):
        return self._ptr[0].reserved
    @reserved.setter
    def reserved(self, reserved):
        self._ptr[0].reserved = reserved

cdef class anon_struct10:
    """
    Attributes
    ----------
    value : unsigned long long


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <cydriver.CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS_st *>_ptr

    def __init__(self, void_ptr _ptr):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].params.fence
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['value : ' + str(self.value)]
            except ValueError:
                str_list += ['value : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def value(self):
        return self._ptr[0].params.fence.value
    @value.setter
    def value(self, unsigned long long value):
        self._ptr[0].params.fence.value = value

cdef class anon_union6:
    """
    Attributes
    ----------
    fence : Any

    reserved : unsigned long long


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <cydriver.CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS_st *>_ptr

    def __init__(self, void_ptr _ptr):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].params.nvSciSync
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['fence : ' + hex(self.fence)]
            except ValueError:
                str_list += ['fence : <ValueError>']
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def fence(self):
        return <void_ptr>self._ptr[0].params.nvSciSync.fence
    @fence.setter
    def fence(self, fence):
        _cyfence = utils.HelperInputVoidPtr(fence)
        self._ptr[0].params.nvSciSync.fence = <void*><void_ptr>_cyfence.cptr
    @property
    def reserved(self):
        return self._ptr[0].params.nvSciSync.reserved
    @reserved.setter
    def reserved(self, unsigned long long reserved):
        self._ptr[0].params.nvSciSync.reserved = reserved

cdef class anon_struct11:
    """
    Attributes
    ----------
    key : unsigned long long


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <cydriver.CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS_st *>_ptr

    def __init__(self, void_ptr _ptr):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].params.keyedMutex
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['key : ' + str(self.key)]
            except ValueError:
                str_list += ['key : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def key(self):
        return self._ptr[0].params.keyedMutex.key
    @key.setter
    def key(self, unsigned long long key):
        self._ptr[0].params.keyedMutex.key = key

cdef class anon_struct12:
    """
    Attributes
    ----------
    fence : anon_struct10

    nvSciSync : anon_union6

    keyedMutex : anon_struct11

    reserved : List[unsigned int]


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <cydriver.CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS_st *>_ptr

    def __init__(self, void_ptr _ptr):
        self._fence = anon_struct10(_ptr=<void_ptr>self._ptr)
        self._nvSciSync = anon_union6(_ptr=<void_ptr>self._ptr)
        self._keyedMutex = anon_struct11(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].params
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['fence :\n' + '\n'.join(['    ' + line for line in str(self.fence).splitlines()])]
            except ValueError:
                str_list += ['fence : <ValueError>']
            try:
                str_list += ['nvSciSync :\n' + '\n'.join(['    ' + line for line in str(self.nvSciSync).splitlines()])]
            except ValueError:
                str_list += ['nvSciSync : <ValueError>']
            try:
                str_list += ['keyedMutex :\n' + '\n'.join(['    ' + line for line in str(self.keyedMutex).splitlines()])]
            except ValueError:
                str_list += ['keyedMutex : <ValueError>']
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def fence(self):
        return self._fence
    @fence.setter
    def fence(self, fence not None : anon_struct10):
        string.memcpy(&self._ptr[0].params.fence, <cydriver.anon_struct10*><void_ptr>fence.getPtr(), sizeof(self._ptr[0].params.fence))
    @property
    def nvSciSync(self):
        return self._nvSciSync
    @nvSciSync.setter
    def nvSciSync(self, nvSciSync not None : anon_union6):
        string.memcpy(&self._ptr[0].params.nvSciSync, <cydriver.anon_union6*><void_ptr>nvSciSync.getPtr(), sizeof(self._ptr[0].params.nvSciSync))
    @property
    def keyedMutex(self):
        return self._keyedMutex
    @keyedMutex.setter
    def keyedMutex(self, keyedMutex not None : anon_struct11):
        string.memcpy(&self._ptr[0].params.keyedMutex, <cydriver.anon_struct11*><void_ptr>keyedMutex.getPtr(), sizeof(self._ptr[0].params.keyedMutex))
    @property
    def reserved(self):
        return self._ptr[0].params.reserved
    @reserved.setter
    def reserved(self, reserved):
        self._ptr[0].params.reserved = reserved

cdef class CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS_st:
    """
    External semaphore signal parameters

    Attributes
    ----------
    params : anon_struct12

    flags : unsigned int
        Only when ::CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS is used to signal
        a CUexternalSemaphore of type
        CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_NVSCISYNC, the valid flag is
        CUDA_EXTERNAL_SEMAPHORE_SIGNAL_SKIP_NVSCIBUF_MEMSYNC which
        indicates that while signaling the CUexternalSemaphore, no memory
        synchronization operations should be performed for any external
        memory object imported as CU_EXTERNAL_MEMORY_HANDLE_TYPE_NVSCIBUF.
        For all other types of CUexternalSemaphore, flags must be zero.
    reserved : List[unsigned int]


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._params = anon_struct12(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['params :\n' + '\n'.join(['    ' + line for line in str(self.params).splitlines()])]
            except ValueError:
                str_list += ['params : <ValueError>']
            try:
                str_list += ['flags : ' + str(self.flags)]
            except ValueError:
                str_list += ['flags : <ValueError>']
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def params(self):
        return self._params
    @params.setter
    def params(self, params not None : anon_struct12):
        string.memcpy(&self._ptr[0].params, <cydriver.anon_struct12*><void_ptr>params.getPtr(), sizeof(self._ptr[0].params))
    @property
    def flags(self):
        return self._ptr[0].flags
    @flags.setter
    def flags(self, unsigned int flags):
        self._ptr[0].flags = flags
    @property
    def reserved(self):
        return self._ptr[0].reserved
    @reserved.setter
    def reserved(self, reserved):
        self._ptr[0].reserved = reserved

cdef class anon_struct13:
    """
    Attributes
    ----------
    value : unsigned long long


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <cydriver.CUDA_EXTERNAL_SEMAPHORE_WAIT_PARAMS_st *>_ptr

    def __init__(self, void_ptr _ptr):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].params.fence
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['value : ' + str(self.value)]
            except ValueError:
                str_list += ['value : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def value(self):
        return self._ptr[0].params.fence.value
    @value.setter
    def value(self, unsigned long long value):
        self._ptr[0].params.fence.value = value

cdef class anon_union7:
    """
    Attributes
    ----------
    fence : Any

    reserved : unsigned long long


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <cydriver.CUDA_EXTERNAL_SEMAPHORE_WAIT_PARAMS_st *>_ptr

    def __init__(self, void_ptr _ptr):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].params.nvSciSync
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['fence : ' + hex(self.fence)]
            except ValueError:
                str_list += ['fence : <ValueError>']
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def fence(self):
        return <void_ptr>self._ptr[0].params.nvSciSync.fence
    @fence.setter
    def fence(self, fence):
        _cyfence = utils.HelperInputVoidPtr(fence)
        self._ptr[0].params.nvSciSync.fence = <void*><void_ptr>_cyfence.cptr
    @property
    def reserved(self):
        return self._ptr[0].params.nvSciSync.reserved
    @reserved.setter
    def reserved(self, unsigned long long reserved):
        self._ptr[0].params.nvSciSync.reserved = reserved

cdef class anon_struct14:
    """
    Attributes
    ----------
    key : unsigned long long

    timeoutMs : unsigned int


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <cydriver.CUDA_EXTERNAL_SEMAPHORE_WAIT_PARAMS_st *>_ptr

    def __init__(self, void_ptr _ptr):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].params.keyedMutex
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['key : ' + str(self.key)]
            except ValueError:
                str_list += ['key : <ValueError>']
            try:
                str_list += ['timeoutMs : ' + str(self.timeoutMs)]
            except ValueError:
                str_list += ['timeoutMs : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def key(self):
        return self._ptr[0].params.keyedMutex.key
    @key.setter
    def key(self, unsigned long long key):
        self._ptr[0].params.keyedMutex.key = key
    @property
    def timeoutMs(self):
        return self._ptr[0].params.keyedMutex.timeoutMs
    @timeoutMs.setter
    def timeoutMs(self, unsigned int timeoutMs):
        self._ptr[0].params.keyedMutex.timeoutMs = timeoutMs

cdef class anon_struct15:
    """
    Attributes
    ----------
    fence : anon_struct13

    nvSciSync : anon_union7

    keyedMutex : anon_struct14

    reserved : List[unsigned int]


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <cydriver.CUDA_EXTERNAL_SEMAPHORE_WAIT_PARAMS_st *>_ptr

    def __init__(self, void_ptr _ptr):
        self._fence = anon_struct13(_ptr=<void_ptr>self._ptr)
        self._nvSciSync = anon_union7(_ptr=<void_ptr>self._ptr)
        self._keyedMutex = anon_struct14(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].params
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['fence :\n' + '\n'.join(['    ' + line for line in str(self.fence).splitlines()])]
            except ValueError:
                str_list += ['fence : <ValueError>']
            try:
                str_list += ['nvSciSync :\n' + '\n'.join(['    ' + line for line in str(self.nvSciSync).splitlines()])]
            except ValueError:
                str_list += ['nvSciSync : <ValueError>']
            try:
                str_list += ['keyedMutex :\n' + '\n'.join(['    ' + line for line in str(self.keyedMutex).splitlines()])]
            except ValueError:
                str_list += ['keyedMutex : <ValueError>']
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def fence(self):
        return self._fence
    @fence.setter
    def fence(self, fence not None : anon_struct13):
        string.memcpy(&self._ptr[0].params.fence, <cydriver.anon_struct13*><void_ptr>fence.getPtr(), sizeof(self._ptr[0].params.fence))
    @property
    def nvSciSync(self):
        return self._nvSciSync
    @nvSciSync.setter
    def nvSciSync(self, nvSciSync not None : anon_union7):
        string.memcpy(&self._ptr[0].params.nvSciSync, <cydriver.anon_union7*><void_ptr>nvSciSync.getPtr(), sizeof(self._ptr[0].params.nvSciSync))
    @property
    def keyedMutex(self):
        return self._keyedMutex
    @keyedMutex.setter
    def keyedMutex(self, keyedMutex not None : anon_struct14):
        string.memcpy(&self._ptr[0].params.keyedMutex, <cydriver.anon_struct14*><void_ptr>keyedMutex.getPtr(), sizeof(self._ptr[0].params.keyedMutex))
    @property
    def reserved(self):
        return self._ptr[0].params.reserved
    @reserved.setter
    def reserved(self, reserved):
        self._ptr[0].params.reserved = reserved

cdef class CUDA_EXTERNAL_SEMAPHORE_WAIT_PARAMS_st:
    """
    External semaphore wait parameters

    Attributes
    ----------
    params : anon_struct15

    flags : unsigned int
        Only when ::CUDA_EXTERNAL_SEMAPHORE_WAIT_PARAMS is used to wait on
        a CUexternalSemaphore of type
        CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_NVSCISYNC, the valid flag is
        CUDA_EXTERNAL_SEMAPHORE_WAIT_SKIP_NVSCIBUF_MEMSYNC which indicates
        that while waiting for the CUexternalSemaphore, no memory
        synchronization operations should be performed for any external
        memory object imported as CU_EXTERNAL_MEMORY_HANDLE_TYPE_NVSCIBUF.
        For all other types of CUexternalSemaphore, flags must be zero.
    reserved : List[unsigned int]


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUDA_EXTERNAL_SEMAPHORE_WAIT_PARAMS_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._params = anon_struct15(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['params :\n' + '\n'.join(['    ' + line for line in str(self.params).splitlines()])]
            except ValueError:
                str_list += ['params : <ValueError>']
            try:
                str_list += ['flags : ' + str(self.flags)]
            except ValueError:
                str_list += ['flags : <ValueError>']
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def params(self):
        return self._params
    @params.setter
    def params(self, params not None : anon_struct15):
        string.memcpy(&self._ptr[0].params, <cydriver.anon_struct15*><void_ptr>params.getPtr(), sizeof(self._ptr[0].params))
    @property
    def flags(self):
        return self._ptr[0].flags
    @flags.setter
    def flags(self, unsigned int flags):
        self._ptr[0].flags = flags
    @property
    def reserved(self):
        return self._ptr[0].reserved
    @reserved.setter
    def reserved(self, reserved):
        self._ptr[0].reserved = reserved

cdef class CUDA_EXT_SEM_SIGNAL_NODE_PARAMS_st:
    """
    Semaphore signal node parameters

    Attributes
    ----------
    extSemArray : CUexternalSemaphore
        Array of external semaphore handles.
    paramsArray : CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS
        Array of external semaphore signal parameters.
    numExtSems : unsigned int
        Number of handles and parameters supplied in extSemArray and
        paramsArray.

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUDA_EXT_SEM_SIGNAL_NODE_PARAMS_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        if self._extSemArray is not NULL:
            free(self._extSemArray)
        if self._paramsArray is not NULL:
            free(self._paramsArray)
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['extSemArray : ' + str(self.extSemArray)]
            except ValueError:
                str_list += ['extSemArray : <ValueError>']
            try:
                str_list += ['paramsArray : ' + str(self.paramsArray)]
            except ValueError:
                str_list += ['paramsArray : <ValueError>']
            try:
                str_list += ['numExtSems : ' + str(self.numExtSems)]
            except ValueError:
                str_list += ['numExtSems : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def extSemArray(self):
        arrs = [<void_ptr>self._ptr[0].extSemArray + x*sizeof(cydriver.CUexternalSemaphore) for x in range(self._extSemArray_length)]
        return [CUexternalSemaphore(_ptr=arr) for arr in arrs]
    @extSemArray.setter
    def extSemArray(self, val):
        if len(val) == 0:
            free(self._extSemArray)
            self._extSemArray_length = 0
            self._ptr[0].extSemArray = NULL
        else:
            if self._extSemArray_length != <size_t>len(val):
                free(self._extSemArray)
                self._extSemArray = <cydriver.CUexternalSemaphore*> calloc(len(val), sizeof(cydriver.CUexternalSemaphore))
                if self._extSemArray is NULL:
                    raise MemoryError('Failed to allocate length x size memory: ' + str(len(val)) + 'x' + str(sizeof(cydriver.CUexternalSemaphore)))
                self._extSemArray_length = <size_t>len(val)
                self._ptr[0].extSemArray = self._extSemArray
            for idx in range(len(val)):
                self._extSemArray[idx] = (<CUexternalSemaphore>val[idx])._ptr[0]

    @property
    def paramsArray(self):
        arrs = [<void_ptr>self._ptr[0].paramsArray + x*sizeof(cydriver.CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS) for x in range(self._paramsArray_length)]
        return [CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS(_ptr=arr) for arr in arrs]
    @paramsArray.setter
    def paramsArray(self, val):
        if len(val) == 0:
            free(self._paramsArray)
            self._paramsArray_length = 0
            self._ptr[0].paramsArray = NULL
        else:
            if self._paramsArray_length != <size_t>len(val):
                free(self._paramsArray)
                self._paramsArray = <cydriver.CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS*> calloc(len(val), sizeof(cydriver.CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS))
                if self._paramsArray is NULL:
                    raise MemoryError('Failed to allocate length x size memory: ' + str(len(val)) + 'x' + str(sizeof(cydriver.CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS)))
                self._paramsArray_length = <size_t>len(val)
                self._ptr[0].paramsArray = self._paramsArray
            for idx in range(len(val)):
                string.memcpy(&self._paramsArray[idx], (<CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS>val[idx])._ptr, sizeof(cydriver.CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS))

    @property
    def numExtSems(self):
        return self._ptr[0].numExtSems
    @numExtSems.setter
    def numExtSems(self, unsigned int numExtSems):
        self._ptr[0].numExtSems = numExtSems

cdef class CUDA_EXT_SEM_WAIT_NODE_PARAMS_st:
    """
    Semaphore wait node parameters

    Attributes
    ----------
    extSemArray : CUexternalSemaphore
        Array of external semaphore handles.
    paramsArray : CUDA_EXTERNAL_SEMAPHORE_WAIT_PARAMS
        Array of external semaphore wait parameters.
    numExtSems : unsigned int
        Number of handles and parameters supplied in extSemArray and
        paramsArray.

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUDA_EXT_SEM_WAIT_NODE_PARAMS_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        if self._extSemArray is not NULL:
            free(self._extSemArray)
        if self._paramsArray is not NULL:
            free(self._paramsArray)
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['extSemArray : ' + str(self.extSemArray)]
            except ValueError:
                str_list += ['extSemArray : <ValueError>']
            try:
                str_list += ['paramsArray : ' + str(self.paramsArray)]
            except ValueError:
                str_list += ['paramsArray : <ValueError>']
            try:
                str_list += ['numExtSems : ' + str(self.numExtSems)]
            except ValueError:
                str_list += ['numExtSems : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def extSemArray(self):
        arrs = [<void_ptr>self._ptr[0].extSemArray + x*sizeof(cydriver.CUexternalSemaphore) for x in range(self._extSemArray_length)]
        return [CUexternalSemaphore(_ptr=arr) for arr in arrs]
    @extSemArray.setter
    def extSemArray(self, val):
        if len(val) == 0:
            free(self._extSemArray)
            self._extSemArray_length = 0
            self._ptr[0].extSemArray = NULL
        else:
            if self._extSemArray_length != <size_t>len(val):
                free(self._extSemArray)
                self._extSemArray = <cydriver.CUexternalSemaphore*> calloc(len(val), sizeof(cydriver.CUexternalSemaphore))
                if self._extSemArray is NULL:
                    raise MemoryError('Failed to allocate length x size memory: ' + str(len(val)) + 'x' + str(sizeof(cydriver.CUexternalSemaphore)))
                self._extSemArray_length = <size_t>len(val)
                self._ptr[0].extSemArray = self._extSemArray
            for idx in range(len(val)):
                self._extSemArray[idx] = (<CUexternalSemaphore>val[idx])._ptr[0]

    @property
    def paramsArray(self):
        arrs = [<void_ptr>self._ptr[0].paramsArray + x*sizeof(cydriver.CUDA_EXTERNAL_SEMAPHORE_WAIT_PARAMS) for x in range(self._paramsArray_length)]
        return [CUDA_EXTERNAL_SEMAPHORE_WAIT_PARAMS(_ptr=arr) for arr in arrs]
    @paramsArray.setter
    def paramsArray(self, val):
        if len(val) == 0:
            free(self._paramsArray)
            self._paramsArray_length = 0
            self._ptr[0].paramsArray = NULL
        else:
            if self._paramsArray_length != <size_t>len(val):
                free(self._paramsArray)
                self._paramsArray = <cydriver.CUDA_EXTERNAL_SEMAPHORE_WAIT_PARAMS*> calloc(len(val), sizeof(cydriver.CUDA_EXTERNAL_SEMAPHORE_WAIT_PARAMS))
                if self._paramsArray is NULL:
                    raise MemoryError('Failed to allocate length x size memory: ' + str(len(val)) + 'x' + str(sizeof(cydriver.CUDA_EXTERNAL_SEMAPHORE_WAIT_PARAMS)))
                self._paramsArray_length = <size_t>len(val)
                self._ptr[0].paramsArray = self._paramsArray
            for idx in range(len(val)):
                string.memcpy(&self._paramsArray[idx], (<CUDA_EXTERNAL_SEMAPHORE_WAIT_PARAMS>val[idx])._ptr, sizeof(cydriver.CUDA_EXTERNAL_SEMAPHORE_WAIT_PARAMS))

    @property
    def numExtSems(self):
        return self._ptr[0].numExtSems
    @numExtSems.setter
    def numExtSems(self, unsigned int numExtSems):
        self._ptr[0].numExtSems = numExtSems

cdef class anon_union8:
    """
    Attributes
    ----------
    mipmap : CUmipmappedArray

    array : CUarray


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <cydriver.CUarrayMapInfo_st *>_ptr

    def __init__(self, void_ptr _ptr):
        self._mipmap = CUmipmappedArray(_ptr=<void_ptr>&self._ptr[0].resource.mipmap)
        self._array = CUarray(_ptr=<void_ptr>&self._ptr[0].resource.array)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].resource
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['mipmap : ' + str(self.mipmap)]
            except ValueError:
                str_list += ['mipmap : <ValueError>']
            try:
                str_list += ['array : ' + str(self.array)]
            except ValueError:
                str_list += ['array : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def mipmap(self):
        return self._mipmap
    @mipmap.setter
    def mipmap(self, mipmap):
        cdef cydriver.CUmipmappedArray cymipmap
        if mipmap is None:
            cymipmap = <cydriver.CUmipmappedArray><void_ptr>0
        elif isinstance(mipmap, (CUmipmappedArray,)):
            pmipmap = int(mipmap)
            cymipmap = <cydriver.CUmipmappedArray><void_ptr>pmipmap
        else:
            pmipmap = int(CUmipmappedArray(mipmap))
            cymipmap = <cydriver.CUmipmappedArray><void_ptr>pmipmap
        self._mipmap._ptr[0] = cymipmap
    @property
    def array(self):
        return self._array
    @array.setter
    def array(self, array):
        cdef cydriver.CUarray cyarray
        if array is None:
            cyarray = <cydriver.CUarray><void_ptr>0
        elif isinstance(array, (CUarray,)):
            parray = int(array)
            cyarray = <cydriver.CUarray><void_ptr>parray
        else:
            parray = int(CUarray(array))
            cyarray = <cydriver.CUarray><void_ptr>parray
        self._array._ptr[0] = cyarray

cdef class anon_struct16:
    """
    Attributes
    ----------
    level : unsigned int

    layer : unsigned int

    offsetX : unsigned int

    offsetY : unsigned int

    offsetZ : unsigned int

    extentWidth : unsigned int

    extentHeight : unsigned int

    extentDepth : unsigned int


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <cydriver.CUarrayMapInfo_st *>_ptr

    def __init__(self, void_ptr _ptr):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].subresource.sparseLevel
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['level : ' + str(self.level)]
            except ValueError:
                str_list += ['level : <ValueError>']
            try:
                str_list += ['layer : ' + str(self.layer)]
            except ValueError:
                str_list += ['layer : <ValueError>']
            try:
                str_list += ['offsetX : ' + str(self.offsetX)]
            except ValueError:
                str_list += ['offsetX : <ValueError>']
            try:
                str_list += ['offsetY : ' + str(self.offsetY)]
            except ValueError:
                str_list += ['offsetY : <ValueError>']
            try:
                str_list += ['offsetZ : ' + str(self.offsetZ)]
            except ValueError:
                str_list += ['offsetZ : <ValueError>']
            try:
                str_list += ['extentWidth : ' + str(self.extentWidth)]
            except ValueError:
                str_list += ['extentWidth : <ValueError>']
            try:
                str_list += ['extentHeight : ' + str(self.extentHeight)]
            except ValueError:
                str_list += ['extentHeight : <ValueError>']
            try:
                str_list += ['extentDepth : ' + str(self.extentDepth)]
            except ValueError:
                str_list += ['extentDepth : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def level(self):
        return self._ptr[0].subresource.sparseLevel.level
    @level.setter
    def level(self, unsigned int level):
        self._ptr[0].subresource.sparseLevel.level = level
    @property
    def layer(self):
        return self._ptr[0].subresource.sparseLevel.layer
    @layer.setter
    def layer(self, unsigned int layer):
        self._ptr[0].subresource.sparseLevel.layer = layer
    @property
    def offsetX(self):
        return self._ptr[0].subresource.sparseLevel.offsetX
    @offsetX.setter
    def offsetX(self, unsigned int offsetX):
        self._ptr[0].subresource.sparseLevel.offsetX = offsetX
    @property
    def offsetY(self):
        return self._ptr[0].subresource.sparseLevel.offsetY
    @offsetY.setter
    def offsetY(self, unsigned int offsetY):
        self._ptr[0].subresource.sparseLevel.offsetY = offsetY
    @property
    def offsetZ(self):
        return self._ptr[0].subresource.sparseLevel.offsetZ
    @offsetZ.setter
    def offsetZ(self, unsigned int offsetZ):
        self._ptr[0].subresource.sparseLevel.offsetZ = offsetZ
    @property
    def extentWidth(self):
        return self._ptr[0].subresource.sparseLevel.extentWidth
    @extentWidth.setter
    def extentWidth(self, unsigned int extentWidth):
        self._ptr[0].subresource.sparseLevel.extentWidth = extentWidth
    @property
    def extentHeight(self):
        return self._ptr[0].subresource.sparseLevel.extentHeight
    @extentHeight.setter
    def extentHeight(self, unsigned int extentHeight):
        self._ptr[0].subresource.sparseLevel.extentHeight = extentHeight
    @property
    def extentDepth(self):
        return self._ptr[0].subresource.sparseLevel.extentDepth
    @extentDepth.setter
    def extentDepth(self, unsigned int extentDepth):
        self._ptr[0].subresource.sparseLevel.extentDepth = extentDepth

cdef class anon_struct17:
    """
    Attributes
    ----------
    layer : unsigned int

    offset : unsigned long long

    size : unsigned long long


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <cydriver.CUarrayMapInfo_st *>_ptr

    def __init__(self, void_ptr _ptr):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].subresource.miptail
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['layer : ' + str(self.layer)]
            except ValueError:
                str_list += ['layer : <ValueError>']
            try:
                str_list += ['offset : ' + str(self.offset)]
            except ValueError:
                str_list += ['offset : <ValueError>']
            try:
                str_list += ['size : ' + str(self.size)]
            except ValueError:
                str_list += ['size : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def layer(self):
        return self._ptr[0].subresource.miptail.layer
    @layer.setter
    def layer(self, unsigned int layer):
        self._ptr[0].subresource.miptail.layer = layer
    @property
    def offset(self):
        return self._ptr[0].subresource.miptail.offset
    @offset.setter
    def offset(self, unsigned long long offset):
        self._ptr[0].subresource.miptail.offset = offset
    @property
    def size(self):
        return self._ptr[0].subresource.miptail.size
    @size.setter
    def size(self, unsigned long long size):
        self._ptr[0].subresource.miptail.size = size

cdef class anon_union9:
    """
    Attributes
    ----------
    sparseLevel : anon_struct16

    miptail : anon_struct17


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <cydriver.CUarrayMapInfo_st *>_ptr

    def __init__(self, void_ptr _ptr):
        self._sparseLevel = anon_struct16(_ptr=<void_ptr>self._ptr)
        self._miptail = anon_struct17(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].subresource
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['sparseLevel :\n' + '\n'.join(['    ' + line for line in str(self.sparseLevel).splitlines()])]
            except ValueError:
                str_list += ['sparseLevel : <ValueError>']
            try:
                str_list += ['miptail :\n' + '\n'.join(['    ' + line for line in str(self.miptail).splitlines()])]
            except ValueError:
                str_list += ['miptail : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def sparseLevel(self):
        return self._sparseLevel
    @sparseLevel.setter
    def sparseLevel(self, sparseLevel not None : anon_struct16):
        string.memcpy(&self._ptr[0].subresource.sparseLevel, <cydriver.anon_struct16*><void_ptr>sparseLevel.getPtr(), sizeof(self._ptr[0].subresource.sparseLevel))
    @property
    def miptail(self):
        return self._miptail
    @miptail.setter
    def miptail(self, miptail not None : anon_struct17):
        string.memcpy(&self._ptr[0].subresource.miptail, <cydriver.anon_struct17*><void_ptr>miptail.getPtr(), sizeof(self._ptr[0].subresource.miptail))

cdef class anon_union10:
    """
    Attributes
    ----------
    memHandle : CUmemGenericAllocationHandle


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <cydriver.CUarrayMapInfo_st *>_ptr

    def __init__(self, void_ptr _ptr):
        self._memHandle = CUmemGenericAllocationHandle(_ptr=<void_ptr>&self._ptr[0].memHandle.memHandle)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].memHandle
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['memHandle : ' + str(self.memHandle)]
            except ValueError:
                str_list += ['memHandle : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def memHandle(self):
        return self._memHandle
    @memHandle.setter
    def memHandle(self, memHandle):
        cdef cydriver.CUmemGenericAllocationHandle cymemHandle
        if memHandle is None:
            cymemHandle = <cydriver.CUmemGenericAllocationHandle><void_ptr>0
        elif isinstance(memHandle, (CUmemGenericAllocationHandle)):
            pmemHandle = int(memHandle)
            cymemHandle = <cydriver.CUmemGenericAllocationHandle><void_ptr>pmemHandle
        else:
            pmemHandle = int(CUmemGenericAllocationHandle(memHandle))
            cymemHandle = <cydriver.CUmemGenericAllocationHandle><void_ptr>pmemHandle
        self._memHandle._ptr[0] = cymemHandle

cdef class CUarrayMapInfo_st:
    """
    Specifies the CUDA array or CUDA mipmapped array memory mapping
    information

    Attributes
    ----------
    resourceType : CUresourcetype
        Resource type
    resource : anon_union8

    subresourceType : CUarraySparseSubresourceType
        Sparse subresource type
    subresource : anon_union9

    memOperationType : CUmemOperationType
        Memory operation type
    memHandleType : CUmemHandleType
        Memory handle type
    memHandle : anon_union10

    offset : unsigned long long
        Offset within mip tail  Offset within the memory
    deviceBitMask : unsigned int
        Device ordinal bit mask
    flags : unsigned int
        flags for future use, must be zero now.
    reserved : List[unsigned int]
        Reserved for future use, must be zero now.

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._val_ptr = <cydriver.CUarrayMapInfo_st *>calloc(1, sizeof(cydriver.CUarrayMapInfo_st))
            self._ptr = self._val_ptr
        else:
            self._ptr = <cydriver.CUarrayMapInfo_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._resource = anon_union8(_ptr=<void_ptr>self._ptr)
        self._subresource = anon_union9(_ptr=<void_ptr>self._ptr)
        self._memHandle = anon_union10(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        if self._val_ptr is not NULL:
            free(self._val_ptr)
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['resourceType : ' + str(self.resourceType)]
            except ValueError:
                str_list += ['resourceType : <ValueError>']
            try:
                str_list += ['resource :\n' + '\n'.join(['    ' + line for line in str(self.resource).splitlines()])]
            except ValueError:
                str_list += ['resource : <ValueError>']
            try:
                str_list += ['subresourceType : ' + str(self.subresourceType)]
            except ValueError:
                str_list += ['subresourceType : <ValueError>']
            try:
                str_list += ['subresource :\n' + '\n'.join(['    ' + line for line in str(self.subresource).splitlines()])]
            except ValueError:
                str_list += ['subresource : <ValueError>']
            try:
                str_list += ['memOperationType : ' + str(self.memOperationType)]
            except ValueError:
                str_list += ['memOperationType : <ValueError>']
            try:
                str_list += ['memHandleType : ' + str(self.memHandleType)]
            except ValueError:
                str_list += ['memHandleType : <ValueError>']
            try:
                str_list += ['memHandle :\n' + '\n'.join(['    ' + line for line in str(self.memHandle).splitlines()])]
            except ValueError:
                str_list += ['memHandle : <ValueError>']
            try:
                str_list += ['offset : ' + str(self.offset)]
            except ValueError:
                str_list += ['offset : <ValueError>']
            try:
                str_list += ['deviceBitMask : ' + str(self.deviceBitMask)]
            except ValueError:
                str_list += ['deviceBitMask : <ValueError>']
            try:
                str_list += ['flags : ' + str(self.flags)]
            except ValueError:
                str_list += ['flags : <ValueError>']
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def resourceType(self):
        return CUresourcetype(self._ptr[0].resourceType)
    @resourceType.setter
    def resourceType(self, resourceType not None : CUresourcetype):
        self._ptr[0].resourceType = resourceType.value
    @property
    def resource(self):
        return self._resource
    @resource.setter
    def resource(self, resource not None : anon_union8):
        string.memcpy(&self._ptr[0].resource, <cydriver.anon_union8*><void_ptr>resource.getPtr(), sizeof(self._ptr[0].resource))
    @property
    def subresourceType(self):
        return CUarraySparseSubresourceType(self._ptr[0].subresourceType)
    @subresourceType.setter
    def subresourceType(self, subresourceType not None : CUarraySparseSubresourceType):
        self._ptr[0].subresourceType = subresourceType.value
    @property
    def subresource(self):
        return self._subresource
    @subresource.setter
    def subresource(self, subresource not None : anon_union9):
        string.memcpy(&self._ptr[0].subresource, <cydriver.anon_union9*><void_ptr>subresource.getPtr(), sizeof(self._ptr[0].subresource))
    @property
    def memOperationType(self):
        return CUmemOperationType(self._ptr[0].memOperationType)
    @memOperationType.setter
    def memOperationType(self, memOperationType not None : CUmemOperationType):
        self._ptr[0].memOperationType = memOperationType.value
    @property
    def memHandleType(self):
        return CUmemHandleType(self._ptr[0].memHandleType)
    @memHandleType.setter
    def memHandleType(self, memHandleType not None : CUmemHandleType):
        self._ptr[0].memHandleType = memHandleType.value
    @property
    def memHandle(self):
        return self._memHandle
    @memHandle.setter
    def memHandle(self, memHandle not None : anon_union10):
        string.memcpy(&self._ptr[0].memHandle, <cydriver.anon_union10*><void_ptr>memHandle.getPtr(), sizeof(self._ptr[0].memHandle))
    @property
    def offset(self):
        return self._ptr[0].offset
    @offset.setter
    def offset(self, unsigned long long offset):
        self._ptr[0].offset = offset
    @property
    def deviceBitMask(self):
        return self._ptr[0].deviceBitMask
    @deviceBitMask.setter
    def deviceBitMask(self, unsigned int deviceBitMask):
        self._ptr[0].deviceBitMask = deviceBitMask
    @property
    def flags(self):
        return self._ptr[0].flags
    @flags.setter
    def flags(self, unsigned int flags):
        self._ptr[0].flags = flags
    @property
    def reserved(self):
        return self._ptr[0].reserved
    @reserved.setter
    def reserved(self, reserved):
        self._ptr[0].reserved = reserved

cdef class CUmemLocation_st:
    """
    Specifies a memory location.

    Attributes
    ----------
    type : CUmemLocationType
        Specifies the location type, which modifies the meaning of id.
    id : int
        identifier for a given this location's CUmemLocationType.

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUmemLocation_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['type : ' + str(self.type)]
            except ValueError:
                str_list += ['type : <ValueError>']
            try:
                str_list += ['id : ' + str(self.id)]
            except ValueError:
                str_list += ['id : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def type(self):
        return CUmemLocationType(self._ptr[0].type)
    @type.setter
    def type(self, type not None : CUmemLocationType):
        self._ptr[0].type = type.value
    @property
    def id(self):
        return self._ptr[0].id
    @id.setter
    def id(self, int id):
        self._ptr[0].id = id

cdef class anon_struct18:
    """
    Attributes
    ----------
    compressionType : bytes

    gpuDirectRDMACapable : bytes

    usage : unsigned short

    reserved : bytes


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <cydriver.CUmemAllocationProp_st *>_ptr

    def __init__(self, void_ptr _ptr):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].allocFlags
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['compressionType : ' + str(self.compressionType)]
            except ValueError:
                str_list += ['compressionType : <ValueError>']
            try:
                str_list += ['gpuDirectRDMACapable : ' + str(self.gpuDirectRDMACapable)]
            except ValueError:
                str_list += ['gpuDirectRDMACapable : <ValueError>']
            try:
                str_list += ['usage : ' + str(self.usage)]
            except ValueError:
                str_list += ['usage : <ValueError>']
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def compressionType(self):
        return self._ptr[0].allocFlags.compressionType
    @compressionType.setter
    def compressionType(self, unsigned char compressionType):
        self._ptr[0].allocFlags.compressionType = compressionType
    @property
    def gpuDirectRDMACapable(self):
        return self._ptr[0].allocFlags.gpuDirectRDMACapable
    @gpuDirectRDMACapable.setter
    def gpuDirectRDMACapable(self, unsigned char gpuDirectRDMACapable):
        self._ptr[0].allocFlags.gpuDirectRDMACapable = gpuDirectRDMACapable
    @property
    def usage(self):
        return self._ptr[0].allocFlags.usage
    @usage.setter
    def usage(self, unsigned short usage):
        self._ptr[0].allocFlags.usage = usage
    @property
    def reserved(self):
        return PyBytes_FromStringAndSize(<char*>self._ptr[0].allocFlags.reserved, 4)
    @reserved.setter
    def reserved(self, reserved):
        if len(reserved) != 4:
            raise ValueError("reserved length must be 4, is " + str(len(reserved)))
        for i, b in enumerate(reserved):
            self._ptr[0].allocFlags.reserved[i] = b

cdef class CUmemAllocationProp_st:
    """
    Specifies the allocation properties for a allocation.

    Attributes
    ----------
    type : CUmemAllocationType
        Allocation type
    requestedHandleTypes : CUmemAllocationHandleType
        requested CUmemAllocationHandleType
    location : CUmemLocation
        Location of allocation
    win32HandleMetaData : Any
        Windows-specific POBJECT_ATTRIBUTES required when
        CU_MEM_HANDLE_TYPE_WIN32 is specified. This object atributes
        structure includes security attributes that define the scope of
        which exported allocations may be tranferred to other processes. In
        all other cases, this field is required to be zero.
    allocFlags : anon_struct18


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUmemAllocationProp_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._location = CUmemLocation(_ptr=<void_ptr>&self._ptr[0].location)
        self._allocFlags = anon_struct18(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['type : ' + str(self.type)]
            except ValueError:
                str_list += ['type : <ValueError>']
            try:
                str_list += ['requestedHandleTypes : ' + str(self.requestedHandleTypes)]
            except ValueError:
                str_list += ['requestedHandleTypes : <ValueError>']
            try:
                str_list += ['location :\n' + '\n'.join(['    ' + line for line in str(self.location).splitlines()])]
            except ValueError:
                str_list += ['location : <ValueError>']
            try:
                str_list += ['win32HandleMetaData : ' + hex(self.win32HandleMetaData)]
            except ValueError:
                str_list += ['win32HandleMetaData : <ValueError>']
            try:
                str_list += ['allocFlags :\n' + '\n'.join(['    ' + line for line in str(self.allocFlags).splitlines()])]
            except ValueError:
                str_list += ['allocFlags : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def type(self):
        return CUmemAllocationType(self._ptr[0].type)
    @type.setter
    def type(self, type not None : CUmemAllocationType):
        self._ptr[0].type = type.value
    @property
    def requestedHandleTypes(self):
        return CUmemAllocationHandleType(self._ptr[0].requestedHandleTypes)
    @requestedHandleTypes.setter
    def requestedHandleTypes(self, requestedHandleTypes not None : CUmemAllocationHandleType):
        self._ptr[0].requestedHandleTypes = requestedHandleTypes.value
    @property
    def location(self):
        return self._location
    @location.setter
    def location(self, location not None : CUmemLocation):
        string.memcpy(&self._ptr[0].location, <cydriver.CUmemLocation*><void_ptr>location.getPtr(), sizeof(self._ptr[0].location))
    @property
    def win32HandleMetaData(self):
        return <void_ptr>self._ptr[0].win32HandleMetaData
    @win32HandleMetaData.setter
    def win32HandleMetaData(self, win32HandleMetaData):
        _cywin32HandleMetaData = utils.HelperInputVoidPtr(win32HandleMetaData)
        self._ptr[0].win32HandleMetaData = <void*><void_ptr>_cywin32HandleMetaData.cptr
    @property
    def allocFlags(self):
        return self._allocFlags
    @allocFlags.setter
    def allocFlags(self, allocFlags not None : anon_struct18):
        string.memcpy(&self._ptr[0].allocFlags, <cydriver.anon_struct18*><void_ptr>allocFlags.getPtr(), sizeof(self._ptr[0].allocFlags))

cdef class CUmemAccessDesc_st:
    """
    Memory access descriptor

    Attributes
    ----------
    location : CUmemLocation
        Location on which the request is to change it's accessibility
    flags : CUmemAccess_flags
        ::CUmemProt accessibility flags to set on the request

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUmemAccessDesc_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._location = CUmemLocation(_ptr=<void_ptr>&self._ptr[0].location)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['location :\n' + '\n'.join(['    ' + line for line in str(self.location).splitlines()])]
            except ValueError:
                str_list += ['location : <ValueError>']
            try:
                str_list += ['flags : ' + str(self.flags)]
            except ValueError:
                str_list += ['flags : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def location(self):
        return self._location
    @location.setter
    def location(self, location not None : CUmemLocation):
        string.memcpy(&self._ptr[0].location, <cydriver.CUmemLocation*><void_ptr>location.getPtr(), sizeof(self._ptr[0].location))
    @property
    def flags(self):
        return CUmemAccess_flags(self._ptr[0].flags)
    @flags.setter
    def flags(self, flags not None : CUmemAccess_flags):
        self._ptr[0].flags = flags.value

cdef class CUmemPoolProps_st:
    """
    Specifies the properties of allocations made from the pool.

    Attributes
    ----------
    allocType : CUmemAllocationType
        Allocation type. Currently must be specified as
        CU_MEM_ALLOCATION_TYPE_PINNED
    handleTypes : CUmemAllocationHandleType
        Handle types that will be supported by allocations from the pool.
    location : CUmemLocation
        Location where allocations should reside.
    win32SecurityAttributes : Any
        Windows-specific LPSECURITYATTRIBUTES required when
        CU_MEM_HANDLE_TYPE_WIN32 is specified. This security attribute
        defines the scope of which exported allocations may be tranferred
        to other processes. In all other cases, this field is required to
        be zero.
    reserved : bytes
        reserved for future use, must be 0

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUmemPoolProps_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._location = CUmemLocation(_ptr=<void_ptr>&self._ptr[0].location)
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['allocType : ' + str(self.allocType)]
            except ValueError:
                str_list += ['allocType : <ValueError>']
            try:
                str_list += ['handleTypes : ' + str(self.handleTypes)]
            except ValueError:
                str_list += ['handleTypes : <ValueError>']
            try:
                str_list += ['location :\n' + '\n'.join(['    ' + line for line in str(self.location).splitlines()])]
            except ValueError:
                str_list += ['location : <ValueError>']
            try:
                str_list += ['win32SecurityAttributes : ' + hex(self.win32SecurityAttributes)]
            except ValueError:
                str_list += ['win32SecurityAttributes : <ValueError>']
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def allocType(self):
        return CUmemAllocationType(self._ptr[0].allocType)
    @allocType.setter
    def allocType(self, allocType not None : CUmemAllocationType):
        self._ptr[0].allocType = allocType.value
    @property
    def handleTypes(self):
        return CUmemAllocationHandleType(self._ptr[0].handleTypes)
    @handleTypes.setter
    def handleTypes(self, handleTypes not None : CUmemAllocationHandleType):
        self._ptr[0].handleTypes = handleTypes.value
    @property
    def location(self):
        return self._location
    @location.setter
    def location(self, location not None : CUmemLocation):
        string.memcpy(&self._ptr[0].location, <cydriver.CUmemLocation*><void_ptr>location.getPtr(), sizeof(self._ptr[0].location))
    @property
    def win32SecurityAttributes(self):
        return <void_ptr>self._ptr[0].win32SecurityAttributes
    @win32SecurityAttributes.setter
    def win32SecurityAttributes(self, win32SecurityAttributes):
        _cywin32SecurityAttributes = utils.HelperInputVoidPtr(win32SecurityAttributes)
        self._ptr[0].win32SecurityAttributes = <void*><void_ptr>_cywin32SecurityAttributes.cptr
    @property
    def reserved(self):
        return PyBytes_FromStringAndSize(<char*>self._ptr[0].reserved, 64)
    @reserved.setter
    def reserved(self, reserved):
        if len(reserved) != 64:
            raise ValueError("reserved length must be 64, is " + str(len(reserved)))
        for i, b in enumerate(reserved):
            self._ptr[0].reserved[i] = b

cdef class CUmemPoolPtrExportData_st:
    """
    Opaque data for exporting a pool allocation

    Attributes
    ----------
    reserved : bytes


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUmemPoolPtrExportData_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['reserved : ' + str(self.reserved)]
            except ValueError:
                str_list += ['reserved : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def reserved(self):
        return PyBytes_FromStringAndSize(<char*>self._ptr[0].reserved, 64)
    @reserved.setter
    def reserved(self, reserved):
        if len(reserved) != 64:
            raise ValueError("reserved length must be 64, is " + str(len(reserved)))
        for i, b in enumerate(reserved):
            self._ptr[0].reserved[i] = b

cdef class CUDA_MEM_ALLOC_NODE_PARAMS_st:
    """
    Memory allocation node parameters

    Attributes
    ----------
    poolProps : CUmemPoolProps
        in: location where the allocation should reside (specified in
        ::location). ::handleTypes must be CU_MEM_HANDLE_TYPE_NONE. IPC is
        not supported.
    accessDescs : CUmemAccessDesc
        in: array of memory access descriptors. Used to describe peer GPU
        access
    accessDescCount : size_t
        in: number of memory access descriptors. Must not exceed the number
        of GPUs.
    bytesize : size_t
        in: size in bytes of the requested allocation
    dptr : CUdeviceptr
        out: address of the allocation returned by CUDA

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUDA_MEM_ALLOC_NODE_PARAMS_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._poolProps = CUmemPoolProps(_ptr=<void_ptr>&self._ptr[0].poolProps)
        self._dptr = CUdeviceptr(_ptr=<void_ptr>&self._ptr[0].dptr)
    def __dealloc__(self):
        if self._accessDescs is not NULL:
            free(self._accessDescs)
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['poolProps :\n' + '\n'.join(['    ' + line for line in str(self.poolProps).splitlines()])]
            except ValueError:
                str_list += ['poolProps : <ValueError>']
            try:
                str_list += ['accessDescs : ' + str(self.accessDescs)]
            except ValueError:
                str_list += ['accessDescs : <ValueError>']
            try:
                str_list += ['accessDescCount : ' + str(self.accessDescCount)]
            except ValueError:
                str_list += ['accessDescCount : <ValueError>']
            try:
                str_list += ['bytesize : ' + str(self.bytesize)]
            except ValueError:
                str_list += ['bytesize : <ValueError>']
            try:
                str_list += ['dptr : ' + str(self.dptr)]
            except ValueError:
                str_list += ['dptr : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def poolProps(self):
        return self._poolProps
    @poolProps.setter
    def poolProps(self, poolProps not None : CUmemPoolProps):
        string.memcpy(&self._ptr[0].poolProps, <cydriver.CUmemPoolProps*><void_ptr>poolProps.getPtr(), sizeof(self._ptr[0].poolProps))
    @property
    def accessDescs(self):
        arrs = [<void_ptr>self._ptr[0].accessDescs + x*sizeof(cydriver.CUmemAccessDesc) for x in range(self._accessDescs_length)]
        return [CUmemAccessDesc(_ptr=arr) for arr in arrs]
    @accessDescs.setter
    def accessDescs(self, val):
        if len(val) == 0:
            free(self._accessDescs)
            self._accessDescs_length = 0
            self._ptr[0].accessDescs = NULL
        else:
            if self._accessDescs_length != <size_t>len(val):
                free(self._accessDescs)
                self._accessDescs = <cydriver.CUmemAccessDesc*> calloc(len(val), sizeof(cydriver.CUmemAccessDesc))
                if self._accessDescs is NULL:
                    raise MemoryError('Failed to allocate length x size memory: ' + str(len(val)) + 'x' + str(sizeof(cydriver.CUmemAccessDesc)))
                self._accessDescs_length = <size_t>len(val)
                self._ptr[0].accessDescs = self._accessDescs
            for idx in range(len(val)):
                string.memcpy(&self._accessDescs[idx], (<CUmemAccessDesc>val[idx])._ptr, sizeof(cydriver.CUmemAccessDesc))

    @property
    def accessDescCount(self):
        return self._ptr[0].accessDescCount
    @accessDescCount.setter
    def accessDescCount(self, size_t accessDescCount):
        self._ptr[0].accessDescCount = accessDescCount
    @property
    def bytesize(self):
        return self._ptr[0].bytesize
    @bytesize.setter
    def bytesize(self, size_t bytesize):
        self._ptr[0].bytesize = bytesize
    @property
    def dptr(self):
        return self._dptr
    @dptr.setter
    def dptr(self, dptr):
        cdef cydriver.CUdeviceptr cydptr
        if dptr is None:
            cydptr = <cydriver.CUdeviceptr><void_ptr>0
        elif isinstance(dptr, (CUdeviceptr)):
            pdptr = int(dptr)
            cydptr = <cydriver.CUdeviceptr><void_ptr>pdptr
        else:
            pdptr = int(CUdeviceptr(dptr))
            cydptr = <cydriver.CUdeviceptr><void_ptr>pdptr
        self._dptr._ptr[0] = cydptr

cdef class anon_union11:
    """
    Attributes
    ----------
    pArray : List[CUarray]

    pPitch : List[Any]


    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr):
        self._ptr = <cydriver.CUeglFrame_st *>_ptr

    def __init__(self, void_ptr _ptr):
        pass
    def __dealloc__(self):
        pass
    def getPtr(self):
        return <void_ptr>&self._ptr[0].frame
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['pArray : ' + str(self.pArray)]
            except ValueError:
                str_list += ['pArray : <ValueError>']
            try:
                str_list += ['pPitch : ' + hex(self.pPitch)]
            except ValueError:
                str_list += ['pPitch : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def pArray(self):
        return [CUarray(init_value=<void_ptr>_pArray) for _pArray in self._ptr[0].frame.pArray]
    @pArray.setter
    def pArray(self, pArray : List[CUarray]):
        if len(pArray) != 3:
            raise IndexError('not enough values found during array assignment, expected 3, got', len(pArray))
        pArray = [int(_pArray) for _pArray in pArray]
        for _idx, _pArray in enumerate(pArray):
            self._ptr[0].frame.pArray[_idx] = <cydriver.CUarray><void_ptr>_pArray

    @property
    def pPitch(self):
        return [<void_ptr>_pPitch for _pPitch in self._ptr[0].frame.pPitch]
    @pPitch.setter
    def pPitch(self, pPitch : List[int]):
        if len(pPitch) != 3:
            raise IndexError('not enough values found during array assignment, expected 3, got', len(pPitch))
        pPitch = [<void_ptr>_pPitch for _pPitch in pPitch]
        for _idx, _pPitch in enumerate(pPitch):
            self._ptr[0].frame.pPitch[_idx] = <void*><void_ptr>_pPitch

cdef class CUeglFrame_st:
    """
    CUDA EGLFrame structure Descriptor - structure defining one frame
    of EGL.  Each frame may contain one or more planes depending on
    whether the surface * is Multiplanar or not.

    Attributes
    ----------
    frame : anon_union11

    width : unsigned int
        Width of first plane
    height : unsigned int
        Height of first plane
    depth : unsigned int
        Depth of first plane
    pitch : unsigned int
        Pitch of first plane
    planeCount : unsigned int
        Number of planes
    numChannels : unsigned int
        Number of channels for the plane
    frameType : CUeglFrameType
        Array or Pitch
    eglColorFormat : CUeglColorFormat
        CUDA EGL Color Format
    cuFormat : CUarray_format
        CUDA Array Format

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, void_ptr _ptr = 0):
        if _ptr == 0:
            self._val_ptr = <cydriver.CUeglFrame_st *>calloc(1, sizeof(cydriver.CUeglFrame_st))
            self._ptr = self._val_ptr
        else:
            self._ptr = <cydriver.CUeglFrame_st *>_ptr
    def __init__(self, void_ptr _ptr = 0):
        self._frame = anon_union11(_ptr=<void_ptr>self._ptr)
    def __dealloc__(self):
        if self._val_ptr is not NULL:
            free(self._val_ptr)
    def getPtr(self):
        return <void_ptr>self._ptr
    def __repr__(self):
        if self._ptr is not NULL:
            str_list = []
            try:
                str_list += ['frame :\n' + '\n'.join(['    ' + line for line in str(self.frame).splitlines()])]
            except ValueError:
                str_list += ['frame : <ValueError>']
            try:
                str_list += ['width : ' + str(self.width)]
            except ValueError:
                str_list += ['width : <ValueError>']
            try:
                str_list += ['height : ' + str(self.height)]
            except ValueError:
                str_list += ['height : <ValueError>']
            try:
                str_list += ['depth : ' + str(self.depth)]
            except ValueError:
                str_list += ['depth : <ValueError>']
            try:
                str_list += ['pitch : ' + str(self.pitch)]
            except ValueError:
                str_list += ['pitch : <ValueError>']
            try:
                str_list += ['planeCount : ' + str(self.planeCount)]
            except ValueError:
                str_list += ['planeCount : <ValueError>']
            try:
                str_list += ['numChannels : ' + str(self.numChannels)]
            except ValueError:
                str_list += ['numChannels : <ValueError>']
            try:
                str_list += ['frameType : ' + str(self.frameType)]
            except ValueError:
                str_list += ['frameType : <ValueError>']
            try:
                str_list += ['eglColorFormat : ' + str(self.eglColorFormat)]
            except ValueError:
                str_list += ['eglColorFormat : <ValueError>']
            try:
                str_list += ['cuFormat : ' + str(self.cuFormat)]
            except ValueError:
                str_list += ['cuFormat : <ValueError>']
            return '\n'.join(str_list)
        else:
            return ''
    @property
    def frame(self):
        return self._frame
    @frame.setter
    def frame(self, frame not None : anon_union11):
        string.memcpy(&self._ptr[0].frame, <cydriver.anon_union11*><void_ptr>frame.getPtr(), sizeof(self._ptr[0].frame))
    @property
    def width(self):
        return self._ptr[0].width
    @width.setter
    def width(self, unsigned int width):
        self._ptr[0].width = width
    @property
    def height(self):
        return self._ptr[0].height
    @height.setter
    def height(self, unsigned int height):
        self._ptr[0].height = height
    @property
    def depth(self):
        return self._ptr[0].depth
    @depth.setter
    def depth(self, unsigned int depth):
        self._ptr[0].depth = depth
    @property
    def pitch(self):
        return self._ptr[0].pitch
    @pitch.setter
    def pitch(self, unsigned int pitch):
        self._ptr[0].pitch = pitch
    @property
    def planeCount(self):
        return self._ptr[0].planeCount
    @planeCount.setter
    def planeCount(self, unsigned int planeCount):
        self._ptr[0].planeCount = planeCount
    @property
    def numChannels(self):
        return self._ptr[0].numChannels
    @numChannels.setter
    def numChannels(self, unsigned int numChannels):
        self._ptr[0].numChannels = numChannels
    @property
    def frameType(self):
        return CUeglFrameType(self._ptr[0].frameType)
    @frameType.setter
    def frameType(self, frameType not None : CUeglFrameType):
        self._ptr[0].frameType = frameType.value
    @property
    def eglColorFormat(self):
        return CUeglColorFormat(self._ptr[0].eglColorFormat)
    @eglColorFormat.setter
    def eglColorFormat(self, eglColorFormat not None : CUeglColorFormat):
        self._ptr[0].eglColorFormat = eglColorFormat.value
    @property
    def cuFormat(self):
        return CUarray_format(self._ptr[0].cuFormat)
    @cuFormat.setter
    def cuFormat(self, cuFormat not None : CUarray_format):
        self._ptr[0].cuFormat = cuFormat.value

cdef class cuuint32_t:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, uint32_t init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.cuuint32_t *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<cuuint32_t ' + str(self.__int__()) + '>'
    def __int__(self):
        return <uint32_t>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class cuuint64_t:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, uint64_t init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.cuuint64_t *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<cuuint64_t ' + str(self.__int__()) + '>'
    def __int__(self):
        return <uint64_t>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUdeviceptr_v2:
    """

    CUDA device pointer CUdeviceptr is defined as an unsigned integer type whose size matches the size of a pointer on the target platform.

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, unsigned long long init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUdeviceptr_v2 *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<CUdeviceptr_v2 ' + str(self.__int__()) + '>'
    def __int__(self):
        return <unsigned long long>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUdevice_v1:
    """

    CUDA device

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, int init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUdevice_v1 *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<CUdevice_v1 ' + str(self.__int__()) + '>'
    def __int__(self):
        return <int>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUtexObject_v1:
    """

    An opaque value that represents a CUDA texture object

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, unsigned long long init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUtexObject_v1 *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<CUtexObject_v1 ' + str(self.__int__()) + '>'
    def __int__(self):
        return <unsigned long long>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUsurfObject_v1:
    """

    An opaque value that represents a CUDA surface object

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, unsigned long long init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUsurfObject_v1 *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<CUsurfObject_v1 ' + str(self.__int__()) + '>'
    def __int__(self):
        return <unsigned long long>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class CUmemGenericAllocationHandle_v1:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, unsigned long long init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.CUmemGenericAllocationHandle_v1 *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<CUmemGenericAllocationHandle_v1 ' + str(self.__int__()) + '>'
    def __int__(self):
        return <unsigned long long>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class GLenum:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, unsigned int init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.GLenum *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<GLenum ' + str(self.__int__()) + '>'
    def __int__(self):
        return <unsigned int>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class GLuint:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, unsigned int init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.GLuint *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<GLuint ' + str(self.__int__()) + '>'
    def __int__(self):
        return <unsigned int>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class EGLint:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, unsigned int init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.EGLint *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<EGLint ' + str(self.__int__()) + '>'
    def __int__(self):
        return <unsigned int>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class VdpDevice:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, uint32_t init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.VdpDevice *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<VdpDevice ' + str(self.__int__()) + '>'
    def __int__(self):
        return <uint32_t>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class VdpGetProcAddress:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, unsigned long long init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.VdpGetProcAddress *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<VdpGetProcAddress ' + str(self.__int__()) + '>'
    def __int__(self):
        return <unsigned long long>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class VdpVideoSurface:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, uint32_t init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.VdpVideoSurface *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<VdpVideoSurface ' + str(self.__int__()) + '>'
    def __int__(self):
        return <uint32_t>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

cdef class VdpOutputSurface:
    """

    Methods
    -------
    getPtr()
        Get memory address of class instance

    """
    def __cinit__(self, uint32_t init_value = 0, void_ptr _ptr = 0):
        if _ptr == 0:
            self._ptr = &self.__val
        else:
            self._ptr = <cydriver.VdpOutputSurface *>_ptr
        if init_value:
            self._ptr[0] = init_value
    def __dealloc__(self):
        pass
    def __repr__(self):
        return '<VdpOutputSurface ' + str(self.__int__()) + '>'
    def __int__(self):
        return <uint32_t>self._ptr[0]
    def getPtr(self):
        return <void_ptr>self._ptr

@cython.embedsignature(True)
def cuGetErrorString(error not None : CUresult):
    """ Gets the string description of an error code.

    Sets `*pStr` to the address of a NULL-terminated string description of
    the error code `error`. If the error code is not recognized,
    :py:obj:`~.CUDA_ERROR_INVALID_VALUE` will be returned and `*pStr` will
    be set to the NULL address.

    Parameters
    ----------
    error : :py:obj:`~.CUresult`
        Error code to convert to string

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    pStr : bytes
        Address of the string pointer.

    See Also
    --------
    :py:obj:`~.CUresult`, :py:obj:`~.cudaGetErrorString`
    """
    cdef cydriver.CUresult cyerror = error.value
    cdef const char* pStr = NULL
    err = cydriver.cuGetErrorString(cyerror, &pStr)
    return (CUresult(err), <bytes>pStr if pStr != NULL else None)

@cython.embedsignature(True)
def cuGetErrorName(error not None : CUresult):
    """ Gets the string representation of an error code enum name.

    Sets `*pStr` to the address of a NULL-terminated string representation
    of the name of the enum error code `error`. If the error code is not
    recognized, :py:obj:`~.CUDA_ERROR_INVALID_VALUE` will be returned and
    `*pStr` will be set to the NULL address.

    Parameters
    ----------
    error : :py:obj:`~.CUresult`
        Error code to convert to string

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    pStr : bytes
        Address of the string pointer.

    See Also
    --------
    :py:obj:`~.CUresult`, :py:obj:`~.cudaGetErrorName`
    """
    cdef cydriver.CUresult cyerror = error.value
    cdef const char* pStr = NULL
    err = cydriver.cuGetErrorName(cyerror, &pStr)
    return (CUresult(err), <bytes>pStr if pStr != NULL else None)

@cython.embedsignature(True)
def cuInit(unsigned int Flags):
    """ Initialize the CUDA driver API.

    Initializes the driver API and must be called before any other function
    from the driver API. Currently, the `Flags` parameter must be 0. If
    :py:obj:`~.cuInit()` has not been called, any function from the driver
    API will return :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`.

    Parameters
    ----------
    Flags : unsigned int
        Initialization flag for CUDA.

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`, :py:obj:`~.CUDA_ERROR_SYSTEM_DRIVER_MISMATCH`, :py:obj:`~.CUDA_ERROR_COMPAT_NOT_SUPPORTED_ON_DEVICE`
    """
    err = cydriver.cuInit(Flags)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuDriverGetVersion():
    """ Returns the latest CUDA version supported by driver.

    Returns in `*driverVersion` the version of CUDA supported by the
    driver. The version is returned as (1000 * major + 10 * minor). For
    example, CUDA 9.2 would be represented by 9020.

    This function automatically returns
    :py:obj:`~.CUDA_ERROR_INVALID_VALUE` if `driverVersion` is NULL.

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    driverVersion : int
        Returns the CUDA driver version

    See Also
    --------
    :py:obj:`~.cudaDriverGetVersion`, :py:obj:`~.cudaRuntimeGetVersion`
    """
    cdef int driverVersion = 0
    err = cydriver.cuDriverGetVersion(&driverVersion)
    return (CUresult(err), driverVersion)

@cython.embedsignature(True)
def cuDeviceGet(int ordinal):
    """ Returns a handle to a compute device.

    Returns in `*device` a device handle given an ordinal in the range [0,
    :py:obj:`~.cuDeviceGetCount()`-1].

    Parameters
    ----------
    ordinal : int
        Device number to get handle for

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`
    device : :py:obj:`~.CUdevice`
        Returned device handle

    See Also
    --------
    :py:obj:`~.cuDeviceGetAttribute`, :py:obj:`~.cuDeviceGetCount`, :py:obj:`~.cuDeviceGetName`, :py:obj:`~.cuDeviceGetUuid`, :py:obj:`~.cuDeviceGetLuid`, :py:obj:`~.cuDeviceTotalMem`, :py:obj:`~.cuDeviceGetExecAffinitySupport`
    """
    cdef CUdevice device = CUdevice()
    err = cydriver.cuDeviceGet(<cydriver.CUdevice*>device._ptr, ordinal)
    return (CUresult(err), device)

@cython.embedsignature(True)
def cuDeviceGetCount():
    """ Returns the number of compute-capable devices.

    Returns in `*count` the number of devices with compute capability
    greater than or equal to 2.0 that are available for execution. If there
    is no such device, :py:obj:`~.cuDeviceGetCount()` returns 0.

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    count : int
        Returned number of compute-capable devices

    See Also
    --------
    :py:obj:`~.cuDeviceGetAttribute`, :py:obj:`~.cuDeviceGetName`, :py:obj:`~.cuDeviceGetUuid`, :py:obj:`~.cuDeviceGetLuid`, :py:obj:`~.cuDeviceGet`, :py:obj:`~.cuDeviceTotalMem`, :py:obj:`~.cuDeviceGetExecAffinitySupport`, :py:obj:`~.cudaGetDeviceCount`
    """
    cdef int count = 0
    err = cydriver.cuDeviceGetCount(&count)
    return (CUresult(err), count)

@cython.embedsignature(True)
def cuDeviceGetName(int length, dev):
    """ Returns an identifer string for the device.

    Returns an ASCII string identifying the device `dev` in the NULL-
    terminated string pointed to by `name`. `length` specifies the maximum
    length of the string that may be returned.

    Parameters
    ----------
    length : int
        Maximum length of string to store in `name`
    dev : :py:obj:`~.CUdevice`
        Device to get identifier string for

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`
    name : bytes
        Returned identifier string for the device

    See Also
    --------
    :py:obj:`~.cuDeviceGetAttribute`, :py:obj:`~.cuDeviceGetUuid`, :py:obj:`~.cuDeviceGetLuid`, :py:obj:`~.cuDeviceGetCount`, :py:obj:`~.cuDeviceGet`, :py:obj:`~.cuDeviceTotalMem`, :py:obj:`~.cuDeviceGetExecAffinitySupport`, :py:obj:`~.cudaGetDeviceProperties`
    """
    cdef cydriver.CUdevice cydev
    if dev is None:
        cydev = <cydriver.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cydev = <cydriver.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cydev = <cydriver.CUdevice>pdev
    pyname = b" " * length
    cdef char* name = pyname
    err = cydriver.cuDeviceGetName(name, length, cydev)
    return (CUresult(err), pyname)

@cython.embedsignature(True)
def cuDeviceGetUuid(dev):
    """ Return an UUID for the device.

    Note there is a later version of this API,
    :py:obj:`~.cuDeviceGetUuid_v2`. It will supplant this version in 12.0,
    which is retained for minor version compatibility.

    Returns 16-octets identifing the device `dev` in the structure pointed
    by the `uuid`.

    Parameters
    ----------
    dev : :py:obj:`~.CUdevice`
        Device to get identifier string for

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`
    uuid : :py:obj:`~.CUuuid`
        Returned UUID

    See Also
    --------
    :py:obj:`~.cuDeviceGetUuid_v2` :py:obj:`~.cuDeviceGetAttribute`, :py:obj:`~.cuDeviceGetCount`, :py:obj:`~.cuDeviceGetName`, :py:obj:`~.cuDeviceGetLuid`, :py:obj:`~.cuDeviceGet`, :py:obj:`~.cuDeviceTotalMem`, :py:obj:`~.cuDeviceGetExecAffinitySupport`, :py:obj:`~.cudaGetDeviceProperties`
    """
    cdef cydriver.CUdevice cydev
    if dev is None:
        cydev = <cydriver.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cydev = <cydriver.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cydev = <cydriver.CUdevice>pdev
    cdef CUuuid uuid = CUuuid()
    err = cydriver.cuDeviceGetUuid(<cydriver.CUuuid*>uuid._ptr, cydev)
    return (CUresult(err), uuid)

@cython.embedsignature(True)
def cuDeviceGetUuid_v2(dev):
    """ Return an UUID for the device (11.4+)

    Returns 16-octets identifing the device `dev` in the structure pointed
    by the `uuid`. If the device is in MIG mode, returns its MIG UUID which
    uniquely identifies the subscribed MIG compute instance.

    Parameters
    ----------
    dev : :py:obj:`~.CUdevice`
        Device to get identifier string for

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`
    uuid : :py:obj:`~.CUuuid`
        Returned UUID

    See Also
    --------
    :py:obj:`~.cuDeviceGetAttribute`, :py:obj:`~.cuDeviceGetCount`, :py:obj:`~.cuDeviceGetName`, :py:obj:`~.cuDeviceGetLuid`, :py:obj:`~.cuDeviceGet`, :py:obj:`~.cuDeviceTotalMem`, :py:obj:`~.cudaGetDeviceProperties`
    """
    cdef cydriver.CUdevice cydev
    if dev is None:
        cydev = <cydriver.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cydev = <cydriver.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cydev = <cydriver.CUdevice>pdev
    cdef CUuuid uuid = CUuuid()
    err = cydriver.cuDeviceGetUuid_v2(<cydriver.CUuuid*>uuid._ptr, cydev)
    return (CUresult(err), uuid)

@cython.embedsignature(True)
def cuDeviceGetLuid(dev):
    """ Return an LUID and device node mask for the device.

    Return identifying information (`luid` and `deviceNodeMask`) to allow
    matching device with graphics APIs.

    Parameters
    ----------
    dev : :py:obj:`~.CUdevice`
        Device to get identifier string for

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`
    luid : bytes
        Returned LUID
    deviceNodeMask : unsigned int
        Returned device node mask

    See Also
    --------
    :py:obj:`~.cuDeviceGetAttribute`, :py:obj:`~.cuDeviceGetCount`, :py:obj:`~.cuDeviceGetName`, :py:obj:`~.cuDeviceGet`, :py:obj:`~.cuDeviceTotalMem`, :py:obj:`~.cuDeviceGetExecAffinitySupport`, :py:obj:`~.cudaGetDeviceProperties`
    """
    cdef cydriver.CUdevice cydev
    if dev is None:
        cydev = <cydriver.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cydev = <cydriver.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cydev = <cydriver.CUdevice>pdev
    cdef char luid[8]
    cdef unsigned int deviceNodeMask = 0
    err = cydriver.cuDeviceGetLuid(luid, &deviceNodeMask, cydev)
    return (CUresult(err), <bytes>luid, deviceNodeMask)

@cython.embedsignature(True)
def cuDeviceTotalMem(dev):
    """ Returns the total amount of memory on the device.

    Returns in `*bytes` the total amount of memory available on the device
    `dev` in bytes.

    Parameters
    ----------
    dev : :py:obj:`~.CUdevice`
        Device handle

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`
    numbytes : int
        Returned memory available on device in bytes

    See Also
    --------
    :py:obj:`~.cuDeviceGetAttribute`, :py:obj:`~.cuDeviceGetCount`, :py:obj:`~.cuDeviceGetName`, :py:obj:`~.cuDeviceGetUuid`, :py:obj:`~.cuDeviceGet`, :py:obj:`~.cuDeviceGetExecAffinitySupport`, :py:obj:`~.cudaMemGetInfo`
    """
    cdef cydriver.CUdevice cydev
    if dev is None:
        cydev = <cydriver.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cydev = <cydriver.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cydev = <cydriver.CUdevice>pdev
    cdef size_t numbytes = 0
    err = cydriver.cuDeviceTotalMem(&numbytes, cydev)
    return (CUresult(err), numbytes)

@cython.embedsignature(True)
def cuDeviceGetTexture1DLinearMaxWidth(pformat not None : CUarray_format, unsigned numChannels, dev):
    """ Returns the maximum number of elements allocatable in a 1D linear texture for a given texture element size.

    Returns in `maxWidthInElements` the maximum number of texture elements
    allocatable in a 1D linear texture for given `pformat` and
    `numChannels`.

    Parameters
    ----------
    pformat : :py:obj:`~.CUarray_format`
        Texture format.
    numChannels : unsigned
        Number of channels per texture element.
    dev : :py:obj:`~.CUdevice`
        Device handle.

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`
    maxWidthInElements : int
        Returned maximum number of texture elements allocatable for given
        `pformat` and `numChannels`.

    See Also
    --------
    :py:obj:`~.cuDeviceGetAttribute`, :py:obj:`~.cuDeviceGetCount`, :py:obj:`~.cuDeviceGetName`, :py:obj:`~.cuDeviceGetUuid`, :py:obj:`~.cuDeviceGet`, :py:obj:`~.cudaMemGetInfo`, :py:obj:`~.cuDeviceTotalMem`
    """
    cdef cydriver.CUdevice cydev
    if dev is None:
        cydev = <cydriver.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cydev = <cydriver.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cydev = <cydriver.CUdevice>pdev
    cdef size_t maxWidthInElements = 0
    cdef cydriver.CUarray_format cypformat = pformat.value
    err = cydriver.cuDeviceGetTexture1DLinearMaxWidth(&maxWidthInElements, cypformat, numChannels, cydev)
    return (CUresult(err), maxWidthInElements)

@cython.embedsignature(True)
def cuDeviceGetAttribute(attrib not None : CUdevice_attribute, dev):
    """ Returns information about the device.

    Returns in `*pi` the integer value of the attribute `attrib` on device
    `dev`. The supported attributes are:

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_THREADS_PER_BLOCK`: Maximum number
      of threads per block;

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_BLOCK_DIM_X`: Maximum x-dimension
      of a block

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_BLOCK_DIM_Y`: Maximum y-dimension
      of a block

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_BLOCK_DIM_Z`: Maximum z-dimension
      of a block

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_GRID_DIM_X`: Maximum x-dimension
      of a grid

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_GRID_DIM_Y`: Maximum y-dimension
      of a grid

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_GRID_DIM_Z`: Maximum z-dimension
      of a grid

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_SHARED_MEMORY_PER_BLOCK`: Maximum
      amount of shared memory available to a thread block in bytes

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_TOTAL_CONSTANT_MEMORY`: Memory
      available on device for constant variables in a CUDA C kernel in
      bytes

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_WARP_SIZE`: Warp size in threads

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_PITCH`: Maximum pitch in bytes
      allowed by the memory copy functions that involve memory regions
      allocated through :py:obj:`~.cuMemAllocPitch()`

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE1D_WIDTH`: Maximum 1D
      texture width

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE1D_LINEAR_WIDTH`:
      Maximum width for a 1D texture bound to linear memory

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE1D_MIPMAPPED_WIDTH`:
      Maximum mipmapped 1D texture width

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_WIDTH`: Maximum 2D
      texture width

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_HEIGHT`: Maximum 2D
      texture height

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LINEAR_WIDTH`:
      Maximum width for a 2D texture bound to linear memory

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LINEAR_HEIGHT`:
      Maximum height for a 2D texture bound to linear memory

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LINEAR_PITCH`:
      Maximum pitch in bytes for a 2D texture bound to linear memory

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_MIPMAPPED_WIDTH`:
      Maximum mipmapped 2D texture width

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_MIPMAPPED_HEIGHT`:
      Maximum mipmapped 2D texture height

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_WIDTH`: Maximum 3D
      texture width

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_HEIGHT`: Maximum 3D
      texture height

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_DEPTH`: Maximum 3D
      texture depth

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_WIDTH_ALTERNATE`:
      Alternate maximum 3D texture width, 0 if no alternate maximum 3D
      texture size is supported

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_HEIGHT_ALTERNATE`:
      Alternate maximum 3D texture height, 0 if no alternate maximum 3D
      texture size is supported

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE3D_DEPTH_ALTERNATE`:
      Alternate maximum 3D texture depth, 0 if no alternate maximum 3D
      texture size is supported

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURECUBEMAP_WIDTH`: Maximum
      cubemap texture width or height

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE1D_LAYERED_WIDTH`:
      Maximum 1D layered texture width

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE1D_LAYERED_LAYERS`:
      Maximum layers in a 1D layered texture

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LAYERED_WIDTH`:
      Maximum 2D layered texture width

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LAYERED_HEIGHT`:
      Maximum 2D layered texture height

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_LAYERED_LAYERS`:
      Maximum layers in a 2D layered texture

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURECUBEMAP_LAYERED_WIDTH`:
      Maximum cubemap layered texture width or height

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURECUBEMAP_LAYERED_LAYERS`:
      Maximum layers in a cubemap layered texture

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE1D_WIDTH`: Maximum 1D
      surface width

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE2D_WIDTH`: Maximum 2D
      surface width

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE2D_HEIGHT`: Maximum 2D
      surface height

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE3D_WIDTH`: Maximum 3D
      surface width

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE3D_HEIGHT`: Maximum 3D
      surface height

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE3D_DEPTH`: Maximum 3D
      surface depth

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE1D_LAYERED_WIDTH`:
      Maximum 1D layered surface width

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE1D_LAYERED_LAYERS`:
      Maximum layers in a 1D layered surface

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE2D_LAYERED_WIDTH`:
      Maximum 2D layered surface width

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE2D_LAYERED_HEIGHT`:
      Maximum 2D layered surface height

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACE2D_LAYERED_LAYERS`:
      Maximum layers in a 2D layered surface

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACECUBEMAP_WIDTH`: Maximum
      cubemap surface width

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACECUBEMAP_LAYERED_WIDTH`:
      Maximum cubemap layered surface width

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_SURFACECUBEMAP_LAYERED_LAYERS`:
      Maximum layers in a cubemap layered surface

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_REGISTERS_PER_BLOCK`: Maximum
      number of 32-bit registers available to a thread block

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_CLOCK_RATE`: The typical clock
      frequency in kilohertz

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_TEXTURE_ALIGNMENT`: Alignment
      requirement; texture base addresses aligned to
      :py:obj:`~.textureAlign` bytes do not need an offset applied to
      texture fetches

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_TEXTURE_PITCH_ALIGNMENT`: Pitch
      alignment requirement for 2D texture references bound to pitched
      memory

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_GPU_OVERLAP`: 1 if the device can
      concurrently copy memory between host and device while executing a
      kernel, or 0 if not

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MULTIPROCESSOR_COUNT`: Number of
      multiprocessors on the device

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_KERNEL_EXEC_TIMEOUT`: 1 if there is a
      run time limit for kernels executed on the device, or 0 if not

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_INTEGRATED`: 1 if the device is
      integrated with the memory subsystem, or 0 if not

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_CAN_MAP_HOST_MEMORY`: 1 if the device
      can map host memory into the CUDA address space, or 0 if not

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_COMPUTE_MODE`: Compute mode that
      device is currently in. Available modes are as follows:

      - :py:obj:`~.CU_COMPUTEMODE_DEFAULT`: Default mode - Device is not
        restricted and can have multiple CUDA contexts present at a single
        time.

      - :py:obj:`~.CU_COMPUTEMODE_PROHIBITED`: Compute-prohibited mode -
        Device is prohibited from creating new CUDA contexts.

      - :py:obj:`~.CU_COMPUTEMODE_EXCLUSIVE_PROCESS`: Compute-exclusive-
        process mode - Device can have only one context used by a single
        process at a time.

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_CONCURRENT_KERNELS`: 1 if the device
      supports executing multiple kernels within the same context
      simultaneously, or 0 if not. It is not guaranteed that multiple
      kernels will be resident on the device concurrently so this feature
      should not be relied upon for correctness.

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_ECC_ENABLED`: 1 if error correction is
      enabled on the device, 0 if error correction is disabled or not
      supported by the device

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_PCI_BUS_ID`: PCI bus identifier of the
      device

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_PCI_DEVICE_ID`: PCI device (also known
      as slot) identifier of the device

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_PCI_DOMAIN_ID`: PCI domain identifier
      of the device

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_TCC_DRIVER`: 1 if the device is using
      a TCC driver. TCC is only available on Tesla hardware running Windows
      Vista or later

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MEMORY_CLOCK_RATE`: Peak memory clock
      frequency in kilohertz

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_GLOBAL_MEMORY_BUS_WIDTH`: Global
      memory bus width in bits

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_L2_CACHE_SIZE`: Size of L2 cache in
      bytes. 0 if the device doesn't have L2 cache

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_THREADS_PER_MULTIPROCESSOR`:
      Maximum resident threads per multiprocessor

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_UNIFIED_ADDRESSING`: 1 if the device
      shares a unified address space with the host, or 0 if not

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_COMPUTE_CAPABILITY_MAJOR`: Major
      compute capability version number

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_COMPUTE_CAPABILITY_MINOR`: Minor
      compute capability version number

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_GLOBAL_L1_CACHE_SUPPORTED`: 1 if
      device supports caching globals in L1 cache, 0 if caching globals in
      L1 cache is not supported by the device

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_LOCAL_L1_CACHE_SUPPORTED`: 1 if device
      supports caching locals in L1 cache, 0 if caching locals in L1 cache
      is not supported by the device

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_SHARED_MEMORY_PER_MULTIPROCESSOR`:
      Maximum amount of shared memory available to a multiprocessor in
      bytes; this amount is shared by all thread blocks simultaneously
      resident on a multiprocessor

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_REGISTERS_PER_MULTIPROCESSOR`:
      Maximum number of 32-bit registers available to a multiprocessor;
      this number is shared by all thread blocks simultaneously resident on
      a multiprocessor

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MANAGED_MEMORY`: 1 if device supports
      allocating managed memory on this system, 0 if allocating managed
      memory is not supported by the device on this system.

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MULTI_GPU_BOARD`: 1 if device is on a
      multi-GPU board, 0 if not.

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MULTI_GPU_BOARD_GROUP_ID`: Unique
      identifier for a group of devices associated with the same board.
      Devices on the same multi-GPU board will share the same identifier.

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_HOST_NATIVE_ATOMIC_SUPPORTED`: 1 if
      Link between the device and the host supports native atomic
      operations.

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_SINGLE_TO_DOUBLE_PRECISION_PERF_RATIO`:
      Ratio of single precision performance (in floating-point operations
      per second) to double precision performance.

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_PAGEABLE_MEMORY_ACCESS`: Device
      suppports coherently accessing pageable memory without calling
      cudaHostRegister on it.

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_CONCURRENT_MANAGED_ACCESS`: Device can
      coherently access managed memory concurrently with the CPU.

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_COMPUTE_PREEMPTION_SUPPORTED`: Device
      supports Compute Preemption.

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_CAN_USE_HOST_POINTER_FOR_REGISTERED_MEM`:
      Device can access host registered memory at the same virtual address
      as the CPU.

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_SHARED_MEMORY_PER_BLOCK_OPTIN`:
      The maximum per block shared memory size suported on this device.
      This is the maximum value that can be opted into when using the
      :py:obj:`~.cuFuncSetAttribute()` call. For more details see
      :py:obj:`~.CU_FUNC_ATTRIBUTE_MAX_DYNAMIC_SHARED_SIZE_BYTES`

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_PAGEABLE_MEMORY_ACCESS_USES_HOST_PAGE_TABLES`:
      Device accesses pageable memory via the host's page tables.

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_DIRECT_MANAGED_MEM_ACCESS_FROM_HOST`:
      The host can directly access managed memory on the device without
      migration.

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_VIRTUAL_MEMORY_MANAGEMENT_SUPPORTED`:
      Device supports virtual memory management APIs like
      :py:obj:`~.cuMemAddressReserve`, :py:obj:`~.cuMemCreate`,
      :py:obj:`~.cuMemMap` and related APIs

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_HANDLE_TYPE_POSIX_FILE_DESCRIPTOR_SUPPORTED`:
      Device supports exporting memory to a posix file descriptor with
      :py:obj:`~.cuMemExportToShareableHandle`, if requested via
      :py:obj:`~.cuMemCreate`

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_HANDLE_TYPE_WIN32_HANDLE_SUPPORTED`:
      Device supports exporting memory to a Win32 NT handle with
      :py:obj:`~.cuMemExportToShareableHandle`, if requested via
      :py:obj:`~.cuMemCreate`

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_HANDLE_TYPE_WIN32_KMT_HANDLE_SUPPORTED`:
      Device supports exporting memory to a Win32 KMT handle with
      :py:obj:`~.cuMemExportToShareableHandle`, if requested via
      :py:obj:`~.cuMemCreate`

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_BLOCKS_PER_MULTIPROCESSOR`:
      Maximum number of thread blocks that can reside on a multiprocessor

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_GENERIC_COMPRESSION_SUPPORTED`: Device
      supports compressible memory allocation via :py:obj:`~.cuMemCreate`

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_PERSISTING_L2_CACHE_SIZE`: Maximum
      L2 persisting lines capacity setting in bytes

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_ACCESS_POLICY_WINDOW_SIZE`:
      Maximum value of :py:obj:`~.CUaccessPolicyWindow.num_bytes`

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_GPU_DIRECT_RDMA_WITH_CUDA_VMM_SUPPORTED`:
      Device supports specifying the GPUDirect RDMA flag with
      :py:obj:`~.cuMemCreate`.

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_RESERVED_SHARED_MEMORY_PER_BLOCK`:
      Amount of shared memory per block reserved by CUDA driver in bytes

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_SPARSE_CUDA_ARRAY_SUPPORTED`: Device
      supports sparse CUDA arrays and sparse CUDA mipmapped arrays.

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_READ_ONLY_HOST_REGISTER_SUPPORTED`:
      Device supports using the :py:obj:`~.cuMemHostRegister` flag
      :py:obj:`~.CU_MEMHOSTERGISTER_READ_ONLY` to register memory that must
      be mapped as read-only to the GPU

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MEMORY_POOLS_SUPPORTED`: Device
      supports using the :py:obj:`~.cuMemAllocAsync` and
      :py:obj:`~.cuMemPool` family of APIs

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_GPU_DIRECT_RDMA_SUPPORTED`: Device
      supports GPUDirect RDMA APIs, like nvidia_p2p_get_pages (see
      https://docs.nvidia.com/cuda/gpudirect-rdma for more information)

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_GPU_DIRECT_RDMA_FLUSH_WRITES_OPTIONS`:
      The returned attribute shall be interpreted as a bitmask, where the
      individual bits are described by the
      :py:obj:`~.CUflushGPUDirectRDMAWritesOptions` enum

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_GPU_DIRECT_RDMA_WRITES_ORDERING`:
      GPUDirect RDMA writes to the device do not need to be flushed for
      consumers within the scope indicated by the returned attribute. See
      :py:obj:`~.CUGPUDirectRDMAWritesOrdering` for the numerical values
      returned here.

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_MEMPOOL_SUPPORTED_HANDLE_TYPES`:
      Bitmask of handle types supported with mempool based IPC

    - :py:obj:`~.CU_DEVICE_ATTRIBUTE_DEFERRED_MAPPING_CUDA_ARRAY_SUPPORTED`:
      Device supports deferred mapping CUDA arrays and CUDA mipmapped
      arrays.

    Parameters
    ----------
    attrib : :py:obj:`~.CUdevice_attribute`
        Device attribute to query
    dev : :py:obj:`~.CUdevice`
        Device handle

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`
    pi : int
        Returned device attribute value

    See Also
    --------
    :py:obj:`~.cuDeviceGetCount`, :py:obj:`~.cuDeviceGetName`, :py:obj:`~.cuDeviceGetUuid`, :py:obj:`~.cuDeviceGet`, :py:obj:`~.cuDeviceTotalMem`, :py:obj:`~.cuDeviceGetExecAffinitySupport`, :py:obj:`~.cudaDeviceGetAttribute`, :py:obj:`~.cudaGetDeviceProperties`
    """
    cdef cydriver.CUdevice cydev
    if dev is None:
        cydev = <cydriver.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cydev = <cydriver.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cydev = <cydriver.CUdevice>pdev
    cdef int pi = 0
    cdef cydriver.CUdevice_attribute cyattrib = attrib.value
    err = cydriver.cuDeviceGetAttribute(&pi, cyattrib, cydev)
    return (CUresult(err), pi)

@cython.embedsignature(True)
def cuDeviceGetNvSciSyncAttributes(nvSciSyncAttrList, dev, int flags):
    """ Return NvSciSync attributes that this device can support.

    Returns in `nvSciSyncAttrList`, the properties of NvSciSync that this
    CUDA device, `dev` can support. The returned `nvSciSyncAttrList` can be
    used to create an NvSciSync object that matches this device's
    capabilities.

    If NvSciSyncAttrKey_RequiredPerm field in `nvSciSyncAttrList` is
    already set this API will return :py:obj:`~.CUDA_ERROR_INVALID_VALUE`.

    The applications should set `nvSciSyncAttrList` to a valid
    NvSciSyncAttrList failing which this API will return
    :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`.

    The `flags` controls how applications intends to use the NvSciSync
    created from the `nvSciSyncAttrList`. The valid flags are:

    - :py:obj:`~.CUDA_NVSCISYNC_ATTR_SIGNAL`, specifies that the
      applications intends to signal an NvSciSync on this CUDA device.

    - :py:obj:`~.CUDA_NVSCISYNC_ATTR_WAIT`, specifies that the applications
      intends to wait on an NvSciSync on this CUDA device.

    At least one of these flags must be set, failing which the API returns
    :py:obj:`~.CUDA_ERROR_INVALID_VALUE`. Both the flags are orthogonal to
    one another: a developer may set both these flags that allows to set
    both wait and signal specific attributes in the same
    `nvSciSyncAttrList`.

    :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`,
    :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`,
    :py:obj:`~.CUDA_ERROR_INVALID_VALUE`,
    :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`,
    :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`,
    :py:obj:`~.CUDA_ERROR_NOT_SUPPORTED`,
    :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`

    Parameters
    ----------
    nvSciSyncAttrList : Any
        Return NvSciSync attributes supported.
    dev : :py:obj:`~.CUdevice`
        Valid Cuda Device to get NvSciSync attributes for.
    flags : int
        flags describing NvSciSync usage.

    Returns
    -------
    CUresult


    See Also
    --------
    :py:obj:`~.cuImportExternalSemaphore`, :py:obj:`~.cuDestroyExternalSemaphore`, :py:obj:`~.cuSignalExternalSemaphoresAsync`, :py:obj:`~.cuWaitExternalSemaphoresAsync`
    """
    cdef cydriver.CUdevice cydev
    if dev is None:
        cydev = <cydriver.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cydev = <cydriver.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cydev = <cydriver.CUdevice>pdev
    cynvSciSyncAttrList = utils.HelperInputVoidPtr(nvSciSyncAttrList)
    cdef void* cynvSciSyncAttrList_ptr = <void*><void_ptr>cynvSciSyncAttrList.cptr
    err = cydriver.cuDeviceGetNvSciSyncAttributes(cynvSciSyncAttrList_ptr, cydev, flags)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuDeviceSetMemPool(dev, pool):
    """ Sets the current memory pool of a device.

    The memory pool must be local to the specified device.
    :py:obj:`~.cuMemAllocAsync` allocates from the current mempool of the
    provided stream's device. By default, a device's current memory pool is
    its default memory pool.

    Parameters
    ----------
    dev : :py:obj:`~.CUdevice`
        None
    pool : :py:obj:`~.CUmemoryPool` or :py:obj:`~.cudaMemPool_t`
        None

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuDeviceGetDefaultMemPool`, :py:obj:`~.cuDeviceGetMemPool`, :py:obj:`~.cuMemPoolCreate`, :py:obj:`~.cuMemPoolDestroy`, :py:obj:`~.cuMemAllocFromPoolAsync`

    Notes
    -----
    Use :py:obj:`~.cuMemAllocFromPoolAsync` to specify asynchronous allocations from a device different than the one the stream runs on.
    """
    cdef cydriver.CUmemoryPool cypool
    if pool is None:
        cypool = <cydriver.CUmemoryPool><void_ptr>0
    elif isinstance(pool, (CUmemoryPool,)):
        ppool = int(pool)
        cypool = <cydriver.CUmemoryPool><void_ptr>ppool
    else:
        ppool = int(CUmemoryPool(pool))
        cypool = <cydriver.CUmemoryPool><void_ptr>ppool
    cdef cydriver.CUdevice cydev
    if dev is None:
        cydev = <cydriver.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cydev = <cydriver.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cydev = <cydriver.CUdevice>pdev
    err = cydriver.cuDeviceSetMemPool(cydev, cypool)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuDeviceGetMemPool(dev):
    """ Gets the current mempool for a device.

    Returns the last pool provided to :py:obj:`~.cuDeviceSetMemPool` for
    this device or the device's default memory pool if
    :py:obj:`~.cuDeviceSetMemPool` has never been called. By default the
    current mempool is the default mempool for a device. Otherwise the
    returned pool must have been set with :py:obj:`~.cuDeviceSetMemPool`.

    Parameters
    ----------
    dev : :py:obj:`~.CUdevice`
        None

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    pool : :py:obj:`~.CUmemoryPool`
        None

    See Also
    --------
    :py:obj:`~.cuDeviceGetDefaultMemPool`, :py:obj:`~.cuMemPoolCreate`, :py:obj:`~.cuDeviceSetMemPool`
    """
    cdef cydriver.CUdevice cydev
    if dev is None:
        cydev = <cydriver.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cydev = <cydriver.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cydev = <cydriver.CUdevice>pdev
    cdef CUmemoryPool pool = CUmemoryPool()
    err = cydriver.cuDeviceGetMemPool(<cydriver.CUmemoryPool*>pool._ptr, cydev)
    return (CUresult(err), pool)

@cython.embedsignature(True)
def cuDeviceGetDefaultMemPool(dev):
    """ Returns the default mempool of a device.

    The default mempool of a device contains device memory from that
    device.

    Parameters
    ----------
    dev : :py:obj:`~.CUdevice`
        None

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED` :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`, :py:obj:`~.CUDA_ERROR_NOT_SUPPORTED`
    pool_out : :py:obj:`~.CUmemoryPool`
        None

    See Also
    --------
    :py:obj:`~.cuMemAllocAsync`, :py:obj:`~.cuMemPoolTrimTo`, :py:obj:`~.cuMemPoolGetAttribute`, :py:obj:`~.cuMemPoolSetAttribute`, :py:obj:`~.cuMemPoolSetAccess`, :py:obj:`~.cuDeviceGetMemPool`, :py:obj:`~.cuMemPoolCreate`
    """
    cdef cydriver.CUdevice cydev
    if dev is None:
        cydev = <cydriver.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cydev = <cydriver.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cydev = <cydriver.CUdevice>pdev
    cdef CUmemoryPool pool_out = CUmemoryPool()
    err = cydriver.cuDeviceGetDefaultMemPool(<cydriver.CUmemoryPool*>pool_out._ptr, cydev)
    return (CUresult(err), pool_out)

@cython.embedsignature(True)
def cuFlushGPUDirectRDMAWrites(target not None : CUflushGPUDirectRDMAWritesTarget, scope not None : CUflushGPUDirectRDMAWritesScope):
    """ Blocks until remote writes are visible to the specified scope.

    Blocks until GPUDirect RDMA writes to the target context via mappings
    created through APIs like nvidia_p2p_get_pages (see
    https://docs.nvidia.com/cuda/gpudirect-rdma for more information), are
    visible to the specified scope.

    If the scope equals or lies within the scope indicated by
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_GPU_DIRECT_RDMA_WRITES_ORDERING`, the
    call will be a no-op and can be safely omitted for performance. This
    can be determined by comparing the numerical values between the two
    enums, with smaller scopes having smaller values.

    Users may query support for this API via
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_FLUSH_FLUSH_GPU_DIRECT_RDMA_OPTIONS`.

    Parameters
    ----------
    target : :py:obj:`~.CUflushGPUDirectRDMAWritesTarget`
        The target of the operation, see
        :py:obj:`~.CUflushGPUDirectRDMAWritesTarget`
    scope : :py:obj:`~.CUflushGPUDirectRDMAWritesScope`
        The scope of the operation, see
        :py:obj:`~.CUflushGPUDirectRDMAWritesScope`

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`,
    """
    cdef cydriver.CUflushGPUDirectRDMAWritesTarget cytarget = target.value
    cdef cydriver.CUflushGPUDirectRDMAWritesScope cyscope = scope.value
    err = cydriver.cuFlushGPUDirectRDMAWrites(cytarget, cyscope)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuDeviceGetProperties(dev):
    """ Returns properties for a selected device.

    [Deprecated]

    This function was deprecated as of CUDA 5.0 and replaced by
    :py:obj:`~.cuDeviceGetAttribute()`.

    Returns in `*prop` the properties of device `dev`. The
    :py:obj:`~.CUdevprop` structure is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    where:

    - :py:obj:`~.maxThreadsPerBlock` is the maximum number of threads per
      block;

    - :py:obj:`~.maxThreadsDim`[3] is the maximum sizes of each dimension
      of a block;

    - :py:obj:`~.maxGridSize`[3] is the maximum sizes of each dimension of
      a grid;

    - :py:obj:`~.sharedMemPerBlock` is the total amount of shared memory
      available per block in bytes;

    - :py:obj:`~.totalConstantMemory` is the total amount of constant
      memory available on the device in bytes;

    - :py:obj:`~.SIMDWidth` is the warp size;

    - :py:obj:`~.memPitch` is the maximum pitch allowed by the memory copy
      functions that involve memory regions allocated through
      :py:obj:`~.cuMemAllocPitch()`;

    - :py:obj:`~.regsPerBlock` is the total number of registers available
      per block;

    - :py:obj:`~.clockRate` is the clock frequency in kilohertz;

    - :py:obj:`~.textureAlign` is the alignment requirement; texture base
      addresses that are aligned to :py:obj:`~.textureAlign` bytes do not
      need an offset applied to texture fetches.

    Parameters
    ----------
    dev : :py:obj:`~.CUdevice`
        Device to get properties for

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`
    prop : :py:obj:`~.CUdevprop`
        Returned properties of device

    See Also
    --------
    :py:obj:`~.cuDeviceGetAttribute`, :py:obj:`~.cuDeviceGetCount`, :py:obj:`~.cuDeviceGetName`, :py:obj:`~.cuDeviceGetUuid`, :py:obj:`~.cuDeviceGet`, :py:obj:`~.cuDeviceTotalMem`
    """
    cdef cydriver.CUdevice cydev
    if dev is None:
        cydev = <cydriver.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cydev = <cydriver.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cydev = <cydriver.CUdevice>pdev
    cdef CUdevprop prop = CUdevprop()
    err = cydriver.cuDeviceGetProperties(<cydriver.CUdevprop*>prop._ptr, cydev)
    return (CUresult(err), prop)

@cython.embedsignature(True)
def cuDeviceComputeCapability(dev):
    """ Returns the compute capability of the device.

    [Deprecated]

    This function was deprecated as of CUDA 5.0 and its functionality
    superceded by :py:obj:`~.cuDeviceGetAttribute()`.

    Returns in `*major` and `*minor` the major and minor revision numbers
    that define the compute capability of the device `dev`.

    Parameters
    ----------
    dev : :py:obj:`~.CUdevice`
        Device handle

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`
    major : int
        Major revision number
    minor : int
        Minor revision number

    See Also
    --------
    :py:obj:`~.cuDeviceGetAttribute`, :py:obj:`~.cuDeviceGetCount`, :py:obj:`~.cuDeviceGetName`, :py:obj:`~.cuDeviceGetUuid`, :py:obj:`~.cuDeviceGet`, :py:obj:`~.cuDeviceTotalMem`
    """
    cdef cydriver.CUdevice cydev
    if dev is None:
        cydev = <cydriver.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cydev = <cydriver.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cydev = <cydriver.CUdevice>pdev
    cdef int major = 0
    cdef int minor = 0
    err = cydriver.cuDeviceComputeCapability(&major, &minor, cydev)
    return (CUresult(err), major, minor)

@cython.embedsignature(True)
def cuDevicePrimaryCtxRetain(dev):
    """ Retain the primary context on the GPU.

    Retains the primary context on the device. Once the user successfully
    retains the primary context, the primary context will be active and
    available to the user until the user releases it with
    :py:obj:`~.cuDevicePrimaryCtxRelease()` or resets it with
    :py:obj:`~.cuDevicePrimaryCtxReset()`. Unlike :py:obj:`~.cuCtxCreate()`
    the newly retained context is not pushed onto the stack.

    Retaining the primary context for the first time will fail with
    :py:obj:`~.CUDA_ERROR_UNKNOWN` if the compute mode of the device is
    :py:obj:`~.CU_COMPUTEMODE_PROHIBITED`. The function
    :py:obj:`~.cuDeviceGetAttribute()` can be used with
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_COMPUTE_MODE` to determine the compute
    mode of the device. The `nvidia-smi` tool can be used to set the
    compute mode for devices. Documentation for `nvidia-smi` can be
    obtained by passing a -h option to it.

    Please note that the primary context always supports pinned
    allocations. Other flags can be specified by
    :py:obj:`~.cuDevicePrimaryCtxSetFlags()`.

    Parameters
    ----------
    dev : :py:obj:`~.CUdevice`
        Device for which primary context is requested

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_UNKNOWN`
    pctx : :py:obj:`~.CUcontext`
        Returned context handle of the new context

    See Also
    --------
    :py:obj:`~.cuDevicePrimaryCtxRelease`, :py:obj:`~.cuDevicePrimaryCtxSetFlags`, :py:obj:`~.cuCtxCreate`, :py:obj:`~.cuCtxGetApiVersion`, :py:obj:`~.cuCtxGetCacheConfig`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxGetLimit`, :py:obj:`~.cuCtxPopCurrent`, :py:obj:`~.cuCtxPushCurrent`, :py:obj:`~.cuCtxSetCacheConfig`, :py:obj:`~.cuCtxSetLimit`, :py:obj:`~.cuCtxSynchronize`
    """
    cdef cydriver.CUdevice cydev
    if dev is None:
        cydev = <cydriver.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cydev = <cydriver.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cydev = <cydriver.CUdevice>pdev
    cdef CUcontext pctx = CUcontext()
    err = cydriver.cuDevicePrimaryCtxRetain(<cydriver.CUcontext*>pctx._ptr, cydev)
    return (CUresult(err), pctx)

@cython.embedsignature(True)
def cuDevicePrimaryCtxRelease(dev):
    """ Release the primary context on the GPU.

    Releases the primary context interop on the device. A retained context
    should always be released once the user is done using it. The context
    is automatically reset once the last reference to it is released. This
    behavior is different when the primary context was retained by the CUDA
    runtime from CUDA 4.0 and earlier. In this case, the primary context
    remains always active.

    Releasing a primary context that has not been previously retained will
    fail with :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`.

    Please note that unlike :py:obj:`~.cuCtxDestroy()` this method does not
    pop the context from stack in any circumstances.

    Parameters
    ----------
    dev : :py:obj:`~.CUdevice`
        Device which primary context is released

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`

    See Also
    --------
    :py:obj:`~.cuDevicePrimaryCtxRetain`, :py:obj:`~.cuCtxDestroy`, :py:obj:`~.cuCtxGetApiVersion`, :py:obj:`~.cuCtxGetCacheConfig`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxGetLimit`, :py:obj:`~.cuCtxPopCurrent`, :py:obj:`~.cuCtxPushCurrent`, :py:obj:`~.cuCtxSetCacheConfig`, :py:obj:`~.cuCtxSetLimit`, :py:obj:`~.cuCtxSynchronize`
    """
    cdef cydriver.CUdevice cydev
    if dev is None:
        cydev = <cydriver.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cydev = <cydriver.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cydev = <cydriver.CUdevice>pdev
    err = cydriver.cuDevicePrimaryCtxRelease(cydev)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuDevicePrimaryCtxSetFlags(dev, unsigned int flags):
    """ Set flags for the primary context.

    Sets the flags for the primary context on the device overwriting
    perviously set ones.

    The three LSBs of the `flags` parameter can be used to control how the
    OS thread, which owns the CUDA context at the time of an API call,
    interacts with the OS scheduler when waiting for results from the GPU.
    Only one of the scheduling flags can be set when creating a context.

    - :py:obj:`~.CU_CTX_SCHED_SPIN`: Instruct CUDA to actively spin when
      waiting for results from the GPU. This can decrease latency when
      waiting for the GPU, but may lower the performance of CPU threads if
      they are performing work in parallel with the CUDA thread.

    - :py:obj:`~.CU_CTX_SCHED_YIELD`: Instruct CUDA to yield its thread
      when waiting for results from the GPU. This can increase latency when
      waiting for the GPU, but can increase the performance of CPU threads
      performing work in parallel with the GPU.

    - :py:obj:`~.CU_CTX_SCHED_BLOCKING_SYNC`: Instruct CUDA to block the
      CPU thread on a synchronization primitive when waiting for the GPU to
      finish work.

    - :py:obj:`~.CU_CTX_BLOCKING_SYNC`: Instruct CUDA to block the CPU
      thread on a synchronization primitive when waiting for the GPU to
      finish work.   Deprecated: This flag was deprecated as of CUDA 4.0
      and was replaced with :py:obj:`~.CU_CTX_SCHED_BLOCKING_SYNC`.

    - :py:obj:`~.CU_CTX_SCHED_AUTO`: The default value if the `flags`
      parameter is zero, uses a heuristic based on the number of active
      CUDA contexts in the process `C` and the number of logical processors
      in the system `P`. If `C` > `P`, then CUDA will yield to other OS
      threads when waiting for the GPU (:py:obj:`~.CU_CTX_SCHED_YIELD`),
      otherwise CUDA will not yield while waiting for results and actively
      spin on the processor (:py:obj:`~.CU_CTX_SCHED_SPIN`). Additionally,
      on Tegra devices, :py:obj:`~.CU_CTX_SCHED_AUTO` uses a heuristic
      based on the power profile of the platform and may choose
      :py:obj:`~.CU_CTX_SCHED_BLOCKING_SYNC` for low-powered devices.

    - :py:obj:`~.CU_CTX_LMEM_RESIZE_TO_MAX`: Instruct CUDA to not reduce
      local memory after resizing local memory for a kernel. This can
      prevent thrashing by local memory allocations when launching many
      kernels with high local memory usage at the cost of potentially
      increased memory usage.   Deprecated: This flag is deprecated and the
      behavior enabled by this flag is now the default and cannot be
      disabled.

    Parameters
    ----------
    dev : :py:obj:`~.CUdevice`
        Device for which the primary context flags are set
    flags : unsigned int
        New flags for the device

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`,

    See Also
    --------
    :py:obj:`~.cuDevicePrimaryCtxRetain`, :py:obj:`~.cuDevicePrimaryCtxGetState`, :py:obj:`~.cuCtxCreate`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cudaSetDeviceFlags`
    """
    cdef cydriver.CUdevice cydev
    if dev is None:
        cydev = <cydriver.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cydev = <cydriver.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cydev = <cydriver.CUdevice>pdev
    err = cydriver.cuDevicePrimaryCtxSetFlags(cydev, flags)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuDevicePrimaryCtxGetState(dev):
    """ Get the state of the primary context.

    Returns in `*flags` the flags for the primary context of `dev`, and in
    `*active` whether it is active. See
    :py:obj:`~.cuDevicePrimaryCtxSetFlags` for flag values.

    Parameters
    ----------
    dev : :py:obj:`~.CUdevice`
        Device to get primary context flags for

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`,
    flags : unsigned int
        Pointer to store flags
    active : int
        Pointer to store context state; 0 = inactive, 1 = active

    See Also
    --------
    :py:obj:`~.cuDevicePrimaryCtxSetFlags`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cudaGetDeviceFlags`
    """
    cdef cydriver.CUdevice cydev
    if dev is None:
        cydev = <cydriver.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cydev = <cydriver.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cydev = <cydriver.CUdevice>pdev
    cdef unsigned int flags = 0
    cdef int active = 0
    err = cydriver.cuDevicePrimaryCtxGetState(cydev, &flags, &active)
    return (CUresult(err), flags, active)

@cython.embedsignature(True)
def cuDevicePrimaryCtxReset(dev):
    """ Destroy all allocations and reset all state on the primary context.

    Explicitly destroys and cleans up all resources associated with the
    current device in the current process.

    Note that it is responsibility of the calling function to ensure that
    no other module in the process is using the device any more. For that
    reason it is recommended to use :py:obj:`~.cuDevicePrimaryCtxRelease()`
    in most cases. However it is safe for other modules to call
    :py:obj:`~.cuDevicePrimaryCtxRelease()` even after resetting the
    device. Resetting the primary context does not release it, an
    application that has retained the primary context should explicitly
    release its usage.

    Parameters
    ----------
    dev : :py:obj:`~.CUdevice`
        Device for which primary context is destroyed

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`, :py:obj:`~.CUDA_ERROR_PRIMARY_CONTEXT_ACTIVE`

    See Also
    --------
    :py:obj:`~.cuDevicePrimaryCtxRetain`, :py:obj:`~.cuDevicePrimaryCtxRelease`, :py:obj:`~.cuCtxGetApiVersion`, :py:obj:`~.cuCtxGetCacheConfig`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxGetLimit`, :py:obj:`~.cuCtxPopCurrent`, :py:obj:`~.cuCtxPushCurrent`, :py:obj:`~.cuCtxSetCacheConfig`, :py:obj:`~.cuCtxSetLimit`, :py:obj:`~.cuCtxSynchronize`, :py:obj:`~.cudaDeviceReset`
    """
    cdef cydriver.CUdevice cydev
    if dev is None:
        cydev = <cydriver.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cydev = <cydriver.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cydev = <cydriver.CUdevice>pdev
    err = cydriver.cuDevicePrimaryCtxReset(cydev)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuDeviceGetExecAffinitySupport(typename not None : CUexecAffinityType, dev):
    """ Returns information about the execution affinity support of the device.

    Returns in `*pi` whether execution affinity type `typename` is
    supported by device `dev`. The supported types are:

    - :py:obj:`~.CU_EXEC_AFFINITY_TYPE_SM_COUNT`: 1 if context with limited
      SMs is supported by the device, or 0 if not;

    Parameters
    ----------
    typename : :py:obj:`~.CUexecAffinityType`
        Execution affinity type to query
    dev : :py:obj:`~.CUdevice`
        Device handle

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`
    pi : int
        1 if the execution affinity type `typename` is supported by the
        device, or 0 if not

    See Also
    --------
    :py:obj:`~.cuDeviceGetAttribute`, :py:obj:`~.cuDeviceGetCount`, :py:obj:`~.cuDeviceGetName`, :py:obj:`~.cuDeviceGetUuid`, :py:obj:`~.cuDeviceGet`, :py:obj:`~.cuDeviceTotalMem`
    """
    cdef cydriver.CUdevice cydev
    if dev is None:
        cydev = <cydriver.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cydev = <cydriver.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cydev = <cydriver.CUdevice>pdev
    cdef int pi = 0
    cdef cydriver.CUexecAffinityType cytypename = typename.value
    err = cydriver.cuDeviceGetExecAffinitySupport(&pi, cytypename, cydev)
    return (CUresult(err), pi)

@cython.embedsignature(True)
def cuCtxCreate(unsigned int flags, dev):
    """ Create a CUDA context.

    Creates a new CUDA context and associates it with the calling thread.
    The `flags` parameter is described below. The context is created with a
    usage count of 1 and the caller of :py:obj:`~.cuCtxCreate()` must call
    :py:obj:`~.cuCtxDestroy()` or when done using the context. If a context
    is already current to the thread, it is supplanted by the newly created
    context and may be restored by a subsequent call to
    :py:obj:`~.cuCtxPopCurrent()`.

    The three LSBs of the `flags` parameter can be used to control how the
    OS thread, which owns the CUDA context at the time of an API call,
    interacts with the OS scheduler when waiting for results from the GPU.
    Only one of the scheduling flags can be set when creating a context.

    - :py:obj:`~.CU_CTX_SCHED_SPIN`: Instruct CUDA to actively spin when
      waiting for results from the GPU. This can decrease latency when
      waiting for the GPU, but may lower the performance of CPU threads if
      they are performing work in parallel with the CUDA thread.

    - :py:obj:`~.CU_CTX_SCHED_YIELD`: Instruct CUDA to yield its thread
      when waiting for results from the GPU. This can increase latency when
      waiting for the GPU, but can increase the performance of CPU threads
      performing work in parallel with the GPU.

    - :py:obj:`~.CU_CTX_SCHED_BLOCKING_SYNC`: Instruct CUDA to block the
      CPU thread on a synchronization primitive when waiting for the GPU to
      finish work.

    - :py:obj:`~.CU_CTX_BLOCKING_SYNC`: Instruct CUDA to block the CPU
      thread on a synchronization primitive when waiting for the GPU to
      finish work.   Deprecated: This flag was deprecated as of CUDA 4.0
      and was replaced with :py:obj:`~.CU_CTX_SCHED_BLOCKING_SYNC`.

    - :py:obj:`~.CU_CTX_SCHED_AUTO`: The default value if the `flags`
      parameter is zero, uses a heuristic based on the number of active
      CUDA contexts in the process `C` and the number of logical processors
      in the system `P`. If `C` > `P`, then CUDA will yield to other OS
      threads when waiting for the GPU (:py:obj:`~.CU_CTX_SCHED_YIELD`),
      otherwise CUDA will not yield while waiting for results and actively
      spin on the processor (:py:obj:`~.CU_CTX_SCHED_SPIN`). Additionally,
      on Tegra devices, :py:obj:`~.CU_CTX_SCHED_AUTO` uses a heuristic
      based on the power profile of the platform and may choose
      :py:obj:`~.CU_CTX_SCHED_BLOCKING_SYNC` for low-powered devices.

    - :py:obj:`~.CU_CTX_MAP_HOST`: Instruct CUDA to support mapped pinned
      allocations. This flag must be set in order to allocate pinned host
      memory that is accessible to the GPU.

    - :py:obj:`~.CU_CTX_LMEM_RESIZE_TO_MAX`: Instruct CUDA to not reduce
      local memory after resizing local memory for a kernel. This can
      prevent thrashing by local memory allocations when launching many
      kernels with high local memory usage at the cost of potentially
      increased memory usage.   Deprecated: This flag is deprecated and the
      behavior enabled by this flag is now the default and cannot be
      disabled. Instead, the per-thread stack size can be controlled with
      :py:obj:`~.cuCtxSetLimit()`.

    Context creation will fail with :py:obj:`~.CUDA_ERROR_UNKNOWN` if the
    compute mode of the device is :py:obj:`~.CU_COMPUTEMODE_PROHIBITED`.
    The function :py:obj:`~.cuDeviceGetAttribute()` can be used with
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_COMPUTE_MODE` to determine the compute
    mode of the device. The `nvidia-smi` tool can be used to set the
    compute mode for * devices. Documentation for `nvidia-smi` can be
    obtained by passing a -h option to it.

    Parameters
    ----------
    flags : unsigned int
        Context creation flags
    dev : :py:obj:`~.CUdevice`
        Device to create context on

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_UNKNOWN`
    pctx : :py:obj:`~.CUcontext`
        Returned context handle of the new context

    See Also
    --------
    :py:obj:`~.cuCtxDestroy`, :py:obj:`~.cuCtxGetApiVersion`, :py:obj:`~.cuCtxGetCacheConfig`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxGetLimit`, :py:obj:`~.cuCtxPopCurrent`, :py:obj:`~.cuCtxPushCurrent`, :py:obj:`~.cuCtxSetCacheConfig`, :py:obj:`~.cuCtxSetLimit`, :py:obj:`~.cuCtxSynchronize`

    Notes
    -----
    In most cases it is recommended to use :py:obj:`~.cuDevicePrimaryCtxRetain`.
    """
    cdef cydriver.CUdevice cydev
    if dev is None:
        cydev = <cydriver.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cydev = <cydriver.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cydev = <cydriver.CUdevice>pdev
    cdef CUcontext pctx = CUcontext()
    err = cydriver.cuCtxCreate(<cydriver.CUcontext*>pctx._ptr, flags, cydev)
    return (CUresult(err), pctx)

@cython.embedsignature(True)
def cuCtxCreate_v3(paramsArray : Optional[Tuple[CUexecAffinityParam] | List[CUexecAffinityParam]], int numParams, unsigned int flags, dev):
    """ Create a CUDA context with execution affinity.

    Creates a new CUDA context with execution affinity and associates it
    with the calling thread. The `paramsArray` and `flags` parameter are
    described below. The context is created with a usage count of 1 and the
    caller of :py:obj:`~.cuCtxCreate()` must call
    :py:obj:`~.cuCtxDestroy()` or when done using the context. If a context
    is already current to the thread, it is supplanted by the newly created
    context and may be restored by a subsequent call to
    :py:obj:`~.cuCtxPopCurrent()`.

    The type and the amount of execution resource the context can use is
    limited by `paramsArray` and `numParams`. The `paramsArray` is an array
    of `CUexecAffinityParam` and the `numParams` describes the size of the
    array. If two `CUexecAffinityParam` in the array have the same type,
    the latter execution affinity parameter overrides the former execution
    affinity parameter. The supported execution affinity types are:

    - :py:obj:`~.CU_EXEC_AFFINITY_TYPE_SM_COUNT` limits the portion of SMs
      that the context can use. The portion of SMs is specified as the
      number of SMs via `CUexecAffinitySmCount`. This limit will be
      internally rounded up to the next hardware-supported amount. Hence,
      it is imperative to query the actual execution affinity of the
      context via `cuCtxGetExecAffinity` after context creation. Currently,
      this attribute is only supported under Volta+ MPS.

    The three LSBs of the `flags` parameter can be used to control how the
    OS thread, which owns the CUDA context at the time of an API call,
    interacts with the OS scheduler when waiting for results from the GPU.
    Only one of the scheduling flags can be set when creating a context.

    - :py:obj:`~.CU_CTX_SCHED_SPIN`: Instruct CUDA to actively spin when
      waiting for results from the GPU. This can decrease latency when
      waiting for the GPU, but may lower the performance of CPU threads if
      they are performing work in parallel with the CUDA thread.

    - :py:obj:`~.CU_CTX_SCHED_YIELD`: Instruct CUDA to yield its thread
      when waiting for results from the GPU. This can increase latency when
      waiting for the GPU, but can increase the performance of CPU threads
      performing work in parallel with the GPU.

    - :py:obj:`~.CU_CTX_SCHED_BLOCKING_SYNC`: Instruct CUDA to block the
      CPU thread on a synchronization primitive when waiting for the GPU to
      finish work.

    - :py:obj:`~.CU_CTX_BLOCKING_SYNC`: Instruct CUDA to block the CPU
      thread on a synchronization primitive when waiting for the GPU to
      finish work.   Deprecated: This flag was deprecated as of CUDA 4.0
      and was replaced with :py:obj:`~.CU_CTX_SCHED_BLOCKING_SYNC`.

    - :py:obj:`~.CU_CTX_SCHED_AUTO`: The default value if the `flags`
      parameter is zero, uses a heuristic based on the number of active
      CUDA contexts in the process `C` and the number of logical processors
      in the system `P`. If `C` > `P`, then CUDA will yield to other OS
      threads when waiting for the GPU (:py:obj:`~.CU_CTX_SCHED_YIELD`),
      otherwise CUDA will not yield while waiting for results and actively
      spin on the processor (:py:obj:`~.CU_CTX_SCHED_SPIN`). Additionally,
      on Tegra devices, :py:obj:`~.CU_CTX_SCHED_AUTO` uses a heuristic
      based on the power profile of the platform and may choose
      :py:obj:`~.CU_CTX_SCHED_BLOCKING_SYNC` for low-powered devices.

    - :py:obj:`~.CU_CTX_MAP_HOST`: Instruct CUDA to support mapped pinned
      allocations. This flag must be set in order to allocate pinned host
      memory that is accessible to the GPU.

    - :py:obj:`~.CU_CTX_LMEM_RESIZE_TO_MAX`: Instruct CUDA to not reduce
      local memory after resizing local memory for a kernel. This can
      prevent thrashing by local memory allocations when launching many
      kernels with high local memory usage at the cost of potentially
      increased memory usage.   Deprecated: This flag is deprecated and the
      behavior enabled by this flag is now the default and cannot be
      disabled. Instead, the per-thread stack size can be controlled with
      :py:obj:`~.cuCtxSetLimit()`.

    Context creation will fail with :py:obj:`~.CUDA_ERROR_UNKNOWN` if the
    compute mode of the device is :py:obj:`~.CU_COMPUTEMODE_PROHIBITED`.
    The function :py:obj:`~.cuDeviceGetAttribute()` can be used with
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_COMPUTE_MODE` to determine the compute
    mode of the device. The `nvidia-smi` tool can be used to set the
    compute mode for * devices. Documentation for `nvidia-smi` can be
    obtained by passing a -h option to it.

    Parameters
    ----------
    paramsArray : List[:py:obj:`~.CUexecAffinityParam`]
        Execution affinity parameters
    numParams : int
        Number of execution affinity parameters
    flags : unsigned int
        Context creation flags
    dev : :py:obj:`~.CUdevice`
        Device to create context on

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_UNSUPPORTED_EXEC_AFFINITY`, :py:obj:`~.CUDA_ERROR_UNKNOWN`
    pctx : :py:obj:`~.CUcontext`
        Returned context handle of the new context

    See Also
    --------
    :py:obj:`~.cuCtxDestroy`, :py:obj:`~.cuCtxGetApiVersion`, :py:obj:`~.cuCtxGetCacheConfig`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxGetLimit`, :py:obj:`~.cuCtxPopCurrent`, :py:obj:`~.cuCtxPushCurrent`, :py:obj:`~.cuCtxSetCacheConfig`, :py:obj:`~.cuCtxSetLimit`, :py:obj:`~.cuCtxSynchronize`, :py:obj:`~.CUexecAffinityParam`
    """
    cdef cydriver.CUdevice cydev
    if dev is None:
        cydev = <cydriver.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cydev = <cydriver.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cydev = <cydriver.CUdevice>pdev
    paramsArray = [] if paramsArray is None else paramsArray
    if not all(isinstance(_x, (CUexecAffinityParam,)) for _x in paramsArray):
        raise TypeError("Argument 'paramsArray' is not instance of type (expected Tuple[cydriver.CUexecAffinityParam,] or List[cydriver.CUexecAffinityParam,]")
    cdef CUcontext pctx = CUcontext()
    cdef cydriver.CUexecAffinityParam* cyparamsArray = NULL
    if len(paramsArray) > 0:
        cyparamsArray = <cydriver.CUexecAffinityParam*> calloc(len(paramsArray), sizeof(cydriver.CUexecAffinityParam))
        if cyparamsArray is NULL:
            raise MemoryError('Failed to allocate length x size memory: ' + str(len(paramsArray)) + 'x' + str(sizeof(cydriver.CUexecAffinityParam)))
        for idx in range(len(paramsArray)):
            string.memcpy(&cyparamsArray[idx], (<CUexecAffinityParam>paramsArray[idx])._ptr, sizeof(cydriver.CUexecAffinityParam))
    err = cydriver.cuCtxCreate_v3(<cydriver.CUcontext*>pctx._ptr, (<CUexecAffinityParam>paramsArray[0])._ptr if len(paramsArray) == 1 else cyparamsArray, numParams, flags, cydev)
    if cyparamsArray is not NULL:
        free(cyparamsArray)
    return (CUresult(err), pctx)

@cython.embedsignature(True)
def cuCtxDestroy(ctx):
    """ Destroy a CUDA context.

    Destroys the CUDA context specified by `ctx`. The context `ctx` will be
    destroyed regardless of how many threads it is current to. It is the
    responsibility of the calling function to ensure that no API call
    issues using `ctx` while :py:obj:`~.cuCtxDestroy()` is executing.

    Destroys and cleans up all resources associated with the context. It is
    the caller's responsibility to ensure that the context or its resources
    are not accessed or passed in subsequent API calls and doing so will
    result in undefined behavior. These resources include CUDA types such
    as :py:obj:`~.CUmodule`, :py:obj:`~.CUfunction`, :py:obj:`~.CUstream`,
    :py:obj:`~.CUevent`, :py:obj:`~.CUarray`, :py:obj:`~.CUmipmappedArray`,
    :py:obj:`~.CUtexObject`, :py:obj:`~.CUsurfObject`,
    :py:obj:`~.CUtexref`, :py:obj:`~.CUsurfref`,
    :py:obj:`~.CUgraphicsResource`, :py:obj:`~.CUlinkState`,
    :py:obj:`~.CUexternalMemory` and :py:obj:`~.CUexternalSemaphore`.

    If `ctx` is current to the calling thread then `ctx` will also be
    popped from the current thread's context stack (as though
    :py:obj:`~.cuCtxPopCurrent()` were called). If `ctx` is current to
    other threads, then `ctx` will remain current to those threads, and
    attempting to access `ctx` from those threads will result in the error
    :py:obj:`~.CUDA_ERROR_CONTEXT_IS_DESTROYED`.

    Parameters
    ----------
    ctx : :py:obj:`~.CUcontext`
        Context to destroy

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuCtxCreate`, :py:obj:`~.cuCtxGetApiVersion`, :py:obj:`~.cuCtxGetCacheConfig`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxGetLimit`, :py:obj:`~.cuCtxPopCurrent`, :py:obj:`~.cuCtxPushCurrent`, :py:obj:`~.cuCtxSetCacheConfig`, :py:obj:`~.cuCtxSetLimit`, :py:obj:`~.cuCtxSynchronize`
    """
    cdef cydriver.CUcontext cyctx
    if ctx is None:
        cyctx = <cydriver.CUcontext><void_ptr>0
    elif isinstance(ctx, (CUcontext,)):
        pctx = int(ctx)
        cyctx = <cydriver.CUcontext><void_ptr>pctx
    else:
        pctx = int(CUcontext(ctx))
        cyctx = <cydriver.CUcontext><void_ptr>pctx
    err = cydriver.cuCtxDestroy(cyctx)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuCtxPushCurrent(ctx):
    """ Pushes a context on the current CPU thread.

    Pushes the given context `ctx` onto the CPU thread's stack of current
    contexts. The specified context becomes the CPU thread's current
    context, so all CUDA functions that operate on the current context are
    affected.

    The previous current context may be made current again by calling
    :py:obj:`~.cuCtxDestroy()` or :py:obj:`~.cuCtxPopCurrent()`.

    Parameters
    ----------
    ctx : :py:obj:`~.CUcontext`
        Context to push

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuCtxCreate`, :py:obj:`~.cuCtxDestroy`, :py:obj:`~.cuCtxGetApiVersion`, :py:obj:`~.cuCtxGetCacheConfig`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxGetLimit`, :py:obj:`~.cuCtxPopCurrent`, :py:obj:`~.cuCtxSetCacheConfig`, :py:obj:`~.cuCtxSetLimit`, :py:obj:`~.cuCtxSynchronize`
    """
    cdef cydriver.CUcontext cyctx
    if ctx is None:
        cyctx = <cydriver.CUcontext><void_ptr>0
    elif isinstance(ctx, (CUcontext,)):
        pctx = int(ctx)
        cyctx = <cydriver.CUcontext><void_ptr>pctx
    else:
        pctx = int(CUcontext(ctx))
        cyctx = <cydriver.CUcontext><void_ptr>pctx
    err = cydriver.cuCtxPushCurrent(cyctx)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuCtxPopCurrent():
    """ Pops the current CUDA context from the current CPU thread.

    Pops the current CUDA context from the CPU thread and passes back the
    old context handle in `*pctx`. That context may then be made current to
    a different CPU thread by calling :py:obj:`~.cuCtxPushCurrent()`.

    If a context was current to the CPU thread before
    :py:obj:`~.cuCtxCreate()` or :py:obj:`~.cuCtxPushCurrent()` was called,
    this function makes that context current to the CPU thread again.

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`
    pctx : :py:obj:`~.CUcontext`
        Returned popped context handle

    See Also
    --------
    :py:obj:`~.cuCtxCreate`, :py:obj:`~.cuCtxDestroy`, :py:obj:`~.cuCtxGetApiVersion`, :py:obj:`~.cuCtxGetCacheConfig`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxGetLimit`, :py:obj:`~.cuCtxPushCurrent`, :py:obj:`~.cuCtxSetCacheConfig`, :py:obj:`~.cuCtxSetLimit`, :py:obj:`~.cuCtxSynchronize`
    """
    cdef CUcontext pctx = CUcontext()
    err = cydriver.cuCtxPopCurrent(<cydriver.CUcontext*>pctx._ptr)
    return (CUresult(err), pctx)

@cython.embedsignature(True)
def cuCtxSetCurrent(ctx):
    """ Binds the specified CUDA context to the calling CPU thread.

    Binds the specified CUDA context to the calling CPU thread. If `ctx` is
    NULL then the CUDA context previously bound to the calling CPU thread
    is unbound and :py:obj:`~.CUDA_SUCCESS` is returned.

    If there exists a CUDA context stack on the calling CPU thread, this
    will replace the top of that stack with `ctx`. If `ctx` is NULL then
    this will be equivalent to popping the top of the calling CPU thread's
    CUDA context stack (or a no-op if the calling CPU thread's CUDA context
    stack is empty).

    Parameters
    ----------
    ctx : :py:obj:`~.CUcontext`
        Context to bind to the calling CPU thread

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`

    See Also
    --------
    :py:obj:`~.cuCtxGetCurrent`, :py:obj:`~.cuCtxCreate`, :py:obj:`~.cuCtxDestroy`, :py:obj:`~.cudaSetDevice`
    """
    cdef cydriver.CUcontext cyctx
    if ctx is None:
        cyctx = <cydriver.CUcontext><void_ptr>0
    elif isinstance(ctx, (CUcontext,)):
        pctx = int(ctx)
        cyctx = <cydriver.CUcontext><void_ptr>pctx
    else:
        pctx = int(CUcontext(ctx))
        cyctx = <cydriver.CUcontext><void_ptr>pctx
    err = cydriver.cuCtxSetCurrent(cyctx)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuCtxGetCurrent():
    """ Returns the CUDA context bound to the calling CPU thread.

    Returns in `*pctx` the CUDA context bound to the calling CPU thread. If
    no context is bound to the calling CPU thread then `*pctx` is set to
    NULL and :py:obj:`~.CUDA_SUCCESS` is returned.

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`,
    pctx : :py:obj:`~.CUcontext`
        Returned context handle

    See Also
    --------
    :py:obj:`~.cuCtxSetCurrent`, :py:obj:`~.cuCtxCreate`, :py:obj:`~.cuCtxDestroy`, :py:obj:`~.cudaGetDevice`
    """
    cdef CUcontext pctx = CUcontext()
    err = cydriver.cuCtxGetCurrent(<cydriver.CUcontext*>pctx._ptr)
    return (CUresult(err), pctx)

@cython.embedsignature(True)
def cuCtxGetDevice():
    """ Returns the device ID for the current context.

    Returns in `*device` the ordinal of the current context's device.

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`,
    device : :py:obj:`~.CUdevice`
        Returned device ID for the current context

    See Also
    --------
    :py:obj:`~.cuCtxCreate`, :py:obj:`~.cuCtxDestroy`, :py:obj:`~.cuCtxGetApiVersion`, :py:obj:`~.cuCtxGetCacheConfig`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxGetLimit`, :py:obj:`~.cuCtxPopCurrent`, :py:obj:`~.cuCtxPushCurrent`, :py:obj:`~.cuCtxSetCacheConfig`, :py:obj:`~.cuCtxSetLimit`, :py:obj:`~.cuCtxSynchronize`, :py:obj:`~.cudaGetDevice`
    """
    cdef CUdevice device = CUdevice()
    err = cydriver.cuCtxGetDevice(<cydriver.CUdevice*>device._ptr)
    return (CUresult(err), device)

@cython.embedsignature(True)
def cuCtxGetFlags():
    """ Returns the flags for the current context.

    Returns in `*flags` the flags of the current context. See
    :py:obj:`~.cuCtxCreate` for flag values.

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`,
    flags : unsigned int
        Pointer to store flags of current context

    See Also
    --------
    :py:obj:`~.cuCtxCreate`, :py:obj:`~.cuCtxGetApiVersion`, :py:obj:`~.cuCtxGetCacheConfig`, :py:obj:`~.cuCtxGetCurrent`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetLimit`, :py:obj:`~.cuCtxGetSharedMemConfig`, :py:obj:`~.cuCtxGetStreamPriorityRange`, :py:obj:`~.cudaGetDeviceFlags`
    """
    cdef unsigned int flags = 0
    err = cydriver.cuCtxGetFlags(&flags)
    return (CUresult(err), flags)

@cython.embedsignature(True)
def cuCtxSynchronize():
    """ Block for a context's tasks to complete.

    Blocks until the device has completed all preceding requested tasks.
    :py:obj:`~.cuCtxSynchronize()` returns an error if one of the preceding
    tasks failed. If the context was created with the
    :py:obj:`~.CU_CTX_SCHED_BLOCKING_SYNC` flag, the CPU thread will block
    until the GPU context has finished its work.

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`

    See Also
    --------
    :py:obj:`~.cuCtxCreate`, :py:obj:`~.cuCtxDestroy`, :py:obj:`~.cuCtxGetApiVersion`, :py:obj:`~.cuCtxGetCacheConfig`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxGetLimit`, :py:obj:`~.cuCtxPopCurrent`, :py:obj:`~.cuCtxPushCurrent`, :py:obj:`~.cuCtxSetCacheConfig`, :py:obj:`~.cuCtxSetLimit`, :py:obj:`~.cudaDeviceSynchronize`
    """
    err = cydriver.cuCtxSynchronize()
    return (CUresult(err),)

@cython.embedsignature(True)
def cuCtxSetLimit(limit not None : CUlimit, size_t value):
    """ Set resource limits.

    Setting `limit` to `value` is a request by the application to update
    the current limit maintained by the context. The driver is free to
    modify the requested value to meet h/w requirements (this could be
    clamping to minimum or maximum values, rounding up to nearest element
    size, etc). The application can use :py:obj:`~.cuCtxGetLimit()` to find
    out exactly what the limit has been set to.

    Setting each :py:obj:`~.CUlimit` has its own specific restrictions, so
    each is discussed here.

    - :py:obj:`~.CU_LIMIT_STACK_SIZE` controls the stack size in bytes of
      each GPU thread. The driver automatically increases the per-thread
      stack size for each kernel launch as needed. This size isn't reset
      back to the original value after each launch. Setting this value will
      take effect immediately, and if necessary, the device will block
      until all preceding requested tasks are complete.

    - :py:obj:`~.CU_LIMIT_PRINTF_FIFO_SIZE` controls the size in bytes of
      the FIFO used by the :py:obj:`~.printf()` device system call. Setting
      :py:obj:`~.CU_LIMIT_PRINTF_FIFO_SIZE` must be performed before
      launching any kernel that uses the :py:obj:`~.printf()` device system
      call, otherwise :py:obj:`~.CUDA_ERROR_INVALID_VALUE` will be
      returned.

    - :py:obj:`~.CU_LIMIT_MALLOC_HEAP_SIZE` controls the size in bytes of
      the heap used by the :py:obj:`~.malloc()` and :py:obj:`~.free()`
      device system calls. Setting :py:obj:`~.CU_LIMIT_MALLOC_HEAP_SIZE`
      must be performed before launching any kernel that uses the
      :py:obj:`~.malloc()` or :py:obj:`~.free()` device system calls,
      otherwise :py:obj:`~.CUDA_ERROR_INVALID_VALUE` will be returned.

    - :py:obj:`~.CU_LIMIT_DEV_RUNTIME_SYNC_DEPTH` controls the maximum
      nesting depth of a grid at which a thread can safely call
      :py:obj:`~.cudaDeviceSynchronize()`. Setting this limit must be
      performed before any launch of a kernel that uses the device runtime
      and calls :py:obj:`~.cudaDeviceSynchronize()` above the default sync
      depth, two levels of grids. Calls to
      :py:obj:`~.cudaDeviceSynchronize()` will fail with error code
      :py:obj:`~.cudaErrorSyncDepthExceeded` if the limitation is violated.
      This limit can be set smaller than the default or up the maximum
      launch depth of 24. When setting this limit, keep in mind that
      additional levels of sync depth require the driver to reserve large
      amounts of device memory which can no longer be used for user
      allocations. If these reservations of device memory fail,
      :py:obj:`~.cuCtxSetLimit()` will return
      :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, and the limit can be reset to a
      lower value. This limit is only applicable to devices of compute
      capability 3.5 and higher. Attempting to set this limit on devices of
      compute capability less than 3.5 will result in the error
      :py:obj:`~.CUDA_ERROR_UNSUPPORTED_LIMIT` being returned.

    - :py:obj:`~.CU_LIMIT_DEV_RUNTIME_PENDING_LAUNCH_COUNT` controls the
      maximum number of outstanding device runtime launches that can be
      made from the current context. A grid is outstanding from the point
      of launch up until the grid is known to have been completed. Device
      runtime launches which violate this limitation fail and return
      :py:obj:`~.cudaErrorLaunchPendingCountExceeded` when
      :py:obj:`~.cudaGetLastError()` is called after launch. If more
      pending launches than the default (2048 launches) are needed for a
      module using the device runtime, this limit can be increased. Keep in
      mind that being able to sustain additional pending launches will
      require the driver to reserve larger amounts of device memory upfront
      which can no longer be used for allocations. If these reservations
      fail, :py:obj:`~.cuCtxSetLimit()` will return
      :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, and the limit can be reset to a
      lower value. This limit is only applicable to devices of compute
      capability 3.5 and higher. Attempting to set this limit on devices of
      compute capability less than 3.5 will result in the error
      :py:obj:`~.CUDA_ERROR_UNSUPPORTED_LIMIT` being returned.

    - :py:obj:`~.CU_LIMIT_MAX_L2_FETCH_GRANULARITY` controls the L2 cache
      fetch granularity. Values can range from 0B to 128B. This is purely a
      performence hint and it can be ignored or clamped depending on the
      platform.

    - :py:obj:`~.CU_LIMIT_PERSISTING_L2_CACHE_SIZE` controls size in bytes
      availabe for persisting L2 cache. This is purely a performance hint
      and it can be ignored or clamped depending on the platform.

    Parameters
    ----------
    limit : :py:obj:`~.CUlimit`
        Limit to set
    value : size_t
        Size of limit

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_UNSUPPORTED_LIMIT`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`

    See Also
    --------
    :py:obj:`~.cuCtxCreate`, :py:obj:`~.cuCtxDestroy`, :py:obj:`~.cuCtxGetApiVersion`, :py:obj:`~.cuCtxGetCacheConfig`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxGetLimit`, :py:obj:`~.cuCtxPopCurrent`, :py:obj:`~.cuCtxPushCurrent`, :py:obj:`~.cuCtxSetCacheConfig`, :py:obj:`~.cuCtxSynchronize`, :py:obj:`~.cudaDeviceSetLimit`
    """
    cdef cydriver.CUlimit cylimit = limit.value
    err = cydriver.cuCtxSetLimit(cylimit, value)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuCtxGetLimit(limit not None : CUlimit):
    """ Returns resource limits.

    Returns in `*pvalue` the current size of `limit`. The supported
    :py:obj:`~.CUlimit` values are:

    - :py:obj:`~.CU_LIMIT_STACK_SIZE`: stack size in bytes of each GPU
      thread.

    - :py:obj:`~.CU_LIMIT_PRINTF_FIFO_SIZE`: size in bytes of the FIFO used
      by the :py:obj:`~.printf()` device system call.

    - :py:obj:`~.CU_LIMIT_MALLOC_HEAP_SIZE`: size in bytes of the heap used
      by the :py:obj:`~.malloc()` and :py:obj:`~.free()` device system
      calls.

    - :py:obj:`~.CU_LIMIT_DEV_RUNTIME_SYNC_DEPTH`: maximum grid depth at
      which a thread can issue the device runtime call
      :py:obj:`~.cudaDeviceSynchronize()` to wait on child grid launches to
      complete.

    - :py:obj:`~.CU_LIMIT_DEV_RUNTIME_PENDING_LAUNCH_COUNT`: maximum number
      of outstanding device runtime launches that can be made from this
      context.

    - :py:obj:`~.CU_LIMIT_MAX_L2_FETCH_GRANULARITY`: L2 cache fetch
      granularity.

    - :py:obj:`~.CU_LIMIT_PERSISTING_L2_CACHE_SIZE`: Persisting L2 cache
      size in bytes

    Parameters
    ----------
    limit : :py:obj:`~.CUlimit`
        Limit to query

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_UNSUPPORTED_LIMIT`
    pvalue : int
        Returned size of limit

    See Also
    --------
    :py:obj:`~.cuCtxCreate`, :py:obj:`~.cuCtxDestroy`, :py:obj:`~.cuCtxGetApiVersion`, :py:obj:`~.cuCtxGetCacheConfig`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxPopCurrent`, :py:obj:`~.cuCtxPushCurrent`, :py:obj:`~.cuCtxSetCacheConfig`, :py:obj:`~.cuCtxSetLimit`, :py:obj:`~.cuCtxSynchronize`, :py:obj:`~.cudaDeviceGetLimit`
    """
    cdef size_t pvalue = 0
    cdef cydriver.CUlimit cylimit = limit.value
    err = cydriver.cuCtxGetLimit(&pvalue, cylimit)
    return (CUresult(err), pvalue)

@cython.embedsignature(True)
def cuCtxGetCacheConfig():
    """ Returns the preferred cache configuration for the current context.

    On devices where the L1 cache and shared memory use the same hardware
    resources, this function returns through `pconfig` the preferred cache
    configuration for the current context. This is only a preference. The
    driver will use the requested configuration if possible, but it is free
    to choose a different configuration if required to execute functions.

    This will return a `pconfig` of :py:obj:`~.CU_FUNC_CACHE_PREFER_NONE`
    on devices where the size of the L1 cache and shared memory are fixed.

    The supported cache configurations are:

    - :py:obj:`~.CU_FUNC_CACHE_PREFER_NONE`: no preference for shared
      memory or L1 (default)

    - :py:obj:`~.CU_FUNC_CACHE_PREFER_SHARED`: prefer larger shared memory
      and smaller L1 cache

    - :py:obj:`~.CU_FUNC_CACHE_PREFER_L1`: prefer larger L1 cache and
      smaller shared memory

    - :py:obj:`~.CU_FUNC_CACHE_PREFER_EQUAL`: prefer equal sized L1 cache
      and shared memory

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    pconfig : :py:obj:`~.CUfunc_cache`
        Returned cache configuration

    See Also
    --------
    :py:obj:`~.cuCtxCreate`, :py:obj:`~.cuCtxDestroy`, :py:obj:`~.cuCtxGetApiVersion`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxGetLimit`, :py:obj:`~.cuCtxPopCurrent`, :py:obj:`~.cuCtxPushCurrent`, :py:obj:`~.cuCtxSetCacheConfig`, :py:obj:`~.cuCtxSetLimit`, :py:obj:`~.cuCtxSynchronize`, :py:obj:`~.cuFuncSetCacheConfig`, :py:obj:`~.cudaDeviceGetCacheConfig`
    """
    cdef cydriver.CUfunc_cache pconfig
    err = cydriver.cuCtxGetCacheConfig(&pconfig)
    return (CUresult(err), CUfunc_cache(pconfig))

@cython.embedsignature(True)
def cuCtxSetCacheConfig(config not None : CUfunc_cache):
    """ Sets the preferred cache configuration for the current context.

    On devices where the L1 cache and shared memory use the same hardware
    resources, this sets through `config` the preferred cache configuration
    for the current context. This is only a preference. The driver will use
    the requested configuration if possible, but it is free to choose a
    different configuration if required to execute the function. Any
    function preference set via :py:obj:`~.cuFuncSetCacheConfig()` will be
    preferred over this context-wide setting. Setting the context-wide
    cache configuration to :py:obj:`~.CU_FUNC_CACHE_PREFER_NONE` will cause
    subsequent kernel launches to prefer to not change the cache
    configuration unless required to launch the kernel.

    This setting does nothing on devices where the size of the L1 cache and
    shared memory are fixed.

    Launching a kernel with a different preference than the most recent
    preference setting may insert a device-side synchronization point.

    The supported cache configurations are:

    - :py:obj:`~.CU_FUNC_CACHE_PREFER_NONE`: no preference for shared
      memory or L1 (default)

    - :py:obj:`~.CU_FUNC_CACHE_PREFER_SHARED`: prefer larger shared memory
      and smaller L1 cache

    - :py:obj:`~.CU_FUNC_CACHE_PREFER_L1`: prefer larger L1 cache and
      smaller shared memory

    - :py:obj:`~.CU_FUNC_CACHE_PREFER_EQUAL`: prefer equal sized L1 cache
      and shared memory

    Parameters
    ----------
    config : :py:obj:`~.CUfunc_cache`
        Requested cache configuration

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuCtxCreate`, :py:obj:`~.cuCtxDestroy`, :py:obj:`~.cuCtxGetApiVersion`, :py:obj:`~.cuCtxGetCacheConfig`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxGetLimit`, :py:obj:`~.cuCtxPopCurrent`, :py:obj:`~.cuCtxPushCurrent`, :py:obj:`~.cuCtxSetLimit`, :py:obj:`~.cuCtxSynchronize`, :py:obj:`~.cuFuncSetCacheConfig`, :py:obj:`~.cudaDeviceSetCacheConfig`
    """
    cdef cydriver.CUfunc_cache cyconfig = config.value
    err = cydriver.cuCtxSetCacheConfig(cyconfig)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuCtxGetSharedMemConfig():
    """ Returns the current shared memory configuration for the current context.

    This function will return in `pConfig` the current size of shared
    memory banks in the current context. On devices with configurable
    shared memory banks, :py:obj:`~.cuCtxSetSharedMemConfig` can be used to
    change this setting, so that all subsequent kernel launches will by
    default use the new bank size. When :py:obj:`~.cuCtxGetSharedMemConfig`
    is called on devices without configurable shared memory, it will return
    the fixed bank size of the hardware.

    The returned bank configurations can be either:

    - :py:obj:`~.CU_SHARED_MEM_CONFIG_FOUR_BYTE_BANK_SIZE`: shared memory
      bank width is four bytes.

    - :py:obj:`~.CU_SHARED_MEM_CONFIG_EIGHT_BYTE_BANK_SIZE`: shared memory
      bank width will eight bytes.

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    pConfig : :py:obj:`~.CUsharedconfig`
        returned shared memory configuration

    See Also
    --------
    :py:obj:`~.cuCtxCreate`, :py:obj:`~.cuCtxDestroy`, :py:obj:`~.cuCtxGetApiVersion`, :py:obj:`~.cuCtxGetCacheConfig`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxGetLimit`, :py:obj:`~.cuCtxPopCurrent`, :py:obj:`~.cuCtxPushCurrent`, :py:obj:`~.cuCtxSetLimit`, :py:obj:`~.cuCtxSynchronize`, :py:obj:`~.cuCtxGetSharedMemConfig`, :py:obj:`~.cuFuncSetCacheConfig`, :py:obj:`~.cudaDeviceGetSharedMemConfig`
    """
    cdef cydriver.CUsharedconfig pConfig
    err = cydriver.cuCtxGetSharedMemConfig(&pConfig)
    return (CUresult(err), CUsharedconfig(pConfig))

@cython.embedsignature(True)
def cuCtxSetSharedMemConfig(config not None : CUsharedconfig):
    """ Sets the shared memory configuration for the current context.

    On devices with configurable shared memory banks, this function will
    set the context's shared memory bank size which is used for subsequent
    kernel launches.

    Changed the shared memory configuration between launches may insert a
    device side synchronization point between those launches.

    Changing the shared memory bank size will not increase shared memory
    usage or affect occupancy of kernels, but may have major effects on
    performance. Larger bank sizes will allow for greater potential
    bandwidth to shared memory, but will change what kinds of accesses to
    shared memory will result in bank conflicts.

    This function will do nothing on devices with fixed shared memory bank
    size.

    The supported bank configurations are:

    - :py:obj:`~.CU_SHARED_MEM_CONFIG_DEFAULT_BANK_SIZE`: set bank width to
      the default initial setting (currently, four bytes).

    - :py:obj:`~.CU_SHARED_MEM_CONFIG_FOUR_BYTE_BANK_SIZE`: set shared
      memory bank width to be natively four bytes.

    - :py:obj:`~.CU_SHARED_MEM_CONFIG_EIGHT_BYTE_BANK_SIZE`: set shared
      memory bank width to be natively eight bytes.

    Parameters
    ----------
    config : :py:obj:`~.CUsharedconfig`
        requested shared memory configuration

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuCtxCreate`, :py:obj:`~.cuCtxDestroy`, :py:obj:`~.cuCtxGetApiVersion`, :py:obj:`~.cuCtxGetCacheConfig`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxGetLimit`, :py:obj:`~.cuCtxPopCurrent`, :py:obj:`~.cuCtxPushCurrent`, :py:obj:`~.cuCtxSetLimit`, :py:obj:`~.cuCtxSynchronize`, :py:obj:`~.cuCtxGetSharedMemConfig`, :py:obj:`~.cuFuncSetCacheConfig`, :py:obj:`~.cudaDeviceSetSharedMemConfig`
    """
    cdef cydriver.CUsharedconfig cyconfig = config.value
    err = cydriver.cuCtxSetSharedMemConfig(cyconfig)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuCtxGetApiVersion(ctx):
    """ Gets the context's API version.

    Returns a version number in `version` corresponding to the capabilities
    of the context (e.g. 3010 or 3020), which library developers can use to
    direct callers to a specific API version. If `ctx` is NULL, returns the
    API version used to create the currently bound context.

    Note that new API versions are only introduced when context
    capabilities are changed that break binary compatibility, so the API
    version and driver version may be different. For example, it is valid
    for the API version to be 3020 while the driver version is 4020.

    Parameters
    ----------
    ctx : :py:obj:`~.CUcontext`
        Context to check

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_UNKNOWN`
    version : unsigned int
        Pointer to version

    See Also
    --------
    :py:obj:`~.cuCtxCreate`, :py:obj:`~.cuCtxDestroy`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxGetLimit`, :py:obj:`~.cuCtxPopCurrent`, :py:obj:`~.cuCtxPushCurrent`, :py:obj:`~.cuCtxSetCacheConfig`, :py:obj:`~.cuCtxSetLimit`, :py:obj:`~.cuCtxSynchronize`
    """
    cdef cydriver.CUcontext cyctx
    if ctx is None:
        cyctx = <cydriver.CUcontext><void_ptr>0
    elif isinstance(ctx, (CUcontext,)):
        pctx = int(ctx)
        cyctx = <cydriver.CUcontext><void_ptr>pctx
    else:
        pctx = int(CUcontext(ctx))
        cyctx = <cydriver.CUcontext><void_ptr>pctx
    cdef unsigned int version = 0
    err = cydriver.cuCtxGetApiVersion(cyctx, &version)
    return (CUresult(err), version)

@cython.embedsignature(True)
def cuCtxGetStreamPriorityRange():
    """ Returns numerical values that correspond to the least and greatest stream priorities.

    Returns in `*leastPriority` and `*greatestPriority` the numerical
    values that correspond to the least and greatest stream priorities
    respectively. Stream priorities follow a convention where lower numbers
    imply greater priorities. The range of meaningful stream priorities is
    given by [`*greatestPriority`, `*leastPriority`]. If the user attempts
    to create a stream with a priority value that is outside the meaningful
    range as specified by this API, the priority is automatically clamped
    down or up to either `*leastPriority` or `*greatestPriority`
    respectively. See :py:obj:`~.cuStreamCreateWithPriority` for details on
    creating a priority stream. A NULL may be passed in for
    `*leastPriority` or `*greatestPriority` if the value is not desired.

    This function will return '0' in both `*leastPriority` and
    `*greatestPriority` if the current context's device does not support
    stream priorities (see :py:obj:`~.cuDeviceGetAttribute`).

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`,
    leastPriority : int
        Pointer to an int in which the numerical value for least stream
        priority is returned
    greatestPriority : int
        Pointer to an int in which the numerical value for greatest stream
        priority is returned

    See Also
    --------
    :py:obj:`~.cuStreamCreateWithPriority`, :py:obj:`~.cuStreamGetPriority`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxSetLimit`, :py:obj:`~.cuCtxSynchronize`, :py:obj:`~.cudaDeviceGetStreamPriorityRange`
    """
    cdef int leastPriority = 0
    cdef int greatestPriority = 0
    err = cydriver.cuCtxGetStreamPriorityRange(&leastPriority, &greatestPriority)
    return (CUresult(err), leastPriority, greatestPriority)

@cython.embedsignature(True)
def cuCtxResetPersistingL2Cache():
    """ Resets all persisting lines in cache to normal status.

    :py:obj:`~.cuCtxResetPersistingL2Cache` Resets all persisting lines in
    cache to normal status. Takes effect on function return.

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_NOT_SUPPORTED`

    See Also
    --------
    :py:obj:`~.CUaccessPolicyWindow`
    """
    err = cydriver.cuCtxResetPersistingL2Cache()
    return (CUresult(err),)

@cython.embedsignature(True)
def cuCtxGetExecAffinity(typename not None : CUexecAffinityType):
    """ Returns the execution affinity setting for the current context.

    Returns in `*pExecAffinity` the current value of `typename`. The
    supported :py:obj:`~.CUexecAffinityType` values are:

    - :py:obj:`~.CU_EXEC_AFFINITY_TYPE_SM_COUNT`: number of SMs the context
      is limited to use.

    Parameters
    ----------
    typename : :py:obj:`~.CUexecAffinityType`
        Execution affinity type to query

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_UNSUPPORTED_EXEC_AFFINITY`
    pExecAffinity : :py:obj:`~.CUexecAffinityParam`
        Returned execution affinity

    See Also
    --------
    :py:obj:`~.CUexecAffinityParam`
    """
    cdef CUexecAffinityParam pExecAffinity = CUexecAffinityParam()
    cdef cydriver.CUexecAffinityType cytypename = typename.value
    err = cydriver.cuCtxGetExecAffinity(<cydriver.CUexecAffinityParam*>pExecAffinity._ptr, cytypename)
    return (CUresult(err), pExecAffinity)

@cython.embedsignature(True)
def cuCtxAttach(unsigned int flags):
    """ Increment a context's usage-count.

    [Deprecated]

    Note that this function is deprecated and should not be used.

    Increments the usage count of the context and passes back a context
    handle in `*pctx` that must be passed to :py:obj:`~.cuCtxDetach()` when
    the application is done with the context. :py:obj:`~.cuCtxAttach()`
    fails if there is no context current to the thread.

    Currently, the `flags` parameter must be 0.

    Parameters
    ----------
    flags : unsigned int
        Context attach flags (must be 0)

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    pctx : :py:obj:`~.CUcontext`
        Returned context handle of the current context

    See Also
    --------
    :py:obj:`~.cuCtxCreate`, :py:obj:`~.cuCtxDestroy`, :py:obj:`~.cuCtxDetach`, :py:obj:`~.cuCtxGetApiVersion`, :py:obj:`~.cuCtxGetCacheConfig`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxGetLimit`, :py:obj:`~.cuCtxPopCurrent`, :py:obj:`~.cuCtxPushCurrent`, :py:obj:`~.cuCtxSetCacheConfig`, :py:obj:`~.cuCtxSetLimit`, :py:obj:`~.cuCtxSynchronize`
    """
    cdef CUcontext pctx = CUcontext()
    err = cydriver.cuCtxAttach(<cydriver.CUcontext*>pctx._ptr, flags)
    return (CUresult(err), pctx)

@cython.embedsignature(True)
def cuCtxDetach(ctx):
    """ Decrement a context's usage-count.

    [Deprecated]

    Note that this function is deprecated and should not be used.

    Decrements the usage count of the context `ctx`, and destroys the
    context if the usage count goes to 0. The context must be a handle that
    was passed back by :py:obj:`~.cuCtxCreate()` or
    :py:obj:`~.cuCtxAttach()`, and must be current to the calling thread.

    Parameters
    ----------
    ctx : :py:obj:`~.CUcontext`
        Context to destroy

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`

    See Also
    --------
    :py:obj:`~.cuCtxCreate`, :py:obj:`~.cuCtxDestroy`, :py:obj:`~.cuCtxGetApiVersion`, :py:obj:`~.cuCtxGetCacheConfig`, :py:obj:`~.cuCtxGetDevice`, :py:obj:`~.cuCtxGetFlags`, :py:obj:`~.cuCtxGetLimit`, :py:obj:`~.cuCtxPopCurrent`, :py:obj:`~.cuCtxPushCurrent`, :py:obj:`~.cuCtxSetCacheConfig`, :py:obj:`~.cuCtxSetLimit`, :py:obj:`~.cuCtxSynchronize`
    """
    cdef cydriver.CUcontext cyctx
    if ctx is None:
        cyctx = <cydriver.CUcontext><void_ptr>0
    elif isinstance(ctx, (CUcontext,)):
        pctx = int(ctx)
        cyctx = <cydriver.CUcontext><void_ptr>pctx
    else:
        pctx = int(CUcontext(ctx))
        cyctx = <cydriver.CUcontext><void_ptr>pctx
    err = cydriver.cuCtxDetach(cyctx)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuModuleLoad(char* fname):
    """ Loads a compute module.

    Takes a filename `fname` and loads the corresponding module `module`
    into the current context. The CUDA driver API does not attempt to
    lazily allocate the resources needed by a module; if the memory for
    functions and data (constant and global) needed by the module cannot be
    allocated, :py:obj:`~.cuModuleLoad()` fails. The file should be a
    `cubin` file as output by nvcc, or a `PTX` file either as output by
    nvcc or handwritten, or a `fatbin` file as output by nvcc from
    toolchain 4.0 or later.

    Parameters
    ----------
    fname : bytes
        Filename of module to load

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_PTX`, :py:obj:`~.CUDA_ERROR_UNSUPPORTED_PTX_VERSION`, :py:obj:`~.CUDA_ERROR_NOT_FOUND`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_FILE_NOT_FOUND`, :py:obj:`~.CUDA_ERROR_NO_BINARY_FOR_GPU`, :py:obj:`~.CUDA_ERROR_SHARED_OBJECT_SYMBOL_NOT_FOUND`, :py:obj:`~.CUDA_ERROR_SHARED_OBJECT_INIT_FAILED`, :py:obj:`~.CUDA_ERROR_JIT_COMPILER_NOT_FOUND`
    module : :py:obj:`~.CUmodule`
        Returned module

    See Also
    --------
    :py:obj:`~.cuModuleGetFunction`, :py:obj:`~.cuModuleGetGlobal`, :py:obj:`~.cuModuleGetTexRef`, :py:obj:`~.cuModuleLoadData`, :py:obj:`~.cuModuleLoadDataEx`, :py:obj:`~.cuModuleLoadFatBinary`, :py:obj:`~.cuModuleUnload`
    """
    cdef CUmodule module = CUmodule()
    err = cydriver.cuModuleLoad(<cydriver.CUmodule*>module._ptr, fname)
    return (CUresult(err), module)

@cython.embedsignature(True)
def cuModuleLoadData(image):
    """ Load a module's data.

    Takes a pointer `image` and loads the corresponding module `module`
    into the current context. The pointer may be obtained by mapping a
    `cubin` or `PTX` or `fatbin` file, passing a `cubin` or `PTX` or
    `fatbin` file as a NULL-terminated text string, or incorporating a
    `cubin` or `fatbin` object into the executable resources and using
    operating system calls such as Windows `FindResource()` to obtain the
    pointer.

    Parameters
    ----------
    image : Any
        Module data to load

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_PTX`, :py:obj:`~.CUDA_ERROR_UNSUPPORTED_PTX_VERSION`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_NO_BINARY_FOR_GPU`, :py:obj:`~.CUDA_ERROR_SHARED_OBJECT_SYMBOL_NOT_FOUND`, :py:obj:`~.CUDA_ERROR_SHARED_OBJECT_INIT_FAILED`, :py:obj:`~.CUDA_ERROR_JIT_COMPILER_NOT_FOUND`
    module : :py:obj:`~.CUmodule`
        Returned module

    See Also
    --------
    :py:obj:`~.cuModuleGetFunction`, :py:obj:`~.cuModuleGetGlobal`, :py:obj:`~.cuModuleGetTexRef`, :py:obj:`~.cuModuleLoad`, :py:obj:`~.cuModuleLoadDataEx`, :py:obj:`~.cuModuleLoadFatBinary`, :py:obj:`~.cuModuleUnload`
    """
    cdef CUmodule module = CUmodule()
    cyimage = utils.HelperInputVoidPtr(image)
    cdef void* cyimage_ptr = <void*><void_ptr>cyimage.cptr
    err = cydriver.cuModuleLoadData(<cydriver.CUmodule*>module._ptr, cyimage_ptr)
    return (CUresult(err), module)

@cython.embedsignature(True)
def cuModuleLoadDataEx(image, unsigned int numOptions, options : Optional[Tuple[CUjit_option] | List[CUjit_option]], optionValues : Optional[Tuple[Any] | List[Any]]):
    """ Load a module's data with options.

    Takes a pointer `image` and loads the corresponding module `module`
    into the current context. The pointer may be obtained by mapping a
    `cubin` or `PTX` or `fatbin` file, passing a `cubin` or `PTX` or
    `fatbin` file as a NULL-terminated text string, or incorporating a
    `cubin` or `fatbin` object into the executable resources and using
    operating system calls such as Windows `FindResource()` to obtain the
    pointer. Options are passed as an array via `options` and any
    corresponding parameters are passed in `optionValues`. The number of
    total options is supplied via `numOptions`. Any outputs will be
    returned via `optionValues`.

    Parameters
    ----------
    image : Any
        Module data to load
    numOptions : unsigned int
        Number of options
    options : List[:py:obj:`~.CUjit_option`]
        Options for JIT
    optionValues : List[Any]
        Option values for JIT

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_PTX`, :py:obj:`~.CUDA_ERROR_UNSUPPORTED_PTX_VERSION`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_NO_BINARY_FOR_GPU`, :py:obj:`~.CUDA_ERROR_SHARED_OBJECT_SYMBOL_NOT_FOUND`, :py:obj:`~.CUDA_ERROR_SHARED_OBJECT_INIT_FAILED`, :py:obj:`~.CUDA_ERROR_JIT_COMPILER_NOT_FOUND`
    module : :py:obj:`~.CUmodule`
        Returned module

    See Also
    --------
    :py:obj:`~.cuModuleGetFunction`, :py:obj:`~.cuModuleGetGlobal`, :py:obj:`~.cuModuleGetTexRef`, :py:obj:`~.cuModuleLoad`, :py:obj:`~.cuModuleLoadData`, :py:obj:`~.cuModuleLoadFatBinary`, :py:obj:`~.cuModuleUnload`
    """
    optionValues = [] if optionValues is None else optionValues
    options = [] if options is None else options
    if not all(isinstance(_x, (CUjit_option)) for _x in options):
        raise TypeError("Argument 'options' is not instance of type (expected Tuple[cydriver.CUjit_option] or List[cydriver.CUjit_option]")
    cdef CUmodule module = CUmodule()
    cyimage = utils.HelperInputVoidPtr(image)
    cdef void* cyimage_ptr = <void*><void_ptr>cyimage.cptr
    if numOptions > len(options): raise RuntimeError("List is too small: " + str(len(options)) + " < " + str(numOptions))
    if numOptions > len(optionValues): raise RuntimeError("List is too small: " + str(len(optionValues)) + " < " + str(numOptions))
    cdef vector[cydriver.CUjit_option] cyoptions = [pyoptions.value for pyoptions in (options)]
    pylist = [utils.HelperCUjit_option(pyoptions, pyoptionValues) for pyoptions, pyoptionValues in zip(options, optionValues)]
    cdef utils.InputVoidPtrPtrHelper voidStarHelperoptionValues = utils.InputVoidPtrPtrHelper(pylist)
    err = cydriver.cuModuleLoadDataEx(<cydriver.CUmodule*>module._ptr, cyimage_ptr, numOptions, cyoptions.data(), <void**><void_ptr>voidStarHelperoptionValues.cptr)
    return (CUresult(err), module)

@cython.embedsignature(True)
def cuModuleLoadFatBinary(fatCubin):
    """ Load a module's data.

    Takes a pointer `fatCubin` and loads the corresponding module `module`
    into the current context. The pointer represents a `fat binary` object,
    which is a collection of different `cubin` and/or `PTX` files, all
    representing the same device code, but compiled and optimized for
    different architectures.

    Prior to CUDA 4.0, there was no documented API for constructing and
    using fat binary objects by programmers. Starting with CUDA 4.0, fat
    binary objects can be constructed by providing the `-fatbin option` to
    nvcc. More information can be found in the nvcc document.

    Parameters
    ----------
    fatCubin : Any
        Fat binary to load

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_PTX`, :py:obj:`~.CUDA_ERROR_UNSUPPORTED_PTX_VERSION`, :py:obj:`~.CUDA_ERROR_NOT_FOUND`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_NO_BINARY_FOR_GPU`, :py:obj:`~.CUDA_ERROR_SHARED_OBJECT_SYMBOL_NOT_FOUND`, :py:obj:`~.CUDA_ERROR_SHARED_OBJECT_INIT_FAILED`, :py:obj:`~.CUDA_ERROR_JIT_COMPILER_NOT_FOUND`
    module : :py:obj:`~.CUmodule`
        Returned module

    See Also
    --------
    :py:obj:`~.cuModuleGetFunction`, :py:obj:`~.cuModuleGetGlobal`, :py:obj:`~.cuModuleGetTexRef`, :py:obj:`~.cuModuleLoad`, :py:obj:`~.cuModuleLoadData`, :py:obj:`~.cuModuleLoadDataEx`, :py:obj:`~.cuModuleUnload`
    """
    cdef CUmodule module = CUmodule()
    cyfatCubin = utils.HelperInputVoidPtr(fatCubin)
    cdef void* cyfatCubin_ptr = <void*><void_ptr>cyfatCubin.cptr
    err = cydriver.cuModuleLoadFatBinary(<cydriver.CUmodule*>module._ptr, cyfatCubin_ptr)
    return (CUresult(err), module)

@cython.embedsignature(True)
def cuModuleUnload(hmod):
    """ Unloads a module.

    Unloads a module `hmod` from the current context.

    Parameters
    ----------
    hmod : :py:obj:`~.CUmodule`
        Module to unload

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuModuleGetFunction`, :py:obj:`~.cuModuleGetGlobal`, :py:obj:`~.cuModuleGetTexRef`, :py:obj:`~.cuModuleLoad`, :py:obj:`~.cuModuleLoadData`, :py:obj:`~.cuModuleLoadDataEx`, :py:obj:`~.cuModuleLoadFatBinary`
    """
    cdef cydriver.CUmodule cyhmod
    if hmod is None:
        cyhmod = <cydriver.CUmodule><void_ptr>0
    elif isinstance(hmod, (CUmodule,)):
        phmod = int(hmod)
        cyhmod = <cydriver.CUmodule><void_ptr>phmod
    else:
        phmod = int(CUmodule(hmod))
        cyhmod = <cydriver.CUmodule><void_ptr>phmod
    err = cydriver.cuModuleUnload(cyhmod)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuModuleGetLoadingMode():
    """ Query lazy loading mode.

    Returns lazy loading mode Module loading mode is controlled by
    CUDA_MODULE_LOADING env variable

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`,
    mode : :py:obj:`~.CUmoduleLoadingMode`
        Returns the lazy loading mode

    See Also
    --------
    :py:obj:`~.cuModuleLoad`,
    """
    cdef cydriver.CUmoduleLoadingMode mode
    err = cydriver.cuModuleGetLoadingMode(&mode)
    return (CUresult(err), CUmoduleLoadingMode(mode))

@cython.embedsignature(True)
def cuModuleGetFunction(hmod, char* name):
    """ Returns a function handle.

    Returns in `*hfunc` the handle of the function of name `name` located
    in module `hmod`. If no function of that name exists,
    :py:obj:`~.cuModuleGetFunction()` returns
    :py:obj:`~.CUDA_ERROR_NOT_FOUND`.

    Parameters
    ----------
    hmod : :py:obj:`~.CUmodule`
        Module to retrieve function from
    name : bytes
        Name of function to retrieve

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_NOT_FOUND`
    hfunc : :py:obj:`~.CUfunction`
        Returned function handle

    See Also
    --------
    :py:obj:`~.cuModuleGetGlobal`, :py:obj:`~.cuModuleGetTexRef`, :py:obj:`~.cuModuleLoad`, :py:obj:`~.cuModuleLoadData`, :py:obj:`~.cuModuleLoadDataEx`, :py:obj:`~.cuModuleLoadFatBinary`, :py:obj:`~.cuModuleUnload`
    """
    cdef cydriver.CUmodule cyhmod
    if hmod is None:
        cyhmod = <cydriver.CUmodule><void_ptr>0
    elif isinstance(hmod, (CUmodule,)):
        phmod = int(hmod)
        cyhmod = <cydriver.CUmodule><void_ptr>phmod
    else:
        phmod = int(CUmodule(hmod))
        cyhmod = <cydriver.CUmodule><void_ptr>phmod
    cdef CUfunction hfunc = CUfunction()
    err = cydriver.cuModuleGetFunction(<cydriver.CUfunction*>hfunc._ptr, cyhmod, name)
    return (CUresult(err), hfunc)

@cython.embedsignature(True)
def cuModuleGetGlobal(hmod, char* name):
    """ Returns a global pointer from a module.

    Returns in `*dptr` and `*bytes` the base pointer and size of the global
    of name `name` located in module `hmod`. If no variable of that name
    exists, :py:obj:`~.cuModuleGetGlobal()` returns
    :py:obj:`~.CUDA_ERROR_NOT_FOUND`. Both parameters `dptr` and `numbytes`
    are optional. If one of them is NULL, it is ignored.

    Parameters
    ----------
    hmod : :py:obj:`~.CUmodule`
        Module to retrieve global from
    name : bytes
        Name of global to retrieve

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_NOT_FOUND`
    dptr : :py:obj:`~.CUdeviceptr`
        Returned global device pointer
    numbytes : int
        Returned global size in bytes

    See Also
    --------
    :py:obj:`~.cuModuleGetFunction`, :py:obj:`~.cuModuleGetTexRef`, :py:obj:`~.cuModuleLoad`, :py:obj:`~.cuModuleLoadData`, :py:obj:`~.cuModuleLoadDataEx`, :py:obj:`~.cuModuleLoadFatBinary`, :py:obj:`~.cuModuleUnload`, :py:obj:`~.cudaGetSymbolAddress`, :py:obj:`~.cudaGetSymbolSize`
    """
    cdef cydriver.CUmodule cyhmod
    if hmod is None:
        cyhmod = <cydriver.CUmodule><void_ptr>0
    elif isinstance(hmod, (CUmodule,)):
        phmod = int(hmod)
        cyhmod = <cydriver.CUmodule><void_ptr>phmod
    else:
        phmod = int(CUmodule(hmod))
        cyhmod = <cydriver.CUmodule><void_ptr>phmod
    cdef CUdeviceptr dptr = CUdeviceptr()
    cdef size_t numbytes = 0
    err = cydriver.cuModuleGetGlobal(<cydriver.CUdeviceptr*>dptr._ptr, &numbytes, cyhmod, name)
    return (CUresult(err), dptr, numbytes)

@cython.embedsignature(True)
def cuModuleGetTexRef(hmod, char* name):
    """ Returns a handle to a texture reference.

    Returns in `*pTexRef` the handle of the texture reference of name
    `name` in the module `hmod`. If no texture reference of that name
    exists, :py:obj:`~.cuModuleGetTexRef()` returns
    :py:obj:`~.CUDA_ERROR_NOT_FOUND`. This texture reference handle should
    not be destroyed, since it will be destroyed when the module is
    unloaded.

    Parameters
    ----------
    hmod : :py:obj:`~.CUmodule`
        Module to retrieve texture reference from
    name : bytes
        Name of texture reference to retrieve

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_NOT_FOUND`
    pTexRef : :py:obj:`~.CUtexref`
        Returned texture reference

    See Also
    --------
    :py:obj:`~.cuModuleGetFunction`, :py:obj:`~.cuModuleGetGlobal`, :py:obj:`~.cuModuleGetSurfRef`, :py:obj:`~.cuModuleLoad`, :py:obj:`~.cuModuleLoadData`, :py:obj:`~.cuModuleLoadDataEx`, :py:obj:`~.cuModuleLoadFatBinary`, :py:obj:`~.cuModuleUnload`, :py:obj:`~.cudaGetTextureReference`
    """
    cdef cydriver.CUmodule cyhmod
    if hmod is None:
        cyhmod = <cydriver.CUmodule><void_ptr>0
    elif isinstance(hmod, (CUmodule,)):
        phmod = int(hmod)
        cyhmod = <cydriver.CUmodule><void_ptr>phmod
    else:
        phmod = int(CUmodule(hmod))
        cyhmod = <cydriver.CUmodule><void_ptr>phmod
    cdef CUtexref pTexRef = CUtexref()
    err = cydriver.cuModuleGetTexRef(<cydriver.CUtexref*>pTexRef._ptr, cyhmod, name)
    return (CUresult(err), pTexRef)

@cython.embedsignature(True)
def cuModuleGetSurfRef(hmod, char* name):
    """ Returns a handle to a surface reference.

    Returns in `*pSurfRef` the handle of the surface reference of name
    `name` in the module `hmod`. If no surface reference of that name
    exists, :py:obj:`~.cuModuleGetSurfRef()` returns
    :py:obj:`~.CUDA_ERROR_NOT_FOUND`.

    Parameters
    ----------
    hmod : :py:obj:`~.CUmodule`
        Module to retrieve surface reference from
    name : bytes
        Name of surface reference to retrieve

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_NOT_FOUND`
    pSurfRef : :py:obj:`~.CUsurfref`
        Returned surface reference

    See Also
    --------
    :py:obj:`~.cuModuleGetFunction`, :py:obj:`~.cuModuleGetGlobal`, :py:obj:`~.cuModuleGetTexRef`, :py:obj:`~.cuModuleLoad`, :py:obj:`~.cuModuleLoadData`, :py:obj:`~.cuModuleLoadDataEx`, :py:obj:`~.cuModuleLoadFatBinary`, :py:obj:`~.cuModuleUnload`, :py:obj:`~.cudaGetSurfaceReference`
    """
    cdef cydriver.CUmodule cyhmod
    if hmod is None:
        cyhmod = <cydriver.CUmodule><void_ptr>0
    elif isinstance(hmod, (CUmodule,)):
        phmod = int(hmod)
        cyhmod = <cydriver.CUmodule><void_ptr>phmod
    else:
        phmod = int(CUmodule(hmod))
        cyhmod = <cydriver.CUmodule><void_ptr>phmod
    cdef CUsurfref pSurfRef = CUsurfref()
    err = cydriver.cuModuleGetSurfRef(<cydriver.CUsurfref*>pSurfRef._ptr, cyhmod, name)
    return (CUresult(err), pSurfRef)

@cython.embedsignature(True)
def cuLinkCreate(unsigned int numOptions, options : Optional[Tuple[CUjit_option] | List[CUjit_option]], optionValues : Optional[Tuple[Any] | List[Any]]):
    """ Creates a pending JIT linker invocation.

    If the call is successful, the caller owns the returned CUlinkState,
    which should eventually be destroyed with :py:obj:`~.cuLinkDestroy`.
    The device code machine size (32 or 64 bit) will match the calling
    application.

    Both linker and compiler options may be specified. Compiler options
    will be applied to inputs to this linker action which must be compiled
    from PTX. The options :py:obj:`~.CU_JIT_WALL_TIME`,
    :py:obj:`~.CU_JIT_INFO_LOG_BUFFER_SIZE_BYTES`, and
    :py:obj:`~.CU_JIT_ERROR_LOG_BUFFER_SIZE_BYTES` will accumulate data
    until the CUlinkState is destroyed.

    `optionValues` must remain valid for the life of the CUlinkState if
    output options are used. No other references to inputs are maintained
    after this call returns.

    Parameters
    ----------
    numOptions : unsigned int
        Size of options arrays
    options : List[:py:obj:`~.CUjit_option`]
        Array of linker and compiler options
    optionValues : List[Any]
        Array of option values, each cast to void *

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_JIT_COMPILER_NOT_FOUND`
    stateOut : :py:obj:`~.CUlinkState`
        On success, this will contain a CUlinkState to specify and complete
        this action

    See Also
    --------
    :py:obj:`~.cuLinkAddData`, :py:obj:`~.cuLinkAddFile`, :py:obj:`~.cuLinkComplete`, :py:obj:`~.cuLinkDestroy`
    """
    optionValues = [] if optionValues is None else optionValues
    options = [] if options is None else options
    if not all(isinstance(_x, (CUjit_option)) for _x in options):
        raise TypeError("Argument 'options' is not instance of type (expected Tuple[cydriver.CUjit_option] or List[cydriver.CUjit_option]")
    if numOptions > len(options): raise RuntimeError("List is too small: " + str(len(options)) + " < " + str(numOptions))
    if numOptions > len(optionValues): raise RuntimeError("List is too small: " + str(len(optionValues)) + " < " + str(numOptions))
    cdef vector[cydriver.CUjit_option] cyoptions = [pyoptions.value for pyoptions in (options)]
    pylist = [utils.HelperCUjit_option(pyoptions, pyoptionValues) for pyoptions, pyoptionValues in zip(options, optionValues)]
    cdef utils.InputVoidPtrPtrHelper voidStarHelperoptionValues = utils.InputVoidPtrPtrHelper(pylist)
    cdef CUlinkState stateOut = CUlinkState()
    err = cydriver.cuLinkCreate(numOptions, cyoptions.data(), <void**><void_ptr>voidStarHelperoptionValues.cptr, stateOut._ptr)
    stateOut._keepalive.append(voidStarHelperoptionValues)
    for option in pylist:
        stateOut._keepalive.append(option)
    return (CUresult(err), stateOut)

@cython.embedsignature(True)
def cuLinkAddData(state, typename not None : CUjitInputType, data, size_t size, char* name, unsigned int numOptions, options : Optional[Tuple[CUjit_option] | List[CUjit_option]], optionValues : Optional[Tuple[Any] | List[Any]]):
    """ Add an input to a pending linker invocation.

    Ownership of `data` is retained by the caller. No reference is retained
    to any inputs after this call returns.

    This method accepts only compiler options, which are used if the data
    must be compiled from PTX, and does not accept any of
    :py:obj:`~.CU_JIT_WALL_TIME`, :py:obj:`~.CU_JIT_INFO_LOG_BUFFER`,
    :py:obj:`~.CU_JIT_ERROR_LOG_BUFFER`,
    :py:obj:`~.CU_JIT_TARGET_FROM_CUCONTEXT`, or :py:obj:`~.CU_JIT_TARGET`.

    Parameters
    ----------
    state : :py:obj:`~.CUlinkState`
        A pending linker action.
    typename : :py:obj:`~.CUjitInputType`
        The type of the input data.
    data : Any
        The input data. PTX must be NULL-terminated.
    size : size_t
        The length of the input data.
    name : bytes
        An optional name for this input in log messages.
    numOptions : unsigned int
        Size of options.
    options : List[:py:obj:`~.CUjit_option`]
        Options to be applied only for this input (overrides options from
        :py:obj:`~.cuLinkCreate`).
    optionValues : List[Any]
        Array of option values, each cast to void *.

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_IMAGE`, :py:obj:`~.CUDA_ERROR_INVALID_PTX`, :py:obj:`~.CUDA_ERROR_UNSUPPORTED_PTX_VERSION`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_NO_BINARY_FOR_GPU`

    See Also
    --------
    :py:obj:`~.cuLinkCreate`, :py:obj:`~.cuLinkAddFile`, :py:obj:`~.cuLinkComplete`, :py:obj:`~.cuLinkDestroy`
    """
    optionValues = [] if optionValues is None else optionValues
    options = [] if options is None else options
    if not all(isinstance(_x, (CUjit_option)) for _x in options):
        raise TypeError("Argument 'options' is not instance of type (expected Tuple[cydriver.CUjit_option] or List[cydriver.CUjit_option]")
    cdef cydriver.CUlinkState cystate
    if state is None:
        cystate = <cydriver.CUlinkState><void_ptr>0
    elif isinstance(state, (CUlinkState,)):
        pstate = int(state)
        cystate = <cydriver.CUlinkState><void_ptr>pstate
    else:
        pstate = int(CUlinkState(state))
        cystate = <cydriver.CUlinkState><void_ptr>pstate
    cdef cydriver.CUjitInputType cytypename = typename.value
    cydata = utils.HelperInputVoidPtr(data)
    cdef void* cydata_ptr = <void*><void_ptr>cydata.cptr
    if numOptions > len(options): raise RuntimeError("List is too small: " + str(len(options)) + " < " + str(numOptions))
    if numOptions > len(optionValues): raise RuntimeError("List is too small: " + str(len(optionValues)) + " < " + str(numOptions))
    cdef vector[cydriver.CUjit_option] cyoptions = [pyoptions.value for pyoptions in (options)]
    pylist = [utils.HelperCUjit_option(pyoptions, pyoptionValues) for pyoptions, pyoptionValues in zip(options, optionValues)]
    cdef utils.InputVoidPtrPtrHelper voidStarHelperoptionValues = utils.InputVoidPtrPtrHelper(pylist)
    err = cydriver.cuLinkAddData(cystate, cytypename, cydata_ptr, size, name, numOptions, cyoptions.data(), <void**><void_ptr>voidStarHelperoptionValues.cptr)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuLinkAddFile(state, typename not None : CUjitInputType, char* path, unsigned int numOptions, options : Optional[Tuple[CUjit_option] | List[CUjit_option]], optionValues : Optional[Tuple[Any] | List[Any]]):
    """ Add a file input to a pending linker invocation.

    No reference is retained to any inputs after this call returns.

    This method accepts only compiler options, which are used if the input
    must be compiled from PTX, and does not accept any of
    :py:obj:`~.CU_JIT_WALL_TIME`, :py:obj:`~.CU_JIT_INFO_LOG_BUFFER`,
    :py:obj:`~.CU_JIT_ERROR_LOG_BUFFER`,
    :py:obj:`~.CU_JIT_TARGET_FROM_CUCONTEXT`, or :py:obj:`~.CU_JIT_TARGET`.

    This method is equivalent to invoking :py:obj:`~.cuLinkAddData` on the
    contents of the file.

    Parameters
    ----------
    state : :py:obj:`~.CUlinkState`
        A pending linker action
    typename : :py:obj:`~.CUjitInputType`
        The type of the input data
    path : bytes
        Path to the input file
    numOptions : unsigned int
        Size of options
    options : List[:py:obj:`~.CUjit_option`]
        Options to be applied only for this input (overrides options from
        :py:obj:`~.cuLinkCreate`)
    optionValues : List[Any]
        Array of option values, each cast to void *

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_FILE_NOT_FOUND` :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_IMAGE`, :py:obj:`~.CUDA_ERROR_INVALID_PTX`, :py:obj:`~.CUDA_ERROR_UNSUPPORTED_PTX_VERSION`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_NO_BINARY_FOR_GPU`

    See Also
    --------
    :py:obj:`~.cuLinkCreate`, :py:obj:`~.cuLinkAddData`, :py:obj:`~.cuLinkComplete`, :py:obj:`~.cuLinkDestroy`
    """
    optionValues = [] if optionValues is None else optionValues
    options = [] if options is None else options
    if not all(isinstance(_x, (CUjit_option)) for _x in options):
        raise TypeError("Argument 'options' is not instance of type (expected Tuple[cydriver.CUjit_option] or List[cydriver.CUjit_option]")
    cdef cydriver.CUlinkState cystate
    if state is None:
        cystate = <cydriver.CUlinkState><void_ptr>0
    elif isinstance(state, (CUlinkState,)):
        pstate = int(state)
        cystate = <cydriver.CUlinkState><void_ptr>pstate
    else:
        pstate = int(CUlinkState(state))
        cystate = <cydriver.CUlinkState><void_ptr>pstate
    cdef cydriver.CUjitInputType cytypename = typename.value
    if numOptions > len(options): raise RuntimeError("List is too small: " + str(len(options)) + " < " + str(numOptions))
    if numOptions > len(optionValues): raise RuntimeError("List is too small: " + str(len(optionValues)) + " < " + str(numOptions))
    cdef vector[cydriver.CUjit_option] cyoptions = [pyoptions.value for pyoptions in (options)]
    pylist = [utils.HelperCUjit_option(pyoptions, pyoptionValues) for pyoptions, pyoptionValues in zip(options, optionValues)]
    cdef utils.InputVoidPtrPtrHelper voidStarHelperoptionValues = utils.InputVoidPtrPtrHelper(pylist)
    err = cydriver.cuLinkAddFile(cystate, cytypename, path, numOptions, cyoptions.data(), <void**><void_ptr>voidStarHelperoptionValues.cptr)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuLinkComplete(state):
    """ Complete a pending linker invocation.

    Completes the pending linker action and returns the cubin image for the
    linked device code, which can be used with
    :py:obj:`~.cuModuleLoadData`. The cubin is owned by `state`, so it
    should be loaded before `state` is destroyed via
    :py:obj:`~.cuLinkDestroy`. This call does not destroy `state`.

    Parameters
    ----------
    state : :py:obj:`~.CUlinkState`
        A pending linker invocation

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`
    cubinOut : Any
        On success, this will point to the output image
    sizeOut : int
        Optional parameter to receive the size of the generated image

    See Also
    --------
    :py:obj:`~.cuLinkCreate`, :py:obj:`~.cuLinkAddData`, :py:obj:`~.cuLinkAddFile`, :py:obj:`~.cuLinkDestroy`, :py:obj:`~.cuModuleLoadData`
    """
    cdef cydriver.CUlinkState cystate
    if state is None:
        cystate = <cydriver.CUlinkState><void_ptr>0
    elif isinstance(state, (CUlinkState,)):
        pstate = int(state)
        cystate = <cydriver.CUlinkState><void_ptr>pstate
    else:
        pstate = int(CUlinkState(state))
        cystate = <cydriver.CUlinkState><void_ptr>pstate
    cdef void_ptr cubinOut = 0
    cdef size_t sizeOut = 0
    err = cydriver.cuLinkComplete(cystate, <void**>&cubinOut, &sizeOut)
    return (CUresult(err), cubinOut, sizeOut)

@cython.embedsignature(True)
def cuLinkDestroy(state):
    """ Destroys state for a JIT linker invocation.

    Parameters
    ----------
    state : :py:obj:`~.CUlinkState`
        State object for the linker invocation

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`

    See Also
    --------
    :py:obj:`~.cuLinkCreate`
    """
    cdef cydriver.CUlinkState cystate
    if state is None:
        cystate = <cydriver.CUlinkState><void_ptr>0
    elif isinstance(state, (CUlinkState,)):
        pstate = int(state)
        cystate = <cydriver.CUlinkState><void_ptr>pstate
    else:
        pstate = int(CUlinkState(state))
        cystate = <cydriver.CUlinkState><void_ptr>pstate
    err = cydriver.cuLinkDestroy(cystate)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemGetInfo():
    """ Gets free and total memory.

    Returns in `*total` the total amount of memory available to the the
    current context. Returns in `*free` the amount of memory on the device
    that is free according to the OS. CUDA is not guaranteed to be able to
    allocate all of the memory that the OS reports as free. In a multi-
    tenet situation, free estimate returned is prone to race condition
    where a new allocation/free done by a different process or a different
    thread in the same process between the time when free memory was
    estimated and reported, will result in deviation in free value reported
    and actual free memory.

    The integrated GPU on Tegra shares memory with CPU and other component
    of the SoC. The free and total values returned by the API excludes the
    SWAP memory space maintained by the OS on some platforms. The OS may
    move some of the memory pages into swap area as the GPU or CPU allocate
    or access memory. See Tegra app note on how to calculate total and free
    memory on Tegra.

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    free : int
        Returned free memory in bytes
    total : int
        Returned total memory in bytes

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaMemGetInfo`
    """
    cdef size_t free = 0
    cdef size_t total = 0
    err = cydriver.cuMemGetInfo(&free, &total)
    return (CUresult(err), free, total)

@cython.embedsignature(True)
def cuMemAlloc(size_t bytesize):
    """ Allocates device memory.

    Allocates `bytesize` bytes of linear memory on the device and returns
    in `*dptr` a pointer to the allocated memory. The allocated memory is
    suitably aligned for any kind of variable. The memory is not cleared.
    If `bytesize` is 0, :py:obj:`~.cuMemAlloc()` returns
    :py:obj:`~.CUDA_ERROR_INVALID_VALUE`.

    Parameters
    ----------
    bytesize : size_t
        Requested allocation size in bytes

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`
    dptr : :py:obj:`~.CUdeviceptr`
        Returned device pointer

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaMalloc`
    """
    cdef CUdeviceptr dptr = CUdeviceptr()
    err = cydriver.cuMemAlloc(<cydriver.CUdeviceptr*>dptr._ptr, bytesize)
    return (CUresult(err), dptr)

@cython.embedsignature(True)
def cuMemAllocPitch(size_t WidthInBytes, size_t Height, unsigned int ElementSizeBytes):
    """ Allocates pitched device memory.

    Allocates at least `WidthInBytes` * `Height` bytes of linear memory on
    the device and returns in `*dptr` a pointer to the allocated memory.
    The function may pad the allocation to ensure that corresponding
    pointers in any given row will continue to meet the alignment
    requirements for coalescing as the address is updated from row to row.
    `ElementSizeBytes` specifies the size of the largest reads and writes
    that will be performed on the memory range. `ElementSizeBytes` may be
    4, 8 or 16 (since coalesced memory transactions are not possible on
    other data sizes). If `ElementSizeBytes` is smaller than the actual
    read/write size of a kernel, the kernel will run correctly, but
    possibly at reduced speed. The pitch returned in `*pPitch` by
    :py:obj:`~.cuMemAllocPitch()` is the width in bytes of the allocation.
    The intended usage of pitch is as a separate parameter of the
    allocation, used to compute addresses within the 2D array. Given the
    row and column of an array element of type T, the address is computed
    as:

    **View CUDA Toolkit Documentation for a C++ code example**

    The pitch returned by :py:obj:`~.cuMemAllocPitch()` is guaranteed to
    work with :py:obj:`~.cuMemcpy2D()` under all circumstances. For
    allocations of 2D arrays, it is recommended that programmers consider
    performing pitch allocations using :py:obj:`~.cuMemAllocPitch()`. Due
    to alignment restrictions in the hardware, this is especially true if
    the application will be performing 2D memory copies between different
    regions of device memory (whether linear memory or CUDA arrays).

    The byte alignment of the pitch returned by
    :py:obj:`~.cuMemAllocPitch()` is guaranteed to match or exceed the
    alignment requirement for texture binding with
    :py:obj:`~.cuTexRefSetAddress2D()`.

    Parameters
    ----------
    WidthInBytes : size_t
        Requested allocation width in bytes
    Height : size_t
        Requested allocation height in rows
    ElementSizeBytes : unsigned int
        Size of largest reads/writes for range

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`
    dptr : :py:obj:`~.CUdeviceptr`
        Returned device pointer
    pPitch : int
        Returned pitch of allocation in bytes

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaMallocPitch`
    """
    cdef CUdeviceptr dptr = CUdeviceptr()
    cdef size_t pPitch = 0
    err = cydriver.cuMemAllocPitch(<cydriver.CUdeviceptr*>dptr._ptr, &pPitch, WidthInBytes, Height, ElementSizeBytes)
    return (CUresult(err), dptr, pPitch)

@cython.embedsignature(True)
def cuMemFree(dptr):
    """ Frees device memory.

    Frees the memory space pointed to by `dptr`, which must have been
    returned by a previous call to one of the following memory allocation
    APIs - :py:obj:`~.cuMemAlloc()`, :py:obj:`~.cuMemAllocPitch()`,
    :py:obj:`~.cuMemAllocManaged()`, :py:obj:`~.cuMemAllocAsync()`,
    :py:obj:`~.cuMemAllocFromPoolAsync()`

    Note - This API will not perform any implict synchronization when the
    pointer was allocated with :py:obj:`~.cuMemAllocAsync` or
    :py:obj:`~.cuMemAllocFromPoolAsync`. Callers must ensure that all
    accesses to the pointer have completed before invoking
    :py:obj:`~.cuMemFree`. For best performance and memory reuse, users
    should use :py:obj:`~.cuMemFreeAsync` to free memory allocated via the
    stream ordered memory allocator.

    Parameters
    ----------
    dptr : :py:obj:`~.CUdeviceptr`
        Pointer to memory to free

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemAllocManaged`, :py:obj:`~.cuMemAllocAsync`, :py:obj:`~.cuMemAllocFromPoolAsync`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemFreeAsync`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaFree`
    """
    cdef cydriver.CUdeviceptr cydptr
    if dptr is None:
        cydptr = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(dptr, (CUdeviceptr,)):
        pdptr = int(dptr)
        cydptr = <cydriver.CUdeviceptr><void_ptr>pdptr
    else:
        pdptr = int(CUdeviceptr(dptr))
        cydptr = <cydriver.CUdeviceptr><void_ptr>pdptr
    err = cydriver.cuMemFree(cydptr)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemGetAddressRange(dptr):
    """ Get information on memory allocations.

    Returns the base address in `*pbase` and size in `*psize` of the
    allocation by :py:obj:`~.cuMemAlloc()` or :py:obj:`~.cuMemAllocPitch()`
    that contains the input pointer `dptr`. Both parameters `pbase` and
    `psize` are optional. If one of them is NULL, it is ignored.

    Parameters
    ----------
    dptr : :py:obj:`~.CUdeviceptr`
        Device pointer to query

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_NOT_FOUND`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    pbase : :py:obj:`~.CUdeviceptr`
        Returned base address
    psize : int
        Returned size of device memory allocation

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`
    """
    cdef cydriver.CUdeviceptr cydptr
    if dptr is None:
        cydptr = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(dptr, (CUdeviceptr,)):
        pdptr = int(dptr)
        cydptr = <cydriver.CUdeviceptr><void_ptr>pdptr
    else:
        pdptr = int(CUdeviceptr(dptr))
        cydptr = <cydriver.CUdeviceptr><void_ptr>pdptr
    cdef CUdeviceptr pbase = CUdeviceptr()
    cdef size_t psize = 0
    err = cydriver.cuMemGetAddressRange(<cydriver.CUdeviceptr*>pbase._ptr, &psize, cydptr)
    return (CUresult(err), pbase, psize)

@cython.embedsignature(True)
def cuMemAllocHost(size_t bytesize):
    """ Allocates page-locked host memory.

    Allocates `bytesize` bytes of host memory that is page-locked and
    accessible to the device. The driver tracks the virtual memory ranges
    allocated with this function and automatically accelerates calls to
    functions such as :py:obj:`~.cuMemcpy()`. Since the memory can be
    accessed directly by the device, it can be read or written with much
    higher bandwidth than pageable memory obtained with functions such as
    :py:obj:`~.malloc()`. Allocating excessive amounts of memory with
    :py:obj:`~.cuMemAllocHost()` may degrade system performance, since it
    reduces the amount of memory available to the system for paging. As a
    result, this function is best used sparingly to allocate staging areas
    for data exchange between host and device.

    Note all host memory allocated using :py:obj:`~.cuMemHostAlloc()` will
    automatically be immediately accessible to all contexts on all devices
    which support unified addressing (as may be queried using
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_UNIFIED_ADDRESSING`). The device pointer
    that may be used to access this host memory from those contexts is
    always equal to the returned host pointer `*pp`. See :py:obj:`~.Unified
    Addressing` for additional details.

    Parameters
    ----------
    bytesize : size_t
        Requested allocation size in bytes

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`
    pp : Any
        Returned host pointer to page-locked memory

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaMallocHost`
    """
    cdef void_ptr pp = 0
    err = cydriver.cuMemAllocHost(<void**>&pp, bytesize)
    return (CUresult(err), pp)

@cython.embedsignature(True)
def cuMemFreeHost(p):
    """ Frees page-locked host memory.

    Frees the memory space pointed to by `p`, which must have been returned
    by a previous call to :py:obj:`~.cuMemAllocHost()`.

    Parameters
    ----------
    p : Any
        Pointer to memory to free

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaFreeHost`
    """
    cyp = utils.HelperInputVoidPtr(p)
    cdef void* cyp_ptr = <void*><void_ptr>cyp.cptr
    err = cydriver.cuMemFreeHost(cyp_ptr)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemHostAlloc(size_t bytesize, unsigned int Flags):
    """ Allocates page-locked host memory.

    Allocates `bytesize` bytes of host memory that is page-locked and
    accessible to the device. The driver tracks the virtual memory ranges
    allocated with this function and automatically accelerates calls to
    functions such as :py:obj:`~.cuMemcpyHtoD()`. Since the memory can be
    accessed directly by the device, it can be read or written with much
    higher bandwidth than pageable memory obtained with functions such as
    :py:obj:`~.malloc()`. Allocating excessive amounts of pinned memory may
    degrade system performance, since it reduces the amount of memory
    available to the system for paging. As a result, this function is best
    used sparingly to allocate staging areas for data exchange between host
    and device.

    The `Flags` parameter enables different options to be specified that
    affect the allocation, as follows.

    - :py:obj:`~.CU_MEMHOSTALLOC_PORTABLE`: The memory returned by this
      call will be considered as pinned memory by all CUDA contexts, not
      just the one that performed the allocation.

    - :py:obj:`~.CU_MEMHOSTALLOC_DEVICEMAP`: Maps the allocation into the
      CUDA address space. The device pointer to the memory may be obtained
      by calling :py:obj:`~.cuMemHostGetDevicePointer()`.

    - :py:obj:`~.CU_MEMHOSTALLOC_WRITECOMBINED`: Allocates the memory as
      write-combined (WC). WC memory can be transferred across the PCI
      Express bus more quickly on some system configurations, but cannot be
      read efficiently by most CPUs. WC memory is a good option for buffers
      that will be written by the CPU and read by the GPU via mapped pinned
      memory or host->device transfers.

    All of these flags are orthogonal to one another: a developer may
    allocate memory that is portable, mapped and/or write-combined with no
    restrictions.

    The :py:obj:`~.CU_MEMHOSTALLOC_DEVICEMAP` flag may be specified on CUDA
    contexts for devices that do not support mapped pinned memory. The
    failure is deferred to :py:obj:`~.cuMemHostGetDevicePointer()` because
    the memory may be mapped into other CUDA contexts via the
    :py:obj:`~.CU_MEMHOSTALLOC_PORTABLE` flag.

    The memory allocated by this function must be freed with
    :py:obj:`~.cuMemFreeHost()`.

    Note all host memory allocated using :py:obj:`~.cuMemHostAlloc()` will
    automatically be immediately accessible to all contexts on all devices
    which support unified addressing (as may be queried using
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_UNIFIED_ADDRESSING`). Unless the flag
    :py:obj:`~.CU_MEMHOSTALLOC_WRITECOMBINED` is specified, the device
    pointer that may be used to access this host memory from those contexts
    is always equal to the returned host pointer `*pp`. If the flag
    :py:obj:`~.CU_MEMHOSTALLOC_WRITECOMBINED` is specified, then the
    function :py:obj:`~.cuMemHostGetDevicePointer()` must be used to query
    the device pointer, even if the context supports unified addressing.
    See :py:obj:`~.Unified Addressing` for additional details.

    Parameters
    ----------
    bytesize : size_t
        Requested allocation size in bytes
    Flags : unsigned int
        Flags for allocation request

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`
    pp : Any
        Returned host pointer to page-locked memory

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaHostAlloc`
    """
    cdef void_ptr pp = 0
    err = cydriver.cuMemHostAlloc(<void**>&pp, bytesize, Flags)
    return (CUresult(err), pp)

@cython.embedsignature(True)
def cuMemHostGetDevicePointer(p, unsigned int Flags):
    """ Passes back device pointer of mapped pinned memory.

    Passes back the device pointer `pdptr` corresponding to the mapped,
    pinned host buffer `p` allocated by :py:obj:`~.cuMemHostAlloc`.

    :py:obj:`~.cuMemHostGetDevicePointer()` will fail if the
    :py:obj:`~.CU_MEMHOSTALLOC_DEVICEMAP` flag was not specified at the
    time the memory was allocated, or if the function is called on a GPU
    that does not support mapped pinned memory.

    For devices that have a non-zero value for the device attribute
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_CAN_USE_HOST_POINTER_FOR_REGISTERED_MEM`,
    the memory can also be accessed from the device using the host pointer
    `p`. The device pointer returned by
    :py:obj:`~.cuMemHostGetDevicePointer()` may or may not match the
    original host pointer `p` and depends on the devices visible to the
    application. If all devices visible to the application have a non-zero
    value for the device attribute, the device pointer returned by
    :py:obj:`~.cuMemHostGetDevicePointer()` will match the original pointer
    `p`. If any device visible to the application has a zero value for the
    device attribute, the device pointer returned by
    :py:obj:`~.cuMemHostGetDevicePointer()` will not match the original
    host pointer `p`, but it will be suitable for use on all devices
    provided Unified Virtual Addressing is enabled. In such systems, it is
    valid to access the memory using either pointer on devices that have a
    non-zero value for the device attribute. Note however that such devices
    should access the memory using only one of the two pointers and not
    both.

    `Flags` provides for future releases. For now, it must be set to 0.

    Parameters
    ----------
    p : Any
        Host pointer
    Flags : unsigned int
        Options (must be 0)

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    pdptr : :py:obj:`~.CUdeviceptr`
        Returned device pointer

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaHostGetDevicePointer`
    """
    cdef CUdeviceptr pdptr = CUdeviceptr()
    cyp = utils.HelperInputVoidPtr(p)
    cdef void* cyp_ptr = <void*><void_ptr>cyp.cptr
    err = cydriver.cuMemHostGetDevicePointer(<cydriver.CUdeviceptr*>pdptr._ptr, cyp_ptr, Flags)
    return (CUresult(err), pdptr)

@cython.embedsignature(True)
def cuMemHostGetFlags(p):
    """ Passes back flags that were used for a pinned allocation.

    Passes back the flags `pFlags` that were specified when allocating the
    pinned host buffer `p` allocated by :py:obj:`~.cuMemHostAlloc`.

    :py:obj:`~.cuMemHostGetFlags()` will fail if the pointer does not
    reside in an allocation performed by :py:obj:`~.cuMemAllocHost()` or
    :py:obj:`~.cuMemHostAlloc()`.

    Parameters
    ----------
    p : Any
        Host pointer

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    pFlags : unsigned int
        Returned flags word

    See Also
    --------
    :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cudaHostGetFlags`
    """
    cdef unsigned int pFlags = 0
    cyp = utils.HelperInputVoidPtr(p)
    cdef void* cyp_ptr = <void*><void_ptr>cyp.cptr
    err = cydriver.cuMemHostGetFlags(&pFlags, cyp_ptr)
    return (CUresult(err), pFlags)

@cython.embedsignature(True)
def cuMemAllocManaged(size_t bytesize, unsigned int flags):
    """ Allocates memory that will be automatically managed by the Unified Memory system.

    Allocates `bytesize` bytes of managed memory on the device and returns
    in `*dptr` a pointer to the allocated memory. If the device doesn't
    support allocating managed memory, :py:obj:`~.CUDA_ERROR_NOT_SUPPORTED`
    is returned. Support for managed memory can be queried using the device
    attribute :py:obj:`~.CU_DEVICE_ATTRIBUTE_MANAGED_MEMORY`. The allocated
    memory is suitably aligned for any kind of variable. The memory is not
    cleared. If `bytesize` is 0, :py:obj:`~.cuMemAllocManaged` returns
    :py:obj:`~.CUDA_ERROR_INVALID_VALUE`. The pointer is valid on the CPU
    and on all GPUs in the system that support managed memory. All accesses
    to this pointer must obey the Unified Memory programming model.

    `flags` specifies the default stream association for this allocation.
    `flags` must be one of :py:obj:`~.CU_MEM_ATTACH_GLOBAL` or
    :py:obj:`~.CU_MEM_ATTACH_HOST`. If :py:obj:`~.CU_MEM_ATTACH_GLOBAL` is
    specified, then this memory is accessible from any stream on any
    device. If :py:obj:`~.CU_MEM_ATTACH_HOST` is specified, then the
    allocation should not be accessed from devices that have a zero value
    for the device attribute
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_CONCURRENT_MANAGED_ACCESS`; an explicit
    call to :py:obj:`~.cuStreamAttachMemAsync` will be required to enable
    access on such devices.

    If the association is later changed via
    :py:obj:`~.cuStreamAttachMemAsync` to a single stream, the default
    association as specifed during :py:obj:`~.cuMemAllocManaged` is
    restored when that stream is destroyed. For managed variables, the
    default association is always :py:obj:`~.CU_MEM_ATTACH_GLOBAL`. Note
    that destroying a stream is an asynchronous operation, and as a result,
    the change to default association won't happen until all work in the
    stream has completed.

    Memory allocated with :py:obj:`~.cuMemAllocManaged` should be released
    with :py:obj:`~.cuMemFree`.

    Device memory oversubscription is possible for GPUs that have a non-
    zero value for the device attribute
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_CONCURRENT_MANAGED_ACCESS`. Managed
    memory on such GPUs may be evicted from device memory to host memory at
    any time by the Unified Memory driver in order to make room for other
    allocations.

    In a multi-GPU system where all GPUs have a non-zero value for the
    device attribute
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_CONCURRENT_MANAGED_ACCESS`, managed
    memory may not be populated when this API returns and instead may be
    populated on access. In such systems, managed memory can migrate to any
    processor's memory at any time. The Unified Memory driver will employ
    heuristics to maintain data locality and prevent excessive page faults
    to the extent possible. The application can also guide the driver about
    memory usage patterns via :py:obj:`~.cuMemAdvise`. The application can
    also explicitly migrate memory to a desired processor's memory via
    :py:obj:`~.cuMemPrefetchAsync`.

    In a multi-GPU system where all of the GPUs have a zero value for the
    device attribute
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_CONCURRENT_MANAGED_ACCESS` and all the
    GPUs have peer-to-peer support with each other, the physical storage
    for managed memory is created on the GPU which is active at the time
    :py:obj:`~.cuMemAllocManaged` is called. All other GPUs will reference
    the data at reduced bandwidth via peer mappings over the PCIe bus. The
    Unified Memory driver does not migrate memory among such GPUs.

    In a multi-GPU system where not all GPUs have peer-to-peer support with
    each other and where the value of the device attribute
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_CONCURRENT_MANAGED_ACCESS` is zero for
    at least one of those GPUs, the location chosen for physical storage of
    managed memory is system-dependent.

    - On Linux, the location chosen will be device memory as long as the
      current set of active contexts are on devices that either have peer-
      to-peer support with each other or have a non-zero value for the
      device attribute
      :py:obj:`~.CU_DEVICE_ATTRIBUTE_CONCURRENT_MANAGED_ACCESS`. If there
      is an active context on a GPU that does not have a non-zero value for
      that device attribute and it does not have peer-to-peer support with
      the other devices that have active contexts on them, then the
      location for physical storage will be 'zero-copy' or host memory.
      Note that this means that managed memory that is located in device
      memory is migrated to host memory if a new context is created on a
      GPU that doesn't have a non-zero value for the device attribute and
      does not support peer-to-peer with at least one of the other devices
      that has an active context. This in turn implies that context
      creation may fail if there is insufficient host memory to migrate all
      managed allocations.

    - On Windows, the physical storage is always created in 'zero-copy' or
      host memory. All GPUs will reference the data at reduced bandwidth
      over the PCIe bus. In these circumstances, use of the environment
      variable CUDA_VISIBLE_DEVICES is recommended to restrict CUDA to only
      use those GPUs that have peer-to-peer support. Alternatively, users
      can also set CUDA_MANAGED_FORCE_DEVICE_ALLOC to a non-zero value to
      force the driver to always use device memory for physical storage.
      When this environment variable is set to a non-zero value, all
      contexts created in that process on devices that support managed
      memory have to be peer-to-peer compatible with each other. Context
      creation will fail if a context is created on a device that supports
      managed memory and is not peer-to-peer compatible with any of the
      other managed memory supporting devices on which contexts were
      previously created, even if those contexts have been destroyed. These
      environment variables are described in the CUDA programming guide
      under the "CUDA environment variables" section.

    - On ARM, managed memory is not available on discrete gpu with Drive
      PX-2.

    Parameters
    ----------
    bytesize : size_t
        Requested allocation size in bytes
    flags : unsigned int
        Must be one of :py:obj:`~.CU_MEM_ATTACH_GLOBAL` or
        :py:obj:`~.CU_MEM_ATTACH_HOST`

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_NOT_SUPPORTED`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`
    dptr : :py:obj:`~.CUdeviceptr`
        Returned device pointer

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cuDeviceGetAttribute`, :py:obj:`~.cuStreamAttachMemAsync`, :py:obj:`~.cudaMallocManaged`
    """
    cdef CUdeviceptr dptr = CUdeviceptr()
    err = cydriver.cuMemAllocManaged(<cydriver.CUdeviceptr*>dptr._ptr, bytesize, flags)
    return (CUresult(err), dptr)

@cython.embedsignature(True)
def cuDeviceGetByPCIBusId(char* pciBusId):
    """ Returns a handle to a compute device.

    Returns in `*device` a device handle given a PCI bus ID string.

    where `domain`, `bus`, `device`, and `function` are all hexadecimal
    values

    Parameters
    ----------
    pciBusId : bytes
        String in one of the following forms:

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`
    dev : :py:obj:`~.CUdevice`
        Returned device handle

    See Also
    --------
    :py:obj:`~.cuDeviceGet`, :py:obj:`~.cuDeviceGetAttribute`, :py:obj:`~.cuDeviceGetPCIBusId`, :py:obj:`~.cudaDeviceGetByPCIBusId`
    """
    cdef CUdevice dev = CUdevice()
    err = cydriver.cuDeviceGetByPCIBusId(<cydriver.CUdevice*>dev._ptr, pciBusId)
    return (CUresult(err), dev)

@cython.embedsignature(True)
def cuDeviceGetPCIBusId(int length, dev):
    """ Returns a PCI Bus Id string for the device.

    Returns an ASCII string identifying the device `dev` in the NULL-
    terminated string pointed to by `pciBusId`. `length` specifies the
    maximum length of the string that may be returned.

    where `domain`, `bus`, `device`, and `function` are all hexadecimal
    values. pciBusId should be large enough to store 13 characters
    including the NULL-terminator.

    Parameters
    ----------
    length : int
        Maximum length of string to store in `name`
    dev : :py:obj:`~.CUdevice`
        Device to get identifier string for

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`
    pciBusId : bytes
        Returned identifier string for the device in the following format

    See Also
    --------
    :py:obj:`~.cuDeviceGet`, :py:obj:`~.cuDeviceGetAttribute`, :py:obj:`~.cuDeviceGetByPCIBusId`, :py:obj:`~.cudaDeviceGetPCIBusId`
    """
    cdef cydriver.CUdevice cydev
    if dev is None:
        cydev = <cydriver.CUdevice>0
    elif isinstance(dev, (CUdevice,)):
        pdev = int(dev)
        cydev = <cydriver.CUdevice>pdev
    else:
        pdev = int(CUdevice(dev))
        cydev = <cydriver.CUdevice>pdev
    pypciBusId = b" " * length
    cdef char* pciBusId = pypciBusId
    err = cydriver.cuDeviceGetPCIBusId(pciBusId, length, cydev)
    return (CUresult(err), pypciBusId)

@cython.embedsignature(True)
def cuIpcGetEventHandle(event):
    """ Gets an interprocess handle for a previously allocated event.

    Takes as input a previously allocated event. This event must have been
    created with the :py:obj:`~.CU_EVENT_INTERPROCESS` and
    :py:obj:`~.CU_EVENT_DISABLE_TIMING` flags set. This opaque handle may
    be copied into other processes and opened with
    :py:obj:`~.cuIpcOpenEventHandle` to allow efficient hardware
    synchronization between GPU work in different processes.

    After the event has been opened in the importing process,
    :py:obj:`~.cuEventRecord`, :py:obj:`~.cuEventSynchronize`,
    :py:obj:`~.cuStreamWaitEvent` and :py:obj:`~.cuEventQuery` may be used
    in either process. Performing operations on the imported event after
    the exported event has been freed with :py:obj:`~.cuEventDestroy` will
    result in undefined behavior.

    IPC functionality is restricted to devices with support for unified
    addressing on Linux and Windows operating systems. IPC functionality on
    Windows is restricted to GPUs in TCC mode

    Parameters
    ----------
    event : :py:obj:`~.CUevent` or :py:obj:`~.cudaEvent_t`
        Event allocated with :py:obj:`~.CU_EVENT_INTERPROCESS` and
        :py:obj:`~.CU_EVENT_DISABLE_TIMING` flags.

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_MAP_FAILED`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    pHandle : :py:obj:`~.CUipcEventHandle`
        Pointer to a user allocated CUipcEventHandle in which to return the
        opaque event handle

    See Also
    --------
    :py:obj:`~.cuEventCreate`, :py:obj:`~.cuEventDestroy`, :py:obj:`~.cuEventSynchronize`, :py:obj:`~.cuEventQuery`, :py:obj:`~.cuStreamWaitEvent`, :py:obj:`~.cuIpcOpenEventHandle`, :py:obj:`~.cuIpcGetMemHandle`, :py:obj:`~.cuIpcOpenMemHandle`, :py:obj:`~.cuIpcCloseMemHandle`, :py:obj:`~.cudaIpcGetEventHandle`
    """
    cdef cydriver.CUevent cyevent
    if event is None:
        cyevent = <cydriver.CUevent><void_ptr>0
    elif isinstance(event, (CUevent,)):
        pevent = int(event)
        cyevent = <cydriver.CUevent><void_ptr>pevent
    else:
        pevent = int(CUevent(event))
        cyevent = <cydriver.CUevent><void_ptr>pevent
    cdef CUipcEventHandle pHandle = CUipcEventHandle()
    err = cydriver.cuIpcGetEventHandle(<cydriver.CUipcEventHandle*>pHandle._ptr, cyevent)
    return (CUresult(err), pHandle)

@cython.embedsignature(True)
def cuIpcOpenEventHandle(handle not None : CUipcEventHandle):
    """ Opens an interprocess event handle for use in the current process.

    Opens an interprocess event handle exported from another process with
    :py:obj:`~.cuIpcGetEventHandle`. This function returns a
    :py:obj:`~.CUevent` that behaves like a locally created event with the
    :py:obj:`~.CU_EVENT_DISABLE_TIMING` flag specified. This event must be
    freed with :py:obj:`~.cuEventDestroy`.

    Performing operations on the imported event after the exported event
    has been freed with :py:obj:`~.cuEventDestroy` will result in undefined
    behavior.

    IPC functionality is restricted to devices with support for unified
    addressing on Linux and Windows operating systems. IPC functionality on
    Windows is restricted to GPUs in TCC mode

    Parameters
    ----------
    handle : :py:obj:`~.CUipcEventHandle`
        Interprocess handle to open

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_MAP_FAILED`, :py:obj:`~.CUDA_ERROR_PEER_ACCESS_UNSUPPORTED`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    phEvent : :py:obj:`~.CUevent`
        Returns the imported event

    See Also
    --------
    :py:obj:`~.cuEventCreate`, :py:obj:`~.cuEventDestroy`, :py:obj:`~.cuEventSynchronize`, :py:obj:`~.cuEventQuery`, :py:obj:`~.cuStreamWaitEvent`, :py:obj:`~.cuIpcGetEventHandle`, :py:obj:`~.cuIpcGetMemHandle`, :py:obj:`~.cuIpcOpenMemHandle`, :py:obj:`~.cuIpcCloseMemHandle`, :py:obj:`~.cudaIpcOpenEventHandle`
    """
    cdef CUevent phEvent = CUevent()
    err = cydriver.cuIpcOpenEventHandle(<cydriver.CUevent*>phEvent._ptr, handle._ptr[0])
    return (CUresult(err), phEvent)

@cython.embedsignature(True)
def cuIpcGetMemHandle(dptr):
    """ Gets an interprocess memory handle for an existing device memory allocation.

    Takes a pointer to the base of an existing device memory allocation
    created with :py:obj:`~.cuMemAlloc` and exports it for use in another
    process. This is a lightweight operation and may be called multiple
    times on an allocation without adverse effects.

    If a region of memory is freed with :py:obj:`~.cuMemFree` and a
    subsequent call to :py:obj:`~.cuMemAlloc` returns memory with the same
    device address, :py:obj:`~.cuIpcGetMemHandle` will return a unique
    handle for the new memory.

    IPC functionality is restricted to devices with support for unified
    addressing on Linux and Windows operating systems. IPC functionality on
    Windows is restricted to GPUs in TCC mode

    Parameters
    ----------
    dptr : :py:obj:`~.CUdeviceptr`
        Base pointer to previously allocated device memory

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_MAP_FAILED`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    pHandle : :py:obj:`~.CUipcMemHandle`
        Pointer to user allocated :py:obj:`~.CUipcMemHandle` to return the
        handle in.

    See Also
    --------
    :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuIpcGetEventHandle`, :py:obj:`~.cuIpcOpenEventHandle`, :py:obj:`~.cuIpcOpenMemHandle`, :py:obj:`~.cuIpcCloseMemHandle`, :py:obj:`~.cudaIpcGetMemHandle`
    """
    cdef cydriver.CUdeviceptr cydptr
    if dptr is None:
        cydptr = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(dptr, (CUdeviceptr,)):
        pdptr = int(dptr)
        cydptr = <cydriver.CUdeviceptr><void_ptr>pdptr
    else:
        pdptr = int(CUdeviceptr(dptr))
        cydptr = <cydriver.CUdeviceptr><void_ptr>pdptr
    cdef CUipcMemHandle pHandle = CUipcMemHandle()
    err = cydriver.cuIpcGetMemHandle(<cydriver.CUipcMemHandle*>pHandle._ptr, cydptr)
    return (CUresult(err), pHandle)

@cython.embedsignature(True)
def cuIpcOpenMemHandle(handle not None : CUipcMemHandle, unsigned int Flags):
    """ Opens an interprocess memory handle exported from another process and returns a device pointer usable in the local process.

    Maps memory exported from another process with
    :py:obj:`~.cuIpcGetMemHandle` into the current device address space.
    For contexts on different devices :py:obj:`~.cuIpcOpenMemHandle` can
    attempt to enable peer access between the devices as if the user called
    :py:obj:`~.cuCtxEnablePeerAccess`. This behavior is controlled by the
    :py:obj:`~.CU_IPC_MEM_LAZY_ENABLE_PEER_ACCESS` flag.
    :py:obj:`~.cuDeviceCanAccessPeer` can determine if a mapping is
    possible.

    Contexts that may open :py:obj:`~.CUipcMemHandles` are restricted in
    the following way. :py:obj:`~.CUipcMemHandles` from each
    :py:obj:`~.CUdevice` in a given process may only be opened by one
    :py:obj:`~.CUcontext` per :py:obj:`~.CUdevice` per other process.

    If the memory handle has already been opened by the current context,
    the reference count on the handle is incremented by 1 and the existing
    device pointer is returned.

    Memory returned from :py:obj:`~.cuIpcOpenMemHandle` must be freed with
    :py:obj:`~.cuIpcCloseMemHandle`.

    Calling :py:obj:`~.cuMemFree` on an exported memory region before
    calling :py:obj:`~.cuIpcCloseMemHandle` in the importing context will
    result in undefined behavior.

    IPC functionality is restricted to devices with support for unified
    addressing on Linux and Windows operating systems. IPC functionality on
    Windows is restricted to GPUs in TCC mode

    Parameters
    ----------
    handle : :py:obj:`~.CUipcMemHandle`
        :py:obj:`~.CUipcMemHandle` to open
    Flags : unsigned int
        Flags for this operation. Must be specified as
        :py:obj:`~.CU_IPC_MEM_LAZY_ENABLE_PEER_ACCESS`

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_MAP_FAILED`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_TOO_MANY_PEERS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    pdptr : :py:obj:`~.CUdeviceptr`
        Returned device pointer

    See Also
    --------
    :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuIpcGetEventHandle`, :py:obj:`~.cuIpcOpenEventHandle`, :py:obj:`~.cuIpcGetMemHandle`, :py:obj:`~.cuIpcCloseMemHandle`, :py:obj:`~.cuCtxEnablePeerAccess`, :py:obj:`~.cuDeviceCanAccessPeer`, :py:obj:`~.cudaIpcOpenMemHandle`

    Notes
    -----
    No guarantees are made about the address returned in `*pdptr`. In particular, multiple processes may not receive the same address for the same `handle`.
    """
    cdef CUdeviceptr pdptr = CUdeviceptr()
    err = cydriver.cuIpcOpenMemHandle(<cydriver.CUdeviceptr*>pdptr._ptr, handle._ptr[0], Flags)
    return (CUresult(err), pdptr)

@cython.embedsignature(True)
def cuIpcCloseMemHandle(dptr):
    """ Attempts to close memory mapped with :py:obj:`~.cuIpcOpenMemHandle`.

    Decrements the reference count of the memory returned by
    :py:obj:`~.cuIpcOpenMemHandle` by 1. When the reference count reaches
    0, this API unmaps the memory. The original allocation in the exporting
    process as well as imported mappings in other processes will be
    unaffected.

    Any resources used to enable peer access will be freed if this is the
    last mapping using them.

    IPC functionality is restricted to devices with support for unified
    addressing on Linux and Windows operating systems. IPC functionality on
    Windows is restricted to GPUs in TCC mode

    Parameters
    ----------
    dptr : :py:obj:`~.CUdeviceptr`
        Device pointer returned by :py:obj:`~.cuIpcOpenMemHandle`

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_MAP_FAILED`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuIpcGetEventHandle`, :py:obj:`~.cuIpcOpenEventHandle`, :py:obj:`~.cuIpcGetMemHandle`, :py:obj:`~.cuIpcOpenMemHandle`, :py:obj:`~.cudaIpcCloseMemHandle`
    """
    cdef cydriver.CUdeviceptr cydptr
    if dptr is None:
        cydptr = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(dptr, (CUdeviceptr,)):
        pdptr = int(dptr)
        cydptr = <cydriver.CUdeviceptr><void_ptr>pdptr
    else:
        pdptr = int(CUdeviceptr(dptr))
        cydptr = <cydriver.CUdeviceptr><void_ptr>pdptr
    err = cydriver.cuIpcCloseMemHandle(cydptr)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemHostRegister(p, size_t bytesize, unsigned int Flags):
    """ Registers an existing host memory range for use by CUDA.

    Page-locks the memory range specified by `p` and `bytesize` and maps it
    for the device(s) as specified by `Flags`. This memory range also is
    added to the same tracking mechanism as :py:obj:`~.cuMemHostAlloc` to
    automatically accelerate calls to functions such as
    :py:obj:`~.cuMemcpyHtoD()`. Since the memory can be accessed directly
    by the device, it can be read or written with much higher bandwidth
    than pageable memory that has not been registered. Page-locking
    excessive amounts of memory may degrade system performance, since it
    reduces the amount of memory available to the system for paging. As a
    result, this function is best used sparingly to register staging areas
    for data exchange between host and device.

    The `Flags` parameter enables different options to be specified that
    affect the allocation, as follows.

    - :py:obj:`~.CU_MEMHOSTREGISTER_PORTABLE`: The memory returned by this
      call will be considered as pinned memory by all CUDA contexts, not
      just the one that performed the allocation.

    - :py:obj:`~.CU_MEMHOSTREGISTER_DEVICEMAP`: Maps the allocation into
      the CUDA address space. The device pointer to the memory may be
      obtained by calling :py:obj:`~.cuMemHostGetDevicePointer()`.

    - :py:obj:`~.CU_MEMHOSTREGISTER_IOMEMORY`: The pointer is treated as
      pointing to some I/O memory space, e.g. the PCI Express resource of a
      3rd party device.

    - :py:obj:`~.CU_MEMHOSTREGISTER_READ_ONLY`: The pointer is treated as
      pointing to memory that is considered read-only by the device. On
      platforms without
      :py:obj:`~.CU_DEVICE_ATTRIBUTE_PAGEABLE_MEMORY_ACCESS_USES_HOST_PAGE_TABLES`,
      this flag is required in order to register memory mapped to the CPU
      as read-only. Support for the use of this flag can be queried from
      the device attribute
      :py:obj:`~.CU_DEVICE_ATTRIBUTE_READ_ONLY_HOST_REGISTER_SUPPORTED`.
      Using this flag with a current context associated with a device that
      does not have this attribute set will cause
      :py:obj:`~.cuMemHostRegister` to error with CUDA_ERROR_NOT_SUPPORTED.

    All of these flags are orthogonal to one another: a developer may page-
    lock memory that is portable or mapped with no restrictions.

    The :py:obj:`~.CU_MEMHOSTREGISTER_DEVICEMAP` flag may be specified on
    CUDA contexts for devices that do not support mapped pinned memory. The
    failure is deferred to :py:obj:`~.cuMemHostGetDevicePointer()` because
    the memory may be mapped into other CUDA contexts via the
    :py:obj:`~.CU_MEMHOSTREGISTER_PORTABLE` flag.

    For devices that have a non-zero value for the device attribute
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_CAN_USE_HOST_POINTER_FOR_REGISTERED_MEM`,
    the memory can also be accessed from the device using the host pointer
    `p`. The device pointer returned by
    :py:obj:`~.cuMemHostGetDevicePointer()` may or may not match the
    original host pointer `ptr` and depends on the devices visible to the
    application. If all devices visible to the application have a non-zero
    value for the device attribute, the device pointer returned by
    :py:obj:`~.cuMemHostGetDevicePointer()` will match the original pointer
    `ptr`. If any device visible to the application has a zero value for
    the device attribute, the device pointer returned by
    :py:obj:`~.cuMemHostGetDevicePointer()` will not match the original
    host pointer `ptr`, but it will be suitable for use on all devices
    provided Unified Virtual Addressing is enabled. In such systems, it is
    valid to access the memory using either pointer on devices that have a
    non-zero value for the device attribute. Note however that such devices
    should access the memory using only of the two pointers and not both.

    The memory page-locked by this function must be unregistered with
    :py:obj:`~.cuMemHostUnregister()`.

    Parameters
    ----------
    p : Any
        Host pointer to memory to page-lock
    bytesize : size_t
        Size in bytes of the address range to page-lock
    Flags : unsigned int
        Flags for allocation request

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_HOST_MEMORY_ALREADY_REGISTERED`, :py:obj:`~.CUDA_ERROR_NOT_PERMITTED`, :py:obj:`~.CUDA_ERROR_NOT_SUPPORTED`

    See Also
    --------
    :py:obj:`~.cuMemHostUnregister`, :py:obj:`~.cuMemHostGetFlags`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cudaHostRegister`
    """
    cyp = utils.HelperInputVoidPtr(p)
    cdef void* cyp_ptr = <void*><void_ptr>cyp.cptr
    err = cydriver.cuMemHostRegister(cyp_ptr, bytesize, Flags)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemHostUnregister(p):
    """ Unregisters a memory range that was registered with cuMemHostRegister.

    Unmaps the memory range whose base address is specified by `p`, and
    makes it pageable again.

    The base address must be the same one specified to
    :py:obj:`~.cuMemHostRegister()`.

    Parameters
    ----------
    p : Any
        Host pointer to memory to unregister

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_HOST_MEMORY_NOT_REGISTERED`,

    See Also
    --------
    :py:obj:`~.cuMemHostRegister`, :py:obj:`~.cudaHostUnregister`
    """
    cyp = utils.HelperInputVoidPtr(p)
    cdef void* cyp_ptr = <void*><void_ptr>cyp.cptr
    err = cydriver.cuMemHostUnregister(cyp_ptr)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpy(dst, src, size_t ByteCount):
    """ Copies memory.

    Copies data between two pointers. `dst` and `src` are base pointers of
    the destination and source, respectively. `ByteCount` specifies the
    number of bytes to copy. Note that this function infers the type of the
    transfer (host to host, host to device, device to device, or device to
    host) from the pointer values. This function is only allowed in
    contexts which support unified addressing.

    Parameters
    ----------
    dst : :py:obj:`~.CUdeviceptr`
        Destination unified virtual address space pointer
    src : :py:obj:`~.CUdeviceptr`
        Source unified virtual address space pointer
    ByteCount : size_t
        Size of memory copy in bytes

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaMemcpy`, :py:obj:`~.cudaMemcpyToSymbol`, :py:obj:`~.cudaMemcpyFromSymbol`
    """
    cdef cydriver.CUdeviceptr cysrc
    if src is None:
        cysrc = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(src, (CUdeviceptr,)):
        psrc = int(src)
        cysrc = <cydriver.CUdeviceptr><void_ptr>psrc
    else:
        psrc = int(CUdeviceptr(src))
        cysrc = <cydriver.CUdeviceptr><void_ptr>psrc
    cdef cydriver.CUdeviceptr cydst
    if dst is None:
        cydst = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(dst, (CUdeviceptr,)):
        pdst = int(dst)
        cydst = <cydriver.CUdeviceptr><void_ptr>pdst
    else:
        pdst = int(CUdeviceptr(dst))
        cydst = <cydriver.CUdeviceptr><void_ptr>pdst
    err = cydriver.cuMemcpy(cydst, cysrc, ByteCount)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpyPeer(dstDevice, dstContext, srcDevice, srcContext, size_t ByteCount):
    """ Copies device memory between two contexts.

    Copies from device memory in one context to device memory in another
    context. `dstDevice` is the base device pointer of the destination
    memory and `dstContext` is the destination context. `srcDevice` is the
    base device pointer of the source memory and `srcContext` is the source
    pointer. `ByteCount` specifies the number of bytes to copy.

    Parameters
    ----------
    dstDevice : :py:obj:`~.CUdeviceptr`
        Destination device pointer
    dstContext : :py:obj:`~.CUcontext`
        Destination context
    srcDevice : :py:obj:`~.CUdeviceptr`
        Source device pointer
    srcContext : :py:obj:`~.CUcontext`
        Source context
    ByteCount : size_t
        Size of memory copy in bytes

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpy3DPeer`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyPeerAsync`, :py:obj:`~.cuMemcpy3DPeerAsync`, :py:obj:`~.cudaMemcpyPeer`
    """
    cdef cydriver.CUcontext cysrcContext
    if srcContext is None:
        cysrcContext = <cydriver.CUcontext><void_ptr>0
    elif isinstance(srcContext, (CUcontext,)):
        psrcContext = int(srcContext)
        cysrcContext = <cydriver.CUcontext><void_ptr>psrcContext
    else:
        psrcContext = int(CUcontext(srcContext))
        cysrcContext = <cydriver.CUcontext><void_ptr>psrcContext
    cdef cydriver.CUdeviceptr cysrcDevice
    if srcDevice is None:
        cysrcDevice = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(srcDevice, (CUdeviceptr,)):
        psrcDevice = int(srcDevice)
        cysrcDevice = <cydriver.CUdeviceptr><void_ptr>psrcDevice
    else:
        psrcDevice = int(CUdeviceptr(srcDevice))
        cysrcDevice = <cydriver.CUdeviceptr><void_ptr>psrcDevice
    cdef cydriver.CUcontext cydstContext
    if dstContext is None:
        cydstContext = <cydriver.CUcontext><void_ptr>0
    elif isinstance(dstContext, (CUcontext,)):
        pdstContext = int(dstContext)
        cydstContext = <cydriver.CUcontext><void_ptr>pdstContext
    else:
        pdstContext = int(CUcontext(dstContext))
        cydstContext = <cydriver.CUcontext><void_ptr>pdstContext
    cdef cydriver.CUdeviceptr cydstDevice
    if dstDevice is None:
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(dstDevice, (CUdeviceptr,)):
        pdstDevice = int(dstDevice)
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
    else:
        pdstDevice = int(CUdeviceptr(dstDevice))
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
    err = cydriver.cuMemcpyPeer(cydstDevice, cydstContext, cysrcDevice, cysrcContext, ByteCount)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpyHtoD(dstDevice, srcHost, size_t ByteCount):
    """ Copies memory from Host to Device.

    Copies from host memory to device memory. `dstDevice` and `srcHost` are
    the base addresses of the destination and source, respectively.
    `ByteCount` specifies the number of bytes to copy.

    Parameters
    ----------
    dstDevice : :py:obj:`~.CUdeviceptr`
        Destination device pointer
    srcHost : Any
        Source host pointer
    ByteCount : size_t
        Size of memory copy in bytes

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaMemcpy`, :py:obj:`~.cudaMemcpyToSymbol`
    """
    cdef cydriver.CUdeviceptr cydstDevice
    if dstDevice is None:
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(dstDevice, (CUdeviceptr,)):
        pdstDevice = int(dstDevice)
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
    else:
        pdstDevice = int(CUdeviceptr(dstDevice))
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
    cysrcHost = utils.HelperInputVoidPtr(srcHost)
    cdef void* cysrcHost_ptr = <void*><void_ptr>cysrcHost.cptr
    err = cydriver.cuMemcpyHtoD(cydstDevice, cysrcHost_ptr, ByteCount)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpyDtoH(dstHost, srcDevice, size_t ByteCount):
    """ Copies memory from Device to Host.

    Copies from device to host memory. `dstHost` and `srcDevice` specify
    the base pointers of the destination and source, respectively.
    `ByteCount` specifies the number of bytes to copy.

    Parameters
    ----------
    dstHost : Any
        Destination host pointer
    srcDevice : :py:obj:`~.CUdeviceptr`
        Source device pointer
    ByteCount : size_t
        Size of memory copy in bytes

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaMemcpy`, :py:obj:`~.cudaMemcpyFromSymbol`
    """
    cdef cydriver.CUdeviceptr cysrcDevice
    if srcDevice is None:
        cysrcDevice = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(srcDevice, (CUdeviceptr,)):
        psrcDevice = int(srcDevice)
        cysrcDevice = <cydriver.CUdeviceptr><void_ptr>psrcDevice
    else:
        psrcDevice = int(CUdeviceptr(srcDevice))
        cysrcDevice = <cydriver.CUdeviceptr><void_ptr>psrcDevice
    cydstHost = utils.HelperInputVoidPtr(dstHost)
    cdef void* cydstHost_ptr = <void*><void_ptr>cydstHost.cptr
    err = cydriver.cuMemcpyDtoH(cydstHost_ptr, cysrcDevice, ByteCount)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpyDtoD(dstDevice, srcDevice, size_t ByteCount):
    """ Copies memory from Device to Device.

    Copies from device memory to device memory. `dstDevice` and `srcDevice`
    are the base pointers of the destination and source, respectively.
    `ByteCount` specifies the number of bytes to copy.

    Parameters
    ----------
    dstDevice : :py:obj:`~.CUdeviceptr`
        Destination device pointer
    srcDevice : :py:obj:`~.CUdeviceptr`
        Source device pointer
    ByteCount : size_t
        Size of memory copy in bytes

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaMemcpy`, :py:obj:`~.cudaMemcpyToSymbol`, :py:obj:`~.cudaMemcpyFromSymbol`
    """
    cdef cydriver.CUdeviceptr cysrcDevice
    if srcDevice is None:
        cysrcDevice = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(srcDevice, (CUdeviceptr,)):
        psrcDevice = int(srcDevice)
        cysrcDevice = <cydriver.CUdeviceptr><void_ptr>psrcDevice
    else:
        psrcDevice = int(CUdeviceptr(srcDevice))
        cysrcDevice = <cydriver.CUdeviceptr><void_ptr>psrcDevice
    cdef cydriver.CUdeviceptr cydstDevice
    if dstDevice is None:
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(dstDevice, (CUdeviceptr,)):
        pdstDevice = int(dstDevice)
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
    else:
        pdstDevice = int(CUdeviceptr(dstDevice))
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
    err = cydriver.cuMemcpyDtoD(cydstDevice, cysrcDevice, ByteCount)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpyDtoA(dstArray, size_t dstOffset, srcDevice, size_t ByteCount):
    """ Copies memory from Device to Array.

    Copies from device memory to a 1D CUDA array. `dstArray` and
    `dstOffset` specify the CUDA array handle and starting index of the
    destination data. `srcDevice` specifies the base pointer of the source.
    `ByteCount` specifies the number of bytes to copy.

    Parameters
    ----------
    dstArray : :py:obj:`~.CUarray`
        Destination array
    dstOffset : size_t
        Offset in bytes of destination array
    srcDevice : :py:obj:`~.CUdeviceptr`
        Source device pointer
    ByteCount : size_t
        Size of memory copy in bytes

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaMemcpyToArray`
    """
    cdef cydriver.CUdeviceptr cysrcDevice
    if srcDevice is None:
        cysrcDevice = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(srcDevice, (CUdeviceptr,)):
        psrcDevice = int(srcDevice)
        cysrcDevice = <cydriver.CUdeviceptr><void_ptr>psrcDevice
    else:
        psrcDevice = int(CUdeviceptr(srcDevice))
        cysrcDevice = <cydriver.CUdeviceptr><void_ptr>psrcDevice
    cdef cydriver.CUarray cydstArray
    if dstArray is None:
        cydstArray = <cydriver.CUarray><void_ptr>0
    elif isinstance(dstArray, (CUarray,)):
        pdstArray = int(dstArray)
        cydstArray = <cydriver.CUarray><void_ptr>pdstArray
    else:
        pdstArray = int(CUarray(dstArray))
        cydstArray = <cydriver.CUarray><void_ptr>pdstArray
    err = cydriver.cuMemcpyDtoA(cydstArray, dstOffset, cysrcDevice, ByteCount)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpyAtoD(dstDevice, srcArray, size_t srcOffset, size_t ByteCount):
    """ Copies memory from Array to Device.

    Copies from one 1D CUDA array to device memory. `dstDevice` specifies
    the base pointer of the destination and must be naturally aligned with
    the CUDA array elements. `srcArray` and `srcOffset` specify the CUDA
    array handle and the offset in bytes into the array where the copy is
    to begin. `ByteCount` specifies the number of bytes to copy and must be
    evenly divisible by the array element size.

    Parameters
    ----------
    dstDevice : :py:obj:`~.CUdeviceptr`
        Destination device pointer
    srcArray : :py:obj:`~.CUarray`
        Source array
    srcOffset : size_t
        Offset in bytes of source array
    ByteCount : size_t
        Size of memory copy in bytes

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaMemcpyFromArray`
    """
    cdef cydriver.CUarray cysrcArray
    if srcArray is None:
        cysrcArray = <cydriver.CUarray><void_ptr>0
    elif isinstance(srcArray, (CUarray,)):
        psrcArray = int(srcArray)
        cysrcArray = <cydriver.CUarray><void_ptr>psrcArray
    else:
        psrcArray = int(CUarray(srcArray))
        cysrcArray = <cydriver.CUarray><void_ptr>psrcArray
    cdef cydriver.CUdeviceptr cydstDevice
    if dstDevice is None:
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(dstDevice, (CUdeviceptr,)):
        pdstDevice = int(dstDevice)
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
    else:
        pdstDevice = int(CUdeviceptr(dstDevice))
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
    err = cydriver.cuMemcpyAtoD(cydstDevice, cysrcArray, srcOffset, ByteCount)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpyHtoA(dstArray, size_t dstOffset, srcHost, size_t ByteCount):
    """ Copies memory from Host to Array.

    Copies from host memory to a 1D CUDA array. `dstArray` and `dstOffset`
    specify the CUDA array handle and starting offset in bytes of the
    destination data. `pSrc` specifies the base address of the source.
    `ByteCount` specifies the number of bytes to copy.

    Parameters
    ----------
    dstArray : :py:obj:`~.CUarray`
        Destination array
    dstOffset : size_t
        Offset in bytes of destination array
    srcHost : Any
        Source host pointer
    ByteCount : size_t
        Size of memory copy in bytes

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaMemcpyToArray`
    """
    cdef cydriver.CUarray cydstArray
    if dstArray is None:
        cydstArray = <cydriver.CUarray><void_ptr>0
    elif isinstance(dstArray, (CUarray,)):
        pdstArray = int(dstArray)
        cydstArray = <cydriver.CUarray><void_ptr>pdstArray
    else:
        pdstArray = int(CUarray(dstArray))
        cydstArray = <cydriver.CUarray><void_ptr>pdstArray
    cysrcHost = utils.HelperInputVoidPtr(srcHost)
    cdef void* cysrcHost_ptr = <void*><void_ptr>cysrcHost.cptr
    err = cydriver.cuMemcpyHtoA(cydstArray, dstOffset, cysrcHost_ptr, ByteCount)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpyAtoH(dstHost, srcArray, size_t srcOffset, size_t ByteCount):
    """ Copies memory from Array to Host.

    Copies from one 1D CUDA array to host memory. `dstHost` specifies the
    base pointer of the destination. `srcArray` and `srcOffset` specify the
    CUDA array handle and starting offset in bytes of the source data.
    `ByteCount` specifies the number of bytes to copy.

    Parameters
    ----------
    dstHost : Any
        Destination device pointer
    srcArray : :py:obj:`~.CUarray`
        Source array
    srcOffset : size_t
        Offset in bytes of source array
    ByteCount : size_t
        Size of memory copy in bytes

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaMemcpyFromArray`
    """
    cdef cydriver.CUarray cysrcArray
    if srcArray is None:
        cysrcArray = <cydriver.CUarray><void_ptr>0
    elif isinstance(srcArray, (CUarray,)):
        psrcArray = int(srcArray)
        cysrcArray = <cydriver.CUarray><void_ptr>psrcArray
    else:
        psrcArray = int(CUarray(srcArray))
        cysrcArray = <cydriver.CUarray><void_ptr>psrcArray
    cydstHost = utils.HelperInputVoidPtr(dstHost)
    cdef void* cydstHost_ptr = <void*><void_ptr>cydstHost.cptr
    err = cydriver.cuMemcpyAtoH(cydstHost_ptr, cysrcArray, srcOffset, ByteCount)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpyAtoA(dstArray, size_t dstOffset, srcArray, size_t srcOffset, size_t ByteCount):
    """ Copies memory from Array to Array.

    Copies from one 1D CUDA array to another. `dstArray` and `srcArray`
    specify the handles of the destination and source CUDA arrays for the
    copy, respectively. `dstOffset` and `srcOffset` specify the destination
    and source offsets in bytes into the CUDA arrays. `ByteCount` is the
    number of bytes to be copied. The size of the elements in the CUDA
    arrays need not be the same format, but the elements must be the same
    size; and count must be evenly divisible by that size.

    Parameters
    ----------
    dstArray : :py:obj:`~.CUarray`
        Destination array
    dstOffset : size_t
        Offset in bytes of destination array
    srcArray : :py:obj:`~.CUarray`
        Source array
    srcOffset : size_t
        Offset in bytes of source array
    ByteCount : size_t
        Size of memory copy in bytes

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaMemcpyArrayToArray`
    """
    cdef cydriver.CUarray cysrcArray
    if srcArray is None:
        cysrcArray = <cydriver.CUarray><void_ptr>0
    elif isinstance(srcArray, (CUarray,)):
        psrcArray = int(srcArray)
        cysrcArray = <cydriver.CUarray><void_ptr>psrcArray
    else:
        psrcArray = int(CUarray(srcArray))
        cysrcArray = <cydriver.CUarray><void_ptr>psrcArray
    cdef cydriver.CUarray cydstArray
    if dstArray is None:
        cydstArray = <cydriver.CUarray><void_ptr>0
    elif isinstance(dstArray, (CUarray,)):
        pdstArray = int(dstArray)
        cydstArray = <cydriver.CUarray><void_ptr>pdstArray
    else:
        pdstArray = int(CUarray(dstArray))
        cydstArray = <cydriver.CUarray><void_ptr>pdstArray
    err = cydriver.cuMemcpyAtoA(cydstArray, dstOffset, cysrcArray, srcOffset, ByteCount)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpy2D(pCopy : Optional[CUDA_MEMCPY2D]):
    """ Copies memory for 2D arrays.

    Perform a 2D memory copy according to the parameters specified in
    `pCopy`. The :py:obj:`~.CUDA_MEMCPY2D` structure is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    where:

    - :py:obj:`~.srcMemoryType` and :py:obj:`~.dstMemoryType` specify the
      type of memory of the source and destination, respectively;
      :py:obj:`~.CUmemorytype_enum` is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    If :py:obj:`~.srcMemoryType` is :py:obj:`~.CU_MEMORYTYPE_UNIFIED`,
    :py:obj:`~.srcDevice` and :py:obj:`~.srcPitch` specify the (unified
    virtual address space) base address of the source data and the bytes
    per row to apply. :py:obj:`~.srcArray` is ignored. This value may be
    used only if unified addressing is supported in the calling context.

    If :py:obj:`~.srcMemoryType` is :py:obj:`~.CU_MEMORYTYPE_HOST`,
    :py:obj:`~.srcHost` and :py:obj:`~.srcPitch` specify the (host) base
    address of the source data and the bytes per row to apply.
    :py:obj:`~.srcArray` is ignored.

    If :py:obj:`~.srcMemoryType` is :py:obj:`~.CU_MEMORYTYPE_DEVICE`,
    :py:obj:`~.srcDevice` and :py:obj:`~.srcPitch` specify the (device)
    base address of the source data and the bytes per row to apply.
    :py:obj:`~.srcArray` is ignored.

    If :py:obj:`~.srcMemoryType` is :py:obj:`~.CU_MEMORYTYPE_ARRAY`,
    :py:obj:`~.srcArray` specifies the handle of the source data.
    :py:obj:`~.srcHost`, :py:obj:`~.srcDevice` and :py:obj:`~.srcPitch` are
    ignored.

    If :py:obj:`~.dstMemoryType` is :py:obj:`~.CU_MEMORYTYPE_HOST`,
    :py:obj:`~.dstHost` and :py:obj:`~.dstPitch` specify the (host) base
    address of the destination data and the bytes per row to apply.
    :py:obj:`~.dstArray` is ignored.

    If :py:obj:`~.dstMemoryType` is :py:obj:`~.CU_MEMORYTYPE_UNIFIED`,
    :py:obj:`~.dstDevice` and :py:obj:`~.dstPitch` specify the (unified
    virtual address space) base address of the source data and the bytes
    per row to apply. :py:obj:`~.dstArray` is ignored. This value may be
    used only if unified addressing is supported in the calling context.

    If :py:obj:`~.dstMemoryType` is :py:obj:`~.CU_MEMORYTYPE_DEVICE`,
    :py:obj:`~.dstDevice` and :py:obj:`~.dstPitch` specify the (device)
    base address of the destination data and the bytes per row to apply.
    :py:obj:`~.dstArray` is ignored.

    If :py:obj:`~.dstMemoryType` is :py:obj:`~.CU_MEMORYTYPE_ARRAY`,
    :py:obj:`~.dstArray` specifies the handle of the destination data.
    :py:obj:`~.dstHost`, :py:obj:`~.dstDevice` and :py:obj:`~.dstPitch` are
    ignored.

    - :py:obj:`~.srcXInBytes` and :py:obj:`~.srcY` specify the base address
      of the source data for the copy.

    For host pointers, the starting address is

    **View CUDA Toolkit Documentation for a C++ code example**

    For device pointers, the starting address is

    **View CUDA Toolkit Documentation for a C++ code example**

    For CUDA arrays, :py:obj:`~.srcXInBytes` must be evenly divisible by
    the array element size.

    - :py:obj:`~.dstXInBytes` and :py:obj:`~.dstY` specify the base address
      of the destination data for the copy.

    For host pointers, the base address is

    **View CUDA Toolkit Documentation for a C++ code example**

    For device pointers, the starting address is

    **View CUDA Toolkit Documentation for a C++ code example**

    For CUDA arrays, :py:obj:`~.dstXInBytes` must be evenly divisible by
    the array element size.

    - :py:obj:`~.WidthInBytes` and :py:obj:`~.Height` specify the width (in
      bytes) and height of the 2D copy being performed.

    - If specified, :py:obj:`~.srcPitch` must be greater than or equal to
      :py:obj:`~.WidthInBytes` + :py:obj:`~.srcXInBytes`, and
      :py:obj:`~.dstPitch` must be greater than or equal to
      :py:obj:`~.WidthInBytes` + dstXInBytes.

    :py:obj:`~.cuMemcpy2D()` returns an error if any pitch is greater than
    the maximum allowed (:py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_PITCH`).
    :py:obj:`~.cuMemAllocPitch()` passes back pitches that always work with
    :py:obj:`~.cuMemcpy2D()`. On intra-device memory copies (device to
    device, CUDA array to device, CUDA array to CUDA array),
    :py:obj:`~.cuMemcpy2D()` may fail for pitches not computed by
    :py:obj:`~.cuMemAllocPitch()`. :py:obj:`~.cuMemcpy2DUnaligned()` does
    not have this restriction, but may run significantly slower in the
    cases where :py:obj:`~.cuMemcpy2D()` would have returned an error code.

    Parameters
    ----------
    pCopy : :py:obj:`~.CUDA_MEMCPY2D`
        Parameters for the memory copy

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaMemcpy2D`, :py:obj:`~.cudaMemcpy2DToArray`, :py:obj:`~.cudaMemcpy2DFromArray`
    """
    cdef cydriver.CUDA_MEMCPY2D* cypCopy_ptr = pCopy._ptr if pCopy != None else NULL
    err = cydriver.cuMemcpy2D(cypCopy_ptr)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpy2DUnaligned(pCopy : Optional[CUDA_MEMCPY2D]):
    """ Copies memory for 2D arrays.

    Perform a 2D memory copy according to the parameters specified in
    `pCopy`. The :py:obj:`~.CUDA_MEMCPY2D` structure is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    where:

    - :py:obj:`~.srcMemoryType` and :py:obj:`~.dstMemoryType` specify the
      type of memory of the source and destination, respectively;
      :py:obj:`~.CUmemorytype_enum` is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    If :py:obj:`~.srcMemoryType` is :py:obj:`~.CU_MEMORYTYPE_UNIFIED`,
    :py:obj:`~.srcDevice` and :py:obj:`~.srcPitch` specify the (unified
    virtual address space) base address of the source data and the bytes
    per row to apply. :py:obj:`~.srcArray` is ignored. This value may be
    used only if unified addressing is supported in the calling context.

    If :py:obj:`~.srcMemoryType` is :py:obj:`~.CU_MEMORYTYPE_HOST`,
    :py:obj:`~.srcHost` and :py:obj:`~.srcPitch` specify the (host) base
    address of the source data and the bytes per row to apply.
    :py:obj:`~.srcArray` is ignored.

    If :py:obj:`~.srcMemoryType` is :py:obj:`~.CU_MEMORYTYPE_DEVICE`,
    :py:obj:`~.srcDevice` and :py:obj:`~.srcPitch` specify the (device)
    base address of the source data and the bytes per row to apply.
    :py:obj:`~.srcArray` is ignored.

    If :py:obj:`~.srcMemoryType` is :py:obj:`~.CU_MEMORYTYPE_ARRAY`,
    :py:obj:`~.srcArray` specifies the handle of the source data.
    :py:obj:`~.srcHost`, :py:obj:`~.srcDevice` and :py:obj:`~.srcPitch` are
    ignored.

    If :py:obj:`~.dstMemoryType` is :py:obj:`~.CU_MEMORYTYPE_UNIFIED`,
    :py:obj:`~.dstDevice` and :py:obj:`~.dstPitch` specify the (unified
    virtual address space) base address of the source data and the bytes
    per row to apply. :py:obj:`~.dstArray` is ignored. This value may be
    used only if unified addressing is supported in the calling context.

    If :py:obj:`~.dstMemoryType` is :py:obj:`~.CU_MEMORYTYPE_HOST`,
    :py:obj:`~.dstHost` and :py:obj:`~.dstPitch` specify the (host) base
    address of the destination data and the bytes per row to apply.
    :py:obj:`~.dstArray` is ignored.

    If :py:obj:`~.dstMemoryType` is :py:obj:`~.CU_MEMORYTYPE_DEVICE`,
    :py:obj:`~.dstDevice` and :py:obj:`~.dstPitch` specify the (device)
    base address of the destination data and the bytes per row to apply.
    :py:obj:`~.dstArray` is ignored.

    If :py:obj:`~.dstMemoryType` is :py:obj:`~.CU_MEMORYTYPE_ARRAY`,
    :py:obj:`~.dstArray` specifies the handle of the destination data.
    :py:obj:`~.dstHost`, :py:obj:`~.dstDevice` and :py:obj:`~.dstPitch` are
    ignored.

    - :py:obj:`~.srcXInBytes` and :py:obj:`~.srcY` specify the base address
      of the source data for the copy.

    For host pointers, the starting address is

    **View CUDA Toolkit Documentation for a C++ code example**

    For device pointers, the starting address is

    **View CUDA Toolkit Documentation for a C++ code example**

    For CUDA arrays, :py:obj:`~.srcXInBytes` must be evenly divisible by
    the array element size.

    - :py:obj:`~.dstXInBytes` and :py:obj:`~.dstY` specify the base address
      of the destination data for the copy.

    For host pointers, the base address is

    **View CUDA Toolkit Documentation for a C++ code example**

    For device pointers, the starting address is

    **View CUDA Toolkit Documentation for a C++ code example**

    For CUDA arrays, :py:obj:`~.dstXInBytes` must be evenly divisible by
    the array element size.

    - :py:obj:`~.WidthInBytes` and :py:obj:`~.Height` specify the width (in
      bytes) and height of the 2D copy being performed.

    - If specified, :py:obj:`~.srcPitch` must be greater than or equal to
      :py:obj:`~.WidthInBytes` + :py:obj:`~.srcXInBytes`, and
      :py:obj:`~.dstPitch` must be greater than or equal to
      :py:obj:`~.WidthInBytes` + dstXInBytes.

    :py:obj:`~.cuMemcpy2D()` returns an error if any pitch is greater than
    the maximum allowed (:py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_PITCH`).
    :py:obj:`~.cuMemAllocPitch()` passes back pitches that always work with
    :py:obj:`~.cuMemcpy2D()`. On intra-device memory copies (device to
    device, CUDA array to device, CUDA array to CUDA array),
    :py:obj:`~.cuMemcpy2D()` may fail for pitches not computed by
    :py:obj:`~.cuMemAllocPitch()`. :py:obj:`~.cuMemcpy2DUnaligned()` does
    not have this restriction, but may run significantly slower in the
    cases where :py:obj:`~.cuMemcpy2D()` would have returned an error code.

    Parameters
    ----------
    pCopy : :py:obj:`~.CUDA_MEMCPY2D`
        Parameters for the memory copy

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaMemcpy2D`, :py:obj:`~.cudaMemcpy2DToArray`, :py:obj:`~.cudaMemcpy2DFromArray`
    """
    cdef cydriver.CUDA_MEMCPY2D* cypCopy_ptr = pCopy._ptr if pCopy != None else NULL
    err = cydriver.cuMemcpy2DUnaligned(cypCopy_ptr)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpy3D(pCopy : Optional[CUDA_MEMCPY3D]):
    """ Copies memory for 3D arrays.

    Perform a 3D memory copy according to the parameters specified in
    `pCopy`. The :py:obj:`~.CUDA_MEMCPY3D` structure is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    where:

    - :py:obj:`~.srcMemoryType` and :py:obj:`~.dstMemoryType` specify the
      type of memory of the source and destination, respectively;
      :py:obj:`~.CUmemorytype_enum` is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    If :py:obj:`~.srcMemoryType` is :py:obj:`~.CU_MEMORYTYPE_UNIFIED`,
    :py:obj:`~.srcDevice` and :py:obj:`~.srcPitch` specify the (unified
    virtual address space) base address of the source data and the bytes
    per row to apply. :py:obj:`~.srcArray` is ignored. This value may be
    used only if unified addressing is supported in the calling context.

    If :py:obj:`~.srcMemoryType` is :py:obj:`~.CU_MEMORYTYPE_HOST`,
    :py:obj:`~.srcHost`, :py:obj:`~.srcPitch` and :py:obj:`~.srcHeight`
    specify the (host) base address of the source data, the bytes per row,
    and the height of each 2D slice of the 3D array. :py:obj:`~.srcArray`
    is ignored.

    If :py:obj:`~.srcMemoryType` is :py:obj:`~.CU_MEMORYTYPE_DEVICE`,
    :py:obj:`~.srcDevice`, :py:obj:`~.srcPitch` and :py:obj:`~.srcHeight`
    specify the (device) base address of the source data, the bytes per
    row, and the height of each 2D slice of the 3D array.
    :py:obj:`~.srcArray` is ignored.

    If :py:obj:`~.srcMemoryType` is :py:obj:`~.CU_MEMORYTYPE_ARRAY`,
    :py:obj:`~.srcArray` specifies the handle of the source data.
    :py:obj:`~.srcHost`, :py:obj:`~.srcDevice`, :py:obj:`~.srcPitch` and
    :py:obj:`~.srcHeight` are ignored.

    If :py:obj:`~.dstMemoryType` is :py:obj:`~.CU_MEMORYTYPE_UNIFIED`,
    :py:obj:`~.dstDevice` and :py:obj:`~.dstPitch` specify the (unified
    virtual address space) base address of the source data and the bytes
    per row to apply. :py:obj:`~.dstArray` is ignored. This value may be
    used only if unified addressing is supported in the calling context.

    If :py:obj:`~.dstMemoryType` is :py:obj:`~.CU_MEMORYTYPE_HOST`,
    :py:obj:`~.dstHost` and :py:obj:`~.dstPitch` specify the (host) base
    address of the destination data, the bytes per row, and the height of
    each 2D slice of the 3D array. :py:obj:`~.dstArray` is ignored.

    If :py:obj:`~.dstMemoryType` is :py:obj:`~.CU_MEMORYTYPE_DEVICE`,
    :py:obj:`~.dstDevice` and :py:obj:`~.dstPitch` specify the (device)
    base address of the destination data, the bytes per row, and the height
    of each 2D slice of the 3D array. :py:obj:`~.dstArray` is ignored.

    If :py:obj:`~.dstMemoryType` is :py:obj:`~.CU_MEMORYTYPE_ARRAY`,
    :py:obj:`~.dstArray` specifies the handle of the destination data.
    :py:obj:`~.dstHost`, :py:obj:`~.dstDevice`, :py:obj:`~.dstPitch` and
    :py:obj:`~.dstHeight` are ignored.

    - :py:obj:`~.srcXInBytes`, :py:obj:`~.srcY` and :py:obj:`~.srcZ`
      specify the base address of the source data for the copy.

    For host pointers, the starting address is

    **View CUDA Toolkit Documentation for a C++ code example**

    For device pointers, the starting address is

    **View CUDA Toolkit Documentation for a C++ code example**

    For CUDA arrays, :py:obj:`~.srcXInBytes` must be evenly divisible by
    the array element size.

    - dstXInBytes, :py:obj:`~.dstY` and :py:obj:`~.dstZ` specify the base
      address of the destination data for the copy.

    For host pointers, the base address is

    **View CUDA Toolkit Documentation for a C++ code example**

    For device pointers, the starting address is

    **View CUDA Toolkit Documentation for a C++ code example**

    For CUDA arrays, :py:obj:`~.dstXInBytes` must be evenly divisible by
    the array element size.

    - :py:obj:`~.WidthInBytes`, :py:obj:`~.Height` and :py:obj:`~.Depth`
      specify the width (in bytes), height and depth of the 3D copy being
      performed.

    - If specified, :py:obj:`~.srcPitch` must be greater than or equal to
      :py:obj:`~.WidthInBytes` + :py:obj:`~.srcXInBytes`, and
      :py:obj:`~.dstPitch` must be greater than or equal to
      :py:obj:`~.WidthInBytes` + dstXInBytes.

    - If specified, :py:obj:`~.srcHeight` must be greater than or equal to
      :py:obj:`~.Height` + :py:obj:`~.srcY`, and :py:obj:`~.dstHeight` must
      be greater than or equal to :py:obj:`~.Height` + :py:obj:`~.dstY`.

    :py:obj:`~.cuMemcpy3D()` returns an error if any pitch is greater than
    the maximum allowed (:py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_PITCH`).

    The :py:obj:`~.srcLOD` and :py:obj:`~.dstLOD` members of the
    :py:obj:`~.CUDA_MEMCPY3D` structure must be set to 0.

    Parameters
    ----------
    pCopy : :py:obj:`~.CUDA_MEMCPY3D`
        Parameters for the memory copy

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaMemcpy3D`
    """
    cdef cydriver.CUDA_MEMCPY3D* cypCopy_ptr = pCopy._ptr if pCopy != None else NULL
    err = cydriver.cuMemcpy3D(cypCopy_ptr)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpy3DPeer(pCopy : Optional[CUDA_MEMCPY3D_PEER]):
    """ Copies memory between contexts.

    Perform a 3D memory copy according to the parameters specified in
    `pCopy`. See the definition of the :py:obj:`~.CUDA_MEMCPY3D_PEER`
    structure for documentation of its parameters.

    Parameters
    ----------
    pCopy : :py:obj:`~.CUDA_MEMCPY3D_PEER`
        Parameters for the memory copy

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyPeer`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyPeerAsync`, :py:obj:`~.cuMemcpy3DPeerAsync`, :py:obj:`~.cudaMemcpy3DPeer`
    """
    cdef cydriver.CUDA_MEMCPY3D_PEER* cypCopy_ptr = pCopy._ptr if pCopy != None else NULL
    err = cydriver.cuMemcpy3DPeer(cypCopy_ptr)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpyAsync(dst, src, size_t ByteCount, hStream):
    """ Copies memory asynchronously.

    Copies data between two pointers. `dst` and `src` are base pointers of
    the destination and source, respectively. `ByteCount` specifies the
    number of bytes to copy. Note that this function infers the type of the
    transfer (host to host, host to device, device to device, or device to
    host) from the pointer values. This function is only allowed in
    contexts which support unified addressing.

    Parameters
    ----------
    dst : :py:obj:`~.CUdeviceptr`
        Destination unified virtual address space pointer
    src : :py:obj:`~.CUdeviceptr`
        Source unified virtual address space pointer
    ByteCount : size_t
        Size of memory copy in bytes
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D8Async`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D16Async`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD2D32Async`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD8Async`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD16Async`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cuMemsetD32Async`, :py:obj:`~.cudaMemcpyAsync`, :py:obj:`~.cudaMemcpyToSymbolAsync`, :py:obj:`~.cudaMemcpyFromSymbolAsync`
    """
    cdef cydriver.CUstream cyhStream
    if hStream is None:
        cyhStream = <cydriver.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    cdef cydriver.CUdeviceptr cysrc
    if src is None:
        cysrc = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(src, (CUdeviceptr,)):
        psrc = int(src)
        cysrc = <cydriver.CUdeviceptr><void_ptr>psrc
    else:
        psrc = int(CUdeviceptr(src))
        cysrc = <cydriver.CUdeviceptr><void_ptr>psrc
    cdef cydriver.CUdeviceptr cydst
    if dst is None:
        cydst = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(dst, (CUdeviceptr,)):
        pdst = int(dst)
        cydst = <cydriver.CUdeviceptr><void_ptr>pdst
    else:
        pdst = int(CUdeviceptr(dst))
        cydst = <cydriver.CUdeviceptr><void_ptr>pdst
    err = cydriver.cuMemcpyAsync(cydst, cysrc, ByteCount, cyhStream)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpyPeerAsync(dstDevice, dstContext, srcDevice, srcContext, size_t ByteCount, hStream):
    """ Copies device memory between two contexts asynchronously.

    Copies from device memory in one context to device memory in another
    context. `dstDevice` is the base device pointer of the destination
    memory and `dstContext` is the destination context. `srcDevice` is the
    base device pointer of the source memory and `srcContext` is the source
    pointer. `ByteCount` specifies the number of bytes to copy.

    Parameters
    ----------
    dstDevice : :py:obj:`~.CUdeviceptr`
        Destination device pointer
    dstContext : :py:obj:`~.CUcontext`
        Destination context
    srcDevice : :py:obj:`~.CUdeviceptr`
        Source device pointer
    srcContext : :py:obj:`~.CUcontext`
        Source context
    ByteCount : size_t
        Size of memory copy in bytes
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`

    See Also
    --------
    :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyPeer`, :py:obj:`~.cuMemcpy3DPeer`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpy3DPeerAsync`, :py:obj:`~.cudaMemcpyPeerAsync`
    """
    cdef cydriver.CUstream cyhStream
    if hStream is None:
        cyhStream = <cydriver.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    cdef cydriver.CUcontext cysrcContext
    if srcContext is None:
        cysrcContext = <cydriver.CUcontext><void_ptr>0
    elif isinstance(srcContext, (CUcontext,)):
        psrcContext = int(srcContext)
        cysrcContext = <cydriver.CUcontext><void_ptr>psrcContext
    else:
        psrcContext = int(CUcontext(srcContext))
        cysrcContext = <cydriver.CUcontext><void_ptr>psrcContext
    cdef cydriver.CUdeviceptr cysrcDevice
    if srcDevice is None:
        cysrcDevice = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(srcDevice, (CUdeviceptr,)):
        psrcDevice = int(srcDevice)
        cysrcDevice = <cydriver.CUdeviceptr><void_ptr>psrcDevice
    else:
        psrcDevice = int(CUdeviceptr(srcDevice))
        cysrcDevice = <cydriver.CUdeviceptr><void_ptr>psrcDevice
    cdef cydriver.CUcontext cydstContext
    if dstContext is None:
        cydstContext = <cydriver.CUcontext><void_ptr>0
    elif isinstance(dstContext, (CUcontext,)):
        pdstContext = int(dstContext)
        cydstContext = <cydriver.CUcontext><void_ptr>pdstContext
    else:
        pdstContext = int(CUcontext(dstContext))
        cydstContext = <cydriver.CUcontext><void_ptr>pdstContext
    cdef cydriver.CUdeviceptr cydstDevice
    if dstDevice is None:
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(dstDevice, (CUdeviceptr,)):
        pdstDevice = int(dstDevice)
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
    else:
        pdstDevice = int(CUdeviceptr(dstDevice))
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
    err = cydriver.cuMemcpyPeerAsync(cydstDevice, cydstContext, cysrcDevice, cysrcContext, ByteCount, cyhStream)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpyHtoDAsync(dstDevice, srcHost, size_t ByteCount, hStream):
    """ Copies memory from Host to Device.

    Copies from host memory to device memory. `dstDevice` and `srcHost` are
    the base addresses of the destination and source, respectively.
    `ByteCount` specifies the number of bytes to copy.

    Parameters
    ----------
    dstDevice : :py:obj:`~.CUdeviceptr`
        Destination device pointer
    srcHost : Any
        Source host pointer
    ByteCount : size_t
        Size of memory copy in bytes
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D8Async`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D16Async`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD2D32Async`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD8Async`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD16Async`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cuMemsetD32Async`, :py:obj:`~.cudaMemcpyAsync`, :py:obj:`~.cudaMemcpyToSymbolAsync`
    """
    cdef cydriver.CUstream cyhStream
    if hStream is None:
        cyhStream = <cydriver.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    cdef cydriver.CUdeviceptr cydstDevice
    if dstDevice is None:
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(dstDevice, (CUdeviceptr,)):
        pdstDevice = int(dstDevice)
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
    else:
        pdstDevice = int(CUdeviceptr(dstDevice))
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
    cysrcHost = utils.HelperInputVoidPtr(srcHost)
    cdef void* cysrcHost_ptr = <void*><void_ptr>cysrcHost.cptr
    err = cydriver.cuMemcpyHtoDAsync(cydstDevice, cysrcHost_ptr, ByteCount, cyhStream)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpyDtoHAsync(dstHost, srcDevice, size_t ByteCount, hStream):
    """ Copies memory from Device to Host.

    Copies from device to host memory. `dstHost` and `srcDevice` specify
    the base pointers of the destination and source, respectively.
    `ByteCount` specifies the number of bytes to copy.

    Parameters
    ----------
    dstHost : Any
        Destination host pointer
    srcDevice : :py:obj:`~.CUdeviceptr`
        Source device pointer
    ByteCount : size_t
        Size of memory copy in bytes
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D8Async`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D16Async`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD2D32Async`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD8Async`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD16Async`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cuMemsetD32Async`, :py:obj:`~.cudaMemcpyAsync`, :py:obj:`~.cudaMemcpyFromSymbolAsync`
    """
    cdef cydriver.CUstream cyhStream
    if hStream is None:
        cyhStream = <cydriver.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    cdef cydriver.CUdeviceptr cysrcDevice
    if srcDevice is None:
        cysrcDevice = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(srcDevice, (CUdeviceptr,)):
        psrcDevice = int(srcDevice)
        cysrcDevice = <cydriver.CUdeviceptr><void_ptr>psrcDevice
    else:
        psrcDevice = int(CUdeviceptr(srcDevice))
        cysrcDevice = <cydriver.CUdeviceptr><void_ptr>psrcDevice
    cydstHost = utils.HelperInputVoidPtr(dstHost)
    cdef void* cydstHost_ptr = <void*><void_ptr>cydstHost.cptr
    err = cydriver.cuMemcpyDtoHAsync(cydstHost_ptr, cysrcDevice, ByteCount, cyhStream)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpyDtoDAsync(dstDevice, srcDevice, size_t ByteCount, hStream):
    """ Copies memory from Device to Device.

    Copies from device memory to device memory. `dstDevice` and `srcDevice`
    are the base pointers of the destination and source, respectively.
    `ByteCount` specifies the number of bytes to copy.

    Parameters
    ----------
    dstDevice : :py:obj:`~.CUdeviceptr`
        Destination device pointer
    srcDevice : :py:obj:`~.CUdeviceptr`
        Source device pointer
    ByteCount : size_t
        Size of memory copy in bytes
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D8Async`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D16Async`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD2D32Async`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD8Async`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD16Async`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cuMemsetD32Async`, :py:obj:`~.cudaMemcpyAsync`, :py:obj:`~.cudaMemcpyToSymbolAsync`, :py:obj:`~.cudaMemcpyFromSymbolAsync`
    """
    cdef cydriver.CUstream cyhStream
    if hStream is None:
        cyhStream = <cydriver.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    cdef cydriver.CUdeviceptr cysrcDevice
    if srcDevice is None:
        cysrcDevice = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(srcDevice, (CUdeviceptr,)):
        psrcDevice = int(srcDevice)
        cysrcDevice = <cydriver.CUdeviceptr><void_ptr>psrcDevice
    else:
        psrcDevice = int(CUdeviceptr(srcDevice))
        cysrcDevice = <cydriver.CUdeviceptr><void_ptr>psrcDevice
    cdef cydriver.CUdeviceptr cydstDevice
    if dstDevice is None:
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(dstDevice, (CUdeviceptr,)):
        pdstDevice = int(dstDevice)
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
    else:
        pdstDevice = int(CUdeviceptr(dstDevice))
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
    err = cydriver.cuMemcpyDtoDAsync(cydstDevice, cysrcDevice, ByteCount, cyhStream)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpyHtoAAsync(dstArray, size_t dstOffset, srcHost, size_t ByteCount, hStream):
    """ Copies memory from Host to Array.

    Copies from host memory to a 1D CUDA array. `dstArray` and `dstOffset`
    specify the CUDA array handle and starting offset in bytes of the
    destination data. `srcHost` specifies the base address of the source.
    `ByteCount` specifies the number of bytes to copy.

    Parameters
    ----------
    dstArray : :py:obj:`~.CUarray`
        Destination array
    dstOffset : size_t
        Offset in bytes of destination array
    srcHost : Any
        Source host pointer
    ByteCount : size_t
        Size of memory copy in bytes
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D8Async`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D16Async`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD2D32Async`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD8Async`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD16Async`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cuMemsetD32Async`, :py:obj:`~.cudaMemcpyToArrayAsync`
    """
    cdef cydriver.CUstream cyhStream
    if hStream is None:
        cyhStream = <cydriver.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    cdef cydriver.CUarray cydstArray
    if dstArray is None:
        cydstArray = <cydriver.CUarray><void_ptr>0
    elif isinstance(dstArray, (CUarray,)):
        pdstArray = int(dstArray)
        cydstArray = <cydriver.CUarray><void_ptr>pdstArray
    else:
        pdstArray = int(CUarray(dstArray))
        cydstArray = <cydriver.CUarray><void_ptr>pdstArray
    cysrcHost = utils.HelperInputVoidPtr(srcHost)
    cdef void* cysrcHost_ptr = <void*><void_ptr>cysrcHost.cptr
    err = cydriver.cuMemcpyHtoAAsync(cydstArray, dstOffset, cysrcHost_ptr, ByteCount, cyhStream)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpyAtoHAsync(dstHost, srcArray, size_t srcOffset, size_t ByteCount, hStream):
    """ Copies memory from Array to Host.

    Copies from one 1D CUDA array to host memory. `dstHost` specifies the
    base pointer of the destination. `srcArray` and `srcOffset` specify the
    CUDA array handle and starting offset in bytes of the source data.
    `ByteCount` specifies the number of bytes to copy.

    Parameters
    ----------
    dstHost : Any
        Destination pointer
    srcArray : :py:obj:`~.CUarray`
        Source array
    srcOffset : size_t
        Offset in bytes of source array
    ByteCount : size_t
        Size of memory copy in bytes
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D8Async`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D16Async`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD2D32Async`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD8Async`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD16Async`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cuMemsetD32Async`, :py:obj:`~.cudaMemcpyFromArrayAsync`
    """
    cdef cydriver.CUstream cyhStream
    if hStream is None:
        cyhStream = <cydriver.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    cdef cydriver.CUarray cysrcArray
    if srcArray is None:
        cysrcArray = <cydriver.CUarray><void_ptr>0
    elif isinstance(srcArray, (CUarray,)):
        psrcArray = int(srcArray)
        cysrcArray = <cydriver.CUarray><void_ptr>psrcArray
    else:
        psrcArray = int(CUarray(srcArray))
        cysrcArray = <cydriver.CUarray><void_ptr>psrcArray
    cydstHost = utils.HelperInputVoidPtr(dstHost)
    cdef void* cydstHost_ptr = <void*><void_ptr>cydstHost.cptr
    err = cydriver.cuMemcpyAtoHAsync(cydstHost_ptr, cysrcArray, srcOffset, ByteCount, cyhStream)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpy2DAsync(pCopy : Optional[CUDA_MEMCPY2D], hStream):
    """ Copies memory for 2D arrays.

    Perform a 2D memory copy according to the parameters specified in
    `pCopy`. The :py:obj:`~.CUDA_MEMCPY2D` structure is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    where:

    - :py:obj:`~.srcMemoryType` and :py:obj:`~.dstMemoryType` specify the
      type of memory of the source and destination, respectively;
      :py:obj:`~.CUmemorytype_enum` is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    If :py:obj:`~.srcMemoryType` is :py:obj:`~.CU_MEMORYTYPE_HOST`,
    :py:obj:`~.srcHost` and :py:obj:`~.srcPitch` specify the (host) base
    address of the source data and the bytes per row to apply.
    :py:obj:`~.srcArray` is ignored.

    If :py:obj:`~.srcMemoryType` is :py:obj:`~.CU_MEMORYTYPE_UNIFIED`,
    :py:obj:`~.srcDevice` and :py:obj:`~.srcPitch` specify the (unified
    virtual address space) base address of the source data and the bytes
    per row to apply. :py:obj:`~.srcArray` is ignored. This value may be
    used only if unified addressing is supported in the calling context.

    If :py:obj:`~.srcMemoryType` is :py:obj:`~.CU_MEMORYTYPE_DEVICE`,
    :py:obj:`~.srcDevice` and :py:obj:`~.srcPitch` specify the (device)
    base address of the source data and the bytes per row to apply.
    :py:obj:`~.srcArray` is ignored.

    If :py:obj:`~.srcMemoryType` is :py:obj:`~.CU_MEMORYTYPE_ARRAY`,
    :py:obj:`~.srcArray` specifies the handle of the source data.
    :py:obj:`~.srcHost`, :py:obj:`~.srcDevice` and :py:obj:`~.srcPitch` are
    ignored.

    If :py:obj:`~.dstMemoryType` is :py:obj:`~.CU_MEMORYTYPE_UNIFIED`,
    :py:obj:`~.dstDevice` and :py:obj:`~.dstPitch` specify the (unified
    virtual address space) base address of the source data and the bytes
    per row to apply. :py:obj:`~.dstArray` is ignored. This value may be
    used only if unified addressing is supported in the calling context.

    If :py:obj:`~.dstMemoryType` is :py:obj:`~.CU_MEMORYTYPE_HOST`,
    :py:obj:`~.dstHost` and :py:obj:`~.dstPitch` specify the (host) base
    address of the destination data and the bytes per row to apply.
    :py:obj:`~.dstArray` is ignored.

    If :py:obj:`~.dstMemoryType` is :py:obj:`~.CU_MEMORYTYPE_DEVICE`,
    :py:obj:`~.dstDevice` and :py:obj:`~.dstPitch` specify the (device)
    base address of the destination data and the bytes per row to apply.
    :py:obj:`~.dstArray` is ignored.

    If :py:obj:`~.dstMemoryType` is :py:obj:`~.CU_MEMORYTYPE_ARRAY`,
    :py:obj:`~.dstArray` specifies the handle of the destination data.
    :py:obj:`~.dstHost`, :py:obj:`~.dstDevice` and :py:obj:`~.dstPitch` are
    ignored.

    - :py:obj:`~.srcXInBytes` and :py:obj:`~.srcY` specify the base address
      of the source data for the copy.

    For host pointers, the starting address is

    **View CUDA Toolkit Documentation for a C++ code example**

    For device pointers, the starting address is

    **View CUDA Toolkit Documentation for a C++ code example**

    For CUDA arrays, :py:obj:`~.srcXInBytes` must be evenly divisible by
    the array element size.

    - :py:obj:`~.dstXInBytes` and :py:obj:`~.dstY` specify the base address
      of the destination data for the copy.

    For host pointers, the base address is

    **View CUDA Toolkit Documentation for a C++ code example**

    For device pointers, the starting address is

    **View CUDA Toolkit Documentation for a C++ code example**

    For CUDA arrays, :py:obj:`~.dstXInBytes` must be evenly divisible by
    the array element size.

    - :py:obj:`~.WidthInBytes` and :py:obj:`~.Height` specify the width (in
      bytes) and height of the 2D copy being performed.

    - If specified, :py:obj:`~.srcPitch` must be greater than or equal to
      :py:obj:`~.WidthInBytes` + :py:obj:`~.srcXInBytes`, and
      :py:obj:`~.dstPitch` must be greater than or equal to
      :py:obj:`~.WidthInBytes` + dstXInBytes.

    - If specified, :py:obj:`~.srcPitch` must be greater than or equal to
      :py:obj:`~.WidthInBytes` + :py:obj:`~.srcXInBytes`, and
      :py:obj:`~.dstPitch` must be greater than or equal to
      :py:obj:`~.WidthInBytes` + dstXInBytes.

    - If specified, :py:obj:`~.srcHeight` must be greater than or equal to
      :py:obj:`~.Height` + :py:obj:`~.srcY`, and :py:obj:`~.dstHeight` must
      be greater than or equal to :py:obj:`~.Height` + :py:obj:`~.dstY`.

    :py:obj:`~.cuMemcpy2DAsync()` returns an error if any pitch is greater
    than the maximum allowed (:py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_PITCH`).
    :py:obj:`~.cuMemAllocPitch()` passes back pitches that always work with
    :py:obj:`~.cuMemcpy2D()`. On intra-device memory copies (device to
    device, CUDA array to device, CUDA array to CUDA array),
    :py:obj:`~.cuMemcpy2DAsync()` may fail for pitches not computed by
    :py:obj:`~.cuMemAllocPitch()`.

    Parameters
    ----------
    pCopy : :py:obj:`~.CUDA_MEMCPY2D`
        Parameters for the memory copy
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D8Async`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D16Async`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD2D32Async`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD8Async`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD16Async`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cuMemsetD32Async`, :py:obj:`~.cudaMemcpy2DAsync`, :py:obj:`~.cudaMemcpy2DToArrayAsync`, :py:obj:`~.cudaMemcpy2DFromArrayAsync`
    """
    cdef cydriver.CUstream cyhStream
    if hStream is None:
        cyhStream = <cydriver.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    cdef cydriver.CUDA_MEMCPY2D* cypCopy_ptr = pCopy._ptr if pCopy != None else NULL
    err = cydriver.cuMemcpy2DAsync(cypCopy_ptr, cyhStream)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpy3DAsync(pCopy : Optional[CUDA_MEMCPY3D], hStream):
    """ Copies memory for 3D arrays.

    Perform a 3D memory copy according to the parameters specified in
    `pCopy`. The :py:obj:`~.CUDA_MEMCPY3D` structure is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    where:

    - :py:obj:`~.srcMemoryType` and :py:obj:`~.dstMemoryType` specify the
      type of memory of the source and destination, respectively;
      :py:obj:`~.CUmemorytype_enum` is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    If :py:obj:`~.srcMemoryType` is :py:obj:`~.CU_MEMORYTYPE_UNIFIED`,
    :py:obj:`~.srcDevice` and :py:obj:`~.srcPitch` specify the (unified
    virtual address space) base address of the source data and the bytes
    per row to apply. :py:obj:`~.srcArray` is ignored. This value may be
    used only if unified addressing is supported in the calling context.

    If :py:obj:`~.srcMemoryType` is :py:obj:`~.CU_MEMORYTYPE_HOST`,
    :py:obj:`~.srcHost`, :py:obj:`~.srcPitch` and :py:obj:`~.srcHeight`
    specify the (host) base address of the source data, the bytes per row,
    and the height of each 2D slice of the 3D array. :py:obj:`~.srcArray`
    is ignored.

    If :py:obj:`~.srcMemoryType` is :py:obj:`~.CU_MEMORYTYPE_DEVICE`,
    :py:obj:`~.srcDevice`, :py:obj:`~.srcPitch` and :py:obj:`~.srcHeight`
    specify the (device) base address of the source data, the bytes per
    row, and the height of each 2D slice of the 3D array.
    :py:obj:`~.srcArray` is ignored.

    If :py:obj:`~.srcMemoryType` is :py:obj:`~.CU_MEMORYTYPE_ARRAY`,
    :py:obj:`~.srcArray` specifies the handle of the source data.
    :py:obj:`~.srcHost`, :py:obj:`~.srcDevice`, :py:obj:`~.srcPitch` and
    :py:obj:`~.srcHeight` are ignored.

    If :py:obj:`~.dstMemoryType` is :py:obj:`~.CU_MEMORYTYPE_UNIFIED`,
    :py:obj:`~.dstDevice` and :py:obj:`~.dstPitch` specify the (unified
    virtual address space) base address of the source data and the bytes
    per row to apply. :py:obj:`~.dstArray` is ignored. This value may be
    used only if unified addressing is supported in the calling context.

    If :py:obj:`~.dstMemoryType` is :py:obj:`~.CU_MEMORYTYPE_HOST`,
    :py:obj:`~.dstHost` and :py:obj:`~.dstPitch` specify the (host) base
    address of the destination data, the bytes per row, and the height of
    each 2D slice of the 3D array. :py:obj:`~.dstArray` is ignored.

    If :py:obj:`~.dstMemoryType` is :py:obj:`~.CU_MEMORYTYPE_DEVICE`,
    :py:obj:`~.dstDevice` and :py:obj:`~.dstPitch` specify the (device)
    base address of the destination data, the bytes per row, and the height
    of each 2D slice of the 3D array. :py:obj:`~.dstArray` is ignored.

    If :py:obj:`~.dstMemoryType` is :py:obj:`~.CU_MEMORYTYPE_ARRAY`,
    :py:obj:`~.dstArray` specifies the handle of the destination data.
    :py:obj:`~.dstHost`, :py:obj:`~.dstDevice`, :py:obj:`~.dstPitch` and
    :py:obj:`~.dstHeight` are ignored.

    - :py:obj:`~.srcXInBytes`, :py:obj:`~.srcY` and :py:obj:`~.srcZ`
      specify the base address of the source data for the copy.

    For host pointers, the starting address is

    **View CUDA Toolkit Documentation for a C++ code example**

    For device pointers, the starting address is

    **View CUDA Toolkit Documentation for a C++ code example**

    For CUDA arrays, :py:obj:`~.srcXInBytes` must be evenly divisible by
    the array element size.

    - dstXInBytes, :py:obj:`~.dstY` and :py:obj:`~.dstZ` specify the base
      address of the destination data for the copy.

    For host pointers, the base address is

    **View CUDA Toolkit Documentation for a C++ code example**

    For device pointers, the starting address is

    **View CUDA Toolkit Documentation for a C++ code example**

    For CUDA arrays, :py:obj:`~.dstXInBytes` must be evenly divisible by
    the array element size.

    - :py:obj:`~.WidthInBytes`, :py:obj:`~.Height` and :py:obj:`~.Depth`
      specify the width (in bytes), height and depth of the 3D copy being
      performed.

    - If specified, :py:obj:`~.srcPitch` must be greater than or equal to
      :py:obj:`~.WidthInBytes` + :py:obj:`~.srcXInBytes`, and
      :py:obj:`~.dstPitch` must be greater than or equal to
      :py:obj:`~.WidthInBytes` + dstXInBytes.

    - If specified, :py:obj:`~.srcHeight` must be greater than or equal to
      :py:obj:`~.Height` + :py:obj:`~.srcY`, and :py:obj:`~.dstHeight` must
      be greater than or equal to :py:obj:`~.Height` + :py:obj:`~.dstY`.

    :py:obj:`~.cuMemcpy3DAsync()` returns an error if any pitch is greater
    than the maximum allowed (:py:obj:`~.CU_DEVICE_ATTRIBUTE_MAX_PITCH`).

    The :py:obj:`~.srcLOD` and :py:obj:`~.dstLOD` members of the
    :py:obj:`~.CUDA_MEMCPY3D` structure must be set to 0.

    Parameters
    ----------
    pCopy : :py:obj:`~.CUDA_MEMCPY3D`
        Parameters for the memory copy
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D8Async`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D16Async`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD2D32Async`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD8Async`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD16Async`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cuMemsetD32Async`, :py:obj:`~.cudaMemcpy3DAsync`
    """
    cdef cydriver.CUstream cyhStream
    if hStream is None:
        cyhStream = <cydriver.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    cdef cydriver.CUDA_MEMCPY3D* cypCopy_ptr = pCopy._ptr if pCopy != None else NULL
    err = cydriver.cuMemcpy3DAsync(cypCopy_ptr, cyhStream)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemcpy3DPeerAsync(pCopy : Optional[CUDA_MEMCPY3D_PEER], hStream):
    """ Copies memory between contexts asynchronously.

    Perform a 3D memory copy according to the parameters specified in
    `pCopy`. See the definition of the :py:obj:`~.CUDA_MEMCPY3D_PEER`
    structure for documentation of its parameters.

    Parameters
    ----------
    pCopy : :py:obj:`~.CUDA_MEMCPY3D_PEER`
        Parameters for the memory copy
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyPeer`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyPeerAsync`, :py:obj:`~.cuMemcpy3DPeerAsync`, :py:obj:`~.cudaMemcpy3DPeerAsync`
    """
    cdef cydriver.CUstream cyhStream
    if hStream is None:
        cyhStream = <cydriver.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    cdef cydriver.CUDA_MEMCPY3D_PEER* cypCopy_ptr = pCopy._ptr if pCopy != None else NULL
    err = cydriver.cuMemcpy3DPeerAsync(cypCopy_ptr, cyhStream)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemsetD8(dstDevice, unsigned char uc, size_t N):
    """ Initializes device memory.

    Sets the memory range of `N` 8-bit values to the specified value `uc`.

    Parameters
    ----------
    dstDevice : :py:obj:`~.CUdeviceptr`
        Destination device pointer
    uc : unsigned char
        Value to set
    N : size_t
        Number of elements

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D8Async`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D16Async`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD2D32Async`, :py:obj:`~.cuMemsetD8Async`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD16Async`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cuMemsetD32Async`, :py:obj:`~.cudaMemset`
    """
    cdef cydriver.CUdeviceptr cydstDevice
    if dstDevice is None:
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(dstDevice, (CUdeviceptr,)):
        pdstDevice = int(dstDevice)
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
    else:
        pdstDevice = int(CUdeviceptr(dstDevice))
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
    err = cydriver.cuMemsetD8(cydstDevice, uc, N)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemsetD16(dstDevice, unsigned short us, size_t N):
    """ Initializes device memory.

    Sets the memory range of `N` 16-bit values to the specified value `us`.
    The `dstDevice` pointer must be two byte aligned.

    Parameters
    ----------
    dstDevice : :py:obj:`~.CUdeviceptr`
        Destination device pointer
    us : unsigned short
        Value to set
    N : size_t
        Number of elements

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D8Async`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D16Async`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD2D32Async`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD8Async`, :py:obj:`~.cuMemsetD16Async`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cuMemsetD32Async`, :py:obj:`~.cudaMemset`
    """
    cdef cydriver.CUdeviceptr cydstDevice
    if dstDevice is None:
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(dstDevice, (CUdeviceptr,)):
        pdstDevice = int(dstDevice)
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
    else:
        pdstDevice = int(CUdeviceptr(dstDevice))
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
    err = cydriver.cuMemsetD16(cydstDevice, us, N)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemsetD32(dstDevice, unsigned int ui, size_t N):
    """ Initializes device memory.

    Sets the memory range of `N` 32-bit values to the specified value `ui`.
    The `dstDevice` pointer must be four byte aligned.

    Parameters
    ----------
    dstDevice : :py:obj:`~.CUdeviceptr`
        Destination device pointer
    ui : unsigned int
        Value to set
    N : size_t
        Number of elements

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D8Async`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D16Async`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD2D32Async`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD8Async`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD16Async`, :py:obj:`~.cuMemsetD32Async`, :py:obj:`~.cudaMemset`
    """
    cdef cydriver.CUdeviceptr cydstDevice
    if dstDevice is None:
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(dstDevice, (CUdeviceptr,)):
        pdstDevice = int(dstDevice)
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
    else:
        pdstDevice = int(CUdeviceptr(dstDevice))
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
    err = cydriver.cuMemsetD32(cydstDevice, ui, N)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemsetD2D8(dstDevice, size_t dstPitch, unsigned char uc, size_t Width, size_t Height):
    """ Initializes device memory.

    Sets the 2D memory range of `Width` 8-bit values to the specified value
    `uc`. `Height` specifies the number of rows to set, and `dstPitch`
    specifies the number of bytes between each row. This function performs
    fastest when the pitch is one that has been passed back by
    :py:obj:`~.cuMemAllocPitch()`.

    Parameters
    ----------
    dstDevice : :py:obj:`~.CUdeviceptr`
        Destination device pointer
    dstPitch : size_t
        Pitch of destination device pointer(Unused if `Height` is 1)
    uc : unsigned char
        Value to set
    Width : size_t
        Width of row
    Height : size_t
        Number of rows

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8Async`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D16Async`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD2D32Async`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD8Async`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD16Async`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cuMemsetD32Async`, :py:obj:`~.cudaMemset2D`
    """
    cdef cydriver.CUdeviceptr cydstDevice
    if dstDevice is None:
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(dstDevice, (CUdeviceptr,)):
        pdstDevice = int(dstDevice)
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
    else:
        pdstDevice = int(CUdeviceptr(dstDevice))
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
    err = cydriver.cuMemsetD2D8(cydstDevice, dstPitch, uc, Width, Height)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemsetD2D16(dstDevice, size_t dstPitch, unsigned short us, size_t Width, size_t Height):
    """ Initializes device memory.

    Sets the 2D memory range of `Width` 16-bit values to the specified
    value `us`. `Height` specifies the number of rows to set, and
    `dstPitch` specifies the number of bytes between each row. The
    `dstDevice` pointer and `dstPitch` offset must be two byte aligned.
    This function performs fastest when the pitch is one that has been
    passed back by :py:obj:`~.cuMemAllocPitch()`.

    Parameters
    ----------
    dstDevice : :py:obj:`~.CUdeviceptr`
        Destination device pointer
    dstPitch : size_t
        Pitch of destination device pointer(Unused if `Height` is 1)
    us : unsigned short
        Value to set
    Width : size_t
        Width of row
    Height : size_t
        Number of rows

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D8Async`, :py:obj:`~.cuMemsetD2D16Async`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD2D32Async`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD8Async`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD16Async`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cuMemsetD32Async`, :py:obj:`~.cudaMemset2D`
    """
    cdef cydriver.CUdeviceptr cydstDevice
    if dstDevice is None:
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(dstDevice, (CUdeviceptr,)):
        pdstDevice = int(dstDevice)
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
    else:
        pdstDevice = int(CUdeviceptr(dstDevice))
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
    err = cydriver.cuMemsetD2D16(cydstDevice, dstPitch, us, Width, Height)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemsetD2D32(dstDevice, size_t dstPitch, unsigned int ui, size_t Width, size_t Height):
    """ Initializes device memory.

    Sets the 2D memory range of `Width` 32-bit values to the specified
    value `ui`. `Height` specifies the number of rows to set, and
    `dstPitch` specifies the number of bytes between each row. The
    `dstDevice` pointer and `dstPitch` offset must be four byte aligned.
    This function performs fastest when the pitch is one that has been
    passed back by :py:obj:`~.cuMemAllocPitch()`.

    Parameters
    ----------
    dstDevice : :py:obj:`~.CUdeviceptr`
        Destination device pointer
    dstPitch : size_t
        Pitch of destination device pointer(Unused if `Height` is 1)
    ui : unsigned int
        Value to set
    Width : size_t
        Width of row
    Height : size_t
        Number of rows

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D8Async`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D16Async`, :py:obj:`~.cuMemsetD2D32Async`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD8Async`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD16Async`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cuMemsetD32Async`, :py:obj:`~.cudaMemset2D`
    """
    cdef cydriver.CUdeviceptr cydstDevice
    if dstDevice is None:
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(dstDevice, (CUdeviceptr,)):
        pdstDevice = int(dstDevice)
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
    else:
        pdstDevice = int(CUdeviceptr(dstDevice))
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
    err = cydriver.cuMemsetD2D32(cydstDevice, dstPitch, ui, Width, Height)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemsetD8Async(dstDevice, unsigned char uc, size_t N, hStream):
    """ Sets device memory.

    Sets the memory range of `N` 8-bit values to the specified value `uc`.

    Parameters
    ----------
    dstDevice : :py:obj:`~.CUdeviceptr`
        Destination device pointer
    uc : unsigned char
        Value to set
    N : size_t
        Number of elements
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D8Async`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D16Async`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD2D32Async`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD16Async`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cuMemsetD32Async`, :py:obj:`~.cudaMemsetAsync`
    """
    cdef cydriver.CUstream cyhStream
    if hStream is None:
        cyhStream = <cydriver.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    cdef cydriver.CUdeviceptr cydstDevice
    if dstDevice is None:
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(dstDevice, (CUdeviceptr,)):
        pdstDevice = int(dstDevice)
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
    else:
        pdstDevice = int(CUdeviceptr(dstDevice))
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
    err = cydriver.cuMemsetD8Async(cydstDevice, uc, N, cyhStream)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemsetD16Async(dstDevice, unsigned short us, size_t N, hStream):
    """ Sets device memory.

    Sets the memory range of `N` 16-bit values to the specified value `us`.
    The `dstDevice` pointer must be two byte aligned.

    Parameters
    ----------
    dstDevice : :py:obj:`~.CUdeviceptr`
        Destination device pointer
    us : unsigned short
        Value to set
    N : size_t
        Number of elements
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D8Async`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D16Async`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD2D32Async`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD8Async`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cuMemsetD32Async`, :py:obj:`~.cudaMemsetAsync`
    """
    cdef cydriver.CUstream cyhStream
    if hStream is None:
        cyhStream = <cydriver.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    cdef cydriver.CUdeviceptr cydstDevice
    if dstDevice is None:
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(dstDevice, (CUdeviceptr,)):
        pdstDevice = int(dstDevice)
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
    else:
        pdstDevice = int(CUdeviceptr(dstDevice))
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
    err = cydriver.cuMemsetD16Async(cydstDevice, us, N, cyhStream)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemsetD32Async(dstDevice, unsigned int ui, size_t N, hStream):
    """ Sets device memory.

    Sets the memory range of `N` 32-bit values to the specified value `ui`.
    The `dstDevice` pointer must be four byte aligned.

    Parameters
    ----------
    dstDevice : :py:obj:`~.CUdeviceptr`
        Destination device pointer
    ui : unsigned int
        Value to set
    N : size_t
        Number of elements
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D8Async`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D16Async`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD2D32Async`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD8Async`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD16Async`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaMemsetAsync`
    """
    cdef cydriver.CUstream cyhStream
    if hStream is None:
        cyhStream = <cydriver.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    cdef cydriver.CUdeviceptr cydstDevice
    if dstDevice is None:
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(dstDevice, (CUdeviceptr,)):
        pdstDevice = int(dstDevice)
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
    else:
        pdstDevice = int(CUdeviceptr(dstDevice))
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
    err = cydriver.cuMemsetD32Async(cydstDevice, ui, N, cyhStream)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemsetD2D8Async(dstDevice, size_t dstPitch, unsigned char uc, size_t Width, size_t Height, hStream):
    """ Sets device memory.

    Sets the 2D memory range of `Width` 8-bit values to the specified value
    `uc`. `Height` specifies the number of rows to set, and `dstPitch`
    specifies the number of bytes between each row. This function performs
    fastest when the pitch is one that has been passed back by
    :py:obj:`~.cuMemAllocPitch()`.

    Parameters
    ----------
    dstDevice : :py:obj:`~.CUdeviceptr`
        Destination device pointer
    dstPitch : size_t
        Pitch of destination device pointer(Unused if `Height` is 1)
    uc : unsigned char
        Value to set
    Width : size_t
        Width of row
    Height : size_t
        Number of rows
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D16Async`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD2D32Async`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD8Async`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD16Async`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cuMemsetD32Async`, :py:obj:`~.cudaMemset2DAsync`
    """
    cdef cydriver.CUstream cyhStream
    if hStream is None:
        cyhStream = <cydriver.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    cdef cydriver.CUdeviceptr cydstDevice
    if dstDevice is None:
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(dstDevice, (CUdeviceptr,)):
        pdstDevice = int(dstDevice)
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
    else:
        pdstDevice = int(CUdeviceptr(dstDevice))
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
    err = cydriver.cuMemsetD2D8Async(cydstDevice, dstPitch, uc, Width, Height, cyhStream)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemsetD2D16Async(dstDevice, size_t dstPitch, unsigned short us, size_t Width, size_t Height, hStream):
    """ Sets device memory.

    Sets the 2D memory range of `Width` 16-bit values to the specified
    value `us`. `Height` specifies the number of rows to set, and
    `dstPitch` specifies the number of bytes between each row. The
    `dstDevice` pointer and `dstPitch` offset must be two byte aligned.
    This function performs fastest when the pitch is one that has been
    passed back by :py:obj:`~.cuMemAllocPitch()`.

    Parameters
    ----------
    dstDevice : :py:obj:`~.CUdeviceptr`
        Destination device pointer
    dstPitch : size_t
        Pitch of destination device pointer(Unused if `Height` is 1)
    us : unsigned short
        Value to set
    Width : size_t
        Width of row
    Height : size_t
        Number of rows
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D8Async`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD2D32Async`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD8Async`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD16Async`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cuMemsetD32Async`, :py:obj:`~.cudaMemset2DAsync`
    """
    cdef cydriver.CUstream cyhStream
    if hStream is None:
        cyhStream = <cydriver.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    cdef cydriver.CUdeviceptr cydstDevice
    if dstDevice is None:
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(dstDevice, (CUdeviceptr,)):
        pdstDevice = int(dstDevice)
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
    else:
        pdstDevice = int(CUdeviceptr(dstDevice))
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
    err = cydriver.cuMemsetD2D16Async(cydstDevice, dstPitch, us, Width, Height, cyhStream)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemsetD2D32Async(dstDevice, size_t dstPitch, unsigned int ui, size_t Width, size_t Height, hStream):
    """ Sets device memory.

    Sets the 2D memory range of `Width` 32-bit values to the specified
    value `ui`. `Height` specifies the number of rows to set, and
    `dstPitch` specifies the number of bytes between each row. The
    `dstDevice` pointer and `dstPitch` offset must be four byte aligned.
    This function performs fastest when the pitch is one that has been
    passed back by :py:obj:`~.cuMemAllocPitch()`.

    Parameters
    ----------
    dstDevice : :py:obj:`~.CUdeviceptr`
        Destination device pointer
    dstPitch : size_t
        Pitch of destination device pointer(Unused if `Height` is 1)
    ui : unsigned int
        Value to set
    Width : size_t
        Width of row
    Height : size_t
        Number of rows
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D8Async`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D16Async`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD8Async`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD16Async`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cuMemsetD32Async`, :py:obj:`~.cudaMemset2DAsync`
    """
    cdef cydriver.CUstream cyhStream
    if hStream is None:
        cyhStream = <cydriver.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    cdef cydriver.CUdeviceptr cydstDevice
    if dstDevice is None:
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(dstDevice, (CUdeviceptr,)):
        pdstDevice = int(dstDevice)
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
    else:
        pdstDevice = int(CUdeviceptr(dstDevice))
        cydstDevice = <cydriver.CUdeviceptr><void_ptr>pdstDevice
    err = cydriver.cuMemsetD2D32Async(cydstDevice, dstPitch, ui, Width, Height, cyhStream)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuArrayCreate(pAllocateArray : Optional[CUDA_ARRAY_DESCRIPTOR]):
    """ Creates a 1D or 2D CUDA array.

    Creates a CUDA array according to the :py:obj:`~.CUDA_ARRAY_DESCRIPTOR`
    structure `pAllocateArray` and returns a handle to the new CUDA array
    in `*pHandle`. The :py:obj:`~.CUDA_ARRAY_DESCRIPTOR` is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    where:

    - `Width`, and `Height` are the width, and height of the CUDA array (in
      elements); the CUDA array is one-dimensional if height is 0, two-
      dimensional otherwise;

    - :py:obj:`~.Format` specifies the format of the elements;
      :py:obj:`~.CUarray_format` is defined as:

    - **View CUDA Toolkit Documentation for a C++ code example**

    - `NumChannels` specifies the number of packed components per CUDA
      array element; it may be 1, 2, or 4;

    Here are examples of CUDA array descriptions:

    Description for a CUDA array of 2048 floats:

    **View CUDA Toolkit Documentation for a C++ code example**

    Description for a 64 x 64 CUDA array of floats:

    **View CUDA Toolkit Documentation for a C++ code example**

    Description for a `width` x `height` CUDA array of 64-bit, 4x16-bit
    float16's:

    **View CUDA Toolkit Documentation for a C++ code example**

    Description for a `width` x `height` CUDA array of 16-bit elements,
    each of which is two 8-bit unsigned chars:

    **View CUDA Toolkit Documentation for a C++ code example**

    Parameters
    ----------
    pAllocateArray : :py:obj:`~.CUDA_ARRAY_DESCRIPTOR`
        Array descriptor

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_UNKNOWN`
    pHandle : :py:obj:`~.CUarray`
        Returned array

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaMallocArray`
    """
    cdef CUarray pHandle = CUarray()
    cdef cydriver.CUDA_ARRAY_DESCRIPTOR* cypAllocateArray_ptr = pAllocateArray._ptr if pAllocateArray != None else NULL
    err = cydriver.cuArrayCreate(<cydriver.CUarray*>pHandle._ptr, cypAllocateArray_ptr)
    return (CUresult(err), pHandle)

@cython.embedsignature(True)
def cuArrayGetDescriptor(hArray):
    """ Get a 1D or 2D CUDA array descriptor.

    Returns in `*pArrayDescriptor` a descriptor containing information on
    the format and dimensions of the CUDA array `hArray`. It is useful for
    subroutines that have been passed a CUDA array, but need to know the
    CUDA array parameters for validation or other purposes.

    Parameters
    ----------
    hArray : :py:obj:`~.CUarray`
        Array to get descriptor of

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`
    pArrayDescriptor : :py:obj:`~.CUDA_ARRAY_DESCRIPTOR`
        Returned array descriptor

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaArrayGetInfo`
    """
    cdef cydriver.CUarray cyhArray
    if hArray is None:
        cyhArray = <cydriver.CUarray><void_ptr>0
    elif isinstance(hArray, (CUarray,)):
        phArray = int(hArray)
        cyhArray = <cydriver.CUarray><void_ptr>phArray
    else:
        phArray = int(CUarray(hArray))
        cyhArray = <cydriver.CUarray><void_ptr>phArray
    cdef CUDA_ARRAY_DESCRIPTOR pArrayDescriptor = CUDA_ARRAY_DESCRIPTOR()
    err = cydriver.cuArrayGetDescriptor(<cydriver.CUDA_ARRAY_DESCRIPTOR*>pArrayDescriptor._ptr, cyhArray)
    return (CUresult(err), pArrayDescriptor)

@cython.embedsignature(True)
def cuArrayGetSparseProperties(array):
    """ Returns the layout properties of a sparse CUDA array.

    Returns the layout properties of a sparse CUDA array in
    `sparseProperties` If the CUDA array is not allocated with flag
    :py:obj:`~.CUDA_ARRAY3D_SPARSE` :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    will be returned.

    If the returned value in :py:obj:`~.CUDA_ARRAY_SPARSE_PROPERTIES.flags`
    contains :py:obj:`~.CU_ARRAY_SPARSE_PROPERTIES_SINGLE_MIPTAIL`, then
    :py:obj:`~.CUDA_ARRAY_SPARSE_PROPERTIES.miptailSize` represents the
    total size of the array. Otherwise, it will be zero. Also, the returned
    value in :py:obj:`~.CUDA_ARRAY_SPARSE_PROPERTIES.miptailFirstLevel` is
    always zero. Note that the `array` must have been allocated using
    :py:obj:`~.cuArrayCreate` or :py:obj:`~.cuArray3DCreate`. For CUDA
    arrays obtained using :py:obj:`~.cuMipmappedArrayGetLevel`,
    :py:obj:`~.CUDA_ERROR_INVALID_VALUE` will be returned. Instead,
    :py:obj:`~.cuMipmappedArrayGetSparseProperties` must be used to obtain
    the sparse properties of the entire CUDA mipmapped array to which
    `array` belongs to.

    Parameters
    ----------
    array : :py:obj:`~.CUarray`
        CUDA array to get the sparse properties of

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS` :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    sparseProperties : :py:obj:`~.CUDA_ARRAY_SPARSE_PROPERTIES`
        Pointer to :py:obj:`~.CUDA_ARRAY_SPARSE_PROPERTIES`

    See Also
    --------
    :py:obj:`~.cuMipmappedArrayGetSparseProperties`, :py:obj:`~.cuMemMapArrayAsync`
    """
    cdef cydriver.CUarray cyarray
    if array is None:
        cyarray = <cydriver.CUarray><void_ptr>0
    elif isinstance(array, (CUarray,)):
        parray = int(array)
        cyarray = <cydriver.CUarray><void_ptr>parray
    else:
        parray = int(CUarray(array))
        cyarray = <cydriver.CUarray><void_ptr>parray
    cdef CUDA_ARRAY_SPARSE_PROPERTIES sparseProperties = CUDA_ARRAY_SPARSE_PROPERTIES()
    err = cydriver.cuArrayGetSparseProperties(<cydriver.CUDA_ARRAY_SPARSE_PROPERTIES*>sparseProperties._ptr, cyarray)
    return (CUresult(err), sparseProperties)

@cython.embedsignature(True)
def cuMipmappedArrayGetSparseProperties(mipmap):
    """ Returns the layout properties of a sparse CUDA mipmapped array.

    Returns the sparse array layout properties in `sparseProperties` If the
    CUDA mipmapped array is not allocated with flag
    :py:obj:`~.CUDA_ARRAY3D_SPARSE` :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    will be returned.

    For non-layered CUDA mipmapped arrays,
    :py:obj:`~.CUDA_ARRAY_SPARSE_PROPERTIES.miptailSize` returns the size
    of the mip tail region. The mip tail region includes all mip levels
    whose width, height or depth is less than that of the tile. For layered
    CUDA mipmapped arrays, if
    :py:obj:`~.CUDA_ARRAY_SPARSE_PROPERTIES.flags` contains
    :py:obj:`~.CU_ARRAY_SPARSE_PROPERTIES_SINGLE_MIPTAIL`, then
    :py:obj:`~.CUDA_ARRAY_SPARSE_PROPERTIES.miptailSize` specifies the size
    of the mip tail of all layers combined. Otherwise,
    :py:obj:`~.CUDA_ARRAY_SPARSE_PROPERTIES.miptailSize` specifies mip tail
    size per layer. The returned value of
    :py:obj:`~.CUDA_ARRAY_SPARSE_PROPERTIES.miptailFirstLevel` is valid
    only if :py:obj:`~.CUDA_ARRAY_SPARSE_PROPERTIES.miptailSize` is non-
    zero.

    Parameters
    ----------
    mipmap : :py:obj:`~.CUmipmappedArray`
        CUDA mipmapped array to get the sparse properties of

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS` :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    sparseProperties : :py:obj:`~.CUDA_ARRAY_SPARSE_PROPERTIES`
        Pointer to :py:obj:`~.CUDA_ARRAY_SPARSE_PROPERTIES`

    See Also
    --------
    :py:obj:`~.cuArrayGetSparseProperties`, :py:obj:`~.cuMemMapArrayAsync`
    """
    cdef cydriver.CUmipmappedArray cymipmap
    if mipmap is None:
        cymipmap = <cydriver.CUmipmappedArray><void_ptr>0
    elif isinstance(mipmap, (CUmipmappedArray,)):
        pmipmap = int(mipmap)
        cymipmap = <cydriver.CUmipmappedArray><void_ptr>pmipmap
    else:
        pmipmap = int(CUmipmappedArray(mipmap))
        cymipmap = <cydriver.CUmipmappedArray><void_ptr>pmipmap
    cdef CUDA_ARRAY_SPARSE_PROPERTIES sparseProperties = CUDA_ARRAY_SPARSE_PROPERTIES()
    err = cydriver.cuMipmappedArrayGetSparseProperties(<cydriver.CUDA_ARRAY_SPARSE_PROPERTIES*>sparseProperties._ptr, cymipmap)
    return (CUresult(err), sparseProperties)

@cython.embedsignature(True)
def cuArrayGetMemoryRequirements(array, device):
    """ Returns the memory requirements of a CUDA array.

    Returns the memory requirements of a CUDA array in `memoryRequirements`
    If the CUDA array is not allocated with flag
    :py:obj:`~.CUDA_ARRAY3D_DEFERRED_MAPPING`
    :py:obj:`~.CUDA_ERROR_INVALID_VALUE` will be returned.

    The returned value in :py:obj:`~.CUDA_ARRAY_MEMORY_REQUIREMENTS.size`
    represents the total size of the CUDA array. The returned value in
    :py:obj:`~.CUDA_ARRAY_MEMORY_REQUIREMENTS.alignment` represents the
    alignment necessary for mapping the CUDA array.

    Parameters
    ----------
    array : :py:obj:`~.CUarray`
        CUDA array to get the memory requirements of
    device : :py:obj:`~.CUdevice`
        Device to get the memory requirements for

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS` :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    memoryRequirements : :py:obj:`~.CUDA_ARRAY_MEMORY_REQUIREMENTS`
        Pointer to :py:obj:`~.CUDA_ARRAY_MEMORY_REQUIREMENTS`

    See Also
    --------
    :py:obj:`~.cuMipmappedArrayGetMemoryRequirements`, :py:obj:`~.cuMemMapArrayAsync`
    """
    cdef cydriver.CUdevice cydevice
    if device is None:
        cydevice = <cydriver.CUdevice>0
    elif isinstance(device, (CUdevice,)):
        pdevice = int(device)
        cydevice = <cydriver.CUdevice>pdevice
    else:
        pdevice = int(CUdevice(device))
        cydevice = <cydriver.CUdevice>pdevice
    cdef cydriver.CUarray cyarray
    if array is None:
        cyarray = <cydriver.CUarray><void_ptr>0
    elif isinstance(array, (CUarray,)):
        parray = int(array)
        cyarray = <cydriver.CUarray><void_ptr>parray
    else:
        parray = int(CUarray(array))
        cyarray = <cydriver.CUarray><void_ptr>parray
    cdef CUDA_ARRAY_MEMORY_REQUIREMENTS memoryRequirements = CUDA_ARRAY_MEMORY_REQUIREMENTS()
    err = cydriver.cuArrayGetMemoryRequirements(<cydriver.CUDA_ARRAY_MEMORY_REQUIREMENTS*>memoryRequirements._ptr, cyarray, cydevice)
    return (CUresult(err), memoryRequirements)

@cython.embedsignature(True)
def cuMipmappedArrayGetMemoryRequirements(mipmap, device):
    """ Returns the memory requirements of a CUDA mipmapped array.

    Returns the memory requirements of a CUDA mipmapped array in
    `memoryRequirements` If the CUDA mipmapped array is not allocated with
    flag :py:obj:`~.CUDA_ARRAY3D_DEFERRED_MAPPING`
    :py:obj:`~.CUDA_ERROR_INVALID_VALUE` will be returned.

    The returned value in :py:obj:`~.CUDA_ARRAY_MEMORY_REQUIREMENTS.size`
    represents the total size of the CUDA mipmapped array. The returned
    value in :py:obj:`~.CUDA_ARRAY_MEMORY_REQUIREMENTS.alignment`
    represents the alignment necessary for mapping the CUDA mipmapped
    array.

    Parameters
    ----------
    mipmap : :py:obj:`~.CUmipmappedArray`
        CUDA mipmapped array to get the memory requirements of
    device : :py:obj:`~.CUdevice`
        Device to get the memory requirements for

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS` :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    memoryRequirements : :py:obj:`~.CUDA_ARRAY_MEMORY_REQUIREMENTS`
        Pointer to :py:obj:`~.CUDA_ARRAY_MEMORY_REQUIREMENTS`

    See Also
    --------
    :py:obj:`~.cuArrayGetMemoryRequirements`, :py:obj:`~.cuMemMapArrayAsync`
    """
    cdef cydriver.CUdevice cydevice
    if device is None:
        cydevice = <cydriver.CUdevice>0
    elif isinstance(device, (CUdevice,)):
        pdevice = int(device)
        cydevice = <cydriver.CUdevice>pdevice
    else:
        pdevice = int(CUdevice(device))
        cydevice = <cydriver.CUdevice>pdevice
    cdef cydriver.CUmipmappedArray cymipmap
    if mipmap is None:
        cymipmap = <cydriver.CUmipmappedArray><void_ptr>0
    elif isinstance(mipmap, (CUmipmappedArray,)):
        pmipmap = int(mipmap)
        cymipmap = <cydriver.CUmipmappedArray><void_ptr>pmipmap
    else:
        pmipmap = int(CUmipmappedArray(mipmap))
        cymipmap = <cydriver.CUmipmappedArray><void_ptr>pmipmap
    cdef CUDA_ARRAY_MEMORY_REQUIREMENTS memoryRequirements = CUDA_ARRAY_MEMORY_REQUIREMENTS()
    err = cydriver.cuMipmappedArrayGetMemoryRequirements(<cydriver.CUDA_ARRAY_MEMORY_REQUIREMENTS*>memoryRequirements._ptr, cymipmap, cydevice)
    return (CUresult(err), memoryRequirements)

@cython.embedsignature(True)
def cuArrayGetPlane(hArray, unsigned int planeIdx):
    """ Gets a CUDA array plane from a CUDA array.

    Returns in `pPlaneArray` a CUDA array that represents a single format
    plane of the CUDA array `hArray`.

    If `planeIdx` is greater than the maximum number of planes in this
    array or if the array does not have a multi-planar format e.g:
    :py:obj:`~.CU_AD_FORMAT_NV12`, then
    :py:obj:`~.CUDA_ERROR_INVALID_VALUE` is returned.

    Note that if the `hArray` has format :py:obj:`~.CU_AD_FORMAT_NV12`,
    then passing in 0 for `planeIdx` returns a CUDA array of the same size
    as `hArray` but with one channel and
    :py:obj:`~.CU_AD_FORMAT_UNSIGNED_INT8` as its format. If 1 is passed
    for `planeIdx`, then the returned CUDA array has half the height and
    width of `hArray` with two channels and
    :py:obj:`~.CU_AD_FORMAT_UNSIGNED_INT8` as its format.

    Parameters
    ----------
    hArray : :py:obj:`~.CUarray`
        Multiplanar CUDA array
    planeIdx : unsigned int
        Plane index

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`
    pPlaneArray : :py:obj:`~.CUarray`
        Returned CUDA array referenced by the `planeIdx`

    See Also
    --------
    :py:obj:`~.cuArrayCreate`, :py:obj:`~.cudaGetArrayPlane`
    """
    cdef cydriver.CUarray cyhArray
    if hArray is None:
        cyhArray = <cydriver.CUarray><void_ptr>0
    elif isinstance(hArray, (CUarray,)):
        phArray = int(hArray)
        cyhArray = <cydriver.CUarray><void_ptr>phArray
    else:
        phArray = int(CUarray(hArray))
        cyhArray = <cydriver.CUarray><void_ptr>phArray
    cdef CUarray pPlaneArray = CUarray()
    err = cydriver.cuArrayGetPlane(<cydriver.CUarray*>pPlaneArray._ptr, cyhArray, planeIdx)
    return (CUresult(err), pPlaneArray)

@cython.embedsignature(True)
def cuArrayDestroy(hArray):
    """ Destroys a CUDA array.

    Destroys the CUDA array `hArray`.

    Parameters
    ----------
    hArray : :py:obj:`~.CUarray`
        Array to destroy

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_ARRAY_IS_MAPPED`, :py:obj:`~.CUDA_ERROR_CONTEXT_IS_DESTROYED`

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaFreeArray`
    """
    cdef cydriver.CUarray cyhArray
    if hArray is None:
        cyhArray = <cydriver.CUarray><void_ptr>0
    elif isinstance(hArray, (CUarray,)):
        phArray = int(hArray)
        cyhArray = <cydriver.CUarray><void_ptr>phArray
    else:
        phArray = int(CUarray(hArray))
        cyhArray = <cydriver.CUarray><void_ptr>phArray
    err = cydriver.cuArrayDestroy(cyhArray)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuArray3DCreate(pAllocateArray : Optional[CUDA_ARRAY3D_DESCRIPTOR]):
    """ Creates a 3D CUDA array.

    Creates a CUDA array according to the
    :py:obj:`~.CUDA_ARRAY3D_DESCRIPTOR` structure `pAllocateArray` and
    returns a handle to the new CUDA array in `*pHandle`. The
    :py:obj:`~.CUDA_ARRAY3D_DESCRIPTOR` is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    where:

    - `Width`, `Height`, and `Depth` are the width, height, and depth of
      the CUDA array (in elements); the following types of CUDA arrays can
      be allocated:

      - A 1D array is allocated if `Height` and `Depth` extents are both
        zero.

      - A 2D array is allocated if only `Depth` extent is zero.

      - A 3D array is allocated if all three extents are non-zero.

      - A 1D layered CUDA array is allocated if only `Height` is zero and
        the :py:obj:`~.CUDA_ARRAY3D_LAYERED` flag is set. Each layer is a
        1D array. The number of layers is determined by the depth extent.

      - A 2D layered CUDA array is allocated if all three extents are non-
        zero and the :py:obj:`~.CUDA_ARRAY3D_LAYERED` flag is set. Each
        layer is a 2D array. The number of layers is determined by the
        depth extent.

      - A cubemap CUDA array is allocated if all three extents are non-zero
        and the :py:obj:`~.CUDA_ARRAY3D_CUBEMAP` flag is set. `Width` must
        be equal to `Height`, and `Depth` must be six. A cubemap is a
        special type of 2D layered CUDA array, where the six layers
        represent the six faces of a cube. The order of the six layers in
        memory is the same as that listed in
        :py:obj:`~.CUarray_cubemap_face`.

      - A cubemap layered CUDA array is allocated if all three extents are
        non-zero, and both, :py:obj:`~.CUDA_ARRAY3D_CUBEMAP` and
        :py:obj:`~.CUDA_ARRAY3D_LAYERED` flags are set. `Width` must be
        equal to `Height`, and `Depth` must be a multiple of six. A cubemap
        layered CUDA array is a special type of 2D layered CUDA array that
        consists of a collection of cubemaps. The first six layers
        represent the first cubemap, the next six layers form the second
        cubemap, and so on.

    - :py:obj:`~.Format` specifies the format of the elements;
      :py:obj:`~.CUarray_format` is defined as:

    - **View CUDA Toolkit Documentation for a C++ code example**

    - `NumChannels` specifies the number of packed components per CUDA
      array element; it may be 1, 2, or 4;

    - :py:obj:`~.Flags` may be set to

      - :py:obj:`~.CUDA_ARRAY3D_LAYERED` to enable creation of layered CUDA
        arrays. If this flag is set, `Depth` specifies the number of
        layers, not the depth of a 3D array.

      - :py:obj:`~.CUDA_ARRAY3D_SURFACE_LDST` to enable surface references
        to be bound to the CUDA array. If this flag is not set,
        :py:obj:`~.cuSurfRefSetArray` will fail when attempting to bind the
        CUDA array to a surface reference.

      - :py:obj:`~.CUDA_ARRAY3D_CUBEMAP` to enable creation of cubemaps. If
        this flag is set, `Width` must be equal to `Height`, and `Depth`
        must be six. If the :py:obj:`~.CUDA_ARRAY3D_LAYERED` flag is also
        set, then `Depth` must be a multiple of six.

      - :py:obj:`~.CUDA_ARRAY3D_TEXTURE_GATHER` to indicate that the CUDA
        array will be used for texture gather. Texture gather can only be
        performed on 2D CUDA arrays.

    `Width`, `Height` and `Depth` must meet certain size requirements as
    listed in the following table. All values are specified in elements.
    Note that for brevity's sake, the full name of the device attribute is
    not specified. For ex., TEXTURE1D_WIDTH refers to the device attribute
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE1D_WIDTH`.

    Note that 2D CUDA arrays have different size requirements if the
    :py:obj:`~.CUDA_ARRAY3D_TEXTURE_GATHER` flag is set. `Width` and
    `Height` must not be greater than
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_GATHER_WIDTH` and
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE2D_GATHER_HEIGHT`
    respectively, in that case.

    **View CUDA Toolkit Documentation for a table example**

    Here are examples of CUDA array descriptions:

    Description for a CUDA array of 2048 floats:

    **View CUDA Toolkit Documentation for a C++ code example**

    Description for a 64 x 64 CUDA array of floats:

    **View CUDA Toolkit Documentation for a C++ code example**

    Description for a `width` x `height` x `depth` CUDA array of 64-bit,
    4x16-bit float16's:

    **View CUDA Toolkit Documentation for a C++ code example**

    Parameters
    ----------
    pAllocateArray : :py:obj:`~.CUDA_ARRAY3D_DESCRIPTOR`
        3D array descriptor

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_UNKNOWN`
    pHandle : :py:obj:`~.CUarray`
        Returned array

    See Also
    --------
    :py:obj:`~.cuArray3DGetDescriptor`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaMalloc3DArray`
    """
    cdef CUarray pHandle = CUarray()
    cdef cydriver.CUDA_ARRAY3D_DESCRIPTOR* cypAllocateArray_ptr = pAllocateArray._ptr if pAllocateArray != None else NULL
    err = cydriver.cuArray3DCreate(<cydriver.CUarray*>pHandle._ptr, cypAllocateArray_ptr)
    return (CUresult(err), pHandle)

@cython.embedsignature(True)
def cuArray3DGetDescriptor(hArray):
    """ Get a 3D CUDA array descriptor.

    Returns in `*pArrayDescriptor` a descriptor containing information on
    the format and dimensions of the CUDA array `hArray`. It is useful for
    subroutines that have been passed a CUDA array, but need to know the
    CUDA array parameters for validation or other purposes.

    This function may be called on 1D and 2D arrays, in which case the
    `Height` and/or `Depth` members of the descriptor struct will be set to
    0.

    Parameters
    ----------
    hArray : :py:obj:`~.CUarray`
        3D array to get descriptor of

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_CONTEXT_IS_DESTROYED`
    pArrayDescriptor : :py:obj:`~.CUDA_ARRAY3D_DESCRIPTOR`
        Returned 3D array descriptor

    See Also
    --------
    :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArrayDestroy`, :py:obj:`~.cuArrayGetDescriptor`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemAllocPitch`, :py:obj:`~.cuMemcpy2D`, :py:obj:`~.cuMemcpy2DAsync`, :py:obj:`~.cuMemcpy2DUnaligned`, :py:obj:`~.cuMemcpy3D`, :py:obj:`~.cuMemcpy3DAsync`, :py:obj:`~.cuMemcpyAtoA`, :py:obj:`~.cuMemcpyAtoD`, :py:obj:`~.cuMemcpyAtoH`, :py:obj:`~.cuMemcpyAtoHAsync`, :py:obj:`~.cuMemcpyDtoA`, :py:obj:`~.cuMemcpyDtoD`, :py:obj:`~.cuMemcpyDtoDAsync`, :py:obj:`~.cuMemcpyDtoH`, :py:obj:`~.cuMemcpyDtoHAsync`, :py:obj:`~.cuMemcpyHtoA`, :py:obj:`~.cuMemcpyHtoAAsync`, :py:obj:`~.cuMemcpyHtoD`, :py:obj:`~.cuMemcpyHtoDAsync`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemGetAddressRange`, :py:obj:`~.cuMemGetInfo`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostGetDevicePointer`, :py:obj:`~.cuMemsetD2D8`, :py:obj:`~.cuMemsetD2D16`, :py:obj:`~.cuMemsetD2D32`, :py:obj:`~.cuMemsetD8`, :py:obj:`~.cuMemsetD16`, :py:obj:`~.cuMemsetD32`, :py:obj:`~.cudaArrayGetInfo`
    """
    cdef cydriver.CUarray cyhArray
    if hArray is None:
        cyhArray = <cydriver.CUarray><void_ptr>0
    elif isinstance(hArray, (CUarray,)):
        phArray = int(hArray)
        cyhArray = <cydriver.CUarray><void_ptr>phArray
    else:
        phArray = int(CUarray(hArray))
        cyhArray = <cydriver.CUarray><void_ptr>phArray
    cdef CUDA_ARRAY3D_DESCRIPTOR pArrayDescriptor = CUDA_ARRAY3D_DESCRIPTOR()
    err = cydriver.cuArray3DGetDescriptor(<cydriver.CUDA_ARRAY3D_DESCRIPTOR*>pArrayDescriptor._ptr, cyhArray)
    return (CUresult(err), pArrayDescriptor)

@cython.embedsignature(True)
def cuMipmappedArrayCreate(pMipmappedArrayDesc : Optional[CUDA_ARRAY3D_DESCRIPTOR], unsigned int numMipmapLevels):
    """ Creates a CUDA mipmapped array.

    Creates a CUDA mipmapped array according to the
    :py:obj:`~.CUDA_ARRAY3D_DESCRIPTOR` structure `pMipmappedArrayDesc` and
    returns a handle to the new CUDA mipmapped array in `*pHandle`.
    `numMipmapLevels` specifies the number of mipmap levels to be
    allocated. This value is clamped to the range [1, 1 +
    floor(log2(max(width, height, depth)))].

    The :py:obj:`~.CUDA_ARRAY3D_DESCRIPTOR` is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    where:

    - `Width`, `Height`, and `Depth` are the width, height, and depth of
      the CUDA array (in elements); the following types of CUDA arrays can
      be allocated:

      - A 1D mipmapped array is allocated if `Height` and `Depth` extents
        are both zero.

      - A 2D mipmapped array is allocated if only `Depth` extent is zero.

      - A 3D mipmapped array is allocated if all three extents are non-
        zero.

      - A 1D layered CUDA mipmapped array is allocated if only `Height` is
        zero and the :py:obj:`~.CUDA_ARRAY3D_LAYERED` flag is set. Each
        layer is a 1D array. The number of layers is determined by the
        depth extent.

      - A 2D layered CUDA mipmapped array is allocated if all three extents
        are non-zero and the :py:obj:`~.CUDA_ARRAY3D_LAYERED` flag is set.
        Each layer is a 2D array. The number of layers is determined by the
        depth extent.

      - A cubemap CUDA mipmapped array is allocated if all three extents
        are non-zero and the :py:obj:`~.CUDA_ARRAY3D_CUBEMAP` flag is set.
        `Width` must be equal to `Height`, and `Depth` must be six. A
        cubemap is a special type of 2D layered CUDA array, where the six
        layers represent the six faces of a cube. The order of the six
        layers in memory is the same as that listed in
        :py:obj:`~.CUarray_cubemap_face`.

      - A cubemap layered CUDA mipmapped array is allocated if all three
        extents are non-zero, and both, :py:obj:`~.CUDA_ARRAY3D_CUBEMAP`
        and :py:obj:`~.CUDA_ARRAY3D_LAYERED` flags are set. `Width` must be
        equal to `Height`, and `Depth` must be a multiple of six. A cubemap
        layered CUDA array is a special type of 2D layered CUDA array that
        consists of a collection of cubemaps. The first six layers
        represent the first cubemap, the next six layers form the second
        cubemap, and so on.

    - :py:obj:`~.Format` specifies the format of the elements;
      :py:obj:`~.CUarray_format` is defined as:

    - **View CUDA Toolkit Documentation for a C++ code example**

    - `NumChannels` specifies the number of packed components per CUDA
      array element; it may be 1, 2, or 4;

    - :py:obj:`~.Flags` may be set to

      - :py:obj:`~.CUDA_ARRAY3D_LAYERED` to enable creation of layered CUDA
        mipmapped arrays. If this flag is set, `Depth` specifies the number
        of layers, not the depth of a 3D array.

      - :py:obj:`~.CUDA_ARRAY3D_SURFACE_LDST` to enable surface references
        to be bound to individual mipmap levels of the CUDA mipmapped
        array. If this flag is not set, :py:obj:`~.cuSurfRefSetArray` will
        fail when attempting to bind a mipmap level of the CUDA mipmapped
        array to a surface reference.

    - :py:obj:`~.CUDA_ARRAY3D_CUBEMAP` to enable creation of mipmapped
    cubemaps. If this flag is set, `Width` must be equal to `Height`, and
    `Depth` must be six. If the :py:obj:`~.CUDA_ARRAY3D_LAYERED` flag is
    also set, then `Depth` must be a multiple of six.

      - :py:obj:`~.CUDA_ARRAY3D_TEXTURE_GATHER` to indicate that the CUDA
        mipmapped array will be used for texture gather. Texture gather can
        only be performed on 2D CUDA mipmapped arrays.

    `Width`, `Height` and `Depth` must meet certain size requirements as
    listed in the following table. All values are specified in elements.
    Note that for brevity's sake, the full name of the device attribute is
    not specified. For ex., TEXTURE1D_MIPMAPPED_WIDTH refers to the device
    attribute
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_MAXIMUM_TEXTURE1D_MIPMAPPED_WIDTH`.

    **View CUDA Toolkit Documentation for a table example**

    Parameters
    ----------
    pMipmappedArrayDesc : :py:obj:`~.CUDA_ARRAY3D_DESCRIPTOR`
        mipmapped array descriptor
    numMipmapLevels : unsigned int
        Number of mipmap levels

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_UNKNOWN`
    pHandle : :py:obj:`~.CUmipmappedArray`
        Returned mipmapped array

    See Also
    --------
    :py:obj:`~.cuMipmappedArrayDestroy`, :py:obj:`~.cuMipmappedArrayGetLevel`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cudaMallocMipmappedArray`
    """
    cdef CUmipmappedArray pHandle = CUmipmappedArray()
    cdef cydriver.CUDA_ARRAY3D_DESCRIPTOR* cypMipmappedArrayDesc_ptr = pMipmappedArrayDesc._ptr if pMipmappedArrayDesc != None else NULL
    err = cydriver.cuMipmappedArrayCreate(<cydriver.CUmipmappedArray*>pHandle._ptr, cypMipmappedArrayDesc_ptr, numMipmapLevels)
    return (CUresult(err), pHandle)

@cython.embedsignature(True)
def cuMipmappedArrayGetLevel(hMipmappedArray, unsigned int level):
    """ Gets a mipmap level of a CUDA mipmapped array.

    Returns in `*pLevelArray` a CUDA array that represents a single mipmap
    level of the CUDA mipmapped array `hMipmappedArray`.

    If `level` is greater than the maximum number of levels in this
    mipmapped array, :py:obj:`~.CUDA_ERROR_INVALID_VALUE` is returned.

    Parameters
    ----------
    hMipmappedArray : :py:obj:`~.CUmipmappedArray`
        CUDA mipmapped array
    level : unsigned int
        Mipmap level

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`
    pLevelArray : :py:obj:`~.CUarray`
        Returned mipmap level CUDA array

    See Also
    --------
    :py:obj:`~.cuMipmappedArrayCreate`, :py:obj:`~.cuMipmappedArrayDestroy`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cudaGetMipmappedArrayLevel`
    """
    cdef cydriver.CUmipmappedArray cyhMipmappedArray
    if hMipmappedArray is None:
        cyhMipmappedArray = <cydriver.CUmipmappedArray><void_ptr>0
    elif isinstance(hMipmappedArray, (CUmipmappedArray,)):
        phMipmappedArray = int(hMipmappedArray)
        cyhMipmappedArray = <cydriver.CUmipmappedArray><void_ptr>phMipmappedArray
    else:
        phMipmappedArray = int(CUmipmappedArray(hMipmappedArray))
        cyhMipmappedArray = <cydriver.CUmipmappedArray><void_ptr>phMipmappedArray
    cdef CUarray pLevelArray = CUarray()
    err = cydriver.cuMipmappedArrayGetLevel(<cydriver.CUarray*>pLevelArray._ptr, cyhMipmappedArray, level)
    return (CUresult(err), pLevelArray)

@cython.embedsignature(True)
def cuMipmappedArrayDestroy(hMipmappedArray):
    """ Destroys a CUDA mipmapped array.

    Destroys the CUDA mipmapped array `hMipmappedArray`.

    Parameters
    ----------
    hMipmappedArray : :py:obj:`~.CUmipmappedArray`
        Mipmapped array to destroy

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_ARRAY_IS_MAPPED`, :py:obj:`~.CUDA_ERROR_CONTEXT_IS_DESTROYED`

    See Also
    --------
    :py:obj:`~.cuMipmappedArrayCreate`, :py:obj:`~.cuMipmappedArrayGetLevel`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cudaFreeMipmappedArray`
    """
    cdef cydriver.CUmipmappedArray cyhMipmappedArray
    if hMipmappedArray is None:
        cyhMipmappedArray = <cydriver.CUmipmappedArray><void_ptr>0
    elif isinstance(hMipmappedArray, (CUmipmappedArray,)):
        phMipmappedArray = int(hMipmappedArray)
        cyhMipmappedArray = <cydriver.CUmipmappedArray><void_ptr>phMipmappedArray
    else:
        phMipmappedArray = int(CUmipmappedArray(hMipmappedArray))
        cyhMipmappedArray = <cydriver.CUmipmappedArray><void_ptr>phMipmappedArray
    err = cydriver.cuMipmappedArrayDestroy(cyhMipmappedArray)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemGetHandleForAddressRange(dptr, size_t size, handleType not None : CUmemRangeHandleType, unsigned long long flags):
    """ Retrieve handle for an address range.

    Get a handle of the specified type to an address range. The address
    range must have been obtained by a prior call to either
    :py:obj:`~.cuMemAlloc` or :py:obj:`~.cuMemAddressReserve`. If the
    address range was obtained via :py:obj:`~.cuMemAddressReserve`, it must
    also be fully mapped via :py:obj:`~.cuMemMap`.

    Users must ensure the `dptr` and `size` are aligned to the host page
    size.

    When requesting
    CUmemRangeHandleType::CU_MEM_RANGE_HANDLE_TYPE_DMA_BUF_FD, users are
    expected to query for dma_buf support for the platform by using
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_DMA_BUF_SUPPORTED` device attribute
    before calling this API. The `handle` will be interpreted as a pointer
    to an integer to store the dma_buf file descriptor. Users must ensure
    the entire address range is backed and mapped when the address range is
    allocated by :py:obj:`~.cuMemAddressReserve`. All the physical
    allocations backing the address range must be resident on the same
    device and have identical allocation properties. Users are also
    expected to retrieve a new handle every time the underlying physical
    allocation(s) corresponding to a previously queried VA range are
    changed.

    Parameters
    ----------
    dptr : :py:obj:`~.CUdeviceptr`
        Pointer to a valid CUDA device allocation. Must be aligned to host
        page size.
    size : size_t
        Length of the address range. Must be aligned to host page size.
    handleType : :py:obj:`~.CUmemRangeHandleType`
        Type of handle requested (defines type and size of the `handle`
        output parameter)
    flags : unsigned long long
        Reserved, must be zero

    Returns
    -------
    CUresult
        CUDA_SUCCESS CUDA_ERROR_INVALID_VALUE CUDA_ERROR_NOT_SUPPORTED
    handle : Any
        Pointer to the location where the returned handle will be stored.
    """
    cdef cydriver.CUdeviceptr cydptr
    if dptr is None:
        cydptr = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(dptr, (CUdeviceptr,)):
        pdptr = int(dptr)
        cydptr = <cydriver.CUdeviceptr><void_ptr>pdptr
    else:
        pdptr = int(CUdeviceptr(dptr))
        cydptr = <cydriver.CUdeviceptr><void_ptr>pdptr
    cdef int handle = 0
    cdef void* cyhandle_ptr = <void*>&handle
    cdef cydriver.CUmemRangeHandleType cyhandleType = handleType.value
    err = cydriver.cuMemGetHandleForAddressRange(cyhandle_ptr, cydptr, size, cyhandleType, flags)
    return (CUresult(err), handle)

@cython.embedsignature(True)
def cuMemAddressReserve(size_t size, size_t alignment, addr, unsigned long long flags):
    """ Allocate an address range reservation.

    Reserves a virtual address range based on the given parameters, giving
    the starting address of the range in `ptr`. This API requires a system
    that supports UVA. The size and address parameters must be a multiple
    of the host page size and the alignment must be a power of two or zero
    for default alignment.

    Parameters
    ----------
    size : size_t
        Size of the reserved virtual address range requested
    alignment : size_t
        Alignment of the reserved virtual address range requested
    addr : :py:obj:`~.CUdeviceptr`
        Fixed starting address range requested
    flags : unsigned long long
        Currently unused, must be zero

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_PERMITTED`, :py:obj:`~.CUDA_ERROR_NOT_SUPPORTED`
    ptr : :py:obj:`~.CUdeviceptr`
        Resulting pointer to start of virtual address range allocated

    See Also
    --------
    :py:obj:`~.cuMemAddressFree`
    """
    cdef cydriver.CUdeviceptr cyaddr
    if addr is None:
        cyaddr = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(addr, (CUdeviceptr,)):
        paddr = int(addr)
        cyaddr = <cydriver.CUdeviceptr><void_ptr>paddr
    else:
        paddr = int(CUdeviceptr(addr))
        cyaddr = <cydriver.CUdeviceptr><void_ptr>paddr
    cdef CUdeviceptr ptr = CUdeviceptr()
    err = cydriver.cuMemAddressReserve(<cydriver.CUdeviceptr*>ptr._ptr, size, alignment, cyaddr, flags)
    return (CUresult(err), ptr)

@cython.embedsignature(True)
def cuMemAddressFree(ptr, size_t size):
    """ Free an address range reservation.

    Frees a virtual address range reserved by cuMemAddressReserve. The size
    must match what was given to memAddressReserve and the ptr given must
    match what was returned from memAddressReserve.

    Parameters
    ----------
    ptr : :py:obj:`~.CUdeviceptr`
        Starting address of the virtual address range to free
    size : size_t
        Size of the virtual address region to free

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_PERMITTED`, :py:obj:`~.CUDA_ERROR_NOT_SUPPORTED`

    See Also
    --------
    :py:obj:`~.cuMemAddressReserve`
    """
    cdef cydriver.CUdeviceptr cyptr
    if ptr is None:
        cyptr = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(ptr, (CUdeviceptr,)):
        pptr = int(ptr)
        cyptr = <cydriver.CUdeviceptr><void_ptr>pptr
    else:
        pptr = int(CUdeviceptr(ptr))
        cyptr = <cydriver.CUdeviceptr><void_ptr>pptr
    err = cydriver.cuMemAddressFree(cyptr, size)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemCreate(size_t size, prop : Optional[CUmemAllocationProp], unsigned long long flags):
    """ Create a CUDA memory handle representing a memory allocation of a given size described by the given properties.

    This creates a memory allocation on the target device specified through
    the `prop` strcuture. The created allocation will not have any device
    or host mappings. The generic memory `handle` for the allocation can be
    mapped to the address space of calling process via
    :py:obj:`~.cuMemMap`. This handle cannot be transmitted directly to
    other processes (see :py:obj:`~.cuMemExportToShareableHandle`). On
    Windows, the caller must also pass an LPSECURITYATTRIBUTE in `prop` to
    be associated with this handle which limits or allows access to this
    handle for a recepient process (see
    :py:obj:`~.CUmemAllocationProp.win32HandleMetaData` for more). The
    `size` of this allocation must be a multiple of the the value given via
    :py:obj:`~.cuMemGetAllocationGranularity` with the
    :py:obj:`~.CU_MEM_ALLOC_GRANULARITY_MINIMUM` flag. If
    :py:obj:`~.CUmemAllocationProp`::allocFlags::usage contains
    :py:obj:`~.CU_MEM_CREATE_USAGE_TILE_POOL` flag then the memory
    allocation is intended only to be used as backing tile pool for sparse
    CUDA arrays and sparse CUDA mipmapped arrays. (see
    :py:obj:`~.cuMemMapArrayAsync`).

    Parameters
    ----------
    size : size_t
        Size of the allocation requested
    prop : :py:obj:`~.CUmemAllocationProp`
        Properties of the allocation to create.
    flags : unsigned long long
        flags for future use, must be zero now.

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_PERMITTED`, :py:obj:`~.CUDA_ERROR_NOT_SUPPORTED`
    handle : :py:obj:`~.CUmemGenericAllocationHandle`
        Value of handle returned. All operations on this allocation are to
        be performed using this handle.

    See Also
    --------
    :py:obj:`~.cuMemRelease`, :py:obj:`~.cuMemExportToShareableHandle`, :py:obj:`~.cuMemImportFromShareableHandle`
    """
    cdef CUmemGenericAllocationHandle handle = CUmemGenericAllocationHandle()
    cdef cydriver.CUmemAllocationProp* cyprop_ptr = prop._ptr if prop != None else NULL
    err = cydriver.cuMemCreate(<cydriver.CUmemGenericAllocationHandle*>handle._ptr, size, cyprop_ptr, flags)
    return (CUresult(err), handle)

@cython.embedsignature(True)
def cuMemRelease(handle):
    """ Release a memory handle representing a memory allocation which was previously allocated through cuMemCreate.

    Frees the memory that was allocated on a device through cuMemCreate.

    The memory allocation will be freed when all outstanding mappings to
    the memory are unmapped and when all outstanding references to the
    handle (including it's shareable counterparts) are also released. The
    generic memory handle can be freed when there are still outstanding
    mappings made with this handle. Each time a recepient process imports a
    shareable handle, it needs to pair it with :py:obj:`~.cuMemRelease` for
    the handle to be freed. If `handle` is not a valid handle the behavior
    is undefined.

    Parameters
    ----------
    handle : :py:obj:`~.CUmemGenericAllocationHandle`
        Value of handle which was returned previously by cuMemCreate.

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_PERMITTED`, :py:obj:`~.CUDA_ERROR_NOT_SUPPORTED`

    See Also
    --------
    :py:obj:`~.cuMemCreate`
    """
    cdef cydriver.CUmemGenericAllocationHandle cyhandle
    if handle is None:
        cyhandle = <cydriver.CUmemGenericAllocationHandle><void_ptr>0
    elif isinstance(handle, (CUmemGenericAllocationHandle,)):
        phandle = int(handle)
        cyhandle = <cydriver.CUmemGenericAllocationHandle><void_ptr>phandle
    else:
        phandle = int(CUmemGenericAllocationHandle(handle))
        cyhandle = <cydriver.CUmemGenericAllocationHandle><void_ptr>phandle
    err = cydriver.cuMemRelease(cyhandle)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemMap(ptr, size_t size, size_t offset, handle, unsigned long long flags):
    """ Maps an allocation handle to a reserved virtual address range.

    Maps bytes of memory represented by `handle` starting from byte
    `offset` to `size` to address range [`addr`, `addr` + `size`]. This
    range must be an address reservation previously reserved with
    :py:obj:`~.cuMemAddressReserve`, and `offset` + `size` must be less
    than the size of the memory allocation. Both `ptr`, `size`, and
    `offset` must be a multiple of the value given via
    :py:obj:`~.cuMemGetAllocationGranularity` with the
    :py:obj:`~.CU_MEM_ALLOC_GRANULARITY_MINIMUM` flag.

    Please note calling :py:obj:`~.cuMemMap` does not make the address
    accessible, the caller needs to update accessibility of a contiguous
    mapped VA range by calling :py:obj:`~.cuMemSetAccess`.

    Once a recipient process obtains a shareable memory handle from
    :py:obj:`~.cuMemImportFromShareableHandle`, the process must use
    :py:obj:`~.cuMemMap` to map the memory into its address ranges before
    setting accessibility with :py:obj:`~.cuMemSetAccess`.

    :py:obj:`~.cuMemMap` can only create mappings on VA range reservations
    that are not currently mapped.

    Parameters
    ----------
    ptr : :py:obj:`~.CUdeviceptr`
        Address where memory will be mapped.
    size : size_t
        Size of the memory mapping.
    offset : size_t
        Offset into the memory represented by
    handle : :py:obj:`~.CUmemGenericAllocationHandle`
        Handle to a shareable memory
    flags : unsigned long long
        flags for future use, must be zero now.

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_PERMITTED`, :py:obj:`~.CUDA_ERROR_NOT_SUPPORTED`

    See Also
    --------
    :py:obj:`~.cuMemUnmap`, :py:obj:`~.cuMemSetAccess`, :py:obj:`~.cuMemCreate`, :py:obj:`~.cuMemAddressReserve`, :py:obj:`~.cuMemImportFromShareableHandle`
    """
    cdef cydriver.CUmemGenericAllocationHandle cyhandle
    if handle is None:
        cyhandle = <cydriver.CUmemGenericAllocationHandle><void_ptr>0
    elif isinstance(handle, (CUmemGenericAllocationHandle,)):
        phandle = int(handle)
        cyhandle = <cydriver.CUmemGenericAllocationHandle><void_ptr>phandle
    else:
        phandle = int(CUmemGenericAllocationHandle(handle))
        cyhandle = <cydriver.CUmemGenericAllocationHandle><void_ptr>phandle
    cdef cydriver.CUdeviceptr cyptr
    if ptr is None:
        cyptr = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(ptr, (CUdeviceptr,)):
        pptr = int(ptr)
        cyptr = <cydriver.CUdeviceptr><void_ptr>pptr
    else:
        pptr = int(CUdeviceptr(ptr))
        cyptr = <cydriver.CUdeviceptr><void_ptr>pptr
    err = cydriver.cuMemMap(cyptr, size, offset, cyhandle, flags)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemMapArrayAsync(mapInfoList : Optional[Tuple[CUarrayMapInfo] | List[CUarrayMapInfo]], unsigned int count, hStream):
    """ Maps or unmaps subregions of sparse CUDA arrays and sparse CUDA mipmapped arrays.

    Performs map or unmap operations on subregions of sparse CUDA arrays
    and sparse CUDA mipmapped arrays. Each operation is specified by a
    :py:obj:`~.CUarrayMapInfo` entry in the `mapInfoList` array of size
    `count`. The structure :py:obj:`~.CUarrayMapInfo` is defined as follow:

    **View CUDA Toolkit Documentation for a C++ code example**

    where :py:obj:`~.CUarrayMapInfo.resourceType` specifies the type of
    resource to be operated on. If :py:obj:`~.CUarrayMapInfo.resourceType`
    is set to :py:obj:`~.CUresourcetype`::CU_RESOURCE_TYPE_ARRAY then
    :py:obj:`~.CUarrayMapInfo`::resource::array must be set to a valid
    sparse CUDA array handle. The CUDA array must be either a 2D, 2D
    layered or 3D CUDA array and must have been allocated using
    :py:obj:`~.cuArrayCreate` or :py:obj:`~.cuArray3DCreate` with the flag
    :py:obj:`~.CUDA_ARRAY3D_SPARSE` or
    :py:obj:`~.CUDA_ARRAY3D_DEFERRED_MAPPING`. For CUDA arrays obtained
    using :py:obj:`~.cuMipmappedArrayGetLevel`,
    :py:obj:`~.CUDA_ERROR_INVALID_VALUE` will be returned. If
    :py:obj:`~.CUarrayMapInfo.resourceType` is set to
    :py:obj:`~.CUresourcetype`::CU_RESOURCE_TYPE_MIPMAPPED_ARRAY then
    :py:obj:`~.CUarrayMapInfo`::resource::mipmap must be set to a valid
    sparse CUDA mipmapped array handle. The CUDA mipmapped array must be
    either a 2D, 2D layered or 3D CUDA mipmapped array and must have been
    allocated using :py:obj:`~.cuMipmappedArrayCreate` with the flag
    :py:obj:`~.CUDA_ARRAY3D_SPARSE` or
    :py:obj:`~.CUDA_ARRAY3D_DEFERRED_MAPPING`.

    :py:obj:`~.CUarrayMapInfo.subresourceType` specifies the type of
    subresource within the resource.
    :py:obj:`~.CUarraySparseSubresourceType_enum` is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    where
    :py:obj:`~.CUarraySparseSubresourceType`::CU_ARRAY_SPARSE_SUBRESOURCE_TYPE_SPARSE_LEVEL
    indicates a sparse-miplevel which spans at least one tile in every
    dimension. The remaining miplevels which are too small to span at least
    one tile in any dimension constitute the mip tail region as indicated
    by
    :py:obj:`~.CUarraySparseSubresourceType`::CU_ARRAY_SPARSE_SUBRESOURCE_TYPE_MIPTAIL
    subresource type.

    If :py:obj:`~.CUarrayMapInfo.subresourceType` is set to
    :py:obj:`~.CUarraySparseSubresourceType`::CU_ARRAY_SPARSE_SUBRESOURCE_TYPE_SPARSE_LEVEL
    then :py:obj:`~.CUarrayMapInfo`::subresource::sparseLevel struct must
    contain valid array subregion offsets and extents. The
    :py:obj:`~.CUarrayMapInfo`::subresource::sparseLevel::offsetX,
    :py:obj:`~.CUarrayMapInfo`::subresource::sparseLevel::offsetY and
    :py:obj:`~.CUarrayMapInfo`::subresource::sparseLevel::offsetZ must
    specify valid X, Y and Z offsets respectively. The
    :py:obj:`~.CUarrayMapInfo`::subresource::sparseLevel::extentWidth,
    :py:obj:`~.CUarrayMapInfo`::subresource::sparseLevel::extentHeight and
    :py:obj:`~.CUarrayMapInfo`::subresource::sparseLevel::extentDepth must
    specify valid width, height and depth extents respectively. These
    offsets and extents must be aligned to the corresponding tile
    dimension. For CUDA mipmapped arrays
    :py:obj:`~.CUarrayMapInfo`::subresource::sparseLevel::level must
    specify a valid mip level index. Otherwise, must be zero. For layered
    CUDA arrays and layered CUDA mipmapped arrays
    :py:obj:`~.CUarrayMapInfo`::subresource::sparseLevel::layer must
    specify a valid layer index. Otherwise, must be zero.
    :py:obj:`~.CUarrayMapInfo`::subresource::sparseLevel::offsetZ must be
    zero and
    :py:obj:`~.CUarrayMapInfo`::subresource::sparseLevel::extentDepth must
    be set to 1 for 2D and 2D layered CUDA arrays and CUDA mipmapped
    arrays. Tile extents can be obtained by calling
    :py:obj:`~.cuArrayGetSparseProperties` and
    :py:obj:`~.cuMipmappedArrayGetSparseProperties`

    If :py:obj:`~.CUarrayMapInfo.subresourceType` is set to
    :py:obj:`~.CUarraySparseSubresourceType`::CU_ARRAY_SPARSE_SUBRESOURCE_TYPE_MIPTAIL
    then :py:obj:`~.CUarrayMapInfo`::subresource::miptail struct must
    contain valid mip tail offset in
    :py:obj:`~.CUarrayMapInfo`::subresource::miptail::offset and size in
    :py:obj:`~.CUarrayMapInfo`::subresource::miptail::size. Both, mip tail
    offset and mip tail size must be aligned to the tile size. For layered
    CUDA mipmapped arrays which don't have the flag
    :py:obj:`~.CU_ARRAY_SPARSE_PROPERTIES_SINGLE_MIPTAIL` set in
    :py:obj:`~.CUDA_ARRAY_SPARSE_PROPERTIES.flags` as returned by
    :py:obj:`~.cuMipmappedArrayGetSparseProperties`,
    :py:obj:`~.CUarrayMapInfo`::subresource::miptail::layer must specify a
    valid layer index. Otherwise, must be zero.

    If :py:obj:`~.CUarrayMapInfo`::resource::array or
    :py:obj:`~.CUarrayMapInfo`::resource::mipmap was created with
    :py:obj:`~.CUDA_ARRAY3D_DEFERRED_MAPPING` flag set the
    :py:obj:`~.CUarrayMapInfo.subresourceType` and the contents of
    :py:obj:`~.CUarrayMapInfo`::subresource will be ignored.

    :py:obj:`~.CUarrayMapInfo.memOperationType` specifies the type of
    operation. :py:obj:`~.CUmemOperationType` is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    If :py:obj:`~.CUarrayMapInfo.memOperationType` is set to
    :py:obj:`~.CUmemOperationType`::CU_MEM_OPERATION_TYPE_MAP then the
    subresource will be mapped onto the tile pool memory specified by
    :py:obj:`~.CUarrayMapInfo`::memHandle at offset
    :py:obj:`~.CUarrayMapInfo.offset`. The tile pool allocation has to be
    created by specifying the :py:obj:`~.CU_MEM_CREATE_USAGE_TILE_POOL`
    flag when calling :py:obj:`~.cuMemCreate`. Also,
    :py:obj:`~.CUarrayMapInfo.memHandleType` must be set to
    :py:obj:`~.CUmemHandleType`::CU_MEM_HANDLE_TYPE_GENERIC.

    If :py:obj:`~.CUarrayMapInfo.memOperationType` is set to
    :py:obj:`~.CUmemOperationType`::CU_MEM_OPERATION_TYPE_UNMAP then an
    unmapping operation is performed. :py:obj:`~.CUarrayMapInfo`::memHandle
    must be NULL.

    :py:obj:`~.CUarrayMapInfo.deviceBitMask` specifies the list of devices
    that must map or unmap physical memory. Currently, this mask must have
    exactly one bit set, and the corresponding device must match the device
    associated with the stream. If
    :py:obj:`~.CUarrayMapInfo.memOperationType` is set to
    :py:obj:`~.CUmemOperationType`::CU_MEM_OPERATION_TYPE_MAP, the device
    must also match the device associated with the tile pool memory
    allocation as specified by :py:obj:`~.CUarrayMapInfo`::memHandle.

    :py:obj:`~.CUarrayMapInfo.flags` and
    :py:obj:`~.CUarrayMapInfo.reserved`[] are unused and must be set to
    zero.

    Parameters
    ----------
    mapInfoList : List[:py:obj:`~.CUarrayMapInfo`]
        List of :py:obj:`~.CUarrayMapInfo`
    count : unsigned int
        Count of :py:obj:`~.CUarrayMapInfo` in `mapInfoList`
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream identifier for the stream to use for map or unmap operations

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`

    See Also
    --------
    :py:obj:`~.cuMipmappedArrayCreate`, :py:obj:`~.cuArrayCreate`, :py:obj:`~.cuArray3DCreate`, :py:obj:`~.cuMemCreate`, :py:obj:`~.cuArrayGetSparseProperties`, :py:obj:`~.cuMipmappedArrayGetSparseProperties`
    """
    cdef cydriver.CUstream cyhStream
    if hStream is None:
        cyhStream = <cydriver.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    mapInfoList = [] if mapInfoList is None else mapInfoList
    if not all(isinstance(_x, (CUarrayMapInfo,)) for _x in mapInfoList):
        raise TypeError("Argument 'mapInfoList' is not instance of type (expected Tuple[cydriver.CUarrayMapInfo,] or List[cydriver.CUarrayMapInfo,]")
    cdef cydriver.CUarrayMapInfo* cymapInfoList = NULL
    if len(mapInfoList) > 0:
        cymapInfoList = <cydriver.CUarrayMapInfo*> calloc(len(mapInfoList), sizeof(cydriver.CUarrayMapInfo))
        if cymapInfoList is NULL:
            raise MemoryError('Failed to allocate length x size memory: ' + str(len(mapInfoList)) + 'x' + str(sizeof(cydriver.CUarrayMapInfo)))
        for idx in range(len(mapInfoList)):
            string.memcpy(&cymapInfoList[idx], (<CUarrayMapInfo>mapInfoList[idx])._ptr, sizeof(cydriver.CUarrayMapInfo))
    if count > len(mapInfoList): raise RuntimeError("List is too small: " + str(len(mapInfoList)) + " < " + str(count))
    err = cydriver.cuMemMapArrayAsync((<CUarrayMapInfo>mapInfoList[0])._ptr if len(mapInfoList) == 1 else cymapInfoList, count, cyhStream)
    if cymapInfoList is not NULL:
        free(cymapInfoList)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemUnmap(ptr, size_t size):
    """ Unmap the backing memory of a given address range.

    The range must be the entire contiguous address range that was mapped
    to. In other words, :py:obj:`~.cuMemUnmap` cannot unmap a sub-range of
    an address range mapped by :py:obj:`~.cuMemCreate` /
    :py:obj:`~.cuMemMap`. Any backing memory allocations will be freed if
    there are no existing mappings and there are no unreleased memory
    handles.

    When :py:obj:`~.cuMemUnmap` returns successfully the address range is
    converted to an address reservation and can be used for a future calls
    to :py:obj:`~.cuMemMap`. Any new mapping to this virtual address will
    need to have access granted through :py:obj:`~.cuMemSetAccess`, as all
    mappings start with no accessibility setup.

    Parameters
    ----------
    ptr : :py:obj:`~.CUdeviceptr`
        Starting address for the virtual address range to unmap
    size : size_t
        Size of the virtual address range to unmap

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_PERMITTED`, :py:obj:`~.CUDA_ERROR_NOT_SUPPORTED`

    See Also
    --------
    :py:obj:`~.cuMemCreate`, :py:obj:`~.cuMemAddressReserve`
    """
    cdef cydriver.CUdeviceptr cyptr
    if ptr is None:
        cyptr = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(ptr, (CUdeviceptr,)):
        pptr = int(ptr)
        cyptr = <cydriver.CUdeviceptr><void_ptr>pptr
    else:
        pptr = int(CUdeviceptr(ptr))
        cyptr = <cydriver.CUdeviceptr><void_ptr>pptr
    err = cydriver.cuMemUnmap(cyptr, size)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemSetAccess(ptr, size_t size, desc : Optional[Tuple[CUmemAccessDesc] | List[CUmemAccessDesc]], size_t count):
    """ Set the access flags for each location specified in `desc` for the given virtual address range.

    Given the virtual address range via `ptr` and `size`, and the locations
    in the array given by `desc` and `count`, set the access flags for the
    target locations. The range must be a fully mapped address range
    containing all allocations created by :py:obj:`~.cuMemMap` /
    :py:obj:`~.cuMemCreate`.

    Parameters
    ----------
    ptr : :py:obj:`~.CUdeviceptr`
        Starting address for the virtual address range
    size : size_t
        Length of the virtual address range
    desc : List[:py:obj:`~.CUmemAccessDesc`]
        Array of :py:obj:`~.CUmemAccessDesc` that describe how to change
        the
    count : size_t
        Number of :py:obj:`~.CUmemAccessDesc` in `desc`

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`, :py:obj:`~.CUDA_ERROR_NOT_SUPPORTED`

    See Also
    --------
    :py:obj:`~.cuMemSetAccess`, :py:obj:`~.cuMemCreate`, :py:obj:`~.py`:obj:`~.cuMemMap`
    """
    desc = [] if desc is None else desc
    if not all(isinstance(_x, (CUmemAccessDesc,)) for _x in desc):
        raise TypeError("Argument 'desc' is not instance of type (expected Tuple[cydriver.CUmemAccessDesc,] or List[cydriver.CUmemAccessDesc,]")
    cdef cydriver.CUdeviceptr cyptr
    if ptr is None:
        cyptr = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(ptr, (CUdeviceptr,)):
        pptr = int(ptr)
        cyptr = <cydriver.CUdeviceptr><void_ptr>pptr
    else:
        pptr = int(CUdeviceptr(ptr))
        cyptr = <cydriver.CUdeviceptr><void_ptr>pptr
    cdef cydriver.CUmemAccessDesc* cydesc = NULL
    if len(desc) > 0:
        cydesc = <cydriver.CUmemAccessDesc*> calloc(len(desc), sizeof(cydriver.CUmemAccessDesc))
        if cydesc is NULL:
            raise MemoryError('Failed to allocate length x size memory: ' + str(len(desc)) + 'x' + str(sizeof(cydriver.CUmemAccessDesc)))
        for idx in range(len(desc)):
            string.memcpy(&cydesc[idx], (<CUmemAccessDesc>desc[idx])._ptr, sizeof(cydriver.CUmemAccessDesc))
    if count > <size_t>len(desc): raise RuntimeError("List is too small: " + str(len(desc)) + " < " + str(count))
    err = cydriver.cuMemSetAccess(cyptr, size, (<CUmemAccessDesc>desc[0])._ptr if len(desc) == 1 else cydesc, count)
    if cydesc is not NULL:
        free(cydesc)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemGetAccess(location : Optional[CUmemLocation], ptr):
    """ Get the access `flags` set for the given `location` and `ptr`.

    Parameters
    ----------
    location : :py:obj:`~.CUmemLocation`
        Location in which to check the flags for
    ptr : :py:obj:`~.CUdeviceptr`
        Address in which to check the access flags for

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_PERMITTED`, :py:obj:`~.CUDA_ERROR_NOT_SUPPORTED`
    flags : unsigned long long
        Flags set for this location

    See Also
    --------
    :py:obj:`~.cuMemSetAccess`
    """
    cdef cydriver.CUdeviceptr cyptr
    if ptr is None:
        cyptr = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(ptr, (CUdeviceptr,)):
        pptr = int(ptr)
        cyptr = <cydriver.CUdeviceptr><void_ptr>pptr
    else:
        pptr = int(CUdeviceptr(ptr))
        cyptr = <cydriver.CUdeviceptr><void_ptr>pptr
    cdef unsigned long long flags = 0
    cdef cydriver.CUmemLocation* cylocation_ptr = location._ptr if location != None else NULL
    err = cydriver.cuMemGetAccess(&flags, cylocation_ptr, cyptr)
    return (CUresult(err), flags)

@cython.embedsignature(True)
def cuMemExportToShareableHandle(handle, handleType not None : CUmemAllocationHandleType, unsigned long long flags):
    """ Exports an allocation to a requested shareable handle type.

    Given a CUDA memory handle, create a shareable memory allocation handle
    that can be used to share the memory with other processes. The
    recipient process can convert the shareable handle back into a CUDA
    memory handle using :py:obj:`~.cuMemImportFromShareableHandle` and map
    it with :py:obj:`~.cuMemMap`. The implementation of what this handle is
    and how it can be transferred is defined by the requested handle type
    in `handleType`

    Once all shareable handles are closed and the allocation is released,
    the allocated memory referenced will be released back to the OS and
    uses of the CUDA handle afterward will lead to undefined behavior.

    This API can also be used in conjunction with other APIs (e.g. Vulkan,
    OpenGL) that support importing memory from the shareable type

    Parameters
    ----------
    handle : :py:obj:`~.CUmemGenericAllocationHandle`
        CUDA handle for the memory allocation
    handleType : :py:obj:`~.CUmemAllocationHandleType`
        Type of shareable handle requested (defines type and size of the
        `shareableHandle` output parameter)
    flags : unsigned long long
        Reserved, must be zero

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_PERMITTED`, :py:obj:`~.CUDA_ERROR_NOT_SUPPORTED`
    shareableHandle : Any
        Pointer to the location in which to store the requested handle type

    See Also
    --------
    :py:obj:`~.cuMemImportFromShareableHandle`
    """
    cdef cydriver.CUmemGenericAllocationHandle cyhandle
    if handle is None:
        cyhandle = <cydriver.CUmemGenericAllocationHandle><void_ptr>0
    elif isinstance(handle, (CUmemGenericAllocationHandle,)):
        phandle = int(handle)
        cyhandle = <cydriver.CUmemGenericAllocationHandle><void_ptr>phandle
    else:
        phandle = int(CUmemGenericAllocationHandle(handle))
        cyhandle = <cydriver.CUmemGenericAllocationHandle><void_ptr>phandle
    cdef utils.HelperCUmemAllocationHandleType cyshareableHandle = utils.HelperCUmemAllocationHandleType(handleType)
    cdef void* cyshareableHandle_ptr = <void*><void_ptr>cyshareableHandle.cptr
    cdef cydriver.CUmemAllocationHandleType cyhandleType = handleType.value
    err = cydriver.cuMemExportToShareableHandle(cyshareableHandle_ptr, cyhandle, cyhandleType, flags)
    return (CUresult(err), cyshareableHandle.pyObj())

@cython.embedsignature(True)
def cuMemImportFromShareableHandle(osHandle, shHandleType not None : CUmemAllocationHandleType):
    """ Imports an allocation from a requested shareable handle type.

    If the current process cannot support the memory described by this
    shareable handle, this API will error as CUDA_ERROR_NOT_SUPPORTED.

    Parameters
    ----------
    osHandle : Any
        Shareable Handle representing the memory allocation that is to be
        imported.
    shHandleType : :py:obj:`~.CUmemAllocationHandleType`
        handle type of the exported handle
        :py:obj:`~.CUmemAllocationHandleType`.

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_PERMITTED`, :py:obj:`~.CUDA_ERROR_NOT_SUPPORTED`
    handle : :py:obj:`~.CUmemGenericAllocationHandle`
        CUDA Memory handle for the memory allocation.

    See Also
    --------
    :py:obj:`~.cuMemExportToShareableHandle`, :py:obj:`~.cuMemMap`, :py:obj:`~.cuMemRelease`

    Notes
    -----
    Importing shareable handles exported from some graphics APIs(VUlkan, OpenGL, etc) created on devices under an SLI group may not be supported, and thus this API will return CUDA_ERROR_NOT_SUPPORTED. There is no guarantee that the contents of `handle` will be the same CUDA memory handle for the same given OS shareable handle, or the same underlying allocation.
    """
    cdef CUmemGenericAllocationHandle handle = CUmemGenericAllocationHandle()
    cyosHandle = utils.HelperInputVoidPtr(osHandle)
    cdef void* cyosHandle_ptr = <void*><void_ptr>cyosHandle.cptr
    cdef cydriver.CUmemAllocationHandleType cyshHandleType = shHandleType.value
    err = cydriver.cuMemImportFromShareableHandle(<cydriver.CUmemGenericAllocationHandle*>handle._ptr, cyosHandle_ptr, cyshHandleType)
    return (CUresult(err), handle)

@cython.embedsignature(True)
def cuMemGetAllocationGranularity(prop : Optional[CUmemAllocationProp], option not None : CUmemAllocationGranularity_flags):
    """ Calculates either the minimal or recommended granularity.

    Calculates either the minimal or recommended granularity for a given
    allocation specification and returns it in granularity. This
    granularity can be used as a multiple for alignment, size, or address
    mapping.

    Parameters
    ----------
    prop : :py:obj:`~.CUmemAllocationProp`
        Property for which to determine the granularity for
    option : :py:obj:`~.CUmemAllocationGranularity_flags`
        Determines which granularity to return

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_PERMITTED`, :py:obj:`~.CUDA_ERROR_NOT_SUPPORTED`
    granularity : int
        Returned granularity.

    See Also
    --------
    :py:obj:`~.cuMemCreate`, :py:obj:`~.cuMemMap`
    """
    cdef size_t granularity = 0
    cdef cydriver.CUmemAllocationProp* cyprop_ptr = prop._ptr if prop != None else NULL
    cdef cydriver.CUmemAllocationGranularity_flags cyoption = option.value
    err = cydriver.cuMemGetAllocationGranularity(&granularity, cyprop_ptr, cyoption)
    return (CUresult(err), granularity)

@cython.embedsignature(True)
def cuMemGetAllocationPropertiesFromHandle(handle):
    """ Retrieve the contents of the property structure defining properties for this handle.

    Parameters
    ----------
    handle : :py:obj:`~.CUmemGenericAllocationHandle`
        Handle which to perform the query on

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_PERMITTED`, :py:obj:`~.CUDA_ERROR_NOT_SUPPORTED`
    prop : :py:obj:`~.CUmemAllocationProp`
        Pointer to a properties structure which will hold the information
        about this handle

    See Also
    --------
    :py:obj:`~.cuMemCreate`, :py:obj:`~.cuMemImportFromShareableHandle`
    """
    cdef cydriver.CUmemGenericAllocationHandle cyhandle
    if handle is None:
        cyhandle = <cydriver.CUmemGenericAllocationHandle><void_ptr>0
    elif isinstance(handle, (CUmemGenericAllocationHandle,)):
        phandle = int(handle)
        cyhandle = <cydriver.CUmemGenericAllocationHandle><void_ptr>phandle
    else:
        phandle = int(CUmemGenericAllocationHandle(handle))
        cyhandle = <cydriver.CUmemGenericAllocationHandle><void_ptr>phandle
    cdef CUmemAllocationProp prop = CUmemAllocationProp()
    err = cydriver.cuMemGetAllocationPropertiesFromHandle(<cydriver.CUmemAllocationProp*>prop._ptr, cyhandle)
    return (CUresult(err), prop)

@cython.embedsignature(True)
def cuMemRetainAllocationHandle(addr):
    """ Given an address `addr`, returns the allocation handle of the backing memory allocation.

    The handle is guaranteed to be the same handle value used to map the
    memory. If the address requested is not mapped, the function will fail.
    The returned handle must be released with corresponding number of calls
    to :py:obj:`~.cuMemRelease`.

    Parameters
    ----------
    addr : Any
        Memory address to query, that has been mapped previously.

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_PERMITTED`, :py:obj:`~.CUDA_ERROR_NOT_SUPPORTED`
    handle : :py:obj:`~.CUmemGenericAllocationHandle`
        CUDA Memory handle for the backing memory allocation.

    See Also
    --------
    :py:obj:`~.cuMemCreate`, :py:obj:`~.cuMemRelease`, :py:obj:`~.cuMemMap`

    Notes
    -----
    The address `addr`, can be any address in a range previously mapped by :py:obj:`~.cuMemMap`, and not necessarily the start address.
    """
    cdef CUmemGenericAllocationHandle handle = CUmemGenericAllocationHandle()
    cyaddr = utils.HelperInputVoidPtr(addr)
    cdef void* cyaddr_ptr = <void*><void_ptr>cyaddr.cptr
    err = cydriver.cuMemRetainAllocationHandle(<cydriver.CUmemGenericAllocationHandle*>handle._ptr, cyaddr_ptr)
    return (CUresult(err), handle)

@cython.embedsignature(True)
def cuMemFreeAsync(dptr, hStream):
    """ Frees memory with stream ordered semantics.

    Inserts a free operation into `hStream`. The allocation must not be
    accessed after stream execution reaches the free. After this API
    returns, accessing the memory from any subsequent work launched on the
    GPU or querying its pointer attributes results in undefined behavior.

    Parameters
    ----------
    dptr : :py:obj:`~.CUdeviceptr`
        memory to free
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        The stream establishing the stream ordering contract.

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT` (default stream specified with no current context), :py:obj:`~.CUDA_ERROR_NOT_SUPPORTED`

    Notes
    -----
    During stream capture, this function results in the creation of a free node and must therefore be passed the address of a graph allocation.
    """
    cdef cydriver.CUstream cyhStream
    if hStream is None:
        cyhStream = <cydriver.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    cdef cydriver.CUdeviceptr cydptr
    if dptr is None:
        cydptr = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(dptr, (CUdeviceptr,)):
        pdptr = int(dptr)
        cydptr = <cydriver.CUdeviceptr><void_ptr>pdptr
    else:
        pdptr = int(CUdeviceptr(dptr))
        cydptr = <cydriver.CUdeviceptr><void_ptr>pdptr
    err = cydriver.cuMemFreeAsync(cydptr, cyhStream)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemAllocAsync(size_t bytesize, hStream):
    """ Allocates memory with stream ordered semantics.

    Inserts an allocation operation into `hStream`. A pointer to the
    allocated memory is returned immediately in *dptr. The allocation must
    not be accessed until the the allocation operation completes. The
    allocation comes from the memory pool current to the stream's device.

    Parameters
    ----------
    bytesize : size_t
        Number of bytes to allocate
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        The stream establishing the stream ordering contract and the memory
        pool to allocate from

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT` (default stream specified with no current context), :py:obj:`~.CUDA_ERROR_NOT_SUPPORTED`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`
    dptr : :py:obj:`~.CUdeviceptr`
        Returned device pointer

    See Also
    --------
    :py:obj:`~.cuMemAllocFromPoolAsync`, :py:obj:`~.cuMemFreeAsync`, :py:obj:`~.cuDeviceSetMemPool`, :py:obj:`~.cuDeviceGetDefaultMemPool`, :py:obj:`~.cuDeviceGetMemPool`, :py:obj:`~.cuMemPoolCreate`, :py:obj:`~.cuMemPoolSetAccess`, :py:obj:`~.cuMemPoolSetAttribute`

    Notes
    -----
    The default memory pool of a device contains device memory from that device.

    Basic stream ordering allows future work submitted into the same stream to use the allocation. Stream query, stream synchronize, and CUDA events can be used to guarantee that the allocation operation completes before work submitted in a separate stream runs.

    During stream capture, this function results in the creation of an allocation node. In this case, the allocation is owned by the graph instead of the memory pool. The memory pool's properties are used to set the node's creation parameters.
    """
    cdef cydriver.CUstream cyhStream
    if hStream is None:
        cyhStream = <cydriver.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    cdef CUdeviceptr dptr = CUdeviceptr()
    err = cydriver.cuMemAllocAsync(<cydriver.CUdeviceptr*>dptr._ptr, bytesize, cyhStream)
    return (CUresult(err), dptr)

@cython.embedsignature(True)
def cuMemPoolTrimTo(pool, size_t minBytesToKeep):
    """ Tries to release memory back to the OS.

    Releases memory back to the OS until the pool contains fewer than
    minBytesToKeep reserved bytes, or there is no more memory that the
    allocator can safely release. The allocator cannot release OS
    allocations that back outstanding asynchronous allocations. The OS
    allocations may happen at different granularity from the user
    allocations.

    Parameters
    ----------
    pool : :py:obj:`~.CUmemoryPool` or :py:obj:`~.cudaMemPool_t`
        The memory pool to trim
    minBytesToKeep : size_t
        If the pool has less than minBytesToKeep reserved, the TrimTo
        operation is a no-op. Otherwise the pool will be guaranteed to have
        at least minBytesToKeep bytes reserved after the operation.

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuMemAllocAsync`, :py:obj:`~.cuMemFreeAsync`, :py:obj:`~.cuDeviceGetDefaultMemPool`, :py:obj:`~.cuDeviceGetMemPool`, :py:obj:`~.cuMemPoolCreate`

    Notes
    -----
    : Allocations that have not been freed count as outstanding.

    : Allocations that have been asynchronously freed but whose completion has not been observed on the host (eg. by a synchronize) can count as outstanding.
    """
    cdef cydriver.CUmemoryPool cypool
    if pool is None:
        cypool = <cydriver.CUmemoryPool><void_ptr>0
    elif isinstance(pool, (CUmemoryPool,)):
        ppool = int(pool)
        cypool = <cydriver.CUmemoryPool><void_ptr>ppool
    else:
        ppool = int(CUmemoryPool(pool))
        cypool = <cydriver.CUmemoryPool><void_ptr>ppool
    err = cydriver.cuMemPoolTrimTo(cypool, minBytesToKeep)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemPoolSetAttribute(pool, attr not None : CUmemPool_attribute, value):
    """ Sets attributes of a memory pool.

    Supported attributes are:

    - :py:obj:`~.CU_MEMPOOL_ATTR_RELEASE_THRESHOLD`: (value type =
      cuuint64_t) Amount of reserved memory in bytes to hold onto before
      trying to release memory back to the OS. When more than the release
      threshold bytes of memory are held by the memory pool, the allocator
      will try to release memory back to the OS on the next call to stream,
      event or context synchronize. (default 0)

    - :py:obj:`~.CU_MEMPOOL_ATTR_REUSE_FOLLOW_EVENT_DEPENDENCIES`: (value
      type = int) Allow :py:obj:`~.cuMemAllocAsync` to use memory
      asynchronously freed in another stream as long as a stream ordering
      dependency of the allocating stream on the free action exists. Cuda
      events and null stream interactions can create the required stream
      ordered dependencies. (default enabled)

    - :py:obj:`~.CU_MEMPOOL_ATTR_REUSE_ALLOW_OPPORTUNISTIC`: (value type =
      int) Allow reuse of already completed frees when there is no
      dependency between the free and allocation. (default enabled)

    - :py:obj:`~.CU_MEMPOOL_ATTR_REUSE_ALLOW_INTERNAL_DEPENDENCIES`: (value
      type = int) Allow :py:obj:`~.cuMemAllocAsync` to insert new stream
      dependencies in order to establish the stream ordering required to
      reuse a piece of memory released by :py:obj:`~.cuMemFreeAsync`
      (default enabled).

    - :py:obj:`~.CU_MEMPOOL_ATTR_RESERVED_MEM_HIGH`: (value type =
      cuuint64_t) Reset the high watermark that tracks the amount of
      backing memory that was allocated for the memory pool. It is illegal
      to set this attribute to a non-zero value.

    - :py:obj:`~.CU_MEMPOOL_ATTR_USED_MEM_HIGH`: (value type = cuuint64_t)
      Reset the high watermark that tracks the amount of used memory that
      was allocated for the memory pool.

    Parameters
    ----------
    pool : :py:obj:`~.CUmemoryPool` or :py:obj:`~.cudaMemPool_t`
        The memory pool to modify
    attr : :py:obj:`~.CUmemPool_attribute`
        The attribute to modify
    value : Any
        Pointer to the value to assign

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuMemAllocAsync`, :py:obj:`~.cuMemFreeAsync`, :py:obj:`~.cuDeviceGetDefaultMemPool`, :py:obj:`~.cuDeviceGetMemPool`, :py:obj:`~.cuMemPoolCreate`
    """
    cdef cydriver.CUmemoryPool cypool
    if pool is None:
        cypool = <cydriver.CUmemoryPool><void_ptr>0
    elif isinstance(pool, (CUmemoryPool,)):
        ppool = int(pool)
        cypool = <cydriver.CUmemoryPool><void_ptr>ppool
    else:
        ppool = int(CUmemoryPool(pool))
        cypool = <cydriver.CUmemoryPool><void_ptr>ppool
    cdef cydriver.CUmemPool_attribute cyattr = attr.value
    cdef utils.HelperCUmemPool_attribute cyvalue = utils.HelperCUmemPool_attribute(attr, value, is_getter=False)
    cdef void* cyvalue_ptr = <void*><void_ptr>cyvalue.cptr
    err = cydriver.cuMemPoolSetAttribute(cypool, cyattr, cyvalue_ptr)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemPoolGetAttribute(pool, attr not None : CUmemPool_attribute):
    """ Gets attributes of a memory pool.

    Supported attributes are:

    - :py:obj:`~.CU_MEMPOOL_ATTR_RELEASE_THRESHOLD`: (value type =
      cuuint64_t) Amount of reserved memory in bytes to hold onto before
      trying to release memory back to the OS. When more than the release
      threshold bytes of memory are held by the memory pool, the allocator
      will try to release memory back to the OS on the next call to stream,
      event or context synchronize. (default 0)

    - :py:obj:`~.CU_MEMPOOL_ATTR_REUSE_FOLLOW_EVENT_DEPENDENCIES`: (value
      type = int) Allow :py:obj:`~.cuMemAllocAsync` to use memory
      asynchronously freed in another stream as long as a stream ordering
      dependency of the allocating stream on the free action exists. Cuda
      events and null stream interactions can create the required stream
      ordered dependencies. (default enabled)

    - :py:obj:`~.CU_MEMPOOL_ATTR_REUSE_ALLOW_OPPORTUNISTIC`: (value type =
      int) Allow reuse of already completed frees when there is no
      dependency between the free and allocation. (default enabled)

    - :py:obj:`~.CU_MEMPOOL_ATTR_REUSE_ALLOW_INTERNAL_DEPENDENCIES`: (value
      type = int) Allow :py:obj:`~.cuMemAllocAsync` to insert new stream
      dependencies in order to establish the stream ordering required to
      reuse a piece of memory released by :py:obj:`~.cuMemFreeAsync`
      (default enabled).

    - :py:obj:`~.CU_MEMPOOL_ATTR_RESERVED_MEM_CURRENT`: (value type =
      cuuint64_t) Amount of backing memory currently allocated for the
      mempool

    - :py:obj:`~.CU_MEMPOOL_ATTR_RESERVED_MEM_HIGH`: (value type =
      cuuint64_t) High watermark of backing memory allocated for the
      mempool since the last time it was reset.

    - :py:obj:`~.CU_MEMPOOL_ATTR_USED_MEM_CURRENT`: (value type =
      cuuint64_t) Amount of memory from the pool that is currently in use
      by the application.

    - :py:obj:`~.CU_MEMPOOL_ATTR_USED_MEM_HIGH`: (value type = cuuint64_t)
      High watermark of the amount of memory from the pool that was in use
      by the application.

    Parameters
    ----------
    pool : :py:obj:`~.CUmemoryPool` or :py:obj:`~.cudaMemPool_t`
        The memory pool to get attributes of
    attr : :py:obj:`~.CUmemPool_attribute`
        The attribute to get

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    value : Any
        Retrieved value

    See Also
    --------
    :py:obj:`~.cuMemAllocAsync`, :py:obj:`~.cuMemFreeAsync`, :py:obj:`~.cuDeviceGetDefaultMemPool`, :py:obj:`~.cuDeviceGetMemPool`, :py:obj:`~.cuMemPoolCreate`
    """
    cdef cydriver.CUmemoryPool cypool
    if pool is None:
        cypool = <cydriver.CUmemoryPool><void_ptr>0
    elif isinstance(pool, (CUmemoryPool,)):
        ppool = int(pool)
        cypool = <cydriver.CUmemoryPool><void_ptr>ppool
    else:
        ppool = int(CUmemoryPool(pool))
        cypool = <cydriver.CUmemoryPool><void_ptr>ppool
    cdef cydriver.CUmemPool_attribute cyattr = attr.value
    cdef utils.HelperCUmemPool_attribute cyvalue = utils.HelperCUmemPool_attribute(attr, 0, is_getter=True)
    cdef void* cyvalue_ptr = <void*><void_ptr>cyvalue.cptr
    err = cydriver.cuMemPoolGetAttribute(cypool, cyattr, cyvalue_ptr)
    return (CUresult(err), cyvalue.pyObj())

@cython.embedsignature(True)
def cuMemPoolSetAccess(pool, map : Optional[Tuple[CUmemAccessDesc] | List[CUmemAccessDesc]], size_t count):
    """ Controls visibility of pools between devices.

    Parameters
    ----------
    pool : :py:obj:`~.CUmemoryPool` or :py:obj:`~.cudaMemPool_t`
        The pool being modified
    map : List[:py:obj:`~.CUmemAccessDesc`]
        Array of access descriptors. Each descriptor instructs the access
        to enable for a single gpu.
    count : size_t
        Number of descriptors in the map array.

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuMemAllocAsync`, :py:obj:`~.cuMemFreeAsync`, :py:obj:`~.cuDeviceGetDefaultMemPool`, :py:obj:`~.cuDeviceGetMemPool`, :py:obj:`~.cuMemPoolCreate`
    """
    map = [] if map is None else map
    if not all(isinstance(_x, (CUmemAccessDesc,)) for _x in map):
        raise TypeError("Argument 'map' is not instance of type (expected Tuple[cydriver.CUmemAccessDesc,] or List[cydriver.CUmemAccessDesc,]")
    cdef cydriver.CUmemoryPool cypool
    if pool is None:
        cypool = <cydriver.CUmemoryPool><void_ptr>0
    elif isinstance(pool, (CUmemoryPool,)):
        ppool = int(pool)
        cypool = <cydriver.CUmemoryPool><void_ptr>ppool
    else:
        ppool = int(CUmemoryPool(pool))
        cypool = <cydriver.CUmemoryPool><void_ptr>ppool
    cdef cydriver.CUmemAccessDesc* cymap = NULL
    if len(map) > 0:
        cymap = <cydriver.CUmemAccessDesc*> calloc(len(map), sizeof(cydriver.CUmemAccessDesc))
        if cymap is NULL:
            raise MemoryError('Failed to allocate length x size memory: ' + str(len(map)) + 'x' + str(sizeof(cydriver.CUmemAccessDesc)))
        for idx in range(len(map)):
            string.memcpy(&cymap[idx], (<CUmemAccessDesc>map[idx])._ptr, sizeof(cydriver.CUmemAccessDesc))
    if count > <size_t>len(map): raise RuntimeError("List is too small: " + str(len(map)) + " < " + str(count))
    err = cydriver.cuMemPoolSetAccess(cypool, (<CUmemAccessDesc>map[0])._ptr if len(map) == 1 else cymap, count)
    if cymap is not NULL:
        free(cymap)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemPoolGetAccess(memPool, location : Optional[CUmemLocation]):
    """ Returns the accessibility of a pool from a device.

    Returns the accessibility of the pool's memory from the specified
    location.

    Parameters
    ----------
    memPool : :py:obj:`~.CUmemoryPool` or :py:obj:`~.cudaMemPool_t`
        the pool being queried
    location : :py:obj:`~.CUmemLocation`
        the location accessing the pool

    Returns
    -------
    CUresult

    flags : :py:obj:`~.CUmemAccess_flags`
        the accessibility of the pool from the specified location

    See Also
    --------
    :py:obj:`~.cuMemAllocAsync`, :py:obj:`~.cuMemFreeAsync`, :py:obj:`~.cuDeviceGetDefaultMemPool`, :py:obj:`~.cuDeviceGetMemPool`, :py:obj:`~.cuMemPoolCreate`
    """
    cdef cydriver.CUmemoryPool cymemPool
    if memPool is None:
        cymemPool = <cydriver.CUmemoryPool><void_ptr>0
    elif isinstance(memPool, (CUmemoryPool,)):
        pmemPool = int(memPool)
        cymemPool = <cydriver.CUmemoryPool><void_ptr>pmemPool
    else:
        pmemPool = int(CUmemoryPool(memPool))
        cymemPool = <cydriver.CUmemoryPool><void_ptr>pmemPool
    cdef cydriver.CUmemAccess_flags flags
    cdef cydriver.CUmemLocation* cylocation_ptr = location._ptr if location != None else NULL
    err = cydriver.cuMemPoolGetAccess(&flags, cymemPool, cylocation_ptr)
    return (CUresult(err), CUmemAccess_flags(flags))

@cython.embedsignature(True)
def cuMemPoolCreate(poolProps : Optional[CUmemPoolProps]):
    """ Creates a memory pool.

    Creates a CUDA memory pool and returns the handle in `pool`. The
    `poolProps` determines the properties of the pool such as the backing
    device and IPC capabilities.

    By default, the pool's memory will be accessible from the device it is
    allocated on.

    Parameters
    ----------
    poolProps : :py:obj:`~.CUmemPoolProps`
        None

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`, :py:obj:`~.CUDA_ERROR_NOT_SUPPORTED`
    pool : :py:obj:`~.CUmemoryPool`
        None

    See Also
    --------
    :py:obj:`~.cuDeviceSetMemPool`, :py:obj:`~.cuDeviceGetMemPool`, :py:obj:`~.cuDeviceGetDefaultMemPool`, :py:obj:`~.cuMemAllocFromPoolAsync`, :py:obj:`~.cuMemPoolExportToShareableHandle`

    Notes
    -----
    Specifying CU_MEM_HANDLE_TYPE_NONE creates a memory pool that will not support IPC.
    """
    cdef CUmemoryPool pool = CUmemoryPool()
    cdef cydriver.CUmemPoolProps* cypoolProps_ptr = poolProps._ptr if poolProps != None else NULL
    err = cydriver.cuMemPoolCreate(<cydriver.CUmemoryPool*>pool._ptr, cypoolProps_ptr)
    return (CUresult(err), pool)

@cython.embedsignature(True)
def cuMemPoolDestroy(pool):
    """ Destroys the specified memory pool.

    If any pointers obtained from this pool haven't been freed or the pool
    has free operations that haven't completed when
    :py:obj:`~.cuMemPoolDestroy` is invoked, the function will return
    immediately and the resources associated with the pool will be released
    automatically once there are no more outstanding allocations.

    Destroying the current mempool of a device sets the default mempool of
    that device as the current mempool for that device.

    Parameters
    ----------
    pool : :py:obj:`~.CUmemoryPool` or :py:obj:`~.cudaMemPool_t`
        None

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuMemFreeAsync`, :py:obj:`~.cuDeviceSetMemPool`, :py:obj:`~.cuDeviceGetMemPool`, :py:obj:`~.cuDeviceGetDefaultMemPool`, :py:obj:`~.cuMemPoolCreate`

    Notes
    -----
    A device's default memory pool cannot be destroyed.
    """
    cdef cydriver.CUmemoryPool cypool
    if pool is None:
        cypool = <cydriver.CUmemoryPool><void_ptr>0
    elif isinstance(pool, (CUmemoryPool,)):
        ppool = int(pool)
        cypool = <cydriver.CUmemoryPool><void_ptr>ppool
    else:
        ppool = int(CUmemoryPool(pool))
        cypool = <cydriver.CUmemoryPool><void_ptr>ppool
    err = cydriver.cuMemPoolDestroy(cypool)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemAllocFromPoolAsync(size_t bytesize, pool, hStream):
    """ Allocates memory from a specified pool with stream ordered semantics.

    Inserts an allocation operation into `hStream`. A pointer to the
    allocated memory is returned immediately in *dptr. The allocation must
    not be accessed until the the allocation operation completes. The
    allocation comes from the specified memory pool.

    Parameters
    ----------
    bytesize : size_t
        Number of bytes to allocate
    pool : :py:obj:`~.CUmemoryPool` or :py:obj:`~.cudaMemPool_t`
        The pool to allocate from
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        The stream establishing the stream ordering semantic

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT` (default stream specified with no current context), :py:obj:`~.CUDA_ERROR_NOT_SUPPORTED`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`
    dptr : :py:obj:`~.CUdeviceptr`
        Returned device pointer

    See Also
    --------
    :py:obj:`~.cuMemAllocAsync`, :py:obj:`~.cuMemFreeAsync`, :py:obj:`~.cuDeviceGetDefaultMemPool`, :py:obj:`~.cuDeviceGetMemPool`, :py:obj:`~.cuMemPoolCreate`, :py:obj:`~.cuMemPoolSetAccess`, :py:obj:`~.cuMemPoolSetAttribute`

    Notes
    -----
    During stream capture, this function results in the creation of an allocation node. In this case, the allocation is owned by the graph instead of the memory pool. The memory pool's properties are used to set the node's creation parameters.
    """
    cdef cydriver.CUstream cyhStream
    if hStream is None:
        cyhStream = <cydriver.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    cdef cydriver.CUmemoryPool cypool
    if pool is None:
        cypool = <cydriver.CUmemoryPool><void_ptr>0
    elif isinstance(pool, (CUmemoryPool,)):
        ppool = int(pool)
        cypool = <cydriver.CUmemoryPool><void_ptr>ppool
    else:
        ppool = int(CUmemoryPool(pool))
        cypool = <cydriver.CUmemoryPool><void_ptr>ppool
    cdef CUdeviceptr dptr = CUdeviceptr()
    err = cydriver.cuMemAllocFromPoolAsync(<cydriver.CUdeviceptr*>dptr._ptr, bytesize, cypool, cyhStream)
    return (CUresult(err), dptr)

@cython.embedsignature(True)
def cuMemPoolExportToShareableHandle(pool, handleType not None : CUmemAllocationHandleType, unsigned long long flags):
    """ Exports a memory pool to the requested handle type.

    Given an IPC capable mempool, create an OS handle to share the pool
    with another process. A recipient process can convert the shareable
    handle into a mempool with
    :py:obj:`~.cuMemPoolImportFromShareableHandle`. Individual pointers can
    then be shared with the :py:obj:`~.cuMemPoolExportPointer` and
    :py:obj:`~.cuMemPoolImportPointer` APIs. The implementation of what the
    shareable handle is and how it can be transferred is defined by the
    requested handle type.

    Parameters
    ----------
    pool : :py:obj:`~.CUmemoryPool` or :py:obj:`~.cudaMemPool_t`
        pool to export
    handleType : :py:obj:`~.CUmemAllocationHandleType`
        the type of handle to create
    flags : unsigned long long
        must be 0

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`
    handle_out : Any
        Returned OS handle

    See Also
    --------
    :py:obj:`~.cuMemPoolImportFromShareableHandle`, :py:obj:`~.cuMemPoolExportPointer`, :py:obj:`~.cuMemPoolImportPointer`, :py:obj:`~.cuMemAllocAsync`, :py:obj:`~.cuMemFreeAsync`, :py:obj:`~.cuDeviceGetDefaultMemPool`, :py:obj:`~.cuDeviceGetMemPool`, :py:obj:`~.cuMemPoolCreate`, :py:obj:`~.cuMemPoolSetAccess`, :py:obj:`~.cuMemPoolSetAttribute`

    Notes
    -----
    : To create an IPC capable mempool, create a mempool with a CUmemAllocationHandleType other than CU_MEM_HANDLE_TYPE_NONE.
    """
    cdef cydriver.CUmemoryPool cypool
    if pool is None:
        cypool = <cydriver.CUmemoryPool><void_ptr>0
    elif isinstance(pool, (CUmemoryPool,)):
        ppool = int(pool)
        cypool = <cydriver.CUmemoryPool><void_ptr>ppool
    else:
        ppool = int(CUmemoryPool(pool))
        cypool = <cydriver.CUmemoryPool><void_ptr>ppool
    cdef utils.HelperCUmemAllocationHandleType cyhandle_out = utils.HelperCUmemAllocationHandleType(handleType)
    cdef void* cyhandle_out_ptr = <void*><void_ptr>cyhandle_out.cptr
    cdef cydriver.CUmemAllocationHandleType cyhandleType = handleType.value
    err = cydriver.cuMemPoolExportToShareableHandle(cyhandle_out_ptr, cypool, cyhandleType, flags)
    return (CUresult(err), cyhandle_out.pyObj())

@cython.embedsignature(True)
def cuMemPoolImportFromShareableHandle(handle, handleType not None : CUmemAllocationHandleType, unsigned long long flags):
    """ imports a memory pool from a shared handle.

    Specific allocations can be imported from the imported pool with
    cuMemPoolImportPointer.

    Parameters
    ----------
    handle : Any
        OS handle of the pool to open
    handleType : :py:obj:`~.CUmemAllocationHandleType`
        The type of handle being imported
    flags : unsigned long long
        must be 0

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`
    pool_out : :py:obj:`~.CUmemoryPool`
        Returned memory pool

    See Also
    --------
    :py:obj:`~.cuMemPoolExportToShareableHandle`, :py:obj:`~.cuMemPoolExportPointer`, :py:obj:`~.cuMemPoolImportPointer`

    Notes
    -----
    Imported memory pools do not support creating new allocations. As such imported memory pools may not be used in cuDeviceSetMemPool or :py:obj:`~.cuMemAllocFromPoolAsync` calls.
    """
    cdef CUmemoryPool pool_out = CUmemoryPool()
    cyhandle = utils.HelperInputVoidPtr(handle)
    cdef void* cyhandle_ptr = <void*><void_ptr>cyhandle.cptr
    cdef cydriver.CUmemAllocationHandleType cyhandleType = handleType.value
    err = cydriver.cuMemPoolImportFromShareableHandle(<cydriver.CUmemoryPool*>pool_out._ptr, cyhandle_ptr, cyhandleType, flags)
    return (CUresult(err), pool_out)

@cython.embedsignature(True)
def cuMemPoolExportPointer(ptr):
    """ Export data to share a memory pool allocation between processes.

    Constructs `shareData_out` for sharing a specific allocation from an
    already shared memory pool. The recipient process can import the
    allocation with the :py:obj:`~.cuMemPoolImportPointer` api. The data is
    not a handle and may be shared through any IPC mechanism.

    Parameters
    ----------
    ptr : :py:obj:`~.CUdeviceptr`
        pointer to memory being exported

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`
    shareData_out : :py:obj:`~.CUmemPoolPtrExportData`
        Returned export data

    See Also
    --------
    :py:obj:`~.cuMemPoolExportToShareableHandle`, :py:obj:`~.cuMemPoolImportFromShareableHandle`, :py:obj:`~.cuMemPoolImportPointer`
    """
    cdef cydriver.CUdeviceptr cyptr
    if ptr is None:
        cyptr = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(ptr, (CUdeviceptr,)):
        pptr = int(ptr)
        cyptr = <cydriver.CUdeviceptr><void_ptr>pptr
    else:
        pptr = int(CUdeviceptr(ptr))
        cyptr = <cydriver.CUdeviceptr><void_ptr>pptr
    cdef CUmemPoolPtrExportData shareData_out = CUmemPoolPtrExportData()
    err = cydriver.cuMemPoolExportPointer(<cydriver.CUmemPoolPtrExportData*>shareData_out._ptr, cyptr)
    return (CUresult(err), shareData_out)

@cython.embedsignature(True)
def cuMemPoolImportPointer(pool, shareData : Optional[CUmemPoolPtrExportData]):
    """ Import a memory pool allocation from another process.

    Returns in `ptr_out` a pointer to the imported memory. The imported
    memory must not be accessed before the allocation operation completes
    in the exporting process. The imported memory must be freed from all
    importing processes before being freed in the exporting process. The
    pointer may be freed with cuMemFree or cuMemFreeAsync. If
    cuMemFreeAsync is used, the free must be completed on the importing
    process before the free operation on the exporting process.

    Parameters
    ----------
    pool : :py:obj:`~.CUmemoryPool` or :py:obj:`~.cudaMemPool_t`
        pool from which to import
    shareData : :py:obj:`~.CUmemPoolPtrExportData`
        data specifying the memory to import

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`
    ptr_out : :py:obj:`~.CUdeviceptr`
        pointer to imported memory

    See Also
    --------
    :py:obj:`~.cuMemPoolExportToShareableHandle`, :py:obj:`~.cuMemPoolImportFromShareableHandle`, :py:obj:`~.cuMemPoolExportPointer`

    Notes
    -----
    The cuMemFreeAsync api may be used in the exporting process before the cuMemFreeAsync operation completes in its stream as long as the cuMemFreeAsync in the exporting process specifies a stream with a stream dependency on the importing process's cuMemFreeAsync.
    """
    cdef cydriver.CUmemoryPool cypool
    if pool is None:
        cypool = <cydriver.CUmemoryPool><void_ptr>0
    elif isinstance(pool, (CUmemoryPool,)):
        ppool = int(pool)
        cypool = <cydriver.CUmemoryPool><void_ptr>ppool
    else:
        ppool = int(CUmemoryPool(pool))
        cypool = <cydriver.CUmemoryPool><void_ptr>ppool
    cdef CUdeviceptr ptr_out = CUdeviceptr()
    cdef cydriver.CUmemPoolPtrExportData* cyshareData_ptr = shareData._ptr if shareData != None else NULL
    err = cydriver.cuMemPoolImportPointer(<cydriver.CUdeviceptr*>ptr_out._ptr, cypool, cyshareData_ptr)
    return (CUresult(err), ptr_out)

@cython.embedsignature(True)
def cuPointerGetAttribute(attribute not None : CUpointer_attribute, ptr):
    """ Returns information about a pointer.

    The supported attributes are:

    - :py:obj:`~.CU_POINTER_ATTRIBUTE_CONTEXT`:

    - Returns in `*data` the :py:obj:`~.CUcontext` in which `ptr` was
      allocated or registered. The type of `data` must be
      :py:obj:`~.CUcontext` *.

    - If `ptr` was not allocated by, mapped by, or registered with a
      :py:obj:`~.CUcontext` which uses unified virtual addressing then
      :py:obj:`~.CUDA_ERROR_INVALID_VALUE` is returned.

    - :py:obj:`~.CU_POINTER_ATTRIBUTE_MEMORY_TYPE`:

    - Returns in `*data` the physical memory type of the memory that `ptr`
      addresses as a :py:obj:`~.CUmemorytype` enumerated value. The type of
      `data` must be unsigned int.

    - If `ptr` addresses device memory then `*data` is set to
      :py:obj:`~.CU_MEMORYTYPE_DEVICE`. The particular :py:obj:`~.CUdevice`
      on which the memory resides is the :py:obj:`~.CUdevice` of the
      :py:obj:`~.CUcontext` returned by the
      :py:obj:`~.CU_POINTER_ATTRIBUTE_CONTEXT` attribute of `ptr`.

    - If `ptr` addresses host memory then `*data` is set to
      :py:obj:`~.CU_MEMORYTYPE_HOST`.

    - If `ptr` was not allocated by, mapped by, or registered with a
      :py:obj:`~.CUcontext` which uses unified virtual addressing then
      :py:obj:`~.CUDA_ERROR_INVALID_VALUE` is returned.

    - If the current :py:obj:`~.CUcontext` does not support unified virtual
      addressing then :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT` is returned.

    - :py:obj:`~.CU_POINTER_ATTRIBUTE_DEVICE_POINTER`:

    - Returns in `*data` the device pointer value through which `ptr` may
      be accessed by kernels running in the current :py:obj:`~.CUcontext`.
      The type of `data` must be CUdeviceptr *.

    - If there exists no device pointer value through which kernels running
      in the current :py:obj:`~.CUcontext` may access `ptr` then
      :py:obj:`~.CUDA_ERROR_INVALID_VALUE` is returned.

    - If there is no current :py:obj:`~.CUcontext` then
      :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT` is returned.

    - Except in the exceptional disjoint addressing cases discussed below,
      the value returned in `*data` will equal the input value `ptr`.

    - :py:obj:`~.CU_POINTER_ATTRIBUTE_HOST_POINTER`:

    - Returns in `*data` the host pointer value through which `ptr` may be
      accessed by by the host program. The type of `data` must be void **.
      If there exists no host pointer value through which the host program
      may directly access `ptr` then :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
      is returned.

    - Except in the exceptional disjoint addressing cases discussed below,
      the value returned in `*data` will equal the input value `ptr`.

    - :py:obj:`~.CU_POINTER_ATTRIBUTE_P2P_TOKENS`:

    - Returns in `*data` two tokens for use with the nv-p2p.h Linux kernel
      interface. `data` must be a struct of type
      CUDA_POINTER_ATTRIBUTE_P2P_TOKENS.

    - `ptr` must be a pointer to memory obtained from
      :py:obj:`~.py`:obj:`~.cuMemAlloc()`. Note that p2pToken and
      vaSpaceToken are only valid for the lifetime of the source
      allocation. A subsequent allocation at the same address may return
      completely different tokens. Querying this attribute has a side
      effect of setting the attribute
      :py:obj:`~.CU_POINTER_ATTRIBUTE_SYNC_MEMOPS` for the region of memory
      that `ptr` points to.

    - :py:obj:`~.CU_POINTER_ATTRIBUTE_SYNC_MEMOPS`:

    - A boolean attribute which when set, ensures that synchronous memory
      operations initiated on the region of memory that `ptr` points to
      will always synchronize. See further documentation in the section
      titled "API synchronization behavior" to learn more about cases when
      synchronous memory operations can exhibit asynchronous behavior.

    - :py:obj:`~.CU_POINTER_ATTRIBUTE_BUFFER_ID`:

    - Returns in `*data` a buffer ID which is guaranteed to be unique
      within the process. `data` must point to an unsigned long long.

    - `ptr` must be a pointer to memory obtained from a CUDA memory
      allocation API. Every memory allocation from any of the CUDA memory
      allocation APIs will have a unique ID over a process lifetime.
      Subsequent allocations do not reuse IDs from previous freed
      allocations. IDs are only unique within a single process.

    - :py:obj:`~.CU_POINTER_ATTRIBUTE_IS_MANAGED`:

    - Returns in `*data` a boolean that indicates whether the pointer
      points to managed memory or not.

    - If `ptr` is not a valid CUDA pointer then
      :py:obj:`~.CUDA_ERROR_INVALID_VALUE` is returned.

    - :py:obj:`~.CU_POINTER_ATTRIBUTE_DEVICE_ORDINAL`:

    - Returns in `*data` an integer representing a device ordinal of a
      device against which the memory was allocated or registered.

    - :py:obj:`~.CU_POINTER_ATTRIBUTE_IS_LEGACY_CUDA_IPC_CAPABLE`:

    - Returns in `*data` a boolean that indicates if this pointer maps to
      an allocation that is suitable for :py:obj:`~.cudaIpcGetMemHandle`.

    - :py:obj:`~.CU_POINTER_ATTRIBUTE_RANGE_START_ADDR`:

    - Returns in `*data` the starting address for the allocation referenced
      by the device pointer `ptr`. Note that this is not necessarily the
      address of the mapped region, but the address of the mappable address
      range `ptr` references (e.g. from :py:obj:`~.cuMemAddressReserve`).

    - :py:obj:`~.CU_POINTER_ATTRIBUTE_RANGE_SIZE`:

    - Returns in `*data` the size for the allocation referenced by the
      device pointer `ptr`. Note that this is not necessarily the size of
      the mapped region, but the size of the mappable address range `ptr`
      references (e.g. from :py:obj:`~.cuMemAddressReserve`). To retrieve
      the size of the mapped region, see :py:obj:`~.cuMemGetAddressRange`

    - :py:obj:`~.CU_POINTER_ATTRIBUTE_MAPPED`:

    - Returns in `*data` a boolean that indicates if this pointer is in a
      valid address range that is mapped to a backing allocation.

    - :py:obj:`~.CU_POINTER_ATTRIBUTE_ALLOWED_HANDLE_TYPES`:

    - Returns a bitmask of the allowed handle types for an allocation that
      may be passed to :py:obj:`~.cuMemExportToShareableHandle`.

    - :py:obj:`~.CU_POINTER_ATTRIBUTE_MEMPOOL_HANDLE`:

    - Returns in `*data` the handle to the mempool that the allocation was
      obtained from.

    Note that for most allocations in the unified virtual address space the
    host and device pointer for accessing the allocation will be the same.
    The exceptions to this are

    - user memory registered using :py:obj:`~.cuMemHostRegister`

    - host memory allocated using :py:obj:`~.cuMemHostAlloc` with the
      :py:obj:`~.CU_MEMHOSTALLOC_WRITECOMBINED` flag For these types of
      allocation there will exist separate, disjoint host and device
      addresses for accessing the allocation. In particular

    - The host address will correspond to an invalid unmapped device
      address (which will result in an exception if accessed from the
      device)

    - The device address will correspond to an invalid unmapped host
      address (which will result in an exception if accessed from the
      host). For these types of allocations, querying
      :py:obj:`~.CU_POINTER_ATTRIBUTE_HOST_POINTER` and
      :py:obj:`~.CU_POINTER_ATTRIBUTE_DEVICE_POINTER` may be used to
      retrieve the host and device addresses from either address.

    Parameters
    ----------
    attribute : :py:obj:`~.CUpointer_attribute`
        Pointer attribute to query
    ptr : :py:obj:`~.CUdeviceptr`
        Pointer

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`
    data : Any
        Returned pointer attribute value

    See Also
    --------
    :py:obj:`~.cuPointerSetAttribute`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostRegister`, :py:obj:`~.cuMemHostUnregister`, :py:obj:`~.cudaPointerGetAttributes`
    """
    cdef cydriver.CUdeviceptr cyptr
    if ptr is None:
        cyptr = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(ptr, (CUdeviceptr,)):
        pptr = int(ptr)
        cyptr = <cydriver.CUdeviceptr><void_ptr>pptr
    else:
        pptr = int(CUdeviceptr(ptr))
        cyptr = <cydriver.CUdeviceptr><void_ptr>pptr
    cdef utils.HelperCUpointer_attribute cydata = utils.HelperCUpointer_attribute(attribute, 0, is_getter=True)
    cdef void* cydata_ptr = <void*><void_ptr>cydata.cptr
    cdef cydriver.CUpointer_attribute cyattribute = attribute.value
    err = cydriver.cuPointerGetAttribute(cydata_ptr, cyattribute, cyptr)
    return (CUresult(err), cydata.pyObj())

@cython.embedsignature(True)
def cuMemPrefetchAsync(devPtr, size_t count, dstDevice, hStream):
    """ Prefetches memory to the specified destination device.

    Prefetches memory to the specified destination device. `devPtr` is the
    base device pointer of the memory to be prefetched and `dstDevice` is
    the destination device. `count` specifies the number of bytes to copy.
    `hStream` is the stream in which the operation is enqueued. The memory
    range must refer to managed memory allocated via
    :py:obj:`~.cuMemAllocManaged` or declared via managed variables.

    Passing in CU_DEVICE_CPU for `dstDevice` will prefetch the data to host
    memory. If `dstDevice` is a GPU, then the device attribute
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_CONCURRENT_MANAGED_ACCESS` must be non-
    zero. Additionally, `hStream` must be associated with a device that has
    a non-zero value for the device attribute
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_CONCURRENT_MANAGED_ACCESS`.

    The start address and end address of the memory range will be rounded
    down and rounded up respectively to be aligned to CPU page size before
    the prefetch operation is enqueued in the stream.

    If no physical memory has been allocated for this region, then this
    memory region will be populated and mapped on the destination device.
    If there's insufficient memory to prefetch the desired region, the
    Unified Memory driver may evict pages from other
    :py:obj:`~.cuMemAllocManaged` allocations to host memory in order to
    make room. Device memory allocated using :py:obj:`~.cuMemAlloc` or
    :py:obj:`~.cuArrayCreate` will not be evicted.

    By default, any mappings to the previous location of the migrated pages
    are removed and mappings for the new location are only setup on
    `dstDevice`. The exact behavior however also depends on the settings
    applied to this memory range via :py:obj:`~.cuMemAdvise` as described
    below:

    If :py:obj:`~.CU_MEM_ADVISE_SET_READ_MOSTLY` was set on any subset of
    this memory range, then that subset will create a read-only copy of the
    pages on `dstDevice`.

    If :py:obj:`~.CU_MEM_ADVISE_SET_PREFERRED_LOCATION` was called on any
    subset of this memory range, then the pages will be migrated to
    `dstDevice` even if `dstDevice` is not the preferred location of any
    pages in the memory range.

    If :py:obj:`~.CU_MEM_ADVISE_SET_ACCESSED_BY` was called on any subset
    of this memory range, then mappings to those pages from all the
    appropriate processors are updated to refer to the new location if
    establishing such a mapping is possible. Otherwise, those mappings are
    cleared.

    Note that this API is not required for functionality and only serves to
    improve performance by allowing the application to migrate data to a
    suitable location before it is accessed. Memory accesses to this range
    are always coherent and are allowed even when the data is actively
    being migrated.

    Note that this function is asynchronous with respect to the host and
    all work on other devices.

    Parameters
    ----------
    devPtr : :py:obj:`~.CUdeviceptr`
        Pointer to be prefetched
    count : size_t
        Size in bytes
    dstDevice : :py:obj:`~.CUdevice`
        Destination device to prefetch to
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream to enqueue prefetch operation

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`

    See Also
    --------
    :py:obj:`~.cuMemcpy`, :py:obj:`~.cuMemcpyPeer`, :py:obj:`~.cuMemcpyAsync`, :py:obj:`~.cuMemcpy3DPeerAsync`, :py:obj:`~.cuMemAdvise`, :py:obj:`~.cudaMemPrefetchAsync`
    """
    cdef cydriver.CUstream cyhStream
    if hStream is None:
        cyhStream = <cydriver.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    cdef cydriver.CUdevice cydstDevice
    if dstDevice is None:
        cydstDevice = <cydriver.CUdevice>0
    elif isinstance(dstDevice, (CUdevice,)):
        pdstDevice = int(dstDevice)
        cydstDevice = <cydriver.CUdevice>pdstDevice
    else:
        pdstDevice = int(CUdevice(dstDevice))
        cydstDevice = <cydriver.CUdevice>pdstDevice
    cdef cydriver.CUdeviceptr cydevPtr
    if devPtr is None:
        cydevPtr = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(devPtr, (CUdeviceptr,)):
        pdevPtr = int(devPtr)
        cydevPtr = <cydriver.CUdeviceptr><void_ptr>pdevPtr
    else:
        pdevPtr = int(CUdeviceptr(devPtr))
        cydevPtr = <cydriver.CUdeviceptr><void_ptr>pdevPtr
    err = cydriver.cuMemPrefetchAsync(cydevPtr, count, cydstDevice, cyhStream)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemAdvise(devPtr, size_t count, advice not None : CUmem_advise, device):
    """ Advise about the usage of a given memory range.

    Advise the Unified Memory subsystem about the usage pattern for the
    memory range starting at `devPtr` with a size of `count` bytes. The
    start address and end address of the memory range will be rounded down
    and rounded up respectively to be aligned to CPU page size before the
    advice is applied. The memory range must refer to managed memory
    allocated via :py:obj:`~.cuMemAllocManaged` or declared via managed
    variables. The memory range could also refer to system-allocated
    pageable memory provided it represents a valid, host-accessible region
    of memory and all additional constraints imposed by `advice` as
    outlined below are also satisfied. Specifying an invalid system-
    allocated pageable memory range results in an error being returned.

    The `advice` parameter can take the following values:

    - :py:obj:`~.CU_MEM_ADVISE_SET_READ_MOSTLY`: This implies that the data
      is mostly going to be read from and only occasionally written to. Any
      read accesses from any processor to this region will create a read-
      only copy of at least the accessed pages in that processor's memory.
      Additionally, if :py:obj:`~.cuMemPrefetchAsync` is called on this
      region, it will create a read-only copy of the data on the
      destination processor. If any processor writes to this region, all
      copies of the corresponding page will be invalidated except for the
      one where the write occurred. The `device` argument is ignored for
      this advice. Note that for a page to be read-duplicated, the
      accessing processor must either be the CPU or a GPU that has a non-
      zero value for the device attribute
      :py:obj:`~.CU_DEVICE_ATTRIBUTE_CONCURRENT_MANAGED_ACCESS`. Also, if a
      context is created on a device that does not have the device
      attribute :py:obj:`~.CU_DEVICE_ATTRIBUTE_CONCURRENT_MANAGED_ACCESS`
      set, then read-duplication will not occur until all such contexts are
      destroyed. If the memory region refers to valid system-allocated
      pageable memory, then the accessing device must have a non-zero value
      for the device attribute
      :py:obj:`~.CU_DEVICE_ATTRIBUTE_PAGEABLE_MEMORY_ACCESS` for a read-
      only copy to be created on that device. Note however that if the
      accessing device also has a non-zero value for the device attribute
      :py:obj:`~.CU_DEVICE_ATTRIBUTE_PAGEABLE_MEMORY_ACCESS_USES_HOST_PAGE_TABLES`,
      then setting this advice will not create a read-only copy when that
      device accesses this memory region.

    - :py:obj:`~.CU_MEM_ADVISE_UNSET_READ_MOSTLY`: Undoes the effect of
      :py:obj:`~.CU_MEM_ADVISE_SET_READ_MOSTLY` and also prevents the
      Unified Memory driver from attempting heuristic read-duplication on
      the memory range. Any read-duplicated copies of the data will be
      collapsed into a single copy. The location for the collapsed copy
      will be the preferred location if the page has a preferred location
      and one of the read-duplicated copies was resident at that location.
      Otherwise, the location chosen is arbitrary.

    - :py:obj:`~.CU_MEM_ADVISE_SET_PREFERRED_LOCATION`: This advice sets
      the preferred location for the data to be the memory belonging to
      `device`. Passing in CU_DEVICE_CPU for `device` sets the preferred
      location as host memory. If `device` is a GPU, then it must have a
      non-zero value for the device attribute
      :py:obj:`~.CU_DEVICE_ATTRIBUTE_CONCURRENT_MANAGED_ACCESS`. Setting
      the preferred location does not cause data to migrate to that
      location immediately. Instead, it guides the migration policy when a
      fault occurs on that memory region. If the data is already in its
      preferred location and the faulting processor can establish a mapping
      without requiring the data to be migrated, then data migration will
      be avoided. On the other hand, if the data is not in its preferred
      location or if a direct mapping cannot be established, then it will
      be migrated to the processor accessing it. It is important to note
      that setting the preferred location does not prevent data prefetching
      done using :py:obj:`~.cuMemPrefetchAsync`. Having a preferred
      location can override the page thrash detection and resolution logic
      in the Unified Memory driver. Normally, if a page is detected to be
      constantly thrashing between for example host and device memory, the
      page may eventually be pinned to host memory by the Unified Memory
      driver. But if the preferred location is set as device memory, then
      the page will continue to thrash indefinitely. If
      :py:obj:`~.CU_MEM_ADVISE_SET_READ_MOSTLY` is also set on this memory
      region or any subset of it, then the policies associated with that
      advice will override the policies of this advice, unless read
      accesses from `device` will not result in a read-only copy being
      created on that device as outlined in description for the advice
      :py:obj:`~.CU_MEM_ADVISE_SET_READ_MOSTLY`. If the memory region
      refers to valid system-allocated pageable memory, then `device` must
      have a non-zero value for the device attribute
      :py:obj:`~.CU_DEVICE_ATTRIBUTE_PAGEABLE_MEMORY_ACCESS`. Additionally,
      if `device` has a non-zero value for the device attribute
      :py:obj:`~.CU_DEVICE_ATTRIBUTE_PAGEABLE_MEMORY_ACCESS_USES_HOST_PAGE_TABLES`,
      then this call has no effect. Note however that this behavior may
      change in the future.

    - :py:obj:`~.CU_MEM_ADVISE_UNSET_PREFERRED_LOCATION`: Undoes the effect
      of :py:obj:`~.CU_MEM_ADVISE_SET_PREFERRED_LOCATION` and changes the
      preferred location to none.

    - :py:obj:`~.CU_MEM_ADVISE_SET_ACCESSED_BY`: This advice implies that
      the data will be accessed by `device`. Passing in
      :py:obj:`~.CU_DEVICE_CPU` for `device` will set the advice for the
      CPU. If `device` is a GPU, then the device attribute
      :py:obj:`~.CU_DEVICE_ATTRIBUTE_CONCURRENT_MANAGED_ACCESS` must be
      non-zero. This advice does not cause data migration and has no impact
      on the location of the data per se. Instead, it causes the data to
      always be mapped in the specified processor's page tables, as long as
      the location of the data permits a mapping to be established. If the
      data gets migrated for any reason, the mappings are updated
      accordingly. This advice is recommended in scenarios where data
      locality is not important, but avoiding faults is. Consider for
      example a system containing multiple GPUs with peer-to-peer access
      enabled, where the data located on one GPU is occasionally accessed
      by peer GPUs. In such scenarios, migrating data over to the other
      GPUs is not as important because the accesses are infrequent and the
      overhead of migration may be too high. But preventing faults can
      still help improve performance, and so having a mapping set up in
      advance is useful. Note that on CPU access of this data, the data may
      be migrated to host memory because the CPU typically cannot access
      device memory directly. Any GPU that had the
      :py:obj:`~.CU_MEM_ADVISE_SET_ACCESSED_BY` flag set for this data will
      now have its mapping updated to point to the page in host memory. If
      :py:obj:`~.CU_MEM_ADVISE_SET_READ_MOSTLY` is also set on this memory
      region or any subset of it, then the policies associated with that
      advice will override the policies of this advice. Additionally, if
      the preferred location of this memory region or any subset of it is
      also `device`, then the policies associated with
      :py:obj:`~.CU_MEM_ADVISE_SET_PREFERRED_LOCATION` will override the
      policies of this advice. If the memory region refers to valid system-
      allocated pageable memory, then `device` must have a non-zero value
      for the device attribute
      :py:obj:`~.CU_DEVICE_ATTRIBUTE_PAGEABLE_MEMORY_ACCESS`. Additionally,
      if `device` has a non-zero value for the device attribute
      :py:obj:`~.CU_DEVICE_ATTRIBUTE_PAGEABLE_MEMORY_ACCESS_USES_HOST_PAGE_TABLES`,
      then this call has no effect.

    - :py:obj:`~.CU_MEM_ADVISE_UNSET_ACCESSED_BY`: Undoes the effect of
      :py:obj:`~.CU_MEM_ADVISE_SET_ACCESSED_BY`. Any mappings to the data
      from `device` may be removed at any time causing accesses to result
      in non-fatal page faults. If the memory region refers to valid
      system-allocated pageable memory, then `device` must have a non-zero
      value for the device attribute
      :py:obj:`~.CU_DEVICE_ATTRIBUTE_PAGEABLE_MEMORY_ACCESS`. Additionally,
      if `device` has a non-zero value for the device attribute
      :py:obj:`~.CU_DEVICE_ATTRIBUTE_PAGEABLE_MEMORY_ACCESS_USES_HOST_PAGE_TABLES`,
      then this call has no effect.

    Parameters
    ----------
    devPtr : :py:obj:`~.CUdeviceptr`
        Pointer to memory to set the advice for
    count : size_t
        Size in bytes of the memory range
    advice : :py:obj:`~.CUmem_advise`
        Advice to be applied for the specified memory range
    device : :py:obj:`~.CUdevice`
        Device to apply the advice for

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`

    See Also
    --------
    :py:obj:`~.cuMemcpy`, :py:obj:`~.cuMemcpyPeer`, :py:obj:`~.cuMemcpyAsync`, :py:obj:`~.cuMemcpy3DPeerAsync`, :py:obj:`~.cuMemPrefetchAsync`, :py:obj:`~.cudaMemAdvise`
    """
    cdef cydriver.CUdevice cydevice
    if device is None:
        cydevice = <cydriver.CUdevice>0
    elif isinstance(device, (CUdevice,)):
        pdevice = int(device)
        cydevice = <cydriver.CUdevice>pdevice
    else:
        pdevice = int(CUdevice(device))
        cydevice = <cydriver.CUdevice>pdevice
    cdef cydriver.CUdeviceptr cydevPtr
    if devPtr is None:
        cydevPtr = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(devPtr, (CUdeviceptr,)):
        pdevPtr = int(devPtr)
        cydevPtr = <cydriver.CUdeviceptr><void_ptr>pdevPtr
    else:
        pdevPtr = int(CUdeviceptr(devPtr))
        cydevPtr = <cydriver.CUdeviceptr><void_ptr>pdevPtr
    cdef cydriver.CUmem_advise cyadvice = advice.value
    err = cydriver.cuMemAdvise(cydevPtr, count, cyadvice, cydevice)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuMemRangeGetAttribute(size_t dataSize, attribute not None : CUmem_range_attribute, devPtr, size_t count):
    """ Query an attribute of a given memory range.

    Query an attribute about the memory range starting at `devPtr` with a
    size of `count` bytes. The memory range must refer to managed memory
    allocated via :py:obj:`~.cuMemAllocManaged` or declared via managed
    variables.

    The `attribute` parameter can take the following values:

    - :py:obj:`~.CU_MEM_RANGE_ATTRIBUTE_READ_MOSTLY`: If this attribute is
      specified, `data` will be interpreted as a 32-bit integer, and
      `dataSize` must be 4. The result returned will be 1 if all pages in
      the given memory range have read-duplication enabled, or 0 otherwise.

    - :py:obj:`~.CU_MEM_RANGE_ATTRIBUTE_PREFERRED_LOCATION`: If this
      attribute is specified, `data` will be interpreted as a 32-bit
      integer, and `dataSize` must be 4. The result returned will be a GPU
      device id if all pages in the memory range have that GPU as their
      preferred location, or it will be CU_DEVICE_CPU if all pages in the
      memory range have the CPU as their preferred location, or it will be
      CU_DEVICE_INVALID if either all the pages don't have the same
      preferred location or some of the pages don't have a preferred
      location at all. Note that the actual location of the pages in the
      memory range at the time of the query may be different from the
      preferred location.

    - :py:obj:`~.CU_MEM_RANGE_ATTRIBUTE_ACCESSED_BY`: If this attribute is
      specified, `data` will be interpreted as an array of 32-bit integers,
      and `dataSize` must be a non-zero multiple of 4. The result returned
      will be a list of device ids that had
      :py:obj:`~.CU_MEM_ADVISE_SET_ACCESSED_BY` set for that entire memory
      range. If any device does not have that advice set for the entire
      memory range, that device will not be included. If `data` is larger
      than the number of devices that have that advice set for that memory
      range, CU_DEVICE_INVALID will be returned in all the extra space
      provided. For ex., if `dataSize` is 12 (i.e. `data` has 3 elements)
      and only device 0 has the advice set, then the result returned will
      be { 0, CU_DEVICE_INVALID, CU_DEVICE_INVALID }. If `data` is smaller
      than the number of devices that have that advice set, then only as
      many devices will be returned as can fit in the array. There is no
      guarantee on which specific devices will be returned, however.

    - :py:obj:`~.CU_MEM_RANGE_ATTRIBUTE_LAST_PREFETCH_LOCATION`: If this
      attribute is specified, `data` will be interpreted as a 32-bit
      integer, and `dataSize` must be 4. The result returned will be the
      last location to which all pages in the memory range were prefetched
      explicitly via :py:obj:`~.cuMemPrefetchAsync`. This will either be a
      GPU id or CU_DEVICE_CPU depending on whether the last location for
      prefetch was a GPU or the CPU respectively. If any page in the memory
      range was never explicitly prefetched or if all pages were not
      prefetched to the same location, CU_DEVICE_INVALID will be returned.
      Note that this simply returns the last location that the applicaton
      requested to prefetch the memory range to. It gives no indication as
      to whether the prefetch operation to that location has completed or
      even begun.

    Parameters
    ----------
    dataSize : size_t
        Array containing the size of data
    attribute : :py:obj:`~.CUmem_range_attribute`
        The attribute to query
    devPtr : :py:obj:`~.CUdeviceptr`
        Start of the range to query
    count : size_t
        Size of the range to query

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`
    data : Any
        A pointers to a memory location where the result of each attribute
        query will be written to.

    See Also
    --------
    :py:obj:`~.cuMemRangeGetAttributes`, :py:obj:`~.cuMemPrefetchAsync`, :py:obj:`~.cuMemAdvise`, :py:obj:`~.cudaMemRangeGetAttribute`
    """
    cdef cydriver.CUdeviceptr cydevPtr
    if devPtr is None:
        cydevPtr = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(devPtr, (CUdeviceptr,)):
        pdevPtr = int(devPtr)
        cydevPtr = <cydriver.CUdeviceptr><void_ptr>pdevPtr
    else:
        pdevPtr = int(CUdeviceptr(devPtr))
        cydevPtr = <cydriver.CUdeviceptr><void_ptr>pdevPtr
    cdef utils.HelperCUmem_range_attribute cydata = utils.HelperCUmem_range_attribute(attribute, dataSize)
    cdef void* cydata_ptr = <void*><void_ptr>cydata.cptr
    cdef cydriver.CUmem_range_attribute cyattribute = attribute.value
    err = cydriver.cuMemRangeGetAttribute(cydata_ptr, dataSize, cyattribute, cydevPtr, count)
    return (CUresult(err), cydata.pyObj())

@cython.embedsignature(True)
def cuMemRangeGetAttributes(dataSizes : Tuple[int] | List[int], attributes : Optional[Tuple[CUmem_range_attribute] | List[CUmem_range_attribute]], size_t numAttributes, devPtr, size_t count):
    """ Query attributes of a given memory range.

    Query attributes of the memory range starting at `devPtr` with a size
    of `count` bytes. The memory range must refer to managed memory
    allocated via :py:obj:`~.cuMemAllocManaged` or declared via managed
    variables. The `attributes` array will be interpreted to have
    `numAttributes` entries. The `dataSizes` array will also be interpreted
    to have `numAttributes` entries. The results of the query will be
    stored in `data`.

    The list of supported attributes are given below. Please refer to
    :py:obj:`~.cuMemRangeGetAttribute` for attribute descriptions and
    restrictions.

    - :py:obj:`~.CU_MEM_RANGE_ATTRIBUTE_READ_MOSTLY`

    - :py:obj:`~.CU_MEM_RANGE_ATTRIBUTE_PREFERRED_LOCATION`

    - :py:obj:`~.CU_MEM_RANGE_ATTRIBUTE_ACCESSED_BY`

    - :py:obj:`~.CU_MEM_RANGE_ATTRIBUTE_LAST_PREFETCH_LOCATION`

    Parameters
    ----------
    dataSizes : List[int]
        Array containing the sizes of each result
    attributes : List[:py:obj:`~.CUmem_range_attribute`]
        An array of attributes to query (numAttributes and the number of
        attributes in this array should match)
    numAttributes : size_t
        Number of attributes to query
    devPtr : :py:obj:`~.CUdeviceptr`
        Start of the range to query
    count : size_t
        Size of the range to query

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`
    data : List[Any]
        A two-dimensional array containing pointers to memory locations
        where the result of each attribute query will be written to.

    See Also
    --------
    :py:obj:`~.cuMemRangeGetAttribute`, :py:obj:`~.cuMemAdvise`, :py:obj:`~.cuMemPrefetchAsync`, :py:obj:`~.cudaMemRangeGetAttributes`
    """
    cdef cydriver.CUdeviceptr cydevPtr
    if devPtr is None:
        cydevPtr = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(devPtr, (CUdeviceptr,)):
        pdevPtr = int(devPtr)
        cydevPtr = <cydriver.CUdeviceptr><void_ptr>pdevPtr
    else:
        pdevPtr = int(CUdeviceptr(devPtr))
        cydevPtr = <cydriver.CUdeviceptr><void_ptr>pdevPtr
    attributes = [] if attributes is None else attributes
    if not all(isinstance(_x, (CUmem_range_attribute)) for _x in attributes):
        raise TypeError("Argument 'attributes' is not instance of type (expected Tuple[cydriver.CUmem_range_attribute] or List[cydriver.CUmem_range_attribute]")
    if not all(isinstance(_x, (int)) for _x in dataSizes):
        raise TypeError("Argument 'dataSizes' is not instance of type (expected Tuple[int] or List[int]")
    pylist = [utils.HelperCUmem_range_attribute(pyattributes, pydataSizes) for (pyattributes, pydataSizes) in zip(attributes, dataSizes)]
    cdef utils.InputVoidPtrPtrHelper voidStarHelperdata = utils.InputVoidPtrPtrHelper(pylist)
    cdef void** cyvoidStarHelper_ptr = <void**><void_ptr>voidStarHelperdata.cptr
    cdef vector[size_t] cydataSizes = dataSizes
    cdef vector[cydriver.CUmem_range_attribute] cyattributes = [pyattributes.value for pyattributes in (attributes)]
    if numAttributes > <size_t>len(dataSizes): raise RuntimeError("List is too small: " + str(len(dataSizes)) + " < " + str(numAttributes))
    if numAttributes > <size_t>len(attributes): raise RuntimeError("List is too small: " + str(len(attributes)) + " < " + str(numAttributes))
    err = cydriver.cuMemRangeGetAttributes(cyvoidStarHelper_ptr, cydataSizes.data(), cyattributes.data(), numAttributes, cydevPtr, count)
    return (CUresult(err), [obj.pyObj() for obj in pylist])

@cython.embedsignature(True)
def cuPointerSetAttribute(value, attribute not None : CUpointer_attribute, ptr):
    """ Set attributes on a previously allocated memory region.

    The supported attributes are:

    - :py:obj:`~.CU_POINTER_ATTRIBUTE_SYNC_MEMOPS`:

    - A boolean attribute that can either be set (1) or unset (0). When
      set, the region of memory that `ptr` points to is guaranteed to
      always synchronize memory operations that are synchronous. If there
      are some previously initiated synchronous memory operations that are
      pending when this attribute is set, the function does not return
      until those memory operations are complete. See further documentation
      in the section titled "API synchronization behavior" to learn more
      about cases when synchronous memory operations can exhibit
      asynchronous behavior. `value` will be considered as a pointer to an
      unsigned integer to which this attribute is to be set.

    Parameters
    ----------
    value : Any
        Pointer to memory containing the value to be set
    attribute : :py:obj:`~.CUpointer_attribute`
        Pointer attribute to set
    ptr : :py:obj:`~.CUdeviceptr`
        Pointer to a memory region allocated using CUDA memory allocation
        APIs

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`

    See Also
    --------
    :py:obj:`~.cuPointerGetAttribute`, :py:obj:`~.cuPointerGetAttributes`, :py:obj:`~.cuMemAlloc`, :py:obj:`~.cuMemFree`, :py:obj:`~.cuMemAllocHost`, :py:obj:`~.cuMemFreeHost`, :py:obj:`~.cuMemHostAlloc`, :py:obj:`~.cuMemHostRegister`, :py:obj:`~.cuMemHostUnregister`
    """
    cdef cydriver.CUdeviceptr cyptr
    if ptr is None:
        cyptr = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(ptr, (CUdeviceptr,)):
        pptr = int(ptr)
        cyptr = <cydriver.CUdeviceptr><void_ptr>pptr
    else:
        pptr = int(CUdeviceptr(ptr))
        cyptr = <cydriver.CUdeviceptr><void_ptr>pptr
    cdef utils.HelperCUpointer_attribute cyvalue = utils.HelperCUpointer_attribute(attribute, value, is_getter=False)
    cdef void* cyvalue_ptr = <void*><void_ptr>cyvalue.cptr
    cdef cydriver.CUpointer_attribute cyattribute = attribute.value
    err = cydriver.cuPointerSetAttribute(cyvalue_ptr, cyattribute, cyptr)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuPointerGetAttributes(unsigned int numAttributes, attributes : Optional[Tuple[CUpointer_attribute] | List[CUpointer_attribute]], ptr):
    """ Returns information about a pointer.

    The supported attributes are (refer to
    :py:obj:`~.cuPointerGetAttribute` for attribute descriptions and
    restrictions):

    - :py:obj:`~.CU_POINTER_ATTRIBUTE_CONTEXT`

    - :py:obj:`~.CU_POINTER_ATTRIBUTE_MEMORY_TYPE`

    - :py:obj:`~.CU_POINTER_ATTRIBUTE_DEVICE_POINTER`

    - :py:obj:`~.CU_POINTER_ATTRIBUTE_HOST_POINTER`

    - :py:obj:`~.CU_POINTER_ATTRIBUTE_SYNC_MEMOPS`

    - :py:obj:`~.CU_POINTER_ATTRIBUTE_BUFFER_ID`

    - :py:obj:`~.CU_POINTER_ATTRIBUTE_IS_MANAGED`

    - :py:obj:`~.CU_POINTER_ATTRIBUTE_DEVICE_ORDINAL`

    - :py:obj:`~.CU_POINTER_ATTRIBUTE_RANGE_START_ADDR`

    - :py:obj:`~.CU_POINTER_ATTRIBUTE_RANGE_SIZE`

    - :py:obj:`~.CU_POINTER_ATTRIBUTE_MAPPED`

    - :py:obj:`~.CU_POINTER_ATTRIBUTE_IS_LEGACY_CUDA_IPC_CAPABLE`

    - :py:obj:`~.CU_POINTER_ATTRIBUTE_ALLOWED_HANDLE_TYPES`

    - :py:obj:`~.CU_POINTER_ATTRIBUTE_MEMPOOL_HANDLE`

    Unlike :py:obj:`~.cuPointerGetAttribute`, this function will not return
    an error when the `ptr` encountered is not a valid CUDA pointer.
    Instead, the attributes are assigned default NULL values and
    CUDA_SUCCESS is returned.

    If `ptr` was not allocated by, mapped by, or registered with a
    :py:obj:`~.CUcontext` which uses UVA (Unified Virtual Addressing),
    :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT` is returned.

    Parameters
    ----------
    numAttributes : unsigned int
        Number of attributes to query
    attributes : List[:py:obj:`~.CUpointer_attribute`]
        An array of attributes to query (numAttributes and the number of
        attributes in this array should match)
    ptr : :py:obj:`~.CUdeviceptr`
        Pointer to query

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_DEVICE`
    data : List[Any]
        A two-dimensional array containing pointers to memory locations
        where the result of each attribute query will be written to.

    See Also
    --------
    :py:obj:`~.cuPointerGetAttribute`, :py:obj:`~.cuPointerSetAttribute`, :py:obj:`~.cudaPointerGetAttributes`
    """
    cdef cydriver.CUdeviceptr cyptr
    if ptr is None:
        cyptr = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(ptr, (CUdeviceptr,)):
        pptr = int(ptr)
        cyptr = <cydriver.CUdeviceptr><void_ptr>pptr
    else:
        pptr = int(CUdeviceptr(ptr))
        cyptr = <cydriver.CUdeviceptr><void_ptr>pptr
    attributes = [] if attributes is None else attributes
    if not all(isinstance(_x, (CUpointer_attribute)) for _x in attributes):
        raise TypeError("Argument 'attributes' is not instance of type (expected Tuple[cydriver.CUpointer_attribute] or List[cydriver.CUpointer_attribute]")
    if numAttributes > len(attributes): raise RuntimeError("List is too small: " + str(len(attributes)) + " < " + str(numAttributes))
    cdef vector[cydriver.CUpointer_attribute] cyattributes = [pyattributes.value for pyattributes in (attributes)]
    pylist = [utils.HelperCUpointer_attribute(pyattributes, 0, is_getter=True) for pyattributes in attributes]
    cdef utils.InputVoidPtrPtrHelper voidStarHelperdata = utils.InputVoidPtrPtrHelper(pylist)
    cdef void** cyvoidStarHelper_ptr = <void**><void_ptr>voidStarHelperdata.cptr
    err = cydriver.cuPointerGetAttributes(numAttributes, cyattributes.data(), cyvoidStarHelper_ptr, cyptr)
    return (CUresult(err), [obj.pyObj() for obj in pylist])

@cython.embedsignature(True)
def cuStreamCreate(unsigned int Flags):
    """ Create a stream.

    Creates a stream and returns a handle in `phStream`. The `Flags`
    argument determines behaviors of the stream.

    Valid values for `Flags` are:

    - :py:obj:`~.CU_STREAM_DEFAULT`: Default stream creation flag.

    - :py:obj:`~.CU_STREAM_NON_BLOCKING`: Specifies that work running in
      the created stream may run concurrently with work in stream 0 (the
      NULL stream), and that the created stream should perform no implicit
      synchronization with stream 0.

    Parameters
    ----------
    Flags : unsigned int
        Parameters for stream creation

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`
    phStream : :py:obj:`~.CUstream`
        Returned newly created stream

    See Also
    --------
    :py:obj:`~.cuStreamDestroy`, :py:obj:`~.cuStreamCreateWithPriority`, :py:obj:`~.cuStreamGetPriority`, :py:obj:`~.cuStreamGetFlags`, :py:obj:`~.cuStreamWaitEvent`, :py:obj:`~.cuStreamQuery`, :py:obj:`~.cuStreamSynchronize`, :py:obj:`~.cuStreamAddCallback`, :py:obj:`~.cudaStreamCreate`, :py:obj:`~.cudaStreamCreateWithFlags`
    """
    cdef CUstream phStream = CUstream()
    err = cydriver.cuStreamCreate(<cydriver.CUstream*>phStream._ptr, Flags)
    return (CUresult(err), phStream)

@cython.embedsignature(True)
def cuStreamCreateWithPriority(unsigned int flags, int priority):
    """ Create a stream with the given priority.

    Creates a stream with the specified priority and returns a handle in
    `phStream`. This API alters the scheduler priority of work in the
    stream. Work in a higher priority stream may preempt work already
    executing in a low priority stream.

    `priority` follows a convention where lower numbers represent higher
    priorities. '0' represents default priority. The range of meaningful
    numerical priorities can be queried using
    :py:obj:`~.cuCtxGetStreamPriorityRange`. If the specified priority is
    outside the numerical range returned by
    :py:obj:`~.cuCtxGetStreamPriorityRange`, it will automatically be
    clamped to the lowest or the highest number in the range.

    Parameters
    ----------
    flags : unsigned int
        Flags for stream creation. See :py:obj:`~.cuStreamCreate` for a
        list of valid flags
    priority : int
        Stream priority. Lower numbers represent higher priorities. See
        :py:obj:`~.cuCtxGetStreamPriorityRange` for more information about
        meaningful stream priorities that can be passed.

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`
    phStream : :py:obj:`~.CUstream`
        Returned newly created stream

    See Also
    --------
    :py:obj:`~.cuStreamDestroy`, :py:obj:`~.cuStreamCreate`, :py:obj:`~.cuStreamGetPriority`, :py:obj:`~.cuCtxGetStreamPriorityRange`, :py:obj:`~.cuStreamGetFlags`, :py:obj:`~.cuStreamWaitEvent`, :py:obj:`~.cuStreamQuery`, :py:obj:`~.cuStreamSynchronize`, :py:obj:`~.cuStreamAddCallback`, :py:obj:`~.cudaStreamCreateWithPriority`

    Notes
    -----
    Stream priorities are supported only on GPUs with compute capability 3.5 or higher.

    In the current implementation, only compute kernels launched in priority streams are affected by the stream's priority. Stream priorities have no effect on host-to-device and device-to-host memory operations.
    """
    cdef CUstream phStream = CUstream()
    err = cydriver.cuStreamCreateWithPriority(<cydriver.CUstream*>phStream._ptr, flags, priority)
    return (CUresult(err), phStream)

@cython.embedsignature(True)
def cuStreamGetPriority(hStream):
    """ Query the priority of a given stream.

    Query the priority of a stream created using :py:obj:`~.cuStreamCreate`
    or :py:obj:`~.cuStreamCreateWithPriority` and return the priority in
    `priority`. Note that if the stream was created with a priority outside
    the numerical range returned by
    :py:obj:`~.cuCtxGetStreamPriorityRange`, this function returns the
    clamped priority. See :py:obj:`~.cuStreamCreateWithPriority` for
    details about priority clamping.

    Parameters
    ----------
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Handle to the stream to be queried

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`
    priority : int
        Pointer to a signed integer in which the stream's priority is
        returned

    See Also
    --------
    :py:obj:`~.cuStreamDestroy`, :py:obj:`~.cuStreamCreate`, :py:obj:`~.cuStreamCreateWithPriority`, :py:obj:`~.cuCtxGetStreamPriorityRange`, :py:obj:`~.cuStreamGetFlags`, :py:obj:`~.cudaStreamGetPriority`
    """
    cdef cydriver.CUstream cyhStream
    if hStream is None:
        cyhStream = <cydriver.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    cdef int priority = 0
    err = cydriver.cuStreamGetPriority(cyhStream, &priority)
    return (CUresult(err), priority)

@cython.embedsignature(True)
def cuStreamGetFlags(hStream):
    """ Query the flags of a given stream.

    Query the flags of a stream created using :py:obj:`~.cuStreamCreate` or
    :py:obj:`~.cuStreamCreateWithPriority` and return the flags in `flags`.

    Parameters
    ----------
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Handle to the stream to be queried

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`
    flags : unsigned int
        Pointer to an unsigned integer in which the stream's flags are
        returned The value returned in `flags` is a logical 'OR' of all
        flags that were used while creating this stream. See
        :py:obj:`~.cuStreamCreate` for the list of valid flags

    See Also
    --------
    :py:obj:`~.cuStreamDestroy`, :py:obj:`~.cuStreamCreate`, :py:obj:`~.cuStreamGetPriority`, :py:obj:`~.cudaStreamGetFlags`
    """
    cdef cydriver.CUstream cyhStream
    if hStream is None:
        cyhStream = <cydriver.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    cdef unsigned int flags = 0
    err = cydriver.cuStreamGetFlags(cyhStream, &flags)
    return (CUresult(err), flags)

@cython.embedsignature(True)
def cuStreamGetCtx(hStream):
    """ Query the context associated with a stream.

    Returns the CUDA context that the stream is associated with.

    The stream handle `hStream` can refer to any of the following:

    - a stream created via any of the CUDA driver APIs such as
      :py:obj:`~.cuStreamCreate` and
      :py:obj:`~.cuStreamCreateWithPriority`, or their runtime API
      equivalents such as :py:obj:`~.cudaStreamCreate`,
      :py:obj:`~.cudaStreamCreateWithFlags` and
      :py:obj:`~.cudaStreamCreateWithPriority`. The returned context is the
      context that was active in the calling thread when the stream was
      created. Passing an invalid handle will result in undefined behavior.

    - any of the special streams such as the NULL stream,
      :py:obj:`~.CU_STREAM_LEGACY` and :py:obj:`~.CU_STREAM_PER_THREAD`.
      The runtime API equivalents of these are also accepted, which are
      NULL, :py:obj:`~.cudaStreamLegacy` and
      :py:obj:`~.cudaStreamPerThread` respectively. Specifying any of the
      special handles will return the context current to the calling
      thread. If no context is current to the calling thread,
      :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT` is returned.

    Parameters
    ----------
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Handle to the stream to be queried

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`,
    pctx : :py:obj:`~.CUcontext`
        Returned context associated with the stream

    See Also
    --------
    :py:obj:`~.cuStreamDestroy`, :py:obj:`~.cuStreamCreateWithPriority`, :py:obj:`~.cuStreamGetPriority`, :py:obj:`~.cuStreamGetFlags`, :py:obj:`~.cuStreamWaitEvent`, :py:obj:`~.cuStreamQuery`, :py:obj:`~.cuStreamSynchronize`, :py:obj:`~.cuStreamAddCallback`, :py:obj:`~.cudaStreamCreate`, :py:obj:`~.cudaStreamCreateWithFlags`
    """
    cdef cydriver.CUstream cyhStream
    if hStream is None:
        cyhStream = <cydriver.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    cdef CUcontext pctx = CUcontext()
    err = cydriver.cuStreamGetCtx(cyhStream, <cydriver.CUcontext*>pctx._ptr)
    return (CUresult(err), pctx)

@cython.embedsignature(True)
def cuStreamWaitEvent(hStream, hEvent, unsigned int Flags):
    """ Make a compute stream wait on an event.

    Makes all future work submitted to `hStream` wait for all work captured
    in `hEvent`. See :py:obj:`~.cuEventRecord()` for details on what is
    captured by an event. The synchronization will be performed efficiently
    on the device when applicable. `hEvent` may be from a different context
    or device than `hStream`.

    flags include:

    - :py:obj:`~.CU_EVENT_WAIT_DEFAULT`: Default event creation flag.

    - :py:obj:`~.CU_EVENT_WAIT_EXTERNAL`: Event is captured in the graph as
      an external event node when performing stream capture. This flag is
      invalid outside of stream capture.

    Parameters
    ----------
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream to wait
    hEvent : :py:obj:`~.CUevent` or :py:obj:`~.cudaEvent_t`
        Event to wait on (may not be NULL)
    Flags : unsigned int
        See :py:obj:`~.CUevent_capture_flags`

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`,

    See Also
    --------
    :py:obj:`~.cuStreamCreate`, :py:obj:`~.cuEventRecord`, :py:obj:`~.cuStreamQuery`, :py:obj:`~.cuStreamSynchronize`, :py:obj:`~.cuStreamAddCallback`, :py:obj:`~.cuStreamDestroy`, :py:obj:`~.cudaStreamWaitEvent`
    """
    cdef cydriver.CUevent cyhEvent
    if hEvent is None:
        cyhEvent = <cydriver.CUevent><void_ptr>0
    elif isinstance(hEvent, (CUevent,)):
        phEvent = int(hEvent)
        cyhEvent = <cydriver.CUevent><void_ptr>phEvent
    else:
        phEvent = int(CUevent(hEvent))
        cyhEvent = <cydriver.CUevent><void_ptr>phEvent
    cdef cydriver.CUstream cyhStream
    if hStream is None:
        cyhStream = <cydriver.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    err = cydriver.cuStreamWaitEvent(cyhStream, cyhEvent, Flags)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuStreamAddCallback(hStream, callback, userData, unsigned int flags):
    """ Add a callback to a compute stream.

    Adds a callback to be called on the host after all currently enqueued
    items in the stream have completed. For each cuStreamAddCallback call,
    the callback will be executed exactly once. The callback will block
    later work in the stream until it is finished.

    The callback may be passed :py:obj:`~.CUDA_SUCCESS` or an error code.
    In the event of a device error, all subsequently executed callbacks
    will receive an appropriate :py:obj:`~.CUresult`.

    Callbacks must not make any CUDA API calls. Attempting to use a CUDA
    API will result in :py:obj:`~.CUDA_ERROR_NOT_PERMITTED`. Callbacks must
    not perform any synchronization that may depend on outstanding device
    work or other callbacks that are not mandated to run earlier. Callbacks
    without a mandated order (in independent streams) execute in undefined
    order and may be serialized.

    For the purposes of Unified Memory, callback execution makes a number
    of guarantees:

    - The callback stream is considered idle for the duration of the
      callback. Thus, for example, a callback may always use memory
      attached to the callback stream.

    - The start of execution of a callback has the same effect as
      synchronizing an event recorded in the same stream immediately prior
      to the callback. It thus synchronizes streams which have been
      "joined" prior to the callback.

    - Adding device work to any stream does not have the effect of making
      the stream active until all preceding host functions and stream
      callbacks have executed. Thus, for example, a callback might use
      global attached memory even if work has been added to another stream,
      if the work has been ordered behind the callback with an event.

    - Completion of a callback does not cause a stream to become active
      except as described above. The callback stream will remain idle if no
      device work follows the callback, and will remain idle across
      consecutive callbacks without device work in between. Thus, for
      example, stream synchronization can be done by signaling from a
      callback at the end of the stream.

    Parameters
    ----------
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream to add callback to
    callback : :py:obj:`~.CUstreamCallback`
        The function to call once preceding stream operations are complete
    userData : Any
        User specified data to be passed to the callback function
    flags : unsigned int
        Reserved for future use, must be 0

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_NOT_SUPPORTED`

    See Also
    --------
    :py:obj:`~.cuStreamCreate`, :py:obj:`~.cuStreamQuery`, :py:obj:`~.cuStreamSynchronize`, :py:obj:`~.cuStreamWaitEvent`, :py:obj:`~.cuStreamDestroy`, :py:obj:`~.cuMemAllocManaged`, :py:obj:`~.cuStreamAttachMemAsync`, :py:obj:`~.cuStreamLaunchHostFunc`, :py:obj:`~.cudaStreamAddCallback`

    Notes
    -----
    This function is slated for eventual deprecation and removal. If you do not require the callback to execute in case of a device error, consider using :py:obj:`~.cuLaunchHostFunc`. Additionally, this function is not supported with :py:obj:`~.cuStreamBeginCapture` and :py:obj:`~.cuStreamEndCapture`, unlike :py:obj:`~.cuLaunchHostFunc`.
    """
    cdef cydriver.CUstreamCallback cycallback
    if callback is None:
        cycallback = <cydriver.CUstreamCallback><void_ptr>0
    elif isinstance(callback, (CUstreamCallback,)):
        pcallback = int(callback)
        cycallback = <cydriver.CUstreamCallback><void_ptr>pcallback
    else:
        pcallback = int(CUstreamCallback(callback))
        cycallback = <cydriver.CUstreamCallback><void_ptr>pcallback
    cdef cydriver.CUstream cyhStream
    if hStream is None:
        cyhStream = <cydriver.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    cyuserData = utils.HelperInputVoidPtr(userData)
    cdef void* cyuserData_ptr = <void*><void_ptr>cyuserData.cptr
    err = cydriver.cuStreamAddCallback(cyhStream, cycallback, cyuserData_ptr, flags)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuStreamBeginCapture(hStream, mode not None : CUstreamCaptureMode):
    """ Begins graph capture on a stream.

    Begin graph capture on `hStream`. When a stream is in capture mode, all
    operations pushed into the stream will not be executed, but will
    instead be captured into a graph, which will be returned via
    :py:obj:`~.cuStreamEndCapture`. Capture may not be initiated if
    `stream` is CU_STREAM_LEGACY. Capture must be ended on the same stream
    in which it was initiated, and it may only be initiated if the stream
    is not already in capture mode. The capture mode may be queried via
    :py:obj:`~.cuStreamIsCapturing`. A unique id representing the capture
    sequence may be queried via :py:obj:`~.cuStreamGetCaptureInfo`.

    If `mode` is not :py:obj:`~.CU_STREAM_CAPTURE_MODE_RELAXED`,
    :py:obj:`~.cuStreamEndCapture` must be called on this stream from the
    same thread.

    Parameters
    ----------
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream in which to initiate capture
    mode : :py:obj:`~.CUstreamCaptureMode`
        Controls the interaction of this capture sequence with other API
        calls that are potentially unsafe. For more details see
        :py:obj:`~.cuThreadExchangeStreamCaptureMode`.

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuStreamCreate`, :py:obj:`~.cuStreamIsCapturing`, :py:obj:`~.cuStreamEndCapture`, :py:obj:`~.cuThreadExchangeStreamCaptureMode`

    Notes
    -----
    Kernels captured using this API must not use texture and surface references. Reading or writing through any texture or surface reference is undefined behavior. This restriction does not apply to texture and surface objects.
    """
    cdef cydriver.CUstream cyhStream
    if hStream is None:
        cyhStream = <cydriver.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    cdef cydriver.CUstreamCaptureMode cymode = mode.value
    err = cydriver.cuStreamBeginCapture(cyhStream, cymode)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuThreadExchangeStreamCaptureMode(mode not None : CUstreamCaptureMode):
    """ Swaps the stream capture interaction mode for a thread.

    Sets the calling thread's stream capture interaction mode to the value
    contained in `*mode`, and overwrites `*mode` with the previous mode for
    the thread. To facilitate deterministic behavior across function or
    module boundaries, callers are encouraged to use this API in a push-pop
    fashion:

    **View CUDA Toolkit Documentation for a C++ code example**

    During stream capture (see :py:obj:`~.cuStreamBeginCapture`), some
    actions, such as a call to :py:obj:`~.cudaMalloc`, may be unsafe. In
    the case of :py:obj:`~.cudaMalloc`, the operation is not enqueued
    asynchronously to a stream, and is not observed by stream capture.
    Therefore, if the sequence of operations captured via
    :py:obj:`~.cuStreamBeginCapture` depended on the allocation being
    replayed whenever the graph is launched, the captured graph would be
    invalid.

    Therefore, stream capture places restrictions on API calls that can be
    made within or concurrently to a
    :py:obj:`~.cuStreamBeginCapture`-:py:obj:`~.cuStreamEndCapture`
    sequence. This behavior can be controlled via this API and flags to
    :py:obj:`~.cuStreamBeginCapture`.

    A thread's mode is one of the following:

    - `CU_STREAM_CAPTURE_MODE_GLOBAL:` This is the default mode. If the
      local thread has an ongoing capture sequence that was not initiated
      with `CU_STREAM_CAPTURE_MODE_RELAXED` at `cuStreamBeginCapture`, or
      if any other thread has a concurrent capture sequence initiated with
      `CU_STREAM_CAPTURE_MODE_GLOBAL`, this thread is prohibited from
      potentially unsafe API calls.

    - `CU_STREAM_CAPTURE_MODE_THREAD_LOCAL:` If the local thread has an
      ongoing capture sequence not initiated with
      `CU_STREAM_CAPTURE_MODE_RELAXED`, it is prohibited from potentially
      unsafe API calls. Concurrent capture sequences in other threads are
      ignored.

    - `CU_STREAM_CAPTURE_MODE_RELAXED:` The local thread is not prohibited
      from potentially unsafe API calls. Note that the thread is still
      prohibited from API calls which necessarily conflict with stream
      capture, for example, attempting :py:obj:`~.cuEventQuery` on an event
      that was last recorded inside a capture sequence.

    Parameters
    ----------
    mode : :py:obj:`~.CUstreamCaptureMode`
        Pointer to mode value to swap with the current mode

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`
    mode : :py:obj:`~.CUstreamCaptureMode`
        Pointer to mode value to swap with the current mode

    See Also
    --------
    :py:obj:`~.cuStreamBeginCapture`
    """
    cdef cydriver.CUstreamCaptureMode cymode = mode.value
    err = cydriver.cuThreadExchangeStreamCaptureMode(&cymode)
    return (CUresult(err), CUstreamCaptureMode(cymode))

@cython.embedsignature(True)
def cuStreamEndCapture(hStream):
    """ Ends capture on a stream, returning the captured graph.

    End capture on `hStream`, returning the captured graph via `phGraph`.
    Capture must have been initiated on `hStream` via a call to
    :py:obj:`~.cuStreamBeginCapture`. If capture was invalidated, due to a
    violation of the rules of stream capture, then a NULL graph will be
    returned.

    If the `mode` argument to :py:obj:`~.cuStreamBeginCapture` was not
    :py:obj:`~.CU_STREAM_CAPTURE_MODE_RELAXED`, this call must be from the
    same thread as :py:obj:`~.cuStreamBeginCapture`.

    Parameters
    ----------
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream to query

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_STREAM_CAPTURE_WRONG_THREAD`
    phGraph : :py:obj:`~.CUgraph`
        The captured graph

    See Also
    --------
    :py:obj:`~.cuStreamCreate`, :py:obj:`~.cuStreamBeginCapture`, :py:obj:`~.cuStreamIsCapturing`
    """
    cdef cydriver.CUstream cyhStream
    if hStream is None:
        cyhStream = <cydriver.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    cdef CUgraph phGraph = CUgraph()
    err = cydriver.cuStreamEndCapture(cyhStream, <cydriver.CUgraph*>phGraph._ptr)
    return (CUresult(err), phGraph)

@cython.embedsignature(True)
def cuStreamIsCapturing(hStream):
    """ Returns a stream's capture status.

    Return the capture status of `hStream` via `captureStatus`. After a
    successful call, `*captureStatus` will contain one of the following:

    - :py:obj:`~.CU_STREAM_CAPTURE_STATUS_NONE`: The stream is not
      capturing.

    - :py:obj:`~.CU_STREAM_CAPTURE_STATUS_ACTIVE`: The stream is capturing.

    - :py:obj:`~.CU_STREAM_CAPTURE_STATUS_INVALIDATED`: The stream was
      capturing but an error has invalidated the capture sequence. The
      capture sequence must be terminated with
      :py:obj:`~.cuStreamEndCapture` on the stream where it was initiated
      in order to continue using `hStream`.

    Note that, if this is called on :py:obj:`~.CU_STREAM_LEGACY` (the "null
    stream") while a blocking stream in the same context is capturing, it
    will return :py:obj:`~.CUDA_ERROR_STREAM_CAPTURE_IMPLICIT` and
    `*captureStatus` is unspecified after the call. The blocking stream
    capture is not invalidated.

    When a blocking stream is capturing, the legacy stream is in an
    unusable state until the blocking stream capture is terminated. The
    legacy stream is not supported for stream capture, but attempted use
    would have an implicit dependency on the capturing stream(s).

    Parameters
    ----------
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream to query

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_STREAM_CAPTURE_IMPLICIT`
    captureStatus : :py:obj:`~.CUstreamCaptureStatus`
        Returns the stream's capture status

    See Also
    --------
    :py:obj:`~.cuStreamCreate`, :py:obj:`~.cuStreamBeginCapture`, :py:obj:`~.cuStreamEndCapture`
    """
    cdef cydriver.CUstream cyhStream
    if hStream is None:
        cyhStream = <cydriver.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    cdef cydriver.CUstreamCaptureStatus captureStatus
    err = cydriver.cuStreamIsCapturing(cyhStream, &captureStatus)
    return (CUresult(err), CUstreamCaptureStatus(captureStatus))

@cython.embedsignature(True)
def cuStreamGetCaptureInfo(hStream):
    """ Query capture status of a stream.

    Note there is a later version of this API,
    :py:obj:`~.cuStreamGetCaptureInfo_v2`. It will supplant this version in
    12.0, which is retained for minor version compatibility.

    Query the capture status of a stream and and get an id for the capture
    sequence, which is unique over the lifetime of the process.

    If called on :py:obj:`~.CU_STREAM_LEGACY` (the "null stream") while a
    stream not created with :py:obj:`~.CU_STREAM_NON_BLOCKING` is
    capturing, returns :py:obj:`~.CUDA_ERROR_STREAM_CAPTURE_IMPLICIT`.

    A valid id is returned only if both of the following are true:

    - the call returns CUDA_SUCCESS

    - captureStatus is set to :py:obj:`~.CU_STREAM_CAPTURE_STATUS_ACTIVE`

    Parameters
    ----------
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        None

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_STREAM_CAPTURE_IMPLICIT`
    captureStatus_out : :py:obj:`~.CUstreamCaptureStatus`
        None
    id_out : :py:obj:`~.cuuint64_t`
        None

    See Also
    --------
    :py:obj:`~.cuStreamGetCaptureInfo_v2`, :py:obj:`~.cuStreamBeginCapture`, :py:obj:`~.cuStreamIsCapturing`
    """
    cdef cydriver.CUstream cyhStream
    if hStream is None:
        cyhStream = <cydriver.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    cdef cydriver.CUstreamCaptureStatus captureStatus_out
    cdef cuuint64_t id_out = cuuint64_t()
    err = cydriver.cuStreamGetCaptureInfo(cyhStream, &captureStatus_out, <cydriver.cuuint64_t*>id_out._ptr)
    return (CUresult(err), CUstreamCaptureStatus(captureStatus_out), id_out)

@cython.embedsignature(True)
def cuStreamGetCaptureInfo_v2(hStream):
    """ Query a stream's capture state (11.3+)

    Query stream state related to stream capture.

    If called on :py:obj:`~.CU_STREAM_LEGACY` (the "null stream") while a
    stream not created with :py:obj:`~.CU_STREAM_NON_BLOCKING` is
    capturing, returns :py:obj:`~.CUDA_ERROR_STREAM_CAPTURE_IMPLICIT`.

    Valid data (other than capture status) is returned only if both of the
    following are true:

    - the call returns CUDA_SUCCESS

    - the returned capture status is
      :py:obj:`~.CU_STREAM_CAPTURE_STATUS_ACTIVE`

    This version of cuStreamGetCaptureInfo is introduced in CUDA 11.3 and
    will supplant the previous version in 12.0. Developers requiring
    compatibility across minor versions to CUDA 11.0 (driver version 445)
    should use :py:obj:`~.cuStreamGetCaptureInfo` or include a fallback
    path.

    Parameters
    ----------
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        The stream to query

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_STREAM_CAPTURE_IMPLICIT`
    captureStatus_out : :py:obj:`~.CUstreamCaptureStatus`
        Location to return the capture status of the stream; required
    id_out : :py:obj:`~.cuuint64_t`
        Optional location to return an id for the capture sequence, which
        is unique over the lifetime of the process
    graph_out : :py:obj:`~.CUgraph`
        Optional location to return the graph being captured into. All
        operations other than destroy and node removal are permitted on the
        graph while the capture sequence is in progress. This API does not
        transfer ownership of the graph, which is transferred or destroyed
        at :py:obj:`~.cuStreamEndCapture`. Note that the graph handle may
        be invalidated before end of capture for certain errors. Nodes that
        are or become unreachable from the original stream at
        :py:obj:`~.cuStreamEndCapture` due to direct actions on the graph
        do not trigger :py:obj:`~.CUDA_ERROR_STREAM_CAPTURE_UNJOINED`.
    dependencies_out : List[:py:obj:`~.CUgraphNode`]
        Optional location to store a pointer to an array of nodes. The next
        node to be captured in the stream will depend on this set of nodes,
        absent operations such as event wait which modify this set. The
        array pointer is valid until the next API call which operates on
        the stream or until end of capture. The node handles may be copied
        out and are valid until they or the graph is destroyed. The driver-
        owned array may also be passed directly to APIs that operate on the
        graph (not the stream) without copying.
    numDependencies_out : int
        Optional location to store the size of the array returned in
        dependencies_out.

    See Also
    --------
    :py:obj:`~.cuStreamGetCaptureInfo`, :py:obj:`~.cuStreamBeginCapture`, :py:obj:`~.cuStreamIsCapturing`, :py:obj:`~.cuStreamUpdateCaptureDependencies`
    """
    cdef cydriver.CUstream cyhStream
    if hStream is None:
        cyhStream = <cydriver.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    cdef cydriver.CUstreamCaptureStatus captureStatus_out
    cdef cuuint64_t id_out = cuuint64_t()
    cdef CUgraph graph_out = CUgraph()
    cdef const cydriver.CUgraphNode* cydependencies_out = NULL
    pydependencies_out = []
    cdef size_t numDependencies_out = 0
    err = cydriver.cuStreamGetCaptureInfo_v2(cyhStream, &captureStatus_out, <cydriver.cuuint64_t*>id_out._ptr, <cydriver.CUgraph*>graph_out._ptr, &cydependencies_out, &numDependencies_out)
    if CUresult(err) == CUresult(0):
        pydependencies_out = [CUgraphNode(init_value=<void_ptr>cydependencies_out[idx]) for idx in range(numDependencies_out)]
    return (CUresult(err), CUstreamCaptureStatus(captureStatus_out), id_out, graph_out, pydependencies_out, numDependencies_out)

@cython.embedsignature(True)
def cuStreamUpdateCaptureDependencies(hStream, dependencies : Optional[Tuple[CUgraphNode] | List[CUgraphNode]], size_t numDependencies, unsigned int flags):
    """ Update the set of dependencies in a capturing stream (11.3+)

    Modifies the dependency set of a capturing stream. The dependency set
    is the set of nodes that the next captured node in the stream will
    depend on.

    Valid flags are :py:obj:`~.CU_STREAM_ADD_CAPTURE_DEPENDENCIES` and
    :py:obj:`~.CU_STREAM_SET_CAPTURE_DEPENDENCIES`. These control whether
    the set passed to the API is added to the existing set or replaces it.
    A flags value of 0 defaults to
    :py:obj:`~.CU_STREAM_ADD_CAPTURE_DEPENDENCIES`.

    Nodes that are removed from the dependency set via this API do not
    result in :py:obj:`~.CUDA_ERROR_STREAM_CAPTURE_UNJOINED` if they are
    unreachable from the stream at :py:obj:`~.cuStreamEndCapture`.

    Returns :py:obj:`~.CUDA_ERROR_ILLEGAL_STATE` if the stream is not
    capturing.

    This API is new in CUDA 11.3. Developers requiring compatibility across
    minor versions to CUDA 11.0 should not use this API or provide a
    fallback.

    Parameters
    ----------
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        None
    dependencies : List[:py:obj:`~.CUgraphNode`]
        None
    numDependencies : size_t
        None
    flags : unsigned int
        None

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_ILLEGAL_STATE`

    See Also
    --------
    :py:obj:`~.cuStreamBeginCapture`, :py:obj:`~.cuStreamGetCaptureInfo`, :py:obj:`~.cuStreamGetCaptureInfo_v2`
    """
    dependencies = [] if dependencies is None else dependencies
    if not all(isinstance(_x, (CUgraphNode,)) for _x in dependencies):
        raise TypeError("Argument 'dependencies' is not instance of type (expected Tuple[cydriver.CUgraphNode,] or List[cydriver.CUgraphNode,]")
    cdef cydriver.CUstream cyhStream
    if hStream is None:
        cyhStream = <cydriver.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    cdef cydriver.CUgraphNode* cydependencies = NULL
    if len(dependencies) > 0:
        cydependencies = <cydriver.CUgraphNode*> calloc(len(dependencies), sizeof(cydriver.CUgraphNode))
        if cydependencies is NULL:
            raise MemoryError('Failed to allocate length x size memory: ' + str(len(dependencies)) + 'x' + str(sizeof(cydriver.CUgraphNode)))
        else:
            for idx in range(len(dependencies)):
                cydependencies[idx] = <cydriver.CUgraphNode>(<CUgraphNode>dependencies[idx])._ptr[0]
    if numDependencies > <size_t>len(dependencies): raise RuntimeError("List is too small: " + str(len(dependencies)) + " < " + str(numDependencies))
    err = cydriver.cuStreamUpdateCaptureDependencies(cyhStream, <cydriver.CUgraphNode*>(<CUgraphNode>dependencies[0])._ptr if len(dependencies) == 1 else cydependencies, numDependencies, flags)
    if cydependencies is not NULL:
        free(cydependencies)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuStreamAttachMemAsync(hStream, dptr, size_t length, unsigned int flags):
    """ Attach memory to a stream asynchronously.

    Enqueues an operation in `hStream` to specify stream association of
    `length` bytes of memory starting from `dptr`. This function is a
    stream-ordered operation, meaning that it is dependent on, and will
    only take effect when, previous work in stream has completed. Any
    previous association is automatically replaced.

    `dptr` must point to one of the following types of memories:

    - managed memory declared using the managed keyword or allocated with
      :py:obj:`~.cuMemAllocManaged`.

    - a valid host-accessible region of system-allocated pageable memory.
      This type of memory may only be specified if the device associated
      with the stream reports a non-zero value for the device attribute
      :py:obj:`~.CU_DEVICE_ATTRIBUTE_PAGEABLE_MEMORY_ACCESS`.

    For managed allocations, `length` must be either zero or the entire
    allocation's size. Both indicate that the entire allocation's stream
    association is being changed. Currently, it is not possible to change
    stream association for a portion of a managed allocation.

    For pageable host allocations, `length` must be non-zero.

    The stream association is specified using `flags` which must be one of
    :py:obj:`~.CUmemAttach_flags`. If the :py:obj:`~.CU_MEM_ATTACH_GLOBAL`
    flag is specified, the memory can be accessed by any stream on any
    device. If the :py:obj:`~.CU_MEM_ATTACH_HOST` flag is specified, the
    program makes a guarantee that it won't access the memory on the device
    from any stream on a device that has a zero value for the device
    attribute :py:obj:`~.CU_DEVICE_ATTRIBUTE_CONCURRENT_MANAGED_ACCESS`. If
    the :py:obj:`~.CU_MEM_ATTACH_SINGLE` flag is specified and `hStream` is
    associated with a device that has a zero value for the device attribute
    :py:obj:`~.CU_DEVICE_ATTRIBUTE_CONCURRENT_MANAGED_ACCESS`, the program
    makes a guarantee that it will only access the memory on the device
    from `hStream`. It is illegal to attach singly to the NULL stream,
    because the NULL stream is a virtual global stream and not a specific
    stream. An error will be returned in this case.

    When memory is associated with a single stream, the Unified Memory
    system will allow CPU access to this memory region so long as all
    operations in `hStream` have completed, regardless of whether other
    streams are active. In effect, this constrains exclusive ownership of
    the managed memory region by an active GPU to per-stream activity
    instead of whole-GPU activity.

    Accessing memory on the device from streams that are not associated
    with it will produce undefined results. No error checking is performed
    by the Unified Memory system to ensure that kernels launched into other
    streams do not access this region.

    It is a program's responsibility to order calls to
    :py:obj:`~.cuStreamAttachMemAsync` via events, synchronization or other
    means to ensure legal access to memory at all times. Data visibility
    and coherency will be changed appropriately for all kernels which
    follow a stream-association change.

    If `hStream` is destroyed while data is associated with it, the
    association is removed and the association reverts to the default
    visibility of the allocation as specified at
    :py:obj:`~.cuMemAllocManaged`. For managed variables, the default
    association is always :py:obj:`~.CU_MEM_ATTACH_GLOBAL`. Note that
    destroying a stream is an asynchronous operation, and as a result, the
    change to default association won't happen until all work in the stream
    has completed.

    Parameters
    ----------
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream in which to enqueue the attach operation
    dptr : :py:obj:`~.CUdeviceptr`
        Pointer to memory (must be a pointer to managed memory or to a
        valid host-accessible region of system-allocated pageable memory)
    length : size_t
        Length of memory
    flags : unsigned int
        Must be one of :py:obj:`~.CUmemAttach_flags`

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_NOT_SUPPORTED`

    See Also
    --------
    :py:obj:`~.cuStreamCreate`, :py:obj:`~.cuStreamQuery`, :py:obj:`~.cuStreamSynchronize`, :py:obj:`~.cuStreamWaitEvent`, :py:obj:`~.cuStreamDestroy`, :py:obj:`~.cuMemAllocManaged`, :py:obj:`~.cudaStreamAttachMemAsync`
    """
    cdef cydriver.CUdeviceptr cydptr
    if dptr is None:
        cydptr = <cydriver.CUdeviceptr><void_ptr>0
    elif isinstance(dptr, (CUdeviceptr,)):
        pdptr = int(dptr)
        cydptr = <cydriver.CUdeviceptr><void_ptr>pdptr
    else:
        pdptr = int(CUdeviceptr(dptr))
        cydptr = <cydriver.CUdeviceptr><void_ptr>pdptr
    cdef cydriver.CUstream cyhStream
    if hStream is None:
        cyhStream = <cydriver.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    err = cydriver.cuStreamAttachMemAsync(cyhStream, cydptr, length, flags)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuStreamQuery(hStream):
    """ Determine status of a compute stream.

    Returns :py:obj:`~.CUDA_SUCCESS` if all operations in the stream
    specified by `hStream` have completed, or
    :py:obj:`~.CUDA_ERROR_NOT_READY` if not.

    For the purposes of Unified Memory, a return value of
    :py:obj:`~.CUDA_SUCCESS` is equivalent to having called
    :py:obj:`~.cuStreamSynchronize()`.

    Parameters
    ----------
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream to query status of

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_NOT_READY`

    See Also
    --------
    :py:obj:`~.cuStreamCreate`, :py:obj:`~.cuStreamWaitEvent`, :py:obj:`~.cuStreamDestroy`, :py:obj:`~.cuStreamSynchronize`, :py:obj:`~.cuStreamAddCallback`, :py:obj:`~.cudaStreamQuery`
    """
    cdef cydriver.CUstream cyhStream
    if hStream is None:
        cyhStream = <cydriver.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    err = cydriver.cuStreamQuery(cyhStream)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuStreamSynchronize(hStream):
    """ Wait until a stream's tasks are completed.

    Waits until the device has completed all operations in the stream
    specified by `hStream`. If the context was created with the
    :py:obj:`~.CU_CTX_SCHED_BLOCKING_SYNC` flag, the CPU thread will block
    until the stream is finished with all of its tasks.

    \note_null_stream

    Parameters
    ----------
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream to wait for

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`

    See Also
    --------
    :py:obj:`~.cuStreamCreate`, :py:obj:`~.cuStreamDestroy`, :py:obj:`~.cuStreamWaitEvent`, :py:obj:`~.cuStreamQuery`, :py:obj:`~.cuStreamAddCallback`, :py:obj:`~.cudaStreamSynchronize`
    """
    cdef cydriver.CUstream cyhStream
    if hStream is None:
        cyhStream = <cydriver.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    err = cydriver.cuStreamSynchronize(cyhStream)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuStreamDestroy(hStream):
    """ Destroys a stream.

    Destroys the stream specified by `hStream`.

    In case the device is still doing work in the stream `hStream` when
    :py:obj:`~.cuStreamDestroy()` is called, the function will return
    immediately and the resources associated with `hStream` will be
    released automatically once the device has completed all work in
    `hStream`.

    Parameters
    ----------
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream to destroy

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`

    See Also
    --------
    :py:obj:`~.cuStreamCreate`, :py:obj:`~.cuStreamWaitEvent`, :py:obj:`~.cuStreamQuery`, :py:obj:`~.cuStreamSynchronize`, :py:obj:`~.cuStreamAddCallback`, :py:obj:`~.cudaStreamDestroy`
    """
    cdef cydriver.CUstream cyhStream
    if hStream is None:
        cyhStream = <cydriver.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    err = cydriver.cuStreamDestroy(cyhStream)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuStreamCopyAttributes(dst, src):
    """ Copies attributes from source stream to destination stream.

    Copies attributes from source stream `src` to destination stream `dst`.
    Both streams must have the same context.

    Parameters
    ----------
    dst : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Destination stream
    src : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Source stream For list of attributes see :py:obj:`~.CUstreamAttrID`

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.CUaccessPolicyWindow`
    """
    cdef cydriver.CUstream cysrc
    if src is None:
        cysrc = <cydriver.CUstream><void_ptr>0
    elif isinstance(src, (CUstream,)):
        psrc = int(src)
        cysrc = <cydriver.CUstream><void_ptr>psrc
    else:
        psrc = int(CUstream(src))
        cysrc = <cydriver.CUstream><void_ptr>psrc
    cdef cydriver.CUstream cydst
    if dst is None:
        cydst = <cydriver.CUstream><void_ptr>0
    elif isinstance(dst, (CUstream,)):
        pdst = int(dst)
        cydst = <cydriver.CUstream><void_ptr>pdst
    else:
        pdst = int(CUstream(dst))
        cydst = <cydriver.CUstream><void_ptr>pdst
    err = cydriver.cuStreamCopyAttributes(cydst, cysrc)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuStreamGetAttribute(hStream, attr not None : CUstreamAttrID):
    """ Queries stream attribute.

    Queries attribute `attr` from `hStream` and stores it in corresponding
    member of `value_out`.

    Parameters
    ----------
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`

    attr : :py:obj:`~.CUstreamAttrID`


    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`
    value_out : :py:obj:`~.CUstreamAttrValue`


    See Also
    --------
    :py:obj:`~.CUaccessPolicyWindow`
    """
    cdef cydriver.CUstream cyhStream
    if hStream is None:
        cyhStream = <cydriver.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    cdef cydriver.CUstreamAttrID cyattr = attr.value
    cdef CUstreamAttrValue value_out = CUstreamAttrValue()
    err = cydriver.cuStreamGetAttribute(cyhStream, cyattr, <cydriver.CUstreamAttrValue*>value_out._ptr)
    return (CUresult(err), value_out)

@cython.embedsignature(True)
def cuStreamSetAttribute(hStream, attr not None : CUstreamAttrID, value : Optional[CUstreamAttrValue]):
    """ Sets stream attribute.

    Sets attribute `attr` on `hStream` from corresponding attribute of
    `value`. The updated attribute will be applied to subsequent work
    submitted to the stream. It will not affect previously submitted work.

    Parameters
    ----------
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`

    attr : :py:obj:`~.CUstreamAttrID`

    value : :py:obj:`~.CUstreamAttrValue`


    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`

    See Also
    --------
    :py:obj:`~.CUaccessPolicyWindow`
    """
    cdef cydriver.CUstream cyhStream
    if hStream is None:
        cyhStream = <cydriver.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    cdef cydriver.CUstreamAttrID cyattr = attr.value
    cdef cydriver.CUstreamAttrValue* cyvalue_ptr = value._ptr if value != None else NULL
    err = cydriver.cuStreamSetAttribute(cyhStream, cyattr, cyvalue_ptr)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuEventCreate(unsigned int Flags):
    """ Creates an event.

    Creates an event *phEvent for the current context with the flags
    specified via `Flags`. Valid flags include:

    - :py:obj:`~.CU_EVENT_DEFAULT`: Default event creation flag.

    - :py:obj:`~.CU_EVENT_BLOCKING_SYNC`: Specifies that the created event
      should use blocking synchronization. A CPU thread that uses
      :py:obj:`~.cuEventSynchronize()` to wait on an event created with
      this flag will block until the event has actually been recorded.

    - :py:obj:`~.CU_EVENT_DISABLE_TIMING`: Specifies that the created event
      does not need to record timing data. Events created with this flag
      specified and the :py:obj:`~.CU_EVENT_BLOCKING_SYNC` flag not
      specified will provide the best performance when used with
      :py:obj:`~.cuStreamWaitEvent()` and :py:obj:`~.cuEventQuery()`.

    - :py:obj:`~.CU_EVENT_INTERPROCESS`: Specifies that the created event
      may be used as an interprocess event by
      :py:obj:`~.cuIpcGetEventHandle()`. :py:obj:`~.CU_EVENT_INTERPROCESS`
      must be specified along with :py:obj:`~.CU_EVENT_DISABLE_TIMING`.

    Parameters
    ----------
    Flags : unsigned int
        Event creation flags

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_OUT_OF_MEMORY`
    phEvent : :py:obj:`~.CUevent`
        Returns newly created event

    See Also
    --------
    :py:obj:`~.cuEventRecord`, :py:obj:`~.cuEventQuery`, :py:obj:`~.cuEventSynchronize`, :py:obj:`~.cuEventDestroy`, :py:obj:`~.cuEventElapsedTime`, :py:obj:`~.cudaEventCreate`, :py:obj:`~.cudaEventCreateWithFlags`
    """
    cdef CUevent phEvent = CUevent()
    err = cydriver.cuEventCreate(<cydriver.CUevent*>phEvent._ptr, Flags)
    return (CUresult(err), phEvent)

@cython.embedsignature(True)
def cuEventRecord(hEvent, hStream):
    """ Records an event.

    Captures in `hEvent` the contents of `hStream` at the time of this
    call. `hEvent` and `hStream` must be from the same context. Calls such
    as :py:obj:`~.cuEventQuery()` or :py:obj:`~.cuStreamWaitEvent()` will
    then examine or wait for completion of the work that was captured. Uses
    of `hStream` after this call do not modify `hEvent`. See note on
    default stream behavior for what is captured in the default case.

    :py:obj:`~.cuEventRecord()` can be called multiple times on the same
    event and will overwrite the previously captured state. Other APIs such
    as :py:obj:`~.cuStreamWaitEvent()` use the most recently captured state
    at the time of the API call, and are not affected by later calls to
    :py:obj:`~.cuEventRecord()`. Before the first call to
    :py:obj:`~.cuEventRecord()`, an event represents an empty set of work,
    so for example :py:obj:`~.cuEventQuery()` would return
    :py:obj:`~.CUDA_SUCCESS`.

    Parameters
    ----------
    hEvent : :py:obj:`~.CUevent` or :py:obj:`~.cudaEvent_t`
        Event to record
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream to record event for

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuEventCreate`, :py:obj:`~.cuEventQuery`, :py:obj:`~.cuEventSynchronize`, :py:obj:`~.cuStreamWaitEvent`, :py:obj:`~.cuEventDestroy`, :py:obj:`~.cuEventElapsedTime`, :py:obj:`~.cudaEventRecord`, :py:obj:`~.cuEventRecordWithFlags`
    """
    cdef cydriver.CUstream cyhStream
    if hStream is None:
        cyhStream = <cydriver.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    cdef cydriver.CUevent cyhEvent
    if hEvent is None:
        cyhEvent = <cydriver.CUevent><void_ptr>0
    elif isinstance(hEvent, (CUevent,)):
        phEvent = int(hEvent)
        cyhEvent = <cydriver.CUevent><void_ptr>phEvent
    else:
        phEvent = int(CUevent(hEvent))
        cyhEvent = <cydriver.CUevent><void_ptr>phEvent
    err = cydriver.cuEventRecord(cyhEvent, cyhStream)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuEventRecordWithFlags(hEvent, hStream, unsigned int flags):
    """ Records an event.

    Captures in `hEvent` the contents of `hStream` at the time of this
    call. `hEvent` and `hStream` must be from the same context. Calls such
    as :py:obj:`~.cuEventQuery()` or :py:obj:`~.cuStreamWaitEvent()` will
    then examine or wait for completion of the work that was captured. Uses
    of `hStream` after this call do not modify `hEvent`. See note on
    default stream behavior for what is captured in the default case.

    :py:obj:`~.cuEventRecordWithFlags()` can be called multiple times on
    the same event and will overwrite the previously captured state. Other
    APIs such as :py:obj:`~.cuStreamWaitEvent()` use the most recently
    captured state at the time of the API call, and are not affected by
    later calls to :py:obj:`~.cuEventRecordWithFlags()`. Before the first
    call to :py:obj:`~.cuEventRecordWithFlags()`, an event represents an
    empty set of work, so for example :py:obj:`~.cuEventQuery()` would
    return :py:obj:`~.CUDA_SUCCESS`.

    flags include:

    - :py:obj:`~.CU_EVENT_RECORD_DEFAULT`: Default event creation flag.

    - :py:obj:`~.CU_EVENT_RECORD_EXTERNAL`: Event is captured in the graph
      as an external event node when performing stream capture. This flag
      is invalid outside of stream capture.

    Parameters
    ----------
    hEvent : :py:obj:`~.CUevent` or :py:obj:`~.cudaEvent_t`
        Event to record
    hStream : :py:obj:`~.CUstream` or :py:obj:`~.cudaStream_t`
        Stream to record event for
    flags : unsigned int
        See :py:obj:`~.CUevent_capture_flags`

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`

    See Also
    --------
    :py:obj:`~.cuEventCreate`, :py:obj:`~.cuEventQuery`, :py:obj:`~.cuEventSynchronize`, :py:obj:`~.cuStreamWaitEvent`, :py:obj:`~.cuEventDestroy`, :py:obj:`~.cuEventElapsedTime`, :py:obj:`~.cuEventRecord`, :py:obj:`~.cudaEventRecord`
    """
    cdef cydriver.CUstream cyhStream
    if hStream is None:
        cyhStream = <cydriver.CUstream><void_ptr>0
    elif isinstance(hStream, (CUstream,)):
        phStream = int(hStream)
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    else:
        phStream = int(CUstream(hStream))
        cyhStream = <cydriver.CUstream><void_ptr>phStream
    cdef cydriver.CUevent cyhEvent
    if hEvent is None:
        cyhEvent = <cydriver.CUevent><void_ptr>0
    elif isinstance(hEvent, (CUevent,)):
        phEvent = int(hEvent)
        cyhEvent = <cydriver.CUevent><void_ptr>phEvent
    else:
        phEvent = int(CUevent(hEvent))
        cyhEvent = <cydriver.CUevent><void_ptr>phEvent
    err = cydriver.cuEventRecordWithFlags(cyhEvent, cyhStream, flags)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuEventQuery(hEvent):
    """ Queries an event's status.

    Queries the status of all work currently captured by `hEvent`. See
    :py:obj:`~.cuEventRecord()` for details on what is captured by an
    event.

    Returns :py:obj:`~.CUDA_SUCCESS` if all captured work has been
    completed, or :py:obj:`~.CUDA_ERROR_NOT_READY` if any captured work is
    incomplete.

    For the purposes of Unified Memory, a return value of
    :py:obj:`~.CUDA_SUCCESS` is equivalent to having called
    :py:obj:`~.cuEventSynchronize()`.

    Parameters
    ----------
    hEvent : :py:obj:`~.CUevent` or :py:obj:`~.cudaEvent_t`
        Event to query

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_NOT_READY`

    See Also
    --------
    :py:obj:`~.cuEventCreate`, :py:obj:`~.cuEventRecord`, :py:obj:`~.cuEventSynchronize`, :py:obj:`~.cuEventDestroy`, :py:obj:`~.cuEventElapsedTime`, :py:obj:`~.cudaEventQuery`
    """
    cdef cydriver.CUevent cyhEvent
    if hEvent is None:
        cyhEvent = <cydriver.CUevent><void_ptr>0
    elif isinstance(hEvent, (CUevent,)):
        phEvent = int(hEvent)
        cyhEvent = <cydriver.CUevent><void_ptr>phEvent
    else:
        phEvent = int(CUevent(hEvent))
        cyhEvent = <cydriver.CUevent><void_ptr>phEvent
    err = cydriver.cuEventQuery(cyhEvent)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuEventSynchronize(hEvent):
    """ Waits for an event to complete.

    Waits until the completion of all work currently captured in `hEvent`.
    See :py:obj:`~.cuEventRecord()` for details on what is captured by an
    event.

    Waiting for an event that was created with the
    :py:obj:`~.CU_EVENT_BLOCKING_SYNC` flag will cause the calling CPU
    thread to block until the event has been completed by the device. If
    the :py:obj:`~.CU_EVENT_BLOCKING_SYNC` flag has not been set, then the
    CPU thread will busy-wait until the event has been completed by the
    device.

    Parameters
    ----------
    hEvent : :py:obj:`~.CUevent` or :py:obj:`~.cudaEvent_t`
        Event to wait for

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`

    See Also
    --------
    :py:obj:`~.cuEventCreate`, :py:obj:`~.cuEventRecord`, :py:obj:`~.cuEventQuery`, :py:obj:`~.cuEventDestroy`, :py:obj:`~.cuEventElapsedTime`, :py:obj:`~.cudaEventSynchronize`
    """
    cdef cydriver.CUevent cyhEvent
    if hEvent is None:
        cyhEvent = <cydriver.CUevent><void_ptr>0
    elif isinstance(hEvent, (CUevent,)):
        phEvent = int(hEvent)
        cyhEvent = <cydriver.CUevent><void_ptr>phEvent
    else:
        phEvent = int(CUevent(hEvent))
        cyhEvent = <cydriver.CUevent><void_ptr>phEvent
    err = cydriver.cuEventSynchronize(cyhEvent)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuEventDestroy(hEvent):
    """ Destroys an event.

    Destroys the event specified by `hEvent`.

    An event may be destroyed before it is complete (i.e., while
    :py:obj:`~.cuEventQuery()` would return
    :py:obj:`~.CUDA_ERROR_NOT_READY`). In this case, the call does not
    block on completion of the event, and any associated resources will
    automatically be released asynchronously at completion.

    Parameters
    ----------
    hEvent : :py:obj:`~.CUevent` or :py:obj:`~.cudaEvent_t`
        Event to destroy

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`

    See Also
    --------
    :py:obj:`~.cuEventCreate`, :py:obj:`~.cuEventRecord`, :py:obj:`~.cuEventQuery`, :py:obj:`~.cuEventSynchronize`, :py:obj:`~.cuEventElapsedTime`, :py:obj:`~.cudaEventDestroy`
    """
    cdef cydriver.CUevent cyhEvent
    if hEvent is None:
        cyhEvent = <cydriver.CUevent><void_ptr>0
    elif isinstance(hEvent, (CUevent,)):
        phEvent = int(hEvent)
        cyhEvent = <cydriver.CUevent><void_ptr>phEvent
    else:
        phEvent = int(CUevent(hEvent))
        cyhEvent = <cydriver.CUevent><void_ptr>phEvent
    err = cydriver.cuEventDestroy(cyhEvent)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuEventElapsedTime(hStart, hEnd):
    """ Computes the elapsed time between two events.

    Computes the elapsed time between two events (in milliseconds with a
    resolution of around 0.5 microseconds).

    If either event was last recorded in a non-NULL stream, the resulting
    time may be greater than expected (even if both used the same stream
    handle). This happens because the :py:obj:`~.cuEventRecord()` operation
    takes place asynchronously and there is no guarantee that the measured
    latency is actually just between the two events. Any number of other
    different stream operations could execute in between the two measured
    events, thus altering the timing in a significant way.

    If :py:obj:`~.cuEventRecord()` has not been called on either event then
    :py:obj:`~.CUDA_ERROR_INVALID_HANDLE` is returned. If
    :py:obj:`~.cuEventRecord()` has been called on both events but one or
    both of them has not yet been completed (that is,
    :py:obj:`~.cuEventQuery()` would return
    :py:obj:`~.CUDA_ERROR_NOT_READY` on at least one of the events),
    :py:obj:`~.CUDA_ERROR_NOT_READY` is returned. If either event was
    created with the :py:obj:`~.CU_EVENT_DISABLE_TIMING` flag, then this
    function will return :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`.

    Parameters
    ----------
    hStart : :py:obj:`~.CUevent` or :py:obj:`~.cudaEvent_t`
        Starting event
    hEnd : :py:obj:`~.CUevent` or :py:obj:`~.cudaEvent_t`
        Ending event

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_DEINITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_CONTEXT`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`, :py:obj:`~.CUDA_ERROR_NOT_READY`, :py:obj:`~.CUDA_ERROR_UNKNOWN`
    pMilliseconds : float
        Time between `hStart` and `hEnd` in ms

    See Also
    --------
    :py:obj:`~.cuEventCreate`, :py:obj:`~.cuEventRecord`, :py:obj:`~.cuEventQuery`, :py:obj:`~.cuEventSynchronize`, :py:obj:`~.cuEventDestroy`, :py:obj:`~.cudaEventElapsedTime`
    """
    cdef cydriver.CUevent cyhEnd
    if hEnd is None:
        cyhEnd = <cydriver.CUevent><void_ptr>0
    elif isinstance(hEnd, (CUevent,)):
        phEnd = int(hEnd)
        cyhEnd = <cydriver.CUevent><void_ptr>phEnd
    else:
        phEnd = int(CUevent(hEnd))
        cyhEnd = <cydriver.CUevent><void_ptr>phEnd
    cdef cydriver.CUevent cyhStart
    if hStart is None:
        cyhStart = <cydriver.CUevent><void_ptr>0
    elif isinstance(hStart, (CUevent,)):
        phStart = int(hStart)
        cyhStart = <cydriver.CUevent><void_ptr>phStart
    else:
        phStart = int(CUevent(hStart))
        cyhStart = <cydriver.CUevent><void_ptr>phStart
    cdef float pMilliseconds = 0
    err = cydriver.cuEventElapsedTime(&pMilliseconds, cyhStart, cyhEnd)
    return (CUresult(err), pMilliseconds)

@cython.embedsignature(True)
def cuImportExternalMemory(memHandleDesc : Optional[CUDA_EXTERNAL_MEMORY_HANDLE_DESC]):
    """ Imports an external memory object.

    Imports an externally allocated memory object and returns a handle to
    that in `extMem_out`.

    The properties of the handle being imported must be described in
    `memHandleDesc`. The :py:obj:`~.CUDA_EXTERNAL_MEMORY_HANDLE_DESC`
    structure is defined as follows:

    **View CUDA Toolkit Documentation for a C++ code example**

    where :py:obj:`~.CUDA_EXTERNAL_MEMORY_HANDLE_DESC.type` specifies the
    type of handle being imported. :py:obj:`~.CUexternalMemoryHandleType`
    is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    If :py:obj:`~.CUDA_EXTERNAL_MEMORY_HANDLE_DESC.type` is
    :py:obj:`~.CU_EXTERNAL_MEMORY_HANDLE_TYPE_OPAQUE_FD`, then
    :py:obj:`~.CUDA_EXTERNAL_MEMORY_HANDLE_DESC`::handle::fd must be a
    valid file descriptor referencing a memory object. Ownership of the
    file descriptor is transferred to the CUDA driver when the handle is
    imported successfully. Performing any operations on the file descriptor
    after it is imported results in undefined behavior.

    If :py:obj:`~.CUDA_EXTERNAL_MEMORY_HANDLE_DESC.type` is
    :py:obj:`~.CU_EXTERNAL_MEMORY_HANDLE_TYPE_OPAQUE_WIN32`, then exactly
    one of
    :py:obj:`~.CUDA_EXTERNAL_MEMORY_HANDLE_DESC`::handle::win32::handle and
    :py:obj:`~.CUDA_EXTERNAL_MEMORY_HANDLE_DESC`::handle::win32::name must
    not be NULL. If
    :py:obj:`~.CUDA_EXTERNAL_MEMORY_HANDLE_DESC`::handle::win32::handle is
    not NULL, then it must represent a valid shared NT handle that
    references a memory object. Ownership of this handle is not transferred
    to CUDA after the import operation, so the application must release the
    handle using the appropriate system call. If
    :py:obj:`~.CUDA_EXTERNAL_MEMORY_HANDLE_DESC`::handle::win32::name is
    not NULL, then it must point to a NULL-terminated array of UTF-16
    characters that refers to a memory object.

    If :py:obj:`~.CUDA_EXTERNAL_MEMORY_HANDLE_DESC.type` is
    :py:obj:`~.CU_EXTERNAL_MEMORY_HANDLE_TYPE_OPAQUE_WIN32_KMT`, then
    :py:obj:`~.CUDA_EXTERNAL_MEMORY_HANDLE_DESC`::handle::win32::handle
    must be non-NULL and
    :py:obj:`~.CUDA_EXTERNAL_MEMORY_HANDLE_DESC`::handle::win32::name must
    be NULL. The handle specified must be a globally shared KMT handle.
    This handle does not hold a reference to the underlying object, and
    thus will be invalid when all references to the memory object are
    destroyed.

    If :py:obj:`~.CUDA_EXTERNAL_MEMORY_HANDLE_DESC.type` is
    :py:obj:`~.CU_EXTERNAL_MEMORY_HANDLE_TYPE_D3D12_HEAP`, then exactly one
    of :py:obj:`~.CUDA_EXTERNAL_MEMORY_HANDLE_DESC`::handle::win32::handle
    and :py:obj:`~.CUDA_EXTERNAL_MEMORY_HANDLE_DESC`::handle::win32::name
    must not be NULL. If
    :py:obj:`~.CUDA_EXTERNAL_MEMORY_HANDLE_DESC`::handle::win32::handle is
    not NULL, then it must represent a valid shared NT handle that is
    returned by ID3D12Device::CreateSharedHandle when referring to a
    ID3D12Heap object. This handle holds a reference to the underlying
    object. If
    :py:obj:`~.CUDA_EXTERNAL_MEMORY_HANDLE_DESC`::handle::win32::name is
    not NULL, then it must point to a NULL-terminated array of UTF-16
    characters that refers to a ID3D12Heap object.

    If :py:obj:`~.CUDA_EXTERNAL_MEMORY_HANDLE_DESC.type` is
    :py:obj:`~.CU_EXTERNAL_MEMORY_HANDLE_TYPE_D3D12_RESOURCE`, then exactly
    one of
    :py:obj:`~.CUDA_EXTERNAL_MEMORY_HANDLE_DESC`::handle::win32::handle and
    :py:obj:`~.CUDA_EXTERNAL_MEMORY_HANDLE_DESC`::handle::win32::name must
    not be NULL. If
    :py:obj:`~.CUDA_EXTERNAL_MEMORY_HANDLE_DESC`::handle::win32::handle is
    not NULL, then it must represent a valid shared NT handle that is
    returned by ID3D12Device::CreateSharedHandle when referring to a
    ID3D12Resource object. This handle holds a reference to the underlying
    object. If
    :py:obj:`~.CUDA_EXTERNAL_MEMORY_HANDLE_DESC`::handle::win32::name is
    not NULL, then it must point to a NULL-terminated array of UTF-16
    characters that refers to a ID3D12Resource object.

    If :py:obj:`~.CUDA_EXTERNAL_MEMORY_HANDLE_DESC.type` is
    :py:obj:`~.CU_EXTERNAL_MEMORY_HANDLE_TYPE_D3D11_RESOURCE`, then
    :py:obj:`~.CUDA_EXTERNAL_MEMORY_HANDLE_DESC`::handle::win32::handle
    must represent a valid shared NT handle that is returned by
    IDXGIResource1::CreateSharedHandle when referring to a ID3D11Resource
    object. If
    :py:obj:`~.CUDA_EXTERNAL_MEMORY_HANDLE_DESC`::handle::win32::name is
    not NULL, then it must point to a NULL-terminated array of UTF-16
    characters that refers to a ID3D11Resource object.

    If :py:obj:`~.CUDA_EXTERNAL_MEMORY_HANDLE_DESC.type` is
    :py:obj:`~.CU_EXTERNAL_MEMORY_HANDLE_TYPE_D3D11_RESOURCE_KMT`, then
    :py:obj:`~.CUDA_EXTERNAL_MEMORY_HANDLE_DESC`::handle::win32::handle
    must represent a valid shared KMT handle that is returned by
    IDXGIResource::GetSharedHandle when referring to a ID3D11Resource
    object and
    :py:obj:`~.CUDA_EXTERNAL_MEMORY_HANDLE_DESC`::handle::win32::name must
    be NULL.

    If :py:obj:`~.CUDA_EXTERNAL_MEMORY_HANDLE_DESC.type` is
    :py:obj:`~.CU_EXTERNAL_MEMORY_HANDLE_TYPE_NVSCIBUF`, then
    :py:obj:`~.CUDA_EXTERNAL_MEMORY_HANDLE_DESC`::handle::nvSciBufObject
    must be non-NULL and reference a valid NvSciBuf object. If the NvSciBuf
    object imported into CUDA is also mapped by other drivers, then the
    application must use :py:obj:`~.cuWaitExternalSemaphoresAsync` or
    :py:obj:`~.cuSignalExternalSemaphoresAsync` as appropriate barriers to
    maintain coherence between CUDA and the other drivers. See
    :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_SIGNAL_SKIP_NVSCIBUF_MEMSYNC` and
    :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_WAIT_SKIP_NVSCIBUF_MEMSYNC` for
    memory synchronization.

    The size of the memory object must be specified in
    :py:obj:`~.CUDA_EXTERNAL_MEMORY_HANDLE_DESC.size`.

    Specifying the flag :py:obj:`~.CUDA_EXTERNAL_MEMORY_DEDICATED` in
    :py:obj:`~.CUDA_EXTERNAL_MEMORY_HANDLE_DESC.flags` indicates that the
    resource is a dedicated resource. The definition of what a dedicated
    resource is outside the scope of this extension. This flag must be set
    if :py:obj:`~.CUDA_EXTERNAL_MEMORY_HANDLE_DESC.type` is one of the
    following: :py:obj:`~.CU_EXTERNAL_MEMORY_HANDLE_TYPE_D3D12_RESOURCE`
    :py:obj:`~.CU_EXTERNAL_MEMORY_HANDLE_TYPE_D3D11_RESOURCE`
    :py:obj:`~.CU_EXTERNAL_MEMORY_HANDLE_TYPE_D3D11_RESOURCE_KMT`

    Parameters
    ----------
    memHandleDesc : :py:obj:`~.CUDA_EXTERNAL_MEMORY_HANDLE_DESC`
        Memory import handle descriptor

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`
    extMem_out : :py:obj:`~.CUexternalMemory`
        Returned handle to an external memory object

    See Also
    --------
    :py:obj:`~.cuDestroyExternalMemory`, :py:obj:`~.cuExternalMemoryGetMappedBuffer`, :py:obj:`~.cuExternalMemoryGetMappedMipmappedArray`

    Notes
    -----
    If the Vulkan memory imported into CUDA is mapped on the CPU then the application must use vkInvalidateMappedMemoryRanges/vkFlushMappedMemoryRanges as well as appropriate Vulkan pipeline barriers to maintain coherence between CPU and GPU. For more information on these APIs, please refer to "Synchronization
    and Cache Control" chapter from Vulkan specification.
    """
    cdef CUexternalMemory extMem_out = CUexternalMemory()
    cdef cydriver.CUDA_EXTERNAL_MEMORY_HANDLE_DESC* cymemHandleDesc_ptr = memHandleDesc._ptr if memHandleDesc != None else NULL
    err = cydriver.cuImportExternalMemory(<cydriver.CUexternalMemory*>extMem_out._ptr, cymemHandleDesc_ptr)
    return (CUresult(err), extMem_out)

@cython.embedsignature(True)
def cuExternalMemoryGetMappedBuffer(extMem, bufferDesc : Optional[CUDA_EXTERNAL_MEMORY_BUFFER_DESC]):
    """ Maps a buffer onto an imported memory object.

    Maps a buffer onto an imported memory object and returns a device
    pointer in `devPtr`.

    The properties of the buffer being mapped must be described in
    `bufferDesc`. The :py:obj:`~.CUDA_EXTERNAL_MEMORY_BUFFER_DESC`
    structure is defined as follows:

    **View CUDA Toolkit Documentation for a C++ code example**

    where :py:obj:`~.CUDA_EXTERNAL_MEMORY_BUFFER_DESC.offset` is the offset
    in the memory object where the buffer's base address is.
    :py:obj:`~.CUDA_EXTERNAL_MEMORY_BUFFER_DESC.size` is the size of the
    buffer. :py:obj:`~.CUDA_EXTERNAL_MEMORY_BUFFER_DESC.flags` must be
    zero.

    The offset and size have to be suitably aligned to match the
    requirements of the external API. Mapping two buffers whose ranges
    overlap may or may not result in the same virtual address being
    returned for the overlapped portion. In such cases, the application
    must ensure that all accesses to that region from the GPU are volatile.
    Otherwise writes made via one address are not guaranteed to be visible
    via the other address, even if they're issued by the same thread. It is
    recommended that applications map the combined range instead of mapping
    separate buffers and then apply the appropriate offsets to the returned
    pointer to derive the individual buffers.

    The returned pointer `devPtr` must be freed using
    :py:obj:`~.cuMemFree`.

    Parameters
    ----------
    extMem : :py:obj:`~.CUexternalMemory`
        Handle to external memory object
    bufferDesc : :py:obj:`~.CUDA_EXTERNAL_MEMORY_BUFFER_DESC`
        Buffer descriptor

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`
    devPtr : :py:obj:`~.CUdeviceptr`
        Returned device pointer to buffer

    See Also
    --------
    :py:obj:`~.cuImportExternalMemory`, :py:obj:`~.cuDestroyExternalMemory`, :py:obj:`~.cuExternalMemoryGetMappedMipmappedArray`
    """
    cdef cydriver.CUexternalMemory cyextMem
    if extMem is None:
        cyextMem = <cydriver.CUexternalMemory><void_ptr>0
    elif isinstance(extMem, (CUexternalMemory,)):
        pextMem = int(extMem)
        cyextMem = <cydriver.CUexternalMemory><void_ptr>pextMem
    else:
        pextMem = int(CUexternalMemory(extMem))
        cyextMem = <cydriver.CUexternalMemory><void_ptr>pextMem
    cdef CUdeviceptr devPtr = CUdeviceptr()
    cdef cydriver.CUDA_EXTERNAL_MEMORY_BUFFER_DESC* cybufferDesc_ptr = bufferDesc._ptr if bufferDesc != None else NULL
    err = cydriver.cuExternalMemoryGetMappedBuffer(<cydriver.CUdeviceptr*>devPtr._ptr, cyextMem, cybufferDesc_ptr)
    return (CUresult(err), devPtr)

@cython.embedsignature(True)
def cuExternalMemoryGetMappedMipmappedArray(extMem, mipmapDesc : Optional[CUDA_EXTERNAL_MEMORY_MIPMAPPED_ARRAY_DESC]):
    """ Maps a CUDA mipmapped array onto an external memory object.

    Maps a CUDA mipmapped array onto an external object and returns a
    handle to it in `mipmap`.

    The properties of the CUDA mipmapped array being mapped must be
    described in `mipmapDesc`. The structure
    :py:obj:`~.CUDA_EXTERNAL_MEMORY_MIPMAPPED_ARRAY_DESC` is defined as
    follows:

    **View CUDA Toolkit Documentation for a C++ code example**

    where :py:obj:`~.CUDA_EXTERNAL_MEMORY_MIPMAPPED_ARRAY_DESC.offset` is
    the offset in the memory object where the base level of the mipmap
    chain is.
    :py:obj:`~.CUDA_EXTERNAL_MEMORY_MIPMAPPED_ARRAY_DESC.arrayDesc`
    describes the format, dimensions and type of the base level of the
    mipmap chain. For further details on these parameters, please refer to
    the documentation for :py:obj:`~.cuMipmappedArrayCreate`. Note that if
    the mipmapped array is bound as a color target in the graphics API,
    then the flag :py:obj:`~.CUDA_ARRAY3D_COLOR_ATTACHMENT` must be
    specified in
    :py:obj:`~.CUDA_EXTERNAL_MEMORY_MIPMAPPED_ARRAY_DESC`::arrayDesc::Flags.
    :py:obj:`~.CUDA_EXTERNAL_MEMORY_MIPMAPPED_ARRAY_DESC.numLevels`
    specifies the total number of levels in the mipmap chain.

    If `extMem` was imported from a handle of type
    :py:obj:`~.CU_EXTERNAL_MEMORY_HANDLE_TYPE_NVSCIBUF`, then
    :py:obj:`~.CUDA_EXTERNAL_MEMORY_MIPMAPPED_ARRAY_DESC.numLevels` must be
    equal to 1.

    The returned CUDA mipmapped array must be freed using
    :py:obj:`~.cuMipmappedArrayDestroy`.

    Parameters
    ----------
    extMem : :py:obj:`~.CUexternalMemory`
        Handle to external memory object
    mipmapDesc : :py:obj:`~.CUDA_EXTERNAL_MEMORY_MIPMAPPED_ARRAY_DESC`
        CUDA array descriptor

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_VALUE`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`
    mipmap : :py:obj:`~.CUmipmappedArray`
        Returned CUDA mipmapped array

    See Also
    --------
    :py:obj:`~.cuImportExternalMemory`, :py:obj:`~.cuDestroyExternalMemory`, :py:obj:`~.cuExternalMemoryGetMappedBuffer`

    Notes
    -----
    On Tegra devices, this API will always attempt to do a compressed mapping when the `extMem` is imported from a handle of type :py:obj:`~.CU_EXTERNAL_MEMORY_HANDLE_TYPE_OPAQUE_FD`
    """
    cdef cydriver.CUexternalMemory cyextMem
    if extMem is None:
        cyextMem = <cydriver.CUexternalMemory><void_ptr>0
    elif isinstance(extMem, (CUexternalMemory,)):
        pextMem = int(extMem)
        cyextMem = <cydriver.CUexternalMemory><void_ptr>pextMem
    else:
        pextMem = int(CUexternalMemory(extMem))
        cyextMem = <cydriver.CUexternalMemory><void_ptr>pextMem
    cdef CUmipmappedArray mipmap = CUmipmappedArray()
    cdef cydriver.CUDA_EXTERNAL_MEMORY_MIPMAPPED_ARRAY_DESC* cymipmapDesc_ptr = mipmapDesc._ptr if mipmapDesc != None else NULL
    err = cydriver.cuExternalMemoryGetMappedMipmappedArray(<cydriver.CUmipmappedArray*>mipmap._ptr, cyextMem, cymipmapDesc_ptr)
    return (CUresult(err), mipmap)

@cython.embedsignature(True)
def cuDestroyExternalMemory(extMem):
    """ Destroys an external memory object.

    Destroys the specified external memory object. Any existing buffers and
    CUDA mipmapped arrays mapped onto this object must no longer be used
    and must be explicitly freed using :py:obj:`~.cuMemFree` and
    :py:obj:`~.cuMipmappedArrayDestroy` respectively.

    Parameters
    ----------
    extMem : :py:obj:`~.CUexternalMemory`
        External memory object to be destroyed

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`

    See Also
    --------
    :py:obj:`~.cuImportExternalMemory`, :py:obj:`~.cuExternalMemoryGetMappedBuffer`, :py:obj:`~.cuExternalMemoryGetMappedMipmappedArray`
    """
    cdef cydriver.CUexternalMemory cyextMem
    if extMem is None:
        cyextMem = <cydriver.CUexternalMemory><void_ptr>0
    elif isinstance(extMem, (CUexternalMemory,)):
        pextMem = int(extMem)
        cyextMem = <cydriver.CUexternalMemory><void_ptr>pextMem
    else:
        pextMem = int(CUexternalMemory(extMem))
        cyextMem = <cydriver.CUexternalMemory><void_ptr>pextMem
    err = cydriver.cuDestroyExternalMemory(cyextMem)
    return (CUresult(err),)

@cython.embedsignature(True)
def cuImportExternalSemaphore(semHandleDesc : Optional[CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC]):
    """ Imports an external semaphore.

    Imports an externally allocated synchronization object and returns a
    handle to that in `extSem_out`.

    The properties of the handle being imported must be described in
    `semHandleDesc`. The :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC` is
    defined as follows:

    **View CUDA Toolkit Documentation for a C++ code example**

    where :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC.type` specifies
    the type of handle being imported.
    :py:obj:`~.CUexternalSemaphoreHandleType` is defined as:

    **View CUDA Toolkit Documentation for a C++ code example**

    If :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC.type` is
    :py:obj:`~.CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_OPAQUE_FD`, then
    :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC`::handle::fd must be a
    valid file descriptor referencing a synchronization object. Ownership
    of the file descriptor is transferred to the CUDA driver when the
    handle is imported successfully. Performing any operations on the file
    descriptor after it is imported results in undefined behavior.

    If :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC.type` is
    :py:obj:`~.CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_OPAQUE_WIN32`, then
    exactly one of
    :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC`::handle::win32::handle
    and
    :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC`::handle::win32::name
    must not be NULL. If
    :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC`::handle::win32::handle
    is not NULL, then it must represent a valid shared NT handle that
    references a synchronization object. Ownership of this handle is not
    transferred to CUDA after the import operation, so the application must
    release the handle using the appropriate system call. If
    :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC`::handle::win32::name is
    not NULL, then it must name a valid synchronization object.

    If :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC.type` is
    :py:obj:`~.CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_OPAQUE_WIN32_KMT`, then
    :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC`::handle::win32::handle
    must be non-NULL and
    :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC`::handle::win32::name
    must be NULL. The handle specified must be a globally shared KMT
    handle. This handle does not hold a reference to the underlying object,
    and thus will be invalid when all references to the synchronization
    object are destroyed.

    If :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC.type` is
    :py:obj:`~.CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_D3D12_FENCE`, then exactly
    one of
    :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC`::handle::win32::handle
    and
    :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC`::handle::win32::name
    must not be NULL. If
    :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC`::handle::win32::handle
    is not NULL, then it must represent a valid shared NT handle that is
    returned by ID3D12Device::CreateSharedHandle when referring to a
    ID3D12Fence object. This handle holds a reference to the underlying
    object. If
    :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC`::handle::win32::name is
    not NULL, then it must name a valid synchronization object that refers
    to a valid ID3D12Fence object.

    If :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC.type` is
    :py:obj:`~.CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_D3D11_FENCE`, then
    :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC`::handle::win32::handle
    represents a valid shared NT handle that is returned by
    ID3D11Fence::CreateSharedHandle. If
    :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC`::handle::win32::name is
    not NULL, then it must name a valid synchronization object that refers
    to a valid ID3D11Fence object.

    If :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC.type` is
    :py:obj:`~.CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_NVSCISYNC`, then
    :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC`::handle::nvSciSyncObj
    represents a valid NvSciSyncObj.

    :py:obj:`~.CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_D3D11_KEYED_MUTEX`, then
    :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC`::handle::win32::handle
    represents a valid shared NT handle that is returned by
    IDXGIResource1::CreateSharedHandle when referring to a IDXGIKeyedMutex
    object. If
    :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC`::handle::win32::name is
    not NULL, then it must name a valid synchronization object that refers
    to a valid IDXGIKeyedMutex object.

    If :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC.type` is
    :py:obj:`~.CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_D3D11_KEYED_MUTEX_KMT`,
    then
    :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC`::handle::win32::handle
    represents a valid shared KMT handle that is returned by
    IDXGIResource::GetSharedHandle when referring to a IDXGIKeyedMutex
    object and
    :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC`::handle::win32::name
    must be NULL.

    If :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC.type` is
    :py:obj:`~.CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_TIMELINE_SEMAPHORE_FD`,
    then :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC`::handle::fd must
    be a valid file descriptor referencing a synchronization object.
    Ownership of the file descriptor is transferred to the CUDA driver when
    the handle is imported successfully. Performing any operations on the
    file descriptor after it is imported results in undefined behavior.

    If :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC.type` is
    :py:obj:`~.CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_TIMELINE_SEMAPHORE_WIN32`,
    then exactly one of
    :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC`::handle::win32::handle
    and
    :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC`::handle::win32::name
    must not be NULL. If
    :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC`::handle::win32::handle
    is not NULL, then it must represent a valid shared NT handle that
    references a synchronization object. Ownership of this handle is not
    transferred to CUDA after the import operation, so the application must
    release the handle using the appropriate system call. If
    :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC`::handle::win32::name is
    not NULL, then it must name a valid synchronization object.

    Parameters
    ----------
    semHandleDesc : :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC`
        Semaphore import handle descriptor

    Returns
    -------
    CUresult
        :py:obj:`~.CUDA_SUCCESS`, :py:obj:`~.CUDA_ERROR_NOT_INITIALIZED`, :py:obj:`~.CUDA_ERROR_NOT_SUPPORTED`, :py:obj:`~.CUDA_ERROR_INVALID_HANDLE`
    extSem_out : :py:obj:`~.CUexternalSemaphore`
        Returned handle to an external semaphore

    See Also
    --------
    :py:obj:`~.cuDestroyExternalSemaphore`, :py:obj:`~.cuSignalExternalSemaphoresAsync`, :py:obj:`~.cuWaitExternalSemaphoresAsync`
    """
    cdef CUexternalSemaphore extSem_out = CUexternalSemaphore()
    cdef cydriver.CUDA_EXTERNAL_SEMAPHORE_HANDLE_DESC* cysemHandleDesc_ptr = semHandleDesc._ptr if semHandleDesc != None else NULL
    err = cydriver.cuImportExternalSemaphore(<cydriver.CUexternalSemaphore*>extSem_out._ptr, cysemHandleDesc_ptr)
    return (CUresult(err), extSem_out)

@cython.embedsignature(True)
def cuSignalExternalSemaphoresAsync(extSemArray : Optional[Tuple[CUexternalSemaphore] | List[CUexternalSemaphore]], paramsArray : Optional[Tuple[CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS] | List[CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS]], unsigned int numExtSems, stream):
    """ Signals a set of external semaphore objects.

    Enqueues a signal operation on a set of externally allocated semaphore
    object in the specified stream. The operations will be executed when
    all prior operations in the stream complete.

    The exact semantics of signaling a semaphore depends on the type of the
    object.

    If the semaphore object is any one of the following types:
    :py:obj:`~.CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_OPAQUE_FD`,
    :py:obj:`~.CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_OPAQUE_WIN32`,
    :py:obj:`~.CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_OPAQUE_WIN32_KMT` then
    signaling the semaphore will set it to the signaled state.

    If the semaphore object is any one of the following types:
    :py:obj:`~.CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_D3D12_FENCE`,
    :py:obj:`~.CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_D3D11_FENCE`,
    :py:obj:`~.CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_TIMELINE_SEMAPHORE_FD`,
    :py:obj:`~.CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_TIMELINE_SEMAPHORE_WIN32`
    then the semaphore will be set to the value specified in
    :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS`::params::fence::value.

    If the semaphore object is of the type
    :py:obj:`~.CU_EXTERNAL_SEMAPHORE_HANDLE_TYPE_NVSCISYNC` this API sets
    :py:obj:`~.CUDA_EXTERNAL_SEMAPHORE_SIGNAL_PARAMS`::params::nvSciSync::fence
    to a value that can be used by subsequent waiters of the same NvSciSync
    object to order operations with those