"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.App = void 0;
const cxapi = require("@aws-cdk/cx-api");
const tree_metadata_1 = require("./private/tree-metadata");
const stage_1 = require("./stage");
const APP_SYMBOL = Symbol.for('@aws-cdk/core.App');
/**
 * A construct which represents an entire CDK app. This construct is normally
 * the root of the construct tree.
 *
 * You would normally define an `App` instance in your program's entrypoint,
 * then define constructs where the app is used as the parent scope.
 *
 * After all the child constructs are defined within the app, you should call
 * `app.synth()` which will emit a "cloud assembly" from this app into the
 * directory specified by `outdir`. Cloud assemblies includes artifacts such as
 * CloudFormation templates and assets that are needed to deploy this app into
 * the AWS cloud.
 *
 * @see https://docs.aws.amazon.com/cdk/latest/guide/apps.html
 */
class App extends stage_1.Stage {
    /**
     * Checks if an object is an instance of the `App` class.
     * @returns `true` if `obj` is an `App`.
     * @param obj The object to evaluate
     */
    static isApp(obj) {
        return APP_SYMBOL in obj;
    }
    /**
     * Initializes a CDK application.
     * @param props initialization properties
     */
    constructor(props = {}) {
        var _a;
        super(undefined, '', {
            outdir: (_a = props.outdir) !== null && _a !== void 0 ? _a : process.env[cxapi.OUTDIR_ENV],
        });
        Object.defineProperty(this, APP_SYMBOL, { value: true });
        this.loadContext(props.context);
        if (props.stackTraces === false) {
            this.node.setContext(cxapi.DISABLE_METADATA_STACK_TRACE, true);
        }
        if (props.runtimeInfo === false) {
            this.node.setContext(cxapi.DISABLE_VERSION_REPORTING, true);
        }
        const autoSynth = props.autoSynth !== undefined ? props.autoSynth : cxapi.OUTDIR_ENV in process.env;
        if (autoSynth) {
            // synth() guarantuees it will only execute once, so a default of 'true'
            // doesn't bite manual calling of the function.
            process.once('beforeExit', () => this.synth());
        }
        if (props.treeMetadata === undefined || props.treeMetadata) {
            new tree_metadata_1.TreeMetadata(this);
        }
    }
    loadContext(defaults = {}) {
        // prime with defaults passed through constructor
        for (const [k, v] of Object.entries(defaults)) {
            this.node.setContext(k, v);
        }
        // read from environment
        const contextJson = process.env[cxapi.CONTEXT_ENV];
        const contextFromEnvironment = contextJson
            ? JSON.parse(contextJson)
            : {};
        for (const [k, v] of Object.entries(contextFromEnvironment)) {
            this.node.setContext(k, v);
        }
    }
}
exports.App = App;
//# sourceMappingURL=data:application/json;base64,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