/**
 * Bundling options
 *
 * @experimental
 */
export interface BundlingOptions {
    /**
     * The Docker image where the command will run.
     */
    readonly image: BundlingDockerImage;
    /**
     * The command to run in the container.
     *
     * @example ['npm', 'install']
     *
     * @see https://docs.docker.com/engine/reference/run/
     *
     * @default - run the command defined in the image
     */
    readonly command?: string[];
    /**
     * Additional Docker volumes to mount.
     *
     * @default - no additional volumes are mounted
     */
    readonly volumes?: DockerVolume[];
    /**
     * The environment variables to pass to the container.
     *
     * @default - no environment variables.
     */
    readonly environment?: {
        [key: string]: string;
    };
    /**
     * Working directory inside the container.
     *
     * @default /asset-input
     */
    readonly workingDirectory?: string;
    /**
     * The user to use when running the container.
     *
     *   user | user:group | uid | uid:gid | user:gid | uid:group
     *
     * @see https://docs.docker.com/engine/reference/run/#user
     *
     * @default - uid:gid of the current user or 1000:1000 on Windows
     */
    readonly user?: string;
}
/**
 * A Docker image used for asset bundling
 */
export declare class BundlingDockerImage {
    readonly image: string;
    /**
     * Reference an image on DockerHub or another online registry.
     *
     * @param image the image name
     */
    static fromRegistry(image: string): BundlingDockerImage;
    /**
     * Reference an image that's built directly from sources on disk.
     *
     * @param path The path to the directory containing the Docker file
     * @param options Docker build options
     */
    static fromAsset(path: string, options?: DockerBuildOptions): BundlingDockerImage;
    /** @param image The Docker image */
    private constructor();
}
/**
 * A Docker volume
 */
export interface DockerVolume {
    /**
     * The path to the file or directory on the host machine
     */
    readonly hostPath: string;
    /**
     * The path where the file or directory is mounted in the container
     */
    readonly containerPath: string;
}
/**
 * Docker build options
 */
export interface DockerBuildOptions {
    /**
     * Build args
     *
     * @default - no build args
     */
    readonly buildArgs?: {
        [key: string]: string;
    };
}
