"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FromCloudFormation = void 0;
const cfn_fn_1 = require("./cfn-fn");
const cfn_pseudo_1 = require("./cfn-pseudo");
const cfn_resource_policy_1 = require("./cfn-resource-policy");
const token_1 = require("./token");
/**
 * This class contains functions for translating from a pure CFN value
 * (like a JS object { "Ref": "Bucket" })
 * to a form CDK understands
 * (like Fn.ref('Bucket')).
 *
 * While this file not exported from the module
 * (to not make it part of the public API),
 * it is directly referenced in the generated L1 code,
 * so any renames of it need to be reflected in cfn2ts/codegen.ts as well.
 *
 * @experimental
 */
class FromCloudFormation {
    static parseValue(cfnValue) {
        return parseCfnValueToCdkValue(cfnValue);
    }
    // nothing to for any but return it
    static getAny(value) { return value; }
    // nothing to do - if 'value' is not a boolean or a Token,
    // a validator should report that at runtime
    static getBoolean(value) { return value; }
    static getDate(value) {
        // if the date is a deploy-time value, just return it
        if (token_1.isResolvableObject(value)) {
            return value;
        }
        // if the date has been given as a string, convert it
        if (typeof value === 'string') {
            return new Date(value);
        }
        // all other cases - just return the value,
        // if it's not a Date, a validator should catch it
        return value;
    }
    static getString(value) {
        // if the string is a deploy-time value, serialize it to a Token
        if (token_1.isResolvableObject(value)) {
            return value.toString();
        }
        // in all other cases, just return the input,
        // and let a validator handle it if it's not a string
        return value;
    }
    static getNumber(value) {
        // if the string is a deploy-time value, serialize it to a Token
        if (token_1.isResolvableObject(value)) {
            return token_1.Token.asNumber(value);
        }
        // in all other cases, just return the input,
        // and let a validator handle it if it's not a number
        return value;
    }
    static getStringArray(value) {
        // if the array is a deploy-time value, serialize it to a Token
        if (token_1.isResolvableObject(value)) {
            return token_1.Token.asList(value);
        }
        // in all other cases, delegate to the standard mapping logic
        return this.getArray(value, this.getString);
    }
    static getArray(value, mapper) {
        if (!Array.isArray(value)) {
            // break the type system, and just return the given value,
            // which hopefully will be reported as invalid by the validator
            // of the property we're transforming
            // (unless it's a deploy-time value,
            // which we can't map over at build time anyway)
            return value;
        }
        return value.map(mapper);
    }
    static getMap(value, mapper) {
        if (typeof value !== 'object') {
            // if the input is not a map (= object in JS land),
            // just return it, and let the validator of this property handle it
            // (unless it's a deploy-time value,
            // which we can't map over at build time anyway)
            return value;
        }
        const ret = {};
        for (const [key, val] of Object.entries(value)) {
            ret[key] = mapper(val);
        }
        return ret;
    }
    static parseCreationPolicy(policy) {
        if (typeof policy !== 'object') {
            return undefined;
        }
        // change simple JS values to their CDK equivalents
        policy = FromCloudFormation.parseValue(policy);
        return undefinedIfAllValuesAreEmpty({
            autoScalingCreationPolicy: parseAutoScalingCreationPolicy(policy.AutoScalingCreationPolicy),
            resourceSignal: parseResourceSignal(policy.ResourceSignal),
        });
        function parseAutoScalingCreationPolicy(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return undefinedIfAllValuesAreEmpty({
                minSuccessfulInstancesPercent: FromCloudFormation.getNumber(p.MinSuccessfulInstancesPercent),
            });
        }
        function parseResourceSignal(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return undefinedIfAllValuesAreEmpty({
                count: FromCloudFormation.getNumber(p.Count),
                timeout: FromCloudFormation.getString(p.Timeout),
            });
        }
    }
    static parseUpdatePolicy(policy) {
        if (typeof policy !== 'object') {
            return undefined;
        }
        // change simple JS values to their CDK equivalents
        policy = FromCloudFormation.parseValue(policy);
        return undefinedIfAllValuesAreEmpty({
            autoScalingReplacingUpdate: parseAutoScalingReplacingUpdate(policy.AutoScalingReplacingUpdate),
            autoScalingRollingUpdate: parseAutoScalingRollingUpdate(policy.AutoScalingRollingUpdate),
            autoScalingScheduledAction: parseAutoScalingScheduledAction(policy.AutoScalingScheduledAction),
            codeDeployLambdaAliasUpdate: parseCodeDeployLambdaAliasUpdate(policy.CodeDeployLambdaAliasUpdate),
            enableVersionUpgrade: policy.EnableVersionUpgrade,
            useOnlineResharding: policy.UseOnlineResharding,
        });
        function parseAutoScalingReplacingUpdate(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return undefinedIfAllValuesAreEmpty({
                willReplace: p.WillReplace,
            });
        }
        function parseAutoScalingRollingUpdate(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return undefinedIfAllValuesAreEmpty({
                maxBatchSize: FromCloudFormation.getNumber(p.MaxBatchSize),
                minInstancesInService: FromCloudFormation.getNumber(p.MinInstancesInService),
                minSuccessfulInstancesPercent: FromCloudFormation.getNumber(p.MinSuccessfulInstancesPercent),
                pauseTime: FromCloudFormation.getString(p.PauseTime),
                suspendProcesses: FromCloudFormation.getStringArray(p.SuspendProcesses),
                waitOnResourceSignals: p.WaitOnResourceSignals,
            });
        }
        function parseCodeDeployLambdaAliasUpdate(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return {
                beforeAllowTrafficHook: FromCloudFormation.getString(p.BeforeAllowTrafficHook),
                afterAllowTrafficHook: FromCloudFormation.getString(p.AfterAllowTrafficHook),
                applicationName: FromCloudFormation.getString(p.ApplicationName),
                deploymentGroupName: FromCloudFormation.getString(p.DeploymentGroupName),
            };
        }
        function parseAutoScalingScheduledAction(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return undefinedIfAllValuesAreEmpty({
                ignoreUnmodifiedGroupSizeProperties: p.IgnoreUnmodifiedGroupSizeProperties,
            });
        }
    }
    static parseDeletionPolicy(policy) {
        switch (policy) {
            case null: return undefined;
            case undefined: return undefined;
            case 'Delete': return cfn_resource_policy_1.CfnDeletionPolicy.DELETE;
            case 'Retain': return cfn_resource_policy_1.CfnDeletionPolicy.RETAIN;
            case 'Snapshot': return cfn_resource_policy_1.CfnDeletionPolicy.SNAPSHOT;
            default: throw new Error(`Unrecognized DeletionPolicy '${policy}'`);
        }
    }
    static getCfnTag(tag) {
        return tag == null
            ? {} // break the type system - this should be detected at runtime by a tag validator
            : {
                key: tag.Key,
                value: tag.Value,
            };
    }
}
exports.FromCloudFormation = FromCloudFormation;
function parseCfnValueToCdkValue(cfnValue) {
    // == null captures undefined as well
    if (cfnValue == null) {
        return undefined;
    }
    // if we have any late-bound values,
    // just return them
    if (token_1.isResolvableObject(cfnValue)) {
        return cfnValue;
    }
    if (Array.isArray(cfnValue)) {
        return cfnValue.map(el => parseCfnValueToCdkValue(el));
    }
    if (typeof cfnValue === 'object') {
        // an object can be either a CFN intrinsic, or an actual object
        const cfnIntrinsic = parseIfCfnIntrinsic(cfnValue);
        if (cfnIntrinsic) {
            return cfnIntrinsic;
        }
        const ret = {};
        for (const [key, val] of Object.entries(cfnValue)) {
            ret[key] = parseCfnValueToCdkValue(val);
        }
        return ret;
    }
    // in all other cases, just return the input
    return cfnValue;
}
function parseIfCfnIntrinsic(object) {
    var _a;
    const key = looksLikeCfnIntrinsic(object);
    switch (key) {
        case undefined:
            return undefined;
        case 'Ref': {
            // ToDo handle translating logical IDs
            return (_a = specialCaseRefs(object[key])) !== null && _a !== void 0 ? _a : cfn_fn_1.Fn._ref(object[key]);
        }
        case 'Fn::GetAtt': {
            // Fn::GetAtt takes a 2-element list as its argument
            const value = object[key];
            // ToDo same comment here as in Ref above
            return cfn_fn_1.Fn.getAtt((value[0]), value[1]);
        }
        case 'Fn::Join': {
            // Fn::Join takes a 2-element list as its argument,
            // where the first element is the delimiter,
            // and the second is the list of elements to join
            const value = parseCfnValueToCdkValue(object[key]);
            return cfn_fn_1.Fn.join(value[0], value[1]);
        }
        case 'Fn::Cidr': {
            const value = parseCfnValueToCdkValue(object[key]);
            return cfn_fn_1.Fn.cidr(value[0], value[1], value[2]);
        }
        case 'Fn::FindInMap': {
            const value = parseCfnValueToCdkValue(object[key]);
            return cfn_fn_1.Fn.findInMap(value[0], value[1], value[2]);
        }
        case 'Fn::Select': {
            const value = parseCfnValueToCdkValue(object[key]);
            return cfn_fn_1.Fn.select(value[0], value[1]);
        }
        case 'Fn::GetAZs': {
            const value = parseCfnValueToCdkValue(object[key]);
            return cfn_fn_1.Fn.getAzs(value);
        }
        case 'Fn::ImportValue': {
            const value = parseCfnValueToCdkValue(object[key]);
            return cfn_fn_1.Fn.importValue(value);
        }
        case 'Fn::Split': {
            const value = parseCfnValueToCdkValue(object[key]);
            return cfn_fn_1.Fn.split(value[0], value[1]);
        }
        case 'Fn::Transform': {
            const value = parseCfnValueToCdkValue(object[key]);
            return cfn_fn_1.Fn.transform(value.Name, value.Parameters);
        }
        case 'Fn::Base64': {
            const value = parseCfnValueToCdkValue(object[key]);
            return cfn_fn_1.Fn.base64(value);
        }
        case 'Fn::If': {
            // Fn::If takes a 3-element list as its argument
            // ToDo the first argument is the name of the condition,
            // so we will need to retrieve the actual object from the template
            // when we handle preserveLogicalIds=false
            const value = parseCfnValueToCdkValue(object[key]);
            return cfn_fn_1.Fn.conditionIf(value[0], value[1], value[2]);
        }
        case 'Fn::Equals': {
            const value = parseCfnValueToCdkValue(object[key]);
            return cfn_fn_1.Fn.conditionEquals(value[0], value[1]);
        }
        case 'Fn::And': {
            const value = parseCfnValueToCdkValue(object[key]);
            return cfn_fn_1.Fn.conditionAnd(...value);
        }
        case 'Fn::Not': {
            const value = parseCfnValueToCdkValue(object[key]);
            return cfn_fn_1.Fn.conditionNot(value[0]);
        }
        case 'Fn::Or': {
            const value = parseCfnValueToCdkValue(object[key]);
            return cfn_fn_1.Fn.conditionOr(...value);
        }
        default:
            throw new Error(`Unsupported CloudFormation function '${key}'`);
    }
}
function looksLikeCfnIntrinsic(object) {
    const objectKeys = Object.keys(object);
    // a CFN intrinsic is always an object with a single key
    if (objectKeys.length !== 1) {
        return undefined;
    }
    const key = objectKeys[0];
    return key === 'Ref' || key.startsWith('Fn::') ? key : undefined;
}
function specialCaseRefs(value) {
    switch (value) {
        case 'AWS::AccountId': return cfn_pseudo_1.Aws.ACCOUNT_ID;
        case 'AWS::Region': return cfn_pseudo_1.Aws.REGION;
        case 'AWS::Partition': return cfn_pseudo_1.Aws.PARTITION;
        case 'AWS::URLSuffix': return cfn_pseudo_1.Aws.URL_SUFFIX;
        case 'AWS::NotificationARNs': return cfn_pseudo_1.Aws.NOTIFICATION_ARNS;
        case 'AWS::StackId': return cfn_pseudo_1.Aws.STACK_ID;
        case 'AWS::StackName': return cfn_pseudo_1.Aws.STACK_NAME;
        case 'AWS::NoValue': return cfn_pseudo_1.Aws.NO_VALUE;
        default: return undefined;
    }
}
function undefinedIfAllValuesAreEmpty(object) {
    return Object.values(object).some(v => v !== undefined) ? object : undefined;
}
//# sourceMappingURL=data:application/json;base64,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