"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isGeneratedWhenNeededMarker = exports.GeneratedWhenNeededMarker = exports.generatePhysicalName = void 0;
const crypto = require("crypto");
const stack_1 = require("../stack");
const token_1 = require("../token");
const token_map_1 = require("./token-map");
function generatePhysicalName(resource) {
    const stack = stack_1.Stack.of(resource);
    const stackPart = new PrefixNamePart(stack.stackName, 25);
    const idPart = new SuffixNamePart(resource.node.uniqueId, 24);
    const region = stack.region;
    if (token_1.Token.isUnresolved(region) || !region) {
        throw new Error(`Cannot generate a physical name for ${resource.node.path}, because the region is un-resolved or missing`);
    }
    const account = stack.account;
    if (token_1.Token.isUnresolved(account) || !account) {
        throw new Error(`Cannot generate a physical name for ${resource.node.path}, because the account is un-resolved or missing`);
    }
    const parts = [stackPart, idPart]
        .map(part => part.generate());
    const hashLength = 12;
    const sha256 = crypto.createHash('sha256')
        .update(stackPart.bareStr)
        .update(idPart.bareStr)
        .update(region)
        .update(account);
    const hash = sha256.digest('hex').slice(0, hashLength);
    const ret = [...parts, hash].join('');
    return ret.toLowerCase();
}
exports.generatePhysicalName = generatePhysicalName;
class NamePart {
    constructor(bareStr) {
        this.bareStr = bareStr;
    }
}
class PrefixNamePart extends NamePart {
    constructor(bareStr, prefixLength) {
        super(bareStr);
        this.prefixLength = prefixLength;
    }
    generate() {
        return this.bareStr.slice(0, this.prefixLength);
    }
}
class SuffixNamePart extends NamePart {
    constructor(str, suffixLength) {
        super(str);
        this.suffixLength = suffixLength;
    }
    generate() {
        const strLen = this.bareStr.length;
        const startIndex = Math.max(strLen - this.suffixLength, 0);
        return this.bareStr.slice(startIndex, strLen);
    }
}
const GENERATE_IF_NEEDED_SYMBOL = Symbol.for('@aws-cdk/core.<private>.GenerateIfNeeded');
/**
 * This marker token is used by PhysicalName.GENERATE_IF_NEEDED. When that token is passed to the
 * physicalName property of a Resource, it triggers different behavior in the Resource constructor
 * that will allow emission of a generated physical name (when the resource is used across
 * environments) or undefined (when the resource is not shared).
 *
 * This token throws an Error when it is resolved, as a way to prevent inadvertent mis-uses of it.
 */
class GeneratedWhenNeededMarker {
    constructor() {
        this.creationStack = [];
        Object.defineProperty(this, GENERATE_IF_NEEDED_SYMBOL, { value: true });
    }
    resolve(_ctx) {
        throw new Error('Invalid physical name passed to CloudFormation. Use "this.physicalName" instead');
    }
    toString() {
        return 'PhysicalName.GENERATE_IF_NEEDED';
    }
}
exports.GeneratedWhenNeededMarker = GeneratedWhenNeededMarker;
/**
 * Checks whether a stringified token resolves to a `GeneratedWhenNeededMarker`.
 */
function isGeneratedWhenNeededMarker(val) {
    const token = token_map_1.TokenMap.instance().lookupString(val);
    return !!token && GENERATE_IF_NEEDED_SYMBOL in token;
}
exports.isGeneratedWhenNeededMarker = isGeneratedWhenNeededMarker;
//# sourceMappingURL=data:application/json;base64,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