"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DefaultStackSynthesizer = exports.BOOTSTRAP_QUALIFIER_CONTEXT = void 0;
const asset_schema = require("@aws-cdk/cdk-assets-schema");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const cxapi = require("@aws-cdk/cx-api");
const fs = require("fs");
const path = require("path");
const assets_1 = require("../assets");
const cfn_fn_1 = require("../cfn-fn");
const token_1 = require("../token");
const _shared_1 = require("./_shared");
exports.BOOTSTRAP_QUALIFIER_CONTEXT = '@aws-cdk/core:bootstrapQualifier';
/**
 * The minimum bootstrap stack version required by this app.
 */
const MIN_BOOTSTRAP_STACK_VERSION = 3;
/**
 * Uses conventionally named roles and reify asset storage locations
 *
 * This synthesizer is the only StackSynthesizer that generates
 * an asset manifest, and is required to deploy CDK applications using the
 * `@aws-cdk/app-delivery` CI/CD library.
 *
 * Requires the environment to have been bootstrapped with Bootstrap Stack V2.
 */
class DefaultStackSynthesizer {
    constructor(props = {}) {
        this.props = props;
        this.files = {};
        this.dockerImages = {};
    }
    bind(stack) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j;
        this._stack = stack;
        const qualifier = (_b = (_a = this.props.qualifier) !== null && _a !== void 0 ? _a : stack.node.tryGetContext(exports.BOOTSTRAP_QUALIFIER_CONTEXT)) !== null && _b !== void 0 ? _b : DefaultStackSynthesizer.DEFAULT_QUALIFIER;
        // Function to replace placeholders in the input string as much as possible
        //
        // We replace:
        // - ${Qualifier}: always
        // - ${AWS::AccountId}, ${AWS::Region}: only if we have the actual values available
        // - ${AWS::Partition}: never, since we never have the actual partition value.
        const specialize = (s) => {
            s = replaceAll(s, '${Qualifier}', qualifier);
            return cxapi.EnvironmentPlaceholders.replace(s, {
                region: resolvedOr(stack.region, cxapi.EnvironmentPlaceholders.CURRENT_REGION),
                accountId: resolvedOr(stack.account, cxapi.EnvironmentPlaceholders.CURRENT_ACCOUNT),
                partition: cxapi.EnvironmentPlaceholders.CURRENT_PARTITION,
            });
        };
        // tslint:disable:max-line-length
        this.bucketName = specialize((_c = this.props.fileAssetsBucketName) !== null && _c !== void 0 ? _c : DefaultStackSynthesizer.DEFAULT_FILE_ASSETS_BUCKET_NAME);
        this.repositoryName = specialize((_d = this.props.imageAssetsRepositoryName) !== null && _d !== void 0 ? _d : DefaultStackSynthesizer.DEFAULT_IMAGE_ASSETS_REPOSITORY_NAME);
        this._deployRoleArn = specialize((_e = this.props.deployRoleArn) !== null && _e !== void 0 ? _e : DefaultStackSynthesizer.DEFAULT_DEPLOY_ROLE_ARN);
        this._cloudFormationExecutionRoleArn = specialize((_f = this.props.cloudFormationExecutionRole) !== null && _f !== void 0 ? _f : DefaultStackSynthesizer.DEFAULT_CLOUDFORMATION_ROLE_ARN);
        this.fileAssetPublishingRoleArn = specialize((_g = this.props.fileAssetPublishingRoleArn) !== null && _g !== void 0 ? _g : DefaultStackSynthesizer.DEFAULT_FILE_ASSET_PUBLISHING_ROLE_ARN);
        this.imageAssetPublishingRoleArn = specialize((_h = this.props.imageAssetPublishingRoleArn) !== null && _h !== void 0 ? _h : DefaultStackSynthesizer.DEFAULT_IMAGE_ASSET_PUBLISHING_ROLE_ARN);
        this._kmsKeyArnExportName = specialize((_j = this.props.fileAssetKeyArnExportName) !== null && _j !== void 0 ? _j : DefaultStackSynthesizer.DEFAULT_FILE_ASSET_KEY_ARN_EXPORT_NAME);
        // tslint:enable:max-line-length
    }
    addFileAsset(asset) {
        _shared_1.assertBound(this.stack);
        _shared_1.assertBound(this.bucketName);
        _shared_1.assertBound(this._kmsKeyArnExportName);
        const objectKey = asset.sourceHash + (asset.packaging === assets_1.FileAssetPackaging.ZIP_DIRECTORY ? '.zip' : '');
        // Add to manifest
        this.files[asset.sourceHash] = {
            source: {
                path: asset.fileName,
                packaging: asset.packaging,
            },
            destinations: {
                [this.manifestEnvName]: {
                    bucketName: this.bucketName,
                    objectKey,
                    region: resolvedOr(this.stack.region, undefined),
                    assumeRoleArn: this.fileAssetPublishingRoleArn,
                    assumeRoleExternalId: this.props.fileAssetPublishingExternalId,
                },
            },
        };
        const { region, urlSuffix } = stackLocationOrInstrinsics(this.stack);
        const httpUrl = cfnify(`https://s3.${region}.${urlSuffix}/${this.bucketName}/${objectKey}`);
        const s3ObjectUrl = cfnify(`s3://${this.bucketName}/${objectKey}`);
        // Return CFN expression
        return {
            bucketName: cfnify(this.bucketName),
            objectKey,
            httpUrl,
            s3ObjectUrl,
            s3Url: httpUrl,
            kmsKeyArn: cfn_fn_1.Fn.importValue(cfnify(this._kmsKeyArnExportName)),
        };
    }
    addDockerImageAsset(asset) {
        _shared_1.assertBound(this.stack);
        _shared_1.assertBound(this.repositoryName);
        const imageTag = asset.sourceHash;
        // Add to manifest
        this.dockerImages[asset.sourceHash] = {
            source: {
                directory: asset.directoryName,
                dockerBuildArgs: asset.dockerBuildArgs,
                dockerBuildTarget: asset.dockerBuildTarget,
                dockerFile: asset.dockerFile,
            },
            destinations: {
                [this.manifestEnvName]: {
                    repositoryName: this.repositoryName,
                    imageTag,
                    region: resolvedOr(this.stack.region, undefined),
                    assumeRoleArn: this.imageAssetPublishingRoleArn,
                    assumeRoleExternalId: this.props.imageAssetPublishingExternalId,
                },
            },
        };
        const { account, region, urlSuffix } = stackLocationOrInstrinsics(this.stack);
        // Return CFN expression
        return {
            repositoryName: cfnify(this.repositoryName),
            imageUri: cfnify(`${account}.dkr.ecr.${region}.${urlSuffix}/${this.repositoryName}:${imageTag}`),
        };
    }
    synthesizeStackArtifacts(session) {
        _shared_1.assertBound(this.stack);
        // Add the stack's template to the artifact manifest
        const templateManifestUrl = this.addStackTemplateToAssetManifest(session);
        const artifactId = this.writeAssetManifest(session);
        _shared_1.addStackArtifactToAssembly(session, this.stack, {
            assumeRoleArn: this._deployRoleArn,
            cloudFormationExecutionRoleArn: this._cloudFormationExecutionRoleArn,
            stackTemplateAssetObjectUrl: templateManifestUrl,
            requiresBootstrapStackVersion: MIN_BOOTSTRAP_STACK_VERSION,
        }, [artifactId]);
    }
    /**
     * Returns the ARN of the deploy Role.
     */
    get deployRoleArn() {
        if (!this._deployRoleArn) {
            throw new Error('deployRoleArn getter can only be called after the synthesizer has been bound to a Stack');
        }
        return this._deployRoleArn;
    }
    /**
     * Returns the ARN of the CFN execution Role.
     */
    get cloudFormationExecutionRoleArn() {
        if (!this._cloudFormationExecutionRoleArn) {
            throw new Error('cloudFormationExecutionRoleArn getter can only be called after the synthesizer has been bound to a Stack');
        }
        return this._cloudFormationExecutionRoleArn;
    }
    get stack() {
        return this._stack;
    }
    /**
     * Add the stack's template as one of the manifest assets
     *
     * This will make it get uploaded to S3 automatically by S3-assets. Return
     * the manifest URL.
     *
     * (We can't return the location returned from `addFileAsset`, as that
     * contains CloudFormation intrinsics which can't go into the manifest).
     */
    addStackTemplateToAssetManifest(session) {
        _shared_1.assertBound(this.stack);
        const templatePath = path.join(session.assembly.outdir, this.stack.templateFile);
        const template = fs.readFileSync(templatePath, { encoding: 'utf-8' });
        const sourceHash = _shared_1.contentHash(template);
        this.addFileAsset({
            fileName: this.stack.templateFile,
            packaging: assets_1.FileAssetPackaging.FILE,
            sourceHash,
        });
        // We should technically return an 'https://s3.REGION.amazonaws.com[.cn]/name/hash' URL here,
        // because that is what CloudFormation expects to see.
        //
        // However, there's no way for us to actually know the UrlSuffix a priori, so we can't construct it here.
        //
        // Instead, we'll have a protocol with the CLI that we put an 's3://.../...' URL here, and the CLI
        // is going to resolve it to the correct 'https://.../' URL before it gives it to CloudFormation.
        return `s3://${this.bucketName}/${sourceHash}`;
    }
    /**
     * Write an asset manifest to the Cloud Assembly, return the artifact IDs written
     */
    writeAssetManifest(session) {
        _shared_1.assertBound(this.stack);
        const artifactId = `${this.stack.artifactId}.assets`;
        const manifestFile = `${artifactId}.json`;
        const outPath = path.join(session.assembly.outdir, manifestFile);
        const manifest = {
            version: asset_schema.AssetManifestSchema.currentVersion(),
            files: this.files,
            dockerImages: this.dockerImages,
        };
        fs.writeFileSync(outPath, JSON.stringify(manifest, undefined, 2));
        session.assembly.addArtifact(artifactId, {
            type: cxschema.ArtifactType.ASSET_MANIFEST,
            properties: {
                file: manifestFile,
                requiresBootstrapStackVersion: MIN_BOOTSTRAP_STACK_VERSION,
            },
        });
        return artifactId;
    }
    get manifestEnvName() {
        _shared_1.assertBound(this.stack);
        return [
            resolvedOr(this.stack.account, 'current_account'),
            resolvedOr(this.stack.region, 'current_region'),
        ].join('-');
    }
}
exports.DefaultStackSynthesizer = DefaultStackSynthesizer;
/**
 * Default ARN qualifier
 */
DefaultStackSynthesizer.DEFAULT_QUALIFIER = 'hnb659fds';
/**
 * Default CloudFormation role ARN.
 */
DefaultStackSynthesizer.DEFAULT_CLOUDFORMATION_ROLE_ARN = 'arn:${AWS::Partition}:iam::${AWS::AccountId}:role/cdk-${Qualifier}-cfn-exec-role-${AWS::AccountId}-${AWS::Region}';
/**
 * Default deploy role ARN.
 */
DefaultStackSynthesizer.DEFAULT_DEPLOY_ROLE_ARN = 'arn:${AWS::Partition}:iam::${AWS::AccountId}:role/cdk-${Qualifier}-deploy-role-${AWS::AccountId}-${AWS::Region}';
/**
 * Default asset publishing role ARN for file (S3) assets.
 */
DefaultStackSynthesizer.DEFAULT_FILE_ASSET_PUBLISHING_ROLE_ARN = 'arn:${AWS::Partition}:iam::${AWS::AccountId}:role/cdk-${Qualifier}-file-publishing-role-${AWS::AccountId}-${AWS::Region}';
/**
 * Default asset publishing role ARN for image (ECR) assets.
 */
DefaultStackSynthesizer.DEFAULT_IMAGE_ASSET_PUBLISHING_ROLE_ARN = 'arn:${AWS::Partition}:iam::${AWS::AccountId}:role/cdk-${Qualifier}-image-publishing-role-${AWS::AccountId}-${AWS::Region}';
/**
 * Default image assets repository name
 */
DefaultStackSynthesizer.DEFAULT_IMAGE_ASSETS_REPOSITORY_NAME = 'cdk-${Qualifier}-container-assets-${AWS::AccountId}-${AWS::Region}';
/**
 * Default file assets bucket name
 */
DefaultStackSynthesizer.DEFAULT_FILE_ASSETS_BUCKET_NAME = 'cdk-${Qualifier}-assets-${AWS::AccountId}-${AWS::Region}';
/**
 * Name of the CloudFormation Export with the asset key name
 */
DefaultStackSynthesizer.DEFAULT_FILE_ASSET_KEY_ARN_EXPORT_NAME = 'CdkBootstrap-${Qualifier}-FileAssetKeyArn';
/**
 * Return the given value if resolved or fall back to a default
 */
function resolvedOr(x, def) {
    return token_1.Token.isUnresolved(x) ? def : x;
}
/**
 * A "replace-all" function that doesn't require us escaping a literal string to a regex
 */
function replaceAll(s, search, replace) {
    return s.split(search).join(replace);
}
/**
 * If the string still contains placeholders, wrap it in a Fn::Sub so they will be substituted at CFN deployment time
 *
 * (This happens to work because the placeholders we picked map directly onto CFN
 * placeholders. If they didn't we'd have to do a transformation here).
 */
function cfnify(s) {
    return s.indexOf('${') > -1 ? cfn_fn_1.Fn.sub(s) : s;
}
/**
 * Return the stack locations if they're concrete, or the original CFN intrisics otherwise
 *
 * We need to return these instead of the tokenized versions of the strings,
 * since we must accept those same ${AWS::AccountId}/${AWS::Region} placeholders
 * in bucket names and role names (in order to allow environment-agnostic stacks).
 *
 * We'll wrap a single {Fn::Sub} around the final string in order to replace everything,
 * but we can't have the token system render part of the string to {Fn::Join} because
 * the CFN specification doesn't allow the {Fn::Sub} template string to be an arbitrary
 * expression--it must be a string literal.
 */
function stackLocationOrInstrinsics(stack) {
    return {
        account: resolvedOr(stack.account, '${AWS::AccountId}'),
        region: resolvedOr(stack.region, '${AWS::Region}'),
        urlSuffix: resolvedOr(stack.urlSuffix, '${AWS::URLSuffix}'),
    };
}
//# sourceMappingURL=data:application/json;base64,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