"use strict";
const cxapi = require("@aws-cdk/cx-api");
const fs = require("fs");
const lib_1 = require("../../lib");
const evaluate_cfn_1 = require("../evaluate-cfn");
const CFN_CONTEXT = {
    'AWS::Region': 'the_region',
    'AWS::AccountId': 'the_account',
    'AWS::URLSuffix': 'domain.aws',
};
let app;
let stack;
/**
 * Evaluate a possibly string-containing value the same way CFN would do
 *
 * (Be invariant to the specific Fn::Sub or Fn::Join we would output)
 */
function evalCFN(value) {
    return evaluate_cfn_1.evaluateCFN(stack.resolve(value), CFN_CONTEXT);
}
function isAssetManifest(x) {
    return x instanceof cxapi.AssetManifestArtifact;
}
function readAssetManifest(asm) {
    const manifestArtifact = asm.artifacts.filter(isAssetManifest)[0];
    if (!manifestArtifact) {
        throw new Error('no asset manifest in assembly');
    }
    return JSON.parse(fs.readFileSync(manifestArtifact.file, { encoding: 'utf-8' }));
}
module.exports = {
    'setUp'(cb) {
        app = new lib_1.App({
            context: {
                [cxapi.NEW_STYLE_STACK_SYNTHESIS_CONTEXT]: 'true',
            },
        });
        stack = new lib_1.Stack(app, 'Stack');
        cb();
    },
    'stack template is in asset manifest'(test) {
        var _a;
        // GIVEN
        new lib_1.CfnResource(stack, 'Resource', {
            type: 'Some::Resource',
        });
        // WHEN
        const asm = app.synth();
        // THEN -- the S3 url is advertised on the stack artifact
        const stackArtifact = asm.getStackArtifact('Stack');
        test.equals(stackArtifact.stackTemplateAssetObjectUrl, 's3://cdk-hnb659fds-assets-${AWS::AccountId}-${AWS::Region}/4bdae6e3b1b15f08c889d6c9133f24731ee14827a9a9ab9b6b6a9b42b6d34910');
        // THEN - the template is in the asset manifest
        const manifestArtifact = asm.artifacts.filter(isAssetManifest)[0];
        test.ok(manifestArtifact);
        const manifest = JSON.parse(fs.readFileSync(manifestArtifact.file, { encoding: 'utf-8' }));
        const firstFile = (_a = (manifest.files ? manifest.files[Object.keys(manifest.files)[0]] : undefined)) !== null && _a !== void 0 ? _a : {};
        test.deepEqual(firstFile, {
            source: { path: 'Stack.template.json', packaging: 'file' },
            destinations: {
                'current_account-current_region': {
                    bucketName: 'cdk-hnb659fds-assets-${AWS::AccountId}-${AWS::Region}',
                    objectKey: '4bdae6e3b1b15f08c889d6c9133f24731ee14827a9a9ab9b6b6a9b42b6d34910',
                    assumeRoleArn: 'arn:${AWS::Partition}:iam::${AWS::AccountId}:role/cdk-hnb659fds-file-publishing-role-${AWS::AccountId}-${AWS::Region}',
                },
            },
        });
        test.done();
    },
    'add file asset'(test) {
        // WHEN
        const location = stack.synthesizer.addFileAsset({
            fileName: __filename,
            packaging: lib_1.FileAssetPackaging.FILE,
            sourceHash: 'abcdef',
        });
        // THEN - we have a fixed asset location with region placeholders
        test.equals(evalCFN(location.bucketName), 'cdk-hnb659fds-assets-the_account-the_region');
        test.equals(evalCFN(location.s3Url), 'https://s3.the_region.domain.aws/cdk-hnb659fds-assets-the_account-the_region/abcdef');
        // THEN - object key contains source hash somewhere
        test.ok(location.objectKey.indexOf('abcdef') > -1);
        test.done();
    },
    'add docker image asset'(test) {
        // WHEN
        const location = stack.synthesizer.addDockerImageAsset({
            directoryName: '.',
            sourceHash: 'abcdef',
        });
        // THEN - we have a fixed asset location with region placeholders
        test.equals(evalCFN(location.repositoryName), 'cdk-hnb659fds-container-assets-the_account-the_region');
        test.equals(evalCFN(location.imageUri), 'the_account.dkr.ecr.the_region.domain.aws/cdk-hnb659fds-container-assets-the_account-the_region:abcdef');
        test.done();
    },
    'synthesis'(test) {
        var _a, _b;
        // GIVEN
        stack.synthesizer.addFileAsset({
            fileName: __filename,
            packaging: lib_1.FileAssetPackaging.FILE,
            sourceHash: 'abcdef',
        });
        stack.synthesizer.addDockerImageAsset({
            directoryName: '.',
            sourceHash: 'abcdef',
        });
        // WHEN
        const asm = app.synth();
        // THEN - we have an asset manifest with both assets and the stack template in there
        const manifest = readAssetManifest(asm);
        test.equals(Object.keys(manifest.files || {}).length, 2);
        test.equals(Object.keys(manifest.dockerImages || {}).length, 1);
        // THEN - every artifact has an assumeRoleArn
        for (const file of Object.values((_a = manifest.files) !== null && _a !== void 0 ? _a : {})) {
            for (const destination of Object.values(file.destinations)) {
                test.deepEqual(destination.assumeRoleArn, 'arn:${AWS::Partition}:iam::${AWS::AccountId}:role/cdk-hnb659fds-file-publishing-role-${AWS::AccountId}-${AWS::Region}');
            }
        }
        for (const file of Object.values((_b = manifest.dockerImages) !== null && _b !== void 0 ? _b : {})) {
            for (const destination of Object.values(file.destinations)) {
                test.deepEqual(destination.assumeRoleArn, 'arn:${AWS::Partition}:iam::${AWS::AccountId}:role/cdk-hnb659fds-image-publishing-role-${AWS::AccountId}-${AWS::Region}');
            }
        }
        test.done();
    },
    'customize publishing resources'(test) {
        var _a, _b, _c, _d, _e, _f;
        // GIVEN
        const myapp = new lib_1.App();
        // WHEN
        const mystack = new lib_1.Stack(myapp, 'mystack', {
            synthesizer: new lib_1.DefaultStackSynthesizer({
                fileAssetsBucketName: 'file-asset-bucket',
                fileAssetPublishingRoleArn: 'file:role:arn',
                fileAssetPublishingExternalId: 'file-external-id',
                imageAssetsRepositoryName: 'image-ecr-repository',
                imageAssetPublishingRoleArn: 'image:role:arn',
                imageAssetPublishingExternalId: 'image-external-id',
            }),
        });
        mystack.synthesizer.addFileAsset({
            fileName: __filename,
            packaging: lib_1.FileAssetPackaging.FILE,
            sourceHash: 'file-asset-hash',
        });
        mystack.synthesizer.addDockerImageAsset({
            directoryName: '.',
            sourceHash: 'docker-asset-hash',
        });
        // THEN
        const asm = myapp.synth();
        const manifest = readAssetManifest(asm);
        test.deepEqual((_c = (_b = (_a = manifest.files) === null || _a === void 0 ? void 0 : _a['file-asset-hash']) === null || _b === void 0 ? void 0 : _b.destinations) === null || _c === void 0 ? void 0 : _c['current_account-current_region'], {
            bucketName: 'file-asset-bucket',
            objectKey: 'file-asset-hash',
            assumeRoleArn: 'file:role:arn',
            assumeRoleExternalId: 'file-external-id',
        });
        test.deepEqual((_f = (_e = (_d = manifest.dockerImages) === null || _d === void 0 ? void 0 : _d['docker-asset-hash']) === null || _e === void 0 ? void 0 : _e.destinations) === null || _f === void 0 ? void 0 : _f['current_account-current_region'], {
            repositoryName: 'image-ecr-repository',
            imageTag: 'docker-asset-hash',
            assumeRoleArn: 'image:role:arn',
            assumeRoleExternalId: 'image-external-id',
        });
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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