"use strict";
const cxapi = require("@aws-cdk/cx-api");
const fs = require("fs-extra");
const os = require("os");
const path = require("path");
const sinon = require("sinon");
const lib_1 = require("../lib");
const STUB_INPUT_FILE = '/tmp/docker-stub.input';
var DockerStubCommand;
(function (DockerStubCommand) {
    DockerStubCommand["SUCCESS"] = "DOCKER_STUB_SUCCESS";
    DockerStubCommand["FAIL"] = "DOCKER_STUB_FAIL";
    DockerStubCommand["SUCCESS_NO_OUTPUT"] = "DOCKER_STUB_SUCCESS_NO_OUTPUT";
})(DockerStubCommand || (DockerStubCommand = {}));
const userInfo = os.userInfo();
const USER_ARG = `-u ${userInfo.uid}:${userInfo.gid}`;
// this is a way to provide a custom "docker" command for staging.
process.env.CDK_DOCKER = `${__dirname}/docker-stub.sh`;
function readDockerStubInput() {
    const out = fs.readFileSync(STUB_INPUT_FILE, 'utf-8').trim();
    return out
        .replace(/-v ([^:]+):\/asset-input/, '-v /input:/asset-input')
        .replace(/-v ([^:]+):\/asset-output/, '-v /output:/asset-output');
}
module.exports = {
    'tearDown'(cb) {
        if (fs.existsSync(STUB_INPUT_FILE)) {
            fs.unlinkSync(STUB_INPUT_FILE);
        }
        cb();
        sinon.restore();
    },
    'base case'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const sourcePath = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // WHEN
        const staging = new lib_1.AssetStaging(stack, 's1', { sourcePath });
        test.deepEqual(staging.sourceHash, '2f37f937c51e2c191af66acf9b09f548926008ec68c575bd2ee54b6e997c0e00');
        test.deepEqual(staging.sourcePath, sourcePath);
        test.deepEqual(stack.resolve(staging.stagedPath), 'asset.2f37f937c51e2c191af66acf9b09f548926008ec68c575bd2ee54b6e997c0e00');
        test.done();
    },
    'staging can be disabled through context'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        stack.node.setContext(cxapi.DISABLE_ASSET_STAGING_CONTEXT, true);
        const sourcePath = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // WHEN
        const staging = new lib_1.AssetStaging(stack, 's1', { sourcePath });
        test.deepEqual(staging.sourceHash, '2f37f937c51e2c191af66acf9b09f548926008ec68c575bd2ee54b6e997c0e00');
        test.deepEqual(staging.sourcePath, sourcePath);
        test.deepEqual(stack.resolve(staging.stagedPath), sourcePath);
        test.done();
    },
    'files are copied to the output directory during synth'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        const file = path.join(__dirname, 'fs', 'fixtures.tar.gz');
        // WHEN
        new lib_1.AssetStaging(stack, 's1', { sourcePath: directory });
        new lib_1.AssetStaging(stack, 'file', { sourcePath: file });
        // THEN
        const assembly = app.synth();
        test.deepEqual(fs.readdirSync(assembly.directory), [
            'asset.2f37f937c51e2c191af66acf9b09f548926008ec68c575bd2ee54b6e997c0e00',
            'asset.af10ac04b3b607b0f8659c8f0cee8c343025ee75baf0b146f10f0e5311d2c46b.gz',
            'cdk.out',
            'manifest.json',
            'stack.template.json',
            'tree.json',
        ]);
        test.done();
    },
    'allow specifying extra data to include in the source hash'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // WHEN
        const withoutExtra = new lib_1.AssetStaging(stack, 'withoutExtra', { sourcePath: directory });
        const withExtra = new lib_1.AssetStaging(stack, 'withExtra', { sourcePath: directory, extraHash: 'boom' });
        // THEN
        test.notEqual(withoutExtra.sourceHash, withExtra.sourceHash);
        test.deepEqual(withoutExtra.sourceHash, '2f37f937c51e2c191af66acf9b09f548926008ec68c575bd2ee54b6e997c0e00');
        test.deepEqual(withExtra.sourceHash, 'c95c915a5722bb9019e2c725d11868e5a619b55f36172f76bcbcaa8bb2d10c5f');
        test.done();
    },
    'with bundling'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        const ensureDirSyncSpy = sinon.spy(fs, 'ensureDirSync');
        const mkdtempSyncSpy = sinon.spy(fs, 'mkdtempSync');
        // WHEN
        new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            bundling: {
                image: lib_1.BundlingDockerImage.fromRegistry('alpine'),
                command: [DockerStubCommand.SUCCESS],
            },
        });
        // THEN
        const assembly = app.synth();
        test.deepEqual(readDockerStubInput(), `run --rm ${USER_ARG} -v /input:/asset-input -v /output:/asset-output -w /asset-input alpine DOCKER_STUB_SUCCESS`);
        test.deepEqual(fs.readdirSync(assembly.directory), [
            'asset.2f37f937c51e2c191af66acf9b09f548926008ec68c575bd2ee54b6e997c0e00',
            'cdk.out',
            'manifest.json',
            'stack.template.json',
            'tree.json',
        ]);
        // asset is bundled in a directory inside .cdk.staging
        const stagingTmp = path.join('.', '.cdk.staging');
        test.ok(ensureDirSyncSpy.calledWith(stagingTmp));
        test.ok(mkdtempSyncSpy.calledWith(sinon.match(path.join(stagingTmp, 'asset-bundle-'))));
        test.done();
    },
    'bundling throws when /asset-ouput is empty'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // THEN
        test.throws(() => new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            bundling: {
                image: lib_1.BundlingDockerImage.fromRegistry('alpine'),
                command: [DockerStubCommand.SUCCESS_NO_OUTPUT],
            },
        }), /Bundling did not produce any output/);
        test.equal(readDockerStubInput(), `run --rm ${USER_ARG} -v /input:/asset-input -v /output:/asset-output -w /asset-input alpine DOCKER_STUB_SUCCESS_NO_OUTPUT`);
        test.done();
    },
    'bundling with BUNDLE asset hash type'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // WHEN
        const asset = new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            bundling: {
                image: lib_1.BundlingDockerImage.fromRegistry('alpine'),
                command: [DockerStubCommand.SUCCESS],
            },
            assetHashType: lib_1.AssetHashType.BUNDLE,
        });
        // THEN
        test.equal(readDockerStubInput(), `run --rm ${USER_ARG} -v /input:/asset-input -v /output:/asset-output -w /asset-input alpine DOCKER_STUB_SUCCESS`);
        test.equal(asset.assetHash, '33cbf2cae5432438e0f046bc45ba8c3cef7b6afcf47b59d1c183775c1918fb1f');
        test.done();
    },
    'custom hash'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // WHEN
        const asset = new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            assetHash: 'my-custom-hash',
        });
        // THEN
        test.equal(fs.existsSync(STUB_INPUT_FILE), false);
        test.equal(asset.assetHash, 'b9c77053f5b83bbe5ba343bc18e92db939a49017010813225fea91fa892c4823'); // hash of 'my-custom-hash'
        test.done();
    },
    'throws with assetHash and not CUSTOM hash type'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // THEN
        test.throws(() => new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            bundling: {
                image: lib_1.BundlingDockerImage.fromRegistry('alpine'),
                command: [DockerStubCommand.SUCCESS],
            },
            assetHash: 'my-custom-hash',
            assetHashType: lib_1.AssetHashType.BUNDLE,
        }), /Cannot specify `bundle` for `assetHashType`/);
        test.equal(readDockerStubInput(), `run --rm ${USER_ARG} -v /input:/asset-input -v /output:/asset-output -w /asset-input alpine DOCKER_STUB_SUCCESS`);
        test.done();
    },
    'throws with BUNDLE hash type and no bundling'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // THEN
        test.throws(() => new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            assetHashType: lib_1.AssetHashType.BUNDLE,
        }), /Cannot use `AssetHashType.BUNDLE` when `bundling` is not specified/);
        test.equal(fs.existsSync(STUB_INPUT_FILE), false);
        test.done();
    },
    'throws with CUSTOM and no hash'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // THEN
        test.throws(() => new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            assetHashType: lib_1.AssetHashType.CUSTOM,
        }), /`assetHash` must be specified when `assetHashType` is set to `AssetHashType.CUSTOM`/);
        test.equal(fs.existsSync(STUB_INPUT_FILE), false); // "docker" not executed
        test.done();
    },
    'throws when bundling fails'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'stack');
        const directory = path.join(__dirname, 'fs', 'fixtures', 'test1');
        // THEN
        test.throws(() => new lib_1.AssetStaging(stack, 'Asset', {
            sourcePath: directory,
            bundling: {
                image: lib_1.BundlingDockerImage.fromRegistry('this-is-an-invalid-docker-image'),
                command: [DockerStubCommand.FAIL],
            },
        }), /Failed to run bundling Docker image for asset stack\/Asset/);
        test.equal(readDockerStubInput(), `run --rm ${USER_ARG} -v /input:/asset-input -v /output:/asset-output -w /asset-input this-is-an-invalid-docker-image DOCKER_STUB_FAIL`);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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