"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Alias = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const kms_generated_1 = require("./kms.generated");
const REQUIRED_ALIAS_PREFIX = 'alias/';
const DISALLOWED_PREFIX = REQUIRED_ALIAS_PREFIX + 'aws/';
class AliasBase extends core_1.Resource {
    /**
     * The ARN of the key.
     *
     * @stability stable
     */
    get keyArn() {
        return core_1.Stack.of(this).formatArn({
            service: 'kms',
            // aliasName already contains the '/'
            resource: this.aliasName,
        });
    }
    /**
     * The ID of the key (the part that looks something like: 1234abcd-12ab-34cd-56ef-1234567890ab).
     *
     * @stability stable
     */
    get keyId() {
        return this.aliasName;
    }
    /**
     * Defines a new alias for the key.
     *
     * @stability stable
     */
    addAlias(alias) {
        return this.aliasTargetKey.addAlias(alias);
    }
    /**
     * Adds a statement to the KMS key resource policy.
     *
     * @stability stable
     */
    addToResourcePolicy(statement, allowNoOp) {
        return this.aliasTargetKey.addToResourcePolicy(statement, allowNoOp);
    }
    /**
     * Grant the indicated permissions on this key to the given principal.
     *
     * @stability stable
     */
    grant(grantee, ...actions) {
        return this.aliasTargetKey.grant(grantee, ...actions);
    }
    /**
     * Grant decryption permissions using this key to the given principal.
     *
     * @stability stable
     */
    grantDecrypt(grantee) {
        return this.aliasTargetKey.grantDecrypt(grantee);
    }
    /**
     * Grant encryption permissions using this key to the given principal.
     *
     * @stability stable
     */
    grantEncrypt(grantee) {
        return this.aliasTargetKey.grantEncrypt(grantee);
    }
    /**
     * Grant encryption and decryption permissions using this key to the given principal.
     *
     * @stability stable
     */
    grantEncryptDecrypt(grantee) {
        return this.aliasTargetKey.grantEncryptDecrypt(grantee);
    }
}
/**
 * Defines a display name for a customer master key (CMK) in AWS Key Management Service (AWS KMS).
 *
 * Using an alias to refer to a key can help you simplify key
 * management. For example, when rotating keys, you can just update the alias
 * mapping instead of tracking and changing key IDs. For more information, see
 * Working with Aliases in the AWS Key Management Service Developer Guide.
 *
 * You can also add an alias for a key by calling `key.addAlias(alias)`.
 *
 * @stability stable
 * @resource AWS::KMS::Alias
 */
class Alias extends AliasBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        let aliasName = props.aliasName;
        if (!core_1.Token.isUnresolved(aliasName)) {
            if (!aliasName.startsWith(REQUIRED_ALIAS_PREFIX)) {
                aliasName = REQUIRED_ALIAS_PREFIX + aliasName;
            }
            if (aliasName === REQUIRED_ALIAS_PREFIX) {
                throw new Error(`Alias must include a value after "${REQUIRED_ALIAS_PREFIX}": ${aliasName}`);
            }
            if (aliasName.toLocaleLowerCase().startsWith(DISALLOWED_PREFIX)) {
                throw new Error(`Alias cannot start with ${DISALLOWED_PREFIX}: ${aliasName}`);
            }
            if (!aliasName.match(/^[a-zA-Z0-9:/_-]{1,256}$/)) {
                throw new Error('Alias name must be between 1 and 256 characters in a-zA-Z0-9:/_-');
            }
        }
        super(scope, id, {
            physicalName: aliasName,
        });
        this.aliasTargetKey = props.targetKey;
        const resource = new kms_generated_1.CfnAlias(this, 'Resource', {
            aliasName: this.physicalName,
            targetKeyId: this.aliasTargetKey.keyArn,
        });
        this.aliasName = this.getResourceNameAttribute(resource.aliasName);
        if (props.removalPolicy) {
            resource.applyRemovalPolicy(props.removalPolicy);
        }
    }
    /**
     * Import an existing KMS Alias defined outside the CDK app.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs the properties of the referenced KMS Alias.
     * @stability stable
     */
    static fromAliasAttributes(scope, id, attrs) {
        class _Alias extends AliasBase {
            get aliasName() { return attrs.aliasName; }
            get aliasTargetKey() { return attrs.aliasTargetKey; }
        }
        return new _Alias(scope, id);
    }
    /**
     * Import an existing KMS Alias defined outside the CDK app, by the alias name.
     *
     * This method should be used
     * instead of 'fromAliasAttributes' when the underlying KMS Key ARN is not available.
     * This Alias will not have a direct reference to the KMS Key, so addAlias and grant* methods are not supported.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param aliasName The full name of the KMS Alias (e.g., 'alias/aws/s3', 'alias/myKeyAlias').
     * @stability stable
     */
    static fromAliasName(scope, id, aliasName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.keyArn = core_1.Stack.of(this).formatArn({ service: 'kms', resource: aliasName });
                this.keyId = aliasName;
                this.aliasName = aliasName;
            }
            get aliasTargetKey() { throw new Error('Cannot access aliasTargetKey on an Alias imported by Alias.fromAliasName().'); }
            addAlias(_alias) { throw new Error('Cannot call addAlias on an Alias imported by Alias.fromAliasName().'); }
            addToResourcePolicy(_statement, _allowNoOp) {
                return { statementAdded: false };
            }
            grant(grantee, ..._actions) { return iam.Grant.drop(grantee, ''); }
            grantDecrypt(grantee) { return iam.Grant.drop(grantee, ''); }
            grantEncrypt(grantee) { return iam.Grant.drop(grantee, ''); }
            grantEncryptDecrypt(grantee) { return iam.Grant.drop(grantee, ''); }
        }
        return new Import(scope, id);
    }
    /**
     * @stability stable
     */
    generatePhysicalName() {
        return REQUIRED_ALIAS_PREFIX + super.generatePhysicalName();
    }
}
exports.Alias = Alias;
_a = JSII_RTTI_SYMBOL_1;
Alias[_a] = { fqn: "@aws-cdk/aws-kms.Alias", version: "1.106.0" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYWxpYXMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJhbGlhcy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBLHdDQUF3QztBQUN4Qyx3Q0FBc0U7QUFHdEUsbURBQTJDO0FBRTNDLE1BQU0scUJBQXFCLEdBQUcsUUFBUSxDQUFDO0FBQ3ZDLE1BQU0saUJBQWlCLEdBQUcscUJBQXFCLEdBQUcsTUFBTSxDQUFDO0FBZ0R6RCxNQUFlLFNBQVUsU0FBUSxlQUFROzs7Ozs7SUFLdkMsSUFBVyxNQUFNO1FBQ2YsT0FBTyxZQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFNBQVMsQ0FBQztZQUM5QixPQUFPLEVBQUUsS0FBSztZQUNkLHFDQUFxQztZQUNyQyxRQUFRLEVBQUUsSUFBSSxDQUFDLFNBQVM7U0FDekIsQ0FBQyxDQUFDO0lBQ0wsQ0FBQzs7Ozs7O0lBRUQsSUFBVyxLQUFLO1FBQ2QsT0FBTyxJQUFJLENBQUMsU0FBUyxDQUFDO0lBQ3hCLENBQUM7Ozs7OztJQUVNLFFBQVEsQ0FBQyxLQUFhO1FBQzNCLE9BQU8sSUFBSSxDQUFDLGNBQWMsQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDN0MsQ0FBQzs7Ozs7O0lBRU0sbUJBQW1CLENBQUMsU0FBOEIsRUFBRSxTQUFtQjtRQUM1RSxPQUFPLElBQUksQ0FBQyxjQUFjLENBQUMsbUJBQW1CLENBQUMsU0FBUyxFQUFFLFNBQVMsQ0FBQyxDQUFDO0lBQ3ZFLENBQUM7Ozs7OztJQUVNLEtBQUssQ0FBQyxPQUF1QixFQUFFLEdBQUcsT0FBaUI7UUFDeEQsT0FBTyxJQUFJLENBQUMsY0FBYyxDQUFDLEtBQUssQ0FBQyxPQUFPLEVBQUUsR0FBRyxPQUFPLENBQUMsQ0FBQztJQUN4RCxDQUFDOzs7Ozs7SUFFTSxZQUFZLENBQUMsT0FBdUI7UUFDekMsT0FBTyxJQUFJLENBQUMsY0FBYyxDQUFDLFlBQVksQ0FBQyxPQUFPLENBQUMsQ0FBQztJQUNuRCxDQUFDOzs7Ozs7SUFFTSxZQUFZLENBQUMsT0FBdUI7UUFDekMsT0FBTyxJQUFJLENBQUMsY0FBYyxDQUFDLFlBQVksQ0FBQyxPQUFPLENBQUMsQ0FBQztJQUNuRCxDQUFDOzs7Ozs7SUFFTSxtQkFBbUIsQ0FBQyxPQUF1QjtRQUNoRCxPQUFPLElBQUksQ0FBQyxjQUFjLENBQUMsbUJBQW1CLENBQUMsT0FBTyxDQUFDLENBQUM7SUFDMUQsQ0FBQztDQUNGOzs7Ozs7Ozs7Ozs7OztBQTRCRCxNQUFhLEtBQU0sU0FBUSxTQUFTOzs7O0lBK0NsQyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQWlCO1FBQ3pELElBQUksU0FBUyxHQUFHLEtBQUssQ0FBQyxTQUFTLENBQUM7UUFFaEMsSUFBSSxDQUFDLFlBQUssQ0FBQyxZQUFZLENBQUMsU0FBUyxDQUFDLEVBQUU7WUFDbEMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQUMscUJBQXFCLENBQUMsRUFBRTtnQkFDaEQsU0FBUyxHQUFHLHFCQUFxQixHQUFHLFNBQVMsQ0FBQzthQUMvQztZQUVELElBQUksU0FBUyxLQUFLLHFCQUFxQixFQUFFO2dCQUN2QyxNQUFNLElBQUksS0FBSyxDQUFDLHFDQUFxQyxxQkFBcUIsTUFBTSxTQUFTLEVBQUUsQ0FBQyxDQUFDO2FBQzlGO1lBRUQsSUFBSSxTQUFTLENBQUMsaUJBQWlCLEVBQUUsQ0FBQyxVQUFVLENBQUMsaUJBQWlCLENBQUMsRUFBRTtnQkFDL0QsTUFBTSxJQUFJLEtBQUssQ0FBQywyQkFBMkIsaUJBQWlCLEtBQUssU0FBUyxFQUFFLENBQUMsQ0FBQzthQUMvRTtZQUVELElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLDBCQUEwQixDQUFDLEVBQUU7Z0JBQ2hELE1BQU0sSUFBSSxLQUFLLENBQUMsa0VBQWtFLENBQUMsQ0FBQzthQUNyRjtTQUNGO1FBRUQsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDZixZQUFZLEVBQUUsU0FBUztTQUN4QixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsY0FBYyxHQUFHLEtBQUssQ0FBQyxTQUFTLENBQUM7UUFFdEMsTUFBTSxRQUFRLEdBQUcsSUFBSSx3QkFBUSxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDOUMsU0FBUyxFQUFFLElBQUksQ0FBQyxZQUFZO1lBQzVCLFdBQVcsRUFBRSxJQUFJLENBQUMsY0FBYyxDQUFDLE1BQU07U0FDeEMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLFNBQVMsR0FBRyxJQUFJLENBQUMsd0JBQXdCLENBQUMsUUFBUSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBRW5FLElBQUksS0FBSyxDQUFDLGFBQWEsRUFBRTtZQUN2QixRQUFRLENBQUMsa0JBQWtCLENBQUMsS0FBSyxDQUFDLGFBQWEsQ0FBQyxDQUFDO1NBQ2xEO0lBQ0gsQ0FBQzs7Ozs7Ozs7O0lBNUVNLE1BQU0sQ0FBQyxtQkFBbUIsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFzQjtRQUNwRixNQUFNLE1BQU8sU0FBUSxTQUFTO1lBQzVCLElBQVcsU0FBUyxLQUFLLE9BQU8sS0FBSyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUM7WUFDbEQsSUFBVyxjQUFjLEtBQUssT0FBTyxLQUFLLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQztTQUM3RDtRQUNELE9BQU8sSUFBSSxNQUFNLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBQy9CLENBQUM7Ozs7Ozs7Ozs7Ozs7SUFXTSxNQUFNLENBQUMsYUFBYSxDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLFNBQWlCO1FBQ3pFLE1BQU0sTUFBTyxTQUFRLGVBQVE7WUFBN0I7O2dCQUNrQixXQUFNLEdBQUcsWUFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxTQUFTLENBQUMsRUFBRSxPQUFPLEVBQUUsS0FBSyxFQUFFLFFBQVEsRUFBRSxTQUFTLEVBQUUsQ0FBQyxDQUFDO2dCQUMzRSxVQUFLLEdBQUcsU0FBUyxDQUFDO2dCQUNsQixjQUFTLEdBQUcsU0FBUyxDQUFDO1lBVXhDLENBQUM7WUFUQyxJQUFXLGNBQWMsS0FBVyxNQUFNLElBQUksS0FBSyxDQUFDLDZFQUE2RSxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQzlILFFBQVEsQ0FBQyxNQUFjLElBQVcsTUFBTSxJQUFJLEtBQUssQ0FBQyxxRUFBcUUsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUMzSCxtQkFBbUIsQ0FBQyxVQUErQixFQUFFLFVBQW9CO2dCQUM5RSxPQUFPLEVBQUUsY0FBYyxFQUFFLEtBQUssRUFBRSxDQUFDO1lBQ25DLENBQUM7WUFDTSxLQUFLLENBQUMsT0FBdUIsRUFBRSxHQUFHLFFBQWtCLElBQWUsT0FBTyxHQUFHLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ3hHLFlBQVksQ0FBQyxPQUF1QixJQUFlLE9BQU8sR0FBRyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsT0FBTyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUN4RixZQUFZLENBQUMsT0FBdUIsSUFBZSxPQUFPLEdBQUcsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDeEYsbUJBQW1CLENBQUMsT0FBdUIsSUFBZSxPQUFPLEdBQUcsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7U0FDdkc7UUFFRCxPQUFPLElBQUksTUFBTSxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztJQUMvQixDQUFDOzs7O0lBNENTLG9CQUFvQjtRQUM1QixPQUFPLHFCQUFxQixHQUFHLEtBQUssQ0FBQyxvQkFBb0IsRUFBRSxDQUFDO0lBQzlELENBQUM7O0FBeEZILHNCQXlGQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGlhbSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCB7IFJlbW92YWxQb2xpY3ksIFJlc291cmNlLCBTdGFjaywgVG9rZW4gfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuaW1wb3J0IHsgSUtleSB9IGZyb20gJy4va2V5JztcbmltcG9ydCB7IENmbkFsaWFzIH0gZnJvbSAnLi9rbXMuZ2VuZXJhdGVkJztcblxuY29uc3QgUkVRVUlSRURfQUxJQVNfUFJFRklYID0gJ2FsaWFzLyc7XG5jb25zdCBESVNBTExPV0VEX1BSRUZJWCA9IFJFUVVJUkVEX0FMSUFTX1BSRUZJWCArICdhd3MvJztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIElBbGlhcyBleHRlbmRzIElLZXkge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGFsaWFzTmFtZTogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgYWxpYXNUYXJnZXRLZXk6IElLZXk7XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIEFsaWFzUHJvcHMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgYWxpYXNOYW1lOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHRhcmdldEtleTogSUtleTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcmVtb3ZhbFBvbGljeT86IFJlbW92YWxQb2xpY3k7XG59XG5cbmFic3RyYWN0IGNsYXNzIEFsaWFzQmFzZSBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSUFsaWFzIHtcbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IGFsaWFzTmFtZTogc3RyaW5nO1xuXG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBhbGlhc1RhcmdldEtleTogSUtleTtcblxuICBwdWJsaWMgZ2V0IGtleUFybigpOiBzdHJpbmcge1xuICAgIHJldHVybiBTdGFjay5vZih0aGlzKS5mb3JtYXRBcm4oe1xuICAgICAgc2VydmljZTogJ2ttcycsXG4gICAgICAvLyBhbGlhc05hbWUgYWxyZWFkeSBjb250YWlucyB0aGUgJy8nXG4gICAgICByZXNvdXJjZTogdGhpcy5hbGlhc05hbWUsXG4gICAgfSk7XG4gIH1cblxuICBwdWJsaWMgZ2V0IGtleUlkKCk6IHN0cmluZyB7XG4gICAgcmV0dXJuIHRoaXMuYWxpYXNOYW1lO1xuICB9XG5cbiAgcHVibGljIGFkZEFsaWFzKGFsaWFzOiBzdHJpbmcpOiBBbGlhcyB7XG4gICAgcmV0dXJuIHRoaXMuYWxpYXNUYXJnZXRLZXkuYWRkQWxpYXMoYWxpYXMpO1xuICB9XG5cbiAgcHVibGljIGFkZFRvUmVzb3VyY2VQb2xpY3koc3RhdGVtZW50OiBpYW0uUG9saWN5U3RhdGVtZW50LCBhbGxvd05vT3A/OiBib29sZWFuKTogaWFtLkFkZFRvUmVzb3VyY2VQb2xpY3lSZXN1bHQge1xuICAgIHJldHVybiB0aGlzLmFsaWFzVGFyZ2V0S2V5LmFkZFRvUmVzb3VyY2VQb2xpY3koc3RhdGVtZW50LCBhbGxvd05vT3ApO1xuICB9XG5cbiAgcHVibGljIGdyYW50KGdyYW50ZWU6IGlhbS5JR3JhbnRhYmxlLCAuLi5hY3Rpb25zOiBzdHJpbmdbXSk6IGlhbS5HcmFudCB7XG4gICAgcmV0dXJuIHRoaXMuYWxpYXNUYXJnZXRLZXkuZ3JhbnQoZ3JhbnRlZSwgLi4uYWN0aW9ucyk7XG4gIH1cblxuICBwdWJsaWMgZ3JhbnREZWNyeXB0KGdyYW50ZWU6IGlhbS5JR3JhbnRhYmxlKTogaWFtLkdyYW50IHtcbiAgICByZXR1cm4gdGhpcy5hbGlhc1RhcmdldEtleS5ncmFudERlY3J5cHQoZ3JhbnRlZSk7XG4gIH1cblxuICBwdWJsaWMgZ3JhbnRFbmNyeXB0KGdyYW50ZWU6IGlhbS5JR3JhbnRhYmxlKTogaWFtLkdyYW50IHtcbiAgICByZXR1cm4gdGhpcy5hbGlhc1RhcmdldEtleS5ncmFudEVuY3J5cHQoZ3JhbnRlZSk7XG4gIH1cblxuICBwdWJsaWMgZ3JhbnRFbmNyeXB0RGVjcnlwdChncmFudGVlOiBpYW0uSUdyYW50YWJsZSk6IGlhbS5HcmFudCB7XG4gICAgcmV0dXJuIHRoaXMuYWxpYXNUYXJnZXRLZXkuZ3JhbnRFbmNyeXB0RGVjcnlwdChncmFudGVlKTtcbiAgfVxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIEFsaWFzQXR0cmlidXRlcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGFsaWFzTmFtZTogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGFsaWFzVGFyZ2V0S2V5OiBJS2V5O1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgQWxpYXMgZXh0ZW5kcyBBbGlhc0Jhc2Uge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUFsaWFzQXR0cmlidXRlcyhzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBhdHRyczogQWxpYXNBdHRyaWJ1dGVzKTogSUFsaWFzIHtcbiAgICBjbGFzcyBfQWxpYXMgZXh0ZW5kcyBBbGlhc0Jhc2Uge1xuICAgICAgcHVibGljIGdldCBhbGlhc05hbWUoKSB7IHJldHVybiBhdHRycy5hbGlhc05hbWU7IH1cbiAgICAgIHB1YmxpYyBnZXQgYWxpYXNUYXJnZXRLZXkoKSB7IHJldHVybiBhdHRycy5hbGlhc1RhcmdldEtleTsgfVxuICAgIH1cbiAgICByZXR1cm4gbmV3IF9BbGlhcyhzY29wZSwgaWQpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGZyb21BbGlhc05hbWUoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgYWxpYXNOYW1lOiBzdHJpbmcpOiBJQWxpYXMge1xuICAgIGNsYXNzIEltcG9ydCBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSUFsaWFzIHtcbiAgICAgIHB1YmxpYyByZWFkb25seSBrZXlBcm4gPSBTdGFjay5vZih0aGlzKS5mb3JtYXRBcm4oeyBzZXJ2aWNlOiAna21zJywgcmVzb3VyY2U6IGFsaWFzTmFtZSB9KTtcbiAgICAgIHB1YmxpYyByZWFkb25seSBrZXlJZCA9IGFsaWFzTmFtZTtcbiAgICAgIHB1YmxpYyByZWFkb25seSBhbGlhc05hbWUgPSBhbGlhc05hbWU7XG4gICAgICBwdWJsaWMgZ2V0IGFsaWFzVGFyZ2V0S2V5KCk6IElLZXkgeyB0aHJvdyBuZXcgRXJyb3IoJ0Nhbm5vdCBhY2Nlc3MgYWxpYXNUYXJnZXRLZXkgb24gYW4gQWxpYXMgaW1wb3J0ZWQgYnkgQWxpYXMuZnJvbUFsaWFzTmFtZSgpLicpOyB9XG4gICAgICBwdWJsaWMgYWRkQWxpYXMoX2FsaWFzOiBzdHJpbmcpOiBBbGlhcyB7IHRocm93IG5ldyBFcnJvcignQ2Fubm90IGNhbGwgYWRkQWxpYXMgb24gYW4gQWxpYXMgaW1wb3J0ZWQgYnkgQWxpYXMuZnJvbUFsaWFzTmFtZSgpLicpOyB9XG4gICAgICBwdWJsaWMgYWRkVG9SZXNvdXJjZVBvbGljeShfc3RhdGVtZW50OiBpYW0uUG9saWN5U3RhdGVtZW50LCBfYWxsb3dOb09wPzogYm9vbGVhbik6IGlhbS5BZGRUb1Jlc291cmNlUG9saWN5UmVzdWx0IHtcbiAgICAgICAgcmV0dXJuIHsgc3RhdGVtZW50QWRkZWQ6IGZhbHNlIH07XG4gICAgICB9XG4gICAgICBwdWJsaWMgZ3JhbnQoZ3JhbnRlZTogaWFtLklHcmFudGFibGUsIC4uLl9hY3Rpb25zOiBzdHJpbmdbXSk6IGlhbS5HcmFudCB7IHJldHVybiBpYW0uR3JhbnQuZHJvcChncmFudGVlLCAnJyk7IH1cbiAgICAgIHB1YmxpYyBncmFudERlY3J5cHQoZ3JhbnRlZTogaWFtLklHcmFudGFibGUpOiBpYW0uR3JhbnQgeyByZXR1cm4gaWFtLkdyYW50LmRyb3AoZ3JhbnRlZSwgJycpOyB9XG4gICAgICBwdWJsaWMgZ3JhbnRFbmNyeXB0KGdyYW50ZWU6IGlhbS5JR3JhbnRhYmxlKTogaWFtLkdyYW50IHsgcmV0dXJuIGlhbS5HcmFudC5kcm9wKGdyYW50ZWUsICcnKTsgfVxuICAgICAgcHVibGljIGdyYW50RW5jcnlwdERlY3J5cHQoZ3JhbnRlZTogaWFtLklHcmFudGFibGUpOiBpYW0uR3JhbnQgeyByZXR1cm4gaWFtLkdyYW50LmRyb3AoZ3JhbnRlZSwgJycpOyB9XG4gICAgfVxuXG4gICAgcmV0dXJuIG5ldyBJbXBvcnQoc2NvcGUsIGlkKTtcbiAgfVxuXG4gIHB1YmxpYyByZWFkb25seSBhbGlhc05hbWU6IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IGFsaWFzVGFyZ2V0S2V5OiBJS2V5O1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBBbGlhc1Byb3BzKSB7XG4gICAgbGV0IGFsaWFzTmFtZSA9IHByb3BzLmFsaWFzTmFtZTtcblxuICAgIGlmICghVG9rZW4uaXNVbnJlc29sdmVkKGFsaWFzTmFtZSkpIHtcbiAgICAgIGlmICghYWxpYXNOYW1lLnN0YXJ0c1dpdGgoUkVRVUlSRURfQUxJQVNfUFJFRklYKSkge1xuICAgICAgICBhbGlhc05hbWUgPSBSRVFVSVJFRF9BTElBU19QUkVGSVggKyBhbGlhc05hbWU7XG4gICAgICB9XG5cbiAgICAgIGlmIChhbGlhc05hbWUgPT09IFJFUVVJUkVEX0FMSUFTX1BSRUZJWCkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYEFsaWFzIG11c3QgaW5jbHVkZSBhIHZhbHVlIGFmdGVyIFwiJHtSRVFVSVJFRF9BTElBU19QUkVGSVh9XCI6ICR7YWxpYXNOYW1lfWApO1xuICAgICAgfVxuXG4gICAgICBpZiAoYWxpYXNOYW1lLnRvTG9jYWxlTG93ZXJDYXNlKCkuc3RhcnRzV2l0aChESVNBTExPV0VEX1BSRUZJWCkpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBBbGlhcyBjYW5ub3Qgc3RhcnQgd2l0aCAke0RJU0FMTE9XRURfUFJFRklYfTogJHthbGlhc05hbWV9YCk7XG4gICAgICB9XG5cbiAgICAgIGlmICghYWxpYXNOYW1lLm1hdGNoKC9eW2EtekEtWjAtOTovXy1dezEsMjU2fSQvKSkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0FsaWFzIG5hbWUgbXVzdCBiZSBiZXR3ZWVuIDEgYW5kIDI1NiBjaGFyYWN0ZXJzIGluIGEtekEtWjAtOTovXy0nKTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICBzdXBlcihzY29wZSwgaWQsIHtcbiAgICAgIHBoeXNpY2FsTmFtZTogYWxpYXNOYW1lLFxuICAgIH0pO1xuXG4gICAgdGhpcy5hbGlhc1RhcmdldEtleSA9IHByb3BzLnRhcmdldEtleTtcblxuICAgIGNvbnN0IHJlc291cmNlID0gbmV3IENmbkFsaWFzKHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIGFsaWFzTmFtZTogdGhpcy5waHlzaWNhbE5hbWUsXG4gICAgICB0YXJnZXRLZXlJZDogdGhpcy5hbGlhc1RhcmdldEtleS5rZXlBcm4sXG4gICAgfSk7XG5cbiAgICB0aGlzLmFsaWFzTmFtZSA9IHRoaXMuZ2V0UmVzb3VyY2VOYW1lQXR0cmlidXRlKHJlc291cmNlLmFsaWFzTmFtZSk7XG5cbiAgICBpZiAocHJvcHMucmVtb3ZhbFBvbGljeSkge1xuICAgICAgcmVzb3VyY2UuYXBwbHlSZW1vdmFsUG9saWN5KHByb3BzLnJlbW92YWxQb2xpY3kpO1xuICAgIH1cbiAgfVxuXG4gIHByb3RlY3RlZCBnZW5lcmF0ZVBoeXNpY2FsTmFtZSgpOiBzdHJpbmcge1xuICAgIHJldHVybiBSRVFVSVJFRF9BTElBU19QUkVGSVggKyBzdXBlci5nZW5lcmF0ZVBoeXNpY2FsTmFtZSgpO1xuICB9XG59XG4iXX0=