"""hammad.genai.types.base"""

from abc import ABC, abstractmethod
from typing import (
    Any,
    AsyncIterator,
    Callable,
    Dict,
    Iterator,
    Generic,
    ParamSpec,
    TypeVar,
    TYPE_CHECKING,
    Union,
)

from pydantic import BaseModel, ConfigDict

if TYPE_CHECKING:
    from openai.types.chat import (
        ChatCompletionMessageParam,
        ChatCompletionContentPartParam,
        ChatCompletionMessageParam,
    )


__all__ = [
    "T",
    "P",
    "R",
    "BaseGenAIModelEvent",
    "BaseGenAIModelStream",
    "BaseTool",
    "BaseGenAIModelResponse",
    "BaseGenAIModelSettings",
    "BaseGenAIModel",
]


T = TypeVar("T")
P = ParamSpec("P")
R = TypeVar("R")


class BaseGenAIModelEvent(BaseModel, Generic[T]):
    """Base class for all events that a Generative AI model can
    emit / return.

    This is a base class used only for type hinting and incorporates
    no fields.
    """

    model_config = ConfigDict(
        arbitrary_types_allowed=True,
    )

    def to_message(self) -> "ChatCompletionMessageParam":
        """Converts the event into a message dictionary that is compatible
        with any message interface within the `hammad.genai` module."""
        raise NotImplementedError(
            f"to_message() is not implemented for {self.__class__.__name__}"
        )

    def to_content_part(self) -> "ChatCompletionContentPartParam":
        """Converts the event into a content part dictionary that can be added
        within chat messages."""
        raise NotImplementedError(
            f"to_content_part() is not implemented for {self.__class__.__name__}"
        )


class BaseGenAIModelStream(BaseGenAIModelEvent[T]):
    """Base class for all streams from Generative AI models within the
    `hammad.genai` module.

    This class manages both sync and async streaming.
    """

    model_config = ConfigDict(
        arbitrary_types_allowed=True,
    )

    type: str
    """The type of the model, can be `language_model`, `embedding_model`, 
    `image_model`..."""

    model: str
    """The model that was used to generate the stream."""

    stream: Union[Iterator[T], AsyncIterator[T]] | None = None
    """The streamed content generated by the model."""

    def __iter__(self) -> Iterator[T]:
        raise NotImplementedError(
            f"__iter__() is not implemented for {self.__class__.__name__}"
        )

    def __aiter__(self) -> AsyncIterator[T]:
        raise NotImplementedError(
            f"__aiter__() is not implemented for {self.__class__.__name__}"
        )


class BaseTool(BaseModel, Generic[P, R]):
    """Base class for tools. All generative AI models within the
    `hammad.genai` module can be converted into tools usable by
    agents and language models.
    """

    model_config = ConfigDict(
        arbitrary_types_allowed=True,
    )

    name: str
    """The name of the tool."""

    description: str
    """Description of what the tool does."""

    function: Callable[P, R]
    """The Python function to execute."""

    parameters_json_schema: Dict[str, Any]
    """JSON schema for the tool's parameters."""

    takes_context: bool = False
    """Whether the function expects a context as first parameter."""

    strict: bool = True
    """Whether to enforce strict JSON schema validation."""


class BaseGenAIModelResponse(BaseGenAIModelEvent[T]):
    """Base class for all responses from Generative AI models within the
    `hammad.genai` module.
    """

    model_config = ConfigDict(
        arbitrary_types_allowed=True,
    )

    type: str
    """The type of the model, can be `language_model`, `embedding_model`, 
    `image_model`..."""

    model: str
    """The model that was used to generate the response."""

    output: T
    """The final response or output generated by the model. This can be
    anything from chat messages, embeddings, ..."""


class BaseGenAIModelSettings(BaseModel):
    """Represents the defaults & base for additional settings that
    can be applied to any model within the `hammad.genai` module.
    """


class BaseGenAIModel(BaseModel, ABC):
    """Base class for all Generative AI models available within the
    `hammad.genai` module.

    NOTE:
    All models within this library use `litellm` directly as the
    client, if you dont have a need for any of the opinionation
    given by this package, I would recommend just using
    `litellm` directly.
    """

    model_config = ConfigDict(arbitrary_types_allowed=True, extra="allow")

    model: str | None = None
    """The model to use. This is always in the `litellm` format:
    
    `<provider>/<model>`

    `openai/gpt-4o-mini`
    `openai/text-embedding-3-small`
    """

    base_url: str | None = None
    """A custom base URL to use for the model.
    """

    api_key: str | None = None
    """The API key to use for the model.
    """

    settings: BaseGenAIModelSettings | None = None
    """The default (additional) settings to use when generating outputs
    with this model."""

    @abstractmethod
    def run(
        self,
        *args,
        **kwargs,
    ) -> Any: ...

    @abstractmethod
    async def async_run(
        self,
        *args,
        **kwargs,
    ) -> Any: ...

    def as_tool(
        self,
        *args,
        **kwargs,
    ) -> BaseTool[P, R]:
        raise NotImplementedError(
            f"as_tool() is not implemented for {self.__class__.__name__}"
        )
