"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.IdentitySource = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_verifiedpermissions_1 = require("aws-cdk-lib/aws-verifiedpermissions");
const core_1 = require("aws-cdk-lib/core");
var ConfigurationMode;
(function (ConfigurationMode) {
    ConfigurationMode["COGNITO"] = "COGNITO";
    ConfigurationMode["OIDC_ACCESS_TOKEN"] = "OIDC_ACCESS_TOKEN";
    ConfigurationMode["OIDC_ID_TOKEN"] = "OIDC_ID_TOKEN";
})(ConfigurationMode || (ConfigurationMode = {}));
class IdentitySourceBase extends core_1.Resource {
}
class IdentitySource extends IdentitySourceBase {
    /**
     * Creates Identity Source from its attributes
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs An `IdentitySourceAttributes` object.
     */
    static fromIdentitySourceAttributes(scope, id, attrs) {
        class Import extends IdentitySourceBase {
            constructor(identitySourceId) {
                super(scope, id);
                this.identitySourceId = identitySourceId;
            }
        }
        let identitySourceId;
        identitySourceId = attrs.identitySourceId;
        return new Import(identitySourceId);
    }
    /**
     * Create an Identity Source from its identifier
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param identitySourceId The Identity Source identifier.
     */
    static fromIdentitySourceId(scope, id, identitySourceId) {
        return IdentitySource.fromIdentitySourceAttributes(scope, id, {
            identitySourceId,
        });
    }
    constructor(scope, id, props) {
        super(scope, id);
        if (props.configuration.cognitoUserPoolConfiguration && props.configuration.openIdConnectConfiguration) {
            throw new Error('Only one between cognitoUserPoolConfiguration or openIdConnectConfiguration must be defined');
        }
        let cfnConfiguration;
        let issuer;
        if (props.configuration.cognitoUserPoolConfiguration) {
            this.clientIds = props.configuration.cognitoUserPoolConfiguration.clientIds ?? [];
            this.audiencesOIDC = [];
            const cognitoGroupConfiguration = props.configuration.cognitoUserPoolConfiguration.groupConfiguration?.groupEntityType
                ? {
                    groupEntityType: props.configuration.cognitoUserPoolConfiguration.groupConfiguration.groupEntityType,
                }
                : undefined;
            cfnConfiguration = {
                cognitoUserPoolConfiguration: {
                    clientIds: core_1.Lazy.list({ produce: () => this.clientIds }),
                    userPoolArn: props.configuration.cognitoUserPoolConfiguration.userPool.userPoolArn,
                    groupConfiguration: cognitoGroupConfiguration,
                },
            };
            this.cognitoGroupEntityType = cognitoGroupConfiguration?.groupEntityType;
            issuer = 'COGNITO';
            this.configurationMode = ConfigurationMode.COGNITO;
        }
        else if (props.configuration.openIdConnectConfiguration) {
            if (props.configuration.openIdConnectConfiguration.accessTokenOnly &&
                props.configuration.openIdConnectConfiguration.identityTokenOnly) {
                throw new Error('Exactly one token selection method between accessTokenOnly and identityTokenOnly must be defined');
            }
            let tokenSelection;
            if (props.configuration.openIdConnectConfiguration.accessTokenOnly) {
                if (!props.configuration.openIdConnectConfiguration.accessTokenOnly.audiences ||
                    props.configuration.openIdConnectConfiguration.accessTokenOnly.audiences.length == 0) {
                    throw new Error('At least one audience is expected in OIDC Access token selection mode');
                }
                this.clientIds = [];
                this.audiencesOIDC = props.configuration.openIdConnectConfiguration.accessTokenOnly.audiences;
                tokenSelection = {
                    accessTokenOnly: {
                        audiences: core_1.Lazy.list({ produce: () => this.audiencesOIDC }),
                        principalIdClaim: props.configuration.openIdConnectConfiguration.accessTokenOnly.principalIdClaim,
                    },
                };
                this.principalIdClaimOIDC = props.configuration.openIdConnectConfiguration.accessTokenOnly.principalIdClaim;
                this.configurationMode = ConfigurationMode.OIDC_ACCESS_TOKEN;
            }
            else if (props.configuration.openIdConnectConfiguration.identityTokenOnly) {
                this.clientIds = props.configuration.openIdConnectConfiguration.identityTokenOnly.clientIds ?? [];
                this.audiencesOIDC = [];
                tokenSelection = {
                    identityTokenOnly: {
                        clientIds: core_1.Lazy.list({ produce: () => this.clientIds }),
                        principalIdClaim: props.configuration.openIdConnectConfiguration.identityTokenOnly.principalIdClaim,
                    },
                };
                this.principalIdClaimOIDC = props.configuration.openIdConnectConfiguration.identityTokenOnly.principalIdClaim;
                this.configurationMode = ConfigurationMode.OIDC_ID_TOKEN;
            }
            else {
                throw new Error('One token selection method between accessTokenOnly and identityTokenOnly must be defined');
            }
            cfnConfiguration = {
                openIdConnectConfiguration: {
                    issuer: props.configuration.openIdConnectConfiguration.issuer,
                    entityIdPrefix: props.configuration.openIdConnectConfiguration.entityIdPrefix,
                    groupConfiguration: props.configuration.openIdConnectConfiguration.groupConfiguration ? {
                        groupClaim: props.configuration.openIdConnectConfiguration.groupConfiguration.groupClaim,
                        groupEntityType: props.configuration.openIdConnectConfiguration.groupConfiguration.groupEntityType,
                    } : undefined,
                    tokenSelection: tokenSelection,
                },
            };
            this.groupConfigGroupClaimOIDC = props.configuration.openIdConnectConfiguration.groupConfiguration?.groupClaim;
            this.groupConfigGroupEntityTypeOIDC = props.configuration.openIdConnectConfiguration.groupConfiguration?.groupEntityType;
            issuer = props.configuration.openIdConnectConfiguration.issuer;
        }
        else {
            throw new Error('One Identity provider configuration between cognitoUserPoolConfiguration and openIdConnectConfiguration must be defined');
        }
        this.identitySource = new aws_verifiedpermissions_1.CfnIdentitySource(this, id, {
            configuration: cfnConfiguration,
            policyStoreId: props.policyStore.policyStoreId,
            principalEntityType: props.principalEntityType,
        });
        this.userPoolArn = props.configuration.cognitoUserPoolConfiguration?.userPool.userPoolArn || undefined;
        this.identitySourceId = this.identitySource.attrIdentitySourceId;
        this.issuer = issuer;
        this.policyStore = props.policyStore;
    }
    /**
     * Add a User Pool Client
     * The method can be called only when the Identity Source is configured with Cognito auth provider
     *
     * @param userPoolClient The User Pool Client Construct.
     */
    addUserPoolClient(userPoolClient) {
        if (this.configurationMode != ConfigurationMode.COGNITO) {
            throw new Error('Cannot add User Pool Client when IdentitySource auth provider is not Cognito');
        }
        this.addClientId(userPoolClient.userPoolClientId);
    }
    /**
     * Add a clientId to the list
     * The method can be called only when the Identity Source is configured with one of these configs:
     *  - Cognito auth provider
     *  - OIDC auth provider and ID Token Selection mode
     *
     * @param clientId The clientId to be added
     */
    addClientId(clientId) {
        if (this.configurationMode != ConfigurationMode.COGNITO && this.configurationMode != ConfigurationMode.OIDC_ID_TOKEN) {
            throw new Error('Adding a Client ID is only supported for the auth providers Cognito or OIDC with configured with ID Token');
        }
        this.clientIds.push(clientId);
    }
    /**
     * Add an audience to the list.
     * The method can be called only when the Identity Source is configured with OIDC auth provider and Access Token Selection mode
     *
     * @param audience the audience to be added
     */
    addAudience(audience) {
        if (this.configurationMode != ConfigurationMode.OIDC_ACCESS_TOKEN) {
            throw new Error('Cannot add audience when IdentitySource auth provider is not OIDC with Access Token');
        }
        this.audiencesOIDC.push(audience);
    }
}
exports.IdentitySource = IdentitySource;
_a = JSII_RTTI_SYMBOL_1;
IdentitySource[_a] = { fqn: "@cdklabs/cdk-verified-permissions.IdentitySource", version: "0.2.0" };
//# sourceMappingURL=data:application/json;base64,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