"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyStore = exports.DeletionProtectionMode = exports.ValidationSettingsMode = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const iam = require("aws-cdk-lib/aws-iam");
const aws_verifiedpermissions_1 = require("aws-cdk-lib/aws-verifiedpermissions");
const core_1 = require("aws-cdk-lib/core");
const cedar_helpers_1 = require("./cedar-helpers");
const policy_1 = require("./policy");
const permissions_1 = require("./private/permissions");
const RELEVANT_HTTP_METHODS = ['get', 'post', 'put', 'patch', 'delete', 'head'];
/**
 * Validation Settings mode, according to the Cloudformation PolicyStore resource
 */
var ValidationSettingsMode;
(function (ValidationSettingsMode) {
    ValidationSettingsMode["OFF"] = "OFF";
    ValidationSettingsMode["STRICT"] = "STRICT";
})(ValidationSettingsMode || (exports.ValidationSettingsMode = ValidationSettingsMode = {}));
var DeletionProtectionMode;
(function (DeletionProtectionMode) {
    DeletionProtectionMode["ENABLED"] = "ENABLED";
    DeletionProtectionMode["DISABLED"] = "DISABLED";
})(DeletionProtectionMode || (exports.DeletionProtectionMode = DeletionProtectionMode = {}));
class PolicyStoreBase extends core_1.Resource {
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [this.policyStoreArn],
            scope: this,
        });
    }
    grantAuth(grantee) {
        return this.grant(grantee, ...permissions_1.AUTH_ACTIONS);
    }
    grantRead(grantee) {
        return this.grant(grantee, ...permissions_1.READ_ACTIONS);
    }
    grantWrite(grantee) {
        return this.grant(grantee, ...permissions_1.WRITE_ACTIONS.concat(permissions_1.READ_ACTIONS));
    }
}
class PolicyStore extends PolicyStoreBase {
    /**
     * Create a PolicyStore construct that represents an external policy store via policy store id.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param policyStoreId The PolicyStore's id.
     */
    static fromPolicyStoreId(scope, id, policyStoreId) {
        return PolicyStore.fromPolicyStoreAttributes(scope, id, { policyStoreId });
    }
    /**
     * Create a PolicyStore construct that represents an external PolicyStore via policy store arn.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param policyStoreArn The PolicyStore's ARN.
     */
    static fromPolicyStoreArn(scope, id, policyStoreArn) {
        return PolicyStore.fromPolicyStoreAttributes(scope, id, { policyStoreArn });
    }
    /**
     * Creates a PolicyStore construct that represents an external Policy Store.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `PolicyStoreAttributes` object.
     */
    static fromPolicyStoreAttributes(scope, id, attrs) {
        class Import extends PolicyStoreBase {
            constructor(policyStoreArn, policyStoreId) {
                super(scope, id);
                this.policyStoreArn = policyStoreArn;
                this.policyStoreId = policyStoreId;
            }
        }
        let policyStoreId;
        let policyStoreArn;
        const stack = core_1.Stack.of(scope);
        if (!attrs.policyStoreId) {
            if (!attrs.policyStoreArn) {
                throw new Error('One of policyStoreId or policyStoreArn is required!');
            }
            policyStoreArn = attrs.policyStoreArn;
            const maybeId = stack.splitArn(attrs.policyStoreArn, core_1.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
            if (!maybeId) {
                throw new Error(`ARN for PolicyStore must be in the form: ${core_1.ArnFormat.SLASH_RESOURCE_NAME}`);
            }
            policyStoreId = maybeId;
        }
        else {
            if (attrs.policyStoreArn) {
                throw new Error('Only one of policyStoreArn or policyStoreId can be provided');
            }
            policyStoreId = attrs.policyStoreId;
            policyStoreArn = stack.formatArn({
                resource: 'policy-store',
                resourceName: attrs.policyStoreId,
                service: 'verifiedpermissions',
            });
        }
        return new Import(policyStoreArn, policyStoreId);
    }
    /**
     * This method generates a schema based on an swagger file. It makes the same assumptions and decisions
     * made in the Amazon Verified Permissions console. This feature is built for swagger files generated from an Amazon API Gateway
     * export. It's possible that some swagger files generated by other tools will not work. In that case, please
     * file an issue.
     * @param swaggerFilePath absolute path to a swagger file in the local directory structure, in json format
     * @param groupEntityTypeName optional parameter to specify the group entity type name. If passed, the schema's User type will have a parent of this type.
     */
    static schemaFromOpenApiSpec(swaggerFilePath, groupEntityTypeName) {
        const openApiSpecString = fs.readFileSync(swaggerFilePath, 'utf-8');
        const openApiSpec = JSON.parse(openApiSpecString);
        if (!openApiSpec.paths) {
            throw new Error('Invalid OpenAPI spec - missing paths object');
        }
        const namespace = (0, cedar_helpers_1.cleanUpApiNameForNamespace)(openApiSpec.info.title);
        const pathUrls = Object.keys(openApiSpec.paths);
        const actionNames = [];
        for (const pathUrl of pathUrls) {
            const pathDef = openApiSpec.paths[pathUrl];
            if (!pathDef) {
                continue;
            }
            let pathVerbs = Object.keys(pathDef);
            if (pathVerbs.includes('x-amazon-apigateway-any-method')) {
                pathVerbs = RELEVANT_HTTP_METHODS;
            }
            for (const httpVerb of pathVerbs) {
                if (!RELEVANT_HTTP_METHODS.includes(httpVerb)) {
                    continue;
                }
                const actionName = `${httpVerb} ${pathUrl}`;
                actionNames.push(actionName);
            }
        }
        return (0, cedar_helpers_1.buildSchema)(namespace, actionNames, groupEntityTypeName);
    }
    /**
     * This method generates a schema based on an AWS CDK RestApi construct. It makes the same assumptions
     * and decisions made in the Amazon Verified Permissions console.
     *
     * @param restApi The RestApi construct instance from which to generate the schema.
     * @param groupEntityTypeName Specifies a group entity type name. If passed, the schema's User type will have a parent of this type.
     */
    static schemaFromRestApi(restApi, groupEntityTypeName) {
        const namespace = (0, cedar_helpers_1.cleanUpApiNameForNamespace)(restApi.restApiName);
        const actionNames = [];
        for (const method of restApi.methods) {
            const pathVerb = method.httpMethod.toLowerCase();
            const pathUrl = method.resource.path;
            if (pathVerb === 'any') {
                for (const verb of RELEVANT_HTTP_METHODS) {
                    actionNames.push(`${verb} ${pathUrl}`);
                }
            }
            if (RELEVANT_HTTP_METHODS.includes(pathVerb)) {
                actionNames.push(`${pathVerb} ${pathUrl}`);
            }
        }
        return (0, cedar_helpers_1.buildSchema)(namespace, actionNames, groupEntityTypeName);
    }
    constructor(scope, id, props = {
        validationSettings: {
            mode: ValidationSettingsMode.OFF,
        },
        deletionProtection: DeletionProtectionMode.DISABLED,
    }) {
        super(scope, id);
        if (props.schema) {
            (0, cedar_helpers_1.checkParseSchema)(props.schema.cedarJson);
        }
        this.policyStore = new aws_verifiedpermissions_1.CfnPolicyStore(this, id, {
            schema: props.schema
                ? {
                    cedarJson: props.schema.cedarJson,
                }
                : undefined,
            validationSettings: {
                mode: props.validationSettings.mode,
            },
            description: props.description,
            deletionProtection: {
                mode: (props.deletionProtection) ? props.deletionProtection : DeletionProtectionMode.DISABLED,
            },
            tags: props.tags,
        });
        this.policyStoreArn = this.getResourceArnAttribute(this.policyStore.attrArn, {
            resource: 'policy-store',
            resourceName: this.physicalName,
            service: 'verifiedpermissions',
        });
        this.policyStoreName = this.getResourceNameAttribute(this.policyStore.ref);
        this.policyStoreId = this.policyStore.attrPolicyStoreId;
        this.schema = props.schema;
        this.validationSettings = props.validationSettings;
        this.description = props.description;
        this.deletionProtection = (props.deletionProtection) ? props.deletionProtection : DeletionProtectionMode.DISABLED;
    }
    /**
     * Add multiple policies to the policy store
     *
     * @param policyDefinitions An array of policy options for the policy stores policies.
     * @returns An array of created policy constructs.
     */
    addPolicies(policyDefinitions) {
        let policies = policyDefinitions.map((policyOption) => {
            return new policy_1.Policy(this, policyOption.policyId, {
                policyStore: this,
                definition: policyOption.policyConfiguration,
            });
        });
        return policies;
    }
    /**
     * Takes in an absolute path to a directory containing .cedar files and adds the contents of each
     * .cedar file as policies to this policy store. Parses the policies with cedar-wasm and, if the policy store has a schema,
     * performs semantic validation of the policies as well.
     * @param absolutePath a string representing an absolute path to the directory containing your policies
     * @returns An array of created Policy constructs.
     */
    addPoliciesFromPath(absolutePath) {
        if (!fs.statSync(absolutePath).isDirectory()) {
            throw new Error(`The path ${absolutePath} does not appear to be a directory`);
        }
        const policyFileNames = fs
            .readdirSync(absolutePath)
            .map((f) => path.join(absolutePath, f))
            .filter((f) => !fs.statSync(f).isDirectory() && f.endsWith('.cedar'));
        if (this.validationSettings.mode === ValidationSettingsMode.STRICT) {
            if (!this.schema) {
                throw new Error('A schema must exist when adding policies to a policy store with strict validation mode.');
            }
            for (const policyFile of policyFileNames) {
                const policyStatement = fs.readFileSync(policyFile, 'utf-8');
                (0, cedar_helpers_1.validatePolicy)(policyStatement, this.schema.cedarJson);
            }
        }
        const policies = policyFileNames.map((cedarFile) => policy_1.Policy.fromFile(this, cedarFile, {
            path: cedarFile,
            policyStore: this,
        }));
        return policies;
    }
}
exports.PolicyStore = PolicyStore;
_a = JSII_RTTI_SYMBOL_1;
PolicyStore[_a] = { fqn: "@cdklabs/cdk-verified-permissions.PolicyStore", version: "0.2.0" };
//# sourceMappingURL=data:application/json;base64,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