"""Rest Client APIs"""
import json
from http import HTTPStatus

import requests
import requests.exceptions
import simplejson
from magen_utils_apis.compare_utils import default_full_compare

from .rest_exception_apis import handle_specific_exception
from .rest_exception_apis import RestReturn

__author__ = "repenno@cisco.com"
__copyright__ = "Copyright(c) 2015, Cisco Systems, Inc."
__version__ = "0.2"
__status__ = "alpha"


def known_exceptions(func):
    """
    Known Exceptions decorator.
    wraps a given function into try-except statement

    :param func: function to decorate
    :type func: Callable

    :return: decorated
    :rtype: Callable
    """
    def helper(*args, **kwargs):
        """Actual Decorator for handling known exceptions"""
        try:
            return func(*args, **kwargs)
        except (requests.exceptions.RequestException,
                json.JSONDecodeError,
                simplejson.scanner.JSONDecodeError) as err:
            return handle_specific_exception(err)
        except TypeError as err:
            success = False
            return RestReturn(success=success, message=err.args[0])
    return helper


def assign_message_code(success: bool):
    """
    Assign Http status code and phrase based on given Flag
    :param success: given Flag
    :type success: bool

    :return: Http status code and phrase
    :rtype: tuple
    """
    return (HTTPStatus.OK.phrase, HTTPStatus.OK) if success\
            else (HTTPStatus.INTERNAL_SERVER_ERROR.phrase, HTTPStatus.INTERNAL_SERVER_ERROR)


def success_message_code(response_obj, check_util=None):
    """
    Assign values for resulting response

    :param response_obj: object that was generated by request
    :type response_obj: requests.Response
    :param check_util: custom check
    :type check_util: Callable

    :return: values for result, http status and phrase
    :rtype: tuple
    """
    if check_util:
        success = check_util(response_obj)
        phrase, status = assign_message_code(success)
    else:
        success = True
        phrase, status = response_obj.reason, response_obj.status_code
    return success, phrase, status


class RestClientApis(object):
    """APIs for handling REST Requests"""
    put_json_headers = {'content-type': 'application/json', 'Accept': 'application/json'}
    get_json_headers = {'Accept': 'application/json'}

    @staticmethod
    @known_exceptions
    def http_get_and_check_success(url, check_util=None, verify=True, stream=False, auth=None):
        """
        This function will send a GET request and check if the response is OK.

        :param auth: Basic HTTP auth
        :param stream: Whether to keep connection open in order to stream large files
        :param verify: Verify certs
        :param url: HTTP URL
        :type url: str
        :param check_util: An optional function that performs specific application level checks. The function
            must return boolean and take response object as an argument
        :type check_util: Callable
        :param verify: Flag to provide SSL certificate verification or not
        :type verify: bool

        :return: Rest Respond Object
        """
        with requests.Session() as session:
            get_response = session.get(url, verify=verify, stream=stream, timeout=2.0, auth=auth)
            get_response.raise_for_status()
            if get_response.status_code != HTTPStatus.NO_CONTENT and get_response.text:
                get_resp_json = get_response.json()
            else:
                get_resp_json = None
            success, message, return_code = success_message_code(get_response, check_util)

            rest_return_obj = RestReturn(success=success, message=message, http_status=return_code,
                                         json_body=get_resp_json,
                                         response_object=get_response)
            return rest_return_obj

    @staticmethod
    @known_exceptions
    def http_delete_and_check_success(url, check_util=None, verify=True, auth=None):
        """
        This function performs a DELETE request

        :param auth: Basic HTTP auth
        :param check_util: An optional function that performs specific application level checks. The function
            must return boolean and take response object as an argument

        :param url: URL used by the POST request
        :return: Rest Respond Object
        """
        with requests.Session() as session:
            delete_resp = session.delete(
                url,
                verify=verify,
                stream=False,
                timeout=2.0,
                auth=auth)
            delete_resp.raise_for_status()
            if delete_resp.status_code != HTTPStatus.NO_CONTENT and delete_resp.text:
                delete_resp_json = delete_resp.json()
            else:
                delete_resp_json = None

            success, message, return_code = success_message_code(delete_resp, check_util)

            rest_return_obj = RestReturn(success=success, message=message, http_status=return_code,
                                         json_body=delete_resp_json,
                                         response_object=delete_resp)
            return rest_return_obj

    @staticmethod
    @known_exceptions
    def http_post_and_check_success(url, json_req, check_util=None, timeout=2.0, verify=True, location=True,
                                    auth=None, headers=put_json_headers):
        """
        This function performs a POST request and returns the json body to the caller
        for any further processing or validation

        :param headers: HTTP headers to add to request
        :param auth: Basic HTTP auth
        :param location: Verify is location header is present in the response
        :param verify: Should we verify certificates?
        :param timeout: Request timeout
        :param json_req: JSON to send to server
        :param url: URL used by the POST request
        :param check_util: An optional function that performs specific application level checks. The function
            must return boolean and take response object as an argument

        :return: Rest Respond Object
        """
        with requests.Session() as session:
            post_resp = session.post(
                url,
                verify=verify,
                data=json_req,
                headers=headers,
                stream=False,
                timeout=timeout,
                auth=auth)

            post_resp.raise_for_status()
            # When an resource is created we need the Location header properly returned
            if location and 'Location' not in post_resp.headers:
                success = False
                post_resp_json = None
                message = HTTPStatus.INTERNAL_SERVER_ERROR.phrase
                return_code = HTTPStatus.INTERNAL_SERVER_ERROR
            else:
                if post_resp.status_code != HTTPStatus.NO_CONTENT and post_resp.text:
                    post_resp_json = post_resp.json()
                else:
                    post_resp_json = None

                success, message, return_code = success_message_code(post_resp, check_util)

            rest_return_obj = RestReturn(success=success, message=message, http_status=return_code,
                                         json_body=post_resp_json,
                                         response_object=post_resp)
            return rest_return_obj

    @staticmethod
    @known_exceptions
    def http_put_and_check_success(url, json_req, my_function=None, verify=True, headers=put_json_headers,
                                   params=None):
        """
        This function performs a PUT request and returns the json body to the caller
        for any further processing or validation

        :param verify:
        :param my_function:
        :param params: Query strings to add to request
        :param headers: HTTP headers to add to request
        :param json_req: JSON to send to server
        :param url: URL used by the POST request
        :param check_util: An optional function that performs specific application level checks. The function
            must return boolean and take response object as an argument

        :return: Rest Respond Object
        """
        with requests.Session() as session:
            put_resp = session.put(
                url,
                verify=verify,
                data=json_req,
                headers=headers,
                params=params,
                stream=False,
                timeout=2.0)

            put_resp.raise_for_status()
            post_resp_json = put_resp.json() if put_resp.status_code != HTTPStatus.NO_CONTENT and put_resp.text \
                else None
            if my_function:
                success, message, return_code = my_function(put_resp)
            else:
                success = True
                message = put_resp.reason
                return_code = put_resp.status_code
            rest_return_obj = RestReturn(success=success,
                                         http_status=return_code,
                                         message=message, json_body=post_resp_json)
            return rest_return_obj

    # The remaining methods in this class are for testing purposes.
    @staticmethod
    def http_get_location_header(response_obj):
        """Get Location Header for a recource"""
        return response_obj.response_object.headers['Location']

    @staticmethod
    @known_exceptions
    def http_post_and_compare_get_resp(url, json_req, json_resp, check_util=default_full_compare,
                                       timeout=2.0):
        """
        This function performs a POST requests, extract the Location of the created magen_resource
        from the response. Then it performs a subsequent GET to check if magen_resource was created
        and checks if the response matches the expected response.

        :param timeout: Request timeout
        :param check_util: A domain specific function that validates the json expected with
            the json received from the subsequent GET request.
            The function must return boolean.
            The signature should be my_function(json_expected as dict/str, json_received as dict, excluded_keys)
            excluded_keys are keys values of which are generated dynamically
            and are wanted to be skipped during the comparison
        :param url: URL used by the POST request
        :param json_req: The JSON used in the request
        :param json_resp: The expected JSON response of the subsequent GET request
        :return: Rest Respond Object
        """
        post_resp_obj = RestClientApis.http_post_and_check_success(url, json_req, timeout=timeout)
        if post_resp_obj.success:
            # We get the HTTP Location of the subsequent GET request
            location_header = RestClientApis.http_get_location_header(post_resp_obj)
            get_response_obj = RestClientApis.http_get_and_check_success(location_header)
            if get_response_obj.success:
                json_get_resp = get_response_obj.json_body
                if json_get_resp:
                    success = check_util(json.loads(json_resp), json_get_resp)
                    message, return_code = assign_message_code(success)
                    rest_return_obj = RestReturn(success=success, http_status=return_code, message=message,
                                                 response_object=get_response_obj.response_object)
                    return rest_return_obj

                # No JSON body in server response. From the API client perspective this is a 500.
                rest_return_obj = RestReturn(success=False,
                                             message=HTTPStatus.INTERNAL_SERVER_ERROR.phrase,
                                             response_object=get_response_obj.response_object)
                return rest_return_obj
            # Get failed
            return get_response_obj
        # POST failed
        return post_resp_obj

    @staticmethod
    @known_exceptions
    def http_post_and_compare_resp(url, json_req, expected_post_json_resp,
                                   check_util=default_full_compare):
        """
        This function performs a POST requests and check if the response matches the expected response.

        :param check_util: A domain specific function that validates the json expected with
            the json received from the subsequent GET request.
            The function must return boolean.
            The signature should be my_function(json_expected as dict/str, json_received as dict, excluded_keys)
            excluded_keys are keys values of which are generated dynamically
            and are wanted to be skipped during the comparison
        :param url: URL used by the POST request
        :param json_req: The JSON used in the request
        :param expected_post_json_resp: The expected JSON response of the subsequent GET request
        :return: Rest Respond Object
        """
        post_resp_obj = RestClientApis.http_post_and_check_success(url, json_req)
        if post_resp_obj.success:
            success = check_util(json.loads(expected_post_json_resp), post_resp_obj.json_body)
            message, return_code = assign_message_code(success)
            rest_return_obj = RestReturn(success=success,
                                         http_status=return_code,
                                         message=message,
                                         response_object=post_resp_obj.response_object)
            return rest_return_obj
        # Post failed
        return post_resp_obj

    @staticmethod
    @known_exceptions
    def http_get_and_compare_resp(url, expected_get_json_resp, check_util=default_full_compare):
        """
        Given a URL it performs a GET and checks if response matches expected response

        :param check_util: A domain specific function that validates the json expected with
            the json received from the subsequent GET request.
            The function must return boolean.
            The signature should be my_function(json_expected as dict/str, json_received as dict, excluded_keys)
            excluded_keys are keys values of which are generated dynamically
            and are wanted to be skipped during the comparison
        :param url: URL of GET request
        :param expected_get_json_resp: expected JSON response
        :return: Rest Respond Object
        """
        get_resp_obj = RestClientApis.http_get_and_check_success(url)
        if get_resp_obj.success:
            get_resp_json = get_resp_obj.json_body
            success = check_util(json.loads(expected_get_json_resp), get_resp_json)
            message, return_code = assign_message_code(success)
        else:
            return get_resp_obj

        rest_return_obj = RestReturn(success=success, message=message, http_status=return_code,
                                     json_body=get_resp_json,
                                     response_object=get_resp_obj.response_object)
        return rest_return_obj

    @staticmethod
    @known_exceptions
    def http_delete_and_get_check(url):
        """
        This function will delete a magen_resource and send a GET request and check if magen_resource was actually
        deleted. This is more of a test function since if server said something was deleted, it should have,
        otherwise server isbuggy.

        :param url: HTTP URL
        :type url: str
        :return: Rest Respond Object
        """
        delete_resp_obj = RestClientApis.http_delete_and_check_success(url)
        json_body = delete_resp_obj.json_body
        response_object = delete_resp_obj.response_object
        if delete_resp_obj.success:
            # if delete was successful
            get_resp_obj = RestClientApis.http_get_and_check_success(url)
            response_object = get_resp_obj.response_object
            json_body = get_resp_obj.json_body
            if get_resp_obj.http_status == HTTPStatus.NOT_FOUND:
                # if resource was not found we are good
                success = True
                return_code = HTTPStatus.OK
                message = HTTPStatus.OK.phrase
            else:
                success = False
                return_code = HTTPStatus.INTERNAL_SERVER_ERROR
                message = HTTPStatus.INTERNAL_SERVER_ERROR.phrase
        else:
            success = False
            return_code = delete_resp_obj.http_status
            message = delete_resp_obj.message
        rest_return_obj = RestReturn(success=success, message=message, http_status=return_code,
                                     json_body=json_body, response_object=response_object)
        return rest_return_obj

    @staticmethod
    @known_exceptions
    def http_put_and_compare_get_resp(url, json_req, json_resp, check_util=default_full_compare):
        """
        This function performs a PUT requests, performs a subsequent GET to check if magen_resource was created/updated
        and checks if the response matches the expected response.

        :param check_util: A domain specific function that validates the json expected with
            the json received from the subsequent GET request.
            The function must return boolean.
            The signature should be my_function(json_expected as dict/str, json_received as dict, excluded_keys)
            excluded_keys are keys values of which are generated dynamically
            and are wanted to be skipped during the comparison
        :param url: URL used by the PUT request
        :param json_req: The JSON used in the request
        :param json_resp: The expected JSON response of the subsequent GET request
        :return: Rest Respond Object
        """
        put_resp_obj = RestClientApis.http_put_and_check_success(url, json_req)
        if put_resp_obj.success:
            get_response_obj = RestClientApis.http_get_and_check_success(url)
            if get_response_obj.success:
                json_get_resp = get_response_obj.json_body
                if json_get_resp:
                    success = check_util(json.loads(json_resp), json_get_resp)
                    message, return_code = assign_message_code(success)
                    rest_return_obj = RestReturn(success=success,
                                                 http_status=return_code,
                                                 message=message,
                                                 response_object=get_response_obj.response_object)
                    return rest_return_obj

                # No JSON body in server response. From the API client perspective this is a 500.
                rest_return_obj = RestReturn(success=False,
                                             message=HTTPStatus.INTERNAL_SERVER_ERROR.phrase,
                                             response_object=get_response_obj.response_object)
                return rest_return_obj
            # Get failed
            return get_response_obj
        # POST failed
        return put_resp_obj

    @staticmethod
    @known_exceptions
    def http_put_and_compare_resp(url, json_req, json_resp, check_util=default_full_compare):
        """
        This function performs a PUT requests, checks if the response matches the expected response.

        :param check_util: A domain specific function that validates the json expected with
            the json received from the subsequent GET request.
            The function must return boolean.
            The signature should be my_function(json_expected as dict/str, json_received as dict, excluded_keys)
            excluded_keys are keys values of which are generated dynamically
            and are wanted to be skipped during the comparison
        :param url: URL used by the PUT request
        :param json_req: The JSON used in the request
        :param json_resp: The expected JSON response of the PUT request
        :return: Rest Respond Object
        """
        put_resp_obj = RestClientApis.http_put_and_check_success(url, json_req)
        if put_resp_obj.success:
            json_put_resp = put_resp_obj.json_body
            if json_put_resp:
                success = check_util(json.loads(json_resp), json_put_resp)
                message, return_code = assign_message_code(success)
                rest_return_obj = RestReturn(success=success,
                                             http_status=return_code,
                                             message=message,
                                             response_object=put_resp_obj.response_object)
                return rest_return_obj

            # No JSON body in server response. From the API client perspective this is a 500.
            rest_return_obj = RestReturn(success=False,
                                         message=HTTPStatus.INTERNAL_SERVER_ERROR.phrase)
            return rest_return_obj
        # PUT failed
        return put_resp_obj

    @staticmethod
    @known_exceptions
    def http_patch_and_check_success(url, json_req, check_util=None, timeout=2.0, verify=True,
                                    auth=None, headers=put_json_headers):
        """
        This function performs a POST request and returns the json body to the caller
        for any further processing or validation

        :param headers: HTTP headers to add to request
        :param auth: Basic HTTP auth
        :param location: Verify is location header is present in the response
        :param verify: Should we verify certificates?
        :param timeout: Request timeout
        :param json_req: JSON to send to server
        :param url: URL used by the POST request
        :param check_util: An optional function that performs specific application level checks. The function
            must return boolean and take response object as an argument

        :return: Rest Respond Object
        """
        with requests.Session() as session:
            post_resp = session.patch(
                url,
                verify=verify,
                data=json_req,
                headers=headers,
                stream=False,
                timeout=timeout,
                auth=auth)

            post_resp.raise_for_status()

            if post_resp.status_code != HTTPStatus.NO_CONTENT and post_resp.text:
                post_resp_json = post_resp.json()
            else:
                post_resp_json = None

            success, message, return_code = success_message_code(post_resp, check_util)

            rest_return_obj = RestReturn(success=success, message=message, http_status=return_code,
                                         json_body=post_resp_json,
                                         response_object=post_resp)
            return rest_return_obj
