"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Application = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const ec2 = require("aws-cdk-lib/aws-ec2");
const iam = require("aws-cdk-lib/aws-iam");
const aws_kinesisanalytics_1 = require("aws-cdk-lib/aws-kinesisanalytics");
const logs = require("aws-cdk-lib/aws-logs");
const core = require("aws-cdk-lib/core");
const environment_properties_1 = require("./private/environment-properties");
const flink_application_configuration_1 = require("./private/flink-application-configuration");
const validation_1 = require("./private/validation");
/**
 * Implements the functionality shared between CDK created and imported
 * IApplications.
 */
class ApplicationBase extends core.Resource {
    /** Implement the convenience `IApplication.addToPrincipalPolicy` method. */
    addToRolePolicy(policyStatement) {
        if (this.role) {
            this.role.addToPrincipalPolicy(policyStatement);
            return true;
        }
        return false;
    }
    get connections() {
        if (!this._connections) {
            throw new Error('This Application isn\'t associated with a VPC. Provide a "vpc" prop when creating the Application or "securityGroups" when importing it.');
        }
        return this._connections;
    }
    /**
     * Return a CloudWatch metric associated with this Flink application.
     *
     * @param metricName The name of the metric
     * @param props Customization properties
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/KinesisAnalytics',
            metricName,
            dimensionsMap: { Application: this.applicationName },
            ...props,
        }).attachTo(this);
    }
    /**
     * The number of Kinesis Processing Units that are used to run your stream
     * processing application. The average number of KPUs used each hour
     * determines the billing for your application.
     *
     * Units: Count
     *
     * Reporting Level: Application
     *
     * @default - average over 5 minutes
     */
    metricKpus(props) {
        return this.metric('KPUs', { statistic: 'Average', ...props });
    }
    /**
     * The time elapsed during an outage for failing/recovering jobs.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Application
     *
     * @default - average over 5 minutes
     */
    metricDowntime(props) {
        return this.metric('downtime', { statistic: 'Average', ...props });
    }
    /**
     * The time that the job has been running without interruption.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Application
     *
     * @default - average over 5 minutes
     */
    metricUptime(props) {
        return this.metric('uptime', { statistic: 'Average', ...props });
    }
    /**
     * The total number of times this job has fully restarted since it was
     * submitted. This metric does not measure fine-grained restarts.
     *
     * Units: Count
     *
     * Reporting Level: Application
     *
     * @default - sum over 5 minutes
     */
    metricFullRestarts(props) {
        return this.metric('fullRestarts', { statistic: 'Sum', ...props });
    }
    /**
     * The number of times checkpointing has failed.
     *
     * Units: Count
     *
     * Reporting Level: Application
     *
     * @default - sum over 5 minutes
     */
    metricNumberOfFailedCheckpoints(props) {
        return this.metric('numberOfFailedCheckpoints', { statistic: 'Sum', ...props });
    }
    /**
     * The time it took to complete the last checkpoint.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Application
     *
     * @default - maximum over 5 minutes
     */
    metricLastCheckpointDuration(props) {
        return this.metric('lastCheckpointDuration', { statistic: 'Maximum', ...props });
    }
    /**
     * The total size of the last checkpoint.
     *
     * Units: Bytes
     *
     * Reporting Level: Application
     *
     * @default - maximum over 5 minutes
     */
    metricLastCheckpointSize(props) {
        return this.metric('lastCheckpointSize', { statistic: 'Maximum', ...props });
    }
    /**
     * The overall percentage of CPU utilization across task managers. For
     * example, if there are five task managers, Kinesis Data Analytics publishes
     * five samples of this metric per reporting interval.
     *
     * Units: Percentage
     *
     * Reporting Level: Application
     *
     * @default - average over 5 minutes
     */
    metricCpuUtilization(props) {
        return this.metric('cpuUtilization', { statistic: 'Average', ...props });
    }
    /**
     * Overall heap memory utilization across task managers. For example, if there
     * are five task managers, Kinesis Data Analytics publishes five samples of
     * this metric per reporting interval.
     *
     * Units: Percentage
     *
     * Reporting Level: Application
     *
     * @default - average over 5 minutes
     */
    metricHeapMemoryUtilization(props) {
        return this.metric('heapMemoryUtilization', { statistic: 'Average', ...props });
    }
    /**
     * The total time spent performing old garbage collection operations.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Application
     *
     * @default - sum over 5 minutes
     */
    metricOldGenerationGCTime(props) {
        return this.metric('oldGenerationGCTime', { statistic: 'Sum', ...props });
    }
    /**
     * The total number of old garbage collection operations that have occurred
     * across all task managers.
     *
     * Units: Count
     *
     * Reporting Level: Application
     *
     * @default - sum over 5 minutes
     */
    metricOldGenerationGCCount(props) {
        return this.metric('oldGenerationGCCount', { statistic: 'Sum', ...props });
    }
    /**
     * The total number of live threads used by the application.
     *
     * Units: Count
     *
     * Reporting Level: Application
     *
     * @default - average over 5 minutes
     */
    metricThreadsCount(props) {
        return this.metric('threadsCount', { statistic: 'Average', ...props });
    }
    /**
     * The total number of records this application, operator, or task has
     * received.
     *
     * Units: Count
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default - average over 5 minutes
     */
    metricNumRecordsIn(props) {
        return this.metric('numRecordsIn', { statistic: 'Average', ...props });
    }
    /**
     * The total number of records this application, operator or task has received
     * per second.
     *
     * Units: Count/Second
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default - average over 5 minutes
     */
    metricNumRecordsInPerSecond(props) {
        return this.metric('numRecordsInPerSecond', { statistic: 'Average', ...props });
    }
    /**
     * The total number of records this application, operator or task has emitted.
     *
     * Units: Count
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default - average over 5 minutes
     */
    metricNumRecordsOut(props) {
        return this.metric('numRecordsOut', { statistic: 'Average', ...props });
    }
    /**
     * The total number of records this application, operator or task has emitted
     * per second.
     *
     * Units: Count/Second
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default - average over 5 minutes
     */
    metricNumRecordsOutPerSecond(props) {
        return this.metric('numRecordsOutPerSecond', { statistic: 'Average', ...props });
    }
    /**
     * The number of records this operator or task has dropped due to arriving
     * late.
     *
     * Units: Count
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default - sum over 5 minutes
     */
    metricNumLateRecordsDropped(props) {
        return this.metric('numLateRecordsDropped', { statistic: 'Sum', ...props });
    }
    /**
     * The last watermark this application/operator/task/thread has received.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default - maximum over 5 minutes
     */
    metricCurrentInputWatermark(props) {
        return this.metric('currentInputWatermark', { statistic: 'Maximum', ...props });
    }
    /**
     * The last watermark this application/operator/task/thread has received.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default - maximum over 5 minutes
     */
    metricCurrentOutputWatermark(props) {
        return this.metric('currentOutputWatermark', { statistic: 'Maximum', ...props });
    }
    /**
     * The amount of managed memory currently used.
     *
     * Units: Bytes
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default - average over 5 minutes
     */
    metricManagedMemoryUsed(props) {
        return this.metric('managedMemoryUsed', { statistic: 'Average', ...props });
    }
    /**
     * The total amount of managed memory.
     *
     * Units: Bytes
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default - average over 5 minutes
     */
    metricManagedMemoryTotal(props) {
        return this.metric('managedMemoryTotal', { statistic: 'Average', ...props });
    }
    /**
     * Derived from managedMemoryUsed/managedMemoryTotal.
     *
     * Units: Percentage
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default - average over 5 minutes
     */
    metricManagedMemoryUtilization(props) {
        return this.metric('managedMemoryUtilization', { statistic: 'Average', ...props });
    }
    /**
     * The time (in milliseconds) this task or operator is idle (has no data to
     * process) per second. Idle time excludes back pressured time, so if the task
     * is back pressured it is not idle.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Operator, Task, Parallelism
     *
     * @default - average over 5 minutes
     */
    metricIdleTimeMsPerSecond(props) {
        return this.metric('idleTimeMsPerSecond', { statistic: 'Average', ...props });
    }
    /**
     * The time (in milliseconds) this task or operator is back pressured per
     * second.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Operator, Task, Parallelism
     *
     * @default - average over 5 minutes
     */
    metricBackPressuredTimeMsPerSecond(props) {
        return this.metric('backPressuredTimeMsPerSecond', { statistic: 'Average', ...props });
    }
    /**
     * The time (in milliseconds) this task or operator is busy (neither idle nor
     * back pressured) per second. Can be NaN, if the value could not be
     * calculated.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Operator, Task, Parallelism
     *
     * @default - average over 5 minutes
     */
    metricBusyTimePerMsPerSecond(props) {
        return this.metric('busyTimePerMsPerSecond', { statistic: 'Average', ...props });
    }
}
/**
 * An imported Flink application.
 */
class Import extends ApplicationBase {
    constructor(scope, id, attrs) {
        super(scope, id);
        // Imported applications have no associated role or grantPrincipal
        this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
        this.role = undefined;
        this.applicationArn = attrs.applicationArn;
        const applicationName = core.Stack.of(scope).splitArn(attrs.applicationArn, core.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
        if (!applicationName) {
            throw new Error(`applicationArn for fromApplicationArn (${attrs.applicationArn}) must include resource name`);
        }
        this.applicationName = applicationName;
        const securityGroups = attrs.securityGroups ?? [];
        if (securityGroups.length > 0) {
            this._connections = new ec2.Connections({ securityGroups });
        }
    }
}
/**
 * The L2 construct for Flink Kinesis Data Applications.
 *
 * @resource AWS::KinesisAnalyticsV2::Application
 *
 */
class Application extends ApplicationBase {
    /**
     * Import an existing Flink application defined outside of CDK code by
     * applicationName.
     */
    static fromApplicationName(scope, id, applicationName) {
        const applicationArn = core.Stack.of(scope).formatArn(applicationArnComponents(applicationName));
        return new Import(scope, id, { applicationArn });
    }
    /**
     * Import an existing application defined outside of CDK code by
     * applicationArn.
     */
    static fromApplicationArn(scope, id, applicationArn) {
        return new Import(scope, id, { applicationArn });
    }
    /**
     * Import an existing application defined outside of CDK code.
     */
    static fromApplicationAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_kinesisanalytics_flink_alpha_ApplicationAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromApplicationAttributes);
            }
            throw error;
        }
        return new Import(scope, id, {
            applicationArn: attrs.applicationArn,
            securityGroups: attrs.securityGroups,
        });
    }
    constructor(scope, id, props) {
        super(scope, id, { physicalName: props.applicationName });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_kinesisanalytics_flink_alpha_ApplicationProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Application);
            }
            throw error;
        }
        (0, validation_1.validateFlinkApplicationProps)(props);
        this.role = props.role ?? new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('kinesisanalytics.amazonaws.com'),
        });
        this.grantPrincipal = this.role;
        // Permit metric publishing to CloudWatch
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['cloudwatch:PutMetricData'],
            resources: ['*'],
        }));
        const code = props.code.bind(this);
        code.bucket.grantRead(this);
        let vpcConfigurations;
        if (props.vpc) {
            const securityGroups = props.securityGroups ?? [
                new ec2.SecurityGroup(this, 'SecurityGroup', {
                    vpc: props.vpc,
                }),
            ];
            this._connections = new ec2.Connections({ securityGroups });
            const subnetSelection = props.vpcSubnets ?? {
                subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS,
            };
            vpcConfigurations = [{
                    securityGroupIds: securityGroups.map(sg => sg.securityGroupId),
                    subnetIds: props.vpc.selectSubnets(subnetSelection).subnetIds,
                }];
        }
        const resource = new aws_kinesisanalytics_1.CfnApplicationV2(this, 'Resource', {
            applicationName: props.applicationName,
            runtimeEnvironment: props.runtime.value,
            serviceExecutionRole: this.role.roleArn,
            applicationConfiguration: {
                ...code.applicationCodeConfigurationProperty,
                environmentProperties: (0, environment_properties_1.environmentProperties)(props.propertyGroups),
                flinkApplicationConfiguration: (0, flink_application_configuration_1.flinkApplicationConfiguration)({
                    checkpointingEnabled: props.checkpointingEnabled,
                    checkpointInterval: props.checkpointInterval,
                    minPauseBetweenCheckpoints: props.minPauseBetweenCheckpoints,
                    logLevel: props.logLevel,
                    metricsLevel: props.metricsLevel,
                    autoScalingEnabled: props.autoScalingEnabled,
                    parallelism: props.parallelism,
                    parallelismPerKpu: props.parallelismPerKpu,
                }),
                applicationSnapshotConfiguration: {
                    snapshotsEnabled: props.snapshotsEnabled ?? true,
                },
                vpcConfigurations,
            },
        });
        resource.node.addDependency(this.role);
        const logGroup = props.logGroup ?? new logs.LogGroup(this, 'LogGroup');
        const logStream = new logs.LogStream(this, 'LogStream', { logGroup });
        /* Permit logging */
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['logs:DescribeLogGroups'],
            resources: [
                core.Stack.of(this).formatArn({
                    service: 'logs',
                    resource: 'log-group',
                    arnFormat: core.ArnFormat.COLON_RESOURCE_NAME,
                    resourceName: '*',
                }),
            ],
        }));
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['logs:DescribeLogStreams'],
            resources: [logGroup.logGroupArn],
        }));
        const logStreamArn = `arn:${core.Aws.PARTITION}:logs:${core.Aws.REGION}:${core.Aws.ACCOUNT_ID}:log-group:${logGroup.logGroupName}:log-stream:${logStream.logStreamName}`;
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['logs:PutLogEvents'],
            resources: [logStreamArn],
        }));
        new aws_kinesisanalytics_1.CfnApplicationCloudWatchLoggingOptionV2(this, 'LoggingOption', {
            applicationName: resource.ref,
            cloudWatchLoggingOption: {
                logStreamArn,
            },
        });
        // Permissions required for VPC usage per:
        // https://docs.aws.amazon.com/kinesisanalytics/latest/java/vpc-permissions.html
        if (props.vpc) {
            this.role.addToPrincipalPolicy(new iam.PolicyStatement({
                actions: [
                    'ec2:DescribeVpcs',
                    'ec2:DescribeSubnets',
                    'ec2:DescribeSecurityGroups',
                    'ec2:DescribeDhcpOptions',
                    'ec2:CreateNetworkInterface',
                    'ec2:CreateNetworkInterfacePermission',
                    'ec2:DescribeNetworkInterfaces',
                    'ec2:DeleteNetworkInterface',
                ],
                resources: ['*'],
            }));
        }
        this.applicationName = this.getResourceNameAttribute(resource.ref);
        this.applicationArn = this.getResourceArnAttribute(core.Stack.of(this).formatArn(applicationArnComponents(resource.ref)), applicationArnComponents(this.physicalName));
        resource.applyRemovalPolicy(props.removalPolicy, {
            default: core.RemovalPolicy.DESTROY,
        });
    }
}
exports.Application = Application;
_a = JSII_RTTI_SYMBOL_1;
Application[_a] = { fqn: "@aws-cdk/aws-kinesisanalytics-flink-alpha.Application", version: "2.121.1-alpha.0" };
function applicationArnComponents(resourceName) {
    return {
        service: 'kinesisanalytics',
        resource: 'application',
        resourceName,
    };
}
//# sourceMappingURL=data:application/json;base64,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