#  Copyright 2021 Collate
#  Licensed under the Apache License, Version 2.0 (the "License");
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#  http://www.apache.org/licenses/LICENSE-2.0
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.
"""
Abstract Source definition to build a Workflow
"""
import time
from abc import ABCMeta, abstractmethod
from typing import Any, Dict, Generic, Iterable, List

from pydantic import Field

from metadata.generated.schema.entity.services.connections.metadata.openMetadataConnection import (
    OpenMetadataConnection,
)
from metadata.ingestion.api.closeable import Closeable
from metadata.ingestion.api.common import Entity
from metadata.ingestion.api.status import Status
from metadata.ingestion.ometa.ometa_api import OpenMetadata


class InvalidSourceException(Exception):
    """
    The source config is not getting the expected
    service connection
    """


class SourceStatus(Status):
    """
    Class to handle processed records
    and success %
    """

    source_start_time = time.time()

    success: List[Any] = []
    warnings: List[Dict[str, str]] = Field(default_factory=list)
    filtered: List[Dict[str, str]] = Field(default_factory=list)

    def scanned(self, record: Any) -> None:
        self.records.append(record)

    def warning(self, key: str, reason: str) -> None:
        self.warnings.append({key: reason})

    def filter(self, key: str, reason: str) -> None:
        self.filtered.append({key: reason})

    def calculate_success(self) -> float:
        source_success = max(
            len(self.records), 1
        )  # To avoid ZeroDivisionError using minimum value as 1
        source_failed = len(self.failures)
        return round(source_success * 100 / (source_success + source_failed), 2)


class Source(Closeable, Generic[Entity], metaclass=ABCMeta):
    """
    Abstract source implementation. The workflow will run
    its next_record and pass them to the next step.
    """

    metadata: OpenMetadata
    connection_obj: Any
    service_connection: Any
    status: SourceStatus

    def __init__(self):
        self.status = SourceStatus()

    @classmethod
    @abstractmethod
    def create(
        cls, config_dict: dict, metadata_config: OpenMetadataConnection
    ) -> "Source":
        pass

    @abstractmethod
    def prepare(self):
        pass

    @abstractmethod
    def next_record(self) -> Iterable[Entity]:
        pass

    def get_status(self) -> SourceStatus:
        return self.status

    @abstractmethod
    def test_connection(self) -> None:
        pass
