"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
describe('IAM policy statement', () => {
    describe('from JSON', () => {
        test('parses with no principal', () => {
            // given
            const stack = new core_1.Stack();
            const s = new lib_1.PolicyStatement();
            s.addActions('service:action1', 'service:action2');
            s.addAllResources();
            s.addCondition('key', { equals: 'value' });
            const doc1 = new lib_1.PolicyDocument();
            doc1.addStatements(s);
            // when
            const doc2 = lib_1.PolicyDocument.fromJson(doc1.toJSON());
            // then
            expect(stack.resolve(doc2)).toEqual(stack.resolve(doc1));
        });
        test('parses a given Principal', () => {
            const stack = new core_1.Stack();
            const s = new lib_1.PolicyStatement();
            s.addActions('service:action1', 'service:action2');
            s.addAllResources();
            s.addArnPrincipal('somearn');
            s.addCondition('key', { equals: 'value' });
            const doc1 = new lib_1.PolicyDocument();
            doc1.addStatements(s);
            const doc2 = lib_1.PolicyDocument.fromJson(doc1.toJSON());
            expect(stack.resolve(doc2)).toEqual(stack.resolve(doc1));
        });
        test('parses a given notPrincipal', () => {
            const stack = new core_1.Stack();
            const s = new lib_1.PolicyStatement();
            s.addActions('service:action1', 'service:action2');
            s.addAllResources();
            s.addNotPrincipals(new lib_1.AnyPrincipal());
            s.addCondition('key', { equals: 'value' });
            const doc1 = new lib_1.PolicyDocument();
            doc1.addStatements(s);
            const doc2 = lib_1.PolicyDocument.fromJson(doc1.toJSON());
            expect(stack.resolve(doc2)).toEqual(stack.resolve(doc1));
        });
        test('parses with notAction', () => {
            const stack = new core_1.Stack();
            const s = new lib_1.PolicyStatement();
            s.addNotActions('service:action3');
            s.addAllResources();
            const doc1 = new lib_1.PolicyDocument();
            doc1.addStatements(s);
            const doc2 = lib_1.PolicyDocument.fromJson(doc1.toJSON());
            expect(stack.resolve(doc2)).toEqual(stack.resolve(doc1));
        });
        test('parses with notActions', () => {
            const stack = new core_1.Stack();
            const s = new lib_1.PolicyStatement();
            s.addNotActions('service:action3', 'service:action4');
            s.addAllResources();
            const doc1 = new lib_1.PolicyDocument();
            doc1.addStatements(s);
            const doc2 = lib_1.PolicyDocument.fromJson(doc1.toJSON());
            expect(stack.resolve(doc2)).toEqual(stack.resolve(doc1));
        });
        test('parses with notResource', () => {
            const stack = new core_1.Stack();
            const s = new lib_1.PolicyStatement();
            s.addActions('service:action3', 'service:action4');
            s.addNotResources('resource1');
            const doc1 = new lib_1.PolicyDocument();
            doc1.addStatements(s);
            const doc2 = lib_1.PolicyDocument.fromJson(doc1.toJSON());
            expect(stack.resolve(doc2)).toEqual(stack.resolve(doc1));
        });
        test('parses with notResources', () => {
            const stack = new core_1.Stack();
            const s = new lib_1.PolicyStatement();
            s.addActions('service:action3', 'service:action4');
            s.addNotResources('resource1', 'resource2');
            const doc1 = new lib_1.PolicyDocument();
            doc1.addStatements(s);
            const doc2 = lib_1.PolicyDocument.fromJson(doc1.toJSON());
            expect(stack.resolve(doc2)).toEqual(stack.resolve(doc1));
        });
        test('the kitchen sink', () => {
            const stack = new core_1.Stack();
            const policyDocument = {
                Version: '2012-10-17',
                Statement: [
                    {
                        Sid: 'FirstStatement',
                        Effect: 'Allow',
                        Action: 'iam:ChangePassword',
                        Resource: '*',
                    },
                    {
                        Sid: 'SecondStatement',
                        Effect: 'Allow',
                        Action: 's3:ListAllMyBuckets',
                        Resource: '*',
                    },
                    {
                        Sid: 'ThirdStatement',
                        Effect: 'Allow',
                        Action: [
                            's3:List*',
                            's3:Get*',
                        ],
                        Resource: [
                            'arn:aws:s3:::confidential-data',
                            'arn:aws:s3:::confidential-data/*',
                        ],
                        Condition: { Bool: { 'aws:MultiFactorAuthPresent': 'true' } },
                    },
                ],
            };
            const doc = lib_1.PolicyDocument.fromJson(policyDocument);
            expect(stack.resolve(doc)).toEqual(policyDocument);
        });
        test('throws error with field data being object', () => {
            expect(() => {
                lib_1.PolicyStatement.fromJson({
                    Action: {},
                });
            }).toThrow(/Fields must be either a string or an array of strings/);
        });
        test('throws error with field data being array of non-strings', () => {
            expect(() => {
                lib_1.PolicyStatement.fromJson({
                    Action: [{}],
                });
            }).toThrow(/Fields must be either a string or an array of strings/);
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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