import os
import zipfile

from turbomail import Message
from genshi.template import TextTemplate
from pyjon.utils import get_secure_filename

from pyf.dataflow import component
from pyf.componentized.components import Component
from pyf.componentized.configuration.keys import SimpleKey, CDATAKey
from pyf.services.core.storage import get_storage


class EmailSender(Component):
    """Send files generated by the process to specified recipients via emails.

    This node is to be used post process, not during the process.

    It is controlled by four configuration keys:
      - from: the sender of the email
      - to: the recipient(s) of the email, separated by semi-columns (;)
      - subject: the subject of the email, as a ONE-LINE Genshi text template
      - body: the main body of the email, as a Genshi text template

    An example of usage is:
    <node type="consumer" pluginname="email_sender" name="send_reports">
      <from>noreply@example.com</output_folder>
      <to>john.smith@example.com</to>
      <subject>Process ${process_name} successfully completed</subject>
      <body>
        <![CDATA[
Hi,

The process ${process_name} (var. ${variant_name}) has finished.
Please find enclosed the output files.

Sincerely,
Your friendly server
        ]]>
      </body>
    </node>

    The following variables are available in the subject and body:
      - process_name: the display_name of the tube that was run
      - variant_name: the name of the variant
    """
    name = "email_sender"

    configuration = [
            SimpleKey('from', help_text="Sender's email address"),
            SimpleKey('to', help_text="Recipients' email addresses"),
            SimpleKey('subject', help_text="The subject of the email"),
            CDATAKey('body', help_text="The body of the email"),
            ]

    def __init__(self, config_node, process_name):
        #super(EmailSender, self).__init__(config_node, name)
        self.config_node = config_node
        self.process_name = process_name

        self.storage = get_storage('output')

    @component('IN', 'OUT')
    def launch(self, values, out):
        for event in values:
            to = list()

            to_conf = self.get_config_key('to')
            if bool(to_conf):
                to.extend([addr.strip() for addr in to_conf.split(';')])
            
            for msg in event.history:
                if msg.user:
                    to.append((msg.user.display_name, msg.user.email_address))
            
            if not to:
                raise ValueError(
                        'No recipient specified for the postprocess email')

            # those are the vars we make accessible to the templates generation
            tmpl_vars = dict(
                    process_name=event.tube.display_name,
                    variant_name=event.variant_name,
                    )

            subject_tmpl = TextTemplate(self.get_config_key('subject'))
            body_tmpl = TextTemplate(self.get_config_key('body'))

            message = Message(self.get_config_key('from'), to,
                    subject_tmpl.generate(**tmpl_vars).render())
            message.plain = body_tmpl.generate(**tmpl_vars).render()

            if event.output_files:
                filename = get_secure_filename()
                zf = zipfile.ZipFile(filename, 'w', zipfile.ZIP_DEFLATED)

                for output_file in event.output_files:
                    storage_filename = self.storage.get_filename(output_file.file_uuid).replace('\\', '/')
                    zf.write(str(storage_filename), str(output_file.filename))
                    yield True

                zf.close()
                message.attach(open(filename, 'rb'), "data.zip")
                os.unlink(filename)
            
            message.send()
            yield True

        yield True

