from typing import TypedDict, List, Optional, Literal, Union, Any, Awaitable, Callable

class TokenUsage(TypedDict):
    """
    Represents token usage statistics for AI model requests.
    
    Attributes:
        promptTokens (int): Number of tokens in the input prompt
        completionTokens (int): Number of tokens in the generated completion
        totalTokens (int): Total number of tokens used (prompt + completion)
    """
    promptTokens: int
    completionTokens: int
    totalTokens: int

class ReasoningDetail(TypedDict):
    """
    Represents detailed reasoning information from AI models.
    
    Attributes:
        type (Literal['text', 'redacted']): Type of reasoning detail
        text (Optional[str]): The reasoning text content
        signature (Optional[str]): Cryptographic signature if applicable
        data (Optional[str]): Additional reasoning data
    """
    type: Literal['text', 'redacted']
    text: Optional[str]
    signature: Optional[str]
    data: Optional[str]

class Source(TypedDict):
    """
    Represents a source reference used by the AI model.
    
    Attributes:
        sourceType (Literal['url']): Type of source (currently only 'url')
        id (str): Unique identifier for the source
        url (str): URL of the source
        title (Optional[str]): Title of the source content
        providerMetadata (Optional[Any]): Provider-specific metadata
    """
    sourceType: Literal['url']
    id: str
    url: str
    title: Optional[str]
    providerMetadata: Optional[Any] # LanguageModelV1ProviderMetadata

class GeneratedFile(TypedDict):
    """
    Represents a file generated by the AI model.
    
    Attributes:
        base64 (str): Base64-encoded file content
        uint8Array (bytes): Binary file content as bytes
        mimeType (str): MIME type of the generated file
    """
    base64: str
    uint8Array: bytes # Assuming Uint8Array translates to bytes
    mimeType: str

class ToolCall(TypedDict):
    # Based on OpenAI's tool call structure
    id: str
    type: Literal['function']
    function: dict[str, str]

class ToolResult(TypedDict):
    tool_call_id: str
    result: Any

class Warning(TypedDict):
    # Assuming a simple structure
    message: str

class Response(TypedDict):
    id: str
    model: str
    timestamp: str # Using string for timestamp for simplicity
    headers: Optional[dict[str, str]]

class ResponseMessage(TypedDict):
    role: str
    content: str
    tool_calls: Optional[List[ToolCall]]

class StepResult(TypedDict):
    # This is a bit vague, I'll assume a simple structure for now
    step: int
    tool_calls: Optional[List[ToolCall]]
    tool_results: Optional[List[ToolResult]]

class OnFinishResult(TypedDict):
    finishReason: Literal["stop", "length", "content-filter", "tool-calls", "error", "other", "unknown"]
    usage: TokenUsage
    providerMetadata: Optional[dict[str, dict[str, Any]]]
    text: str
    reasoning: Optional[str]
    reasoningDetails: List[ReasoningDetail]
    sources: List[Source]
    files: List[GeneratedFile]
    toolCalls: List[ToolCall]
    toolResults: List[ToolResult]
    warnings: Optional[List[Warning]]
    response: Response
    messages: List[ResponseMessage]
    steps: List[StepResult]

OnFinish = Callable[[OnFinishResult], Union[Awaitable[None], None]]
