"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const apigw = require("@aws-cdk/aws-apigateway");
const acm = require("@aws-cdk/aws-certificatemanager");
const route53 = require("@aws-cdk/aws-route53");
const core_1 = require("@aws-cdk/core");
const targets = require("../lib");
test('targets.ApiGateway can be used to the default domain of an APIGW', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const cert = new acm.Certificate(stack, 'cert', { domainName: 'example.com' });
    const api = new apigw.RestApi(stack, 'api', {
        domainName: {
            domainName: 'example.com',
            certificate: cert,
        },
    });
    const zone = new route53.HostedZone(stack, 'zone', {
        zoneName: 'example.com',
    });
    api.root.addMethod('GET');
    // WHEN
    new route53.ARecord(stack, 'A', {
        zone,
        target: route53.RecordTarget.fromAlias(new targets.ApiGateway(api)),
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::Route53::RecordSet', {
        Name: 'example.com.',
        Type: 'A',
        AliasTarget: {
            DNSName: {
                'Fn::GetAtt': [
                    'apiCustomDomain64773C4F',
                    'RegionalDomainName',
                ],
            },
            HostedZoneId: {
                'Fn::GetAtt': [
                    'apiCustomDomain64773C4F',
                    'RegionalHostedZoneId',
                ],
            },
        },
        HostedZoneId: {
            Ref: 'zoneEB40FF1E',
        },
    }));
});
test('targets.ApiGatewayDomain can be used to directly reference a domain', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const cert = new acm.Certificate(stack, 'cert', { domainName: 'example.com' });
    const domain = new apigw.DomainName(stack, 'domain', { domainName: 'example.com', certificate: cert });
    const zone = new route53.HostedZone(stack, 'zone', {
        zoneName: 'example.com',
    });
    // WHEN
    new route53.ARecord(stack, 'A', {
        zone,
        target: route53.RecordTarget.fromAlias(new targets.ApiGatewayDomain(domain)),
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::Route53::RecordSet', {
        Name: 'example.com.',
        Type: 'A',
        AliasTarget: {
            DNSName: {
                'Fn::GetAtt': [
                    'domainFBFFA2F6',
                    'RegionalDomainName',
                ],
            },
            HostedZoneId: {
                'Fn::GetAtt': [
                    'domainFBFFA2F6',
                    'RegionalHostedZoneId',
                ],
            },
        },
        HostedZoneId: {
            Ref: 'zoneEB40FF1E',
        },
    }));
});
test('fails if an ApiGateway is used with an API that does not define a domain name', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const api = new apigw.RestApi(stack, 'api');
    const zone = new route53.HostedZone(stack, 'zone', {
        zoneName: 'example.com',
    });
    api.root.addMethod('GET');
    // THEN
    expect(() => {
        new route53.ARecord(stack, 'A', {
            zone,
            target: route53.RecordTarget.fromAlias(new targets.ApiGateway(api)),
        });
    }).toThrow(/API does not define a default domain name/);
});
//# sourceMappingURL=data:application/json;base64,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