"""OTP service"""

import logging
import json
from datetime import datetime, timezone
from typing import Dict, Any
import requests

from ..dto.request.otp_request_dto import OtpRequestDTO
from ..constants.auth_constants import OTP_REQUEST_ID, TRANSACTION_ID
from ..service.auth_service import AuthService

logger = logging.getLogger(__name__)


class OtpService:
    """OTP service"""

    def __init__(self, config: Dict[str, Any], auth_service: AuthService):
        self.config = config
        self.auth_service = auth_service

    def request_otp(self, otp_request_dto: OtpRequestDTO) -> Dict[str, Any]:
        """Request OTP"""
        try:
            otp_request_dto.id = OTP_REQUEST_ID
            otp_request_dto.request_time = self._get_utc_current_datetime_iso_string()
            otp_request_dto.transaction_id = TRANSACTION_ID
            otp_request_dto.version = "1.0"

            if not otp_request_dto.otp_channel:
                otp_request_dto.otp_channel = ["email"]

            req_json = json.dumps(otp_request_dto.to_dict())

            headers = {
                "Content-Type": "application/json",
                "signature": self.auth_service.get_signature(req_json),
                "Authorization": "Null"
            }

            url = self.config.get("ida.otp.url")
            logger.info(f"[OTP] Sending OTP request to URL: {url}")

            response = requests.post(
                url,
                json=otp_request_dto.to_dict(),
                headers=headers,
                verify=self.config.get("ida.ssl.verify", False)
            )

            return response.json()
        except Exception as e:
            logger.error("[OTP] Failed to request OTP", exc_info=True)
            raise


    def _get_utc_current_datetime_iso_string(self) -> str:
        """Get current UTC datetime as ISO string with Z suffix"""
        dt = datetime.now(timezone.utc)
        iso_str = dt.strftime('%Y-%m-%dT%H:%M:%S.%f')[:-3]
        return f"{iso_str}Z"
