import torch
import torch.nn as nn
import torch.utils.model_zoo as model_zoo

from ForensicHub.registry import register_model
from ForensicHub.core.base_model import BaseModel

model_urls = {
    'resnet18': 'https://download.pytorch.org/models/resnet18-5c106cde.pth',
    'resnet34': 'https://download.pytorch.org/models/resnet34-333f7ec4.pth',
    'resnet50': 'https://download.pytorch.org/models/resnet50-19c8e357.pth',
    'resnet101': 'https://download.pytorch.org/models/resnet101-5d3b4d8f.pth',
    'resnet152': 'https://download.pytorch.org/models/resnet152-b121ed2d.pth',
}


def conv3x3(in_planes, out_planes, stride=1):
    """3x3 convolution with padding"""
    return nn.Conv2d(in_planes, out_planes, kernel_size=3, stride=stride,
                     padding=1, bias=False)


def conv1x1(in_planes, out_planes, stride=1):
    """1x1 convolution"""
    return nn.Conv2d(in_planes, out_planes, kernel_size=1, stride=stride, bias=False)


class BasicBlock(BaseModel):
    expansion = 1

    def __init__(self, inplanes, planes, stride=1, downsample=None):
        super(BasicBlock, self).__init__()
        self.conv1 = conv3x3(inplanes, planes, stride)
        self.bn1 = nn.BatchNorm2d(planes)
        self.relu = nn.ReLU(inplace=True)
        self.conv2 = conv3x3(planes, planes)
        self.bn2 = nn.BatchNorm2d(planes)
        self.downsample = downsample
        self.stride = stride

    def forward(self, x):
        identity = x

        out = self.conv1(x)
        out = self.bn1(out)
        out = self.relu(out)

        out = self.conv2(out)
        out = self.bn2(out)

        if self.downsample is not None:
            identity = self.downsample(x)

        out += identity
        out = self.relu(out)

        return out


class Bottleneck(nn.Module):
    expansion = 4

    def __init__(self, inplanes, planes, stride=1, downsample=None):
        super(Bottleneck, self).__init__()
        self.conv1 = conv1x1(inplanes, planes)
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = conv3x3(planes, planes, stride)
        self.bn2 = nn.BatchNorm2d(planes)
        self.conv3 = conv1x1(planes, planes * self.expansion)
        self.bn3 = nn.BatchNorm2d(planes * self.expansion)
        self.relu = nn.ReLU(inplace=True)
        self.downsample = downsample
        self.stride = stride

    def forward(self, x):
        identity = x

        out = self.conv1(x)
        out = self.bn1(out)
        out = self.relu(out)

        out = self.conv2(out)
        out = self.bn2(out)
        out = self.relu(out)

        out = self.conv3(out)
        out = self.bn3(out)

        if self.downsample is not None:
            identity = self.downsample(x)

        out += identity
        out = self.relu(out)

        return out


class ResNet(BaseModel):

    def __init__(self, block, layers, num_classes=1000, zero_init_residual=False):
        super(ResNet, self).__init__()
        self.inplanes = 64
        self.conv1 = nn.Conv2d(3, 64, kernel_size=7, stride=2, padding=3,
                               bias=False)
        self.bn1 = nn.BatchNorm2d(64)
        self.relu = nn.ReLU(inplace=True)
        self.maxpool = nn.MaxPool2d(kernel_size=3, stride=2, padding=1)
        self.layer1 = self._make_layer(block, 64, layers[0])
        self.layer2 = self._make_layer(block, 128, layers[1], stride=2)
        self.layer3 = self._make_layer(block, 256, layers[2], stride=2)
        self.layer4 = self._make_layer(block, 512, layers[3], stride=2)
        self.avgpool = nn.AdaptiveAvgPool2d((1, 1))
        self.fc = nn.Linear(512 * block.expansion, num_classes)
        # print(num_classes)
        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                nn.init.kaiming_normal_(m.weight, mode='fan_out', nonlinearity='relu')
            elif isinstance(m, nn.BatchNorm2d):
                nn.init.constant_(m.weight, 1)
                nn.init.constant_(m.bias, 0)

        # Zero-initialize the last BN in each residual branch,
        # so that the residual branch starts with zeros, and each residual block behaves like an identity.
        # This improves the model by 0.2~0.3% according to https://arxiv.org/abs/1706.02677
        if zero_init_residual:
            for m in self.modules():
                if isinstance(m, Bottleneck):
                    nn.init.constant_(m.bn3.weight, 0)
                elif isinstance(m, BasicBlock):
                    nn.init.constant_(m.bn2.weight, 0)

    def _make_layer(self, block, planes, blocks, stride=1):
        downsample = None
        if stride != 1 or self.inplanes != planes * block.expansion:
            downsample = nn.Sequential(
                conv1x1(self.inplanes, planes * block.expansion, stride),
                nn.BatchNorm2d(planes * block.expansion),
            )

        layers = []
        layers.append(block(self.inplanes, planes, stride, downsample))
        self.inplanes = planes * block.expansion
        for _ in range(1, blocks):
            layers.append(block(self.inplanes, planes))

        return nn.Sequential(*layers)

    def forward(self, image: torch.Tensor, label: torch.Tensor, **kwargs):

        label = label.float()

        x = image

        x = self.conv1(x)
        x = self.bn1(x)
        x = self.relu(x)
        x = self.maxpool(x)

        x = self.layer1(x)
        x = self.layer2(x)
        x = self.layer3(x)
        x = self.layer4(x)

        x = self.avgpool(x)
        x = x.view(x.size(0), -1)
        x = self.fc(x)

        x = x.squeeze()
        combined_loss = torch.nn.functional.binary_cross_entropy_with_logits(x, label)
        data_dict = {
            "backward_loss": combined_loss,
            "pred_label": x.sigmoid(),
            # optional below
            "visual_loss": {
                "combined_loss": combined_loss
            }
        }

        return data_dict

    def load_state_dict(self, pretrained_dict):
        model_dict = self.state_dict()
        filtered_dict = {k: v for k, v in pretrained_dict.items() if not k.startswith("fc.") and k in model_dict}
        model_dict.update(filtered_dict)
        super().load_state_dict(model_dict, strict=False)


@register_model("Resnet18")
class Resnet18(ResNet):
    def __init__(self, pretrained=False, **kwargs):
        super().__init__(block=BasicBlock, layers=[2, 2, 2, 2], **kwargs)
        if pretrained:
            self.load_state_dict(model_zoo.load_url(model_urls['resnet18']))


@register_model("Resnet34")
class Resnet34(ResNet):
    def __init__(self, pretrained=False, **kwargs):
        super().__init__(block=BasicBlock, layers=[3, 4, 6, 3], **kwargs)
        if pretrained:
            self.load_state_dict(model_zoo.load_url(model_urls['resnet34']))


@register_model("Resnet50")
class Resnet50(ResNet):
    def __init__(self, pretrained=False, **kwargs):
        super().__init__(block=Bottleneck, layers=[3, 4, 6, 3], **kwargs)
        if pretrained:
            self.load_state_dict(model_zoo.load_url(model_urls['resnet50']))


@register_model("Resnet101")
class Resnet101(ResNet):
    def __init__(self, pretrained=False, **kwargs):
        super().__init__(block=Bottleneck, layers=[3, 4, 23, 3], **kwargs)
        if pretrained:
            self.load_state_dict(model_zoo.load_url(model_urls['resnet101']))


@register_model("Resnet152")
class Resnet152(ResNet):
    def __init__(self, pretrained=False, **kwargs):
        super().__init__(block=Bottleneck, layers=[3, 8, 36, 3], **kwargs)
        if pretrained:
            self.load_state_dict(model_zoo.load_url(model_urls['resnet152']))
