"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiGatewayToLambda = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cdk = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const apigateway = require("aws-cdk-lib/aws-apigateway");
const iam = require("aws-cdk-lib/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
const resources = require("@aws-solutions-constructs/resources");
class OpenApiGatewayToLambda extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        CheckOpenapiProps(props);
        const apiDefinitionBucket = props.apiDefinitionBucket ?? props.apiDefinitionAsset?.bucket;
        const apiDefinitionKey = props.apiDefinitionKey ?? props.apiDefinitionAsset?.s3ObjectKey;
        // store a counter to be able to uniquely name lambda functions to avoid naming collisions
        let lambdaCounter = 0;
        this.apiLambdaFunctions = props.apiIntegrations.map(apiIntegration => {
            if (apiIntegration.existingLambdaObj && apiIntegration.lambdaFunctionProps) {
                throw new Error(`Error - Cannot provide both lambdaFunctionProps and existingLambdaObj in an ApiIntegrationfor the api integration with id: ${apiIntegration.id}`);
            }
            if (apiIntegration.existingLambdaObj || apiIntegration.lambdaFunctionProps) {
                return {
                    id: apiIntegration.id,
                    lambdaFunction: defaults.buildLambdaFunction(this, {
                        existingLambdaObj: apiIntegration.existingLambdaObj,
                        lambdaFunctionProps: apiIntegration.lambdaFunctionProps
                    }, `${apiIntegration.id}ApiFunction${lambdaCounter++}`)
                };
            }
            else {
                throw new Error(`One of existingLambdaObj or lambdaFunctionProps must be specified for the api integration with id: ${apiIntegration.id}`);
            }
        });
        // Map each id and lambda function pair to the required format for the template writer custom resource
        const apiIntegrationUris = this.apiLambdaFunctions.map(apiLambdaFunction => {
            // the placeholder string that will be replaced in the OpenAPI Definition
            const uriPlaceholderString = apiLambdaFunction.id;
            // the endpoint URI of the backing lambda function, as defined in the API Gateway extensions for OpenAPI here:
            // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-integration.html
            const uriResolvedValue = `arn:${aws_cdk_lib_1.Aws.PARTITION}:apigateway:${aws_cdk_lib_1.Aws.REGION}:lambda:path/2015-03-31/functions/${apiLambdaFunction.lambdaFunction.functionArn}/invocations`;
            return {
                id: uriPlaceholderString,
                value: uriResolvedValue
            };
        });
        // This custom resource will overwrite the string placeholders in the openapi definition with the resolved values of the lambda URIs
        const apiDefinitionWriter = resources.createTemplateWriterCustomResource(this, 'Api', {
            // CheckAlbProps() has confirmed the existence of these values
            templateBucket: apiDefinitionBucket,
            templateKey: apiDefinitionKey,
            templateValues: apiIntegrationUris,
            timeout: props.internalTransformTimeout ?? cdk.Duration.minutes(1),
            memorySize: props.internalTransformMemorySize ?? 1024
        });
        const specRestApiResponse = defaults.CreateSpecRestApi(this, {
            ...props.apiGatewayProps,
            apiDefinition: apigateway.ApiDefinition.fromBucket(apiDefinitionWriter.s3Bucket, apiDefinitionWriter.s3Key)
        }, props.logGroupProps);
        this.apiGateway = specRestApiResponse.api;
        this.apiGatewayCloudWatchRole = specRestApiResponse.role;
        this.apiGatewayLogGroup = specRestApiResponse.logGroup;
        // Redeploy the API any time the incoming API definition changes (from asset or s3 object)
        this.apiGateway.latestDeployment?.addToLogicalId(apiDefinitionKey);
        this.apiLambdaFunctions.forEach(apiLambdaFunction => {
            // Redeploy the API any time one of the lambda functions changes
            this.apiGateway.latestDeployment?.addToLogicalId(apiLambdaFunction.lambdaFunction.functionArn);
            // Grant APIGW invocation rights for each lambda function
            apiLambdaFunction.lambdaFunction.addPermission('PermitAPIGInvocation', {
                principal: new iam.ServicePrincipal('apigateway.amazonaws.com'),
                sourceArn: this.apiGateway.arnForExecuteApi('*')
            });
        });
    }
}
exports.OpenApiGatewayToLambda = OpenApiGatewayToLambda;
_a = JSII_RTTI_SYMBOL_1;
OpenApiGatewayToLambda[_a] = { fqn: "@aws-solutions-constructs/aws-openapigateway-lambda.OpenApiGatewayToLambda", version: "2.61.0" };
function CheckOpenapiProps(props) {
    let errorMessages = '';
    let errorFound = false;
    if (props.apiDefinitionAsset && (props.apiDefinitionBucket || props.apiDefinitionKey)) {
        errorMessages += 'Either apiDefinitionBucket/apiDefinitionKey or apiDefinitionAsset must be specified, but not both\n';
        errorFound = true;
    }
    const apiDefinitionBucket = props.apiDefinitionBucket ?? props.apiDefinitionAsset?.bucket;
    const apiDefinitionKey = props.apiDefinitionKey ?? props.apiDefinitionAsset?.s3ObjectKey;
    if (apiDefinitionBucket === undefined || apiDefinitionKey === undefined) {
        errorMessages += 'Either apiDefinitionBucket/apiDefinitionKey or apiDefinitionAsset must be specified\n';
        errorFound = true;
    }
    if (props.apiIntegrations === undefined || props.apiIntegrations.length < 1) {
        errorMessages += 'At least one ApiIntegration must be specified in the apiIntegrations property\n';
        errorFound = true;
    }
    if (errorFound) {
        throw new Error(errorMessages);
    }
}
//# sourceMappingURL=data:application/json;base64,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