"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const s3 = require("aws-cdk-lib/aws-s3");
const aws_s3_assets_1 = require("aws-cdk-lib/aws-s3-assets");
const path = require("path");
const assertions_1 = require("aws-cdk-lib/assertions");
const core_1 = require("@aws-solutions-constructs/core");
const defaults = require("@aws-solutions-constructs/core");
test('Simple deployment works', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const construct = new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
        apiDefinitionAsset,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                lambdaFunctionProps: {
                    runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                }
            }
        ]
    });
    expect(construct.apiGateway).toBeDefined();
    expect(construct.apiGatewayCloudWatchRole).toBeDefined();
    expect(construct.apiGatewayLogGroup).toBeDefined();
    expect(construct.apiLambdaFunctions.length).toEqual(1);
    expect(construct.apiLambdaFunctions[0].id).toEqual('MessagesHandler');
    expect(construct.apiLambdaFunctions[0].lambdaFunction).toBeDefined();
});
test('API Definition can be specified from Asset', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const construct = new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
        apiDefinitionAsset,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                lambdaFunctionProps: {
                    runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                }
            }
        ]
    });
    expect(construct.apiLambdaFunctions[0].id).toEqual('MessagesHandler');
    expect(construct.apiLambdaFunctions[0].lambdaFunction).toBeDefined();
});
test('API Definition can be specified from S3 Bucket and Key', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionBucket = core_1.CreateScrapBucket(stack, "scrapBucket");
    const apiDefinitionKey = 'api.yaml';
    const construct = new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
        apiDefinitionBucket,
        apiDefinitionKey,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                lambdaFunctionProps: {
                    runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                }
            }
        ]
    });
    expect(construct.apiLambdaFunctions[0].id).toEqual('MessagesHandler');
    expect(construct.apiLambdaFunctions[0].lambdaFunction).toBeDefined();
});
test('Throws error when both api definition asset and s3 object are specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const app = () => {
        new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
            apiDefinitionAsset,
            apiDefinitionBucket: new s3.Bucket(stack, 'ApiDefinitionBucket'),
            apiDefinitionKey: 'key',
            apiIntegrations: [
                {
                    id: 'MessagesHandler',
                    lambdaFunctionProps: {
                        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                        handler: 'index.handler',
                        code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                    }
                }
            ]
        });
    };
    expect(app).toThrowError('Either apiDefinitionBucket/apiDefinitionKey or apiDefinitionAsset must be specified, but not both');
});
test('Multiple Lambda Functions can be specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
        apiDefinitionAsset,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                lambdaFunctionProps: {
                    runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                }
            }, {
                id: 'PhotosHandler',
                lambdaFunctionProps: {
                    runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/photos-lambda`),
                }
            }
        ]
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::Lambda::Function', 4);
});
test('Existing lambda function can be specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const existingLambdaObj = new lambda.Function(stack, 'ExistingLambda', {
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler',
        functionName: 'ExistingLambdaFunction',
        code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
    });
    new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
        apiDefinitionAsset,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                existingLambdaObj
            }, {
                id: 'PhotosHandler',
                lambdaFunctionProps: {
                    runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                    handler: 'index.handler',
                    functionName: 'NewLambdaFunction',
                    code: lambda.Code.fromAsset(`${__dirname}/photos-lambda`),
                }
            }
        ]
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::Lambda::Function', 4);
    template.hasResourceProperties("AWS::Lambda::Function", {
        FunctionName: 'ExistingLambdaFunction'
    });
    template.hasResourceProperties("AWS::Lambda::Function", {
        FunctionName: 'NewLambdaFunction'
    });
});
test('Throws error when neither existingLambdaObj or lambdaFunctionProps is specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const app = () => {
        new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
            apiDefinitionAsset,
            apiIntegrations: [
                {
                    id: 'MessagesHandler'
                }
            ]
        });
    };
    expect(app).toThrowError('One of existingLambdaObj or lambdaFunctionProps must be specified for the api integration with id: MessagesHandler');
});
test('Throws error when no api definition is specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => {
        new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
            apiIntegrations: [
                {
                    id: 'MessagesHandler',
                    lambdaFunctionProps: {
                        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                        handler: 'index.handler',
                        code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                    }
                }
            ]
        });
    };
    expect(app).toThrowError('Either apiDefinitionBucket/apiDefinitionKey or apiDefinitionAsset must be specified');
});
test('Throws error when no api integration is specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const app = () => {
        new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
            apiDefinitionAsset,
            apiIntegrations: []
        });
    };
    expect(app).toThrowError('At least one ApiIntegration must be specified in the apiIntegrations property');
});
test('Throws error when api definition s3 bucket is specified but s3 object key is missing', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionBucket = core_1.CreateScrapBucket(stack, "scrapBucket");
    const app = () => {
        new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
            apiDefinitionBucket,
            apiIntegrations: []
        });
    };
    expect(app).toThrowError('Either apiDefinitionBucket/apiDefinitionKey or apiDefinitionAsset must be specified');
});
test('Throws error when api definition s3 object key is specified but s3 bucket is missing', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => {
        new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
            apiDefinitionKey: 'prefix/api-definition.yml',
            apiIntegrations: []
        });
    };
    expect(app).toThrowError('Either apiDefinitionBucket/apiDefinitionKey or apiDefinitionAsset must be specified');
});
test('Throws error when api is defined as asset and s3 bucket is specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const apiDefinitionBucket = core_1.CreateScrapBucket(stack, "scrapBucket");
    const app = () => {
        new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
            apiDefinitionAsset,
            apiDefinitionBucket,
            apiIntegrations: []
        });
    };
    expect(app).toThrowError('Either apiDefinitionBucket/apiDefinitionKey or apiDefinitionAsset must be specified');
});
test('Throws error when api is defined as asset and s3 key is specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const app = () => {
        new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
            apiDefinitionAsset,
            apiDefinitionKey: 'prefix/api-definition.yml',
            apiIntegrations: []
        });
    };
    expect(app).toThrowError('Either apiDefinitionBucket/apiDefinitionKey or apiDefinitionAsset must be specified');
});
test('Two Constructs create APIs with different names', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
        apiDefinitionAsset,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                lambdaFunctionProps: {
                    runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                }
            }
        ]
    });
    const secondApiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'SecondOpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition-withCognitoAuth.yaml')
    });
    new lib_1.OpenApiGatewayToLambda(stack, 'second-test', {
        apiDefinitionAsset: secondApiDefinitionAsset,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                lambdaFunctionProps: {
                    runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                }
            }
        ]
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::ApiGateway::RestApi", 2);
    template.hasResourceProperties("AWS::ApiGateway::RestApi", {
        Name: {
            "Fn::Join": [
                "",
                [
                    "test-apigateway-lambda-",
                    {
                        "Fn::Select": [
                            2,
                            {
                                "Fn::Split": [
                                    "/",
                                    {
                                        Ref: "AWS::StackId"
                                    }
                                ]
                            }
                        ]
                    }
                ]
            ]
        }
    });
    template.hasResourceProperties("AWS::ApiGateway::RestApi", {
        Name: {
            "Fn::Join": [
                "",
                [
                    "second-test-",
                    {
                        "Fn::Select": [
                            2,
                            {
                                "Fn::Split": [
                                    "/",
                                    {
                                        Ref: "AWS::StackId"
                                    }
                                ]
                            }
                        ]
                    }
                ]
            ]
        }
    });
});
test('Confirm API definition uri is added to function name', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
        apiDefinitionAsset,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                lambdaFunctionProps: {
                    runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                }
            }
        ]
    });
    const template = assertions_1.Template.fromStack(stack);
    const resources = JSON.parse(JSON.stringify(template)).Resources;
    expect(Object.keys(resources).find((element) => {
        return element.includes("MessagesHandler");
    })).toBeTruthy();
});
test('Confirm  that providing both lambdaFunction and functionProps is an error', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const existingLambdaObj = new lambda.Function(stack, 'ExistingLambda', {
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler',
        functionName: 'ExistingLambdaFunction',
        code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
    });
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const props = {
        apiDefinitionAsset,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                lambdaFunctionProps: {
                    runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                },
                existingLambdaObj
            }
        ]
    };
    const app = () => {
        new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', props);
    };
    expect(app).toThrowError(`Error - Cannot provide both lambdaFunctionProps and existingLambdaObj in an ApiIntegrationfor the api integration with id: MessagesHandler`);
});
//# sourceMappingURL=data:application/json;base64,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