import * as cloudwatch from '@aws-cdk/aws-cloudwatch';
import { RemovalPolicy } from '@aws-cdk/core';
import { Construct } from 'constructs';
import { Alias, AliasOptions } from './alias';
import { EventInvokeConfigOptions } from './event-invoke-config';
import { IFunction, QualifiedFunctionBase } from './function-base';
/**
 *
 */
export interface IVersion extends IFunction {
    /**
     * The most recently deployed version of this function.
     *
     * @attribute true
     */
    readonly version: string;
    /**
     * The underlying AWS Lambda function.
     */
    readonly lambda: IFunction;
    /**
     * The ARN of the version for Lambda@Edge.
     */
    readonly edgeArn: string;
    /**
     * Defines an alias for this version.
     *
     * @param aliasName The name of the alias.
     * @param options Alias options.
     */
    addAlias(aliasName: string, options?: AliasOptions): Alias;
}
/**
 * Options for `lambda.Version`.
 */
export interface VersionOptions extends EventInvokeConfigOptions {
    /**
     * SHA256 of the version of the Lambda source code.
     *
     * Specify to validate that you're deploying the right version.
     *
     * @default No validation is performed
     */
    readonly codeSha256?: string;
    /**
     * Description of the version.
     *
     * @default Description of the Lambda
     */
    readonly description?: string;
    /**
     * Specifies a provisioned concurrency configuration for a function's version.
     *
     * @default No provisioned concurrency
     */
    readonly provisionedConcurrentExecutions?: number;
    /**
     * Whether to retain old versions of this function when a new version is created.
     *
     * @default RemovalPolicy.DESTROY
     */
    readonly removalPolicy?: RemovalPolicy;
}
/**
 * Properties for a new Lambda version.
 */
export interface VersionProps extends VersionOptions {
    /**
     * Function to get the value of.
     */
    readonly lambda: IFunction;
}
/**
 *
 */
export interface VersionAttributes {
    /**
     * The version.
     */
    readonly version: string;
    /**
     * The lambda function.
     */
    readonly lambda: IFunction;
}
/**
 * A single newly-deployed version of a Lambda function.
 *
 * This object exists to--at deploy time--query the "then-current" version of
 * the Lambda function that it refers to. This Version object can then be
 * used in `Alias` to refer to a particular deployment of a Lambda.
 *
 * This means that for every new update you deploy to your Lambda (using the
 * CDK and Aliases), you must always create a new Version object. In
 * particular, it must have a different name, so that a new resource is
 * created.
 *
 * If you want to ensure that you're associating the right version with
 * the right deployment, specify the `codeSha256` property while
 * creating the `Version.
 */
export declare class Version extends QualifiedFunctionBase implements IVersion {
    /**
     * Construct a Version object from a Version ARN.
     *
     * @param scope The cdk scope creating this resource.
     * @param id The cdk id of this resource.
     * @param versionArn The version ARN to create this version from.
     */
    static fromVersionArn(scope: Construct, id: string, versionArn: string): IVersion;
    /**
     *
     */
    static fromVersionAttributes(scope: Construct, id: string, attrs: VersionAttributes): IVersion;
    /**
     * The most recently deployed version of this function.
     */
    readonly version: string;
    /**
     * The underlying AWS Lambda function.
     */
    readonly lambda: IFunction;
    /**
     * The ARN fo the function.
     */
    readonly functionArn: string;
    /**
     * The name of the function.
     */
    readonly functionName: string;
    /**
     * The qualifier of the version or alias of this function.
     *
     * A qualifier is the identifier that's appended to a version or alias ARN.
     */
    protected readonly qualifier: string;
    /**
     * Whether the addPermission() call adds any permissions.
     *
     * True for new Lambdas, false for version $LATEST and imported Lambdas
     * from different accounts.
     */
    protected readonly canCreatePermissions = true;
    /**
     *
     */
    constructor(scope: Construct, id: string, props: VersionProps);
    /**
     * The principal this Lambda Function is running as.
     */
    get grantPrincipal(): import("@aws-cdk/aws-iam").IPrincipal;
    /**
     * The IAM role associated with this function.
     *
     * Undefined if the function was imported without a role.
     */
    get role(): import("@aws-cdk/aws-iam").IRole | undefined;
    /**
     * Return the given named metric for this Function.
     */
    metric(metricName: string, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Defines an alias for this version.
     *
     * @param aliasName The name of the alias (e.g. "live").
     * @param options Alias options.
     */
    addAlias(aliasName: string, options?: AliasOptions): Alias;
    /**
     * The ARN of the version for Lambda@Edge.
     */
    get edgeArn(): string;
    /**
     * Validate that the provisionedConcurrentExecutions makes sense
     *
     * Member must have value greater than or equal to 1
     */
    private determineProvisionedConcurrency;
}
/**
 * Given an opaque (token) ARN, returns a CloudFormation expression that extracts the
 * qualifier (= version or alias) from the ARN.
 *
 * Version ARNs look like this:
 *
 *   arn:aws:lambda:region:account-id:function:function-name:qualifier
 *
 * ..which means that in order to extract the `qualifier` component from the ARN, we can
 * split the ARN using ":" and select the component in index 7.
 *
 * @returns `FnSelect(7, FnSplit(':', arn))`
 */
export declare function extractQualifierFromArn(arn: string): string;
