"""
p5.js wrapper for Pyodide-MkDocs-Theme, by Frédéric Zinelli.

Basic use:

1. Import p5 as a namespace (no wildcard imports!):
2. Define the setup and draw callbacks, using calls to the original p6 JS functions,
   using`p5.functionName()` syntax.
3. Call `p5.run(setup, draw, preload, target="div_id")` at the end of the code (`preload`
   is optional / `target` is also optional and defaults to the current PMT option value
   for `args.figure.div_id`).

NOTE:
    * Use the `{{ figure() }}` macro to create the target elements in the page.
    * Use `{{ figure(..., p5_buttons="left") }}` (or "right") to also insert start and stop
    buttons, controlling the sketch event loop.
    * Any number of animation can be built in the page, through any number of IDEs, terminals,
    or py_btns as long as each of them targets a different DOM element.



Example:

```python
import p5

def setup():
    p5.createCanvas(200,200)
    p5.background(0)

def draw():
    p5.circle(p5.mouseX, p5.mouseY, 50)

p5.run(setup, draw)                     # Targets "figure1"
p5.run(setup, draw, target='figure2')   # Second figure with the same behavior
```


Help about p5 use:
    - https://frederic-zinelli.gitlab.io/pyodide-mkdocs-theme/custom/p5/
    - https://p5js.org/reference/


Sources:
    - https://p5js.org/
    - https://github.com/processing/p5.js/wiki/Global-and-instance-mode
    - https://forge.apps.education.fr/basthon/basthon-kernel/-/tree/master/packages/kernel-python3/src/modules/p5/p5?ref_type=heads
      (p5 adaptation for Basthon, by Romain Casati)
"""

# pylint: disable=E0401, C0103, C0116, C0321, C0415, W0105, W0613, W0621





def __define():

    import js
    from pyodide.ffi import to_js
    from pyodide.ffi.wrappers import add_event_listener, remove_event_listener

    from dataclasses import dataclass, field
    from typing import Any, Callable, Dict
    from functools import wraps


    P5_CALLBACKS_NAMES = 'setup draw preload'.split()

    JS_P5  = js.p5
    SKETCH = None
    """
    Current sketch to handle. Automatically sent back when accessing js.p5 attributes.
    Use js.p5 (global mode) if sketch is None.
    """



    @dataclass
    class P5Handler:
        """
        Manage the inner logistic linking python to p5.js, listeners and memory leaks.
        """

        setup:   Callable[[],None]
        draw:    Callable[[],None]
        preload: Callable[[],None]
        container_id: str

        _proxies: list = field(default_factory=list)
        _buttons: dict = field(default_factory=dict)
        _sketch: Any = None
        _p5instance: Any = None



        def __post_init__(self):

            # Make sure the div is emptied (needed in case a default message is in the
            # {{ figure() }} macros):
            js.document.getElementById(self.container_id).replaceChildren()

            log('Create handler')
            proxy = self._to_proxy(self.__call__)   # Already contextualized

            self._p5instance = js.p5.new(proxy, self.container_id)
            log('instance done')



        def __call__(self, js_sketch):

            log('Calling P5Handler')
            self._sketch = js_sketch            # Stored to allow later destruction

            self._handle_buttons(True)
            for prop in P5_CALLBACKS_NAMES:
                self._build_proxy_cbk_with_sketch_rotation(prop, js_sketch)
            log("Done with P5Handler call")



        def _build_proxy_cbk_with_sketch_rotation(self, prop, js_sketch):
            """
            Create and register a JsProxy function, based on the user defined python function,
            wrapped in another function that will ensure the nonlocal SKETCH object is updated
            before the user function is called so that the animation/drawing is always done in
            the right DOM element.
            """
            log(f'{prop}: creating proxy and binding sketch')

            py_cbk = getattr(self, prop) or (lambda *_a,**_kw: None)

            @wraps(py_cbk)
            def py_wrapper(*a, **kw):
                nonlocal SKETCH
                SKETCH = js_sketch
                return py_cbk(*a, **kw)

            # Create and store the JsProxy (setup, draw, ...)
            proxy = self._to_proxy(py_wrapper)

            # Patch js_sketch with the proxy:
            setattr(js_sketch, prop, proxy)
                # (See: https://github.com/processing/p5.js/wiki/Global-and-instance-mode )
                # Note: JS method binding isn't needed here because of the wrapper function:
                # all method calls are going through the module p5.__getattr__ and then are
                # redirected to the js_sketch instance.


        def _to_proxy(self, cbk):
            """
            Create a JsProxy for the given callback and store its reference for later destruction.
            """
            proxy = to_js(cbk)
            self._proxies.append(proxy)
            return proxy



        def start(self, _event): self._sketch.loop()

        def stop(self,  _event): self._sketch.noLoop()

        def _handle_buttons(self, add:bool):
            """
            Add or remove the event listeners for the start and stop buttons (if needed).

            WARNING:
                Removal of event listeners is done an a different method call, meaning the JsProxy
                sent back by `js.document.getElementById` will _NOT_ return the same instance as
                the first time. And because listeners identification is always done using (at some
                point) the id of the JS object, the proxy _HAS_ to be stored with the callback so
                that the listener removal can properly happen.
            """
            if add:
                for prop in ('start','stop'):

                    btn_id = f'{ prop }-btn-{ self.container_id }'
                    elt    = js.document.getElementById(btn_id)
                    if not elt:
                        return

                    cbk = getattr(self, prop)
                    self._buttons[prop] = (elt, cbk)
                    add_event_listener(elt, 'click', cbk)
            else:
                for elt,cbk in self._buttons.values():
                    remove_event_listener(elt, 'click', cbk)
                self._buttons.clear()



        def destroy(self):
            """
            Destroy all proxies + unbind the js objects.
            """
            nonlocal SKETCH

            log('Remove buttons listeners')
            self._handle_buttons(False)

            log('Stop drawing + cleanup p5 instance & DOM')
            # Protect against a possible failure on the previous run, implying no p5 instance:
            if self._p5instance:
                self._p5instance.remove()

            log('Destroy all JsProxies')
            for proxy in self._proxies:
                proxy.destroy()
            self._proxies.clear()

            log('Remove JsProxies references')
            self._p5instance = self._sketch = SKETCH = None





    def log(*a,**kw):
        """ Debugging logger, if needed """
        # print(*a,**kw)




    def __getattr__(prop):
        """
        Enforce p5 contracts within PMT, and handle redirections of p5 functions or constants
        at call time.
        """
        if prop=='__all__':
            raise ImportError(
                "Wildcard imports of p5 is forbidden within Pyodide-MkDocs-Theme context.\n"
                "Import the module as a namespace instead:\n    import p5\n    p5.createCanvas()"
            )
        try:
            return getattr(SKETCH or JS_P5, prop)
        except:
            raise AttributeError(f"p5.{ prop } is not defined") from None




    DEFAULT_ID: str = js.config().argsFigureDivId

    HANDLERS_TRACKER: Dict[str,P5Handler] = {}



    def run(setup:Callable, draw:Callable=None, preload:Callable=None, *, target:str=None):
        """
        Starts a p5 animation with the given callbacks and target element in the DOM.

        Example:

        ```python
        import p5

        def setup():
            p5.createCanvas(200,200)
            p5.background(0)

        def draw():
            p5.circle(p5.mouseX, p5.mouseY, 50)

        p5.run(setup, draw)                     # Targets "figure1"
        p5.run(setup, draw, target='figure2')   # Second figure with the same behavior
        ```

        Help about p5 use:
            - https://p5js.org/reference/
            - https://frederic-zinelli.gitlab.io/pyodide-mkdocs-theme/custom/p5/
        """
        target = target or DEFAULT_ID

        if target in HANDLERS_TRACKER:
            HANDLERS_TRACKER[target].destroy()

        anim = P5Handler(setup, draw, preload, target)
        HANDLERS_TRACKER[target] = anim



    return run, __getattr__

run, __getattr__ = __define()

def __dir__():
    return ['run']
