import argparse
import logging
import multiprocessing as mp
import os
from pprint import pp

from openfilter.filter_runtime.filter import Filter, PROP_EXIT_FLAGS, PROP_EXIT, OBEY_EXIT
from openfilter.filter_runtime.utils import dict_without

from .common import SCRIPT, parse_filters

logger = logging.getLogger(__name__)

logger.setLevel(int(getattr(logging, (os.getenv('LOG_LEVEL') or 'INFO').upper())))


# --- run --------------------------------------------------------------------------------------------------------------

def cmd_run(args):
    try:
        idx = args.index('-')
    except ValueError:
        idx = len(args)

    opts   = args[:idx]
    parser = argparse.ArgumentParser(prog=f'{SCRIPT} run', formatter_class=argparse.RawTextHelpFormatter,
        usage=f"""
usage: {SCRIPT} run [-h] [--ipc] [-s] [-f] [-p {{all,clean,error,none}}] [-o {{all,clean,error,none}}] [--dry] FILTER [FILTER ...]
        """.strip(),
        description="""
Run one or more Filters.

positional arguments:
  FILTER                specified as "- package.filter.Filter [--config_param value ...]"
        """.strip(),
        epilog=f"""
examples:
  Plug into a running pipeline Filter at localhost:5554 and log its output without disturbint it too much:
    {SCRIPT} run - Util --sources tcp://localhost:5554?? --log

  Read a video file and visualize it:
    {SCRIPT} run - VideoIn --sources file://video.mp4 --outputs tcp://0 - Webvis --sources tcp://localhost
      autochain filters:
    {SCRIPT} run - VideoIn --sources file://video.mp4 - Webvis

  Connect via ids:
    {SCRIPT} run - VideoIn --id myvideo --sources file://video.mp4 - webvis --sources myvideo
      autogenerated id:
    {SCRIPT} run - VideoIn --sources file://video.mp4 - Webvis --sources VideoIn

notes:
  * For --outputs, 'tcp://*', 'tcp://0.0.0.0' and 'tcp://0' are all equivalent.
        """.strip(),
    )

    parser.add_argument('--ipc',
        action = 'store_true',
        help   = 'when creating new connections make them ipc:// instead of tcp://',
    )
    parser.add_argument('-s', '--solo',
        action = 'store_true',
        help   = 'run a single Filter in same process',
    )
    parser.add_argument('-f', '--fork',
        action = 'store_true',
        help   = "run Filters using 'fork' method instead of 'spawn', doesn't work with CUDA",
    )
    parser.add_argument('-p', '--prop-exit',
        type    = str,
        default = PROP_EXIT,
        choices = list(PROP_EXIT_FLAGS),
        help    = 'exit conditions Filters will propagate (default: %(default)s)',
    )
    parser.add_argument('-o', '--obey-exit',
        type    = str,
        default = OBEY_EXIT,
        choices = list(PROP_EXIT_FLAGS),
        help    = 'exit conditions Filters will obey (default: %(default)s)',
    )
    parser.add_argument('--dry',
        action = 'store_true',
        help   = 'dry run, just prepare and show the configs that would be used',
    )

    opts = parser.parse_args(opts)

    logger.debug(f'opts: {opts}')

    # parse filters

    filters = parse_filters(args[:idx:-1], opts.ipc)

    # run

    if not opts.fork:
        mp.set_start_method('spawn')

    if opts.solo and len(filters) > 1:
        raise ValueError("can only run a single Filter '--solo'")

    if opts.dry:
        for _, config, name in filters:
            print(f'\n{name}:\n{"-" * (len(name) + 1)}')
            pp(config)

    elif opts.solo:
        filters[0][0].run(dict_without(filters[0][1], '__env_compose'),
            prop_exit=opts.prop_exit, obey_exit=opts.obey_exit)
    else:
        Filter.run_multi([(cls, dict_without(config, '__env_compose')) for cls, config, _ in filters],
            prop_exit=opts.prop_exit, obey_exit=opts.obey_exit)
