# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['imia', 'imia.ext']

package_data = \
{'': ['*']}

extras_require = \
{'aiosqlite': ['aiosqlite>=0.17.0,<0.18.0'],
 'sqlalchemy': ['SQLAlchemy>=1.4.25,<2.0.0']}

setup_kwargs = {
    'name': 'imia',
    'version': '0.5.3',
    'description': 'Full stack authentication library for ASGI.',
    'long_description': '# Imia\n\nImia (belarussian for "a name") is an authentication library for Starlette and FastAPI (python 3.8+).\n\n![PyPI](https://img.shields.io/pypi/v/imia)\n![GitHub Workflow Status](https://img.shields.io/github/workflow/status/alex-oleshkevich/imia/Lint)\n![GitHub](https://img.shields.io/github/license/alex-oleshkevich/imia)\n![Libraries.io dependency status for latest release](https://img.shields.io/librariesio/release/pypi/imia)\n![PyPI - Downloads](https://img.shields.io/pypi/dm/imia)\n![GitHub Release Date](https://img.shields.io/github/release-date/alex-oleshkevich/imia)\n![Lines of code](https://img.shields.io/tokei/lines/github/alex-oleshkevich/imia)\n\n## Production status\n\nThe library is considered in "beta" state thus may contain bugs or security issues, but I actively use it in production.\n\n## Installation\n\nInstall `imia` using PIP or poetry:\n\n```bash\npip install imia\n# or\npoetry add imia\n```\n\n## Features\n\n- Login/logout flows\n- Pluggable authenticators:\n    - WWW-Basic\n    - session\n    - token\n    - bearer token\n    - any token (customizable)\n    - API key\n- Database agnostic user storage\n- Authentication middleware\n    - with fallback strategies:\n        - redirect to an URL\n        - raise an exception\n        - do nothing\n    - with optional URL protection\n    - with option URL exclusion from protection\n- User Impersonation (stateless and stateful)\n- SQLAlchemy 1.4 (async mode) integration\n\n## TODO\n\n* remember me\n\n## A very quick start\n\nIf you are too lazy to read this doc, take a look into `examples/` directory. There you will find several files demoing\nvarious parts of this library.\n\n## How it works?\n\nHere are all moving parts:\n\n1. **UserLike** object, aka "user model" - is an arbitrary class that implements `imia.UserLike` protocol.\n2. **a user provider** - an adapter that loads user model (UserLike object) from the storage (a database).\n3. **an authenticator** - a class that loads user using the user provider from the request (eg. session)\n4. **an authentication middleware** that accepts an HTTP request and calls authenticators for a user model. The\n   middleware always populates `request.auth` with `UserToken`.\n6. **user token** is a class that holds authentication state\n\nWhen a HTTP request reaches your application, an `imia.AuthenticationMiddleware` will start handling it. The middleware\niterates over configured authenticators and stops on the first one that returns non-None value. At this point the\nrequest is considered authenticated. If no authenticators return user model then the middleware will create  _anonymous\nuser token_. The user token available in `request.auth` property. Use `user_token.is_authenticated` token property to\nmake sure that user is authenticated.\n\n## User authentication quick start\n\n1. Create a user model and implement methods defined by `imia.UserLike` protocol.\n2. Create an instance of `imia.UserProvider` that corresponds to your user storage. Feel free to create your own.\n3. Setup one or more authenticators and pass them to the middleware\n4. Add `imia.AuthenticationMiddleware` to your Starlette application\n\nAt this point you are done.\n\nHere is a brief example that uses in-memory provider for demo purpose. For production environment you should use\ndatabase backed providers like `SQLAlchemyORMUserProvider` or  `SQLAlchemyCoreUserProvider`. Also, for simplicity reason\nwe will not implement [login/logout flow](docs/login_logout.md) and will authenticate requests using API keys.\n\n```python\nfrom dataclasses import dataclass, field\n\nfrom starlette.applications import Starlette\nfrom starlette.middleware import Middleware\nfrom starlette.requests import Request\nfrom starlette.responses import JSONResponse\nfrom starlette.routing import Route\n\nfrom imia import APIKeyAuthenticator, AuthenticationMiddleware, InMemoryProvider\n\n\n@dataclass\nclass User:\n    """This is our user model. It may be an ORM model, or any python class, the library does not care of it,\n    it only expects that the class has methods defined by the UserLike protocol."""\n\n    id: str\n    password: str = \'password\'\n    scopes: list[str] = field(default_factory=list)\n\n    def get_display_name(self) -> str:\n        return self.id.split(\'@\')[0].title()\n\n    def get_id(self) -> str:\n        return self.id\n\n    def get_hashed_password(self) -> str:\n        return self.password\n\n    def get_scopes(self) -> list:\n        return self.scopes\n\n\nasync def whoami_view(request: Request) -> JSONResponse:\n    return JSONResponse({\n        \'id\': request.auth.user_id,\n        \'name\': request.auth.display_name,\n    })\n\n\nuser_provider = InMemoryProvider({\n    \'user1@example.com\': User(id=\'user1@example.com\'),\n    \'user2@example.com\': User(id=\'user2@example.com\'),\n})\n\nauthenticators = [\n    APIKeyAuthenticator(user_provider=user_provider),\n]\n\nroutes = [\n    Route(\'/\', whoami_view),\n]\n\nmiddleware = [\n    Middleware(AuthenticationMiddleware, authenticators=authenticators)\n]\n\napp = Starlette(routes=routes, middleware=middleware)\n```\n\nNow save the file to `myapp.py` and run it with [uvicorn](https://uvicorn.org) application server:\n\n```bash\nuvicorn myapp:app\n```\n\nOpen `http://127.0.0.1:8000/` and see that your request is not authenticated and user is anonymous. Let\'s pass API key\nvia query parameters to make the configured APIKeyAuthenticator to load user. This time\nopen `http://127.0.0.1:8000/?apikey=user1@example.com` in your browser. Now the request is fully authenticated as User1\nuser.\n\nFor more details refer to the doc sections below.\n\n## Docs\n\n1. [UserLike protocol (a user model)](docs/userlike_protocol.md)\n2. [Load user from databases using User Providers](docs/user_providers.md)\n6. [Request authentication](docs/authentication.md)\n7. [Built-in authenticators](docs/authenticators.md)\n5. [User token](docs/user_token.md)\n5. [Passwords](docs/password_verification.md)\n4. [Login/Logout flow](docs/login_logout.md)\n8. [User impersontation](docs/impersonation.md)\n\n## Usage\n\nSee [examples/](examples) directory.\n',
    'author': 'Alex Oleshkevich',
    'author_email': 'alex.oleshkevich@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/alex-oleshkevich/imia',
    'packages': packages,
    'package_data': package_data,
    'extras_require': extras_require,
    'python_requires': '>=3.8.0,<4.0.0',
}


setup(**setup_kwargs)
