"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.findUp = exports.nodeMajorVersion = exports.parseStackTrace = void 0;
const fs = require("fs");
const path = require("path");
// From https://github.com/errwischt/stacktrace-parser/blob/master/src/stack-trace-parser.js
const STACK_RE = /^\s*at (?:((?:\[object object\])?[^\\/]+(?: \[as \S+\])?) )?\(?(.*?):(\d+)(?::(\d+))?\)?\s*$/i;
/**
 * Parses the stack trace of an error
 */
function parseStackTrace(error) {
    const err = error || new Error();
    if (!err.stack) {
        return [];
    }
    const lines = err.stack.split('\n');
    const stackTrace = [];
    for (const line of lines) {
        const results = STACK_RE.exec(line);
        if (results) {
            stackTrace.push({
                file: results[2],
                methodName: results[1],
                lineNumber: parseInt(results[3], 10),
                column: parseInt(results[4], 10),
            });
        }
    }
    return stackTrace;
}
exports.parseStackTrace = parseStackTrace;
/**
 * Returns the major version of node installation
 */
function nodeMajorVersion() {
    return parseInt(process.versions.node.split('.')[0], 10);
}
exports.nodeMajorVersion = nodeMajorVersion;
/**
 * Find a file by walking up parent directories
 */
function findUp(name, directory = process.cwd()) {
    const absoluteDirectory = path.resolve(directory);
    if (fs.existsSync(path.join(directory, name))) {
        return directory;
    }
    const { root } = path.parse(absoluteDirectory);
    if (absoluteDirectory === root) {
        return undefined;
    }
    return findUp(name, path.dirname(absoluteDirectory));
}
exports.findUp = findUp;
//# sourceMappingURL=data:application/json;base64,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