"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const package_json_1 = require("delay/package.json");
const fs = require("fs");
const path = require("path");
const bundling_1 = require("../lib/bundling");
const util = require("../lib/util");
jest.mock('@aws-cdk/aws-lambda');
const writeFileSyncMock = jest.spyOn(fs, 'writeFileSync').mockReturnValue();
const existsSyncOriginal = fs.existsSync;
const existsSyncMock = jest.spyOn(fs, 'existsSync');
const originalFindUp = util.findUp;
const findUpMock = jest.spyOn(util, 'findUp').mockImplementation((name, directory) => {
    if (name === 'package.json') {
        return path.join(__dirname, '..');
    }
    return originalFindUp(name, directory);
});
beforeEach(() => {
    jest.clearAllMocks();
});
test('Parcel bundling', () => {
    bundling_1.Bundling.parcel({
        entry: '/project/folder/entry.ts',
        runtime: aws_lambda_1.Runtime.NODEJS_12_X,
        cacheDir: '/cache-dir',
        projectRoot: '/project',
        parcelEnvironment: {
            KEY: 'value',
        },
    });
    // Correctly bundles with parcel
    expect(aws_lambda_1.Code.fromAsset).toHaveBeenCalledWith('/project', {
        assetHashType: core_1.AssetHashType.BUNDLE,
        bundling: expect.objectContaining({
            environment: {
                KEY: 'value',
            },
            volumes: [{ containerPath: '/parcel-cache', hostPath: '/cache-dir' }],
            workingDirectory: '/asset-input/folder',
            command: [
                'bash', '-c', 'parcel build /asset-input/folder/entry.ts --target cdk-lambda --cache-dir /parcel-cache',
            ],
        }),
    });
    // Correctly updates package.json
    const call = writeFileSyncMock.mock.calls[0];
    expect(call[0]).toMatch('package.json');
    expect(JSON.parse(call[1])).toEqual(expect.objectContaining({
        'cdk-lambda': '/asset-output/index.js',
        'targets': {
            'cdk-lambda': {
                context: 'node',
                includeNodeModules: {
                    'aws-sdk': false,
                },
                sourceMap: false,
                minify: false,
                engines: {
                    node: '>= 12',
                },
            },
        },
    }));
    // Searches for the package.json starting in the directory of the entry file
    expect(findUpMock).toHaveBeenCalledWith('package.json', '/project/folder');
});
test('Parcel with Windows paths', () => {
    bundling_1.Bundling.parcel({
        entry: 'C:\\my-project\\lib\\entry.ts',
        runtime: aws_lambda_1.Runtime.NODEJS_12_X,
        cacheDir: '/cache-dir',
        projectRoot: 'C:\\my-project',
    });
    expect(aws_lambda_1.Code.fromAsset).toHaveBeenCalledWith('C:\\my-project', expect.objectContaining({
        bundling: expect.objectContaining({
            command: expect.arrayContaining([
                expect.stringContaining('/lib/entry.ts'),
            ]),
        }),
    }));
});
test('Parcel bundling with externals and dependencies', () => {
    bundling_1.Bundling.parcel({
        entry: '/project/folder/entry.ts',
        runtime: aws_lambda_1.Runtime.NODEJS_12_X,
        projectRoot: '/project',
        externalModules: ['abc'],
        nodeModules: ['delay'],
    });
    // Correctly bundles with parcel
    expect(aws_lambda_1.Code.fromAsset).toHaveBeenCalledWith('/project', {
        assetHashType: core_1.AssetHashType.BUNDLE,
        bundling: expect.objectContaining({
            command: [
                'bash', '-c',
                'parcel build /asset-input/folder/entry.ts --target cdk-lambda && mv /asset-input/.package.json /asset-output/package.json && cd /asset-output && npm install',
            ],
        }),
    });
    // Correctly updates package.json
    const call = writeFileSyncMock.mock.calls[0];
    expect(call[0]).toMatch('package.json');
    expect(JSON.parse(call[1])).toEqual(expect.objectContaining({
        targets: expect.objectContaining({
            'cdk-lambda': expect.objectContaining({
                includeNodeModules: {
                    delay: false,
                    abc: false,
                },
            }),
        }),
    }));
    // Correctly writes dummy package.json
    expect(writeFileSyncMock).toHaveBeenCalledWith('/project/.package.json', JSON.stringify({
        dependencies: {
            delay: package_json_1.version,
        },
    }));
});
test('Detects yarn.lock', () => {
    existsSyncMock.mockImplementation((p) => {
        if (/yarn.lock/.test(p.toString())) {
            return true;
        }
        return existsSyncOriginal(p);
    });
    bundling_1.Bundling.parcel({
        entry: '/project/folder/entry.ts',
        runtime: aws_lambda_1.Runtime.NODEJS_12_X,
        projectRoot: '/project',
        nodeModules: ['delay'],
    });
    // Correctly bundles with parcel
    expect(aws_lambda_1.Code.fromAsset).toHaveBeenCalledWith('/project', {
        assetHashType: core_1.AssetHashType.BUNDLE,
        bundling: expect.objectContaining({
            command: expect.arrayContaining([
                expect.stringMatching(/yarn\.lock.+yarn install/),
            ]),
        }),
    });
});
//# sourceMappingURL=data:application/json;base64,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